(* ---------------------------------------------------------------------------- 

   Title: Transmogrify.m                                                       

   Authors:
     Shaun McCance <shaunm@wolfram.com>
     Stephen Layland <layland@wolfram.com>

   Version: $Id: Transmogrify.m,v 1.136.2.1 2016/09/20 03:51:07 billw Exp $ 

   Discussion:                                                                 
                                                                               
   Transmogrify is an attempt at implementing the idea of XSLT (Extensible     
   Stylesheet Language Translation) in top level Mathematica code.  As XSLT    
   makes single-source documents possible in XML, Transmogrify allows you to   
   convert a Mathematica notebook into any type of XML you desire.  It is      
   currently used to create many of the popular Wolfram websites, like         
   mathworld.wolfram.com, functions.wolfram.com, and documents.wolfram.com. 
   
   Originally implemented by Shaun McCance based off of original ideas         
   by Andy Hunt, it was later completely rewritten by Stephen Layland.
   Then Shaun gutted it and put it back together in its current form.
   May the cycle continue forever.

  ---------------------------------------------------------------------------- *)


(** Usage Statements **)

Transmogrify::usage = "Transmogrify converts Mathematica notebooks into\
 SymbolicXML based on transmogrification rules specified in a XMLTransform.";

TransmogrifyString::usage = "TransmogrifyString converts Mathematica expressions into\
 XML based on transmogrification rules specified in a XMLTransform.";


(* Transmogrify Options *)
AbortOnError::usage = "AbortOnError is an option to Transmogrify which forces\
 Transmogrify to quit processing if it encounters a substantial error.  Setting\
 this to False causes Transmogrify to attempt to continue using some built in\
 guesses."

AtomsReturnSelf::usage = "Option for Transmogrify which explains what to do when\
 Transmogrify tries to recurse over an atomic expression.  If set to false, it will\
 return Null, otherwise it will return the result of SelectSelf[] at that iteration."

AutoRecurse::usage = "Option for Transmogrify that forces automatic recursion\
 to an expressions children if the expression did not change after rule application.
 Note that if returning SelectSelf[] can also cause this to recurse, which you may
 not want."

CreateImages::usage = "Option for Transmogrify to effectively ignore BoxToImage\
 calls when True."

DefaultParameters::usage = "DefaultParameters is an option for Transmogrify that\
 specifies a list of parameters to be used in the XMLTransform code via GetParameter[]."

ExportFormat::usage = "ExportFormat is an option for Transmogrify that tells\
 it what file format to export the resulting transformed expression to."
ExportOptions::usage = "List of Options that will be passed to Export[] when\
 Transmogrify exports the resulting transformed expression."

InputDirDepth::usage = "InputDirDepth is an option for Transmogrify that is\
 only used when calling it on an input directory (batch mode).  For example\
 \n\nTransmogrify[\"outputdir\", \"inputdir\", \"transform\",\
 InputDirDepth->Infinity]\n\nwill tell transmogrify to convert all notebooks\
 in \"inputdir\" and its subdirectories.  See FileNames[]."

MaxExportErrors::usage = "Option for Transmogrify to specify how many errors\
 in generated SymbolicXML code are allowed before not attempting to Export the\
 final result.  When Transmogrify fails to export, the problematic XMLOutput is\
 saved in $XMLOutput, with errors in $XMLErrors.  This option obviously only\
 has any effect if ExportFormat is set to \"XML\" "

OutputExtension::usage = "OuptutExtension is an option for Transmogrify\
 that only has an effect when called in Batch mode.  For example, when called\
 as\n\nTransmogrify[\"outputdir\",\"inputdir\",\"transformfile\",\
 OutputExtension->\"html\"]\n\nAll .nb files in \"inputdir\" will be turned\
 into filename.html"


(* ------------------------------------------------------------- *)

XMLTransform::usage = "XMLTransform is a container for specially formatted\
 transform rules (like XSLT templates) that are used during conversion to XML.\
 It can also contain useful options like IncludeXMLTransforms, IgnoreBoxes, etc."
 
XMLTransformQ::usage = "XMLTransformQ[expr] returns true if expr is an XMLTransform."

IncludeXMLTransforms::usage = "Option for XMLTransform.  IncludeXMLTransforms\
 specifies a list of transforms to be included.  IncludeXMLTransforms can be a\
 single filename or a list of filenames.  See filename specifications in the\
 Transmogrify documentation."

WhichXMLTransform::usage = "WhichXMLTransform[\"filename\"] gives the full path name\
 of \"filename\" that Transmogrify found in $XMLTransformPath"

GetXMLTransform::usage = "GetXMLTransform[f] - imports the XMLTransform specified by\
 the XMLTransform file \"f\", merging all transforms included via the\
 IncludeXMLTransforms option.  See the documentation for available forms of a\
 transformfile name."

(* ------------------------------------------------------------- *)

$OutputFile::usage = "The file currently being written to."
$ExportImageFormat::usage = "Default format that Transmogrify exports Images as.  Default\
 is \"GIF\""
$TransmogrifyDirectory::usage = "$TransmogrifyDirectory is the path to where Transmogrity Add-on is installed."
$XMLTransformPath::usage = "List of paths where Transmogrify looks for XMLTransforms."
$XMLTransforms::usage = "$XMLTransforms returns the list of template names available on $XMLTransformsPath."
$DefaultXMLTransforms::usage = "$DefaultXMLTransforms returns the list of default template names available from Transmogrify's XMLTransforms directory."
$XMLErrors::usage = "List of positions where the last generated SymbolicXML had errors."
$XMLOutput::usage = "Container for last generated SymbolicXML.  Only populated if \
 Transmogrify did not try to Export (see MaxExportErrors)."
$DebugMode::usage = "Boolean flag to indicate Transmogrify debugging status."

(* ------------------------------------------------------------- *)
(* userland XMLTransform functions *)

BoxToImage::usage = "This function is a simple interface to the frontend's ConvertTo\
 functionality.  BoxToImage[expr, fmt, opts]."

NewFile::usage = "NewFile[\"outfile\", expr] processes expr and exports them to\
 outfile, setting the Transmogrify variable for the output file."

(* BoxToImage Options *)
Inline::usage = "Inline->True tells BoxToImage to not draw borders and other cell\
 decorations, allowing for seamless inline images.";
TransparentBackground::usage = "TransparentBackground->True tells BoxToImage to not have a background";
MaxImageSize::usage = "Option for BoxToImage.  MaxImageSize->x will resize all images\
 that are wider than x pixels.  MaxImageSize->{x,y} will resize all images larger\
 than x by y pixels.  MaxImageSize->None or MaxImageSize->Infinity tells BoxToImage\
 to leave the image size alone.  See CropImage for more.";
CropImage::usage = "CropImage -> True tells BoxToImage to crop images larger than\
 the value given in the MaxImageSize option.  CropImage -> False simply rescales the\
 image to fit in the dimensions specified by MaxImageSize.";
StringContent::usage "StringContent -> \"s\" tells BoxToImage to use altnerate string\
 content when called on a string.  This can be used to create images for substrings."
ImageFormat::usage = "ImageFormat is an option to BoxToImage that allows saving the\
 image as a format of your choice.  ImageFormat->Automatic tells BoxToImage to use\
 $ExportImageFormat as the default format type."


GetAbsoluteFileName::usage = "GetAbsoluteFileName retrieves the absolute\
 filename to which Transmogrify is currently writing the converted content. \
 If Transmogrify is not writing to a file, GetAbsoluteFileName returns None."
GetFileName::usage = "GetFileName retrieves the filename to which Transmogrify\
 is currently writing the converted content.  If Transmogrify is not writing to\
 a file, GetFileName returns None."
GetFileNameBase::usage = "GetFileNameBase retrieves the portion of the current\
 filename before the extension."
GetFileExtension::usage = "GetFileExtension returns the extension of the\
 current filename, without the leading dot."

GetOption::usage = "GetOption gets the specified option or list of nested\
 options from a box structure.  If the option is not set, it returns None."
HasOption::usage = "Has Option returns True if the box structure has the\
 specified option or list of nested options set, False otherwise."

$ShowStringCharacters::usage = "";

StringSubstitute::usage = "StringSubstitute[rules] replaces all string elements\
 and substrings of the current expression by applying the pattern->value rules.\
 note that the value need not be a string.  StringSubstitute[expr,rules] does\
 the same thing for a generic expression."

StringValue::usage = "Returns a string of all the String content in an expression."

GetParameter::usage = "GetParameter retrieves the value of a parameter set\
 with the DefaultParameters option."
WithParameters::usage = "Use Block or Module instead, please."


GetStyle::usage = "Get the computed style for the current expression"
GroupOpenQ::usage = "True if the current CellGroup expression is open, otherwise False"
HasStyle::usage = "Whether or not the current expression has a style"

IncrementCounter::usage = "IncrementCounter[\"s\"] - XMLTransform function that\
 returns a string representation of the incremented counter for \"s\"."
DecrementCounter::usage = "DecrementCounter[\"s\"] - XMLTransform function that\
 returns a string representation of the decremented counter for \"s\"."
GetCounter::usage = "GetCounter[\"s\"] - retrieves the current value of the\
 counter for \"s\"."
ResetCounters::usage = "ResetCounters[] resets all counters.  ResetCounter[\"s\"]\
 resets only the counter for \"s\""

SelectSelf::usage = "SelectSelf selects the box structure currently in context"
SelectLiteral::usage = "SelectLiteral[] has been deprecated, use SelectSelf[]"

SelectParent::usage = "SelectParent[] selects the parent of the box structure\
 currently in context.  SelectParent[int] selects the parent plus int ancestors.\
 SelectParent[0] is equivalent to SelectParent[].  Returns None on fail."

SelectAncestors::usage = "SelectAncestors[] selects all successive parent expressions\
 through the document root.  Current expression is represented as Slot[] though a head\
 of Function is not yet applied.  SelectAncestors[selector] returns a list of all\
 Ancestors that match selector."

SelectAncestorsAndSelf::usage = "SelectAncestorsAndSelf selects all ancestor\
 expressions and the expression currently in context.  Effectively, the entire\
 branch of the tree currently being processed."

SelectNearestAncestor::usage = "SelectNearestAncestor[selector] returns the\
 nearest ancestor expression that matches selector. Current expression is\
 represented by Slot[] though a head of Function is not yet applied."

SelectNearestAncestorAndSelf::usage = "Same as SelectNearestAncestor[selector]\
 but with Slot[] replaced by the current expression."

SelectChildren::usage = "SelectChildren[] selects all expressions one level deep inside\
 of the expression currently in context.  SelectChildren[ selector ] only matches\
 the expressions that will match selector.  SelectChildren[ n ] returns the nth
 child, leaving all others unevaluated."

SelectDescendants::usage = "SelectDescendants selects all boxes at any level\
 inside of the expression currently in context. Identical to SelectChildren[]\
 except when a selector is given."

SelectDescendantsAndSelf::usage = "SelectDescendantsAndSelf selects descendant\
 expressions and the expression currently in context."

SelectNearestDescendants::usage = "SelectNearestDescendants[selector] - selects\
 the nearest descendant matching selector.  If more than one expression at the\
 same level match, return all of them wrapped in ExprSequence[]."

SelectSiblings::usage = "SelectSiblings selects the children of the parent of\
 the context box structure."

SelectGridRows::usage = "Select the rows of a Grid or GridBox.  Each row is\
 wrapped in GridRow[]."
SelectGridColumns::usage = "Select the columns of a Grid or GridBox.  Each column\
 is wrapped in GridColumn[]."
GridRow::usage = "Symbolic wrapper for Grid and GridBox rows"
GridColumn::usage = "Symbolic wrapper for Grid and GridBox columns"

Recurse::usage = "Recurse[] is a shorthand for Transmogrify[SelectChildren[]]\
 for use in XMLTransforms."

ExprSequence::usage = "ExprSequence[] is a container symbol that is used by Transmogrify\
 to recursively process mathematica expressions that contain a non-option\
 Sequence[]."
 
TransmogrifyInfo::usage = "TransmogrifyInfo[] gives a bunch of information about the\
 currently loaded version of Transmogrify.  Transmogrify[sym] returns the value for\
 that symbol."

(* XMLTransform functions *)
P
DIV
Ol
Li
H1
H2
H3
H4
TD
Span
A
UL
Br
Img
Image
transmogrifyImage
ImageList
IfCommentSSI


(* -------------------------------------------------------------
                   I m p l e m e n t a t i o n
   ------------------------------------------------------------- *)

(* set up some Debug info *)
If[TrueQ[$DebugMode],
  Print["Transmogrify Debug Mode Enabled"];
  $Profile::usage = "List which holds the results of the \"NBProfile\" trace."
  ,
  Get["Transmogrify`Debug`Null`"];
]

`Information`$CVSVersion = "$Id: Transmogrify.m,v 1.136.2.1 2016/09/20 03:51:07 billw Exp $"

(* ---------------------------------
      Begin Transmogrify`Private`
   --------------------------------- *)
Begin["`Private`"]

$BoxTypes = {
  ActionMenuBox -> {"Children" -> {}},
  AdjustmentBox -> {},
  AnimatorBox -> {"Children" -> {}},
  ArrowBox -> {"Children" -> {}},
  BoxData -> {},
  ButtonBox -> {"Styles" -> (Cases[#, _[BaseStyle, s_] :> s]&)},
  Cell -> {
    "Styles" -> (Cases[Rest[#], _String] &), 
    "NodeID" -> True},
  CellBoundingBox -> {"Children" -> {}},
  CellElementsBoundingBox -> {"Children" -> {}},
  CellGroupData -> {"Styles" -> (Cases[Rest[#[[1, 1]]], _String] &)},
  CheckboxBox -> {"Children" -> {}},
  CircleBox -> {"Children" -> {}},
  ColorSetterBox -> {"Children" -> {}},
  ContentsBoundingBox -> {"Children" -> {}},
  CounterBox -> {"Children" -> {}},
  CuboidBox -> {"Children" -> {}},
  CylinderBox -> {"Children" -> {}},
  DiskBox -> {"Children" -> {}},
  DynamicBox -> {"Children" -> {}},
  DynamicModuleBox -> {"Children" -> {}},
  DynamicWrapperBox -> {"Children" -> {}},
  ErrorBox -> {},
  FormBox -> {"Styles" -> ({#[[2]]}&)},
  FractionBox -> {"Children" -> {{1}, {2}}},
  FrameBox -> {},
  GeometricTransformationbox -> {"Children" -> {}},
  Graphics3DBox -> {"Children" -> {}},
  GraphicsBox -> {"Children" -> {}},
  GraphicsComplex3DBox -> {"Children" -> {}},
  GraphicsComplexBox -> {"Children" -> {}},
  GraphicsData -> {"Children" -> {}},
  GraphicsGridBox -> {"Children" -> {}},
  GraphicsGroupBox -> {"Children" -> {}},
  GridBox -> {},
  GridColumn -> {"Children" -> All},
  GridRow -> {"Children" -> All},
  InputFieldBox -> {"Children" -> {}},
  Inset3DBox -> {"Children" -> {}},
  InsetBox -> {"Children" -> {}},
  InterpretationBox -> {},
  ItemBox -> {"Children" -> {}},
  Line3DBox -> {"Children" -> {}},
  LineBox -> {"Children" -> {}},
  List -> {"Children" -> All},
  LocatorBox -> {"Children" -> {}},
  LocatorPaneBox -> {"Children" -> {}},
  Notebook -> {"NodeID" -> True},
  OpenerBox -> {"Children" -> {}},
  OptionValueBox -> {"Children" -> {}},
  OverscriptBox -> {"Children" -> {{1}, {2}}},
  PaneBox -> {"Children" -> {}},
  PanelBox -> {"Children" -> {}},
  PaneSelectorBox -> {"Children" -> {}},
  Point3DBox -> {"Children" -> {}},
  PointBox -> {"Children" -> {}},
  Polygon3DBox -> {"Children" -> {}},
  PolygonBox -> {"Children" -> {}},
  PopupMenuBox -> {"Children" -> {}},
  ProgressIndicatorBox -> {"Children" -> {}},
  RadicalBox -> {"Children" -> {{1}, {2}}},
  RadioButtonBox -> {"Children" -> {}},
  RasterBox -> {"Children" -> {}},
  RawData -> {},
  RectangleBox -> {"Children" -> {}},
  RotationBox -> {"Children" -> {}},
  RowBox -> {},
  SetterBox -> {"Children" -> {}},
  ShrinkWrapBoundingBox -> {"Children" -> {}},
  Slider2DBox -> {"Children" -> {}},
  SliderBox -> {"Children" -> {}},
  SphereBox -> {"Children" -> {}},
  SqrtBox -> {},
  StyleBox -> {
    "Styles" -> (Cases[Rest[#], _String] &), 
    "NodeID" -> True},
  SubscriptBox -> {"Children" -> {{1}, {2}}},
  SubsuperscriptBox -> {"Children" -> {{1}, {2}, {3}}},
  SuperscriptBox -> {"Children" -> {{1}, {2}}},
  TabViewBox -> {"Children" -> {}},
  TagBox -> {},
  TemplateBox -> {"Children" -> {}},
  Text3DBox -> {"Children" -> {}},
  TextBoundingBox -> {"Children" -> {}},
  TextBox -> {"Children" -> {}},
  TextData -> {},
  TogglerBox -> {"Children" -> {}},
  TooltipBox -> {"Children" -> {{1}, {2}}},
  UnderoverscriptBox -> {"Children" -> {{1}, {2}, {3}}},
  UnderscriptBox -> {"Children" -> {{1}, {2}}},
  ValueBox -> {}
};




(* list of public symbols that will not be Protected|ReadProtected *)
$WriteableSymbols = List@@Unevaluated/@Hold[
  $XMLTransforms,
  $XMLTransformPath,
  $DefaultXMLTransforms,
  $XMLErrors,
  $XMLOutput,
  $ExportImageFormat,
  $Profile,
  $ShowStringCharacters
]

(* munge the context path to add some useful symbols.  Make sure
   System`ConvertersDump` is in the path by calling Import; *)
Import;
$ContextPath = Join[$ContextPath, {"Transmogrify`Debug`"}]

(* some Debug helper shortcuts *)
`trace = Transmogrify`Debug`Private`trace
`traceOpts = Transmogrify`Debug`Private`traceOpts

`bag = Internal`Bag
`stuffBag = Internal`StuffBag
`bagLength = Internal`BagLength
`bagPart = Internal`BagPart

(* --------------------- 
         s e t u p       
   --------------------- *)

(* setup the XMLTransform search path.  Searched from first to last element *)
(* a la GUIKit, find any installed package with an XMLTransforms subdir *)  

$XMLTransformPath = Module[
  {dir, appPaths = {}, appDirs, transformDirs},
    
  dir = If[ NameQ["$InstallationDirectory"], ToExpression["$InstallationDirectory"], $UserAddOnsDirectory];
  If[ StringQ[dir], 
    PrependTo[appPaths, ToFileName[{dir, "AddOns", "ExtraPackages"}]];
    PrependTo[appPaths, ToFileName[{dir, "AddOns", "StandardPackages"}]];
    PrependTo[appPaths, ToFileName[{dir, "AddOns", "Autoload"}]];
    PrependTo[appPaths, ToFileName[{dir, "AddOns", "Applications"}]];
  ];
  dir = If[ NameQ["$BaseDirectory"], ToExpression["$BaseDirectory"], $AddOnsDirectory];
  If[StringQ[dir],
    (* This branch is for 4.2 and later (4.1.5 on Mac OSX). *)
    PrependTo[appPaths, ToFileName[{dir, "Autoload"}]];
    PrependTo[appPaths, ToFileName[{dir, "Applications"}]];
  ];
  dir = If[ NameQ["$UserBaseDirectory"], ToExpression["$UserBaseDirectory"], $UserAddOnsDirectory];
  If[StringQ[dir],
    (* 4.2 and later *)
    PrependTo[appPaths, ToFileName[{dir, "Autoload"}]];
    PrependTo[appPaths, ToFileName[{dir, "Applications"}]];
    ,
    (* else *)
    PrependTo[appPaths, ToFileName[{$PreferencesDirectory, "AddOns", "Autoload"}]];
    PrependTo[appPaths, ToFileName[{$PreferencesDirectory, "AddOns", "Applications"}]];
  ];

  (* FileNames sorts all results so we need to apply this to each appPaths separate to 
     preserve desired path order.  ($TransmogrifyDirectory prepended later) *)
  appDirs = Select[ Flatten[FileNames["*", #]& /@ appPaths], 
    (FileType[#] === Directory && # =!= $TransmogrifyDirectory)&];

  transformDirs = Select[ToFileName[{#, "XMLTransforms"}]& /@ appDirs, (FileType[#] === Directory)&];
  
  (* put these first so Applications don't take precedence over any of these *)
  Join[
    ToFileName[{Sequence@@#,"XMLTransforms"}]&/@
      {
        $TransmogrifyDirectory, (* first check the transforms included in release *)
        {$UserBaseDirectory,"SystemFiles"},     (* then any ones defined by the user *)
        {$BaseDirectory,"SystemFiles"},
                                (* then those defined by the site for all versions *)
        {$InstallationDirectory, "SystemFiles"}
                                (* then those defined by the site for this version *)
      }
    ,
    transformDirs
  ]
]


$XMLTransforms := Select[ FileNames["*.m", $XMLTransformPath, Infinity], XMLTransformQ[#]& ]

$DefaultXMLTransforms := Select[ FileNames["*.m", ToFileName[{$TransmogrifyDirectory}, "XMLTransforms"], Infinity], XMLTransformQ[#]& ]


`trace[{TransformParsing, 1},"Initialized $XMLTransformPath to ",$XMLTransformPath]

(* set default $ExportImageFormat if not already specified. *)
If[ !MatchQ[$ExportImageFormat, _String], $ExportImageFormat = "GIF"]

(* decide what package import/export format to use *)
$PackageFormat = If[$VersionNumber >= 6 && 
  Import =!= Head[Internal`DeactivateMessages[ImportString["blah","Package"],Import::format]],
  "Package", "Expression"]

(* the following variable is used by certain recursive functions to 
   switch between different methods for best efficiency *)
Transmogrify`Internal`$SmallExpressionThreshold = 2000

(* nice helper function to get package info. . . i think this
   can be generalized with something like PackageInfo[Context]
   to be a generic interface to a pac(kage|let)s `Information`
   variables *)
SetAttributes[TransmogrifyInfo,{HoldAll,Listable}]

TransmogrifyInfo[] := {#,TransmogrifyInfo[#]}&/@StringReplace[
  Names["Transmogrify`Information`*"], RegularExpression[".*`"]->""]

TransmogrifyInfo[x_] := Module[
  { s="Transmogrify`Information`"<>ToString@Unevaluated@x },
  If[!MemberQ[Names["Transmogrify`Information`*"],s],
    Print["No information available"],
    ToExpression[s]
  ]
]
TransmogrifyInfo[x__] := TransmogrifyInfo[{x}]

(* -----------------------
      h e l p e r  Q s
   ----------------------- *)

(** XMLTransformQ
  XMLTransform files can be specified in the following ways:

  Transmogrify[stuff,"transformFile"]              
  Transmogrify[stuff,{"dir","transformFile"}]
  Transmogrify[stuff,"/absolute/path/to/transformFile.m"] (for Windows, too)
  Transmogrify[stuff,"~/transformFile.m"]
  Transmogrify[stuff,"http://some/site/transformFile.m"]
*)
XMLTransformQ[{__String},_String]=True
XMLTransformQ[expr_]:= MatchQ[expr,
  (
    _String|
    {__String}|
    {{__String},_String}
  )]
XMLTransformQ[___]=False

TransformQ[_?XMLTransformQ|_XMLTransform]=True
TransformQ[___]=False

(** 
  profileQ
  this function is for debugging purposes and tells the Transmogrify
  engine that it needs to count which rules are matched by the transmogrify
  expression.
**)
profileQ:=(TrueQ[$DebugMode] && MemberQ[Transmogrify`Debug`$TraceList,NBProfile|{NBProfile,___?OptionQ}])

(*****************************************************************************
            T r a n s m o g r i f y   D e f i n i t i o n s

    Transmogrify[] is going to be the only public function that actually 
    returns something.  Every other public function's evaluation will be
    held until actually called from Transmogrify[].  This allows users to
    write Select* functions in their XMLTransform rules and have them not
    be evaluated until they will mean something.

    All other public utility functions should live in Utilities.m
******************************************************************************)


Options[Transmogrify] = {
    AbortOnError -> True,
    AtomsReturnSelf -> True,
    AutoRecurse -> True,
    CreateImages -> True,
    ExportOptions -> {"AttributeQuoting"->"\""},
    ExportFormat -> "XML",
    (*IgnoreBoxes -> {},*)
    InputDirDepth -> 1,
	Literalize -> True,
    MaxExportErrors -> Infinity,
    OutputExtension -> "html"
}

Options[TransmogrifyString] = Options[Transmogrify]


(** Messages **)
Transmogrify::childindex="SelectChildren[`1`] doesn't exist for `2`"
Transmogrify::cantselect = "Can't select the `1` of node `2`.";
Transmogrify::ambiguous = "The box `1` has an ambiguous position.";
Off[Transmogrify::ambiguous];
Transmogrify::noboxinfo = "No information about box type `1`.";
Transmogrify::noparam = "Could not resolve the parameter `1`.";
Transmogrify::nofileop = "When called without a filename argument, Transmogrify\
 cannot use the operation `1`.";
Transmogrify::badxml = "Export as XML failed due to bad markup.  Please fix your transforms.\
 The generated output was saved as $XMLOutput, with error positions as $XMLErrors for your\
 debugging pleasure."
Transmogrify::baddir = "The output file specified is in a directory that couldn't be\
 written to."

SelectGridRows::notgrid = SelectGridColumns::notgrid = "`1` is not a Grid or GridBox expression"


(** GetChildPositions **)
GetChildPositions[expr_] := Module[{boxinfo, cpos},
  boxinfo = Head[expr] /. $BoxTypes;
  If[boxinfo === Head[expr],
    Message[Transmogrify::noboxinfo, ToString[Head[expr]]];
    cpos = All,
    cpos = "Children" /. boxinfo /. "Children" -> {{1}}
    ];
  Which[
    cpos === All,
    cpos = DeleteCases[Position[expr, arg_ /; FreeQ[{Rule, RuleDelayed}, Head[arg]], 1], {0}],
    cpos === {{1}} && Head[First[expr]] === List,
    cpos = {1, #} & /@ Range[Length[First[expr]]]
  ];
  cpos
];


(** ExtractOne **)
ExtractOne[expr_, pos_] := If[Length[pos] == 0, expr, Extract[expr, pos]];


(** GetNodeID **)
GetNodeID[_String] := None;
GetNodeID[expr_] := If[MatchQ[expr, _[___, "NodeID" -> _, ___]],
  First[Cases[expr, ("NodeID" -> id_) :> id]],
  expr];


(** GetNodeSourcePosition *)
GetNodeSourcePosition[expr_, opts___?OptionQ]  := Module[{pos, parents, position, ppos},
  pos = $NodePosition[GetNodeID[expr]];
  If[ListQ[pos], Return[{$Notebook, pos}]];

  parents = Parents /. {opts} /. {Parents -> $Parents};
  position = Position /. {opts} /. {Position -> $Position};
  If[Length[parents] == 0,
    Return[{expr, {{}}}] ];

  ppos = GetNodeSourcePosition[First[parents], Parents -> Rest[parents], Position -> Rest[position]];
  {First[ppos], Append[Last[ppos], First[position]]}
];


(** GetNodePosition **)
GetNodePosition[expr_, opts___?OptionQ] :=
  Module[{spos = GetNodeSourcePosition[expr, opts]},
    If[Head[First[spos]] === Notebook, Last[spos], None]];


(** ParseNotebook **)
ParseNotebook[nb_] :=
  Block[{$NodeIDCounter = 0, $RecursionLimit = 1024}, ParseExpr[nb, {{}}]];


(** ParseExpr **)
ParseExpr[s_String, ___] := s;
ParseExpr[s_Integer, ___] := s;
ParseExpr[expr_, pos_] := Module[{boxinfo, cpos, children, out, nid},
  boxinfo = Head[expr] /. $BoxTypes;
  If[boxinfo === Head[expr],
    Message[Transmogrify::noboxinfo, ToString[Head[expr]]];
    cpos = All,
    cpos = "Children" /. boxinfo /. "Children" -> {{1}}
    ];
  Which[
    cpos === All,
      cpos = DeleteCases[Position[expr, arg_ /; FreeQ[{Rule, RuleDelayed}, Head[arg]], 1], {0}],
    cpos === {{1}} && Head[First[expr]] === List,
      cpos = {1, #} & /@ Range[Length[First[expr]]]
  ];
  If[Length[cpos] > 0,
    children = Map[ParseExpr[Extract[expr, #], Append[pos, #]] &, cpos];
    out = ReplacePart[expr, MapThread[(#1 -> #2) &, {cpos, children}]],
    out = expr
  ];
  If[("NodeID" /. boxinfo) === True && !MatchQ[expr, Cell[_CellGroupData, ___]],
    out = Append[out, "NodeID" -> ++$NodeIDCounter]];
(** FIXME: let's do this instead
  If[("NodeID" /. boxinfo) === True && !MatchQ[expr, Cell[_CellGroupData, ___]],
    If[MatchQ[out, _[___, _[TaggingRules, _], ___]],
      out = Replace[out,
        rh_[TaggingRules, tr_] :> rh[TaggingRules, Append[tr, "NodeID" -> ++$NodeIDCounter]] ],
      out = Append[out, TaggingRules -> {"NodeID" -> ++$NodeIDCounter}]
  ]];
*)
  nid = GetNodeID[out];
  Switch[$NodePosition[nid],
    None,
      $NodePosition[nid] = pos,
    "Ambiguous",
      None,
    _,
      Message[Transmogrify::ambiguous, out]; $NodePosition[nid] = "Ambiguous"
  ];
    
  out
];



(* -----------------------------------------------------
        1-2 Argument definitions return SymbolicXML        
   ----------------------------------------------------- *)

(*  The following definition is a special case of a userland 
    XMLTransform function, the purpose of which is to continue
    recursing if we're called during a rule replacement. It is
    defined first for recursion optimization. *)

Transmogrify[SelectChildren[args___], opts___?OptionQ] :=
Module[{OldPosition = $Position, OldParents, NewParents, NewParentsSet = False},
  OldPosition = $Position;
  OldParents = $Parents;
  Sequence @@ Map[
    Function[{pos}, Module[{expr = ExtractOne[$Self, pos]},
      If[MatchQ[expr, _[___, "NodeID" -> _, ___]],
        Block[{$Parents = {}, $Position = {}},
          MakeItSo[expr, opts]
        ],
        Block[{$Parents, $Position},
          If[!NewParentsSet, NewParents = Prepend[OldParents, $Self]; NewParentsSet = True];
          $Parents = NewParents;
          $Position = Prepend[OldPosition, pos];
          MakeItSo[expr, opts]
        ]
    ]]],
    SelectPositions[SelectChildren, args]
  ]
] /; TrueQ[$Walking];

(* FIXME: we can probably figure out $Parents for other selectors *)
Transmogrify[expr_, opts___?OptionQ] :=
Block[{$Parents = {}, $Position = {}},
  MakeItSo[Evaluate[expr], opts]
] /; TrueQ[$Walking];



(**
  Transmogrify[expr, "transform"|XMLTransform[{...}]]
**)
Transmogrify[expr_ /; Head[expr]=!=String, transform_?TransformQ, opts___?OptionQ] :=
	Transmogrify[None, expr, XMLTransformInit[transform,opts], opts]

(** 
  Transmogrify["infile","transform"|XMLTransform[{...}]] - import expr from file
**)
Transmogrify[exprfile_String, transform_?TransformQ, opts___?OptionQ] := 
	Transmogrify[None, Import[exprfile, "Notebook"], XMLTransformInit[transform,opts], opts]

(* -----------------------------------------------------
        3 Argument definitions export XML to file       
   -----------------------------------------------------*)

(** 
  Transmogrify["output-file", expr, "transform"]
  don't try to overload the transform to be "transform"|XMLTransform[] 
  as that's taken care of in the actual implementation below!
**)
Transmogrify[output_String, expr_/;!StringQ[expr], transformFile_?XMLTransformQ, opts___?OptionQ]:=
    Transmogrify[output, expr, XMLTransformInit[transformFile, opts], opts]

(** 
  Transmogrify["output-file", "input-file", "transform"|XMLTransform[{...}]]
  import expr from file
**)
Transmogrify[filename_String, exprfile_String, transform_?TransformQ, opts___?OptionQ]:= 
    Transmogrify[filename, Import[exprfile, "Notebook"],
      XMLTransformInit[transform,opts],opts
    ]/;FileType[exprfile]===File

(* -----------------------------------------------------
        3 Argument definitions for Batch mode      
   -----------------------------------------------------*)

(** 
  Transmogrify["output-directory", "input-directory", "transform"|XMLTransform[]] - batch mode
**)
Transmogrify[outputdir_String, inputdir_String, transform_?TransformQ, opts___?OptionQ]:= 
Module[
  {nbs, lev},
  lev = InputDirDepth /. {opts} /. Options[Transmogrify];

  If[!ListQ[
    nbs=FileNames["*.nb",inputdir, lev]],Return[$Failed]];
  
  If[nbs === {},
    Message[Transmogrify::badinput,inputdir];
    Return[$Failed]
    ,
    Transmogrify[outputdir, nbs, XMLTransformInit[transform, opts], opts ]
  ]
]/;FileType[inputdir]===Directory

Transmogrify::badinput = "`1` is not a valid input filename or directory."

(**
  Transmogrify["outputdir", {"file1","file2",...}, "transform"]
  note- the actual implementation only takes place when an XMLTransform is given as 
  the third argument.  See below.
**)
Transmogrify[outputdir_String, infiles:{__String}, transform_?XMLTransformQ, opts___?OptionQ]:= 
    Transmogrify[outputdir, infiles, XMLTransformInit[transformFile,opts],opts]

(** 
  Quick Error Cases
  the first one users specify files/directories that don't exist
  the second catches when Import["input-file"] returns $Failed for some reason
**)
Transmogrify[_String,in_String,___]:=(Message[Transmogrify::badinput,in];$Failed)
Transmogrify[_,$Failed,__]=$Failed

(* ----------------------------------------------------
        Actual implementations 
   ---------------------------------------------------- *)
(** 
  Transmogrify["output-directory",{"file1","file2",...},XMLTransform[{...}]]

  This is the actual implementation of Batch mode Transmogrify[]

  All other Batch mode definitions should do whatever they need to do
  to call this definition.
**)

Transmogrify[outdir_String, infiles:{__String}, trans:(_XMLTransform|$CachedTransform), opts___?OptionQ]:=

Module[
  {transform,ext},
  ext = OutputExtension /. {opts} /. Options[Transmogrify];
  If[!StringQ[ext],
    Message[Transmogrify::outext,ext];
    ext = ""
  ];
  (* parse transform once *)
  TransmogrifyInit[trans, opts];
  (* then map Transmogrify over everything *)
  Transmogrify[ToFileName[{outdir}, StringReplace[#,
    {
      StartOfString~~WordCharacter__~~$PathnameSeparator->"",
      "."~~WordCharacter__~~EndOfString->"."
    }]<>ext], Get[#],
    $CachedTransform, opts]&/@infiles
]
 
(** 
  Transmogrify["output-file", expr, XMLTransform[{...}]]
  
  This is the actual implementation for Single mode Transmogrify[]
  It does the following things:

    1: parse the XMLTransform (TransmogrifyInit[])
    2: initialize the pseudo globals : $Ancestors, $Walking, etc.
    3: call the recursive rule replacement engine, MakeItSo[]
    4: export the resulting SymbolicXML to outfile
    5: does some reporting if called under debug's NBProfile mode

**)

Transmogrify[outfile:(_String|None),
	expr_ /;!AtomQ[expr], transform:(_XMLTransform|$Failed|$CachedTransform), 
    opts___?OptionQ] :=
Block[
  { 
    output, ret, $Counters,
    $Notebook, $NodePosition, $NotebookContext,
    $Ancestors=#, $Walking=True,
    $OutputFile = outfile,
    format, exportopts, maxerrs, dir, 
    saveProfile, profileFile
  },

  AppendTo[Attributes[Transmogrify], HoldFirst];

  GetNodeFunctionInit;
  Clear[$NodePosition];
  $NodePosition[_] := None;
  $Notebook = ParseNotebook[expr];
  $NotebookContext = ("c" <> ToString[Hash[$Notebook]] <> "`");
  
  If[ StringQ[$OutputFile], $OutputFile = StringReplace[$OutputFile,"~"->$HomeDirectory] ];

  (* get the directory to which we're going to save files *)
  dir = If[outfile === None || StringFreeQ[ outfile, RegularExpression["^(?:/|[A-Z]:\\\\)"]],
    Directory[],
    Transmogrify`Utilities`Private`createDirectory[DirectoryName[outfile]]
  ];

  (* die if we can't write to a new directory *)
  If[dir===$Failed,
    Message[Transmogrify::baddir,dir];
    Return[$Failed],
    (* otherwise change directories *)
    SetDirectory[dir];
  ];

  (* first initialize the Transmogrify environment *)
  TransmogrifyInit[transform,opts];

  (* now set the user's DefaultParameters, which will override
     anything set in the Transform *)
  If[ Length[params=DefaultParameters /. {opts}]>0,
    Scan[ ($Parameters[#[[1]]] = {#[[2]]})&, params ] ];

  (* get some user opts *)
  {Transmogrify`Internal`$AutoRecurse, atomval, format, exportopts,maxerrs} = 
    {AutoRecurse, AtomsReturnSelf, ExportFormat, ExportOptions, MaxExportErrors} /. 
      {opts} /. Options[Transmogrify];

  (* set up return value for atomic recursion.  don't use Sequence[]! *)
  If[TrueQ[atomval],
    Transmogrify`Internal`$AtomReturnValue := $Self,
    Transmogrify`Internal`$AtomReturnValue = Null
  ];

  (* do the job *)
  output = Transmogrify[$Notebook];
  
  (* remove Nulls.  Evaluate[] to remove single Sequence[] *)
  output = Evaluate[Sequence @@ DeleteCases[{output},Null,Infinity]];

  (* remove any previous definitions to $XML(Output|Errors) just to save space *)
  Clear[$XMLOutput,$XMLErrors];

  (* set up return value *)
  ret = Which[
    (* return SymbolicXML *)
    !StringQ[outfile], output,
    (* export Text for stuff that's already Strings *)
    StringQ[output] || format==="Text",
      Export[ outfile, output, "Text", Sequence @@ exportopts ],
    (* save SymbolicXML to file if valid *)
    format === "XML",
      (* BUG #61406 *)
      Check[ $XMLErrors = XML`SymbolicXMLErrors[output], 
        If[$XMLErrors === {}, $XMLErrors = {{0}}] ];

      If[ Length@$XMLErrors > maxerrs,
        Message[Transmogrify::badxml]; $XMLOutput=output; $Failed,
        Export[ outfile, output, "XML", Sequence @@ exportopts ]
      ],
    (* save all other formats *)
    True,
      Export[ outfile, output, format, Sequence @@ exportopts ]
  ];

  (* report profile *)
  If[profileQ,
    (* save each rule hit in $Profile[$OutputFile] *)
    filebase = If[ StringQ[$OutputFile],
      Transmogrify`Utilities`Private`BaseName[$OutputFile],
      $OutputFile ];
    $Profile[filebase]={First[#],Length[#]}&/@Split@Sort[`bagPart[$TransformMatches,All]];
    `trace[NBProfile,
      "Profile for "<>ToString@filebase<>" finished:  Executed "<>
      ToString@`bagLength[$TransformMatches]<>" rules ("<>ToString@Length@$Profile[filebase]<>" distinct)"];
    `trace[{NBProfile,2},"Profile for file: "<>ToString@filebase<>" : ",
      $Profile[filebase]];

    (* quick hack to allow saving the profile expression to file. *)
    {saveProfile, profileFile} = {"SaveProfile","ProfileFile"} /. `traceOpts[NBProfile] /. Options[NBProfile];

    If[ TrueQ[saveProfile], 
      (
        `trace[ NBProfile, "Saving $Profile to " <> # ];
        Export[ #, $Profile[filebase], $PackageFormat ];
      )&[ StringReplace[profileFile,"%f"->ToString@filebase] ];

      (* clean up to save memory for multiple files *)
      Clear[$Profile]
    ]
  ];
  
  (* debug clean up *)
  Transmogrify`Debug`Private`traceHandlerCleanUp[
    Transmogrify`Debug`$TraceHandler];

  ResetDirectory[];
  Attributes[Transmogrify] = DeleteCases[Attributes[Transmogrify], HoldFirst];

  ret
]

(** TransmogrifyString - addition to mirror ExportString **)
TransmogrifyString[c__,opts___?OptionQ]:=
	If[!MatchQ[#,_Transmogrify], 
      ExportString[#, ExportFormat /. {opts} /. Options[Transmogrify],
        Sequence @@ (ExportOptions/.{opts}/.Options[Transmogrify])
      ]
      ,
      #
    ]& [ Transmogrify[c,opts] ]

(** 
    TransmogrifyInit 
    this function takes care of setting up the environment that
    Transmogrify will use to apply its changes.  It will take care of 
    parsing the transform up into rules for use later on, and set up
    other package globals.

        $Rules                list of rules to apply at each node 
        $Parameters[]         DefaultParameters definitions

        $TransformMatches     DEBUG
**)

(* TODO - design with Andy what these should be if anything *)
defaultRules = {};

TransmogrifyInit[trans:(_XMLTransform|$Failed|$CachedTransform), opts___?OptionQ]:=

Module[
  {transform=trans, transdefs},

  `trace[TransformParsing, "Setting up Transmogrify Variables"];

  (* DEBUG - do some profile setup if need be *)
  If[profileQ,
    $TransformMatches = `bag[]
  ];

  (* initialize counters *)
  ResetCounters[];
  $Counters[_] = 0;

  Which[$CachedTransform === trans,
    `trace[TransformParsing, "XMLTransform Cached!  Not reparsing."];
    Return[],
    
    (* complain and die if XMLTransform is bad *)
    Head[transform]=!=XMLTransform,
      Message[Transmogrify::notrans];
      Abort[]
  ];

  (** delete unnecessary rules from transform if necessary **)
  (* TODO - handle ScreenEnv for multiple files in Batch mode *)
  If[$ScreenStyleEnvironment =!= None,
    transform = DeleteCases[transform,
      _[{_, _, env_/;!MatchQ[env,$ScreenStyleEnvironment|All]},_],
	  {2}]
  ];

  (* add defaultRules that will be applied if nothing else matches *)
  (* NOTE - commented out pending design review *)
  (*
  transform = XMLTransform[
    Join[First[transform], defaultRules],
	Sequence@@Rest[transform]];
  *)

  (* turn XMLTransform[{}] into pattern rules and save it as the
     global $Rules.

     THESE RULES ARE NOT SORTED!
     the current design is to make the user list their rules
     in a manner such that most specific come first, with generic
     catch-all's coming last.  If we should do sorting based off 
     of specificity, that code should go here.  :)

     NOTE - Release[] is necessary due to the way the ScreenEnvironment
     patterns are implemented.  It can go away if the ScreenEnvironment
     option goes away (it's only kinda supported right now anyway)
     
     Also, Release[] is faster than ReleaseHold[] for multiple
     iterations.  For 10^5 iterations on my machine:
        Release@mkPattern[{Cell, "blah", "er"}] => 0.014168 Second
        ReleaseHold@         "        "         => 0.492397 Second
     ReleaseHold, however, will remove HoldComplete's while Release
     won't.
  *)
  $Rules = transform[[1]];
  $Rules[[All,1]] = Release[mkPattern/@$Rules[[All,1]]];

  `trace[TransformParsing,"Parsed "<>ToString@Length[$Rules]<>" rules."];
  `trace[{TransformParsing,2},"List of $Rules: ",$Rules];
 
  (* clear any variables lying around to save memory *)
  Clear[$Parameters,$TagsToCells];
  $Parameters[_] = {};

  (** Merging the options from the transform - $MogOpts is a relic of the
      T0 engine.  
      FIX - figure out how options are going to work now that mogrify[]
            doesn't exist!  DefaultParameters is set up below, but 
            CounterReassignments currently are not.  Figure that out.
      Commented out because it currently does nothing.
  **)
  (*
  $MogOpts = Sequence @@ Join[DeleteCases[{opts},_[DefaultParameters,_]],
    If[(IgnoreBoxes /. {opts}) =!= IgnoreBoxes,
      Options[transform, IgnoreBoxes] /. {} -> {IgnoreBoxes->{}},
      {}
    ],
    If[(CounterReassignments /. {opts}) =!= CounterReassignments,
      Options[transform, CounterReassignments] /. {} -> {CounterReassignments->{}},
      {}
    ]
  ];*)

  (* 
      if DefaultParameters are set in the transform, merge them
      all together, weakest first.  in otherwords, transforms included
      with IncludeXMLTransform will not override any DefaultParameters
      defined in the parent transform.

      use Cases instead of Replace to get _all_ default params.
  *)
  If[Length[
    transdefs = Cases[Options[transform], DefaultParameters ~(Rule|RuleDelayed)~
      _]]>0,
    transdefs = Flatten@Reverse@transdefs[[All,2]];
    Scan[ ($Parameters[#[[1]]] = {#[[2]]})&, transdefs ] 
  ];
](* end TransmogrifyInit[] *)

(** parse
    this perhaps vaguely named function takes care of setting 
    up the variables that will be used by the expression-specific
    functions (Select*, GetStyle, etc.) for the currently scoped 
    expression.  Specifically, they create the following
    pseudo-globals:

      $Children      $Styles    $Options     $GroupOpen
    
    Note that the head ExprSequence[] is wrapped around all sibling
    expressions to make it easy for MakeItSo[] to know to map.

    Add new patterns here to handle arbitrary styles
    and structures.
**)

(* parse is HoldAllComplete to match on HoldExpressions without 
   evaluating things that shouldn't be evaluated.  Actually dealing
   with things that should be held should be left up to the user
   in the Transform? *)
Attributes[parse] = {HoldAllComplete}

(* Held* functions are helper functions from robby to help remove 
   evaluation leaks.  these should be deprecated in favor of 
   OptionPattern[] when oyvind finishes it in 6 *)
SetAttributes[{HeldOptionQ, HeldNonOptionQ}, HoldAllComplete]

HeldOptionQ[expr_] := OptionQ @ Unevaluated[expr]
HeldOptionQ[exprs__] := Thread[Unevaluated @ HeldOptionQ[And[exprs]], And]

HeldNonOptionQ[expr_] := Not @ HeldOptionQ[expr]
HeldNonOptionQ[exprs__] := Thread[Unevaluated @ HeldNonOptionQ[And[exprs]], And]

(* exprs with the standard idea of Cell-type "Styles" *)
parse[ h:(Cell|StyleBox)[stuff_, styles__String, opts___?OptionQ] ] := 
  ($Children = {stuff}; $Styles = {styles}; $Options = {opts};)

(* pseudo-style for CellGroupData[] ... *)
parse[ CellGroupData[stuff:{Cell[_,styles__String,___],___},open___] ]:=
  ($Children = stuff; $Styles = {styles}; $GroupOpen = open)

(* handle CellGroupData[nonstyled-cells].  otherwise Open|Closed might show
   up in $Children *)
parse[ CellGroupData[l_, open___] ] :=
  ($Children = l; $Styles={}; $GroupOpen = open;)

parse[ TagBox[stuff_, tag_, opts___?OptionQ] ] :=
  ($Children = {stuff}; $Styles = {ToString[tag]}; $Options = {opts};)

(* ... and FormBox[] *)
parse[ FormBox[stuff_, style_, opts___?OptionQ] ] :=
  ($Children = {stuff}; $Styles = {style}; $Options = {opts};)

(* all other normal expressions, including things that _should_ have styles
   but don't *)
parse[ h_[stuff_List, opts___?HeldOptionQ] ] := 
  ($Children = stuff; $Styles = {}; $Options = {opts};)

parse[ h_[stuff_, opts___?HeldOptionQ] ] := 
  ($Children = {stuff}; $Styles = {}; $Options = {opts};)

parse[ h_[stuff__, opts___?HeldOptionQ] ] := 
  ($Children = {stuff}; $Styles = {}; $Options = {opts};)

(* TODO - figure out what to do with Hold expressions *)
(* parse[ h_?holdQ[stuff__, opts___?OptionQ] ] := 
    ($Children = {HoldPattern@stuff}; $Styles = {}; $Options= {opts};)
*)

(* anything else that matches should be an Atom *)
parse[_] := ($Children = $Styles = $Options = {};)

(** holdQ[] 
    simple function to find out if a function head has a Hold*
    attribute
**)
holdQ[head_AtomQ] := MemberQ[Attributes[head], 
  HoldFirst|HoldRest|HoldAll|HoldAllComplete]

(** getKids[]
    this is a helper function to recurse through expressions in 
    certain selector functions.  See SelectNearestDescendants[]
    for example.  TODO - obey evaluation rules for held 
    expressions?
**)
getKids[(Cell | StyleBox | FormBox)[stuff_, ___]] := If[ListQ[stuff], stuff, {stuff}];
getKids[CellGroupData[stuff_List,___]] := stuff;
getKids[h_[stuff__, ___?OptionQ]] := {stuff};
getKids[sibs__] := getKids /@ {sibs};
getKids[_] := {};

(** MakeItSo[]
  
    This is the actual engine of Transmogrify that takes care
    of walking the entire expression.  As it does so, it sets 
    up an environment for the current node which will in effect
    define the XMLTransform functions that the user calls.

    Being recursive, this function is where we need to be the
    most efficient.  Use as little evaluations/tests as possible.
    For example, F/@expr is faster than F[#]&/@expr, remove
    unnecessary /; conditionals, etc.
**)
(*Attributes[MakeItSo]={HoldAll}*)

(* NOTE - don't delete this definition!  It exists to stop recursion.
   Sometimes users don't have enough forsight to not tell Transmogrify
   to recurse for atomic expressions.  For example, String :> Recurse[]
   will cause the rule replacement to call Transmogrify[None].  

   Short circuit here by simply returning Null, which is a reasonable
   result if the user forgot to ask for something or $Self, which would
   be more friendly.  This is decided by the AtomsReturnSelf option of
   Transmogrify, which sets up $AtomReturnValue.

   Note that returning $Self here causes an extra level of recursion and
   it would be better to specify an explicit rule in the XMLTransform.
*)
(* I'm commenting this out, and changing the pattern on the next
   definition to lst:{___}, instead of lst:{__}.  There's a big
   scary warning from layland here, but I ran a battery of tests
   that didn't reveal any recursion problems.  It's possible that
   architectural changes I've made have negated the problem.

   The reason I'm commenting this out is that it causes transforms
   to bomb in the perfectly reasonable case the a selection returns
   an empty list.  Instead of outputting nothing, you get a nasty
   cell expression in your output.  If this turns out to cause any
   real problems, we'll have to revisit.

   //shaunm
*)
(*
MakeItSo[{},opts___?OptionQ] := Transmogrify`Internal`$AtomReturnValue
*)

MakeItSo[lst:{___}, opts___?OptionQ] := Block[{ret, tmp=$Ancestors},
  `trace[Recursion,"mapping over " <> ToString@Length@lst <> " nodes"];
  `trace[{Recursion,2}, "Mapping over ", lst];

  (* Apply Sequence to remove the ExprSequence[] wrapper *)
  ret = Sequence @@ Flatten[Function[e, MakeItSo[e, opts]] /@ lst];
  `trace[{Recursion,2}, "Map returned: ", ret];
  ret
]

(* the variables in this Block are pseudo-global, meaning
   that they are local to the current call of Transmogrify[],
   however, every function that MakeItSo calls, including the
   selector functions, will know about them unless they have
   been locally scoped.  
   
   NOTE:
      * this definition should accept a single normal  
        Mathematica expression (with the ExprSequence head already stripped)
      * This definition does not have an expr_ /; Head[expr]=!=ExprSequence
        for tiny performance gain.
*)

MakeItSo[expr_, opts___?OptionQ]:= Block[
  {
    h=Head@expr, out, a, afunc,
    $Self=expr, $Options, $Parent, $Children, $Styles, $Siblings,
    branchAncestors=$Ancestors, $UserWantsSelf=False,
    oldShowStringCharacters=$ShowStringCharacters
  },

If[MatchQ[expr, Cell[___, "Output", ___]], OUTPUT=True, OUTPUT=False];
  {a,$Siblings} = {Ancestor,Sibs} /. {opts} /. {Ancestor->None, Sibs->None};

  `trace[{Recursion,2}, "processing "<>ToString@expr];

  Which[
    MatchQ[$Self, _[___, _[ShowStringCharacters, _], ___]],
    $ShowStringCharacters = First@Cases[$Self, _[ShowStringCharacters, ssc_] :> ssc],
    MatchQ[$Self, _Notebook],
    $ShowStringCharacters = True
  ];

  (* first set up the easy environment stuff *)
  parse[expr];

  (* If expr ain't an Atom, set up $Ancestors for this iteration.  Note 
     that this is treated as an anonymous function to make life easier.
     # can be replaced by the current value of $Children when and if 
     SelectAncestors is called.  See the Selector functions below for
     more. *)
  If[!AtomQ[expr],
    afunc=Function[f,f&][If[a===None,$Ancestors,a]];

    $Ancestors = afunc[Which[
      h===CellGroupData, h[#,$GroupOpen],
      h===TagBox, h[#, Last[expr]],
      HasStyle[], h[#, Sequence@@$Styles, $Options],
      True, h[#, $Options]
    ]]
  ];

  (* do the actual replacement.  We're using {0} so as to
     not interfere with other replacements as we're walking. *)
  out = Replace[ expr, $Rules, {0} ];

  (* if we didn't match anything, continue on until we do.
     this effectively deletes the unmatched expression _head_.  
     Note that the Internal`$AutoRecurse var could be passed as 
     an option to MakeItSo, but repeated rule application slows
     things down so I opted for an Internal` variable instead.  
     That way, if users _really_ want control over switching 
     auto recursion on and off during recursion, they can set 
     the Internal variable in their transform rules. 

     Lastly, the $UserWantsSelf (when i think about you?... :) )
     variable is to make the AtomsReturnSelf play nicely with the
     AutoRecurse functionality.  This in effect allows the 
     AtomsReturnSelf option to be overridden by an explicit 
     XMLTranform rule.  That way, 

      Transmogrify["string",XMLTransform[{String:>SelectSelf[]}]]

     always returns SelectSelf[] regardless of the value of the
     AtomsReturnSelf option.  Note that $UserWantsSelf will be set
     to True anytime SelectSelf is called, but that it will only have
     effect below if out === expr and all the other jazz is true.
  *)
  If[ Transmogrify`Internal`$AutoRecurse && 
        out === expr && !deletedQ[out] && !$UserWantsSelf,
    `trace[Recursion, "No matches for "<>ToString@Head@expr<>" expr.  Recursing!"];
    out = Transmogrify[SelectChildren[]];
    ,
    (* otherwise we matched something; save if we're profiling *)
    If[profileQ,
      `stuffBag[$TransformMatches, expr /. (#->#&/@$Rules[[All,1]])]
      ,
      `trace[TransformRules, ToString@Short@expr<>" matched rule: " <> ToString[expr /. (#->#&/@$Rules[[All,1]])]]
    ];
  ];
  `trace[{Recursion,2}, "Returning ",out];
  (* reset $Ancestors to last branch point *)
  $Ancestors=branchAncestors;

  $ShowStringCharacters = oldShowStringCharacters;

  out
]

(** deletedQ 
    test function to find out if a user deleted an expression
    and its children via a transform rule return value of nothing.
    Note that a generic AtomQ[] statement is not included.  This 
    means that if a user has AutoRecurse->True, and transforms
    an expression to a (non-empty=""|Null) atom, that atom will 
    still be transformable via the autorecursion.  This might not
    be what the user wants, however.  

    TODO - design review
**)
deletedQ = Function[x, MatchQ[Unevaluated@x, HoldPattern[Sequence][]|{}|Null|""], SequenceHold]

(** mkPattern 
    Simple function to turn XMLTransform selectors:

      Head|{Head}|{Head,Style}|{Head,Style,Screen}

    into appropriate pattern rules like:

      _Head, Head[__,style,___?OptionQ]

    TODO - handle options somehow.  deprecate or support
    screen environment syntax : it's never used anyway
    This should also be updated to allow for multiple
    v6 style sequences. Perhaps make definitions for style_List
    and return Sequence@@style in the pattern.
**)

mkPattern[String] = _String;
mkPattern[{head_?AtomQ}|head_?AtomQ] := _head

mkPattern[{Cell, CellGroupData}] := Cell[_CellGroupData, ___?OptionQ];
mkPattern[{Cell, None}] := Cell[c_/;(Head[c]=!=CellGroupData), ___?OptionQ]
mkPattern[{head:(StyleBox|FormBox), None} ] := head[_, ___?OptionQ]
mkPattern[{head:(Cell|StyleBox|FormBox), style_}] := (
  head[__, style, ___?StringQ, ___?OptionQ]/.{All->_(*,None->Sequence[]*)})

(* this three arg form will interfere with the Screenenvironment three arg
   form below, but just added it to see if it would be used. *)
mkPattern[{CellGroupData, None, open_:None}] := CellGroupData[{Cell[_,___?OptionQ],___},open/.None->___]
mkPattern[{CellGroupData, open:(Open|Closed)}] := mkPattern[{CellGroupData, All, open}]
mkPattern[{CellGroupData, style_, open_:None}] := (
  CellGroupData[{Cell[_, style,  ___?OptionQ],___},open] /. {All->_,None->___})

(* three argument selectors. TODO - change screen to options? *)
mkPattern[{head:(Cell|StyleBox), style_, screen_}] := 
  Hold[head[_, Sequence @@ style, ___?OptionQ] /; $ScreenEnvironment === screen]

(* FIXME *)
mkPattern[{CellGroupData, style_, screen_}] = _

(* thread over Alternatives *)
x:mkPattern[_Alternatives] := Thread[Unevaluated@x,Alternatives]

(* thread over Alternatives for list-based shortcuts *)
mkPattern[{x:HoldPattern[Alternatives][__?AtomQ]}] := mkPattern[x]
mkPattern[{HoldPattern[Alternatives][a__?AtomQ],style_}] := 
  Alternatives@@(mkPattern[{#,style}]&/@{a})

(* assume that anything else is a valid pattern
   given by the user *)
mkPattern[x_]:= x


(***********************************************************
            U S E R L A N D   F U N C T I O N S
***********************************************************)
(* these should be pretty straight forward.  any function
   defined here should _not_ be evaluatable unless we're 
   currently traversing the expression.  this is accomplished
   by setting the Transmogrify`Private`$Walking variable to
   True only in the Transmogrify Block.  my tests show that
   repeated applications of TrueQ[] is about twice as fast
   as creating another named function and doing rule 
   replacements. *)


(* filename helpers *)
GetAbsoluteFileName[] := If[
  (* add current directory if output file is relative *)
  StringFreeQ[$OutputFile,RegularExpression["^(?:/|[A-Z]:\\\\)"]],
  ToFileName[{Directory[]}, $OutputFile],
  $OutputFile 
] /; TrueQ[$Walking] && StringQ[$OutputFile]

Unprotect["GetFileName"];
GetFileName[] := $OutputFile /; 
  TrueQ[$Walking] && StringQ[$OutputFile]

GetFileNameBase[] := Transmogrify`Utilities`Private`BaseName[$OutputFile,
  Transmogrify`Utilities`Private`Extension -> False] /; 
  TrueQ[$Walking] && StringQ[$OutputFile]

GetFileExtension[] := Transmogrify`Utilities`Private`FileExtension[$OuputFile] /; 
  TrueQ[$Walking] && StringQ[$OutputFile]

GetAbsoluteFileName[] := (Message[Transmogrify::nofileop, GetAbsoluteFileName];"")
GetFileName[] := (Message[Transmogrify::nofileop, GetFileName];"")
GetFileNameBase[] := (Message[Transmogrify::nofileop, GetFileNameBase];"")
GetFileExtension[] := (Message[Transmogrify::nofileop, GetFileExtension];"")
GetFileExtension[f_String]:= StringReplace[f, Character__~~"."->""];


(* style helpers *)
GetStyle[] := GetStyle[$Self] /; TrueQ[$Walking];
GetStyle[expr_] := Module[{lst},
  lst = GetStyleList[expr];
  If[!ListQ[lst] || lst === {}, None, First[lst]]
] /; TrueQ[$Walking];

GetStyleList[] := GetStyleList[$Self] /; TrueQ[$Walking];
GetStyleList[expr_] := Module[{boxinfo, func},
  boxinfo = Head[expr] /. $BoxTypes;
  If[boxinfo === Head[expr],
    Message[Transmogrify::noboxinfo, ToString[Head[expr]]];
    Return[{}]];
  func = "Styles" /. boxinfo;
  If[func === "Styles", Return[{}]];
  If[ListQ[#],#,{#}]&[func[expr]]
] /; TrueQ[$Walking];

HasStyle[] := HasStyle[$Self] /; TrueQ[$Walking];
HasStyle[expr_] := Module[{lst},
  lst = GetStyleList[expr];
  If[ListQ[lst] && Length[lst] > 0, True, False]
] /; TrueQ[$Walking]

GroupOpenQ[] := !MatchQ[$GroupOpen,Closed] /; TrueQ[$Walking]

(* fast *)
GetOption[x_] := x /. Options[$Self] /. x->None

(* slow *)
GetOption[s__] := (
  (*Message[Transmogrify::obs,GetOption," See Options[] instead."];*)
  If[MatchQ[$Self,
		_@@Fold[{___, (Rule|RuleDelayed)[#2, #1], ___}&, _, Reverse[Flatten[{s}]]] ],
	Fold[(#2 /. Cases[#1, _~(Rule|RuleDelayed)~_])&, $Self, {s}],
    None]
) /; TrueQ[$Walking]

(* fast *)
HasOption[x_] := MemberQ[$Self, x ~ (Rule|RuleDelayed) ~ _]

(* slow *)
HasOption[s__] := MatchQ[$Self,
  _@@Fold[{___, (Rule|RuleDelayed)[#2, #1], ___}&, _, Reverse[Flatten[{s}]]] 
] /; TrueQ[$Walking]

(** StringSubstitute 
    just like StringReplace, except that the righthand side of the 
    replacement rule can be any mathematica expression.  Therefore, 
    if the current string is:

        "i am some text"

    you can do stuff like:

        StringSubstitute[{"i"->I,"am"->a[m],"some"->S[0]^me,"text"->T E[x^t]}]

    which returns the following list:
  
        {\[ImaginaryI]," ",a[m]," ",S[0]^me," ",T E[x^t]}

**)
(* NOTE - repeated calls to this with a large list of rules is SLOW! 
   this is due to how well StringReplace scales for large lists.
   If you are smart about rule replacements, things can be made quite
   fast without this function.  See DocumentationConversion's 
   ReplaceSpecialCharacters[] for a good example of this.
*)
StringSubstitute[list_List] := 
  If[ !StringQ[$Self],$Self, StringSubstitute[$Self, list] ] /; TrueQ[$Walking]
StringSubstitute[expr_String,list_] := Sequence@@StringReplace[expr,list] /; TrueQ[$Walking]

(* -------------- Selector Helpers -------------- *)

(* often used function to display results of a call to Cases.  I'm not
   for sure I like wrapping multiple expressions in ExprSequence or not. *)
formatRes = Switch[#, None|{}, {}, {__}, #, _, {#}]&

(** 
   String Kludge!  (removeStylesAndOptions[])
   Due to the current architecture, calling Select*[_String] will
   return _all_ strings in the expression, including those in "Styles" and 
   options (like CellTags->{"String"}).

   I figure there's no reason why it shouldn't return all strings, because 
   they're still part of the expression.  Since v.1 didn't do this, however,
   the following kludgey definition exists to remove options and string styles.
   This allows us to match as v.1 did.
   
   Note that each selector that uses this definition is currently defined in such
   a way that it allows for both functionalities.  
   
      Select*[_String] gives all strings, regardless if they're styles or not, and

      Select*[String] gives only strings within the "content" section of the 
      expression.

   Feel free to change/delete these definitions if you choose one functionality
   over the other.
*)
removeStylesAndOptions[expr_] := expr //. { 
  (Cell|StyleBox)[stuff_,__String, o___?OptionQ] :> {stuff},
  FormBox[s_,___]:>{s},
  h_[c__,o__?OptionQ] :> {c}
}


(* -------------- Selector Functions -------------- *)


(** Recurse **)
Recurse[] := Transmogrify[SelectChildren[]] /; TrueQ[$Walking]
Recurse[expr_,opts___?OptionQ] := Transmogrify[expr,opts] /; TrueQ[$Walking]


(** SelectorQ **)
Attributes[SelectorQ] = {HoldFirst};
SelectorQ[sel_] := MemberQ[{SelectChildren}, Head[sel]];


(** SelectSelf **)
(* FIXME: what's $UserWantsSel? *)
SelectSelf[] := ($UserWantsSelf=True;$Self) /; TrueQ[$Walking]


(** SelectLiteral **)
SelectLiteral[x___] := x;


(** SelectChildren **)
SelectChildren[args___] := Extract[$Self, SelectPositions[SelectChildren, args]] /; TrueQ[$Walking];

SelectPositions[SelectChildren] := GetChildPositions[$Self];
SelectPositions[SelectChildren, i_Integer] :=
  Module[{ch},
    ch = GetChildPositions[$Self];
    If[0 < Abs[i] <= Length[ch],
      {ch[[i]]},
      Message[Transmogrify::childindex,i,k]; {}
  ]];
SelectPositions[SelectChildren, head_, style_] :=
  SelectPositions[SelectChildren, {head, style}];
SelectPositions[SelectChildren, selector_] :=
  Module[{pos},
    pos = GetChildPositions[$Self];
    Cases[Map[{#, ExtractOne[$Self, #]}&, pos], {i_, mkPattern[selector]} :> i]
  ];

(* FIXME: ExprSequence *)
(*
SelectChildren[ExprSequence[expr_]] := Block[
  {$Children, $Styles, $Options, $GroupOpen},
  parse[expr];
  SelectChildren[]
] /; TrueQ[$Walking]
*)


SelectSiblings[] := Module[{selfpos, parent, sibpos},
  selfpos = $NodePosition[GetNodeID[$Self]];
  If[MemberQ[{"Ambiguous", None}, selfpos],
    Message[Transmogrify::cantselect, "siblings", $Self]; Return[{}]];
  If[selfpos === {{}}, Return[{}]];
  parent = ExtractOne[$Notebook, Flatten[Most[selfpos]]];
  sibpos = GetChildPositions[parent];
  sibpos = DeleteCases[sibpos, Last[selfpos]];
  Extract[parent, sibpos]
] /; TrueQ[$Walking];

SelectSiblings[selector_] :=
  Cases[SelectSiblings[], mkPattern[selector]] /; TrueQ[$Walking];

(** SelectAncestors[]
    this will return only the node heads/styles of the branch of the tree
    that the current node is on.  In other words, SelectAncestors acting at
    node f in the tree below

         a
        /|\
       b c d
      /\    \
     e (f)   g

    will return a,b,# with associated styles/options at each level.  All branch
    content will be ignored, and f will be replaced with Slot[] (#).  See
    SelectAncestorsAndSelf[] for entire branch a,b,f and f's children.
    Note that Function is not applied so people can still pattern match
    on the results (Head[SelectParent[]], etc.)

    The _[#,___]-># is to remove the $Self layer.
**)
SelectAncestors[] := Module[{selfpos},
  selfpos = GetNodePosition[$Self];
  If[MemberQ[{"Ambiguous", None}, selfpos],
    Message[Transmogrify::cantselect, "ancestors", $Self]; Return[{}]];
  If[selfpos === {{}}, Return[{}]];
  Extract[$Notebook,
    Map[
      Flatten[Drop[selfpos, -#]]&,
      Range[Length[selfpos] - 1]] ]
] /; TrueQ[$Walking];

SelectAncestorsAndSelf[] := Module[{selfpos},
  selfpos = GetNodePosition[$Self];
  If[MemberQ[{"Ambiguous", None}, selfpos],
    Message[Transmogrify::cantselect, "ancestors", $Self]; Return[{}]];
  If[selfpos === {{}}, Return[{}]];
  Extract[$Notebook,
    Map[
      Flatten[Drop[selfpos, -#]]&,
      Range[0, Length[selfpos] - 1]] ]
] /; TrueQ[$Walking];

SelectAncestors[selector_] :=
  Cases[SelectAncestors[], mkPattern[selector]] /; TrueQ[$Walking];

SelectAncestorsAndSelf[selector_] :=
  Cases[SelectAncestorsAndSelf[], mkPattern[selector]] /; TrueQ[$Walking];

SelectNearestAncestor[selector_] :=
  If[Length[#] > 0, First[#], None]&[
    Cases[SelectAncestors[], mkPattern[selector], 1, 1]
  ] /; TrueQ[$Walking];

SelectNearestAncestorAndSelf[selector_] :=
  If[Length[#] > 0, First[#], None]&[
    Cases[SelectAncestorsAndSelf[], mkPattern[selector], 1, 1]
  ] /; TrueQ[$Walking];

(** SelectParent[] **)
SelectParent[i_Integer:0] := Module[{selfpos},
  selfpos = GetNodePosition[$Self];
  If[selfpos === {{}}, Return[None]];
  If[MemberQ[{"Ambiguous", None}, selfpos],
    (* Try to use the $Parents Block-scoped variable instead *)
    If[Length[$Parents] > 0,
      If[i >= Length[$Parents],
        Message[SelectParent::badindex,i]; Return[None]];
      Return[Part[$Parents, i]]];
    (* Otherwise there's nothing we can do *)
    Message[Transmogrify::cantselect, "parent", $Self]; Return[None]];
  If[i > Length[selfpos],
    Message[SelectParent::badindex,i]; Return[None]];

  ExtractOne[$Notebook, Flatten[Drop[selfpos, -i]]]
] /; TrueQ[$Walking];

(* FIXME: SelectParent with a selector *)

SelectParent::badindex = "Parent index `1` must be a positive integer."

getDescendants[_String, ___] := {};
getDescendants[expr_, pattern_, nearest_] := Module[{cpos},
  cpos = GetChildPositions[expr];
  Map[
    Function[{pos}, Module[{child, cmatch},
      child = ExtractOne[expr, Flatten[pos]];
      cmatch = MatchQ[child, pattern];
      (* FIXME: put in some recursion optimization on known box types *)
      Apply[Sequence, Join[
        If[cmatch, {child}, {}],
        If[!(cmatch && TrueQ[nearest]),
          getDescendants[child, pattern, nearest],
          {}]
      ]]
    ]],
    cpos]
];

(** SelectDescendants[] **)
SelectDescendants[] :=
  getDescendants[$Self, _, False] /; TrueQ[$Walking];
SelectDescendants[selector_] :=
  getDescendants[$Self, mkPattern[selector], False] /; TrueQ[$Walking];

(** SelectDescendantsAndSelf[] **)
SelectDescendantsAndSelf[] :=
  Join[{$Self}, getDescendants[$Self, _, False]] /; TrueQ[$Walking];
SelectDescendantsAndSelf[selector_] := 
  Join[
    If[MatchQ[$Self, mkPattern[selector]], {$Self}, {}],
    getDescendants[$Self, _, False]
] /; TrueQ[$Walking];
  
SelectNearestDescendants[] :=
  getDescendants[$Self, _, True] /; TrueQ[$Walking];
SelectNearestDescendants[selector_] :=
  getDescendants[$Self, mkPattern[selector], True] /; TrueQ[$Walking];

(** SelectGrid*[] **)

SelectGridRows[] := Module[
  { h=Head@$Self, l },
  If[!MatchQ[h, Grid|GridBox],
    Message[SelectGridRows::notgrid, h];
    Return[None]
  ];

  GridRow@@@$Self[[1]]
] /; TrueQ[$Walking]

SelectGridColumns[] := Module[
  { h=Head@$Self, l },
  If[!MatchQ[h, Grid|GridBox],
    Message[SelectGridRows::notgrid, h];
    Return[None]
  ];

  GridColumn@@@Transpose@$Self[[1]]
] /; TrueQ[$Walking]


(** StringValue[]
   This should return a string of all the _string content_ of an expression.  
   Users should use ToString if they desire strings of non-string expressions. *)
StringValue[] := StringValue[$Self] /; TrueQ[$Walking]
StringValue[ExprSequence[e_]] := StringValue[e]
StringValue[s_String] := s
StringValue[expr_] := Block[{$Children, $Options, $Styles},
  parse[expr];
  StringJoin @@ (SelectDescendantsAndSelf[String] /. None->"")
] /; TrueQ[$Walking]

(** BoxToImage[] - This function is responsible for creating images of cell
    structures that are not easily emulated in symbolic XML.  It uses two
    methods to create the images, both of which involve the frontend.
    
    The default image format is given by the global variable
    $ExportImageFormat.
    
    The flow goes like this:
    
      1. If $ExportImageFormat is either GIF or PNM, go ahead and
      use frontend's ExportPacket (which can handle GIF or PNM only)
      to quickly create an image and get its dimensions.  For this to
      work, parentify[$Self] needs to return a valid FE box BoxData,
      Cell, or Notebook expression.
      
      2. If the image dimensions are larger than the specified
      limits in MaxImageSize, _or_ if we are given an image format
      other than GIF or PNM _or_ if we are given ConversionOptions
      for Export, do the following:
      
        a) use the kernel's ToRasterDataPacket[] to create a
           raster of the expression
        b) perform crop/resize routines on raster (controlled by
           the CropImage option)
        c) use the kernels' FromRasterDataPacket[] to create a Graphics
           object which we then Export[]
        
    The kernel export is usually slower due to overhead of To/FromRDP
    and the fact that the kernel export needs the FE to do dithering
    (which is stupid, but that's the way it is right now)
**)

(* TODO - make generic ExportOptions instead of all these? *)
Options[BoxToImage] = {
    Inline->False,
    ImageFormat->Automatic,
    ImageSize->Automatic,
    Magnification->Automatic,
    MaxImageSize->635,
    CropImage->False,
    StringContent->None,
    ConversionOptions->{}, 
    TransparentBackground->False
}

BoxToImage[opts___?OptionQ] := Module[ { format },
  format = Format /. {opts} /. Options[Transmogrify];
  BoxToImage[StringJoin[
      GetFileBaseName[],"_img_",ToString[IncrementCounter["image"]],".",format
    ], opts ]
] /; TrueQ[$Walking]

BoxToImage[filename_String, opts___?OptionQ]:= BoxToImage[filename,$Self,opts] /; TrueQ[$Walking]

BoxToImage[filename_String, self_, opts___?OptionQ] :=
Module[
  { expr, format, failexpr, fullname = ToFileName[{Directory[]},filename],
    width, height, newwidth, newheight, size, maxsize, crop, convopts, mag,
    takex, takey, strcontent, inline, transparentbackground, isManipulate, baseline = Null, needexport},

  (* don't create images if user requested not to *)
  If[ !TrueQ[ CreateImages /. {opts} /. Options[Transmogrify] ] (* &&
      FileType[fullname] === File*),

    (* some routine to get image dimensions from file?  I just did some tests
       with ImageMagick's MagickWand API, but the overhead of MathLink is probably
       going to be slower than simply reexporting.  Perhaps using the MagickCore
       library would prove faster? *)
    Return[]
  ];
  
  failexpr = {"Filename" -> filename, "URL" -> StringReplace[filename,"\\"->"/"],
	    "Width" -> "0", "Height" -> "0", "Directory"->Directory[]};

  Catch[ (* encloses the rest of the function *)
    `trace[ImageCreation, "Creating image for a "<>ToString[Head@expr]<>" Expression"]; 
    `trace[{ImageCreation,2}, "Expr = "<>ToString@expr];
    `trace[ImageCreation,"FileName: "<>filename];
  
    (* create directory relative to current dir or complain *)
    If[ Transmogrify`Utilities`Private`createDirectory[DirectoryName[fullname]]===$Failed,
      `trace[ImageCreation, "Couldn't create directory "<>fullname];
      Throw[failexpr]
    ];
  
    (** process user options **)
    {format, size, mag, maxsize, crop, convopts, strcontent, inline, transparentbackground} = 
      {
        ImageFormat, ImageSize, Magnification, MaxImageSize, CropImage,
        ConversionOptions, StringContent, Inline, TransparentBackground
      } /. {opts} /. Options[BoxToImage];

	If[StringQ[strcontent], self = strcontent];
  
    Which[ 
      MatchQ[maxsize,None|Infinity], maxsize={Infinity,Infinity},
      MatchQ[maxsize,x_?Positive], maxsize={maxsize,Infinity},
      !MatchQ[maxsize,{x_?Positive,y_?Positive}], maxsize={Infinity,Infinity};
        Message[BoxToImage::badsz,maxsize]
    ];
      
    If[ format === Automatic,
      format=$ExportImageFormat];
      
    (*
       create expression with inherited options.  this depends on the magnification
       option passed in.  Apply magnification at the expression level iff 
       Head of the expression is Cell or StyleBox.  Otherwise apply at 
       the Parent level. TODO - make this less if/then'y.  more general and less
       Magnification specific.
    *)
    isManipulate = (Count[self, Manipulate`InterpretManipulate[_], Infinity] > 0);
    Which[
      (* FIXME: Hack to get GIF exports sizes the same as SWF exports.  Since
         the SWF export sends only the cell, and not an entire Notebook, we
         have to do the same thing, or else stuff might be sized differently.
         There is probably a smarter way of going about this, but I need this
         to work right now for the tech conference. *)
      (Head[self] === Cell) && isManipulate,
        expr = mangleCell[self, inline, transparentbackground];
      ,
      mag===Automatic,
        `trace[ImageCreation,"leaving magnification alone."];
        expr = parentify[self, opts];
      ,
      MatchQ[mag,x_?Positive],
        `trace[ImageCreation,"Setting magnification of expr to ",mag];
        
        If[ MatchQ[expr,_Cell|_StyleBox],
          `trace[ImageCreation, "Applying Magnification inline."];
          expr = parentify[Transmogrify`Utilities`Private`PrependOptions[$Self, Magnification->N@mag]]
          ,
          `trace[ImageCreation, "Applying magnification at parent level"];
          expr = Transmogrify`Utilities`Private`PrependOptions[parentify[$Self, opts], Magnification->N@mag]
        ]
      ,
      True,
        Message[BoxToImage::badmag,mag];
        expr = parentify[self, opts];
    ];
    If[Head[expr] =!= Notebook && !isManipulate,
      If[Head[expr] =!= Cell,
      	If[Head[expr] =!= List, expr = Cell[expr]] ];
      expr = Notebook[Flatten@{expr}, Sequence@@Rest[$Notebook]];
    ];
    
    expr = DeleteCases[expr, Rule[DockedCells, _]];

    (* this is related to the 2x image stuff in Image *)
    If[Positive[mag], expr = expr /. Rule[WindowSize, x_] :> Rule[WindowSize, 2 * x]];

	(* expr = Append[expr, DockedCells->{}]; *)
    (* 
       go ahead and export with ExportPacket if it's a format that ExportPacket
       can handle at Andy's request.
       
       TODO - figure out the most optimal way of exporting images.  FE's 
       ExportPacket is fast, but limited.  Kernel's Export adds more formats, 
       but needs to currently call the frontend in order to dither things.  
       Kernel export should be faster in 6.0, but until then, we have a crazy
       mixture of FE and kernel exports, depending on what the user needs.  
       This should probably cleaned up/optimized, but it works for now...
    *)

    If[ MatchQ[format,"GIF"|"PPM"] && convopts==={},
      `trace[ImageCreation,"Exporting with ExportPacket"];

      (*AppendTo[Global`imgs, expr];*)
      (* handle MathLink changes in v6 *)
        ret = MathLink`CallFrontEnd[ ExportPacket[expr, format, fullname, opts]];
      
      (* give up if ExportPacket did *)
      (* BUG - MathLink`CallFrontEnd[ ExportPacket[] ] returns "$Failed" instead
         of the symbol $Failed. *)
      If[MatchQ[ret, $Failed|"$Failed"],
        `trace[ImageCreation,"Call to ExportPacket Failed!"];
        Message[BoxToImage::failed, Short[self]];
        Throw[failexpr]
      ];
      
      (* get baseline *)
      baseline = Last@ret;
      (* get width/height info from ExportPacket return value *)
      {width, height} = #[[2]]-#[[1]]& /@ Transpose@Round[ret[[2]]]
    ,
      (* otherwise get width/height from ToRDP *)
      `trace[ImageCreation,"Creating raster for format: "<>format];
      rdp = System`ConvertersDump`ToRasterDataPacket[Evaluate@expr,format];
      {height,width} = Switch[Length[dims=Dimensions[rdp[[3]]]],
        2, dims,
        3, Most[dims]
      ];
      needexport=True
    ];
  
    (* if dimensions are larger than maxsize do crop/resize routines *)
    Which[ 
      (* Which branch 1 *)
      Inner[Greater, {width,height}, maxsize, Or],
        `trace[ImageCreation,"Image larger than MaxImageSize"];
  
        (* make rdp if we used ExportPacket earlier *)
        If[Length[rdp]==0,
          rdp = System`ConvertersDump`ToRasterDataPacket[Evaluate@expr,format]];
  
        {newwidth,newheight} = Min@@@Transpose[{{width,height},maxsize}];    
        If[ crop===True,
          `trace[ImageCreation,"Cropping image to "<>ToString@{newwidth,newheight}];
  
          takex = If[width < maxsize[[1]], All, maxsize[[1]]];
          takey = If[height < maxsize[[2]], All, -maxsize[[2]]];
  
          ret = Export[filename,
            System`ConvertersDump`FromRasterDataPacket[
              ReplacePart[rdp,Take[rdp[[3]], takey, takex],3]]
            , format, Sequence@@convopts];
          needexport=False
          ,
          (* otherwise shrink with HTML width height attributes *)
          `trace[ImageCreation,"shrinking with HTML"]
        ]
      ,
      (* Which branch 2 *)
      size=!=Automatic,
        (*
           Use Export[] if ImageSize is different than the original.
           still need RDP because ImageSize options dont work for non-image
           expressions in Export[] 
        *)
        `trace[ImageCreation,"Using FromRDP to Export image with ImageSize->", size];
        ret = Export[
          filename, 
          System`ConvertersDump`FromRasterDataPacket[
            System`ConvertersDump`ToRasterDataPacket[Evaluate@expr,format]],
          format, ImageSize->size, Sequence@@convopts];
        needexport=False;
        If[MatchQ[size, _?Positive],
          {newwidth, newheight} = {size, ""}
          ,
          {newwidth, newheight} = size
        ]
      ,
      True,
        `trace[ImageCreation,"leaving image size alone."];
        {newwidth, newheight} = {width,height}
    ]; (* Which *)

    (* export image if we didn't already *)
    If[ needexport, 
      ret = Export[filename, Evaluate@expr, format, Sequence@@convopts];
    ];
      
    If[ret === $Failed,
      (* true clause *)
      `trace[ImageCreation,"Call to ExportPacket Failed!"];
  	  Throw[failexpr]
    ,
      (* all 2x images must have an even-numbered width and height; if either is
      odd, add a row/column as needed to the right and/or bottom of the image*)
      Block[{i, d},
        If[mag == 2,
          i = Import[filename];
          d = ImageDimensions[i]; (* returns {width, height} *)
          i = Switch[d,
            (* ImagePad[image, {{left, right}, {bottom, top}}, ...] *)
            {_?OddQ,  _?EvenQ}, ImagePad[i, {{0, 1}, {0, 0}}, "Fixed"],
            {_?EvenQ, _?OddQ},  ImagePad[i, {{0, 0}, {1, 0}}, "Fixed"],
            {_?OddQ,  _?OddQ},  ImagePad[i, {{0, 1}, {1, 0}}, "Fixed"],
            {_?EvenQ, _?EvenQ}, i, 
            (* if we somehow end up here, just return the image unchanged: *)
            _, i
          ];
          Export[filename, i]
        ]
      ]
    ];
  
    (* baseline defined? *)
    If[baseline === Null, 0, baseline];

    (* return a list of rules *)
    `trace[ImageCreation, "Leaving BoxToImage"];

    { "Filename" -> filename, "Width" -> ToString@newwidth, "Height" -> ToString@newheight, 
      "Directory"->Directory[], "URL" -> StringReplace[filename,"\\"->"/"], 
      "Baseline"-> ToString@baseline }
  ]
] /; TrueQ[$Walking]

BoxToImage::badsz = "ImageSize specifications should be either a positive real number or a list of two positive real numbers.  `1` is neither"
BoxToImage::badmag = "Magnification->`1` should be a positive real number."
BoxToImage::failed = "BoxToImage cannot create an image of `1`"

(** parentify[] 
    this routine is responsible for preparing certain box structures for 
    image creation.  It does this by simply grabbing the branch to a imageifyable
    parent (BoxData, Cell, Notebook) on the ancestor branch and tweaking some 
    options.  If no imageifyable node is found, return $Failed or try wrapping
    it in BoxData[]... ? FIX
**)

GetNodeFunctionInit := (
ClearAll[GetNodeFunction];
GetNodeFunction[pos_] := GetNodeFunction[pos] = Module[{ex, ch},
  ex = ExtractOne[$Notebook, pos];
  ch = GetChildPositions[ex];
  ex = Delete[ReplacePart[ex, First[ch] -> #], Rest[ch]];
  ex
];
GetNodeFunction[src_, pos_] := GetNodeFunction[src, pos] = Module[{ex, ch},
  ex = ExtractOne[src, pos];
  ch = GetChildPositions[ex];
  ex = Delete[ReplacePart[ex, First[ch] -> #], Rest[ch]];
  ex
];
);

parentify[expr_, opts___?OptionQ] := Module[
  {
    inline = TrueQ[Inline /. {opts} /. Options[BoxToImage]],
    transparentbackground = TrueQ[TransparentBackground /. {opts} /. Options[BoxToImage]],
    new, src, pos, isn, funs
  },

  (* new = If[Head[expr] === Cell, expr, Cell[expr]]; Notebook[{new}] *)

  {src, pos} = GetNodeSourcePosition[$Self];

  If[Head[expr] === Cell && Head[First[expr]] =!= CellGroupData,
    new = mangleCell[expr, inline, transparentbackground],
    new = expr];
  isn = (src === $Notebook);
  funcs = Map[
    Function[{p}, Module[{po, ex, ch},
      po = Flatten[p];
      ex = If[isn, GetNodeFunction[po], GetNodeFunction[src, po]];
      ex = ex /.{SuperscriptBox[a_]:>a};
      If[ExtractOne[src, Append[po, 0]] === Cell &&
         ExtractOne[src, Join[po, {1, 0}]] =!= CellGroupData,
        ex = mangleCell[ex, inline, transparentbackground] ];
      Function[Evaluate@ex]
    ]],
    Rest@Most@FoldList[Append[#1, #2]&, {}, pos]
  ];
  new = Apply[Composition, funcs][new];
  new /. Notebook[c_Cell, o___] :> Notebook[{c}, o]
];

mangleCell[expr_, inline_, transparentbackground_] := Module[{new, optColumnWidths = Automatic},

  If[
    Or[
      Positive@Length@Cases[expr, Rule[CellLabel, "*TableForm*"], Infinity],
      (* a pattern that happens to be present in Dataset outputs: *)
      Positive@Length@Cases[expr, BoxData[TagBox[TagBox[StyleBox[GridBox[{__}, __], __], Deploy], False]], Infinity],
      And[
        Positive@Length@Cases[expr, Cell[_, "Output", ___], Infinity],
        Positive@Length@Cases[expr, TemplateBox[_, "Dataset", ___], Infinity]
      ]
    ],
    optColumnWidths = All];

  new = Transmogrify`Utilities`Private`PrependOptions[expr, {
    (** always **)
    CellMargins->{{0,0},{0,1}},
    ShowCellBracket->False,
    CellContext->$NotebookContext,
    CellOpen->True,
    If[transparentbackground,
      (** only transparentbackground **)
      Unevaluated@Sequence[
        Background->GrayLevel[1,0]
      ],
      Unevaluated@Sequence[]
      ],
    If[inline,
      (** only inline **)
      Unevaluated@Sequence[
        TextAlignment->Left,
        CellDingbat->None,
        ShowCellLabel->False,
        CellFrameLabels->{{None, None}, {None, None}},
        GridBoxOptions->{ ColumnWidths -> optColumnWidths, GridBoxDividers->{} },
        CellFrame->{{0, 0}, {0, 0}},
        CellFrameMargins->{{0, 0}, {0, 0}}
      ],
      Unevaluated@Sequence[]
  ]}];
  If[inline,
    new = DeleteCases[new, _[CellLabel, _]] ];
  new
];



(** NewFile["output-file", expr]
**)
Attributes[NewFile] = {HoldRest};
NewFile[outfile_String, expr_] := Block[
  { 
    $OutputFile = ToFileName[{DirectoryName[$OutputFile]}, outfile]
  },
  
  (* get the directory to which we're going to save files *)
  dir = If[outfile === None || StringFreeQ[outfile, RegularExpression["^(?:/|[A-Z]:\\\\)"]],
    Directory[],
    Transmogrify`Utilities`Private`createDirectory[DirectoryName[outfile]]];

  (* die if we can't write to a new directory *)
  If[dir===$Failed,
    Message[Transmogrify::baddir,dir];
    Return[$Failed],
    (* otherwise change directories *)
    SetDirectory[dir];
  ];

  output = expr;
  
  (* remove Nulls.  Evaluate[] to remove single Sequence[] *)
  output = Evaluate[Sequence @@ DeleteCases[{output},Null,Infinity]];

  (* remove any previous definitions to $XML(Output|Errors) just to save space *)
  Clear[$XMLOutput,$XMLErrors];

  (* set up return value *)
  ret = Which[

    (* export Text for stuff that's already Strings *)
    StringQ[output] || format==="Text",
      Export[ outfile, output, "Text", Sequence @@ exportopts ],

    (* save SymbolicXML to file if valid *)
    format === "XML",
      (* BUG #61406 *)
      Check[ $XMLErrors = XML`SymbolicXMLErrors[output], If[$XMLErrors === {}, $XMLErrors = {{0}}] ];

      If[ Length@$XMLErrors > maxerrs,
        Message[Transmogrify::badxml]; $XMLOutput=output; $Failed,
        Export[ outfile, output, "XML", Sequence @@ exportopts ]
      ],
    (* save all other formats *)
    True,
      Export[ outfile, output, format, Sequence @@ exportopts ]
  ];

  ResetDirectory[];
  ret
]; (* END NewFile *)


GetParameter[s_String] := (
  If[ 
    Length[#] > 0, First[#], Message[Transmogrify::noparam, s]; Null
  ]&[$Parameters[s]]
) /; TrueQ[$Walking] 

(* TODO - deprecate this for blocks instead! *)
SetAttributes[WithParameters, {HoldRest, SequenceHold}]

WithParameters[rules_List, body__] := Module[{params,out},
  params = Map[
    ($Parameters[#[[1]]] = Prepend[$Parameters[#[[1]]],#[[2]]];#[[1]])&,
  rules];
  out = Evaluate[body];
  Scan[
    ($Parameters[#] = Rest[$Parameters[#]])&,params];
  out
] /; TrueQ[$Walking]

(* ToString is automatically applied since it is often used as
   filename counters. *)
IncrementCounter[counter_String] := ToString[++$Counters[counter]] /; TrueQ[$Walking]
DecrementCounter[counter_String] := ToString[--$Counters[counter]] /; TrueQ[$Walking]
GetCounter[counter_String] := ToString[$Counters[counter]] /; TrueQ[$Walking]

ResetCounters[] := $Counters=. /; TrueQ[$Walking]
ResetCounters[counter_String] := ($Counters[counter]=0)  /; TrueQ[$Walking] 


(***********************************************************
      X M L T r a n s f o r m   F U N C T I O N S
 **********************************************************)
(* this section defines the following functions:
      
   XMLTransformInit:  initializes some variables for the current
      transform.  calls getXMLTransform if any of the currently
      loaded files have been modified or if it's a new transform
   findTransform:  searchs $XMLTransformPath for the requested
      file.  returns absolute path.
   getXMLTransform:   imports and merges all IncludeXMLTransform
      references.
   mergeTransforms:   simple utility to merge multiple XMLTransforms
   importXMLTransform:actually imports the file.  Note, this also
      uses the low level $ImportCache to skip over unnecessary
      imports.  It does another FileInformation check, but that's
      ok.
*)

(** XMLTransformInit 
    wrapper for getXMLTransform that takes helps decide if
    we need to do do transform processing at all. 
    this speeds up multiple calls of transmogrify with the
    same transform.  TODO - cache XMLTransforms, too? 

    return $CachedTransform to tell Transmogrify[] to not
    do transform processing.
**)
XMLTransformInit[f_?XMLTransformQ,opts___?OptionQ]:= Module[
  {
    file = findTransform[f]
  },

  Catch[
    If[
      file === $Failed,
      Throw[$Failed]
    ];

    (* see if this file was already requested *)
    If[$LastRequestedTransform === file,

      (* if so, see if we need a reimport *)
      filelist = `bagPart[$XMLTransformList,All];

      (* check to see if any of the Transforms have been modified (or is a URI);
         re-import if so. *)
      Scan[
        If[(Date /. FileInformation[#[[1]]]) > #[[2]] || #[[2]] === Date,
          $XMLTransformList=`bag[];
          `trace[TransformParsing, #[[1]]<>" has changed or is a URL.  Need import"];
          Throw[getXMLTransform[file,opts]]
        ]&, filelist ];

      (* we got here, nothing needs updating, use cached *)
      `trace[TransformParsing, "Using cached transform!"];
      $CachedTransform
      ,
      (* otherwise, this is a new transform, import it. *)
      $LastRequestedTransform = file;
      $XMLTransformList=`bag[];
      `trace[TransformParsing, "New requested transform, importing!"];
      getXMLTransform[file,opts]
    ]
  ]
]

XMLTransformInit[x_XMLTransform,___?OptionQ]:=Module[
  {t = x},
  Which[ 
    (* add shorthand for empty transform *)
    MatchQ[t, XMLTransform[]], 
      t = XMLTransform[{}],
    !MatchQ[t, XMLTransform[l_List,___?OptionQ]],
      Return[$Failed]
  ];
  $XMLTransformList=`bag[];
  If[ $LastRequestedTransform === t,
    `trace[TransformParsing, "Same raw transform requested. Using cached transform!"];
    $CachedTransform,
    `trace[TransformParsing, "New raw transform requested. Saving."];
    $LastRequestedTransform = t
  ]
]

(* see if a file is either a full file name or a URI *)
uriQ[f_String]:=!StringFreeQ[f,RegularExpression["^(?:/|[A-Z]:\\\\|(?:(?i)http|f(?:tp|ile)|paclet)://)"]]

(** 
    findTransform
    this is a helper function for getXMLTransform. it basically 
    finds the first file on $XMLTransformPath that matches the 
    desired filename 
**)

(** findTransform[ URI ] **)
(*  where URI can be an absolute pathname as well. *)
findTransform[f_String?uriQ]:=f

(** findTransform[ filename ] **)
(*  this is the default definition that searches $XMLTransformPath
    for relative filename strings.  returns _Absolute Filename_ *)
findTransform[f_String]:= 
  Block[{$Path=$XMLTransformPath, fn},
    Catch[
      If[(fn = System`Private`FindFile[f]) === $Failed,
        (* try adding a .m if it didn't already have one *)
        If[ StringTake[f,-2]=!=".m" && 
          (fn = System`Private`FindFile[f<>".m"]) =!= $Failed,
          Message[XMLTransform::obsfn,f]
          ,
          (* otherwise fail and let the user know why *)
          Message[Transmogrify::notransfile,f, 
            If[StringFreeQ[f,RegularExpression["^(?:/|[A-Z]:\\\\)"]],
              " in $XMLTransformPath.","."]
          ];
          Throw[$Failed]
        ]
      ];
      fn = System`Private`ExpandFileName[fn];
      (* add current dir for relative pathnames if necessary *)
      If[ Not@StringFreeQ[ fn, RegularExpression["^\\.\\.?[/\\\\]"]],
        fn = ToFileName[{Directory[]}, fn]
      ];
      (* die if we got a directory? *)
      If[ FileType[fn] =!= File, Throw[$Failed] ];
      fn
    ]
  ]

(* platform independent list cases *)
findTransform[{s_String}]:= findTransform[s]
findTransform[{s__String}]:= findTransform[ToFileName[Most@{s},Last@{s}]]
findTransform[{s__String},f_String]:= findTransform[ToFileName[{s},f]]
findTransform[x:{{__String},_String}]:= findTransform[ToFileName@@x]
findTransform[___]:=$Failed

(** WhichXMLTransform - quick/useful public frontend to findTransform **)
WhichXMLTransform[x__] := findTransform[x]/.$Failed->{}

(** GetXMLTransform - quick/useful public frontend to getXMLTransform **)
GetXMLTransform[f_?XMLTransformQ]:= Module[{ff=findTransform[f]},
  If[ff===$Failed, $Failed, getXMLTransform[ff]]]

GetXMLTransform[{s__String},f_String]:= GetXMLTransform[ToFileName[{s},f]]

GetXMLTransform[__]=$Failed

(** getXMLTransform **)
(*
  This function takes an XMLTransform file (result of findTransform[]
  above, i.e. -full filename ) and takes care of actually importing the 
  XMLTransform[] expression.

  Included transforms are merged via the IncludeXMLTransforms option which
  can be in any combination of the normal transform file specifications:
  
  IncludeXMLTransforms->{{"dir","junk.m"},"test.m","/path/to/test2.m"}
  IncludeXMLTransforms->"junk.m"
  IncludeXMLTransforms->"/path/to/junk.m"
  IncludeXMLTransforms->{{"dir","junk.m"}}
  IncludeXMLTransforms->{"file1.m","file2.m"}

  note that IncludeXMLTransforms->{"dir","junk.m"} is ambiguous, so 
  getXMLTransform will try to warn you of this instead of trying 
  {{"dir","junk.m"}} itself.
*)

(* import XMLTransform specified by string from full filename *)
getXMLTransform[$Failed|{},___]=$Failed

getXMLTransform[f_String, fileList_List:{}, opts___?OptionQ] := Module[
  {
    transform, toMerge, newFileList, additionalTransforms,
    abort = AbortOnError /. {opts} /. Options[Transmogrify]
  }, 
  
  Catch[
    Which[
      (* die if this Transform has already been included (or will be) *)
      (* TODO - error message should be more descriptive for the "will be" 
         case *)
      fileList=!={} && MemberQ[fileList,f],
        Message[IncludeXMLTransforms::recurse,f,fileList];
        DieIf[abort];
        Throw[$Failed]
      ,
      (* otherwise import (and die if we couldn't or we got something weird...) *)
      True,
        (* set directory for FileNames[] scoping in Transform *)
        SetDirectory[DirectoryName[f]];
        If[(transform = importXMLTransform[f]) === $Failed,
          DieIf[abort];
          Throw[$Failed]
        ]
    ];
    
    (* now include any requested transforms *)
    If[(toMerge = IncludeXMLTransforms /. Options[transform]) =!= IncludeXMLTransforms,
        transform = DeleteCases[transform, IncludeXMLTransforms~(Rule|RuleDelayed)~_];  
        
        (* loop over all includes to get full URI's *)
        toMerge = Module[{tmp = findTransform[#]},
          If[tmp===$Failed,
            Message[IncludeXMLTransforms::badincl,f,#];
            If[Depth[toMerge]<3,
              Message[IncludeXMLTransforms::dirwarn]];
            DieIf[abort]
            ,
            tmp
          ]]& /@ toMerge;
        
        (* now loop again to actually get each include *)
        additionalTransforms = DeleteCases[
          MapIndexed[
            If[#1=!=$Failed,
              getXMLTransform[#1,Join[{f},Take[toMerge,{1,#2[[1]]-1}],fileList],opts]
            ]&,
            toMerge
          ], Null
        ];

        (* merge transforms together if we got some *)
        If[Length@additionalTransforms > 0,
          transform = 
            mergeTransforms[Join[{transform},additionalTransforms]];
        ]
    ];
    ResetDirectory[];
    transform 
  ]
]

(** mergeTransforms[]
    does exactly what you'd expect
**)
mergeTransforms[l:{_XMLTransform..}]:=
  XMLTransform @@ MapAt[Sequence @@ # &,
    Join @@@ Transpose[
      {First[#], Flatten[{Options[#]}]} & /@ l
    ], {2}]

mergeTransforms[l_List]:=
  mergeTransforms[Cases[l,_XMLTransform,Infinity]]
  
mergeTransforms[{}]=$Failed

(** importXMLTransform - helper function for getXMLTransform **)
(*
    this section of code uses the FastImport function which makes
    use of an $ImportCache for repeated imports of the same file.
    see ImportCache.m for details of how this works.

    if we got here, f _exists_ and is an absolute pathname. 
*)
importXMLTransform[f_String, opts___?OptionQ]:= Module[
  {ext, format, transform},
  
  (* see if it's a dump file *)
  ext = StringReplace[f, RegularExpression[".*\\."]->""];
  
  format = If[ ext === "mx", 
    "Dump"
    ,
    $PackageFormat
  ];
 
  (* import transform if necessary, otherwise get cache data *)
  transform = Transmogrify`ImportCache`FastImport[f, format];

  (* set up current list of current transform files and moddates
     for use in deciding if we need to process templates at all. *)
  `stuffBag[$XMLTransformList,
    {f, First@Cases[Transmogrify`ImportCache`$ImportCache,
        Transmogrify`ImportCache`CacheObject[f,_,moddate_,___]:>moddate]
    }
  ];
  
  If[MatchQ[transform,l_List/;l=!={}],
    (* TODO - something other than imitate 5 behavior *)
    transform=Last@Flatten[{transform},1]
  ];
  
  Switch[ transform,
    (* Import died for some unknown reason.  probably file format
       mismatch *)
    $Failed,
    Message[Transmogrify::imptrans,f];
    $Failed
    ,
    (* import succeeded but we got old stuff *)
    _Template,
    Message[Transmogrify::obstrans,f];
    transform /. {Template->XMLTransform, IncludeTemplates->IncludeXMLTransforms}
    ,
    e_ /; Head@e=!=XMLTransform,
    (* import succeeded but we got bollocks *)
    Message[Transmogrify::badtrans,f];
    $Failed
    ,
    (* otherwise everything's coo *)
    _,
    transform
  ]
]

(** TODO - this was to support IncludeXMLTransform->"*.m" 
  functionality but hasn't been plugged in yet.
**)
(* kinda slow, but handles rudimentary wild cards *)
findFiles[x_, path_List:$XMLTransformPath ]:= 
Catch[
  Scan[ Module[{o = FileNames[#<>x]},
    If[o =!= {}, Throw[o]]
  ]&, path];
  Throw[$Failed]
]

hasWildCardQ[s_String]:= 
  !StringFreeQ[s,RegularExpression["(?<!\\\\)[*?]"]]


(* XMLTransform related error messages *)
Transmogrify::badtrans = "The XMLTransform file `1` did not contain an XMLTransform."
Transmogrify::imptrans= "Could not import the file `1` for some reason.  Make sure that it is a file containing valid Mathematica code."
Transmogrify::obs = "The option or function `1` for Transmogrify is now obsolete. `2` Please\
 see the documentation."

Transmogrify::obstrans = "The Template[] syntax for Transmogrify is now obsolete.\
 Attempting to convert to XMLTransform[].\n\
 Please see the documentation."

Transmogrify::notrans = "You specified an XMLTransform but Transmogrify\
 was not able to correctly process it or one of its included transforms."
Transmogrify::notransfile = "Transmogrify could not find the transform file \"`1`\"."


IncludeXMLTransforms::recurse = "The transform `1` has already been included by\
 one of the following files `2`."
IncludeXMLTransforms::badincl = "While processing the transform `1`, the included\
 transform `2` could not be found."
IncludeXMLTransforms::dirwarn = "Note: Use {{\"dir\",\"file.m\"}} when including single files of the form {\"dir\",\"file.m\"}."

IncludeXMLTransform::expand = "the expansion of \"`1`\" expands to \"`2`\" files."
IncludeXMLTransforms::noincl = "There were some problems processing the included\
 transforms from `1`."

XMLTransform::obsfn = "A transform file for `1` was not found, but one for `1`.m was.
 The filename format of {\"dir\",\"format\"}, where format does not have a full extension\
 is deprecated.  Please use the entire filename extension."

XMLTransform::imgattrs = "Warning: missing `1` attribute within Image tag."; 


(***********************************************************
           U T I L I T Y   F U N C T I O N S
 **********************************************************)

(** DieIf - flow control helper **)
DieIf[ bool_ ]:= If[ bool === True,
  Message[Transmogrify::abort]; 
  SetDirectory[$InitialTransmogrifyDirectory];
  Transmogrify`Debug`Private`traceHandlerCleanUp[Transmogrify`Debug`$TraceHandler];
  Abort[ ]
  ,
  Message[Transmogrify::continue];
]
Transmogrify::abort = "Transmogrify encountered an error and is Aborting (see\
 AbortOnError)"
Transmogrify::continue = "Transmogrify encountered an error but is attempting\
 to continue anyway.  Results may be unpredictable. (see AbortOnError)"


(**************** XMLTransform  Helpers ******************)
getcelltags[n:{_, __}] := Map[(XMLElement["a", {"name" -> #}, {""} ])&, n]; 
getcelltags[n:{_}] := XMLElement["a", {"name" -> n[[1]]}, {""} ]; 
getcelltags[n_String] := XMLElement["a", {"name" -> ToString[n]}, {""} ]; 
getcellID[x_Integer] := XMLElement["a", {"name" -> ToString@x}, {""} ]; 

TagElement[element_String, attr_List, cont_]:=  
XMLElement[element, attr, Flatten@{
  		Which[HasOption[CellTags] && HasOption[CellID],
			Sequence@{(*getcelltags[GetOption[CellTags]], *)getcellID[GetOption[CellID]], cont  },
		HasOption[CellID], Sequence@{getcellID[GetOption[CellID]], cont}, 
		(*HasOption[CellTags], Sequence@{getcelltags[GetOption[CellTags]], cont},*)
		True, cont]
		}];

DIV[ attr_List:{}, cont_ ] := TagElement["div", attr, cont];
P[ attr_List:{}, cont_ ] := TagElement["p", attr, cont];
Ol[ attr_List:{}, cont_ ] := TagElement["ol", attr, cont];
Li[ attr_List:{}, cont_ ] := TagElement["li", attr, cont];
H1[ attr_List:{}, cont_ ] := TagElement["h1", attr, cont];
H2[ attr_List:{}, cont_ ] := TagElement["h2", attr, cont];
H3[ attr_List:{}, cont_ ] := TagElement["h3", attr, cont];
H4[ attr_List:{}, cont_ ] := TagElement["h4", attr, cont];
TD[ attr_List:{}, cont_ ] := TagElement["td", attr, cont];

Unprotect[Span];
Span[ attr_List:{}, cont_ ] := XMLElement["span", attr, Flatten@{cont} ];
A[ attr_List:{}, cont_ ] := XMLElement["a", attr, Flatten@{cont} ];
UL[ attr_List:{}, cont_ ] := XMLElement["ul", attr, Flatten@{cont} ];
(* Li[ attr_List:{}, cont_ ] := XMLElement["li", attr, Flatten@{cont} ]; *)
Br[ attr_List:{} ] := XMLElement["br", attr, {}];
Br[] := XMLElement["br", {}, {}];
Img[ attr_List:{} ] := 
(
(*  If[Not@MemberQ[attr, "height" -> _],
    Message[XMLTransform::imgattrs, "height"] ];
  If[Not@MemberQ[attr, "width" -> _],
    Message[XMLTransform::imgattrs, "width"] ];*)
  XMLElement["img", attr, {}]
);


(** Image **)
Unprotect[transmogrifyImage];
transmogrifyImage[ filename_String, attr_List:{}, expr___, opts___?OptionQ] := Block[
  {
    defaultWidth, defaultHeight, btiWrappedImageReturn, f,
    wrappedWidth, wrappedHeight,
    $ImageResolution = 72, (* default resolution *)
    mag = 2, (* writing 2x files *)
    origdefaultWidth, origdefaultHeight,
    $defaultWindowSize = 550, (* from Export.m *)
    inputQ = False, outputQ = False,
    origwrappedWidth, origwrappedHeight, out
  },

  inputQ  = StringMatchQ[FileBaseName[filename], StartOfString ~~ "I_*"];
  outputQ = StringMatchQ[FileBaseName[filename], StartOfString ~~ "O_*"];

  If[Not[inputQ],
    If[outputQ,
      out = BoxToImage[
        filename,
        expr,
        Sequence @@ {Magnification -> mag, opts}
      ];
      {defaultWidth, defaultHeight} = Ceiling[ImageDimensions[Import[filename]] / 2];
      Return[
        XMLElement["img", {
          "src" -> ("URL" /. #),
          "height" -> ToString[defaultHeight],
          "width" -> ToString[defaultWidth],
          Sequence @@ attr},{}
        ]&[out]
      ]
    ,
      Return[
        XMLElement["img", {
          "src" -> ("URL" /. #),
          "height" -> ("Height" /. #),
          "width" -> ("Width" /. #),
          Sequence @@ attr},{}
        ]&[BoxToImage[filename, expr, opts]]
      ]
    ]
  ,

    (* every call to BTI exports expr to filename and returns information about the procedure *)
    (* btiDefaultImageReturn is used when writing default html *)
    btiDefaultImageReturn = BoxToImage[
      filename,
      expr,
      ConversionOptions -> {
        (* this is supposedly applied to the image before any changes to size or resolution *)
        ImageFormattingWidth -> $defaultWindowSize
      },
      Sequence @@ {Magnification -> mag, opts}
    ];
    {origdefaultWidth, origdefaultHeight} = ImageDimensions[Import[filename]];
    {defaultWidth, defaultHeight} = Ceiling[{origdefaultWidth, origdefaultHeight}/2];
  
    If[defaultWidth > 405, 
      f = StringReplace[filename, FileBaseName[filename] :> FileBaseName[filename]<>"_405"];
      (* note that this return value is essentially worthless, but I keep it to report (below) what's happening through the flow of the function *)
      btiWrappedImageReturn = BoxToImage[
        f,
        expr,
        ConversionOptions->{ImageFormattingWidth -> 405},
        Sequence @@ {Magnification -> mag, opts}
      ];
  
      (*********************************************************************
      note that the output of BoxToImage cannot be trusted when image-changing
      options are passed to BoxToImage, because BTI sends the expression off to
      ToRasterDataPacket, *which does not consider options*!
  
      TODO: To get the wrapped image width for use in html, we have to Import
      the file and find its dimensions.
      *********************************************************************)
  
      {origwrappedWidth, origwrappedHeight} = ImageDimensions[Import[f]];  
      {wrappedWidth, wrappedHeight} = Ceiling[{origwrappedWidth, origwrappedHeight} / 2];

      Return[
        XMLElement[
          "img",
          {
            "src"        -> "",
            "data-src"   -> ("URL" /. btiDefaultImageReturn),
            "data-big"   -> StringJoin[ToString[defaultWidth], " ", ToString[defaultHeight]],
            "data-small" -> StringJoin[ToString[wrappedWidth], " ", ToString[wrappedHeight]],
            Sequence @@ attr
          },
          {}
        ]
      ]
    ,
      Return[
        XMLElement[
          "img",
          {
            "src"    -> ("URL" /. #),
            "height" -> ToString[defaultHeight],
            "width"  -> ToString[defaultWidth],
            Sequence @@ attr
          },
          {}
        ]&[btiDefaultImageReturn]
      ]
    ]
  ]
]

(** image list **)
ImageList[ filename_String, attr_List:{}, expr___, opts___?OptionQ]:=
Join[
 {XMLElement["img",{
    "src"->("URL"/.#),"height"->("Height"/.#),"width"->("Width"/.#),Sequence@@attr},{} ]},
 #
]&[BoxToImage[filename, expr, opts]]



(* If Comment for SSI  *)
Options[IfCommentSSI] = {ExprCommand -> "is_development", Recurse->True};
IfCommentSSI[t_, f___, opts___?OptionQ] := 
DIV[{
  "\n", 
  XMLObject["Comment"]["#if expr='" <> 
    "${is_development} = 1" <> 
    "' "], 
  Recurse @ t, 
  XMLObject["Comment"]["#else"], 
  Recurse @ f, 
  XMLObject["Comment"]["#endif"], 
  "\n"
}];


(* -------------------------------------
              T H E   E N D
   ------------------------------------- *)

End[] (* "`Private`" *)