(******************************************************************************
   Literalize.m

   Contains code for changing the CounterBoxes in a notebook to literal values.

   Revision History:

     1/3/2001 -- First version written by Hormozd Gahvari

     1/4/2001 -- First version cleaned up to make first working version, also 
                 by H. Gahvari

     2/15/2001 -- Added code to handle CounterIncrements or CounterAssignments 
                  options that have been spliced into a cell by hand, again 
                  done by H. Gahvari

	8/13/2003 -- Updates by Andy Hunt to make v5 compatible and remove FE dependencies

     2/9/2004 -- Do CounterAsssignments before CounterIncrements, because
                 that's the way it's supposed to be done.  (Shaun McCance)
******************************************************************************)


Begin["`Literalize`"];

(* <!-- Public
Literalize::usage = "Literalize[nb] converts all CounterBoxes in nb to corresponding literal values."
Literalize::notes = "See also: PreProcessNotebook";
 --> *)

GetStyleSheet::usage = "GetStyleSheet[nb] returns the notebook expression of the style sheet assigned to the notebook.";
 

Begin["`Private`"]

(* --------------------------------------- *
         G e t S t y l e S h e e t
 * --------------------------------------- *)

(* 
   TODO - this should be replaced with a reasonable FE level command.
          talk with daleh/lou and jfultz.
          Investigate v6's NotebookInformation[] for NotebookObjects:

          styleSheetData = 
             NotebookGet@@("StyleDefinitions"/.NotebookInformation[nbobject]);
*)

GetStyleSheet[nb_]:= Module[
  {styleSheetLocations},
	styleSheetLocations = 
      Flatten[ListDirs /@ (StyleSheetPath /. Options[$FrontEnd, StyleSheetPath])];
      
    styleSheetLocations = Join[styleSheetLocations, {
      ToFileName[{$PreferencesDirectory, "SystemFiles", "FrontEnd", "StyleSheets"}],
      Directory[]
	} (*, nbdir*) ];

	Switch[styleSheetName, 
	
	  (* StyleDefinitions is set to a String, meaning that the style sheet is 
		 specified by a notebook in $StyleDirectory. *)
		_String, styleSheetData = Get[styleSheetName, Path->styleSheetLocations];

	  (* The style sheet is in a directory other than $StyleDirectory. *)
		_FrontEnd`FileName, styleSheetData = GetFEFile[styleSheetName];

	  (* The only other possibility is that StyleDefinitions is set to a Notebook 
		 expression, i.e., the style sheet itself. *)
		_, styleSheetData = styleSheetName];

	styleSheetData
	];

GetFEFile[FrontEnd`FileName[d_, f_, o___]]:= Import[ToFileName[d, f], "Notebook"];

(* Get list for directories that include _ *)

(* list dir in fe and kernel filesystems *)
ListDirs[FrontEnd`FileName[{begin___, Verbatim[_], end___}, ___]] :=
   listdirs/@{
     FrontEnd`FileName[{begin, _, end}] /.
       {HoldPattern[$TopDirectory] -> FeTopDir,
       HoldPattern[$PreferencesDirectory] -> FePrefDir},
     FrontEnd`FileName[{begin, _, end}]
   }

FeTopDir :=
   FeTopDir = ToFileName[$TopDirectory]

FePrefDir :=
   FePrefDir = ToFileName[$PreferencesDirectory]

listdirs[FrontEnd`FileName[{begin___, Verbatim[_], end___}, ___]] :=
   Module[{choices},
     choices = Select[FileNames["*", ToFileName[{begin}]],
       (FileType[#]===Directory)&];
     Map[ToFileName[{#, end}]&, choices]
   ]

(* Convert directories that do not include _ *)

(* list dir in fe and kernel filesystems *)
ListDirs[dir_] :=
   listdirs/@{
     dir,
     dir /.
       {HoldPattern[$TopDirectory] -> $TopDirectory,
       HoldPattern[$PreferencesDirectory] -> $PreferencesDirectory}
   }

(* ------------------------------------- *
          L i t e r a l i z e
 * ------------------------------------- *)
Literalize[nb_NotebookObject] := Literalize@NotebookGet@nb;

Literalize[nb_Notebook] :=
  Module[{nbdata, nbcells, oldNBcells, newNBcells, styleDefs, styleNB, styleSheetData, 
          counters, optionsWithCounters, counterStyles, old, new, appendExprs, 
          countData, theStyle, stylePos, countPos, counterRules, increments, 
          assignments, stylePath, counterIncs, nb1, c},

    nbcells = Cases[nb, c_Cell /; Head[c[[1]]] =!= CellGroupData, {1, Infinity}];

	(* Get style sheet assigned to nb *)

    styleSheetData = getStyleSheet[nb];
    
    (* initialize counters that are present in style sheet *)

    counters = 
      List[#, 0] & /@ 
        Union[Join[First /@ Cases[styleSheetData, _CounterBox, {1, Infinity}], 
          If[Head[#[[2]]] === List, 
                Sequence @@ 
                  DeleteCases[Flatten[#[[2]]], _Integer], #[[2]]] & /@ 
            Cases[styleSheetData, 
              r_Rule /; 
                r[[1]] === CounterAssignments || 
                  r[[1]] === CounterIncrements, {1, Infinity}]]];

    (* extract cells from nb that are of styles that factor into automatic numbering *)

    counterStyles = 
      Part[#, 1, 1] & /@ 
        Cases[styleSheetData, 
          c_Cell /; (Head[c[[1]]] =!= CellGroupData && 
            MemberQ[c, 
              r_Rule /; (r[[1]] === CounterAssignments || 
                    r[[1]] === CounterIncrements || 
                    MemberQ[r, _CounterBox, {1, Infinity}]), {1, 
                Infinity}]), {1, Infinity}];

    oldNBcells = Cases[nbcells, c_Cell /; (Length[c] >= 2 && MemberQ[counterStyles, 
      c[[2]]])];

    (* delete inline cells from oldNBcells *)

(*    oldNBcells = 
      DeleteCases[
        MapIndexed[If[! MemberQ[Drop[oldNBcells, #2], #, {1, Infinity}], #] &, 
          oldNBcells], Null]; *)

    (* Set newNBcells to oldNBcells.  From now on, operations will be performed only 
       on newNBcells. *)

    newNBcells = oldNBcells;

    (* get information on cell options to splice into cells that have CounterBoxes in 
       cell options like CellFrameLabels *)

    optionsWithCounters = 
      List[#[[1, 1]], 
          Cases[#, r_Rule /; MemberQ[r, _CounterBox, {1, Infinity}]]] & /@ 
        Cases[styleSheetData, 
          c_Cell /; (Head[c[[1]]] =!= CellGroupData && 
              MemberQ[c, 
                r_Rule /; MemberQ[r, _CounterBox, {1, Infinity}], {1, 
                  Infinity}]), {1, Infinity}];

    (* perform the splicing *)

    Do[old = Cases[newNBcells, c_Cell /; c[[2]] === optionsWithCounters[[i, 1]]];
       appendExprs = Table[optionsWithCounters[[i, 2]], {Length[old]}];
       appendExprs = 
         MapIndexed[
           DeleteCases[#, r_Rule /; MemberQ[old[[First[#2]]], r[[1]], {2}]] &, 
             appendExprs];
       new = DeleteCases[#, {}] & /@ 
         MapIndexed[
           If[appendExprs[[First[#2]]] =!= {}, 
            Append[#, Sequence @@ appendExprs[[First[#2]]]], Append[#, {}]] &,
              old];
       newNBcells = newNBcells /. (Apply[Rule, #] & /@ Transpose[{old, new}]),
    {i, Length[optionsWithCounters]}];

    (* Set up data structure to hold info about which styles increment certain counters 
       and which styles set certain counters.  The structure looks like this: 

       {styleName, {counterIncrements}, {counterAssignments}}

       The list {counterAssignments} list looks like this:

       {{counterName1, assignment}, {counterName2, assignment}, ... } *)

    countData = 
      List[#[[1, 1]], 
        Cases[#, 
          r_Rule /; 
            r[[1]] === CounterAssignments || 
              r[[1]] === CounterIncrements]] & /@ 
      Cases[styleSheetData, 
        c_Cell /; (Head[c[[1]]] =!= CellGroupData && 
            MemberQ[c, 
              r_Rule /; (r[[1]] === CounterAssignments || 
                    r[[1]] === CounterIncrements), {1, Infinity}]), {1, Infinity}];

    countData = 
      List[#[[1]], 
          If[Length[#[[2]]] == 2, 
            Sequence @@ {If[
                  Head[#[[2, 1]]] === List, #[[2, 1]], {#[[2, 1]]}], #[[2, 
                    2]]}, Sequence @@ {#[[2]], {}}]] & /@ 
      MapAt[Part[#, 2] &, countData, Position[countData, _Rule]];

    (* Loop through the cells in newNBcells, updating the counters and making 
       replacements as needed. *)

    Do[theStyle = newNBcells[[i, 2]];
       stylePos = First[Flatten[DeleteCases[Position[countData, theStyle], 
         l_List /; Length[l] > 2]]];

       (* perform counter assignments *)

       assignments = If[MemberQ[newNBcells[[i]], r_Rule /; r[[1]] === CounterAssignments],

         (* current cell has user-defined counter assignments; handle appropriately *)
         Options[newNBcells[[i]], CounterAssignments][[1,2]],

         (* no user-defined counter assignments; proceed normally *)
         countData[[stylePos, 3]]

       ];

       CompoundExpression[
         countPos = First[Flatten[Position[counters, #[[1]]]]],
         counters[[countPos, 2]] = #[[2]]
       ]& /@ assignments;

       (* perform counter increments *)

       If[MemberQ[newNBcells[[i]], r_Rule /; r[[1]] === CounterIncrements],

         (* current cell has user-defined counter increments; handle appropriately *)
         counterIncs = Options[newNBcells[[i]], CounterIncrements][[1,2]];
         increments = If[Head[counterIncs] =!= List, {counterIncs}, counterIncs],

         (* no user-defined counter increments; proceed normally *)
         increments = countData[[stylePos, 2]]

       ];

       CompoundExpression[
         countPos = First[Flatten[Position[counters, #]]],
         ++(counters[[countPos, 2]])
       ]& /@ increments;

       (* replace CounterBoxes with literal values *)

       counterRules = Apply[Rule, #]& /@
         (List[CounterBox[#[[1]]], ToString[#[[2]]]]& /@ counters);
       newNBcells[[i]] = newNBcells[[i]] /. counterRules,        

    {i, Length[newNBcells]}];


    (* Make replacement rules out of oldNBcells and newNBcells, apply the rules, and 
       presto!  No more CounterBoxes. *)

    nb /. (Apply[Rule, #]& /@ Transpose[{oldNBcells, newNBcells}])

  ]
 

End[]
End[]

(*
vi: tw=88 sw=2
*)
