(* -*- Mode: fundamental; tab-width: 3 -*- *)
(* 
    Title: Layout.m
    $Id: Layout.m,v 1.9 2007/07/23 20:19:34 shaunm Exp $

    Context: Transmogrify`Layout`

    Authors: stephen layland 

    Copyright: © February 2005 by stephen layland

    Requirements: 
    Mathematica 5.1 or later for RegularExpression capabilities
    
    Resources: 
    Friedl, Jeffrey E. F. "Mastering Regular Expressions"    

    TODO:

    o  Fix block scoping implementation.  there's probably a better way.

    o  clean up implementation, perhaps break up code to two separate private
       functions, one for Strings the other for XML.
*)

(* -------------------------------------------------------------------------- *)
(*  this was originally written for the Transmogrify nb->XML converter, but   *)
(*  it can be used for any number of generic text based layouts. See        *)
(*  documentation in help browser                                             *)
(* -------------------------------------------------------------------------- *)

(* changed from BeginPackage to create a subcontext of Transmogrify` *)
Begin["`Layout`"];


(* ----------------------------- *)
(*   public methods              *)
(* ----------------------------- *)

(* export usage statements to Transmogrify` context *)
(* <!-- Public 
FillInLayout::usage = "FillInLayout is a utility which allows you to create a\
 structure based file of your choosing, and interpolate specially formatted text\
 as arbitary Mathematica code.  The following different methods exist for interpolating\
 variables from the default scope, from a list, or from a context, respectively:\n
 FillInLayout[\"layoutfile\",\"format\",opts]\n\
 FillInLayout[\"layoutfile\",\"format\",{varlist},opts]\n\
 FillInLayout[\"layoutfile\",\"format\",\"context`\",opts],\n\
 where {varlist} is of the form {\"var1\"->var1,...}, or {\"var1\", \"var2\"->val2,...}.";
 
EmbeddedCodeRule::usage = "This options is a rule that will convert either a\
 string, or a Mathematica expression, of your choice into a _String_ that denotes\
 arbitrary Mathematica code that will be passed off to ToExpression.  Setting this to\
 Automatic causes FillInLayout to recognize <!-- {mcode} --> as Mathematica code for all\
 files imported as SymbolicXML (default), and uses ${mcode} for regular text files. See\
 FillInLayout.nb for examples.";
-->*)

(* ----------------------------- *)
(*   BEGIN IMPLEMENTATION        *)
(* ----------------------------- *)

Begin["`Private`"]

(* default options *)

Options[FillInLayout] = {
    ConversionOptions->Automatic,
    EmbeddedCodeRule->Automatic
}

(* error messages *)
FillInLayout::badrule = "EmbeddedCodeRule needs to be a rule that converts embedded mathematica code into a straight string representation.";


(* FillInLayout["filename", "format", opts] *)
(* this is a Block[] because it can get variables defined in the scope
   that FillInLayout[] was called in *)

FillInLayout[fn_String,formatin_String:"", varlist_List:{}, opts___?OptionQ]:=Block[
  {format=formatin,convopts, structure, pos, vars, rule, position, replacepart, extract,
  replace, singlevars, doubles, singlevals, rules},
  Catch[
    (* snag known options *)
    {convopts, rule} = {ConversionOptions, EmbeddedCodeRule} /. {opts} /.
        Options[FillInLayout];
           
    (*--- get "varN" \[Rule] valN options for replacement ---*)
    (* 
       basically, any Rule that is of the form "string" goes to something 
       these will take precedence over any of the same variables in varlist
    *)

    vars = Cases[Flatten@{opts}, HoldPattern[(Rule | RuleDelayed)[_String, _]]];

    (* use file extension if format wasn't given *)
    If[format === "",
      format = StringReplace[fn, RegularExpression@".*\\.(.+)$" -> "$1"]];

    (* set up automatic conversion options depending on format *)

    If[convopts === Automatic,
      convopts =
        If[
          !StringFreeQ[format, RegularExpression@"ml$", IgnoreCase->True],
          {"IncludeEmbeddedObjects" -> "Comments"}
          ,
          {}
        ]
    ];

    structure = Import[fn, format, Sequence@@convopts];
    If[structure === $Failed, Throw[$Failed]];

    (* handle string replacements *)
      
    (* set up Automatic rule or die if we got a bad one from user *)
    Which[
      rule === Automatic,
      (* rule to turn ${mcode} into mcode.  i love regexes! *)
      rule = RegularExpression[
  "(?x)
     (?<!\\\\)\\${                   # match an opening ${ not preceded by a slash   \n
     (                               # and start capturing the stuff that follows    \n
         (?:[^\\\\\\n](?!\\${))*     # then any non \ or newline character that is not\n
         (?:                         #    followed by an escaped ${                  \n
             \\\\                    # now match any \ or newline that we got to earlier \n
             (?:[^\\\\\\n](?!\\${))* # and any amount of valid embedded code that follows\n
         )*                          # as many times as you can                      \n
     )                               
     }                               # lastly, make sure it ends with a closing } \n
  "
          ]:>"$1"
      ,
      !MatchQ[rule, (_String | _RegularExpression)~(Rule | RuleDelayed)~_],
      Message[FillInLayout::badrule];
      Throw[$Failed]
    ];

    pos = StringPosition[structure, First@rule];

    (* leave now if we didn't find anything to replace *)
    If[pos === {},
      Throw[structure]];
      
    (* If we didn't get any vars passed in, try to get stuff from 
       FillInLayout's scope *)
    If[Length@Join[vars,varlist] == 0,
      vals = ( ToExpression /@ StringReplace[StringTake[structure, pos],rule] )
          /. Null->"";
      (*
         i know you can do this in one simple StringReplace call, but 
         this way the EmbeddedCodeRule pattern is consistent.
      *)
	   (* FIXME *)
      newpos=pos;
      Do[
        structure = StringReplacePart[structure, ToString@vals[[i]], First@newpos];
        newpos = StringPosition[structure,First@rule];
        ,
        {i,Length[vals]}
      ];

      ,
      (* otherwise use only the variables we are asking for that are passed
         in as "var" -> val options or as a varlist argument *)
      
      (* first format all the var types we have *)
      (* "varname" *)
      singlevars = Cases[varlist,_String];
      (* get vals from scope if not set *)
      singlevals = ToExpression/@singlevars;
        
      (* "varname" -> val *)
      doubles = Join[vars,Cases[varlist,_String~(Rule|RuleDelayed)~_]];
      If[Length[singlevars] > 0,
        rules = Join[
          doubles,
          Rule@@@Transpose[{singlevars,singlevals}]
        ];
        ,
        rules = doubles
      ];
        
      (**
        stuff.
         we need rules of the form "string" -> "\"another string\"" for strings, and
         all "'s escaped for normal expressions because the second part of the rule 
         is piped through ToExpression.
      *)
      rules[[All,2]] = ToString[#,FormatType->InputForm]& /@ rules[[All,2]];
      rules[[All,1]] = StartOfString ~~ # ~~ EndOfString & /@ rules[[All,1]];
      vals = ToExpression /@ ( 
          StringReplace[StringReplace[StringTake[structure,pos],rule],rules]
        ) /. Null->"";

	   structure = StringReplacePart[structure, ToString/@vals, pos];
    ];
    structure
  ]
];

FillInLayout[fn_String, format_String, c_String /; !StringFreeQ[c,
  RegularExpression["`\\*?$"]], opts___?OptionQ]:= 
Module[
  {context=c,names,varlist},
  
  If[StringFreeQ[context, "*"~~EndOfString],
    names=Names[context<>"*"]
    ,
    names=Names[context]
  ];

  varlist=Rule@@@({StringReplace[#,context->""],ToExpression[#]}&/@names);
  FillInLayout[fn, format, varlist, opts]
];

(* FillIn and export to file *)
FillInLayout[outfile_String, layoutfile_String, format_String, opts___?OptionQ]:=
  Export[
    outfile,
    FillInLayout[layoutfile,format,opts],
    format, "AttributeQuoting"->"\""
  ];

End[] (*`Private`*)

End[] (*`Layout`*)

