(* :Title: ImportCache.m`*)

(* :Context: ImportCache.m` *)

(* :Authors: stephen layland *)

(* :Copyright: © April 2005 by stephen layland *)

(* :Package Version: 1.0 *)

(* :Requirements: *)

(* subcontext of Transmogrify` *)
BeginPackage["`ImportCache`"];

(* -------------------------------------------------------------------------- *)
(*  this package implements a simple import cache to speed up repeated        *)
(*  imports that can occur in recursive or complex programs.  unlike Needs[]  *)
(*  FastImport is file-sensitive, rather than context sensitive, which allows *)
(*  for more flexibility.                                                     *)
(*                                                                            *)
(*  this was modified for inclusion in the Transmogrify` package, where it    *)
(*  is generously used from a subcontext to parse many transformation files   *)
(*  given by users.                                                           *)
(* -------------------------------------------------------------------------- *)

(* ----------------------------- *)
(*   public methods              *)
(* ----------------------------- *)

(* export usage messages to Transmogrify` *)

(* <!-- Public *)
CachedQ::usage = "CachedQ[file] tells you if a file is stored in the cache or not"
CacheObject::usage = "CacheObject[filename, ...] represents cached data."
CacheSize::usage = "CacheSize[] returns the current size of the cache in Bytes"

FastImport::usage = "FastImport[] is a wrapper for Import that uses a cache to speed\
 up imports.  Syntax and options are the same as Import[]."
ForceImport::usage = "ForceImport->True forces FastImport to import an already\
 cached file."

$ImportCache::usage = "$ImportCache is a list of CacheObjects currently in memory"
CachedFiles::usage = "CachedFiles[] returns a list of files in the ImportCache"

$MaxCachedFiles::usage = "$MaxCachedFiles is the maximum number of files that\
 exist in the $ImportCache.  Set to negative value or Infinity to disable."
$MaxCacheSize::usage = "$MaxCacheSize is the upper limit in Bytes at which the\
 $ImportCache will purge old files.  Set to negative value or Infinity to\
 disable."

PurgeCache::usage = "PurgeCache...well, purges the cache... irreversibly."
(* --> *)

(* ----------------------------- *)
(*   BEGIN IMPLEMENTATION        *)
(* ----------------------------- *)

(* initialization *)
$MaxCacheSize = 256*1024^2 (*512 MB*)
$MaxCachedFiles = 10

If[Head@$ImportCache =!= List || Length@$ImportCache <= 0,
    $ImportCache = {}
]

Begin["`Private`"]

qr[s_String]:= StringReplace[s,"\\"->"\\\\"];

(* quick helper *)
expandFile[f_String]:= If[
  StringFreeQ[f, RegularExpression["^(?:ht|f)tp://"]],
  
  System`Private`ExpandFileName[
    Evaluate@StringReplace[f, 
      RegularExpression["^\\.("<>qr@$PathnameSeparator<>")"]:>
        Directory[]<>"$1"
    ]
  ],
  f
]

CachedFiles[]:= $ImportCache[[All,1]]
CachedQ[f_String]:= MemberQ[$ImportCache, CacheObject[expandFile@f,___]]
CacheSize[]:= Total[$ImportCache[[All,2]]]


ModifiedDate[f_String?CachedQ]:= First@Cases[$ImportCache, c:CacheObject[f,__]:>c[[3]]]

(*
    CacheObjects currently have the following form:

    CacheObject[
        full filename,
        byte count of imported data,
        file modification date,
        time in seconds it took to import,
        how many times this file has been requested,
        how long (in import cycles) has this file been cached
    ]
*)

UpdateCounters[]=UpdateCounters[""]
UpdateCounters[f_String]:= 
  $ImportCache = $ImportCache /. {
    CacheObject[f,stuff__,freq_,age_]:>CacheObject[f,stuff,freq+1,age+1],
    CacheObject[stuff__,age_]:>CacheObject[stuff,age+1]
  }

PurgeCache[]:= (CacheObject=.; CacheData=.; $ImportCache={};)

Options[FastImport] = Join[{ForceImport->False},Options[Import]];
FastImport[file_String, format:(_String|Automatic):Automatic, opts___?OptionQ]:=
Module[
  {
    current=Date/.FileInformation[file], data, f=expandFile[file], 
    force=ForceImport/.{opts}/.Options[FastImport]
  }, 
  Catch[
    Which[
      (* die if we got a weird file *)
      !StringQ[f],
        Throw[$Failed]
      ,
      (* return cached value if we can. if current === Date, it's a URI *)
      force=!=True && CachedQ[f] && ModifiedDate[f] === current && current =!= Date,
        UpdateCounters[f];
        Throw[CacheData[f]]
    ];

    (* if we get here we need to import *)
    impopts = Sequence @@ DeleteCases[{opts},_[ForceImport,_]];
    time = AbsoluteTiming[(data = Import[f, format, impopts];)][[1]]/Second;
    
    (* import failed for some reason *)
    If[MatchQ[data,$Failed|_Import],
      Throw[$Failed]
    ];

    (* setup cache, update counters *)
    CacheData[f] = data;
    UpdateCounters[f];
    
    (* 
       if the file is already cached (i.e. - the file
       was modified after original import), update the 
       cache entry.  hits/age already updated with UpdateCounters[]
    *)
    If[ CachedQ[f],
      $ImportCache = $ImportCache /. CacheObject[f,__, hits_, age_]
        :>CacheObject[f,ByteCount[data],current,time,hits,age]
      ,
      (* otherwise add it to the cache *)
      $ImportCache = Flatten[{
          CacheObject[f,ByteCount[data],current,time,1,1], $ImportCache}]
    ];

    (* invalidate cache entries if need be *)
    If[CacheSize[] > $MaxCacheSize || Length@$ImportCache > $MaxCacheFiles,
      UpdateCache[]
    ];
    Throw[data];
  ]
]

UpdateCache[]:= Block[{},
  (* 
     gives cache entries priorities based on
    
                    age of cache entry
        ------------------------------------------
        (# of requests) (import time) (memory size) 
    
     The smaller this is, the more important that the
     entry stays in cache.  see CacheObject description
     above.
  *)
  
  While[ 
    (CacheSize[] > $MaxCacheSize > 0) ||
    (Length@$ImportCache > $MaxCacheFiles > 0)
    ,
    $ImportCache = Drop[
      $ImportCache[[
        Ordering[
          #1/(#2*#3*#4) & @@@ $ImportCache[[All, {6, 5, 2, 4}]]
        ]]],-1]
  ];
]

End[] (*`Private`*)

EndPackage[] (*`ImportCache` *)

