(* ::Package:: *)

(* :Name: DocumentationLinkTools` *)

(* :CVS Keywords:  $Id: DocumentationLinkTools.m,v 1.17 2010/07/20 18:24:56 jayw Exp $ *)

(* :Title: Link and Path Configuration Utilities for Documentation Creation Palettes  *)

(* :Author: Andre Kuzniarek *)

(* :Copyright: (c) 2005-07, Wolfram Research, Inc. All rights reserved. *)

(* :Mathematica Version: 6.0 *)

(* :Package Version: 0.01 *)

(* :Summary: Link and Path Configuration Utilities for Documentation Creation Palettes. *)



BeginPackage[ "DocumentationTools`"]

setTaggingRulesOption::usage = "Two arg form sets tagging rules in a notebook. One arg form applies selected arguments to $FrontEnd's TaggingRules.";
getTaggingRulesOption::usage = "Two arg form gets tagging rules in a notebook. One arg form lists selected arguments to $FrontEnd's TaggingRules.";
NextNotebook::usage = "Gets the next notebook not having WindowFrame -> \"Palette\" or the messages notebook.";

ExpandToCell::usage = "Does repeated selection expansions from a cursor selection or position inside a cell until the first cell bracket is reached.";

$DocuToolsDir::usage = "Installation directory for the DocumentationTools package.";

$DocuToolsDir::notfound =
"WARNING: The DocuTools installation directory was not found.
Please set the paramater $DocuToolsDir to point to the installation
directory; otherwise, certain features of DocuTools may not work as
expected.";

SetDocuToolsParametersInFEInit::usage = "Utility function for setting DocuTools variables as strings in a \"DocuToolsSettings\" tagging rule in the FE init.";
SetDocuToolsParameters::usage = "Utility function for setting DocuTools variables as strings when starting Mathematica from Workbench.";
DocuToolsParameters::usage = "Utility function that gives values of 5 key DocuTools parameters as rules.";

CellInfo::usage = "Utility function for returning CellInformation packet data.";
dlMessageToConsole::usage = "Utility function for rerouting messages to the console.";
DocuToolsLoader::usage = "Loads DocuTools.m with Get[], if not already loaded. DocuTools.m redefines this to null to so that it can be called in buttons to allow for lazy loading.";
CustomMenusApply::usage = "Updates menus with custom menu additions, stored as MenuSetup.m in FrontEnd/TextResources/PLATFORM. Restores default menus when applied a second time.";

$MVersion::usage = "Sets the version number of Mathematica being documented and appears in New in field of new function pages.";
$ShowPacletVariablesPrompt::usage = "Symbol to specify if the Paclet Variables prompt should be set to be permanently closed.";

PalettePackageDirectory::usage = "Returns path of palette installation directory.";

DocumentationDirSelect::usage = "Browse for directory to set as top level of documentation source tree and store setting in $UserBaseDirectory/DocuToolsPreferences.";
DocumentatioDirConfiguration::usage = "Intialize documentation paths, using base path in $UserBaseDirectory/DocuToolsPreferences, or default value (if user is not a special user), or special user-specific paths.";
$ApplicationName::usage = "The application name corresponding to a package's primary context.";
$LinkBase::usage = "Base of paclet expression in button data for applications and is an application's top directory name. It is usually the same as $ApplicationName but not always, allowing for flexibility in paclet URIs.";
$DocumentationPath::usage = "Path variable for tracking possible locations of documentation files.";
$DocumentationDirectory::usage = "Directory variable for documentation system source files.";
$ApplicationDirectory::usage = "Directory variable for application pack location.";
$MathematicaDocs::usage = "Boolean for determining if docs being worked on are for Mathematica or a paclet (only Mathematica uses System subdir for docs).";
$ReferenceDirectory::usage = "Directory variable for configuring any other pth variables for reference pages.";
$FunctionDirectory::usage = "Directory variable for function pages, called by various tools and style sheet definitions.";
$GuideDirectory::usage = "Directory variable for guide pages, called by various tools and style sheet definitions.";
$HowToDirectory::usage = "Directory variable for how to pages, called by various tools and style sheet definitions.";
$RawGuideDirectory::usage = "Directory variable for autogenerated guide pages in raw reference directory.";
$TutorialDirectory::usage = "Directory variable for tutorial pages, called by various tools and style sheet definitions.";
$CharacterDirectory::usage = "Directory variable for character pages, called by various tools and style sheet definitions.";
$FormatDirectory::usage = "Directory variable for format pages, called by various tools and style sheet definitions.";
$LinksDirectory::usage = "Directory variable for Links packages, called by various tools and style sheet definitions.";
$ExampleDirectory::usage = "Directory variable for example pages, called by the ExampleOpen function and button in function page palette.";
$MathWorldDirectory::usage = "Directory variable for MathWorld, called by various tools and style sheet definitions.";
$FunctionsSiteDirectory::usage = "Directory variable for functions.wolfram.com, called by various tools and style sheet definitions.";
$NKSDirectory::usage = "Directory variable for www.wolframscience.com, called by various tools and style sheet definitions.";
$PackageDirectory::usage = "Directory variable for packages, called by various tools and style sheet definitions.";
$NoteDirectory::usage = "Directory variable for notes, called by various tools and style sheet definitions.";

DocLink::usage = "ButtonFunction for symbol, paclet, or path-specific hyperlinks. Takes button contents and button data as arguments. \
Does the right thing with path variable set to URL[...] or any other similar function. Assumes function link if button data is not present.";
NETDocLink::usage = "ButtonFunction used to open the NETLink help file to specific pages.";

AddPacletDirectoryMapping::usage = "AddPacletDirectoryMapping[ paclet, pacletDirectory] adds a mapping to give the paclet directory for a given paclet";

OldRow::usage = "A utility for constructing rows in grids.";
LoadApplication::usage = "Button function of PacletVariablesPromptOpen for loading the package given by $LinkBase.";
$Old$Path::usage = "Global variable used by LoadApplication.";
$ApplicationLoadPalette::usage = "Progress monitor used by LoadApplication.";
$PresentBeginTime::usage = "Global variable used by LoadApplication.";
PacletVariablesPromptOpen::usage = "PacletVariablesPromptOpen[] generates a dialog that displays the current values of $ApplicationName and $LinkBase and generates a dynamic monitor of these variables from one of its buttons. The dialog only gets generated if $MathematicaDocs === False.";
CurrentPacletMonitorOpen::usage = "Generates a notebook that monitors the current values of $ApplicationName and $LinkBase.";
PreferencesCheck::usage = "If DocuTools has not been called from Workbench and the preferences set for DocuTools variables have not been set, a message is displayed informing the user that preferences are not stored in a different manner and will need to be reset. Otherwise PacletVariablesPromptOpen[] is called.";
PacletVariablesPromptRestore::usage = "PacletVariablesPromptRestore[] restores \"$ShowPacletVariablesPrompt\" to \"True\" in the FE init.";

$AnnotationText::usage = "Global variable used by the Annotation Dialog.";
$AnnotationDialogNotebook = "NotebookObject of the Annotation Dialog.";
GenerateAnnotationDialog::usage = 
"When GenerateAnnotationDialog[] is called from an annotation button in a cell, the Annotation dialog is generated with its input field \
containing the annotation. When called from AnnotationInsert, the Annotation dialog opens with the annotation from the cell containing \
the cursor if there is an annotation allowing annotation editing. The dialog allows the insertion of an annotation if there is none.";
NotebookToActOn::usage = "Option of GenerateAnnotationDialog.";
CellIDToActOn::usage = "Option of GenerateAnnotationDialog.";
AnnotationModify::usage = "AnnotationModify[] is the button function of the OK button of the Annotation dialog. It modifies an annotation in a cell with cell ID (specified by the tagging rule \"AnnotationCellID\" in the dialog) in the notebook specified in dialog by the tagging rule \"AnnotationNotebook\".";
AnnotationInsert::usage = 
  "AnnotationInsert[] is intended to be called from a palette button \
with the cursor inside a cell or at its cell bracket. The annotation \
dialog is opened allowing the insertion of an annotation in a cell if \
there is none and modification of an existing annotation if there is \
one in the cell.";
AnnotationRemove::usage = "A button function that removes an annotation from a cell.";
CalledFromFrameLabel::usage = "Option of AnnotationRemove.";
AnnotationSearch::usage = "AnnotationSearch[dir] searches in the direction dir, which can be \"Down\" or \"Up\", for annotation cells in the input notebook.";

Begin["`Private`"]

(* ExpandToCell does repeated selection expansions from a cursor selection or position inside a cell until the first cell
   bracket is reached. *)

ExpandToCell[x_NotebookObject] := Module[{lnkre},
  While[(LinkWrite[$ParentLink, FrontEnd`CellInformation[x]]; lnkre = LinkRead[$ParentLink]);
   (lnkre =!= $Failed && Not[MemberQ["CursorPosition" /. lnkre, "CellBracket"]]), 
   FrontEndExecute[FrontEnd`SelectionMove[x, All, Cell, AutoScroll -> False]]]]

(*
  Automatically determine $DocuToolsDir, if possible.
*)
If[$VersionNumber < 7, FindFile = System`Private`FindFile];

$DocuToolsDir =
  Quiet @ Check[
    DirectoryName[FindFile["DocumentationTools`"], 2],
    MessageToConsole[$DocuToolsDir::notfound]; $Failed
  ];

getTaggingRulesOption[opt_] := 
   Replace[ Options[$FrontEnd, TaggingRules], {
      { r_[ TaggingRules,
         {opts1___, (Rule|RuleDelayed)[opt, val_], opts2___} ] } :>
         val,
      { r_[ TaggingRules, (Rule|RuleDelayed)[opt, val_] ] } :>
         val,
      {r_[TaggingRules, rh_]} -> None,
      other_ -> $Failed
   }]
   
getTaggingRulesOption[nb_, opt_] := 
 Replace[Options[nb, TaggingRules], 
         {{r_[TaggingRules, {opts1___, (Rule | RuleDelayed)[opt, val_], opts2___}]} :> val, 
          {r_[TaggingRules, (Rule | RuleDelayed)[opt, val_]]} :> val, 
          {r_[TaggingRules, rh_]} -> None, other_ -> $Failed}]

setTaggingRulesOption[newopt:(opt_ -> val_)] :=
   Replace[ Options[$FrontEnd, TaggingRules], {
      { r_[ TaggingRules, tr_] } :>
         SetOptions[ $FrontEnd, TaggingRules ->
            Append[
               Replace[ tr, {
                  None -> {},
                  (Rule|RuleDelayed)[opt, oldval_] -> {},
                  l_List :> DeleteCases[ l, (Rule|RuleDelayed)[opt, oldval_] ],
                  nonList_ :> { nonList }
               }],
               newopt
            ]
         ],
      other_ -> $Failed
   }]
   
setTaggingRulesOption[nb_, newopt : (opt_ -> val_)] := 
 Replace[Options[nb, TaggingRules], {{r_[TaggingRules, tr_]} :> 
    SetOptions[nb, TaggingRules -> 
      Append[Replace[tr, {None -> {}, (Rule | RuleDelayed)[opt, oldval_] -> {}, 
                          l_List :> DeleteCases[l, (Rule | RuleDelayed)[opt, oldval_]], 
                          nonList_ :> {nonList}}], newopt]], other_ -> $Failed}]
   
NextNotebook[] := 
 Module[{i = 1, nbs = Notebooks[]}, 
  While[i <= Length[nbs] && (AbsoluteOptions[nbs[[i]], WindowTitle] === {WindowTitle -> "Messages"} || 
      Options[nbs[[i]], WindowFrame] === {WindowFrame -> "Palette"}), i++]; 
  If[i > Length[nbs], None, nbs[[i]]]]
  
   
ApplicationNameCorrespondingToLinkBase[linkbase_] := 
 If[# === {}, "", StringDrop[#[[1]], -1]] &@
  Cases[Block[{$Packages = {}}, 
    Quiet@Needs[linkbase <> "`"]; $Packages], 
   x_ /; StringCount[x, "`"] === 1 && StringMatchQ[x, __ ~~ "`"]]                    

SetDocuToolsParametersInFEInit[rulelist : {(_String -> _String) ..}] :=
  Module[{taggingrules = Options[$FrontEnd, TaggingRules]},
         Which[taggingrules === {TaggingRules -> None},
               SetOptions[$FrontEnd, TaggingRules -> {"DocuToolsSettings" -> rulelist}],
               Cases[TaggingRules /. taggingrules, a : ("DocuToolsSettings" -> _)] === {},
               SetOptions[$FrontEnd, TaggingRules -> Append[TaggingRules /. taggingrules, "DocuToolsSettings" -> rulelist]],
               True,
               SetOptions[$FrontEnd, TaggingRules -> ((TaggingRules /. taggingrules) /. ("DocuToolsSettings" -> e_) :> ("DocuToolsSettings" -> 
                                                                          Join[DeleteCases[e, Alternatives @@ (#[[1]] -> _ & /@ rulelist)], rulelist]))]]]
ConfigureDocuToolsVariables[] := 
 Module[{taggingrules = Options[$FrontEnd, TaggingRules], gt}, 
        If[(* "DocuToolsSettings" exists as a tagging rule in the FE init.m *)
           taggingrules =!= {TaggingRules -> None} &&
            ("DocuToolsSettings" /. (TaggingRules /. taggingrules)) =!= "DocuToolsSettings",
           
           gt = getTaggingRulesOption["DocuToolsSettings"];
           If[("$ApplicationName" /. gt) =!= "$ApplicationName", $ApplicationName = ("$ApplicationName" /. gt)]; 
           If[("$LinkBase" /. gt) =!= "$LinkBase", $LinkBase = ("$LinkBase" /. gt)]; 
           If[("$ApplicationDirectory" /. gt) =!= "$ApplicationDirectory", $ApplicationDirectory = ("$ApplicationDirectory" /. gt)]; 
           If[("$DocumentationDirectory" /. gt) =!= "$DocumentationDirectory", $DocumentationDirectory = ("$DocumentationDirectory" /. gt)]; 
           If[("$DocumentationPath" /. gt) =!= "$DocumentationPath", $DocumentationPath = ("$DocumentationPath" /. gt)]]]

If[Not[ValueQ[DocumentationTools`Utilities`$Workbench]] && $Notebooks, ConfigureDocuToolsVariables[]]


(* When starting Mathematica in Workbench SetDocuToolsParameters needs to be called at startup setting 4 parameters:
   "$ApplicationName", "$LinkBase", "$DocumentationDirectory", "$ApplicationDirectory". 

   Example:

   SetDocuToolsParameters["$ApplicationName" -> "NumericalAnalysis", 
                          "$LinkBase" -> "NumericalAnalysis", 
  "$DocumentationDirectory" -> "C:\\Documents and Settings\\jayw\\WolframWorkspaces\\Base\\NumericalAnalysis\\NumericalAnalysis\\Documentation\\English\\", 
  "$ApplicationDirectory" -> "C:\\Documents and Settings\\jayw\\WolframWorkspaces\\Base\\NumericalAnalysis\\NumericalAnalysis\\"]
 
 *)
                                                                          

Options[SetDocuToolsParameters] = {"$ApplicationName" -> "", "$LinkBase" -> "", "$DocumentationDirectory" -> "", 
                                   "$ApplicationDirectory" -> ""}
                                   
                                   
SetDocuToolsParameters[OptionsPattern[]] := (DocumentationTools`$ApplicationName = OptionValue["$ApplicationName"];
                                             DocumentationTools`$LinkBase = OptionValue["$LinkBase"];
                                             DocumentationTools`$DocumentationDirectory = OptionValue["$DocumentationDirectory"]; 
                                             DocumentationTools`$ApplicationDirectory = OptionValue["$ApplicationDirectory"];
                                             SetDocuToolsParametersInFEInit[{"$ApplicationName" -> DocumentationTools`$ApplicationName, 
                                                                             "$LinkBase" -> DocumentationTools`$LinkBase, 
                                                                             "$DocumentationDirectory" -> DocumentationTools`$DocumentationDirectory,
                                                                             "$ApplicationDirectory" -> DocumentationTools`$ApplicationDirectory}])                                                                             

DocuToolsParameters[] := {"$ApplicationName" -> DocumentationTools`$ApplicationName, "$LinkBase" -> DocumentationTools`$LinkBase, 
                          "$DocumentationDirectory" -> DocumentationTools`$DocumentationDirectory, "$ApplicationDirectory" -> DocumentationTools`$ApplicationDirectory}
                          


CellInfo[ selNB_NotebookObject] := MathLink`CallFrontEnd[ FrontEnd`CellInformation[ selNB]]



(* Read preferences, or write preferences file if none exists *)
(*
If[StringMatchQ[ToString@FileNames["*", $UserBaseDirectory], "*DocuToolsPrefs*"],
  Get[ToFileName[{$UserBaseDirectory}, "DocuToolsPrefs"]],
  Put[DateString[], ToFileName[{$UserBaseDirectory}, "DocuToolsPrefs"]]]
*)

dlOldNotebookRead[nb_]:= NotebookRead[nb, "WrapBoxesWithBoxData" -> True]


SetAttributes[ dlMessageToConsole, HoldFirst]

dlMessageToConsole[symbolWithValue_, values___] :=
 Module[{presetMessageOptionsValues, cs},
        (* Get current MessageOptions to restore after message is sent to console. *)
        presetMessageOptionsValues = MessageOptions /. Options[$FrontEnd, MessageOptions];
        
        (* New MessageOptions has "KernelMessageAction" with "PrintToConsole". *)
        newMessageOptionsValues = If[(cs = Cases[presetMessageOptionsValues, a : ("KernelMessageAction" -> _)]; cs) === {}, 
                                  Append[presetMessageOptionsValues, "KernelMessageAction" -> {"Beep", "PrintToNotebook"}],
        presetMessageOptionsValues /. ("KernelMessageAction" -> a_) :> ("KernelMessageAction" -> 
                  If[StringQ[a],"PrintToConsole", Append[DeleteCases[a,"PrintToNotebook"],"PrintToConsole"]])]; 
        SetOptions[$FrontEnd, MessageOptions -> newMessageOptionsValues]; 
        Message[symbolWithValue, values];
        SetSelectedNotebook[MessagesNotebook[]];
        (* Restore previous MessageOptions. *)
        SetOptions[$FrontEnd, MessageOptions -> presetMessageOptionsValues]]
        
dlSpecialCharacterQ[x_String] := StringMatchQ[x, "\"\\[" ~~ a__?(Not@MemberQ[{"[", "]"}, #] &) ~~ "]\""]

dlMultipleCellBracketsSelected[x_] := MatchQ[x,{{"Style"->_,__},{"Style"->_,__},___}]
 

$AppDir = DirectoryName @ System`Private`$InputFileName

DocumentationTools`Private`$LoadedDocuTools = "LinkTools"

DocuToolsLoader[] := If[
    DocumentationTools`Private`$LoadedDocuTools === "LinkTools",
      Get[ ToFileName[ DocumentationTools`Private`$AppDir, "DocumentationTools.m"]] (* $AppDir set by CustomMenusApply *)
    ]

(* The FE group fixed the bug where the shortcuts would not work when using AddMenuCommands.
   MenuSetup.m is now simply the list of menu items to be added. Using System`Private`$InputFileName 
   to establish path in case package is loaded in alternate locations. 
   Note that DirectoryName is applied twice to $InputFileName if CustomMenusApply is invoked from init.m *)

CustomMenusApply[] /; (Head[ $FrontEnd] === FrontEndObject) :=
    Module[{menuFile},
      menuFile = ToFileName[{ DocumentationTools`Private`$AppDir, "FrontEnd", "TextResources",
                   Switch[ $OperatingSystem,
                     "MacOSX", "Macintosh",
                     "Unix",   "X",
                     _,        "Windows"]}, "MenuSetup.m"];
      MathLink`CallFrontEnd@FrontEnd`ResetMenusPacket[{Automatic}];
      Get[ menuFile]
      ]

(* This does not work completely within Applications -- shortcuts don't show up,
   though it works in Autoload just fine. The mind boggles: 
   
   If[ $VersionNumber >= 6, CustomMenusApply[]] *)


$MVersion := If[ StringMatchQ[ #, "*."], StringDrop[#, -1], #] &[ ToString @ $VersionNumber]

PalettePackageDirectory[] := 
   StringReplace[
     DirectoryName[
       ToFileName["FileName" /. NotebookInformation[ ButtonNotebook[]]]], 
         RegularExpression[ 
           "FrontEnd" <> 
           If[$OperatingSystem === "Windows", 
             $PathnameSeparator <> $PathnameSeparator,
             $PathnameSeparator] 
           <> ".+"] :> ""]
           
           
PalettePackageDirectory[ pathAddition_String] := 
   StringReplace[
     DirectoryName[
       ToFileName["FileName" /. NotebookInformation[ ButtonNotebook[]]]], 
         RegularExpression[ 
           "FrontEnd" <> 
           If[$OperatingSystem === "Windows", 
             $PathnameSeparator <> $PathnameSeparator,
             $PathnameSeparator] 
           <> ".+"] :> ""] <> pathAddition


(* Reconcile old variable name stored in prefs *)
If[ ValueQ[ $DocumentationPath] && !ValueQ[ $DocumentationDirectory],
  $DocumentationDirectory = $DocumentationPath]
(* End reconcile *)

DocumentationDirSelect::nonstandardDirSel = "The application directory you selected does not contain a documentation layout of the form: Documentation"<>$PathnameSeparator<>$Language;

Options[DocumentationDirSelect] = {"Interactive" -> False}

DocumentationDirSelect[ ] := 
      Which[ 
        ValueQ[ $ApplicationDirectory] && !ValueQ[ DocumentationTools`Utilities`$DocumentationSourceDirectory (* Usually set by Workbench *)], 
          DocumentationDirSelect[ ToFileName[ $ApplicationDirectory], "Interactive" -> False],
        ValueQ[ DocumentationTools`Utilities`$DocumentationSourceDirectory], 
          DocumentationDirSelect[ ToFileName @ DocumentationTools`Utilities`$DocumentationSourceDirectory, "Interactive" -> False],
        True,
          DocumentationDirSelect[ ToFileName @ $HomeDirectory, "Interactive" -> True]
        ]

DocumentationDirSelect[ "Interactive" -> True] :=
      Which[ 
        ValueQ[ $ApplicationDirectory] && !ValueQ[ DocumentationTools`Utilities`$DocumentationSourceDirectory (* Usually set by Workbench *)], 
          DocumentationDirSelect[ ToFileName[ $ApplicationDirectory], "Interactive" -> True],
        ValueQ[ DocumentationTools`Utilities`$DocumentationSourceDirectory], 
          DocumentationDirSelect[ ToFileName @ DocumentationTools`Utilities`$DocumentationSourceDirectory, "Interactive" -> True],
        True,
          DocumentationDirSelect[ ToFileName @ $HomeDirectory, "Interactive" -> True]
        ]
        
DocumentationDirSelect[pagetype_String/;FileType[pagetype]===None, "Interactive" -> True] := (DocumentationDirSelect[ "Interactive" -> True]; DocumentationTools`CreateNewPageDialog[pagetype])
      
DocumentationDirSelect[ startDir_String/;FileType[startDir]===Directory, optsIn___] := 
	Module[{ dir, dirList, opts = {optsIn}}, 

  Which[ 
       (opts =!= {} && ("Interactive" /. opts) === True) || ("Interactive" /. Options[ DocumentationDirSelect]) === True, 
         dir = MathLink`CallFrontEnd[ FrontEnd`DirectoryBrowse[ startDir, "Select the application containing your working documentation layout:"]], (* SystemDialogInput does not work from package *)
       (opts =!= {} && ("Interactive" /. opts) === False) || ("Interactive" /. Options[ DocumentationDirSelect]) === False, 
         dir = ToFileName[ startDir],
       True,
         dir = ToFileName[ startDir]
       ];
     Catch[
       (If[ dir === Null, 
           Throw[ Abort[]],
           dirList = StringSplit[ dir, $PathnameSeparator]];
       Which[ 
         MatchQ[ dirList, {___, "Documentation", $Language}],
           $DocumentationDirectory = dir;
           $ApplicationDirectory = DirectoryName @ DirectoryName @ dir,
         MatchQ[ dirList, {___, $Language}],
           $DocumentationDirectory = dir;
           $ApplicationDirectory = DirectoryName @ dir,
         MatchQ[ dirList, {___, "Documentation"}],
           $DocumentationDirectory = ToFileName[ dir <> $Language];
           $ApplicationDirectory = DirectoryName @ dir,
         {} =!= Quiet[ FileNames[ $Language, dir, 1]],  (* Using Quiet for bug 92226 *)
           $DocumentationDirectory = ToFileName[ {dir, $Language}];
           $ApplicationDirectory = DirectoryName @ dir,
         {} =!= Quiet[ FileNames[ $Language, dir, 2]],
           $DocumentationDirectory = ToFileName[ {dir, "Documentation", $Language}];
           $ApplicationDirectory = dir,
         {} =!= Quiet[ FileNames[ $Language, dir, 3]] 
         && Length @ Position[ Flatten @ StringSplit[ Quiet[ FileNames[ $Language, dir, 3]], $PathnameSeparator], Last[ StringSplit[ dir, $PathnameSeparator]]] == 2,
           $DocumentationDirectory = ToFileName[ {dir, Last[ StringSplit[ dir, $PathnameSeparator]], "Documentation", $Language}];
           $ApplicationDirectory = ToFileName[ {dir, Last[ StringSplit[ dir, $PathnameSeparator]]}],           
         {} === Quiet[ FileNames[ $Language, dir, 2]],
           Throw[ 
             dlMessageToConsole[ DocumentationDirSelect::nonstandardDirSel]],
         True,
           $DocumentationDirectory = dir;
           $ApplicationDirectory = dir
         ]);
	 $LinkBase = Last[ StringSplit[ $ApplicationDirectory, $PathnameSeparator]];
	 (* Removed use of StringMatchQ[_, ""|Whitespace] in the following Which since it causes problems when PacletVariablesPromptOpen[] (which calls
	    DocumentationDirSelect) is in the kernel init.m and DocuTools is in Autoload. *)
     Which[ 
       ValueQ[ DocumentationTools`Utilities`$LinkBase] && StringQ[ DocumentationTools`Utilities`$LinkBase] && Not[DocumentationTools`Utilities`$LinkBase === "" || MatchQ[Characters[DocumentationTools`Utilities`$LinkBase], {" "..}]],
         $LinkBase = DocumentationTools`Utilities`$LinkBase; $ApplicationName = ApplicationNameCorrespondingToLinkBase[$LinkBase], (* DocuTools user is setting link base separately from application name *)
       ValueQ[ DocumentationTools`Utilities`$DocumentationSourceDirectory] && StringQ[ DocumentationTools`Utilities`$DocumentationSourceDirectory] && Not[DocumentationTools`Utilities`$DocumentationSourceDirectory === "" || MatchQ[Characters[DocumentationTools`Utilities`$DocumentationSourceDirectory], {" "..}]],
         Null, (* Workbench is setting $LinkBase *)
       ValueQ[ $LinkBase] && StringQ[$LinkBase] && Not[ $LinkBase === "" || MatchQ[Characters[ $LinkBase], {" " ..}]],
         $ApplicationName = ApplicationNameCorrespondingToLinkBase[$LinkBase], (* application name is set by $LinkBase *)
       True,
         $ApplicationName = $LinkBase]; (* Default fall through as application name is usually the same as link base *)
     
     SetDocuToolsParametersInFEInit[{"$ApplicationName" -> $ApplicationName, "$LinkBase" -> $LinkBase, 
                                     "$ApplicationDirectory" -> $ApplicationDirectory, "$DocumentationDirectory" -> $DocumentationDirectory}];
     
     DocumentationTools`DocumentationPathConfiguration[];
     If[ ValueQ[ DocumentationTools`Utilities`$DocumentationSourceDirectory], (* Reset $InitialDirectory if user browses to new location *)
        DocumentationTools`Utilities`$DocumentationSourceDirectory = $DocumentationDirectory]
     ]]
          


(* Variables for documentation file paths, used by special hyperlink styles/tools: *)

packagesMirrorTest := StringMatchQ[ ToString @ FileNames[ "*", $DocumentationDirectory], "*"<>$Language<>$PathnameSeparator<>"PackagesMirror*"] ||
                      StringMatchQ[ ToString @ FileNames[ "*", $DocumentationDirectory], "*"<>$PathnameSeparator<>"PackagesMirror*"]

$MathematicaDocs        := If[ StringMatchQ[ ToString @ FileNames[ "*", $DocumentationDirectory], "*"<>$PathnameSeparator<>"System*"] || 
                               StringMatchQ[ $DocumentationDirectory, "*Documentation"<>$PathnameSeparator<>"WolframNetwork*"],
                             True, False]
$ReferenceDirectory     := If[ $MathematicaDocs,
                             ToFileName[{ $DocumentationDirectory, "System", "ReferencePages"}],
                             ToFileName[{ $DocumentationDirectory, "ReferencePages"}]]
$FunctionDirectory      := ToFileName[{ $ReferenceDirectory, "Symbols"}]
$CharacterDirectory     := ToFileName[{ $ReferenceDirectory, "Characters"}]
$FormatDirectory        := ToFileName[{ $ReferenceDirectory, "Formats"}]
$GuideDirectory         := If[ $MathematicaDocs,
                             ToFileName[{ $DocumentationDirectory, "System", "Guides"}],
                             ToFileName[{ $DocumentationDirectory, "Guides"}]]
$HowToDirectory         := If[ $MathematicaDocs,
                             ToFileName[{ $DocumentationDirectory, "System", "HowTos"}],
                             ToFileName[{ $DocumentationDirectory, "HowTos"}]]
$TutorialDirectory      := If[ $MathematicaDocs,
                             ToFileName[{ $DocumentationDirectory, "System", "Tutorials"}],
                             ToFileName[{ $DocumentationDirectory, "Tutorials"}]]
$NoteDirectory          := If[ $MathematicaDocs,
                             ToFileName[{ $DocumentationDirectory, "System", "Notes"}],
                             ToFileName[{ $DocumentationDirectory, "Notes"}]]
$ExampleDirectory       := ToFileName[{ DirectoryName @ $DocumentationDirectory, "RawMaterial", "Examples"}]
$RawGuideDirectory      := ToFileName[{ DirectoryName @ $DocumentationDirectory, "RawMaterial", "RawGuides"}]
$PackageDirectory       := Which[ 
                             packagesMirrorTest,
                               ToFileName[{ $DocumentationDirectory, "PackagesMirror"}],
                             $DocumentationDirectory === ToFileName[{$TopDirectory, "Documentation", $Language}],
                               ToFileName[{ $InstallationDirectory, "Packages"}],
                             StringMatchQ[ ToString @ FileNames["*", $DocumentationDirectory], "*Documentation" <> $PathnameSeparator <> $Language <> "*"],
                               DirectoryName @ DirectoryName @ DirectoryName @ $DocumentationDirectory,
                             StringQ @ $DocumentationDirectory,
                               DirectoryName @ $DocumentationDirectory,
                             True,
                               ToFileName[{ $InstallationDirectory, "Packages"}]]
$LinksDirectory         := If[ packagesMirrorTest,
                             ToFileName[{$DocumentationDirectory, "PackagesMirror"}], 
                             ToFileName[{$InstallationDirectory, "SystemFiles", "Links"}]]
                        
$MathWorldDirectory     := URL["http://mathworld.wolfram.com/search/?q="]
$FunctionsSiteDirectory := URL["http://functions.wolfram.com/search/index.cgi?filter=1&q="]
$NKSDirectory           := URL["http://www.wolframscience.com/nksonline/search/?q="]

unsortedUnion[x_] := Tally[x][[All, 1]]

DocumentationPathConfiguration[] :=
 Module[{taggingrules = Options[$FrontEnd, TaggingRules], gt},
    If[ (* Clunky construction to avoid mysterious Throw errors from using Which, during startup when this function is called in init.m *)
        DocumentationTools`AuxiliaryPaths`$UserSpecial === True,
        ($DocumentationDirectory = DocumentationTools`AuxiliaryPaths`$SpecialPath),
     If[ (taggingrules === {TaggingRules -> None}) || 
           ("DocuToolsSettings" /. (TaggingRules /. taggingrules)) === "DocuToolsSettings" || 
             ("$DocumentationDirectory" /. ("DocuToolsSettings" /. (TaggingRules /. taggingrules))) === "$DocumentationDirectory", 
        $DocumentationDirectory = ToFileName[{$TopDirectory, "Documentation", $Language}]]];
(*   If[ !ListQ[ $DocumentationStoredPath], $DocumentationStoredPath = {}];  *)
    $DocumentationPath = unsortedUnion @ Join[
      {$DocumentationDirectory},
      {$LinksDirectory},
      {$PackageDirectory},
      (* Flatten @ {$ApplicationDirectory}, *)
      {
      ToFileName[{ $UserBaseDirectory, "Applications"}],
      ToFileName[{ $UserBaseDirectory, "Autoload"}],
      ToFileName[{ $BaseDirectory, "Applications"}],
      ToFileName[{ $BaseDirectory, "Autoload"}],
      ToFileName[{$TopDirectory}],
      ToFileName[{$TopDirectory, "AddOns", "Applications"}],
      ToFileName[{$TopDirectory, "Documentation", $Language, "Packages"}]
(*      }, $DocumentationStoredPath];   *)
        }, gt=getTaggingRulesOption["DocuToolsSettings"];
           If[gt =!= None && ("$DocumentationPath"/.gt)=!=$DocumentationPath && ListQ["$DocumentationPath"/.gt],("$DocumentationPath"/.gt),{}]];
(*    If[ $DocumentationStoredPath =!= $DocumentationPath,
      $DocumentationStoredPath = $DocumentationPath;
      Save[ ToFileName[ $UserBaseDirectory, "DocuToolsPrefs"], DocumentationTools`$DocumentationStoredPath]]   *)
     
    ]

If[$Notebooks, DocumentationPathConfiguration[]]


fixFileLoc[ str_String] := Which[
   StringMatchQ[ str, "applescript"],
     "AppleScript",
   StringMatchQ[ str, "c"],
     "C",
   StringMatchQ[ str, "frontendobject"],
     "FrontEndObjects",
   StringMatchQ[ str, "menuitem"],
     "MenuItems",
   StringMatchQ[ str, "howto"],
     "HowTos",
   True,
     ToUpperCase @ First @ Characters @ str <> StringDrop[ str, 1] <> "s"];

FirstIfList[ lis_] := 
  Which[
   !ListQ[lis] && StringQ[lis],
     lis,
   lis === {},
     {},
   True,
     First @ lis]

AddPacletDirectoryMapping[ pacletName_, pacletDir_] :=
	HoldPattern[lookupPacletDir[ pacletName]] = pacletDir

lookupPacletDir[ pacletName_] := pacletName

findFile[ pathSegment_List, fileName_String] := findFile[ "System", pathSegment, fileName]

findFile[ paclet_String, pathSegment_List, fileName_String] := 
	Module[ {names, pacletDir},
		pacletDir = lookupPacletDir[ paclet];
		names = 
          {
      		Flatten[ FileNames[ ToFileName[ Join[ {pacletDir}, pathSegment], fileName <> ".nb"], #, Infinity]& /@ $DocumentationPath],
      		Flatten[ FileNames[ ToFileName[ Join[ {pacletDir}, {"Documentation", $Language}, pathSegment], fileName <> ".nb"], #, Infinity]& /@ $DocumentationPath],
         (* The following seems pointless and gratuitous: *)
      		Flatten[ FileNames[ ToFileName[ Join[ If[DocumentationTools`AuxiliaryPaths`$UserSpecial === True, DocumentationTools`AuxiliaryPaths`$SpecialPartialPath, {"Documentation", $Language}], {pacletDir}, pathSegment], fileName <> ".nb"], #, Infinity]& /@ $DocumentationPath]
          };
  		NotebookLocate[{ FirstIfList @ Flatten @ Select[ names, # =!= {} &], None}]
	]

findFilePDF[ paclet_String, pathSegment_List, fileName_String] := 
  NotebookLocate[{ 
    URL[ If[$OperatingSystem === "Windows", #, "file://" <> #] & [
      FirstIfList @ Flatten @ Select[
        {
        FileNames[ ToFileName[ Join[ {paclet}, pathSegment], fileName], $DocumentationPath, Infinity],
        FileNames[ ToFileName[ Join[ {paclet}, {"Documentation", $Language}, pathSegment], fileName], $DocumentationPath, Infinity],
     (* The following seems pointless and gratuitous: *)
        FileNames[ ToFileName[ Join[ If[DocumentationTools`AuxiliaryPaths`$UserSpecial === True, DocumentationTools`AuxiliaryPaths`$SpecialPartialPath, {"Documentation", $Language}], {paclet}, pathSegment], fileName], $DocumentationPath, Infinity]
        }, # =!= {} &]
      ]], 
    None}]

findAnchor[ anchor_] := 
  If[ $Failed === NotebookFind[ InputNotebook[], anchor, Next, CellID],
    NotebookFind[ InputNotebook[], anchor, Next, CellTags]];
   

DocLink[ cont_, data_] := DocLink[ $FunctionDirectory, cont, data]

DocLink[ path_, cont_, data_] := Module[{ uriParts, packageQ, linkPackQ },

    Catch[
    Which[ 
      data === Automatic && MatchQ[ cont, BoxData[_String] | _String], 
        Throw @ NotebookLocate[{ path <> StringReplace[ cont /. { BoxData[ a_] -> a}, {".nb" -> ""}] <> ".nb", None}], (* path is only referenced here *)
      NumberQ[ data],
        Throw @ NotebookFind[ ButtonNotebook[], data, Next, CellID],
      ListQ[ data] && NumberQ[ Last @ data],
        Throw[
         (NotebookLocate[{ First @ data, None}]; (* using NotebookLocate because of new shift-click feature support being added in M6 *)
          NotebookFind[ InputNotebook[], Last @ data, Next, CellID])], (* unlike NotebookOpen, NotebookLocate does not return a notebook object, so we need to assign separately *)
      ListQ[ data],
        Throw @ NotebookLocate[ data],
      StringQ[ data] (* pass through to URI resolution process *),
        Null,
      True,
        Throw[ Null] 
      ];
    
    uriParts = StringSplit[ StringReplace[ data, {"paclet:" -> ""}], {"/", "#"}];
   
    If[ !LowerCaseQ @ First @ uriParts || StringMatchQ[ First @ uriParts, "webM*"] && !StringMatchQ[ First @ uriParts, "Raw*"],
      packageQ = True,
      packageQ = False];
      
    If[ packageQ &&  StringMatchQ[ First @ uriParts, "*Link"], (* Might not actually need this *)
      linkPackQ = True,
      linkPackQ = False];

 (* URI resolution process: *)
    Which[

   (* Hack alert: just doing this for convenience, generalize later: *)
      StringMatchQ[ data, "paclet://DocuTools/Tutorials/*"],
        NotebookLocate[{ ToFileName[{ PalettePackagePath[], "Documentation", "English", "Tutorials"}, StringReplace[ data, "paclet://DocumentationTools/Tutorials/"->""] <> ".nb"], None}],
      StringMatchQ[ data, "paclet://DocuTools/Guides/*"],
        NotebookLocate[{ ToFileName[{ PalettePackagePath[], "Documentation", "English", "Guides"}, StringReplace[ data, "paclet://DocumentationTools/Guides/"->""] <> ".nb"], None}],
      StringMatchQ[ data, "paclet://DemoTools/Tutorials/*"],
        NotebookLocate[{ ToFileName[{ PalettePackagePath[], "Documentation", "English", "Tutorials"}, StringReplace[ data, "paclet://DemoTools/Tutorials/"->""] <> ".nb"], None}],
      StringMatchQ[ data, "paclet://DemoTools/Guides/*"],
        NotebookLocate[{ ToFileName[{ PalettePackagePath[], "Documentation", "English", "Guides"}, StringReplace[ data, "paclet://DemoTools/Guides/"->""] <> ".nb"], None}],
   (* End hack alert *)
         
   (* Recently added PDF handling by request: *)
   
      StringMatchQ[ Last@uriParts, "*.pdf"],
        Which[
          packageQ && Length[ uriParts] == 3 && uriParts[[2]] =!= "ref", (* package guide or tutorial or other *)
            findFilePDF[ First@uriParts, {fixFileLoc[uriParts[[2]]]}, Last@uriParts],
          packageQ && Length[ uriParts] == 4 && uriParts[[2]] =!= "ref", (* package other content type *)
            findFilePDF[ First@uriParts, {fixFileLoc[uriParts[[2]]], uriParts[[3]]}, Last@uriParts]
          ],
   
   (* End PDF handling *)


      packageQ && Length[ uriParts] == 3 && uriParts[[2]] === "ref", (* package symbol *)
        findFile[ First@uriParts, {"ReferencePages", "Symbols"}, Last@uriParts],
        
      packageQ && Length[ uriParts] == 3 && uriParts[[2]] =!= "ref", (* package guide or tutorial or other *)
        findFile[ First@uriParts, {fixFileLoc[uriParts[[2]]]}, Last@uriParts],
 
      packageQ && Length[ uriParts] == 4 && uriParts[[2]] === "ref" && StringMatchQ[ data, "*#*"], (* package symbol w/anchor *)
        findFile[ First@uriParts, {"ReferencePages", "Symbols"}, uriParts[[3]]];
        findAnchor[ Last @ uriParts],
 
      packageQ && Length[ uriParts] == 4 && uriParts[[2]] === "ref", (* package other symbol*)
        findFile[ First@uriParts, {"ReferencePages"}, Last@uriParts],
 
      packageQ && Length[ uriParts] == 4 && uriParts[[2]] =!= "ref" && StringMatchQ[ data, "*#*"], (* package guide or tutorial w/anchor *)
        findFile[ First@uriParts, {fixFileLoc[uriParts[[2]]]}, uriParts[[3]]];
        findAnchor[ Last @ uriParts],
      
      packageQ && Length[ uriParts] == 4 && uriParts[[2]] =!= "ref", (* package other content type *)
        findFile[ First@uriParts, {fixFileLoc[uriParts[[2]]], uriParts[[3]]}, Last@uriParts],
      
      packageQ && Length[ uriParts] == 5 && uriParts[[2]] === "ref",(* package other symbol w/anchor *)
        findFile[ First@uriParts, {"ReferencePages", fixFileLoc[ uriParts[[3]]]}, uriParts[[4]]];
        findAnchor[ Last @ uriParts],
 
      packageQ && Length[ uriParts] == 5 && uriParts[[2]] =!= "ref" && StringMatchQ[ data, "*#*"], (* package other content type w/anchor *)
        findFile[ First@uriParts, {fixFileLoc[uriParts[[2]]], uriParts[[3]]}, uriParts[[4]]];
        findAnchor[ Last @ uriParts],
 
      packageQ && Length[ uriParts] == 5 && uriParts[[2]] =!= "ref", (* package other content with unconventional layout *)
        findFile[ First@uriParts, {fixFileLoc[uriParts[[2]]], uriParts[[3]], uriParts[[4]]},  Last @ uriParts],

      Length[ uriParts] == 2 && First @ uriParts === "RawGuides", (* lingering raw material *)
        NotebookLocate[{ToFileName[{ $RawGuideDirectory}, Last @ uriParts], None}],

      Length[ uriParts] == 2 && (First @ uriParts) === "ref", (* system symbol *)
        findFile[ {"ReferencePages", "Symbols"}, Last@uriParts],

      Length[ uriParts] == 2, (* system guide or tutorial *)
        findFile[ {fixFileLoc[ First @ uriParts]}, Last@uriParts],
 
      Length[ uriParts] == 3 && First @ uriParts === "RawGuides", (* lingering raw material *)
        NotebookLocate[{ToFileName[{ $RawGuideDirectory, uriParts[[2]]}, Last @ uriParts], None}],

      Length[ uriParts] == 3 && (First @ uriParts) === "ref" && !LowerCaseQ[ uriParts[[2]]],  (* system symbol w/anchor *)
        findFile[ {"ReferencePages", "Symbols"}, uriParts[[2]]];
        findAnchor[ Last @ uriParts],
 
      Length[ uriParts] == 3 && (First @ uriParts) === "ref", (* system other symbol *)
        findFile[ {"ReferencePages", fixFileLoc[ uriParts[[2]]]}, Last @ uriParts],
      
      Length[ uriParts] == 4 && (First@uriParts) === "ref",  (* system other symbol w/anchor *)
        findFile[ {"ReferencePages", fixFileLoc[ uriParts[[2]]]}, uriParts[[3]]];
        findAnchor[ Last @ uriParts],
 
      Length[ uriParts] == 3 && LowerCaseQ@First@uriParts, (* system guide or tutorial w/anchor *)
        findFile[ {fixFileLoc[ First @ uriParts]},uriParts[[2]]];
        findAnchor[ Last @ uriParts],
        
   (* End of URI resolution process *)

   (* This extends hyperlink support to just filenames *)
      StringMatchQ[ data, RegularExpression[".+" <> If[ $OperatingSystem === "Windows", "\\\\", "/"] <> ".+"]],  
        NotebookLocate[{ data, None}],
      StringMatchQ[ data, "*.nb"],  
        NotebookLocate[{ ToFileName[{ Directory[]}, data], None}],

      True,
        NotebookLocate[ data]]
   ]]



DocLink::noin = "There is no input notebook.";
DocLink::betwcells = "The cursor is between cells or not inside an input notebook.";
DocLink::mulcell = "Multiple cells have been selected.";
DocLink::cellbrac = "A cell bracket is selected.";
DocLink::nstruc = "This structure cannot be handled by DocLink.";


(* This selects the word within range of the cursor and assumes it is a function name, 
   and opens the corresponding function page: *)

DocLink[ "FunctionSelect"] := Module[{nb = InputNotebook[], ci, cs, sel, funcName},
   Catch[
     If[ nb === $Failed, (* There is no input notebook. *)
       Throw[ dlMessageToConsole[ DocLink::noin]]];
     ci = CellInfo[ nb];
     If[ ci === $Failed, (*The cursor is between cells.*)
       Throw[ dlMessageToConsole[ DocLink::betwcells]]];
     If[ dlMultipleCellBracketsSelected[ ci], 
       Throw[ dlMessageToConsole[ DocLink::mulcell]]];
     If[("CursorPosition" /. ci) === {"CellBracket"}, 
       Throw[ dlMessageToConsole[ DocLink::cellbrac]]];
     If[ dlOldNotebookRead[ nb] === {}, 
       SelectionMove[ nb, All, Word]]; 
     sel = dlOldNotebookRead[nb];
     If[(cs = Cases[sel, ButtonBox[__], Infinity]) =!= {}, 
        funcName = StringReplace[cs[[1, 1]], " " -> ""], 
        If[MatchQ[#, _String] && # =!= "", funcName = StringReplace[#, " " -> ""], Abort[]] &[
         StringJoin[Cases[{sel /. StyleBox[cont_, ___] -> cont}, _String, Infinity]]]];
     DocumentationTools`DocLink[ DocumentationTools`$FunctionDirectory, funcName, Automatic]
     ]]
     

DocLink[ "CharacterSelect"] := 
 Module[{ nb = InputNotebook[], ci, sel, stringpart, selFix, charName}, 
      Catch[
        If[ nb === $Failed, (* There is no input notebook. *)
          Throw[ dlMessageToConsole[ DocLink::noin]]];
        ci = CellInfo[ nb];
        If[ ci === $Failed, (*The cursor is between cells.*)
          Throw[ dlMessageToConsole[ DocLink::betwcells]]];
        If[ dlMultipleCellBracketsSelected[ ci], 
          Throw[ dlMessageToConsole[ DocLink::mulcell]]];
        If[("CursorPosition" /. ci) === {"CellBracket"}, 
          Throw[ dlMessageToConsole[ DocLink::cellbrac]]];, 
        If[ dlOldNotebookRead[ nb] === {}, 
          SelectionMove[ nb, All, Word]]; 
        sel = dlOldNotebookRead[ nb];
        stringpart = Switch[ sel, 
                             _String, sel, 
                             BoxData[_String], sel[[1]], 
                             StyleBox[_String, __], sel[[1]],
                             BoxData[StyleBox[_String, __]], sel[[1, 1]], 
                             ButtonBox[_String, __], sel[[1]], 
                             _, Throw[
                                  dlMessageToConsole[ DocLink::nstruc];
                                  FrontEndExecute[ FrontEnd`FrontEndToken[nb, "MoveNext"]]]];
         selFix = ToString[ stringpart, InputForm, CharacterEncoding -> None]; 
         charName = If[ dlSpecialCharacterQ[ selFix], 
                      StringReplace[ StringTake[ ToString[ selFix, InputForm, CharacterEncoding -> None], {7, -5}], " " -> ""], 
                            StringReplace[selFix, {" " -> "", "\"" -> ""}]]; 
         DocumentationTools`DocLink[ DocumentationTools`$CharacterDirectory, charName, Automatic]
         ]]


DocLink[ "FormatSelect"] := Module[{nb = InputNotebook[], ci, cs, sel, funcName},
   Catch[
     If[ nb === $Failed, (* There is no input notebook. *)
       Throw[ dlMessageToConsole[ DocLink::noin]]];
     ci = CellInfo[ nb];
     If[ ci === $Failed, (*The cursor is between cells.*)
       Throw[ dlMessageToConsole[ DocLink::betwcells]]];
     If[ dlMultipleCellBracketsSelected[ ci], 
       Throw[ dlMessageToConsole[ DocLink::mulcell]]];
     If[("CursorPosition" /. ci) === {"CellBracket"}, 
       Throw[ dlMessageToConsole[ DocLink::cellbrac]]];, 
     If[ dlOldNotebookRead[ nb] === {}, 
       SelectionMove[ nb, All, Word]]; 
     sel = dlOldNotebookRead[nb];
     If[(cs = Cases[sel, ButtonBox[__], Infinity]) =!= {}, 
        funcName = StringReplace[cs[[1, 1]], " " -> ""], 
        If[MatchQ[#, _String] && # =!= "", funcName = StringReplace[#, " " -> ""], Abort[]] &[
         StringJoin[Cases[{sel /. StyleBox[cont_, ___] -> cont}, _String, Infinity]]]];
     DocumentationTools`DocLink[ DocumentationTools`$FormatDirectory, funcName, Automatic]
     ]]


$SymbolName = ""

DocLink[ "FunctionField"] := 
    If[ DocumentationTools`$SymbolName =!= "",
      (DocLink[ BoxData[ DocumentationTools`$SymbolName], Automatic];
       DocumentationTools`$SymbolName = ""),
       DocLink[ "FunctionSelect"]
      ]

DocLink[ "FunctionDialog"] := Module[{ funcDialog },
    DocumentationTools`$SymbolName = "xxxx";
    funcDialog = NotebookPut @
      Notebook[{
        Cell[ BoxData[
          InputFieldBox[ 
            Dynamic[ DocumentationTools`$SymbolName], 
            String, FieldSize -> {58, {1, Infinity}}, ContinuousAction->True]], 
          CellMargins -> {{10, 10}, {10, 10}}]}, 
        ShowCellBracket->False,
        WindowSize -> {400, 50}, 
        WindowMargins -> Automatic,
        WindowTitle -> "Open Function Page",
        WindowFrame -> "Palette",
        WindowElements -> {"CloseBox"},
        WindowFrameElements -> {"CloseBox"},
        Saveable -> False,
        NotebookEventActions -> {"ReturnKeyDown" :> CompoundExpression[
                                                      DocumentationTools`DocLink[ DocumentationTools`$FunctionDirectory, DocumentationTools`$SymbolName, Automatic],
                                                      NotebookClose[ EvaluationNotebook[]]]}];
    SelectionMove[ funcDialog, Before, Notebook];
    SelectionMove[ funcDialog, Next, Cell];
    SelectionMove[ funcDialog, After, CellContents];
    SelectionMove[ funcDialog, Previous, Character, 2];
    SelectionMove[ funcDialog, All, Word];
    SetSelectedNotebook[funcDialog]
    ]

      
NETDocLink::notwin = "The operating system must be Windows to access the help file from this button.";

(* If Run gets suitably enhanced, replace Experimental`NewRun with Run below. *)

NETDocLink[x_, y_] := 
 If[$OperatingSystem === "Windows", 
    Experimental`NewRun[StringJoin["hh.exe mk:@MSITStore:", 
                                   ToFileName[{$InstallationDirectory, "SystemFiles", "Links", "NETLink", "Documentation"},
                                              ".NETLink API.chm"],
                                   "::Wolfram.NETLink.", 
                                   If[y === Automatic, x /. _[a_] :> a, y],
                                   ".html"], 
                        Experimental`Shell -> None, 
                        Experimental`Wait -> False], 
    dlMessageToConsole[NETDocLink::notwin]]

CurrentPacletMonitorOpen::pacvarsundef = "One or both of $ApplicationName and $LinkBase are undefined or are empty strings.";
    
CurrentPacletMonitorOpen[]:=
If[Not@ValueQ[$ApplicationName] || StringMatchQ[$ApplicationName, "" | Whitespace] || Not@ValueQ[$LinkBase] || StringMatchQ[$LinkBase, "" | Whitespace],
   dlMessageToConsole[CurrentPacletMonitorOpen::pacvarsundef],
CreatePalette[
 Column[{Row[{" $ApplicationName: ", 
     Style[Dynamic[$ApplicationName], Bold], " "}], 
   Row[{" $Linkbase: ", 
     Style[Dynamic[$LinkBase], Bold], " "}], 
   Row[{" The application ", 
  Dynamic[Refresh[
    Which[MemberQ[$ContextPath, $ApplicationName <> "`"],
          Style["is", Darker@Green, Bold],
          MemberQ[$Packages, $ApplicationName <> "`"] || MemberQ[$Packages, x_String /; StringMatchQ[x, $ApplicationName <> "`*`"]], 
          Style["is being", Blue, Bold],
          True,
          Style["is not", Bold, Red]], 
    UpdateInterval -> .5]], " loaded. ", 
  Button[Style["Load", 10], LoadApplication[], Method -> "Queued"]}]}, 
  Alignment -> Left], Saveable -> False, 
 WindowMargins -> {{Automatic, 160}, {Automatic, 0}}, 
 WindowTitle -> "Paclet Variables Monitor", WindowSize -> {Fit, Fit}]]

OldRow = Grid[{#}, ##2]&

LoadApplication::notspec = "$ApplicationName has not been specified as a nonempty string.";
LoadApplication::failure = "Unable to load the package specified by $ApplicationName.";

LoadApplication[] := 
 If[StringQ[$ApplicationName] && Not@StringMatchQ[$ApplicationName, "" | Whitespace],
    $Old$Path = $Path;
    If[StringQ[$ApplicationDirectory] && FileType[$ApplicationDirectory] === Directory && Not@MemberQ[$Path, $ApplicationDirectory], 
       AppendTo[$Path, $ApplicationDirectory]];
       $PresentBeginTime = Round[AbsoluteTime[]];
       $LoadApplicationNotebook = NotebookPut[Notebook[{}, Visible -> False,
    NotebookDynamicExpression :> Refresh[Catch[If[(* 1 second has passed since the Load button has been clicked. *)
                                                  Round[AbsoluteTime[]] > DocumentationTools`$PresentBeginTime + 1, 
      Which[(* The package does not exist or has the wrong structure. *)
            Not[MemberQ[$Packages, $ApplicationName <> "`"] || MemberQ[$Packages, x_String /; StringMatchQ[x, $ApplicationName <> "`*`"]]], 
            Throw[DocumentationTools`dlMessageToConsole[LoadApplication::failure]; NotebookClose[]; $Path = DocumentationTools`$Old$Path], 
            Not@MemberQ[$ContextPath, $ApplicationName <> "`"], 
            DocumentationTools`$ApplicationLoadPalette[]; NotebookClose[]]]],
                                         UpdateInterval -> 1]]]; 
       Quiet[Needs[$LinkBase <> "`"]];
       If[MemberQ[Notebooks[], $LoadApplicationNotebook], NotebookClose[$LoadApplicationNotebook]];
       If[Not[MemberQ[$Packages, $ApplicationName <> "`"] || MemberQ[$Packages, x_String /; StringMatchQ[x, $ApplicationName <> "`*`"]]], 
          DocumentationTools`dlMessageToConsole[LoadApplication::failure]; $Path = DocumentationTools`$Old$Path], 
       dlMessageToConsole[LoadApplication::notspec]]
                       
$ApplicationLoadPalette[] := 
 CreatePalette[ProgressIndicator[Dynamic[Clock[Infinity]], Indeterminate, ImageSize -> {300, 30}], WindowMargins -> Automatic, 
  WindowTitle -> "Loading " <> DocumentationTools`$LinkBase, WindowSize -> {300, Fit}, 
  NotebookDynamicExpression :> Dynamic[Refresh[If[MemberQ[$ContextPath, DocumentationTools`$ApplicationName <> "`"], NotebookClose[]],
                                               UpdateInterval -> 1]]]
                          

(***** Needed to avoid StringMatchQ below when PacletVariablesPromptOpen[] is in the kernel init.m and DocuTools is in Autoload. ****)

PromptNotebook[]:=
 Notebook[{Cell["", FontSize -> 1, CellElementSpacings -> {"CellMinHeight" -> 1}, Selectable -> False], 
   Cell["Take note of the current values of the following symbols\nif you are working on package documentation:", "Text", 
        FontFamily -> "Helvetica", FontSize -> 13, 
        Selectable -> False, CellMargins -> {{7, 10}, {5, 5}}], 
   Cell["", FontSize -> 1, CellElementSpacings -> {"CellMinHeight" -> 1}, Selectable -> False], 
   Cell[BoxData[FormBox[ToBoxes@Style[Grid[{{Row[{"$ApplicationName: ", 
                                          Style[#, Bold] &@Dynamic[Refresh[If[Not@ValueQ[$ApplicationName] || $ApplicationName === "" || Union[Characters[$ApplicationName]] === {" "}, 
                                         "not defined", $ApplicationName], UpdateInterval -> .5]]}]}, 
                                     {Row[{"$Linkbase: ", Style[#, Bold] &@Dynamic[Refresh[If[Not@ValueQ[$LinkBase] || $LinkBase === "" || Union[Characters[$LinkBase]] === {" "}, 
                                           "not defined", $LinkBase], UpdateInterval -> .5]]}]}, 
                                     {Style["", Selectable -> False, Editable -> False]},
                                     {Row[{"Display this window at startup: ", 
 				       Checkbox[Dynamic[$ShowPacletVariablesPrompt, (sn = #;
 				           If[$ShowPacletVariablesPrompt, 
 				              $ShowPacletVariablesPrompt = False;
 				              SetDocuToolsParametersInFEInit[{"$ShowPacletVariablesPrompt" -> "False"}], 
 				              $ShowPacletVariablesPrompt = True; 
 				              SetDocuToolsParametersInFEInit[{"$ShowPacletVariablesPrompt" -> "True"}]]) &]]}]}, 
                                     {OldRow[{Tooltip[Button["Load Package", NotebookClose[]; LoadApplication[], Method -> "Queued"], 
                                                      "Loads package specified by $ApplicationName.", ActionDelay -> 0.35], 
                                              Tooltip[Button["Continuous Monitor", CurrentPacletMonitorOpen[]; NotebookClose[EvaluationNotebook[]]], 
                                                      "A small window indicating the current settings for these symbols", ActionDelay -> 0.35], 
                                              Button["Cancel", NotebookClose[EvaluationNotebook[]]]}, 
                                             RowAlignments -> Center]},
                                     {Style["", Selectable -> False, Editable -> False]}}, 
                                   ColumnAlignments -> Left, ColumnSpacings -> .2]], TraditionalForm]], FontFamily -> "Helvetica"], 
   Cell["", FontSize -> 50, CellElementSpacings -> {"CellMinHeight" -> 1}, Selectable -> False], 
   Cell[BoxData@ToBoxes@Dynamic[If[DocumentationTools`Utilities`$Workbench, NotebookClose[EvaluationNotebook[]]], SynchronousUpdating -> False], "Text"]}, 
  WindowSize -> {355, 175},
  WindowFloating -> True,
  WindowMargins -> {{Automatic, Automatic}, {Automatic, Automatic}}, 
  WindowFrame -> "Palette", WindowElements -> {}, 
  WindowFrameElements -> {}, ShowCellBracket -> False, 
  ClosingAutoSave -> False, WindowTitle -> None, Saveable -> False, 
  ShowStringCharacters -> False,
  Editable -> False, 
 Selectable -> False]

PacletVariablesPromptOpen[]:=
 Module[{taggingrules = Options[$FrontEnd, TaggingRules], gt},
 If[Not@ValueQ[DocumentationTools`Utilities`$Workbench] && $MathematicaDocs === False && 
     If[ValueQ[DocumentationTools`Utilities`$PacletVariablesPrompt], DocumentationTools`Utilities`$PacletVariablesPrompt === True, True] &&
        ((taggingrules === {TaggingRules -> None}) || (("DocuToolsSettings" /. (TaggingRules /. taggingrules)) === "DocuToolsSettings") || 
           (gt = getTaggingRulesOption["DocuToolsSettings"]; (("$ShowPacletVariablesPrompt" /. gt) === "$ShowPacletVariablesPrompt") || 
             (("$ShowPacletVariablesPrompt" /. gt) =!= "False"))),

 If[(taggingrules === {TaggingRules -> None}) ||
   (("DocuToolsSettings" /. (TaggingRules /. taggingrules)) === "DocuToolsSettings"), $ApplicationName = ""; $LinkBase = ""];
   
 $ShowPacletVariablesPrompt = True;

 NotebookPut[PromptNotebook[]]]]

PreferencesCheck[]:=
 If[Not@ValueQ[DocumentationTools`Utilities`$Workbench] && 
     (taggingrules = Options[$FrontEnd, TaggingRules]; 
      (taggingrules === {TaggingRules -> None}) || 
      (Cases[Options[$FrontEnd, TaggingRules], a : ("$ShowPacletVariablesPrompt" -> _), 5] =!= {"$ShowPacletVariablesPrompt" -> "False"})),
    
    PacletVariablesPromptOpen[]]
    
    
PacletVariablesPromptRestore::pacmon = "Use the Paclet Monitor button instead.";

PacletVariablesPromptRestore[] := (If[Not@ValueQ[DocumentationTools`Utilities`$Workbench] && $Notebooks,
                                     SetDocuToolsParametersInFEInit[{"$ShowPacletVariablesPrompt" -> "True"}]; 
                                     $ShowPacletVariablesPrompt = True;
                                     NotebookPut[PromptNotebook[]]];
                                   If[ValueQ[DocumentationTools`Utilities`$Workbench] && $Notebooks,
                                      dlMessageToConsole[PacletVariablesPromptRestore::pacmon]])
                                      
(*************** Code for annotations *********************************************)
                                      
$AnnotationText = ""
$AnnotationDialogNotebook = None

GenerateAnnotationDialog::nocellid = "The selected cell lacks a cell id.";

$AnnotationCell = Cell[BoxData[GridBox[{{Cell[BoxData[ButtonBox["  Edit  ", BaseStyle -> "TextAnnotationButton"]]], "", 
                                         Cell[BoxData[ButtonBox[" Delete " , BaseStyle -> "TextAnnotationRemoveButton"]]], ""}},
                                       GridBoxAlignment -> {"Columns" -> {Left, Center, Right, Right}, "Rows" -> {{Automatic}}}, 
                                       GridBoxItemSize -> {"Columns" -> {Scaled[0.2], Scaled[0.58], Scaled[0.2], Scaled[0.02]}}]]]
                                       
$AnnotationCellPattern = Cell[BoxData[GridBox[{{Cell[BoxData[ButtonBox[_String, BaseStyle -> "TextAnnotationButton"]]], "", 
                                                Cell[BoxData[ButtonBox[_String , BaseStyle -> "TextAnnotationRemoveButton"]]], ""}},
                                       GridBoxAlignment -> {"Columns" -> _, "Rows" -> {{Automatic}}}, GridBoxItemSize -> {"Columns" -> _}]]]

Options[GenerateAnnotationDialog] = {NotebookToActOn -> EvaluationNotebook, CellIDToActOn -> None}

GenerateAnnotationDialog[opts___] := 
 Module[{evnb, ceid, nb, re, cellid}, 
  evnb = NotebookToActOn /. {opts} /. Options[GenerateAnnotationDialog];
  ceid = CellIDToActOn /. {opts} /. Options[GenerateAnnotationDialog];
  nb = If[evnb === EvaluationNotebook, ButtonNotebook[], evnb]; 
  If[evnb === EvaluationNotebook, SelectionMove[nb, All, ButtonCell]];
  re = NotebookRead[nb]; 
  $AnnotationText = If[# === {}, "", #[[1]]] &@Cases[re, 
                                                     c : (CellFrameLabels -> {{_, _}, {_, 
                                         Cell[TextData[{a_, $AnnotationCell}], 
                                              "TextAnnotation", CellSize -> {_, Inherited}]}}) :> StringReplace[a, b__ ~~ "\n" ~~ EndOfString :> b],
                                                     Infinity]; 
  If[ceid === None,
     cellid = If[#==={},#,#[[1]]]&@Cases[re, _[CellID, a_] :> a];
     If[cellid === {}, Throw[MessageToConsole[GenerateAnnotationDialog::nocellid]]],
     cellid = ceid];
  
  If[$AnnotationDialogNotebook === None,
     (SelectionMove[#, Before, Notebook];
      SelectionMove[#, Next, Cell, 2];
      SelectionMove[#, After, CellContents];
      FrontEndTokenExecute[#, "Tab"];
      SetSelectedNotebook[#]) &[
   $AnnotationDialogNotebook = NotebookPut[Notebook[{Cell["", FontSize -> 1, CellElementSpacings -> {"CellMinHeight" -> 1}], 
       Cell[BoxData[ToBoxes@Style[
           Grid[{{Style["Annotation:", Bold, Editable -> False]}, 
                {Tooltip[InputField[Dynamic[$AnnotationText], String, FieldSize -> {45, {4, Infinity}}],
                         "Use Shift\[ThinSpace]+\[ThinSpace]\[ThinSpace]Return for new line.", TooltipDelay -> .7]}, 
                {Grid[{{Grid[{{Button[Style["OK", Bold], 
                                      AnnotationModify[], Method -> "Queued"],
                               Button[Style["Cancel", Bold], $AnnotationDialogNotebook = None; NotebookClose[]]}}],
                               Button[Style["Delete", Bold], $AnnotationDialogNotebook = None; AnnotationRemove[]; NotebookClose[]]}}, 
                      ItemSize -> {{37, 8}}, Alignment -> {{Left, Right}, Automatic}]}},
                Alignment -> {{{Left}}, Automatic}], 
           FontFamily -> "Verdana", FontSize -> 11]], 
        CellContext -> "Global`"], 
       Cell["", FontSize -> 1, CellElementSpacings -> {"CellMinHeight" -> 1}]}, 
                             WindowMargins -> {{Automatic, Automatic}, {Automatic, Automatic}}, 
                             WindowToolbars -> {}, 
                             WindowFrameElements -> {},
                             ShowCellBracket -> False, 
                             ClosingAutoSave -> False, 
                             WindowTitle -> "Annotate Cell", 
                          (* NotebookEventActions -> {"ReturnKeyDown" :> ((* To get the focus of the notebook first. *)
                                                                          FrontEndTokenExecute[$AnnotationDialogNotebook, "Tab"]; AnnotationModify[])},
                                                                          *)
                             Saveable -> False, 
                             ShowStringCharacters -> False, 
                             Selectable -> False, 
                             WindowSize -> {Fit, 145}, 
                             WindowFrame -> "Palette", 
                             ScrollingOptions -> {"PagewiseScrolling" -> False, "PagewiseDisplay" -> True, "VerticalScrollRange" -> Fit}, 
                             WindowElements -> {"VerticalScrollBar"},
                             TaggingRules -> {"AnnotationCellID" -> cellid, "AnnotationNotebook" -> nb}]]],
     (SelectionMove[#, Before, Notebook];
      SelectionMove[#, Next, Cell, 2];
      SelectionMove[#, After, CellContents];
      FrontEndTokenExecute[#, "Tab"];
      SetSelectedNotebook[#]) &[$AnnotationDialogNotebook];
      setTaggingRulesOption[$AnnotationDialogNotebook, "AnnotationCellID" -> ceid]; 
      setTaggingRulesOption[$AnnotationDialogNotebook, "AnnotationNotebook" -> evnb]]]
      
AddSpaceAtEnd = If[StringMatchQ[#, __ ~~ (a_ /; Not@StringMatchQ[a, Whitespace]) ~~ EndOfString], # <> " ", #] &
                             
AnnotationModify::inemp = "The input field is empty.";
AnnotationModify::nbclosed = "The notebook you were working with is no longer open.";
AnnotationModify::cellnf = "The cell that was being worked with cannot be found.";
AnnotationModify::finpw = "The cell does not have a finite page width.";
AnnotationModify::lrcm = "The left and right cell margins were not both integers.";
AnnotationModify::noanposs = "It is not possible to annotate this cell.";

AnnotationModify[] := 
 Module[{evnb, cellid, f, opts, c, p, cs, re}, 
  Catch[
   If[StringMatchQ[$AnnotationText, ""|Whitespace], Throw[MessageToConsole[AnnotationModify::inemp]]];
   evnb = getTaggingRulesOption[$AnnotationDialogNotebook, "AnnotationNotebook"];
   If[Not@MemberQ[Last /@ Notebooks[], evnb[[-1]]], Throw[NotebookClose[]; 
                                                          $AnnotationDialogNotebook = None; 
                                                          MessageToConsole[AnnotationModify::nbclosed]]];
   cellid = getTaggingRulesOption[$AnnotationDialogNotebook, "AnnotationCellID"];
   f = NotebookFind[evnb, cellid, All, CellID]; 
   If[f === $Failed, Throw[NotebookClose[]; $AnnotationDialogNotebook = None; MessageToConsole[AnnotationModify::cellnf]]];
   opts = Options[NotebookSelection[evnb], CellFrameLabels];
   
   If[MatchQ[opts, {CellFrameLabels -> {{_, _}, {_, Cell[TextData[{_String, $AnnotationCellPattern}], 
                                                         "TextAnnotation", CellSize -> {_, Inherited}] | None | Inherited}}}],
                                                         
      {c, p} = AbsoluteOptions[NotebookSelection[evnb], {CellMargins, PageWidth}];
      (* If[Not[NumberQ[#] && -Infinity < # < Infinity &[(PageWidth /. p)]], Throw[MessageToConsole[AnnotationModify::finpw]]];
         If[Not@MatchQ[(CellMargins /. c), {{_Integer, _Integer}, _}], Throw[MessageToConsole[AnnotationModify::lrcm]]]; *)
         
      cs = If[Not[NumberQ[#] && -Infinity < # < Infinity &[(PageWidth /. p)]] || Not@MatchQ[(CellMargins /. c), {{_Integer, _Integer}, _}],
              590,
              ((PageWidth /. p) - Total@(CellMargins /. c)[[1]])];
      
      re = NotebookRead[evnb]; 
      NotebookWrite[evnb, 
                    If[FreeQ[re, CellFrameLabels -> _],
                       Insert[re, 
                              (CellFrameLabels -> {{Inherited, Inherited}, {Inherited, 
                               Cell[TextData[{StringJoin[AddSpaceAtEnd@$AnnotationText, "\n"], $AnnotationCell}], "TextAnnotation", CellSize -> {cs, Inherited}]}}),
                              -1],
                        re /. (CellFrameLabels -> {{a_, b_}, {c_, _}}) :> (CellFrameLabels -> {{a, b}, {c, 
                               Cell[TextData[{StringJoin[AddSpaceAtEnd@$AnnotationText, "\n"], $AnnotationCell}], "TextAnnotation", CellSize -> {cs, Inherited}]}})],
                     All];
      FrontEndExecute[FrontEnd`SelectionAddCellTags[evnb, "TextAnnotation"]];
      $AnnotationDialogNotebook = None;
      NotebookClose[];
      SetSelectedNotebook[evnb],
                                                                                                        
      MessageToConsole[AnnotationModify::noanposs]]]]
       
AnnotationInsert::noin = "There is no open input notebook.";
AnnotationInsert::betwcells = "The cursor is between cells or not inside an input notebook.";
AnnotationInsert::mulcell = "Multiple cells have been selected.";
AnnotationInsert::cellidfail = "Cell id creation failed."
AnnotationInsert::noanposs = "It is not possible to annotate this cell.";

AnnotationInsert[] := Module[{nb = InputNotebook[], ci, re, cellid, opts},
   Catch[If[(* There is no input notebook. *)nb === $Failed, Throw[MessageToConsole[AnnotationInsert::noin]]]; 
    ci = CellInfo[nb];
    If[(*The cursor is between cells or not inside an input notebook.*)ci === $Failed, Throw[MessageToConsole[AnnotationInsert::betwcells]]]; 
    If[multipleCellBracketsSelected[ci], Throw[MessageToConsole[AnnotationInsert::mulcell]]]; 
    ExpandToCell[nb];
    If[ToString[Options[nb, "CreateCellID"]] === "{CreateCellID -> False}", SetOptions[nb, "CreateCellID" -> True]];
    
    (* This loses the focus on windows -- typing into the notebook does not even work.
    If[Options[NotebookSelection[nb], CellID] === {CellID -> 0},
       FrontEndExecute[FrontEnd`FrontEndToken["DeleteNext"]];
       FrontEndExecute[FrontEnd`FrontEndToken["Undo"]]]; *)
       
    (* This will create a cell id if none exists. *)
    If[Options[NotebookSelection[nb], CellID] === {CellID -> 0},
       re = NotebookRead[nb];
       NotebookWrite[nb, re, All]];
    If[# === {CellID -> 0}, Throw[MessageToConsole[AnnotationInsert::cellidfail]], cellid = #[[1, 2]]]&[Options[NotebookSelection[nb], CellID]];
       
    opts = Options[NotebookSelection[nb], CellFrameLabels];
    Which[MatchQ[opts, 
                 {CellFrameLabels -> {{_, _}, {_, Cell[TextData[{_String, $AnnotationCellPattern}], 
                                                         "TextAnnotation", CellSize -> {_, Inherited}] | None | Inherited}}}], 
          $AnnotationText = If[# === {}, "", #[[1]]]&@Cases[opts, 
                                                   Cell[TextData[{a_String, _}], "TextAnnotation", _] :> StringReplace[a, b__ ~~ "\n" ~~ EndOfString :> b], 
                                  Infinity];
          GenerateAnnotationDialog[NotebookToActOn -> nb, CellIDToActOn -> cellid], 
          MatchQ[opts, {CellFrameLabels -> {{_, _}, {_, a_ /; Not@MatchQ[a, None | Inherited]}}}], 
          MessageToConsole[AnnotationInsert::noanposs]]]]
    
AnnotationRemove::noin = "There is no open input notebook.";
AnnotationRemove::betwcells = "The cursor is between cells or not inside an input notebook.";
AnnotationRemove::mulcell = "Multiple cells have been selected."; 
AnnotationRemove::notpres = "No annotation is present.";

Options[AnnotationRemove] = {CalledFromFrameLabel -> False}

AnnotationRemove[opts___] := Module[{nb, cffl, ci, optss, re},
  Catch[cffl = CalledFromFrameLabel /. {opts} /. Options[AnnotationRemove];
   nb = If[cffl, InputNotebook[], NextNotebook[]];
   If[(*There is no input notebook.*)nb === $Failed, Throw[MessageToConsole[AnnotationRemove::noin]]];
   If[Not@cffl, ci = CellInfo[nb];
                If[(*The cursor is between cells or not inside an input notebook.*)ci === $Failed, 
                   Throw[MessageToConsole[AnnotationRemove::betwcells]]]; 
                If[multipleCellBracketsSelected[ci], 
                Throw[MessageToConsole[AnnotationRemove::mulcell]]]]; 
   If[Not@cffl, ExpandToCell[nb], SelectionMove[nb, All, ButtonCell]];
   optss = Options[NotebookSelection[nb], CellFrameLabels];
   If[Not@MatchQ[optss, {CellFrameLabels -> {{_, _}, {_, Cell[TextData[{_String, $AnnotationCellPattern}], 
                                                             "TextAnnotation", CellSize -> {_, Inherited}]}}}], 
      MessageToConsole[AnnotationRemove::notpres],
      re = NotebookRead[nb];
      NotebookWrite[nb, re /. {(CellFrameLabels -> {{Inherited, Inherited}, {Inherited, _}}) -> Sequence[],
                               (CellFrameLabels -> {{a_, b_}, {c_, _}}) :> (CellFrameLabels -> {{a, b}, {c, Inherited}})}, All];
      FrontEndExecute[FrontEnd`SelectionRemoveCellTags[nb, "TextAnnotation"]]]]]
      
AnnotationSearch::noin = "There is no open input notebook.";
AnnotationSearch::noannot = "The input notebook contains no annotation cells.";

AnnotationSearch[a : "Up" | "Down"] := 
 Module[{nb = InputNotebook[]}, 
  Catch[If[nb === $Failed, Throw[MessageToConsole[AnnotationSearch::noin]]];; 
        If[NotebookFind[nb, "TextAnnotation", If[a === "Down", Next, Previous], CellTags] === $Failed, 
           Throw[MessageToConsole[AnnotationSearch::noannot]]]]]
           
(*************** End code for annotations *********************************************)


End[]

EndPackage[]
