(* ::Package:: *)

(* :Title: Utils.m *)

(* :Authors:
	Andrew Hunt, Shaun McCance, Todd Gayley
	andy@wolfram.com, shaunm@wolfram.com, tgayley@wolfram.com
*)

(* :Package Version: 0.50 *)

(* :Mathematica Version: 6.0 *)

BeginPackage["DocumentationBuild`Utils`", {
    "DocumentationBuild`Common`",
    "Transmogrify`", "JLink`"
}];

AddSpaces::usage = "Add spaces to content.";
AddSpans::usage = "Add spaces to content.";
AddSpaceToCamelCase::usage = "AddSpaceToCamelCase[str] adds space to camel case e.g. 'SomeFunction' -> 'Some Function'.";
ReplaceCharacters::usage = "";
ItalicizeMoreAboutKeyWords::usage = "";
makeTitleModiferString::usage ="";
StringList::usage = "StringList[list] returns the contents of list as a single string."
ToStringList::usage = "ToStringList[list] returns the contents of list as a string, and empty string for empty list or other expressions."
titlemaker::usage = "";
NumberToLetter::usage = "";
MakeCamelCaseString::usage = "MakeCamelCaseString[str] capitalizes the first letter(s) and deletes spaces in str.";
CapitalizeString::usage = "CapitalizeString[str] returns str in all capital letters.";
LowercaseString::usage = "LowercaseString[str] returns str in all lowercase letters.";
GetQuietly::usage = "Turn off Syntax::newl before using Get.";
GetPrimaryGuide::usage = "";
CoreMathematicaPacletQ::usage = "";
HasOptionsQ::usage = "HasOptionsQ[nb] returns True if the symbol page nb has an options table. Note: no page type checking.";

RepairMetadataCellLabels::usage = "RepairMetadataCellLabels[notebook] will return the nb expr of a notebook with the Categorization and Details cell labels re-added.
                                   RepairMetadataCellLabels[filename] will do the same to a notebook file and overwrite-save that file."

DocumentFlag::usage = "DocumentFlag[nbfile] returns the flag style (\"InternalFlag\"
etc.), if any, that is used in the first cell in the notebook saved in nbfile."

ConvertFileQ::usage = "ConvertFileQ[nbfile] returns True if nb should be converted during documentation build.";

IsManipulateQ::usage = "IsManipulateQ[cell] returns True if cell is a Manipulate expression.";
IsSoundQ::usage = "IsSoundQ[cell] returns True if cell contains a Sound expression.";
IsAudioQ::usage = "IsAudioQ[cell] returns True if cell contains a Audio expression.";

UnsortedUnion::usage = "UnsortedUnion[list] returns a list of unique elements in order in which thery appear."
GetLanguageExtension::usage = "GetLanguageExtension[lang] returns langs two letter language extension.";

DotHTML::usage = "";

GetFirstCellContents::usage = "";

Localized::usage = "Localized[key, lang, args___] looks up the localized content for\
 the key in Localization.m, possibly calling it as a function with args.";

BaseName::usage = "BaseName[file] returns the base file name of a full path\
 filename.  For example, /path/to/file.nb -> file.nb.  BaseName[file,\
 Extension->False] causes the file extension to not be included,\
 /path/to/file.nb -> file";

CellToString::usage = "Converts cell expression to String suitable as content for alt tags";

CreateInputForm::usage = "CreateInputForm[cell] returns the InputForm string of a cell."
Options[CreateInputForm] = {PrintErrors->True};

RemoveEmptyCellGroups::usage = "RemoveEmptyCellGroups[ expr, styleList ] removes all single\
 cells in expr that are normally the header cell for CellGroups as identified by the\
 second argument styleList."

MakeActionMenu::usage = "MakeActionMenu[caption, dat]. The caption is the string data which will\
 be shown. The dat is the item data that DocumentationBuild`Utils`GetMoreAboutSectionList[file]\
 returns, for example.";

GetClassificationBoxes::usage = "";

GetMoreAboutSectionList::usage =
  "GetMoreAboutSectionList[nb] returns a list of 'More About' titles from a symbol notebook.";
GetSeeAlsoSectionList::usage =
  "GetSeeAlsoSectionList[nb] returns a list of 'See Also' functions from a symbol notebook.";
GetTutorialFunctionList::usage =
  "GetTutorialFunctionList[nb] returns a list of functions within Tutorial DefinitionBoxes.";
GetTutorialMoreAboutSectionList::usage =
  "GetTutorialMoreAboutSectionList[nb] returns a list of 'More About' titles from a tutorial notebook.";
GetTutorialsSectionList::usage =
  "GetTutorialsSectionList[nb] returns a list of 'tutorials' from a symbol notebook.";
GetRelatedTutorialsSectionList::usage =
  "GetRelatedTutorialsSectionList[nb] returns a list of 'related tutorials' from a tutorial notebook.";
GetTutorialMoreAboutList::usage =
  "GetTutorialMoreAboutList[nb] returns a list of 'tutorials' from a symbol notebook.";
GetFeaturedExampleMoreAboutSectionList::usage =
  "GetFeaturedExampleMoreAboutSectionList[nb] returns a list of 'related guides' from a notebook.";
GetFeaturedExampleTutorialsSectionList::usage =
  "GetFeaturedExampleTutorialsSectionList[nb] returns a list of 'related guides' from a notebook.";
GetGuideMoreAboutSectionList::usage =
  "GetGuideMoreAboutSectionList[nb] returns a list of 'More About' titles from a guide notebook.";
GetGuideTutorialsSectionList::usage =
  "GetGuideTutorialsSectionList[nb] returns a list of 'tutorials' from a guide notebook.";
GetGuideFunctionList::usage =
  "GetGuideFunctionList[nb] returns an alphabetized list of functions the Guide page..";
GetGuideHowTosList::usage =
  "GetGuideHowTosList[nb] returns a list of how-to links from a guide notebook.";

GetGuidesDirectory::usage =
  "GetGuidesDirectory[dir] returns the path to the 'Guides' directory.";
GetTutorialsDirectory::usage =
  "GetTutorialsDirectory[dir] returns the path to the 'Tutorials' directory.";
GetNotebookIndexQ::usage = 
  "GetNotebookIndexQ[pagetype] returns True or False for whether the pagetype should be indexed.";
GetOverviewButtonLinks::usage = "GetOverviewButtonLinks[uri, language] returns a list previous/next overview links.";
GetValueFromOptions::usage = "returns value of the right side of an option, for example GetValueFromOptions[CoreInformation, paclet, LinkTrails]";

AppendHTMLDirectory::usage = "AppendHTMLDirectory[path] returns ToFileName[{path, 'html'}].";
ExportTOCXML::usage = "ExportTOCXML[filepath, nb, linkbase] returns an xml expression of a documentation TOC nb ready for inclusion in Workbench."
ExportPluginXML::usage = "ExportPluginXML[filepath, nb, linkbase]";
ExportEclipseResourceFiles::usage = "ExportEclipseResourceFiles[dir, nb, linkbase]";

MakeLinkTrailExpr::usage = "";
LinkTrailFromPacletURI::usage = "";
CleanSummaryForWeb::usage = "";

HyperlinkEqualityTest::usage = "Test consecutive ButtonBoxes or StyleBoxes which contain ButtonBoxes for Links with same ButtonData";
ApplyStyleBoxInside::usage = "Utility for MergeButtonBoxes";
MergeButtonBoxes::usage = "Utility for ConvertTextData";
ConvertTextData::usage = "Checks TextData[] using HyperlinkEqualityTest, if so it merges consectutive Style/ButtonBoxes into one ButtonBox";

MakePacletButton::usage = "Turn a normal button box into the appropriate 'paclet:' link."

ConvertButtonBoxesToTemplateBoxes::usage = "ConvertButtonBoxesToTemplateBoxes[nb] changes ButtonBoxes into TemplateBoxes.";

MakeCharacterNotesGrid::usage = "MakeCharacterNotesGrid[charImageCell, notesGroup] constructs the side-by-side character/notes grid for character pages.";

FESaveNotebook::usage = "FESaveNotebook[file, nb] saves the nb to file using the FE.";
FETimeConstrained::usage = "";

GetSearchMetaDataList::usage = "";

GetCellsFromCellGroup::usage = "";

BlockMessageOff::usage =
"BlockMessageOff[{symbol1::tag1, symbol2::tag2, ... }, body] evaluates body with
the specified messages turned off. Messages that were on in the first place are
turned back on after body is evaluated."

BuildLog::usage = "";
ShowImageDifferences::usage = "ShowImageDifferences[objA, objB, type] shows the graphical differences of two objects and type (either NB or URL) defines whether these objects are notebooks or urls"

sameLinkTrailsQ::usage = "";
samePacletNameCellQ::usage = "";
notesInNotesSectionQ::usage = "";
seeAlsoInSeeAlsoSectionQ::usage = "";
moreAboutInMoreAboutSectionQ::usage = "";
corruptedCellsQ::usage = "";
samePulldownMenusQ::usage = "";
samePulldownMenuTextQ::usage = "";
buttonBoxDataMissingQ::usage = "";

PreviousNextElementList::usage = "";
GetBlockStyleList::usage="";
FlattenCellGroups::usage="";
MakeComparisonListingCell::usage="";
MakeComparisonItemCell::usage="";

ApplyConvertGraphicsToBitmapsToNotebook::usage="";
SaveConvertGraphicsToBitmapsToNotebook::usage="";

functionAncestorQ::usage="functionAncestorQ[check, totalexpr, subexpr] returns True if check is a 
         function head that appears as such in the totalexpr as an ancestor of subexpr."

MakeThumbnail::usage = "MakeThumbnail[expr] creates two cells: a SuppressedMaterial preview cell, \
and an AlternateThumbnail cell.  The preview cell shows the 300x300 pixel thumbnail at 150x150 size, \
and the AlternateThumbnail cell provides the image used by the build system to make the thumbnail.  \
A standard frame can be added to the thumbnail image with the Framed and FrameMargins options. \
Typically MakeThumnail[expr, Framed->True] suffices. \
To create a thumbnail of an excerpt of the rendering of 'expr', use the option Excerpt->{{l, b}, f} \
where f is the linear fraction of the excerpt and {l, b} is the scaled coordinate of the left bottom corner of the excerpt. \
For example, Excerpt \[Rule] {{.25, .25}, .5} excerpts the center 100x100 subimage of a 200x200 image and \
Excerpt \[Rule] {{0, 0}, .5} excerpts the bottom left quadrant.";

stringReplaceRepeated::usage = "The old chestnut: run StringReplace repeatedly until the expression no longer changes.";
convertWSMLink::usage = "If a string starts with wsm://, convert it to an http:// link.";
convertModelicaLink::ysage = "If a string starts with modelica://, convert it to an http:// link.";

CheckSeeAlsos::usage = "CheckSeeAlsos[nb] removes links from see alsos that are to future or nonexistent pages.";
CheckGuideInlineListings::usage = "CheckGuideInlineListings[nb] removes links from guide inline listings that are to future or nonexistent pages.";

Begin["`Private`"];


(******************************************************************************)
(** Small Utilities *)


(** AddSpaces **)
AddSpaces[c_String]:= StringReplace[c, {","->", "}];
AddSpaces[c_List]:= AddSpaces /@ c;
AddSpaces[XMLElement[tag_, attr_, c_]]:= XMLElement[tag, attr, AddSpaces /@ c];
AddSpaces[s__String] := AddSpaces[StringJoin[s]]
AddSpaces[expr__ ] := Sequence@@({expr} /. x:(_XMLElement|_String):>AddSpaces[x])

(** AddSpaces **)
AddSpans[c_String]:= If[!StringFreeQ[c, {"|", ",", "[", "]", "{", "}", "(", ")"}], 
	XMLElement["span", {"class"->"specialcharacters"}, Characters[c] /. {
	"|" -> XMLElement["span", {"class" -> "verticalbar"}, {"|"}],
	"," -> XMLElement["span", {"class" -> "comma"}, {","}],
	"[" -> XMLElement["span", {"class" -> "openbracket"}, {"["}],
	"]" -> XMLElement["span", {"class" -> "closebracket"}, {"]"}],
	"{" -> XMLElement["span", {"class" -> "openbrace"}, {"{"}],
	"}" -> XMLElement["span", {"class" -> "closebrace"}, {"}"}],
	"(" -> XMLElement["span", {"class" -> "openparenthesis"}, {"("}],
	")" -> XMLElement["span", {"class" -> "closeparenthesis"}, {")"}]
	}],
	c];
AddSpans[c_List]:= AddSpans /@ c;
AddSpans[XMLElement[tag_, attr_, c_]]:= XMLElement[tag, attr, AddSpans /@ c];
AddSpans[s__String] := AddSpans[StringJoin[s]]
AddSpans[expr__ ] := Sequence@@({expr} /. x:(_XMLElement|_String):>AddSpans[x])

(** AddSpaceToCamelCase **)
AddSpaceToCamelCase[str_String] :=
  Module[{lst, opts},
  	opts = Alternatives["WorldPlot", "GUIKit", "MSP", "XML", "DatabaseLink",
                     "JLink", "NETLink", "MathLink", "ANOVA", "J/Link",
                     "WebServices", "XMLSchema", "webMathematica", "CUDALink", 
                     "TetGenLink", "GPUTools", "SymbolicC", "DLLTools", "OpenCLLink", "FDLLink", 
                     "SystemModeler", "RLink", "HTTPClient", "LibraryLink", "FEM"];
    If[StringMatchQ[str, ___ ~~ opts ~~ ___],
      Return[StringReplace[str, some___ ~~ opt:opts ~~ rest___ :> 
      	AddSpaceToCamelCase[some] <> opt <> If[StringLength[rest] > 0, " ", ""] <> AddSpaceToCamelCase[rest]]] ];
    StringReplace[
      StringReplace[str, 
        RegularExpression["(?<![ 0-9$]|^)([A-Z])"] -> " $1"], 
      {RegularExpression["^\\s"] -> "", RegularExpression["\\s+"] -> " "}]
  ];
AddSpaceToCamelCase[TextData[c___]] := TextData[c];
AddSpaceToCamelCase[s___] := (Message[AddSpaceToCamelCase::args, s]; $Failed);
AddSpaceToCamelCase::args = "Expected string but found: `1`";

(** ReplaceCharacters **)
ReplaceCharacters[str_String, search_String, rep_] :=
  Flatten[StringSplit[str, search -> rep]];
ReplaceCharacters[Cell[TextData[lst_List], r___], search_String, rep_] :=
 Cell[TextData[Flatten[
    If[Head@# === String,
       ReplaceCharacters[#, search, rep],
       #
       ] & /@ lst]], r
  ];
ReplaceCharacters[Cell[str_String, r___], search_String, rep_] :=
  ReplaceCharacters[str, search, rep];

makeTitleModiferString[tm_List] := Module[{},
  Which[
   Length[tm] === 0, "",
   Length[tm] === 1, "(" <> tm[[1]] <> ")",
   Length[tm] > 1, "(" <> StringJoin[Riffle[tm, ", "]] <> ")",
   True, ""]];

(** StringList **)
StringList[s_]:= If[Length[s] > 0, StringJoin[First[s], (", " <> #) & /@ Rest[s]], ""];

ToStringList[lis_List] := Which[Length[lis] < 1, " ",
   Length[lis] === 1, StringJoin[lis],
   True, StringJoin[Riffle[lis, ", "]]
   ];

ToStringList[lis_] := "";

titlemaker[title_String] := 
 Module[{}, 
  ToLowerCase[
   StringReplace[
    title, {" " -> "-", "\[EAcute]" -> "e", "\[Not]\[EAcute]" -> "e", 
     "\[Hyphen]" -> "-", "-" -> "-",  ":"->"-", "&" -> "And", "\[Pi]"->"pi" }]]];
     
NumberToLetter[num_String] := 
 Module[{}, FromCharacterCode[96 + ToExpression[num]]];

(** MakeCamelCaseString **)
MakeCamelCaseString[str_String] := 
  StringReplace[
    StringReplace[str, {
      "&" -> "And",
      "-" -> " ",
      ("/"|"{"|"}"|"'"|"\""|","|":"|"."|"["|"]") -> ""
    }],
    RegularExpression["\\s(\\w)"] :> ToUpperCase["$1"]
  ];
MakeCamelCaseString[c___] := 
  MakeCamelCaseString@Transmogrify`ConvertToString[c];

(** CapitalizeString **)
CapitalizeString[str_String]:= StringReplace[str, RegularExpression["(\\w*)"] :> ToUpperCase["$1"]];
CapitalizeString[expr___]:= (Message[CapitalizeString::str,expr]; expr);
CapitalizeString::str = "Expected String found: `1`";

(** LowercaseString **)
LowercaseString[str_String]:= StringReplace[str, RegularExpression["(\\w*)"] :> ToLowerCase["$1"]];
LowercaseString[expr___]:= (Message[LowercaseString::str,expr]; expr);
LowercaseString::str = "Expected String found: `1`";

(** GetQuietly **)
GetQuietly[name_String] := (Off[Syntax::"newl"]; Get[name]);

(** Get first cell in nb -- Used within DocumentFlag[] **)
FirstCell[ nbFile_String] := GetQuietly[nbFile][[1, 1]];

(** DocumentFlag **)
(* DocumentFlag[ nbfile_String] :=
Replace[ FirstCell[ nbfile],
{
  Cell[_,
    flagStyle:(
      "ObsoleteFlag" |
      "InternalFlag" |
      "TemporaryFlag" |
      "PreviewFlag" | 
      "FutureFlag" |
      "ExcisedFlag"
    ),
    ___] :>
      flagStyle,
  _ -> None
}]; *)

DocumentFlag[nb_String] := DocumentFlag[GetQuietly[nb]];

DocumentFlag[expr_Notebook]:=
Module[{f, g},
  f = Cases[expr, Cell[_, "ObsoleteFlag"|"InternalFlag"|"TemporaryFlag"|"PreviewFlag"|"FutureFlag"|"ExcisedFlag"|"AwaitingFutureDesignReviewFlag", ___], Infinity];
  g = ScreenStyleEnvironment /. Options[expr];
  
  If[Length[f] < 1 && ! MatchQ[g, ScreenStyleEnvironment | "Working"],
    f = Switch[g,
     "InternalDocument", {{"", "InternalFlag"}},
     "ObsoleteObject" | "FutureObject" | "ExcisedObject",
     {{"", StringReplace[g, "Object" -> "Flag"]}},
     _,
     f]
    ];
  
  If[Length@f > 0,
  	If[Length[Cases[expr, "U N D E R   D E V E L O P M E N T", Infinity]]>0, 
  		Switch[Part[List@@First@f,2], 
  			"ObsoleteFlag", None,
  			"InternalFlag", Part[List@@First@f,2],
  			"TemporaryFlag", None,
  			"PreviewFlag", None,
  			"FutureFlag", None,
  			"ExcisedFlag", Part[List@@First@f,2],
  			"AwaitingFutureDesignReviewFlag", None,  
  			_ , Part[List@@First@f,2]], 
  			Part[List@@First@f,2]
  			],  
  	None]
];

(** ConvertFileQ **)
ConvertFileQ[ file_String]:= ConvertFileQ[ GetQuietly@file];

ConvertFileQ[expr_Notebook]:=
Module[{ convert},
  convert = Cases[expr, Rule[TaggingRules, {___, Rule["UndeployInDocsBuild", val_], ___}] :> val, Infinity, 1];
  If[convert === {True}, False, True]
];

ConvertFileQ[infile_String, ignorefiles_List]:=
Module[{},
  ((Length@ignorefiles > 0) && MemberQ[Alternatives@@ignorefiles, StringReplace[infile, {DirectoryName[infile] -> ""}]] )
];

(** UnsortedUnion **)
UnsortedUnion[list_List]:= Reap[Sow[1, list], _, # &][[2]]

(** GetLanguageExtension **)
GetLanguageExtension[ext_String] := Switch[ext, "Japanese", "ja", "ChineseSimplified", "zh", "Spanish", "es", _, "en"];

(** DotHTML **)
DotHTML[s_String] :=
  If[StringMatchQ[s, "*#*"],
    (StringTake[s, #] <> ".html" <> StringDrop[s, #])&[StringPosition[s, "#"][[-1, -1]] - 1],
    s<> ".html" ];

(* get string form of first cell in nb *)
GetFirstCellContents[nb_Notebook] := 
  GetFirstCellContents[Cases[nb, Cell[c_, ___] :> c, Infinity]];
GetFirstCellContents[{l_, ___}] := GetFirstCellContents[l];
GetFirstCellContents[s_String] := s;
GetFirstCellContents[___] := $Failed;

(** IsManipulateQ **)
IsManipulateQ[___]:= False;
IsManipulateQ[c_Cell]:= Not[FreeQ[c, Manipulate`InterpretManipulate]];

(** IsSoundQ **)
IsSoundQ[___]:= False;
IsSoundQ[c_Cell]:= Not[FreeQ[c, Sound]];

(** IsAudioQ **)
IsAudioQ[___]:= False;
IsAudioQ[c_Cell]:= Not[FreeQ[c, Audio]];

(******************************************************************************)
(** Localized **)
$Localization := $Localization = Dispatch[Get[ToFileName[
  {$DocumentationBuildDirectory}, "Localization.m"]]];
Localized::nokey = "Localized could not find the key `1`.";
Localized::nolang = "Localized could not find the `1` translation of `2`.";
Localized[key_, lang_, args___] :=
  Module[{vals, val},
    vals = key /. $Localization;
    If[vals === key,
      Message[Localized::nokey, key];
      Return[None]];
    If[MatchQ[vals, {___, lang -> _, ___}],
        val = lang /. vals,
        Message[Localized::nolang, lang, key];
        val = "English" /. vals
    ];
    If[Length[{args}] > 0, val[args], val]
  ];


(******************************************************************************)
(** GetNotebookIndexQ **)
GetNotebookIndexQ::arg = "String expected as input but value was: `1`";
GetNotebookIndexQ[c___]:= (Message[GetNotebookIndexQ::arg, c]; False);
GetNotebookIndexQ[type_String]:=
  Switch[type,
    (*"Message", False,*)
    (* "HowTo", False, *)
    _, True
  ];


(******************************************************************************)
(** BaseName **)

Options[BaseName] = {"Extension" -> True};
BaseName[f_String,opts___?OptionQ]:= (
  If[("Extension" /. {opts} /. Options[BaseName])===True,
    #,
    StringReplace[#, RegularExpression["\\.\\w+$"] :> ""]
  ]&@ StringReplace[f, DirectoryName[f]->""]
);



(******************************************************************************)
(** CellToString **)
TextCharacterReplacementRules := TextCharacterReplacementRules =
  Get[ToFileName[{$DocumentationBuildDirectory}, "TextCharacterReplacementRules.m"]];

Parenth[x_String?((StringLength[#] == 1)&)] := x;
Parenth[x_, ___] := {"(", x, ")"};

CellToString[mycell_] :=
  StringReplace[
    StringJoin[
      ToString/@Flatten[
        {mycell //. {
          Cell[x__, ("NumberedEquation" | "NumberedEquationArray" | "NumberedEquations"), ___]
            :> {" ", x, " "},
          Cell[x__, ("JSP" | "JPG"), ___] :> {},
          Cell[x__, ___] :> x,
          TextData[x__] :> x,
          RowBox[x__] :> x,
          FormBox[x__, ___] :> x,
          ButtonBox[x__, ___] :> x,
          BoxData[x__] :> x,
          SubsuperscriptBox[x__, y__, z__]
            :> {x, "_", Parenth[y], "^", Parenth[z]},
          SuperscriptBox[x__, y__]
            :> {x, "^", Parenth[y]},
          SubscriptBox[x__, y__]
            :> {x, "_", Parenth[y]},
          FractionBox[x__, y__]
            :> {Parenth[x], "/", Parenth[y]},
          SqrtBox[x__] :> {"sqrt(", x, ")"},
          StyleBox[x__, ___] :> x,
          GridBox[x__, ___]
            :> Riffle[Riffle[#, {" "}]& /@ x, {"; "}],
          UnderoverscriptBox[x__, y__, z__]
            :> {x, "_", Parenth[y], "^", Parenth[z]},
          UnderscriptBox[x__, y__]
            :> {x, "_", Parenth[y]},
          OverscriptBox[x__, y__]
            :> {x, "^", Parenth[y]},
          TagBox[x__, ___] :> x
      }}]
    ],
    TextCharacterReplacementRules];


(******************************************************************************)
(** CreateInputForm **)
CreateInputForm::str = "Incorrect output: `1`";
CreateInputForm[c : Cell[___], opts___] := 
  Developer`UseFrontEnd@Module[{nbObj, newish, nbExpr, expr},
    If[IsManipulateQ[c], " ",
      newish = 1;
      If[newish === 1,
        nbExpr = MathLink`CallFrontEnd[ExportPacket[c, "InputText"]], 
        (* Else *)
        nbObj = NotebookPut@Notebook[{c}];
        FrontEndExecute[FrontEnd`FrontEndToken[nbObj, "SelectAll"]];
        FrontEndExecute[FrontEnd`FrontEndToken[nbObj, "CopySpecial", "InputText"]];
        nbExpr = NotebookGet[ClipboardNotebook[]];
        NotebookClose[nbObj];
      ];
      expr = GetFirstCellContents[nbExpr];
      expr = If[StringQ@expr, 
        expr = Transmogrify`ConvertToString@expr;
        FixedPoint[StringReplace[#, RegularExpression["(\r|\n)+"] :> "\n"]&, expr],
        (* Else *)
        expr];
      If[(Head@expr =!= String) || (StringLength@expr < 1), 
        If[!FAILED,
          FAILED=True;
          Export["CreateInputFormFailed.m", {Global`PageTitleForCreateInputForm, c, nbExpr, expr}];
        ];
        Message[CreateInputForm::str,expr]; $Failed, 
        expr]
  ]];


(******************************************************************************)
(** RemoveEmptyCellGroups **)

RemoveEmptyCellGroups[ expr_, style__String ] := RemoveEmptyCellGroups[ expr, {style} ]

RemoveEmptyCellGroups[ expr_, styles:{__String} ]:= 
  Module[{ex, pos},
    (* This seems to happen from time to time *)
    ex = DeleteCases[expr, Cell[CellGroupData[{}, ___], ___], Infinity];
    (* First we find and delete all appropriately-styled non-inline
       cells that are not the lead cell of a CellGroupData. *)
    pos = Position[ex, Cell[_, Alternatives@@styles, ___], Infinity];
    pos = Select[pos,
        And[
            MemberQ[{CellGroupData, Notebook}, Head@First@Extract[ex, {Drop[#, -2]}]],
            Last[#] =!= 1]&];
    pos = Select[pos,
      (!MatchQ[First[Extract[ex, {#}]], Cell[___, CellTags->"IncludedEmptySection", ___]])&];
    ex = Delete[ex, pos];
    (* Then we delete Cell@CellGroupData expressions containing
       only appropriately-styled lead cells. *)
    While[(
        pos = Position[ex,
          (Cell[CellGroupData[{}, ___], ___] | 
           Cell[CellGroupData[{Cell[_, Alternatives@@styles, ___]}, ___], ___] |
           Cell[CellGroupData[{Cell[_, Alternatives@@styles, ___], Cell[TextData[{}], ___]}, ___], ___]),
          Infinity];
        pos = Select[pos,
          (!MatchQ[First[Extract[ex, {#}]], Cell[___, CellTags->"IncludedEmptySection", ___]])&];
        Length[pos] > 0),
      ex = Delete[ex, pos]
    ];
    ex];


(******************************************************************************)
(** MakeActionMenu **)

Options[MakeActionMenu] = {"Appearance" -> None};

MakeActionMenu[caption_String, style_String, dat_List, opts___?OptionQ] :=
  MakeActionMenu[Cell[TextData[caption]], style, dat, opts];

MakeActionMenu[caption_TextCell, style_String, dat_List, opts___?OptionQ] := 
  Module[{outList = {}, uri, action},
    appear = "Appearance" /. {opts} /. Options[MakeActionMenu];
    ClearAttributes[RuleDelayed, HoldRest];
    Scan[
      Function[{i},
        uri = If[StringMatchQ[dat[[i]][[2]], "paclet*"], dat[[i]][[2]], "paclet:" <> dat[[i]][[2]] ];
        action = ActionXXXX[uri];
        AppendTo[outList, dat[[i]][[1]] :> action];
      ],
      Range[Length[dat]] ];
    SetAttributes[RuleDelayed, HoldRest];
    outList = outList /. ActionXXXX -> Documentation`HelpLookup;
    ActionMenu[Framed[caption], outList, MenuStyle->style, Appearance -> appear, MenuAppearance -> Automatic]
  ];


(******************************************************************************)
(** GetClassificationBoxes **)

GetClassificationBoxes::args = "Incorrect arguments: `1`";

GetClassificationBoxes[nb_String, lang_String:"English", opts:OptionsPattern[]] :=
  GetClassificationBoxes[GetQuietly[nb], lang, opts];

GetClassificationBoxes[nb_Notebook, lang_String:"English", opts:OptionsPattern[]] :=
  GetClassificationBoxes[GetNotebookInfo[nb], lang, opts];

GetClassificationBoxes[info_List, lang_String:"English", opts:OptionsPattern[]] :=
  Module[{type, paclet, pacletQ, flag},
    type = "EntityType" /. info;
    paclet = "PacletName" /. info;
    flag = "Flag" /. info;
    GetClassificationBoxes[type, paclet, flag, lang, opts]
  ];

GetClassificationBoxes["Net Function", ("" | "Mathematica"), _, lang_String:"English", opts:OptionsPattern[]] :=
  Localized["NetFunctionClassification", lang];

GetClassificationBoxes["Character Name", ("" | "Mathematica"), _, lang_String:"English", opts:OptionsPattern[]] :=
  If[TrueQ["Rebranded" /. {opts}], 
  	Localized["CharacterNameClassification_Rebranded", lang], Localized["CharacterNameClassification", lang]];
  	
GetClassificationBoxes["Device Connection", ("" | "Mathematica"), _, lang_String:"English", opts:OptionsPattern[]] :=
  If[TrueQ["Rebranded" /. {opts}], 
  	Localized["DeviceClassification_Rebranded", lang], Localized["DeviceClassification", lang]];
  	
GetClassificationBoxes["Service Connection", ("" | "Mathematica"), _, lang_String:"English", opts:OptionsPattern[]] :=
  If[TrueQ["Rebranded" /. {opts}], 
  	Localized["ServiceClassification_Rebranded", lang], Localized["ServiceClassification", lang]];
  	
GetClassificationBoxes["Embedding Format", ("" | "Mathematica"), _, lang_String:"English", opts:OptionsPattern[]] :=
  If[TrueQ["Rebranded" /. {opts}], 
  	Localized["EmbeddingFormatClassification_Rebranded", lang], Localized["EmbeddingFormatClassification", lang]];
  	
GetClassificationBoxes["Interpreter", ("" | "Mathematica"), _, lang_String:"English", opts:OptionsPattern[]] :=
  If[TrueQ["Rebranded" /. {opts}], 
  	Localized["InterpreterClassification_Rebranded", lang], Localized["InterpreterClassification", lang]];

GetClassificationBoxes["Example", ("" | "Mathematica"), _, lang_String:"English", opts:OptionsPattern[]] :=
  If[TrueQ["Rebranded" /. {opts}], 
  	Localized["ExampleClassification_Rebranded", lang], Localized["ExampleClassification", lang]];

GetClassificationBoxes["File", ("" | "Mathematica"), _, lang_String:"English", opts:OptionsPattern[]]:= 
  If[TrueQ["Rebranded" /. {opts}], 
  	Localized["FileClassification_Rebranded", lang], Localized["FileClassification", lang]];

GetClassificationBoxes["Format", ("" | "Mathematica"), _, lang_String:"English", opts:OptionsPattern[]] :=
  If[TrueQ["Rebranded" /. {opts}], 
  	Localized["FormatClassification_Rebranded", lang], Localized["FormatClassification", lang]];

GetClassificationBoxes[("Front End Object" | "FE Token"),
                       ("" | "Mathematica"),
                       _, lang_String:"English", opts:OptionsPattern[]] :=
  If[TrueQ["Rebranded" /. {opts}], 
  	Localized["FrontEndObjectClassification_Rebranded", lang], Localized["FrontEndObjectClassification", lang]];

GetClassificationBoxes["Guide", "Compatibility Package", _, lang_String:"English", opts:OptionsPattern[]] :=
  If[TrueQ["Rebranded" /. {opts}], 
  	Localized["CompatibilityGuideClassification_Rebranded", lang], Localized["CompatibilityGuideClassification", lang]];

GetClassificationBoxes["HowTo", ("" | "Mathematica"), _, lang_String:"English", opts:OptionsPattern[]] :=
  If[TrueQ["Rebranded" /. {opts}], 
  	Localized["HowToClassification_Rebranded", lang], Localized["HowToClassification", lang]];
  
GetClassificationBoxes["Indicator", ("" | "Mathematica"), _, lang_String:"English", opts:OptionsPattern[]] :=
  If[TrueQ["Rebranded" /. {opts}], 
  	Localized["IndicatorClassification_Rebranded", lang], Localized["IndicatorClassification", lang]];
  
GetClassificationBoxes["MathLink C Function", "Mathematica", "ObsoleteFlag", lang_String:"English", opts:OptionsPattern[]] :=
  If[TrueQ["Rebranded" /. {opts}], 
  	Localized["ObsoleteMathLinkClassification_Rebranded", lang], Localized["ObsoleteMathLinkClassification", lang]];

GetClassificationBoxes["MathLink C Function", "Mathematica", _, lang_String:"English", opts:OptionsPattern[]] :=
  If[TrueQ["Rebranded" /. {opts}], 
  	Localized["MathLinkClassification_Rebranded", lang], Localized["MathLinkClassification", lang]];
  
GetClassificationBoxes["PythonLink Python Function", "Mathematica", _, lang_String:"English", opts:OptionsPattern[]] :=
  If[TrueQ["Rebranded" /. {opts}], 
  	Localized["PythonLinkClassification_Rebranded", lang], Localized["PythonLinkClassification", lang]];
  
GetClassificationBoxes["LibraryLink C Function", "Mathematica", _, lang_String:"English", opts:OptionsPattern[]] :=
  If[TrueQ["Rebranded" /. {opts}], 
  	Localized["CharacterNameClassification_Rebranded", lang], Localized["LibraryLinkClassification", lang]];

GetClassificationBoxes["LibraryLink C Function", "LibraryLink", _, lang_String:"English", opts:OptionsPattern[]] :=
  If[TrueQ["Rebranded" /. {opts}], 
  	Localized["LibraryLinkClassification_Rebranded", lang], Localized["LibraryLinkClassification", lang]];

GetClassificationBoxes["Menu Item", ("" | "Mathematica"), _, lang_String:"English", opts:OptionsPattern[]] :=
  If[TrueQ["Rebranded" /. {opts}], 
  	Localized["MenuItemClassification_Rebranded", lang], Localized["MenuItemClassification", lang]];

GetClassificationBoxes["Message", ("" | "Mathematica"), _, lang_String:"English", opts:OptionsPattern[]] :=
  If[TrueQ["Rebranded" /. {opts}], 
  	Localized["MessageClassification_Rebranded", lang], Localized["MessageClassification", lang]];

GetClassificationBoxes["Note", ("" | "Mathematica"), _, lang_String:"English", opts:OptionsPattern[]] :=
  If[TrueQ["Rebranded" /. {opts}], 
  	Localized["NoteClassification_Rebranded", lang], Localized["NoteClassification", lang]];

GetClassificationBoxes["Overview", ("" | "Mathematica"), _, lang_String:"English", opts:OptionsPattern[]] :=
  If[TrueQ["Rebranded" /. {opts}], 
  	Localized["OverviewClassification_Rebranded", lang], Localized["OverviewClassification", lang]];
  	
GetClassificationBoxes["Program", ("" | "Mathematica"), _, lang_String:"English", opts:OptionsPattern[]] :=
  If[TrueQ["Rebranded" /. {opts}], 
  	Localized["ProgramClassification_Rebranded", lang], Localized["ProgramClassification", lang]];

GetClassificationBoxes["Symbol", "Mathematica", "PreviewFlag", lang_String:"English", opts:OptionsPattern[]]:= 
  If[TrueQ["Rebranded" /. {opts}], 
  	Localized["PreviewClassification_Rebranded", lang], Localized["PreviewClassification", lang]];

GetClassificationBoxes["Symbol", "Mathematica", "AwaitingFutureDesignReviewFlag", lang_String:"English", opts:OptionsPattern[]] :=
  If[TrueQ["Rebranded" /. {opts}], 
  	Localized["FutureDesignReviewClassification_Rebranded", lang], Localized["FutureDesignReviewClassification", lang]];

GetClassificationBoxes["Symbol", "Mathematica", "ObsoleteFlag", lang_String:"English", opts:OptionsPattern[]] :=
  If[TrueQ["Rebranded" /. {opts}], 
  	Localized["ObsoleteSymbolClassification_Rebranded", lang], Localized["ObsoleteSymbolClassification", lang]];

GetClassificationBoxes["Symbol", "Mathematica", _, lang_String:"English", opts:OptionsPattern[]] :=
  If[TrueQ["Rebranded" /. {opts}], 
  	Localized["SymbolClassification_Rebranded", lang], Localized["SymbolClassification", lang]];
  
GetClassificationBoxes["Symbol", "", _, lang_String:"English", opts:OptionsPattern[]] :=
  If[TrueQ["Rebranded" /. {opts}], 
  	Localized["SymbolClassification_Rebranded", lang], Localized["SymbolClassification", lang]];

(*GetClassificationBoxes["Symbol", paclet_, _, lang_String:"English"] :=
  Localized["PacletSymbolClassification", lang][paclet]; *)

GetClassificationBoxes["System Program", ("" | "Mathematica"), _, lang_String:"English", opts:OptionsPattern[]] :=
  If[TrueQ["Rebranded" /. {opts}], 
  	Localized["SystemProgramClassification_Rebranded", lang], Localized["SystemProgramClassification", lang]];

GetClassificationBoxes[("Tutorial" | "Upgrade Information" | "Compatibility Package"),
                       "Compatibility Package", _,
                       lang_String:"English", opts:OptionsPattern[]] :=
  If[TrueQ["Rebranded" /. {opts}], 
  	Localized["CompatibilityInformationClassification_Rebranded", lang], Localized["CompatibilityInformationClassification", lang]];
  
  
GetClassificationBoxes["Tutorial", ("" | "Mathematica"), "ObsoleteFlag", lang_String:"English", opts:OptionsPattern[]] :=
  If[TrueQ["Rebranded" /. {opts}], 
  	Localized["ObsoleteTutorialClassification_Rebranded", lang], Localized["ObsoleteTutorialClassification", lang]];

GetClassificationBoxes["Tutorial", ("" | "Mathematica"), _, lang_String:"English", opts:OptionsPattern[]] :=
  If[TrueQ["Rebranded" /. {opts}], 
  	Localized["TutorialClassification_Rebranded", lang], Localized["TutorialClassification", lang]];

GetClassificationBoxes["Guide", ("" | "Mathematica"), None, lang_String:"English", opts:OptionsPattern[]] :=
  If[TrueQ["Rebranded" /. {opts}], 
  	Localized["MathematicaGuideClassification_Rebranded", lang], Localized["MathematicaGuideClassification", lang]];

GetClassificationBoxes["ExamplePage", _, _, lang_String:"English", opts:OptionsPattern[]] :=
  If[TrueQ["Rebranded" /. {opts}], 
  	Localized["ExamplePage_Rebranded", lang], Localized["ExamplePage", lang]];

(* TODO Handle this catch all routine for localizing M- calssifications *)
GetClassificationBoxes[str_String, ("" | "Mathematica"), None, lang_String:"English", opts:OptionsPattern[]] :=
  TextData[{
    StyleBox["Mathematica", Rule[FontSlant, "Italic"]], " ", str}];

GetClassificationBoxes[type_, paclet_, flag_, lang_String:"English", opts:OptionsPattern[]]:= 
  Module[{pn=paclet, typestr},
    typestr = Localized[type, lang];
    If [(paclet =!= "") && (paclet =!= None),
      (
        pn = If[StringMatchQ[pn, "*Package"], {
            StyleBox[
              AddSpaceToCamelCase@StringReplace[pn, RegularExpression["\\s+Package\\s*$"]->""],
              FontSlant->"Italic"],
            Localized["Package", lang]
          },
          StyleBox[AddSpaceToCamelCase@pn , FontSlant->"Italic"] ];
        TextData[Riffle[Flatten@{flag, pn, typestr} /. {"ObsoleteFlag"->"Obsolete ", None|"" -> Sequence[]}, " "]]
      ),
      StringJoin @@ Riffle[Flatten@{flag, pn, typestr} /. {"ObsoleteFlag"->"Obsolete ", None|"" -> Sequence[]}, " "]
  ]];

GetClassificationBoxes[a___]:= (Message[GetClassificationBoxes::args, a]; $Failed);


(******************************************************************************)
(* Remove single and double quotes for Web meta description *)
CleanSummaryForWeb[{desc_String}]:= (
   {StringReplace[desc, {"'" -> "&apos;", "\"" -> "&quot;"}]}
   );
CleanSummaryForWeb[desc_String]:= (
   StringReplace[desc, {"'" -> "&apos;", "\"" -> "&quot;"}]
   );
(* end CleanSummaryForWeb *)


(******************************************************************************)
(* Igor's code for converting consecutive ButtonBoxes/StyleBoxes with the same ButtonData.*)
HyperlinkEqualityTest[
   StyleBox[ButtonBox[con1_, buttonOpts1___], styopts1___], 
   StyleBox[ButtonBox[con2_, buttonOpts2___], 
    styopts2___]] := (HyperlinkEqualityTest[
    ButtonBox[con1, buttonOpts1], ButtonBox[con2, buttonOpts2]]);

HyperlinkEqualityTest[ButtonBox[con1_, buttonOpts1___], 
   StyleBox[ButtonBox[con2_, buttonOpts2___], 
    styopts2___]] := (HyperlinkEqualityTest[
    ButtonBox[con1, buttonOpts1], ButtonBox[con2, buttonOpts2]]);

HyperlinkEqualityTest[
   StyleBox[ButtonBox[con1_, buttonOpts1___], styopts1___], 
   ButtonBox[con2_, buttonOpts2___]] := (HyperlinkEqualityTest[
    ButtonBox[con1, buttonOpts1], ButtonBox[con2, buttonOpts2]]);

HyperlinkEqualityTest[ButtonBox[contents1_, buttonOpts1___], 
   ButtonBox[contents2_, 
    buttonOpts2___]] := (MemberQ[{buttonOpts1}, 
     BaseStyle -> sty_/;MemberQ[{sty},"Hyperlink" | "Link",Infinity]] && 
    MemberQ[{buttonOpts2}, 
     BaseStyle -> 
      sty_/;MemberQ[{sty},"Hyperlink" | "Link",Infinity]] && ((ButtonData /. {buttonOpts1} /. 
        ButtonData :> contents1) === (ButtonData /. {buttonOpts2} /. 
        ButtonData :> contents2)));

HyperlinkEqualityTest[otherwise__] = False;

(*ApplyStyleBoxInside[]-Takes StyleBoxes that are outside a ButtonBox \
and applies it to the contents of the ButtonBox (places it \
"inside").IgorA.Dec/2007.*)

ApplyStyleBoxInside[
   StyleBox[ButtonBox[contents_, buttonopts___], 
    styopts___]] := (ButtonBox[StyleBox[contents, styopts], 
    buttonopts]);

ApplyStyleBoxInside[
   StyleBox[ButtonBox[contents_, buttonopts___], style : "", 
    styopts___]] := (ButtonBox[StyleBox[contents, style, styopts], 
    buttonopts]);

ApplyStyleBoxInside[else__] := else;

(*MergeButtonBox[]-Extracts the BaseStyle and ButtonData of the first \
ButtonBox extracts the contents of all ButtonBoxes (which already \
have StyleBoxes wrapped around them when necessary),and returns a \
single,new ButtonBox.IgorA.Dec/2007.*)

MergeButtonBoxes[cells_List] := 
  Module[{buttons, firstButton, contents, bstyle, bdata}, 
   buttons = ApplyStyleBoxInside /@ cells;
   firstButton = First@Cases[buttons, _ButtonBox, Infinity];
   bstyle = Options[firstButton, BaseStyle] /. {{} -> "Hyperlink"};
   bdata = Options[firstButton, ButtonData] /. {{} -> "#"};
   contents = Cases[buttons, ButtonBox[con_, bttnopts___] :> con];
   ButtonBox[contents, Sequence @@ Flatten[{bstyle, bdata}]]];
   
   
 ConvertTextData[TextData[cont_]] := 
  Module[{elems, splitelems, newelems},
  	(*ensure the content is a list*)
   elems = If[ListQ@cont, cont, {cont}];
   (*split the list & test hyperlinks*)
   splitelems = Split[elems, (HyperlinkEqualityTest[#1, #2] &)];
   (*merge multistyle links*)
   If[Max[Length /@ splitelems] > 1,
    newelems = 
     Flatten[(If[Length[#] > 1, MergeButtonBoxes[#], #] &) /@ 
       splitelems, 1];
    TextData[
     If[ListQ@newelems, newelems, {newelems}]
     (*leave all other cases*)
      (*TextData[If[ListQ@elems, elems, {elems}]] *)
     ],TextData[cont] 
    ]
   ];

(*end Igor's code.*)

(*  Utility to italicize Mathematica in autogenerated MoreAbouts *)
(* Commenting out since a newer, more generalized version exists just below  --brianv

 ItalicizeMoreAboutKeyWords[moreAboutCell_Cell, keyWord_String] := 
  Module[{keywordLength, output},
   keywordLength = StringLength[keyWord];
   output = 
    moreAboutCell /. {Cell[TextData[ButtonBox[a_String/;StringMatchQ[a, keyWord~~WordBoundary~~__],
              b__]], "MoreAbout", c__]:> 
       Cell[TextData[ButtonBox[{StyleBox[keyWord, FontSlant -> "Italic"], StringDrop[a,11]}, b]], "MoreAbout", c],
      Cell[TextData[ButtonBox[a_String/;StringMatchQ[a, __~~WordBoundary~~keyWord~~WordBoundary~~__], 
          b__]], "MoreAbout", c__]:> 
      Cell[TextData[ButtonBox[{First[StringSplit[a, WordBoundary~~keyWord~~WordBoundary]],
       	StyleBox[keyWord, FontSlant -> "Italic"], 
           Last[StringSplit[a, WordBoundary~~keyWord~~WordBoundary]]}, b]], "MoreAbout", c],
      Cell[TextData[ButtonBox[a_String/;StringMatchQ[a, __~~WordBoundary~~keyWord], 
          b__]], "MoreAbout", c__]:> 
       Cell[TextData[ButtonBox[{StringDrop[a, -keywordLength], StyleBox[keyWord, FontSlant -> "Italic"]}, b]], 
        "MoreAbout", c]};
   output
   ]; *)

ItalicizeMoreAboutKeyWords[macell_Cell, kws_String] :=
 ItalicizeMoreAboutKeyWords[macell, {kws}]

ItalicizeMoreAboutKeyWords[macell_Cell, kws_List] := 
 macell /. 
  Cell[TextData[ButtonBox[s_, rest___]], more___] /; MemberQ[{more}, "MoreAbout"] :> 
   Cell[TextData[
     ButtonBox[
      List @@ StringReplace[s, 
        MapThread[
         Rule, {StringExpression[WordBoundary, #, WordBoundary] & /@ 
           kws, StyleBox[#, FontSlant -> "Italic"] & /@ kws}]], 
      rest]], more]

(******************************************************************************)
(** GetJoinedButtonCaptions - Internal **)
GetJoinedButtonCaptions[expr_] := 
  Module[{boxInstances, link, expr2}, 
  	
  	expr2 = DeleteCases[expr, _["ReplacedText", ___], Infinity];

    boxInstances = 
      Cases[expr2, ButtonBox[c_, ___, ButtonData -> l_, ___] :> {ConvertToString[c], l}, Infinity];
	link = Last[First[boxInstances]];
	
    If[boxInstances === {}, Return[{}]];
    (* return list of pairs { {title, uri}, {...} } *)
    {StringJoin[First[#] & /@ boxInstances], 
    	Which[
    		MatchQ[link, {URL[_], _}], link[[1,1]],
    		True, StringReplace[link, "paclet:" :> ""]
    	]}
  ];


(******************************************************************************)
(** GetMoreAboutSectionList **)

GetMoreAboutSectionList[nb_String, opts___?OptionQ] := 
  GetMoreAboutSectionList[GetQuietly@nb, opts];

GetMoreAboutSectionList[nb_Notebook, opts___?OptionQ] := 
  Module[{cell, expr, url, c, moreAbouts}, 
    moreAbouts = Cases[nb, Cell[__, "MoreAbout", ___], Infinity];
    GetJoinedButtonCaptions /@moreAbouts
  ];


(******************************************************************************)
(** GetSeeAlsoSectionList **)

Options[GetSeeAlsoSectionList] = {"IncludePaclets"->True, "Language"->"English", "BuildMode"->"Normal", "PacletName" -> "Mathematica"};

GetSeeAlsoSectionList[nb_String, opts___?OptionQ] :=
  GetSeeAlsoSectionList[Get@nb, opts]; 

GetSeeAlsoSectionList[nb_Notebook, opts___?OptionQ] :=
  Module[{cell, paclets, lst, tlst, seeAlsoGroup, lang, uri, buildmode, pacletName},
    paclets = "IncludePaclets" /. {opts} /. Options[GetSeeAlsoSectionList];
    lang = "Language" /. {opts} /. Options[GetSeeAlsoSectionList];
    buildmode = "BuildMode" /. {opts} /. Options[GetSeeAlsoSectionList];
    pacletName = "PacletName" /. {opts} /. Options[GetSeeAlsoSectionList];
      
  BuildLog["-- Remove SeeAlso links to flagged notebooks"];
  seeAlsoGroup = DocumentationBuild`Make`Private`GetSeeAlsoGroup[nb, lang];
  Scan[
    Function[{pos}, Module[{cell, origcells, newcells},
      cell = Extract[seeAlsoGroup, pos];
      origcells = Cases[cell,
        Cell[BoxData[_ButtonBox, ___], ___]
          | Cell[BoxData[StyleBox[_ButtonBox, ___], ___], ___] |
          Cell[BoxData[TemplateBox[___], ___], ___],
        Infinity];
      (* TODO-andy If ButtonData exists, it has to be used! *)
      If[buildmode==="Future",
      	newcells = origcells,
      newcells = Select[origcells,
        Function[{bbcell}, Module[{sym, sym2},
          If[MatchQ[bbcell,
                Cell[___, "FutureExample", ___]
                  | Cell[BoxData[StyleBox[___, "FutureExample", ___], ___], ___]],
              False,
            (* Else *)
            sym = Cases[bbcell, _ButtonBox, Infinity, 1];
            sym2 = Cases[bbcell, _TemplateBox, Infinity, 1];
            uri = Cases[sym, ButtonData~(Rule|RuleDelayed)~uri_String :> uri, Infinity, 1];
            If[ (Length @ uri > 0) && StringMatchQ[First@uri, "paclet:*"],
               (* If Button contains ButtonData, use it *)
               DocumentationBuild`Make`Private`ShippingFunctionQ[First@uri, pacletName],
               (* check if the TemplateBox has a URI *)
               If[StringMatchQ[sym2[[1,1,2]],"paclet:*"], uri = sym2[[1,1,2]]; 
                 DocumentationBuild`Make`Private`ShippingFunctionQ[uri, pacletName],
               (* no ButtonData so need to create it *)
               sym = Transmogrify`ConvertToString[sym[[1, 1]]];
               sym = StringReplace[sym, "\""->""];
               sym = "paclet:"<>If[ FreeQ[pacletName, "Mathematica" | ""], pacletName<>"/", ""]<>"ref/" <> sym;
               DocumentationBuild`Make`Private`ShippingFunctionQ[sym, pacletName]
               ]
            ]
      ]]]]];
      If[Length[origcells] =!= Length[newcells],
        seeAlsoGroup = ReplacePart[seeAlsoGroup, pos -> Join[
          Cell[TextData[
            Flatten[Riffle[newcells, {{
              "\[NonBreakingSpace]",
              StyleBox["\[MediumSpace]\[FilledVerySmallSquare]\[MediumSpace]", "InlineSeparator"],
              " "
            }}, 2 ]]
          ]],
          Rest[cell]
      ]]];
    ]],
    Position[seeAlsoGroup, Cell[_TextData, "SeeAlso", ___], Infinity]
  ];
    cell = Cases[seeAlsoGroup, Cell[__, "SeeAlso", ___], Infinity, 1];
    lst = Cases[cell,
      (ButtonBox[c_, o___] :>
        {Transmogrify`ConvertToString[c], 
          If[MemberQ[{o}, ButtonData -> _]||MemberQ[{o}, ButtonData :> _], 
            ButtonData /. Cases[{o}, _~(Rule|RuleDelayed)~_],
            "paclet:ref/" <> Transmogrify`ConvertToString[c] ] }),
      Infinity];
     If[lst === {}, lst = Cases[cell,TemplateBox[c_,___]:>{Transmogrify`ConvertToString[c[[1]]],c[[2]]}, Infinity]];
      
    If[paclets === True, lst, First /@ lst]
  ];


(******************************************************************************)
(** GetTutorialFunctionList **)

Options[GetTutorialFunctionList]= {"IncludePaclets"->False};

GetTutorialFunctionList[nb_, opts___?OptionQ]:= GetTutorialFunctionList[Get@nb, opts]; 
GetTutorialFunctionList[nbExpr_Notebook, opts___?OptionQ]:= 
Module[{cell, paclets, defBoxes, funcURL, c, syms}, 
  paclets = "IncludePaclets" /. {opts} /. Options[GetTutorialFunctionList];
  defBoxes = Cases[nbExpr, Cell[_, "DefinitionBox", ___], Infinity];
  
  defBoxes = defBoxes /. a_TextData -> ConvertTextData[a];
  
  syms = Union@Cases[defBoxes, ButtonBox[c_String, ___, BaseStyle->"Link", ___, ButtonData~(Rule|RuleDelayed)~funcURL_String, ___] :> {c,StringReplace[funcURL, "paclet:" -> ""]}, Infinity];
  If[syms =!= {}, syms,
    Union[Cases[defBoxes, TemplateBox[{n_, uri_},___] :> {n, StringReplace[uri, "paclet:"->""]}, Infinity]]
  ]
];


(******************************************************************************)
(** GetGuideFunctionList **)

Options[GetGuideFunctionList]= {"IncludePaclets"->False};

GetGuideFunctionList[nb_, opts___?OptionQ]:= GetGuideFunctionList[Get@nb, opts]; 
GetGuideFunctionList[nbExpr_Notebook, opts___?OptionQ]:= 
Module[{cell, paclets, guideText, funcURL, c}, 
  paclets = "IncludePaclets" /. {opts} /. Options[GetGuideFunctionList];
  guideText = DeleteCases[
  Cases[nbExpr, Cell[_, "GuideText"|"InlineGuideFunctionListing", ___], Infinity], 
  	StyleBox[_ButtonBox, "FutureExample",___], Infinity];
  
  guideText = guideText /. a_TextData -> ConvertTextData[a];
  
 (* 4/17/14 -- edit to make sure future, etc. ref links don't show up in the Functions 
    drop down in guides *) 
 Union@Select[DeleteCases[
 Cases[guideText, ButtonBox[c_String, ___, 
    BaseStyle -> "Link" | "NewInVersionLink" | {"Link", a_}/;a=!="BrighterFlag", ___, 
    ButtonData~(Rule | RuleDelayed)~funcURL_String, ___]/;StringMatchQ[c,RegularExpression["\\w+"]] :> {c, 
    StringReplace[funcURL, "paclet:" -> ""]}, Infinity], {"_"|"__"|"___", _String}],
    MatchQ["Flag" /. ("paclet:"<>#[[2]] /. DocumentationBuild`Export`$CoreInformation["English"]), None|"None"]&]
];


(******************************************************************************)
(** GetTutorialMoreAboutList **)

Options[GetTutorialMoreAboutList] = {"IncludePaclets"->False};

GetTutorialMoreAboutList[nb_String, opts___?OptionQ] :=
  GetTutorialMoreAboutList[Get@nb, opts];

GetTutorialMoreAboutList[nb_Notebook, opts___?OptionQ] :=
  Module[{cell, expr, url, c}, 
    expr = Cases[nb, Cell[__, "TutorialMoreAbout", ___], Infinity];
    expr = If[opts==="", 
    	(* for web include all cells *)  expr, 
    	(* for nb delete invisible cells *)   DeleteCases[expr, Cell[a__, b_String, c__, 
 CellElementSpacings -> {"ClosedCellHeight" -> 0}, d___], Infinity]];
    GetJoinedButtonCaptions /@expr
  ];

(******************************************************************************)
(** GetTutorialMoreAboutSectionList **)

Options[GetTutorialMoreAboutSectionList] = {"IncludePaclets"->False};

GetTutorialMoreAboutSectionList[nb_String, opts___?OptionQ] :=
  GetTutorialMoreAboutSectionList[GetQuietly@nb, opts];

GetTutorialMoreAboutSectionList[nb_Notebook, opts___?OptionQ] :=
  Module[{expr, c}, 
    expr = Cases[nb, Cell[__, "TutorialMoreAbout", ___], Infinity];
    expr = If[opts==="", 
    	(* for web include all cells *)  expr, 
    	(* for nb delete invisible cells *)   DeleteCases[expr, Cell[a__, b_String, c__, 
 CellElementSpacings -> {"ClosedCellHeight" -> 0}, d___], Infinity]];
    GetJoinedButtonCaptions /@expr
  ];
  
(******************************************************************************)
(** GetGuideMoreAboutSectionList **)

Options[GetGuideMoreAboutSectionList] = {"IncludePaclets"->False};

GetGuideMoreAboutSectionList[nb_String, opts___?OptionQ] :=
  GetGuideMoreAboutSectionList[GetQuietly@nb, opts];

GetGuideMoreAboutSectionList[nb_Notebook, opts___?OptionQ] :=
  Module[{expr}, 
    expr = Cases[nb, Cell[__, "GuideMoreAbout" | "GuideMoreAboutSub", ___], Infinity];
    expr = If[opts==="", 
    	(* for web include all cells *)  expr, 
    	(* for nb delete invisible cells *)   DeleteCases[expr, Cell[a__, b_String, c__, 
 CellElementSpacings -> {"ClosedCellHeight" -> 0}, d___], Infinity]];
    GetJoinedButtonCaptions /@expr
  ];
  
  
(******************************************************************************)
(** GetFeaturedExampleMoreAboutSectionList **)

Options[GetFeaturedExampleMoreAboutSectionList] = {"IncludePaclets"->False};

GetFeaturedExampleMoreAboutSectionList[nb_String, opts___?OptionQ] :=
  GetFeaturedExampleMoreAboutSectionList[GetQuietly@nb, opts];

GetFeaturedExampleMoreAboutSectionList[nb_Notebook, opts___?OptionQ] :=
  Module[{expr}, 
    expr = Cases[nb, Cell[__, "FeaturedExampleMoreAbout", ___], Infinity];
    expr = If[opts==="", 
    	(* for web include all cells *)  expr, 
    	(* for nb delete invisible cells *)   DeleteCases[expr, Cell[a__, b_String, c__, 
 CellElementSpacings -> {"ClosedCellHeight" -> 0}, d___], Infinity]];
    GetJoinedButtonCaptions /@expr
  ];
  
(******************************************************************************)
(** GetFeaturedExampleTutorialsSectionList **)

Options[GetFeaturedExampleTutorialsSectionList] = {"IncludePaclets"->False};

GetFeaturedExampleTutorialsSectionList[nb_String, opts___?OptionQ] :=
  GetFeaturedExampleTutorialsSectionList[GetQuietly@nb, opts];

GetFeaturedExampleTutorialsSectionList[nb_Notebook, opts___?OptionQ] :=
  Module[{expr}, 
    expr = Cases[nb, Cell[__, "FeaturedExampleTutorial", ___], Infinity];
    expr = If[opts==="", 
    	(* for web include all cells *)  expr, 
    	(* for nb delete invisible cells *)   DeleteCases[expr, Cell[a__, b_String, c__, 
 CellElementSpacings -> {"ClosedCellHeight" -> 0}, d___], Infinity]];
    GetJoinedButtonCaptions /@expr
  ];
  
  
(******************************************************************************)
(** GetRelatedTutorialsSectionList **)

Options[GetRelatedTutorialsSectionList] = {"IncludePaclets"->False};

GetRelatedTutorialsSectionList[nb_String, opts___?OptionQ] :=
  GetRelatedTutorialsSectionList[Get@nb, opts];

GetRelatedTutorialsSectionList[nb_Notebook, opts___?OptionQ] :=
  Module[{expr, c}, 
    expr = Cases[nb, Cell[__, "RelatedTutorials", ___], Infinity];
    expr = If[opts==="", 
    	(* for web include all cells *)  expr, 
    	(* for nb delete invisible cells *)   DeleteCases[expr, Cell[a__, b_String, c__, 
 CellElementSpacings -> {"ClosedCellHeight" -> 0}, d___], Infinity]];
    GetJoinedButtonCaptions /@expr
  ];


(******************************************************************************)
(** GetTutorialsSectionList **)

Options[GetTutorialsSectionList] = {"IncludePaclets"->False};

GetTutorialsSectionList[nb_String, opts___?OptionQ] :=
  GetTutorialsSectionList[Get@nb, opts];

GetTutorialsSectionList[nb_Notebook, opts___?OptionQ] :=
  Module[{expr, c}, 
    expr = Cases[nb, Cell[__, "Tutorials", ___], Infinity];
    expr = If[opts==="", 
    	(* for web include all cells *)  expr, 
    	(* for nb delete invisible cells *)   DeleteCases[expr, Cell[a__, b_String, c__, 
 CellElementSpacings -> {"ClosedCellHeight" -> 0}, d___], Infinity]];
    GetJoinedButtonCaptions /@expr
  ];
  
  
(******************************************************************************)
(** GetGuideTutorialsSectionList **)

Options[GetGuideTutorialsSectionList] = {"IncludePaclets"->False};

GetGuideTutorialsSectionList[nb_String, opts___?OptionQ] :=
  GetGuideTutorialsSectionList[Get@nb, opts];

GetGuideTutorialsSectionList[nb_Notebook, opts___?OptionQ] :=
  Module[{expr}, 
    expr = Cases[nb, Cell[__, "GuideTutorial", ___], Infinity];
    expr = If[opts==="", 
    	(* for web include all cells *)  expr, 
    	(* for nb delete invisible cells *)   DeleteCases[expr, Cell[a__, b_String, c__, 
 CellElementSpacings -> {"ClosedCellHeight" -> 0}, d___], Infinity]];
    GetJoinedButtonCaptions /@expr
  ];

(******************************************************************************)
(** GetGuideHowTosList **)

GetGuideHowTosList[nb_String] :=
  GetGuideHowTosList[Get[nb]];
  
GetGuideHowTosList[nb_Notebook] :=
  Module[{expr},
  	expr = Cases[nb, 
      cell : Cell[c_, ___, "GuideRelatedLinks", ___] /; 
        MemberQ[{c}, (s_String /; StringMatchQ[s, "paclet:*howto/*"]), 
          Infinity] :> 
        {Transmogrify`ConvertToString[cell], 
          First[Cases[c, str_String /; StringMatchQ[str, "paclet:*howto/*"], 
           Infinity]]}, 
      Infinity];
    expr
  ];
  	


(******************************************************************************)
(** GetGuidesDirectory **)

GetGuidesDirectory[path_String, language_String: "English"] :=
Module[{dirs, pre},
  pre = If[StringMatchQ[path, "/*"], "/", ""];
  dirs = Join[{pre}, StringSplit[path, "\\" | "/"]];
  Which[
    Last@dirs === "Mathematica", 
		Return@ToFileName[ Join[dirs, {"Documentation", language, "System", "Guides"}]],
	Last@dirs === language, 
		Return@ToFileName[Join[dirs, {"Guides"}]],
	Last@dirs === "Documentation", 
		Return@ToFileName[Join[dirs, {language, "Guides"}]],
	True, 
  path]
  ];

(******************************************************************************)
(** GetTutorialsDirectory **)

GetTutorialsDirectory[path_String, language_String: "English"] :=
Module[{dirs, pre},
  pre = If[StringMatchQ[path, "/*"], "/", ""];
  dirs = Join[{pre}, StringSplit[path, "\\" | "/"]];
  Which[
    Last@dirs === "Mathematica", 
		Return@ToFileName[ Join[dirs, {"Documentation", language, "System", "Tutorials"}]],
	Last@dirs === language, 
		Return@ToFileName[Join[dirs, {"Tutorials"}]],
	Last@dirs === "Documentation", 
		Return@ToFileName[Join[dirs, {language, "Tutorials"}]],
	True, 
  path]
  ];


(******************************************************************************)
(** GetOverviewButtonLinks **)

(*  Get overview tagging rules *)
GetOverviewButtonLinks[uri_String, language___String:"English"]:=
Module[{info, overview, chapters, pos, prev, next, prevnext, cell},
  (* Create overview links *)
  info = DocumentationBuild`Info`GetSourceInformation[uri, language];
  If[OptionQ[info] && MemberQ[info, _["Overview", _]],
    (*  make links *)
    overview = "Overview" /. ("Links" /. info);
    chapters = "Chapters" /. ("Links" /.
        ( DocumentationBuild`Info`GetSourceInformation[overview, language] ) );
    chapters = Cases[chapters, {_String, _String}, Infinity];
    pos = Position[chapters, {_, uri}][[1, 1]];
    prev = If[pos =!= 1,
      ButtonBox[StyleBox["\[FilledLeftTriangle]", "OverviewNavText"],
          BaseStyle -> "Link",
          ButtonData -> chapters[[pos - 1, 2]] ],
        None];
    next = If[pos < Length[chapters],
        ButtonBox[StyleBox["\[FilledRightTriangle]", "OverviewNavText"],
          BaseStyle -> "Link",
          ButtonData -> chapters[[pos + 1, 2]] ],
        None];
    cell = 
        Cell[BoxData[
          ButtonBox[StyleBox[
            If[Head[#] === TextData, ReplacePart[#, {0} -> RowBox], #]&[
              "Title" /. ( DocumentationBuild`Info`GetSourceInformation[overview, language] )], "OverviewNavText"],
            BaseStyle -> "Link",
            ButtonData -> overview]
          ], "Text", FontFamily->"Verdana"];
    prevnext = Cell[TextData[Join[
          If[prev =!= None, {prev}, {}],
          If[prev =!= None && next =!= None, {"\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\[ThickSpace]"}, {}],
          If[next =!= None, {next}, {}]
        ]], "Text", FontFamily->"Verdana"];
     {"toc" -> cell, "prevnext" -> prevnext
     },
     (* No match, return empty list *)
     {}
   ]
];



(******************************************************************************)
(** MakeLinkTrailExpr - Internal **)

MakeLinkTrailCellGroup[ts_List, language_String:"English", opts_List:{}] := 
Module[{ trails=ts, buttons},
(*	trails = trails //. {"paclet:guide/Mathematica", a___} :> {a}; *)
	LogIt["-- Make link trail"];
	LogIt["   Trails: " <> ToString[trails]];
	(* turn list of paclet:* into list of ButtonBoxes *)
	buttons = Flatten@{Map[Function[pacletURI, 
		bb = MakeButtonBoxFromPacletURI[pacletURI, language]; 
				If[bb =!= $Failed, {bb, StyleBox[" > ", "LinkTrailSeparator"]}, $Failed ] ], Flatten@{#}]} & /@ trails;
	(* return if $Failed when making a Button *)
	If[ !FreeQ[buttons, $Failed], 
		LogIt["   Failed!"]; Return@{} ];
	(* if no list of buttons, return *)
	If[ Flatten@{buttons} === {}, Return@{} ];
	(* return cell expr *)	
	Cell[TextData[Flatten@{ buttons }], "LinkTrail", Sequence@@Flatten@{opts, CellID->0} ]
	];
	
MakeLinkTrailExpr[u_String /; !StringMatchQ[u, "paclet:*"], language_String:"English"]:= 
	MakeLinkTrailExpr["paclet:"<>u, language];
MakeLinkTrailExpr[u_String, language_String:"English"]:= 
Module[{trails, count=0, expr, numTrails}, 
	LogIt["-- MakeLinkTrailExpr: "<>u];
	(* trails = LinkTrailFromPacletURI[u]; *)
	trails = DocumentationBuild`Info`GetCoreInformationLinkTrail[u, language];
	(* three trail maximum (as of 10/7/08 -- was 2 max) *)
	trails = If[Length@trails > 3, Take[trails, 3], trails];
	numTrails = Length@trails;
	LogIt["-- Number of trails: "<>ToString@numTrails];
	(* adjust appearance if more the one link trail *)
	
	expr = 
	Which[numTrails === 1, MakeLinkTrailCellGroup[ trails, language],
		  numTrails === 2, Flatten@{MakeLinkTrailCellGroup[ trails[[1]], language, {CellFrame->False, CellMargins->{{Inherited,Inherited},{-3,0}} }], MakeLinkTrailCellGroup[ trails[[2]], language]},
		  numTrails >= 3, Flatten@{Map[MakeLinkTrailCellGroup[ #, language, {CellFrame->False, CellMargins->{{Inherited,Inherited},{-3,0}} }]&, Drop[trails,-1]], MakeLinkTrailCellGroup[ trails[[-1]], language]}
	];
	
		(* return link trail cell *)
	{Cell[CellGroupData[Flatten@{ expr }]]}
];
MakeLinkTrailExpr[u_, ___]:= (Message[MakeLinkTrailExpr::args, u]; "");
MakeLinkTrailExpr::args = "Paclet url expected, but found `1`";


(******************************************************************************)
(** LinkTrailFromPacletURI **)
(*
	LinkTrailFromPacletURI 
	in: LinkTrailFromPacletURI["pacledt:ref/Sin", {optional primary guide list}]
	out: {{"paclet:guide/Mathematica", "paclet:guide/TrigonometricFunctions"}}
*)
LinkTrailFromPacletURI[u_String]:= 
	LinkTrailFromPacletURI[ u, DocumentationBuild`Export`$CoreInformation["English"] ];
LinkTrailFromPacletURI[u_String /; !StringMatchQ[u, "paclet:*"], primaryGuides_]:= 
	LinkTrailFromPacletURI["paclet:"<>u, primaryGuides];
LinkTrailFromPacletURI[u_String /; StringMatchQ[u, "paclet:*"], primaryGuides_]:= 
Module[{trail, uri=u, info, primaryGuide, pg, lst, res},
	LogIt["-- LinkTrailFromPacletURI: "<>u<>", "<>ToString[Length@primaryGuides]];
	(* get the primary guide *)
	pg = GetPrimaryGuide[uri, primaryGuides];
	(* If no primary guide, then return root guide *)
	If[pg === uri, (* XX Root guide rename change XX *)Return[{{"paclet:guide/WolframRoot"}}] ];
	LogIt["-- "];
	trail =
	If[CoreMathematicaPacletQ[uri], 
		(* wri guide/ref/tutorial... *)
		If[StringMatchQ[uri, "paclet:*"], 
			(* compute link trail for wri guides *)
			res = DocumentationBuild`Navigation`LinkTrailListFromFilename @ uri;
			If[res === {}, (* XX Root guide rename change XX *){{"paclet:guide/WolframRoot"}}, res]
		,
			(* uri is not a wri guide... *)
			(* check to see if paclet uri has a primary guide *)
			If[(pg === (* XX Root guide rename change XX *)"paclet:guide/WolframRoot") || (pg === ""),
				(* return link to root guide *)
				(* XX Root guide rename change XX *){ {"paclet:guide/WolframRoot"} }
			,
				(* return root + primary guide *)
				DocumentationBuild`Utils`UnsortedUnion @ { Flatten @ DocumentationBuild`Utils`UnsortedUnion @
					(* XX Root guide rename change XX *)DeleteCases[{"paclet:guide/WolframRoot", GetPrimaryGuide[pg, primaryGuides], pg}, ""|Null|$Failed]  }
			]
		]
	,
		(* return pakage's primary guide *)
		{{pg }}

	]

];
(* error check *)
LinkTrailFromPacletURI[u___]:= (Message[LinkTrailFromPacletURI::arg, u]; $Failed)
LinkTrailFromPacletURI::arg = "Paclet uri expected, but found `1`";


(******************************************************************************)
(** GetPrimaryGuide - Internal **)
MakeLinkTrailCellGroup[ts_List, language_String:"English", opts_List:{}] := 
Module[{ trails=ts, buttons},
(*	trails = trails //. {"paclet:guide/Mathematica", a___} :> {a}; *)
	BuildLog["-- Make link trail"];
	BuildLog["   Trails: " <> ToString[trails]];
	(* turn list of paclet:* into list of ButtonBoxes *)
	buttons = Flatten@{Map[Function[pacletURI, 
		bb = MakeButtonBoxFromPacletURI[pacletURI, language]; 
				If[bb =!= $Failed, {bb, StyleBox[" > ", "LinkTrailSeparator"]}, $Failed ] ], Flatten@{#}]} & /@ trails;
	(* return if $Failed when making a Button *)
	If[ !FreeQ[buttons, $Failed], 
		BuildLog["   Failed!"]; Return@{} ];
	(* if no list of buttons, return *)
	If[ Flatten@{buttons} === {}, Return@{} ];
	(* return cell expr *)
    (* Using Most[ ] here to get rid of the last LinkTrailSeparator *)	
	Cell[TextData[Most[Flatten@{ buttons }]], "LinkTrail", Sequence@@Flatten@{opts, CellID->0} ]
	];


(******************************************************************************)
(** MakeButtonBoxFromPacletURI - Internal **)
MakeButtonBoxFromPacletURI[u_String /; !StringMatchQ[u, "paclet:*"], language_String:"English", opts:OptionsPattern[]]:= 
	MakeButtonBoxFromPacletURI["paclet:"<>u, language, opts];
MakeButtonBoxFromPacletURI[pacletURI_String /; StringMatchQ[pacletURI, "paclet:*"], language_String:"English", opts:OptionsPattern[]]:= 
Module[{title, info, wsmlinkQ},
	wsmlinkQ = TrueQ[StringMatchQ[pacletURI, "paclet:WSMLink/*"]];
	title = If[wsmlinkQ, "SystemModeler Link", DocumentationBuild`Info`GetCoreInformationTitle[pacletURI, language]];
	If[StringMatchQ[pacletURI, (* XX Root guide rename change XX *)"paclet:guide/Mathematica"|"paclet:guide/WolframRoot"],
		StyleBox[ButtonBox["Mathematica", ButtonData -> pacletURI, BaseStyle -> {"Link", "LinkTrail"}], FontSlant -> "Italic"],
		MakePacletButton @
			ButtonBox[title, ButtonData -> pacletURI, BaseStyle -> {"Link", "LinkTrail"}]
	]
];
MakeButtonBoxFromPacletURI::info = "No entry for `1`";
MakeButtonBoxFromPacletURI::title = "No title for `1`";


(******************************************************************************)
(** GetPrimaryGuide - Internal **)
(* 
	GetPrimaryGuide
	in: paclet uri
	out: primary guide paclet uri
*)
(* Hard code specifc package primary guides *)
GetPrimaryGuide[uri_String /; StringMatchQ[uri,"paclet:WSMLink/*"], ___]:= "paclet:WSMLink/guide/WSMLink";
GetPrimaryGuide[uri_String /; StringMatchQ[uri,"paclet:JLink/*"], ___]:= "paclet:JLink/guide/JavaInterface";
GetPrimaryGuide[uri_String /; StringMatchQ[uri,"paclet:NETLink/*"], ___]:= "paclet:NETLink/guide/DotNETInterface";
GetPrimaryGuide[uri_String /; StringMatchQ[uri,"paclet:DatabaseLink/*"], ___]:= "paclet:DatabaseLink/guide/SQLDatabaseOperations";
GetPrimaryGuide[uri_String /; StringMatchQ[uri,"paclet:WebServices/*"], ___]:= "paclet:WebServices/guide/WebServiceOperations";
GetPrimaryGuide[uri_String /; StringMatchQ[uri,"paclet:GUIKit/*"], ___]:= "paclet:GUIKit/guide/GUIKitPackage";
GetPrimaryGuide[uri_String /; StringMatchQ[uri,"paclet:ANOVA/*"], ___]:= "paclet:ANOVA/guide/AnalysisOfVariancePackage";
GetPrimaryGuide[uri_String /; StringMatchQ[uri,"paclet:FEMDocumentation/*"], ___]:= "paclet:FEMDocumentation/guide/FiniteElementMethodGuide";
GetPrimaryGuide[uri_String /; StringMatchQ[uri,"paclet:Compatibility/*"], ___]:= 
Module[{pkg}, 
  If[StringMatchQ[uri, "paclet:Compatibility/guide/StandardPackageCompatibilityGuide"], (* XX Root guide rename change XX *)Return["paclet:guide/WolframRoot"]];
  If[StringMatchQ[uri, "paclet:Compatibility/*UpgradingInformation"], Return["paclet:Compatibility/guide/StandardPackageCompatibilityGuide"]];
  pkg = StringReplace[uri, RegularExpression[ "paclet:Compatibility/(ref|guide|tutorial)/(.+)(/.)*"] -> "$2"];
  "paclet:Compatibility/guide/" <> pkg <> "/" <> pkg <> "UpgradingInformation"
];
GetPrimaryGuide[uri_String /; StringMatchQ[uri,"paclet:ErrorBarPlots/*"], ___]:= "paclet:ErrorBarPlots/guide/ErrorBarPlottingPackage";
GetPrimaryGuide[uri_String /; StringMatchQ[uri,"paclet:VectorAnalysis/*"], ___]:= "paclet:VectorAnalysis/guide/CoordinateSystems";
GetPrimaryGuide[uri_String /; StringMatchQ[uri,"paclet:VectorFieldPlots/*"], ___]:= "paclet:VectorFieldPlots/guide/VectorFieldPlottingPackage";
GetPrimaryGuide[uri_String /; StringMatchQ[uri,"paclet:WorldPlot/*"], ___]:= "paclet:WorldPlot/guide/WorldPlottingPackage";
(* Hard code specifc core M- primary guides *)
GetPrimaryGuide[uri_String /; StringMatchQ[uri,"paclet:ref/c/*"], ___]:= "paclet:guide/WSTPCLanguageFunctions";
GetPrimaryGuide[uri_String /; StringMatchQ[uri,"paclet:ref/character/*"], ___]:= "paclet:guide/SpecialCharacters";
GetPrimaryGuide[uri_String /; StringMatchQ[uri,"paclet:LibraryLink/*"], ___]:= "paclet:guide/LibraryLink";
GetPrimaryGuide[uri_String /; StringMatchQ[uri,"paclet:ref/format/*"], ___]:= "paclet:guide/ImportingAndExporting";
GetPrimaryGuide[uri_String /; StringMatchQ[uri,"paclet:ref/frontendobject/*"], ___]:= "paclet:guide/FrontEndTokens";
GetPrimaryGuide[uri_String /; StringMatchQ[uri,"paclet:ref/menuitem/*"], ___]:= "paclet:guide/MenuItems";
GetPrimaryGuide[uri_String /; StringMatchQ[uri,"paclet:ref/message/*"], ___]:= "paclet:guide/Messages";
GetPrimaryGuide[uri_String /; StringMatchQ[uri,"paclet:ref/indicator/*"], ___]:= "paclet:guide/FinancialVisualization";

(* all other uris *)
GetPrimaryGuide[uri_String /; StringMatchQ[uri,"paclet:*"], primaryGuides_List|Dispatch]:=
	GetPrimaryGuide[uri, "English", primaryGuides ];
GetPrimaryGuide[uri_String /; StringMatchQ[uri,"paclet:*"], language_String:"English"]:=
	GetPrimaryGuide[uri, language, DocumentationBuild`Export`$CoreInformation[language]];
GetPrimaryGuide[uri_String /; StringMatchQ[uri,"paclet:*"], language_String:"English", primaryGuides_]:= 
Module[{info, primaryGuide, packageList, linkBase, suffix},
	BuildLog["-- GetPrimaryGuide: "<>uri];
	{packageList} = {"Packages"} /. Options[GetPrimaryGuide];
	packageList = If[packageList === "Packages", {}, packageList];
	linkBase = LinkBaseFromPacletURI[uri];
	(* only M- Packages have root guides named '<paclet>Package' *)
	suffix = If[MemberQ[packageList, linkBase], "Package", ""];
	BuildLog["-- "<>ToString[{linkBase,suffix}]];

	If[CoreMathematicaPacletQ[uri], 
		(* get paclet info from CoreInformation *)
		info = (uri) /. primaryGuides;
		If[OptionQ@info,
			(* get PrimaryGuide *)
			primaryGuide = "PrimaryGuide" /. info;
			If[!StringMatchQ[primaryGuide, "PrimaryGuide"], 
				(* return PrimaryGuide *)			
				primaryGuide,
				(* no guide, return empty String *)
				(* XX Root guide rename change XX *)"paclet:guide/WolframRoot"
			],
			(* no guide, return empty String *)
			(* XX Root guide rename change XX *)"paclet:guide/WolframRoot"
		],
		StringJoin["paclet:", linkBase, "/guide/", linkBase <> suffix]
	]
];
GetPrimaryGuide[uri_, __]:= (Message[GetPrimaryGuide::args,uri]; $Failed);
GetPrimaryGuide::args = "Expected uri but found: `1`";
Options[GetPrimaryGuide] = {"Packages" -> 
{"Audio", "BarCharts", "Benchmarking", "BlackBodyRadiation", "Calendar", "Combinatorica", \
"ComputationalGeometry", "ComputerArithmetic", "Developer", "EquationTrekker", "Experimental", "FiniteFields", \
"FourierSeries", "FunctionApproximations", "Geodesy", "GraphUtilities", "HierarchicalClustering", "Histograms", \
"HypothesisTesting", "LinearRegression", "MultivariateStatistics", "Music", "NonlinearRegression", "Notation", "NumericalCalculus", \
"NumericalDifferentialEquationAnalysis", "PhysicalConstants", "PieCharts", "PlotLegends", "PolyhedronOperations", "Polytopes", \
"PrimalityProving", "Quaternions", "RegressionCommon", "ResonanceAbsorptionLines", "Splines", "StandardAtmosphere", \
"StatisticalPlots", "Units", "VariationalMethods", "XML"}};


(******************************************************************************)
(** CoreMathematicaPacletQ **)
CoreMathematicaPacletQ[u_String /; StringMatchQ[u,"paclet:*"], opts___?OptionQ]:= 
Module[{coreDirs, lst, uri=u},
	coreDirs = "CoreDirectories" /. {opts} /. Options[CoreMathematicaPacletQ];
	uri = StringReplace[u, "paclet:" -> ""];
	lst = StringSplit[uri, "/"];
	(* does uri contain 1 or more "/"? *)
	If[Length@lst <= 1, Return@False];
	(* simplistic heuristic... *)
	If[MemberQ[coreDirs, First@lst], True, False]
];
CoreMathematicaPacletQ[___]:= False;
Options[CoreMathematicaPacletQ]= {"CoreDirectories"->{"guide","tutorial","note","ref","howto","screencast","preview"} };


(******************************************************************************)
(** HasOptionsQ **)
HasOptionsQ[nb_Notebook, base_String]:=
 Module[{styles = {"3ColumnTableMod", "2ColumnTableMod"}},
 	((*MemberQ[nb, Cell[___, Alternatives@@styles, ___], Infinity] && *)
 	 (TrueQ[Length[DocumentationBuild`SystemResourcesFromDocs`Private`getoptionnames[base]] > 0]))
 ];


(******************************************************************************)
(** RepairMetadataCellLabels **)
RepairMetadataCellLabels[nb_Notebook]:=
 Module[{newnb = nb, catcells, detcells, 
  catlabels = {"Entity Type", "Paclet Name", "Context", "URI"}, 
  detlabels = {"Lead", "Developers", "Authors", "Feature Name", "QA", 
    "DA", "Docs", "Features Page Notes", "Comments"}}, 
  catcells = Cases[newnb, Cell[__, "Categorization", ___], Infinity];
  If[Length[catcells] =!= 4, 
   Return["Not right num: Categorization cells"]];
  detcells = Cases[newnb, Cell[__, "Details", ___], Infinity];
  If[Length[detcells] =!= 9, Return["Not right num: Details cells"]];
  (newnb = 
      ReplaceAll[
      newnb, #[[1]] -> Append[#[[1]], CellLabel -> #[[2]]]]) & /@ 
    Thread[{catcells, catlabels}];
  (newnb = 
      ReplaceAll[
      newnb, #[[1]] -> Append[#[[1]], CellLabel -> #[[2]]]]) & /@ 
    Thread[{detcells, detlabels}]; newnb];
    
RepairMetadataCellLabels[filename_String]:=
 Module[{nbdata = Quiet[Get[filename]], newnb}, 
  newnb = RepairMetadataCellLabels[nbdata];
  If[(nbdata =!= newnb) && (Head[newnb] === Notebook), Export[filename, newnb]]];

(******************************************************************************)
(** LinkBaseFromPacletURI **)
LinkBaseFromPacletURI[u_String /; StringMatchQ[u,"paclet:*"], opts___?OptionQ]:= 
Module[{coreDirs, lst, uri=u},
	If[ CoreMathematicaPacletQ[u], 
		"Mathematica",
		lst = StringSplit[StringReplace[u, "paclet:" -> ""], "/"];
		(* does uri contain 1 or more "/"? *)
		(* TODO: Return empty string? *)
		If[Length@lst < 1, "", First@lst]
	]
];
LinkBaseFromPacletURI[a___]:= (Message[LinkBaseFromPacletURI::arg, a]; $Failed);
LinkBaseFromPacletURI::arg = "Paclet uri expected but found `1`";


(******************************************************************************)
(** MakePacletButton **)
(* convert "FunctionLink", "TutorialLink", etc. into the appropriate
   "paclet:location" format.
*)
MakePacletButton[c___]:= (Message[MakePacletButton::arg,c]; c);

Options[MakePacletButton] = {BaseStyle->"Link", ButtonData->None};

(* Return if ButtonData contains a URL *)
(* MakePacletButton[bb:ButtonBox[__, ButtonData~(Rule|RuleDelayed)~{URL[_],___}, ___]]:= Return@bb; *)
(* Process button *)
MakePacletButton[bb:ButtonBox[content_String, opts___?OptionQ]] :=
  Module[{b, style, data, newdata, buttonFunction},
    {style, data} = {BaseStyle, ButtonData} /. {opts} /. Options[MakePacletButton];
    buttonFunction = Hold[{ButtonFunction} /. {opts} /. Options[MakePacletButton]];
    (* TODO: Handle multiple link styles; For now, take the first *)
    If[style==={}, Return[bb]];
    style = If[Head@style===List, First@Flatten@{style}, style];

    If[style===None, Return[bb]];
    (* external links *)
    If[MatchQ[style, "Hyperlink" ], Return[bb] ];
    If[MemberQ[data, _URL], If[style==="Link"||"NKSLink", Return@ResetOptions[bb, BaseStyle->"Hyperlink"],Return[bb]] ];

    If[StringMatchQ[style, "NKSLink" | "InformationLink" | "MathWorldLink" | "FunctionsLink"],
      b = bb /. _[BaseStyle, _]:>(BaseStyle->"Hyperlink"); Return[b] ];
      
        If[StringMatchQ[style, "AlphaSymbolLink"],
      b = ButtonBox[content, BaseStyle->"Hyperlink", ButtonData->{URL["http://wadocs.wolfram.com/mathdoc/Function/"<>content<>".html"], None}, opts]; Return[b] ];

    If[MatchQ[style, "FunctionLink"|"TutorialLink"|"GuideLink"|"InterfaceLink"|"CharacterLink"|"FormatLink"],
      b = bb /. _[BaseStyle, _]:>(BaseStyle->"Link"),
      b = bb];
    If[MatchQ[style, "ExampleLink"],
      b = ButtonBox["\[NonBreakingSpace]\[RightGuillemet]", BaseStyle->"ExampleLink", opts] ];

    (* just return the button if we have nothing to check over *)
    If[style===None || MatchQ[data, {s_String,_}|s_String/;!StringFreeQ[s,"paclet:"]],
      b = b /. _[ButtonData, p_]:>ButtonData-> StringReplace[p, {"paclet://"->"paclet:"}];
      Return[b]
    ];

    Switch[style,
      "Link"|"FunctionLink"|"NewInVersionLink"(*|"ExampleLink"*), newdata = "paclet:ref/",
      "TutorialLink", newdata = "paclet:tutorials/",
      "GuideLink", newdata = "paclet:guides/",
      "InterfaceLink", newdata = "paclet:interfaces/",
      "CharacterLink", newdata = "paclet:characters/",
      "FormatLink", newdata = "paclet:formats/",
      _ , Return[b]
    ];

    (* add filename base *)
    Switch[style,
      (* get filename from ButtonData for Tutorials *)
      "TutorialLink", 
        Which[
          data===None,
            Message[MakePacletButton::nodata];
            Return[b],
          ListQ[data], newdata = newdata<>data[[1]],
          True, newdata = newdata<>data],

      (* get filename from content for everything else *)
      _, 
      	If[data==="data", Return[b],
      		newdata = newdata <> content ]
    ];
  
    (* add anchor link *)
    If[Length[data]===2,
      newdata = newdata <> "#"<>data[[2]]];
  
    (* Strip nb extension, and turn \[Character]s into "Character"s *)
    newdata = StringReplace[newdata, {".nb" -> "", "\"" ->"" (*, "RawGuides"->"Guides"*),
      RegularExpression["[\\\\\[Backslash]]\\[(.*?)\\]"]->"$1"}];

    (* Links to certain Flags should be kept, but delinkified *)
    If[(DocumentationBuild`Info`GetSourceInformation[newdata, "English", "Flag"] === "InternalFlag") &&
       (FreeQ[{opts},ButtonFunction] && FreeQ[BaseStyle/.{opts},"Link"]),
      Return[Transmogrify`ConvertToString[content]] ];

    (*If[data===None,
      ButtonBox[content, opts, ButtonData->newdata],
      b /. _[ButtonData,_]:>ButtonData->newdata
    ]*)
    If[data===None, 
    Which[
    	MatchQ[buttonFunction, Hold[{ButtonFunction}/. {BaseStyle->"Link",Evaluator->Automatic,ButtonFunction:>(TreeBrowse`DocsNavigatorLookup[True]&)}/. Options[MakePacletButton]]], 
    		(*If[MatchQ[ButtonBox[content,opts], 
              ButtonBox[_, opt___]/;(MemberQ[{opt},BaseStyle->"Link",Infinity]&&FreeQ[{opt},_[ButtonFunction,_],Infinity])],
             TemplateBox[{b[[1]],ButtonData /. Options[b]},"RefLink"],
             ButtonBox[content,opts]],*)
             ButtonBox[content,opts],
    	MatchQ[buttonFunction, Hold[{ButtonFunction}/. {BaseStyle->"Link",Evaluator->Automatic,ButtonFunction:>(TreeBrowse`DocsNavigatorLookup[False]&)}/. Options[MakePacletButton]]], 
    		(*If[MatchQ[ButtonBox[content,opts], 
              ButtonBox[_, opt___]/;(MemberQ[{opt},_[BaseStyle,"Link"],Infinity]&&FreeQ[{opt},_[ButtonFunction,_],Infinity])],
             TemplateBox[{b[[1]],ButtonData /. Options[b]},"RefLink"],
             ButtonBox[content,opts]],*)
             ButtonBox[content,opts],
      	True, (*If[MatchQ[ButtonBox[content, opts, ButtonData->newdata], 
              ButtonBox[_, opt___]/;(MemberQ[{opt},_[BaseStyle,"Link"],Infinity]&&FreeQ[{opt},_[ButtonFunction,_],Infinity])],
             TemplateBox[{b[[1]],newdata},"RefLink"],
             ButtonBox[content, opts, ButtonData->newdata]]*)
             ButtonBox[content, opts, ButtonData->newdata]
    	],
      (* V9 EDIT BELOW (commented out now) *)
      (*If[MatchQ[(b = b /. _[ButtonData,_]:>ButtonData->newdata), 
        ButtonBox[_, opt___]/;(MemberQ[{opt},BaseStyle->"Link",Infinity]&&FreeQ[{opt},_[ButtonFunction,_],Infinity])],
       TemplateBox[{b[[1]],ButtonData /. Options[b]},"RefLink"],
       b]*)
       b /. _[ButtonData,_]:>ButtonData->newdata
    ]
  ];

MakePacletButton::nodata = "Tutorial button for `1` has no ButtonData.";
MakePacletButton::arg = "Incorrect arguments for `1`.";


(******************************************************************************)
(** ConvertButtonBoxesToTemplateBoxes **)

ConvertButtonBoxesToTemplateBoxes[nb_Notebook] :=
 Module[{nbexpr = nb, 
   bluestyles = {"InlineFunctionSans", "InlineFormula", "GuideMoreAbout", 
     "GuideMoreAboutSub", "Tutorials", "MoreAbout", "RelatedLinks", 
     "Subsubsection", "InlineGuideFunction", "InlineCharacterName",
     "TOCChapter", "TOCSection", "TOCSubsection", "TOCSubsubsection", "GuideTOCLink"},
   orangestyles = {"GuideFunctionsSubsection"}, 
   graystyles = {"MenuName"}, pos, anchor, boxstring},
   
  (* Take out anchor bar so it doesn't get messed with -- we'll put it back later *)
  pos = Position[nbexpr, Cell[___, "AnchorBarGrid", ___]];
  anchor = Extract[nbexpr, pos];
  nbexpr = Delete[nbexpr, pos];
  
  (* Blue links and then orange links *)
  (* ButtonBoxes with enclosing StyleBoxes first (e.g. SeeAlso links) *)
  nbexpr = ReplaceAll[nbexpr,
    cell : 
       Cell[dat_[c___], some___, 
        sty : (Alternatives @@ bluestyles), rest___] /; 
      MemberQ[{c}, 
       StyleBox[ButtonBox[___, BaseStyle -> (s_ /; MemberQ[{s}, "Link" | "NewInVersionLink", Infinity]), ___], __], 
       Infinity] :> 
     Cell[BoxData[
       Sequence @@ (ReplaceRepeated[If[dat===TextData, {First[ConvertTextData[dat[c]]]}, {c}], 
          StyleBox[ButtonBox[n_, ___, ButtonData -> uri_, ___],st__] :> 
           TemplateBox[{Cell[TextData[n]], uri}, "RefLink", BaseStyle -> Flatten[Join[{sty}, If[StringMatchQ[sty, "InlineFormula"], {st(*, "CodeFont"*)}, {st}]]]]])], 
      some, sty, rest]];
   (* Now the rest of the ButtonBoxes *)
   nbexpr = ReplaceAll[nbexpr,
    cell : 
       Cell[dat_[c___], some___, 
        sty : (Alternatives @@ bluestyles), rest___] /; 
      MemberQ[{c}, 
       ButtonBox[___, BaseStyle -> (s_ /; MemberQ[{s}, "Link" | "NewInVersionLink", Infinity]), ___], 
       Infinity] :> 
     Cell[BoxData[
       Sequence @@ (ReplaceRepeated[If[dat===TextData, {First[ConvertTextData[dat[c]]]}, {c}], 
          ButtonBox[n_, ___, ButtonData -> uri_, ___] :> 
           TemplateBox[{Cell[If[MemberQ[n, str_String/; !StringFreeQ[str, "\!"], Infinity],
           	boxstring = First@Cases[n, str_String/;!StringFreeQ[str,"\!"], Infinity];
           	TextData[ReplaceAll[n, boxstring -> Cell[BoxData[boxstring]]]],
           	TextData[n]]], uri}, "RefLink", BaseStyle -> If[StringMatchQ[sty,"InlineFormula"],{sty(*, "CodeFont"*)}, sty]]])], 
      some, sty, rest]];
   (* Orange-colored links *)
   nbexpr = ReplaceAll[nbexpr,
    cell : 
       Cell[dat_[c___], some___, 
        sty : (Alternatives @@ orangestyles), rest___] /; 
      MemberQ[{c}, 
       ButtonBox[___, BaseStyle -> (s_ /; MemberQ[{s}, "Link", Infinity]), ___], 
       Infinity] :> 
     Cell[BoxData[
       Sequence @@ (ReplaceRepeated[If[dat===TextData, {First[ConvertTextData[dat[c]]]}, {c}], 
          ButtonBox[n_, ___, ButtonData -> uri_, ___] :> 
           TemplateBox[{Cell[TextData[n]], uri}, "OrangeLink", BaseStyle -> sty]])], 
      some, sty, rest]];

   (* Now convert Hyperlinks *)
   (* StyleBox[ButtonBox[...]] first *)
   nbexpr = ReplaceAll[nbexpr,
    Cell[dat_[c___], some___, sty_String, rest___] /; 
      (MemberQ[{c}, StyleBox[bb:ButtonBox[___, BaseStyle->(s_ /; MemberQ[{s}, "Hyperlink", Infinity]), ___], __] /; 
        MemberQ[bb, URL[___], Infinity], Infinity] && 
         !MatchQ[sty,"HowToScreencastLink"|"Input"|"Output"] &&
         (* Additional exclusion to make sure hyperlinks inside of Databin blobs 
            aren't converted! *)
         FreeQ[{some, sty, rest}, "HowToScreencastLink"|"Input"|"Output", 1]) :>
     Cell[dat[Sequence@@(ReplaceRepeated[If[dat===TextData, {First[ConvertTextData[dat[c]]]}, {c}],
       StyleBox[ButtonBox[n_, ___, BaseStyle->s_, ___, ButtonData -> {URL[url_],___}, ___], st__] :>
        Cell[BoxData[TemplateBox[{Cell[TextData[n]], url}, "WebLink", 
         BaseStyle->Flatten[Join[{sty},{st},Flatten[DeleteCases[{s},"Hyperlink",Infinity]]]]]]]])], some, sty, rest]];

   nbexpr = ReplaceAll[nbexpr,
    Cell[dat_[c___], some___, sty_String, rest___] /; 
      (MemberQ[{c}, bb:ButtonBox[___, BaseStyle->(s_ /; MemberQ[{s}, "Hyperlink", Infinity]), ___] /; 
        MemberQ[bb, URL[___], Infinity], Infinity] && 
         !MatchQ[sty,"HowToScreencastLink"|"Input"|"Output"] &&
         (* Additional exclusion to make sure hyperlinks inside of Databin blobs 
            aren't converted! *)
         FreeQ[{some, sty, rest}, "HowToScreencastLink"|"Input"|"Output", 1]) :>
     Cell[dat[Sequence@@(ReplaceRepeated[If[dat===TextData, {First[ConvertTextData[dat[c]]]}, {c}],
       ButtonBox[n_, ___, BaseStyle->s_, ___, ButtonData -> {URL[url_],___}, ___] :>
        Cell[BoxData[TemplateBox[{Cell[TextData[n]], url}, "WebLink", 
         BaseStyle->Flatten[Join[{sty},Flatten[DeleteCases[{s},"Hyperlink",Infinity]]]]]]]])], some, sty, rest]];
   
   (* Now handle "gray links"; for now, just menu names -- which are all 
      StyleBox[ButtonBox[...], "MenuName"] *)
   nbexpr = ReplaceAll[nbexpr,
   	StyleBox[c_, some___, sty_String, rest___] /;
   	  (MemberQ[{c}, bb:ButtonBox[___] /; 
   	  	MemberQ[bb, s_String /; StringMatchQ[s, "paclet:*ref/menuitem/*"|"paclet:guide/*Menu"], Infinity], Infinity] &&
   	   MemberQ[graystyles, sty]) :>
   	Cell[BoxData[
       Sequence @@ (ReplaceRepeated[{c}, 
          ButtonBox[n_, ___, ButtonData -> uri_, ___] :> 
           TemplateBox[{Cell[TextData[n]], uri}, "MenuGrayLink", BaseStyle -> 
            Flatten[Join[{sty},Cases[{some,rest},_[FontSize,__], Infinity]]]]])], 
      some, sty, rest]
   
   ];
   
   (* Reinsert anchor bar *)
   nbexpr = Insert[nbexpr, anchor, pos];

   ReplaceRepeated[nbexpr, TextData[TemplateBox[c___]]:>BoxData[TemplateBox[c]]]
]


(******************************************************************************)
(** MakeCharacterNotesGrid **)

MakeCharacterNotesGrid[characterImageCell_, notesGroup_] :=
  Module[{chargrid, notescells, notesgrid},
    chargrid = ReplaceAll[First[Cases[characterImageCell, GridBox[___], Infinity, 1]],
      r_[GridBoxSpacings, ___] :> r[GridBoxSpacings, {"Columns" -> {{2}}, "Rows" -> {{3}}}]];
    notescells = Cases[notesGroup, Cell[c___, "Notes", rest___]:>Cell[c, "CharacterNotes", rest], Infinity];
    notesgrid = GridBox[Map[
      {RowBox[Flatten[{Cell["\[FilledSmallSquare]", "CharacterNotesDingbat"], "\[NonBreakingSpace]", #}]]}&, notescells], 
     BaseStyle->{AutoIndent->False},
     GridBoxSpacings->{"Rows"->{{1}},"Columns"->{{0}}},
     GridBoxItemSize->{"Columns"->{{Scaled[.75]}},"Rows"->{{.4}}}];
    Cell[BoxData[GridBox[{{
      chargrid,
      notesgrid}}]], "CharacterImage"]
  ];


(******************************************************************************)
(** FESaveNotebook **)

FESaveNotebook[s___]:= {$Failed;s}
FESaveNotebook[outfile_String, exp_Notebook, opts___?OptionQ]:=
Developer`UseFrontEnd @ 
Module[{nbObj, maxtime},
	BuildLog["     FESaveNotebook"];
  maxtime = MaxTime /. {opts} /. Options[FESaveNotebook];
  nbObj = FETimeConstrained[
    FrontEnd`NotebookPutReturnObject[exp], maxtime];

  If[Head[nbObj] === ReturnPacket, nbObj = First[nbObj]];
  CreateDirectory[DirectoryName@outfile];
  If[Head[nbObj] === NotebookObject,
    FETimeConstrained[
      FrontEnd`NotebookSave[nbObj, outfile], maxtime];
    FETimeConstrained[
      FrontEnd`NotebookClose[nbObj], maxtime];
  ,
  nbObj]
];
Options[FESaveNotebook] = {MaxTime -> 3000};

(******************************************************************************)
(** ApplyConvertToBitmapToNotebook **)

ApplyConvertGraphicsToBitmapsToNotebook[s___]:= {$Failed;s}
ApplyConvertGraphicsToBitmapsToNotebook[exp_Notebook, entityType_String, uri_String, opts___?OptionQ]:=
Module[{nbObj, maxtime, nbexpr},
  maxtime = MaxTime /. {opts} /. Options[ApplyConvertGraphicsToBitmapsToNotebook];
  
  (* If there are no output cells, let's skip the step *)
  If[Cases[exp, Cell[___, "Output", ___], Infinity] =!= {},
    nbObj = NotebookPut[exp];
    If[Head[nbObj] === ReturnPacket, nbObj = First[nbObj]];
    ConvertGraphicsToBitmaps[nbObj, entityType, uri];
    (* CurrentValue[nbObj, {TaggingRules, "MoreInformationCell"}] = Dynamic[CurrentValue[$FrontEnd, {"ToPostScriptOptions", "RemoveColorFromBitmaps"}]]; *)
    If[Head[nbObj] === NotebookObject,
      nbexpr = NotebookGet[nbObj];
      NotebookClose[nbObj];
      nbexpr
    ,
    nbexpr = exp],
    nbexpr = exp];
  nbexpr
];
Options[ApplyConvertGraphicsToBitmapsToNotebook] = {MaxTime -> 3000};

(******************************************************************************)
(** SaveConvertToBitmapToNotebook **)

SaveConvertGraphicsToBitmapsToNotebook[s___]:= {$Failed;s}
SaveConvertGraphicsToBitmapsToNotebook[outfile_String, exp_Notebook, entityType_String, uri_String, opts___?OptionQ]:=
Module[{nbObj, maxtime, savedebug, savetime1, savetime2, savetime3, savetime4, savetime5, outputfile, saferaster},
  maxtime = MaxTime /. {opts} /. Options[SaveConvertGraphicsToBitmapsToNotebook];
  savedebug = "Debug" /. {opts} /. Options[SaveConvertGraphicsToBitmapsToNotebook];
  saferaster = "SafeRaster" /. {opts} /. Options[SaveConvertGraphicsToBitmapsToNotebook];
  
  (* If there are no output cells, let's skip this; the existing fallthrough below of nbObj not being a 
     NotebookObject should catch these cases *)
  If[Cases[exp, Cell[___, "Output", ___], Infinity] =!= {},
  	
  savetime1 = ToString@AbsoluteTiming[
  nbObj = NotebookPut[exp];
  ][[1]];
  
  If[savedebug==="True", Global`AntLog["Running NotebookPut :"<>savetime1]];
  
  If[Head[nbObj] === ReturnPacket, nbObj = First[nbObj]];
  
  If[MemberQ[
  	{"DiscretePlot3D.nb", "Circumsphere.nb", "Insphere.nb", "SphericalShell.nb", "DensityPlot3D.nb"},
  	FileNameTake[outfile]],
         If[savedebug==="True", Global`AntLog["Skipping ConvertGraphicsToBitmaps for this file"]],
         savetime2 = ToString@AbsoluteTiming[ConvertGraphicsToBitmaps[nbObj, entityType, uri, "SafeRaster"->saferaster];][[1]];
         If[savedebug==="True", Global`AntLog["Running ConvertGraphicsToBitmaps :"<>savetime2]]
         ]
   ]; 
   
(*  CurrentValue[nbObj, {TaggingRules, "MoreInformationCell"}] = Dynamic[CurrentValue[$FrontEnd, {"ToPostScriptOptions", "RemoveColorFromBitmaps"}]]; *)
  If[Head[nbObj] === NotebookObject,
  	CreateDirectory[DirectoryName[outfile]];
    savetime3 = ToString@AbsoluteTiming[
    	If[$OperatingSystem === "MacOSX", Export[outfile, nbObj,"NB"], NotebookSave[nbObj, outfile]];
    ][[1]];
    
    If[savedebug==="True", Global`AntLog["Running NotebookSave :"<>savetime3]];
    
    savetime4 = ToString@AbsoluteTiming[
    	NotebookClose[nbObj];
    ][[1]];
    
    If[savedebug==="True", Global`AntLog["Running NotebookClose :"<>savetime4]];
    outfile
  ,
  savetime5 = ToString@AbsoluteTiming[
  If[!DirectoryQ[DirectoryName[outfile]], CreateDirectory[DirectoryName[outfile]]];
  outputfile = Export[outfile,exp,"NB"];
  ][[1]];
  If[savedebug==="True", Global`AntLog["Running Export :"<>savetime5]];
  outputfile
  ]
];
Options[SaveConvertGraphicsToBitmapsToNotebook] = {MaxTime -> 3000, "Debug"->"False", "SafeRaster"->True};


(******************************************************************************)
(** FETimeConstrained **)
FETimeConstrained[func_, time_]:=
MathLink`CallFrontEnd @ FrontEnd`TimeConstrained[func, time];


(******************************************************************************)
(** Marketing Page functions**)
GetBlockStyleList[expr_]:=
Union@Cases[Flatten[{FlattenCellGroups @ expr }], 
                Cell[_, s_String, ss___String, ___?OptionQ] :> s <> If[Length[{ss}] > 0, "*" <> ss, ""], 1];

(* Flatten Cell Groups *)
FlattenCellGroups[Notebook[{c__Cell}, opts___]] := 
	Notebook[Flatten[FlattenCellGroups /@ {c}], opts];

FlattenCellGroups[Cell[CellGroupData[{c__Cell}, ___],___]] := 
	Map[FlattenCellGroups, {c}];

FlattenCellGroups[c_Cell] := c;
FlattenCellGroups[c_List] := 
  FlattenCellGroups @ Cell@CellGroupData@Flatten[{c}];
  
(*  *)
PreviousNextElementList[m_String, l:{___, p_, m_, n_, ___}]:= {p, n};
PreviousNextElementList[m_String, l:{m_, n_, ___, p_}]:= {p, n};
PreviousNextElementList[m_String, l:{n_, ___, p_, m_}]:= {p, n};
PreviousNextElementList[m_String, l:{n_, m_}]:= {n, n};
PreviousNextElementList[m_String, l:{m_, p_}]:= {p, p};
PreviousNextElementList[___]:= {$Failed, $Failed};


MakeComparisonListingCell[Cell[c_, "ComparisonListing", ___]]:= MakeComparisonListingCell @ c;
MakeComparisonListingCell[TextData[lst_List]]:= MakeComparisonListingCell @ lst;
MakeComparisonListingCell[TextData[con_]]:= MakeComparisonListingCell[{con}];
MakeComparisonListingCell[str_]:=
Module[{elems, list},
  (* split list on "," *)
  elems = Transmogrify`ConvertToString @ str;
  elems = StringSplit[ elems, RegularExpression[",\\s*"]];
  elems = DeleteCases[elems, "" | "(" | ")" | "," | ", " | "...)" | ", ...)", Infinity];
  elems = If[Head@#===String, FixedPoint[StringReplace[#, {"(" | ")" | "..." -> ""}]&, #], #]&/@elems;
  elems = DeleteCases[elems, "" | "(" | ")" | "," | ", " | "...)" | ", ...)", Infinity];
(*  list = If[(StringLength@StringJoin@@Take[elems, 3] < 35, Take[elems, 2], Take[elems, 3], elems];*)
  Cell[TextData[ 
    Flatten@Join[{"("}, Riffle[Join[If[Length@elems > 2, Take[elems, 3], elems], {"... )"}], ", "]]], "ComparisonListing"]
];


MakeComparisonItemCell[Cell[con_, sty_String, opts___]]:= 
  Cell[TextData[Flatten@{ MakeComparisonItemCell@con }], sty, opts];
MakeComparisonItemCell[TextData[lst_List]]:= MakeComparisonItemCell @ lst;
MakeComparisonItemCell[TextData[con_]]:= MakeComparisonItemCell[{con}];
MakeComparisonItemCell[con_]:=
Module[{elems, str},
  str = Transmogrify`ConvertToString @ con;
  If[ StringLength@str > 86,
    Print["Too long: "<>ToString@StringLength@str],
    Print["Just right: "<>ToString@StringLength@str]
  ];
  con
];


(******************************************************************************)
(** GetSearchMetaDataList **)
GetSearchMetaDataList[nb_String, opts___?OptionQ]:= 
  GetSearchMetaDataList[GetQuietly@nb, opts];
GetSearchMetaDataList[c___]:= (Message[GetSearchMetaDataList::arg, c]; $Failed);
GetSearchMetaDataList::arg = "Incorrect arg: `1`";
GetSearchMetaDataList[nbExpr_Notebook, opts___?OptionQ]:=
Module[{expr, ret},
  expr = Rest[List @@ nbExpr];
  If[Length@expr > 0,
  	If[ OptionQ[ret = TaggingRules /. expr],
      ret = Flatten[ {"Metadata" /. ret}], {}]
    ,
    Return[{}];
  ];
  If[Length@ret < 2, {}, ret]
];


(******************************************************************************)
(** AppendHTMLDirectory **)
AppendHTMLDirectory[path_String]:=
Module[{dir},
	dir = StringSplit[ path, "/"|"\\"];
 	If[ Last@dir === "html", path, ToFileName[path, "html"] ]
];


(******************************************************************************)
(** ExportEclipseResourceFiles **)
ExportEclipseResourceFiles[path_String, nb_String, linkbase_String:"", opts___?OptionQ] :=
  ExportEclipseResourceFiles[path, Get@nb, linkbase, opts];

ExportEclipseResourceFiles[path_String, nbExpr_Notebook, linkbase_String:"", opts___?OptionQ] :=
Module[{fileList, files},
	fileList = "ReturnFileList" /. {opts} /. {"ReturnFileList" -> True};
	files = ExportTOCXML[ ToFileName[{path}, linkbase<>".xml"], nbExpr, linkbase, "ReturnFileList"->fileList];
	ExportPluginXML[ ToFileName[{path}, "plugin.xml"], nbExpr, linkbase, "FileList"->files];
]

(* *)
ExportTOCXML[file_String, nb_String, linkbase_String:"", opts___?OptionQ] :=
  ExportTOCXML[file, Get@nb, linkbase, opts];

ExportTOCXML[file_String, nbExpr_Notebook, linkbase_String:"", opts___?OptionQ] :=
Module[{ret, DocumentationBuild`$xmlFileList, fileList},
	fileList = "ReturnFileList" /. {opts} /. {"ReturnFileList" -> False};
	DocumentationBuild`$xmlFileList={BaseName[file]};
ret = 
Transmogrify`Transmogrify[file, nbExpr, 
Transmogrify`XMLTransform[{ 
	Notebook :> 
		Module[{children, label }, 
			label = Cases[Transmogrify`SelectSelf[], Cell[con_, "TOCDocumentTitle", ___] :> 
				Transmogrify`ConvertToString@con, Infinity]; 
			children = Cases[Transmogrify`SelectSelf[], Cell[CellGroupData[{Cell[_, "TOCChapter", ___], ___},___],___], Infinity]; 
			(*Transmogrify`Recurse[]; *)
			XMLObject["Document"][{
				XMLObject["Declaration"]["Version" -> "1.0", "Encoding" -> "UTF-8"], 
				XMLObject["ProcessingInstruction"]["NLS", "TYPE=\"org.eclipse.help.toc\""]
				},
				XMLElement["toc", {If[Length@label > 0, "label" -> First@label]}, 
					Flatten@{Transmogrify`Recurse[ children] }], {}]
		], 
	{CellGroupData, All} :> 
		Module[{ce, self, uri},
			self = Transmogrify`SelectSelf[];
			uri = getButtonData[Transmogrify`SelectChildren[1], linkbase];
			uri = If[uri =!= None, {"href" -> "html" <> uri}, {}];
			ce = Transmogrify`ConvertToString @ Transmogrify`SelectChildren[1];
			XMLElement[ "topic", Flatten@{"label" -> ce, uri }, 
				Flatten@{ Transmogrify`Recurse[] }]
		], 
	{Cell, All} :> 
		Module[{uri, label, children},
			(*Print@Transmogrify`SelectSelf[];
			children = Transmogrify`SelectChildren[]; *)
			uri = getButtonData[Transmogrify`SelectSelf[], linkbase]; 
			uri = If[uri =!= None, {"href" -> "html" <> uri}, {}];
			label = Transmogrify`ConvertToString @ Transmogrify`SelectSelf[]; 
			XMLElement[ "topic", Flatten@{"label" -> label,  uri }, {}]
		],
	{String} :> Transmogrify`SelectSelf[]
	}]
];
	If[fileList===True, DocumentationBuild`$xmlFileList, ret]
];


(******************************************************************************)
(** ExportPluginXML **)
ExportPluginXML[file_String, nb_String, linkbase_String:"", opts___?OptionQ] :=
  ExportPluginXML[file, Get@nb, linkbase, opts];

ExportPluginXML[file_String, nbExpr_Notebook, linkbase_String:"", opts___?OptionQ] :=
Module[{ret, xmlFileList, fileList},
	fileList = "FileList" /. {opts} /. {"FileList" -> {} };

	Export[file,
		XMLObject["Document"][{
			XMLObject["Declaration"]["Version" -> "1.0", "Encoding" -> "UTF-8"]
				},
			XMLElement["plugin", {
			"name"->linkbase<>" Documentation", 
			"id"->"com.wolfram.eclipse."<> LowercaseString@linkbase<>".help", 
			"version"->"1.0.0", 
			"provider-name"->"Wolfram Research, Inc."}, {
			XMLElement["extension", {
				"point" -> "org.eclipse.help.toc"
				}, Flatten@{
				If[Length@fileList>0, {
					XMLElement["toc", {"file"->First@fileList, "primary"->"true"}, {}],
					If[Length@fileList>1,
						{XMLElement["toc", {"file"->#}, {}] &/@ Rest@fileList},
						{}] },
					{fileList}]
		}]
			}], {}], "XML"]
];

                                                                                                                                                                                
getTitleFromCell[Cell[CellGroupData[expr_List, ___], ___]] := 
 getTitleFromCell@expr;
getTitleFromCell[CellGroupData[expr_List, ___]] := 
 getTitleFromCell@expr;
getTitleFromCell[{expr_, ___}] := 
 Transmogrify`ConvertToString@expr;
getTitleFromCell[a___] := 
 Transmogrify`ConvertToString@a;
 
getButtonData[{Cell[ce___]}, sty_String] := getButtonData[Cell[ce], sty];
getButtonData[Cell[ce___], sty_String] := 
 Module[{uri}, 
	uri = Union@
		Cases[{ce}, 
		ButtonBox[___, Rule[ButtonData, uri_String], ___] :> uri, Infinity]; 
	uri = If[Length@uri > 0, First@uri, Return[None] ];
	uri = Rest@StringSplit[uri, sty];
	If[Length@uri > 0, DocumentationBuild`Utils`DotHTML[First@uri (*<> ".html"*)], None]
];


(** GetValueFromOptions gets a value from a list of options**)
GetValueFromOptions[opts_, s_, lhs_] := 
  Module[{res}, res = Check[lhs /. (s /. opts), "$Failed"]; 
   If[res === lhs || res === "$Failed", "$Failed", res]];

(******************************************************************************)
(**
    option manipulation helpers
    these are the fastest implementations i could come up with! 

    As you can see, PrependOptions is by far the fastest as it doesn't
    try to delete the original option.  this is fine for most cases where
    they are applied via Replace*.
**)
(** PrependOptions **)
PrependOptions[h_[stuff___,opts___?OptionQ],newopts:{__?OptionQ}]:=
  h[stuff,Sequence@@Flatten@newopts,opts]
PrependOptions[h_[stuff___,opts___?OptionQ], newopts__?OptionQ]:=
  h[stuff,newopts,opts]

(** DeleteOptions **)
DeleteOptions[h_[stuff___,opts___?OptionQ], newopts__?OptionQ]:=
  DeleteOptions[h[stuff,opts],{newopts}]

(* delete the exact option,value pair *)
DeleteOptions[h_[stuff___,opts___?OptionQ], newopts:{__?OptionQ}]:=
  h[stuff, Sequence @@ DeleteCases[{opts},Alternatives@@newopts]]

(* delete all options with the optioname specified in newopts *)
DeleteOptions[h_[stuff___,opts___?OptionQ], newopts:{__}]:=
  h[stuff, Sequence @@ DeleteCases[{opts},_[Alternatives@@newopts,_]]]

(** ResetOptions **)
(* actually reset options. it's faster to define this in terms of 
   delete cases rather than have another call to DeleteOptions *)
ResetOptions[h_[stuff___,opts___?OptionQ], newopts__?OptionQ]:=
  h[stuff, Sequence@@Flatten@{newopts}, Sequence @@ DeleteCases[{opts}, _[Alternatives @@ (Flatten[{newopts}][[All, 1]]), _]]];

ResetOptions[h_[stuff___],newopts__?OptionQ]:=h[stuff,newopts];


(******************************************************************************)
(** fixSplitHyperlinks - Internal **)
fixSplitHyperlinks[nbExpr_Notebook] :=
  Module[{expr, text1, text2, text3, text4, text5, stylebox1, stylebox2, url},
    (**********************************************************************
      5 consecutive hyperlink boxes: button, style, button, style, button.
    **********************************************************************)
    expr = nbExpr;
    expr = expr /. 
      List[
        listleft___,
        ButtonBox[text1_String, Rule[_, _], Rule[ButtonData, List[URL[url_String], None]]],
        StyleBox[ButtonBox[text2_String, Rule[_, _], Rule[ButtonData, List[URL[url_String], None]]], stylebox1___],
        ButtonBox[text3_String, Rule[_, _], Rule[ButtonData, List[URL[url_String], None]]],
        StyleBox[ButtonBox[text4_String, Rule[_, _], Rule[ButtonData, List[URL[url_String], None]]], stylebox2___],
        ButtonBox[text5_String, Rule[_, _], Rule[ButtonData, List[URL[url_String], None]]],
        listright___
      ] ->
      List[
        listleft,
        ButtonBox[
          StyleBox[List[text1, StyleBox[text2, stylebox1], text3, StyleBox[text4, stylebox2], text5]],
          Rule[BaseStyle, "Hyperlink"],
          Rule[ButtonData, List[URL[url], None]]
        ],
        listright
      ];
    (**********************************************************************
      3 consecutive hyperlink boxes: button, style, button.
    **********************************************************************)
    expr = expr /. 
      List[
        listleft___,
        ButtonBox[text1_String, Rule[_, _], Rule[ButtonData, List[URL[url_String], None]]],
        StyleBox[ButtonBox[text2_String, Rule[_, _], Rule[ButtonData, List[URL[url_String], None]]], stylebox1___],
        ButtonBox[text3_String, Rule[_, _], Rule[ButtonData, List[URL[url_String], None]]],
        listright___
      ] ->
      List[
        listleft,
        ButtonBox[
          StyleBox[List[text1, StyleBox[text2, stylebox1], text3]],
          Rule[BaseStyle, "Hyperlink"],
          Rule[ButtonData, List[URL[url], None]]
        ],
        listright
      ]; 
    (**********************************************************************
      3 consecutive hyperlink boxes: style, button, style
    **********************************************************************)
    expr = expr /. 
      List[
        listleft___,
        StyleBox[ButtonBox[text1_String, Rule[_, _], Rule[ButtonData, List[URL[url_String], None]]], stylebox1___],
        ButtonBox[text2_String, Rule[_, _], Rule[ButtonData, List[URL[url_String], None]]],
        StyleBox[ButtonBox[text3_String, Rule[_, _], Rule[ButtonData, List[URL[url_String], None]]], stylebox2___],
        listright___
      ] ->
      List[
        listleft,
        ButtonBox[
          StyleBox[List[StyleBox[text1, stylebox1], text2, StyleBox[text3, stylebox2]]],
          Rule[BaseStyle, "Hyperlink"],
          Rule[ButtonData, List[URL[url], None]]
        ],
        listright
      ]; 
    (**********************************************************************
      2 consecutive hyperlink boxes: style, button
    **********************************************************************)
    expr = expr /. 
      List[
        listleft___,
        StyleBox[ButtonBox[text1_String, Rule[_, _], Rule[ButtonData, List[URL[url_String], None]]], stylebox1___],
        ButtonBox[text2_String, Rule[_, _], Rule[ButtonData, List[URL[url_String], None]]],
        listright___
      ] ->
      List[
        listleft,
        ButtonBox[
          StyleBox[List[StyleBox[text1, stylebox1], text2]],
          Rule[BaseStyle, "Hyperlink"],
          Rule[ButtonData, List[URL[url], None]]
        ],
        listright
      ]; 
    (**********************************************************************
      2 consecutive hyperlink boxes: button, style
    **********************************************************************)
    expr = expr /. 
      List[
        listleft___,
        ButtonBox[text1_String, Rule[_, _], Rule[ButtonData, List[URL[url_String], None]]],
        StyleBox[ButtonBox[text2_String, Rule[_, _], Rule[ButtonData, List[URL[url_String], None]]], stylebox1___],
        listright___
      ] ->
      List[
        listleft,
        ButtonBox[
          StyleBox[List[text1, StyleBox[text2, stylebox1]]],
          Rule[BaseStyle, "Hyperlink"],
          Rule[ButtonData, List[URL[url], None]]
        ],
        listright
      ]; 
    expr]

fixSplitHyperlinks[dir_String] :=
    Module[{manifest, increment, count = 0},
    manifest = FileNames[{"*.nb"}, dir, Infinity];
    increment = Floor[N[Length@manifest / 100]];
    Module[{nb, nbExpr, initialByteCount, finalByteCount},
      (* begin poor man's progress meter *)
      count++;
      If[increment > 1 && Mod[count, increment] === 0,
        Print[ToString@count<> " of "<> ToString@Length@manifest<> ": "<>ToString[100*N[count/Length@manifest]]<>"%"]];
      (* end poor man's progress meter *)
      nb = #;
      nbExpr = Quiet@Get@nb;
      initialByteCount = ByteCount[nbExpr];
      nbExpr = fixSplitHyperlinks[nbExpr];
     (**********************************************************************
       If the notebook expression has changed then Put it into the
       notebook file, overwriting the original notebook.  As of
       November 2007, Export in 6.0.2 Linux M-- will resplit unified
       hyperlinks so use Put here.
      **********************************************************************)
      finalByteCount = ByteCount[nbExpr];
      If[initialByteCount =!= finalByteCount, 
        Put[nbExpr, nb];
        Print[nb<>" changed"]];
    ]& /@ manifest]



(* Flatten Cell Groups *)
flattenCellGroups[Notebook[{c__Cell}, opts___]] := 
	Notebook[Flatten[flattenCellGroups /@ {c}], opts];

flattenCellGroups[Cell[CellGroupData[{c__Cell}, ___],___]] := 
	Map[flattenCellGroups, {c}];

flattenCellGroups[c_Cell] := c;
flattenCellGroups[c_List] := 
  flattenCellGroups @ Cell@CellGroupData@Flatten[{c}];
  

(* create timestamp *)
timestamp:=ToString[(StringForm["`1`/`2`/`3` `4`:"<>
  (#[[1]]<>":"<>#[[2]]&)[
    (If[#<10,"0"<>ToString@#,ToString[#]]&)/@{#5,Round[#6]}], 
  ##]& @@ Date[])]

line="--------------------------------------------------\n"

(* quick logging *)
BuildLog[str_]:=
Module[{log},
	If[TrueQ[DocumentationBuild`Common`$DocumentationBuildDebug],
		log = DocumentationBuild`Utils`Private`timestamp<>"\n"<>ToString[str];
		Which[
			TrueQ[Global`$AntBuildQ], Global`AntLog[log],
			True, Print@log
		]
	]
];



(* Format Messages list *)
formatMessageList[lst_List]:=
Module[{},
	StringJoin @@ Flatten[
		Riffle[{First@# <> " (" <> ToString@Length[#] <> ")"} & /@ 
    			Split@Sort[ToString /@ lst], ", "]
    	]
];


(* GetCellsFromCellGroup within template nb *)
GetCellsFromCellGroup[str_String, grp_String, sty___:"Output", opts___?OptionQ]:=
  GetCellsFromCellGroup[GetQuietly@str, grp, sty, opts];
GetCellsFromCellGroup[expr_Notebook, grp_String, sty___:"Output", opts___?OptionQ]:=
Module[{cells, s, keepHeadCell},
  {keepHeadCell} = {"IncludeHeadCell"} /. {opts} /. {"IncludeHeadCell" -> True};
  PacletBuildLog[line <> " GetCellsFromCellGroup ("<>ToString[grp]<>") " <> line];
  cells = 
    Cases[expr, 
       Cell[CellGroupData[{Cell[_, grp, ___], ___}, ___], ___],
    Infinity];
  (*  *)
  If[!keepHeadCell, 
    cells = Flatten@DeleteCases[cells, Cell[_, grp, ___], Infinity]];
  If[ sty =!= All,
  cells = flattenCellGroups@
      DeleteCases[cells, Cell[_, s_String /; !StringMatchQ[s, sty], ___], Infinity],
    cells = flattenCellGroups@cells
  ];
  PacletBuildLog["   Found "<>ToString[Length[cells]]<>" cell(s)"];
  Flatten@cells
];


(* line breaking routine for XML files *)
indentFunction[{"", "table"}] := Automatic;
indentFunction[{"", "a"}] := False;
indentFunction[{"", "div"}] := Automatic;
indentFunction[{"", "ul"}] := True;
indentFunction[{"", "li"}] := Automatic;
indentFunction[{"", "span"}] := False;
indentFunction[{"", "p"}] := Automatic;



(******************************************************************************)
(* BlockMessageOff *)

SetAttributes[BlockMessageOff, HoldAll];

BlockMessageOff[{msgnames___MessageName}, body_] :=
With[{
   onMsgNames = DeleteCases[ Hold[msgnames], mn_ /; Head[mn] === $Off ]
},
   Scan[Off, onMsgNames];
   First @ {body, Scan[On, onMsgNames] }
];



(******************************************************************************)
(* Image Difference Code*)

GetScreenCapture[url_String] := 
  Module[{}, 
   Import["http://webtestmac.wri.wolfram.com:8080/webMathematica/screencapture.jsp?url=" <> url, {"PNG", "Data"}]];

GetNotebookCapture[file_String] := Module[{nb, ras},
   nb = NotebookOpen[file, Visible -> False];
   FrontEndTokenExecute[nb, "SelectAll"];
   FrontEndTokenExecute[nb, "SelectionOpenAllGroups"];
   ras = Rasterize[nb][[1, 1]];
   NotebookClose[nb]; ras];

HashIt[data_] := Module[{hash},
  hash = Map[IntegerString[Hash[#, "SHA"], 16, 40] &, data];
  Compress[ExportString[hash, "Text"]]
  ];

GetDiff[hashAA_, hashBB_] := Module[{},
	InstallJava[];
   client = JavaNew["org.apache.commons.httpclient.HttpClient"];
   mPost = JavaNew["org.apache.commons.httpclient.methods.PostMethod", "http://webtestmac.wri.wolfram.com:8080/webMathematica/diff.jsp"];
   mPost@addParameter["hash1", hashAA];
   mPost@addParameter["hash2", hashBB];
   client@executeMethod[mPost];
   output = mPost@getResponseBodyAsString[];
   mPost@releaseConnection[]; 
   ImportString[Uncompress[output], "Lines"]];

URLImageDifference[url1_, url2_, type_] := Module[
  {raster1, raster2, dimrast1, dimrast2, rastmax, hash1, hash2, count1 = 1, count2 = 1, hashA, 
   hashB, len1, len2, length, res = {}, command, difference, 
   red = {255, 0, 0}, blue = {0, 0, 255}, black = {0, 0, 0}, 
   green = {0, 255, 0}, blueline, redline, shortblueline, 
   shortredline, shortsameline},
  
Which[type === "URL",
  raster1 = Reverse@GetScreenCapture[url1];
  raster2 = Reverse@GetScreenCapture[url2];
  dimrast1 = Dimensions[raster1][[2]];
  dimrast2 = Dimensions[raster2][[2]];
  rastmax = Min[dimrast1, dimrast2];
  raster1 = Take[raster1, All, {1, rastmax}];
  raster2 = Take[raster2, All, {1, rastmax}]
  ,
  type == "Image",
  raster1 = Reverse@Import[url1, "Data"];
  raster2 = Reverse@Import[url2, "Data"];
  dimrast1 = Dimensions[raster1][[2]];
  dimrast2 = Dimensions[raster2][[2]];
  rastmax = Min[dimrast1, dimrast2];
  raster1 = Take[raster1, All, {1, rastmax}];
  raster2 = Take[raster2, All, {1, rastmax}]
  ,
  type == "NB",
  raster1 = GetNotebookCapture[url1];
  raster2 = GetNotebookCapture[url2];
  dimrast1 = Dimensions[raster1][[2]];
  dimrast2 = Dimensions[raster2][[2]];
  rastmax = Min[dimrast1, dimrast2];
  raster1 = Take[raster1, All, {1, rastmax}];
  raster2 = Take[raster2, All, {1, rastmax}]
  , True,
  raster1 = Reverse@GetScreenCapture[url1];
  raster2 = Reverse@GetScreenCapture[url2];
  dimrast1 = Dimensions[raster1][[2]];
  dimrast2 = Dimensions[raster2][[2]];
  rastmax = Min[dimrast1, dimrast2];
  raster1 = Take[raster1, All, {1, rastmax}];
  raster2 = Take[raster2, All, {1, rastmax}]];
  
If[(length = Length[First[raster1]]) =!= Length[First[raster2]], 
  Print["Captured images are not the same width " <> 
    ToString[{Length[First[raster1]], Length[First[raster2]]}] <> 
    ", Aborting...."]; ];
  
  If[Last[Dimensions[raster1]] === 4, AppendTo[red, 255]; 
   AppendTo[blue, 255]; AppendTo[black, 255]; AppendTo[green, 255]];
  blueline = Table[blue, {length}];
  shortblueline = Join[{black}, Table[blue, {8}], {black}];
  shortredline = Join[{black}, Table[red, {8}], {black}];
  shortsameline = Join[{black}, Table[green, {8}], {black}];
  
  hashA = HashIt[raster1];
  hashB = HashIt[raster2];
  
  difference = GetDiff[hashA, hashB];

If[StringMatchQ[StringJoin[difference], ___ ~~ ">" | "<" | "|" ~~ ___],
  Scan[
   Function[{d},
    Which[
     StringMatchQ[d, ___ ~~ ">" ~~ ___], 
     AppendTo[res, 
      Join[shortblueline, blueline, shortblueline, 
       raster2[[count2++]], shortblueline]],
     StringMatchQ[d, ___ ~~ "<" ~~ ___], 
     AppendTo[res, 
      Join[shortblueline, raster1[[count1++]], shortblueline, 
       blueline, shortblueline]],
     StringMatchQ[d, ___ ~~ "|" ~~ ___], 
     AppendTo[res, 
      Join[shortredline, raster1[[count1++]], shortredline, 
       raster2[[count2++]], shortredline]],
     True, 
     AppendTo[res, 
      Join[shortsameline, raster1[[count1++]], shortsameline, 
       raster2[[count2++]], shortsameline]]
     ]
    ],
   difference], res="$Failed"];
  res
  ];
  
ShowImageDifferences[url1_String, url2_String, type_String:"URL"] := Module[{dimensions, raster, image},
  (*
  ShowImageDifferences["http://www.google.com", "http://www.google.co.uk", "URL"] 
  ShowImageDifferences["/tmp/Mathematica.nb", "/tmp/test/Mathematica.nb", "NB"] 
  *)
  raster=URLImageDifference[url1, url2, type];
  If[raster==="$Failed", "No image difference",
  dimensions = Reverse@Dimensions[raster, 2];
  image = 
   Graphics[Raster[raster, {{0, 0}, dimensions}, {0, 255}], 
    ImageSize -> dimensions]
  ]
  ];
  
sameLinkTrailsQ[nb1_String, nb2_String] := 
  sameLinkTrailsQ[Quiet@Get@nb1, Quiet@Get@nb2];

sameLinkTrailsQ[nb1Expr_Notebook, nb2Expr_Notebook] := 
  Module[{nb1linktrails, nb2linktrails},
   nb1linktrails = 
    Cases[nb1Expr, Cell[___, "LinkTrail", ___], Infinity];
   nb2linktrails = 
    Cases[nb2Expr, Cell[___, "LinkTrail", ___], Infinity];
   nb1linktrails === nb2linktrails];

samePacletNameCellQ[nb1_String, nb2_String] := 
  samePacletNameCellQ[Quiet@Get@nb1, Quiet@Get@nb2];

samePacletNameCellQ[nb1Expr_Notebook, nb2Expr_Notebook] := 
  Module[{nb1pacletnamecell, nb2pacletnamecell},
   nb1pacletnamecell = 
    Cases[nb1Expr, Cell[___, "PacletNameCell", ___], Infinity];
   nb2pacletnamecell = 
    Cases[nb2Expr, Cell[___, "PacletNameCell", ___], Infinity];
   nb1pacletnamecell === nb2pacletnamecell];

notesInNotesSectionQ[nb_String] := notesInNotesSectionQ[Quiet@Get@nb];

notesInNotesSectionQ[nbExpr_Notebook] := 
  Module[{notessection}, 
   notessection = 
    Cases[nbExpr, Cell[___, "NotesSection", ___], Infinity];
   If[Length[notessection] > 0, 
    If[Length[Cases[nbExpr, Cell[___, "Notes", ___], Infinity]] > 0, 
     True, False ], True]];

seeAlsoInSeeAlsoSectionQ[nb_String] := 
  seeAlsoInSeeAlsoSectionQ[Quiet@Get@nb];

seeAlsoInSeeAlsoSectionQ[nbExpr_Notebook] := 
  Module[{seealsosection}, 
   seealsosection = 
    Cases[nbExpr, Cell[___, "SeeAlsoSection", ___], Infinity];
   If[Length[seealsosection] > 0, 
    If[Length[Cases[nbExpr, Cell[___, "SeeAlso", ___], Infinity]] > 0,
      True, False ], True]];

moreAboutInMoreAboutSectionQ[nb_String] := 
  moreAboutInMoreAboutSectionQ[Quiet@Get@nb];

moreAboutInMoreAboutSectionQ[nbExpr_Notebook] := 
  Module[{moreaboutsection}, 
   moreaboutsection = 
    Cases[nbExpr, Cell[___, "MoreAboutSection", ___], Infinity];
   If[Length[moreaboutsection] > 0, 
    If[Length[Cases[nbExpr, Cell[___, "MoreAbout", ___], Infinity]] > 
      0, True, False ], True]];

corruptedCellsQ[nb_String] := corruptedCellsQ[Quiet@Get@nb];

corruptedCellsQ[nbExpr_Notebook] := 
  Module[{documentationbuilderrors}, 
   documentationbuilderrors = 
    StringCases[ToString[nbExpr], "DocumentationBuild`", Infinity];
   If[Length[documentationbuilderrors] > 0, True, False]];

samePulldownMenusQ[nb1_String, nb2_String] := 
  samePulldownMenusQ[Quiet@Get@nb1, Quiet@Get@nb2];

samePulldownMenusQ[nb1Expr_Notebook, nb2Expr_Notebook] := 
  Module[{nb1lpulldown, nb2lpulldown},
   nb1lpulldown = 
    Cases[nb1Expr, Cell[___, "AnchorBarGrid", ___], Infinity];
   nb2lpulldown = 
    Cases[nb2Expr, Cell[___, "AnchorBarGrid", ___], Infinity];
   nb1lpulldown === nb2lpulldown];

samePulldownMenuTextQ[nb1_String, nb2_String] := 
  samePulldownMenuTextQ[Quiet@Get@nb1, Quiet@Get@nb2];

samePulldownMenuTextQ[nb1Expr_Notebook, nb2Expr_Notebook] := 
  Module[{nb1lpulldown, nb2lpulldown},
   nb1lpulldown = 
    Cases[Cases[nb1Expr, Cell[___, "AnchorBarGrid", ___], Infinity], 
     FrameBox[a_, b___] :> a, Infinity];
   nb2lpulldown = 
    Cases[Cases[nb2Expr, Cell[___, "AnchorBarGrid", ___], Infinity], 
     FrameBox[a_, b___] :> a, Infinity];
   nb1lpulldown === nb2lpulldown];

buttonBoxDataMissingQ[nb_String] := 
  buttonBoxDataMissingQ[Quiet@Get@nb];

buttonBoxDataMissingQ[nbExpr_Notebook] := 
  Module[{allButtons, buttonboxamount}, 
   allButtons = Cases[nbExpr, _ButtonBox, {1, \[Infinity]}]; 
   buttonboxamount = 
    Select[allButtons, 
     FreeQ[#, ButtonData | ButtonFunction, {-1}, Heads -> True] &]; 
   If[Length[buttonboxamount] =!= 0, True, False]];


(******************************************************************************)
(* functionAncestorQ *)

functionAncestorQ[check_Symbol, totalexpr_, subexpr_] := 
 Module[{parentlist = {}, superexpr = totalexpr, sub2 = subexpr}, 
  While[! AtomQ[superexpr] && ! MatchQ[superexpr, {}], 
   superexpr = 
    Cases[{totalexpr}, 
     expr : (f_[___, sub2, ___]) :> (AppendTo[parentlist, f]; expr), 
     Infinity, Heads -> True]; 
   If[Length[superexpr] > 0, superexpr = superexpr[[1]]]; 
   sub2 = superexpr]; MemberQ[Flatten@parentlist, check]];
   
functionAncestorQ[check_List, totalexpr_, subexpr_] :=
 Module[{parentlist = {}, superexpr = totalexpr, sub2 = subexpr}, 
  While[! AtomQ[superexpr] && ! MatchQ[superexpr, {}], 
   superexpr = 
    Cases[{totalexpr}, 
     expr : (f_[___, sub2, ___]) :> (AppendTo[parentlist, f]; expr), 
     Infinity, Heads -> True]; 
   If[Length[superexpr] > 0, superexpr = superexpr[[1]]]; 
   sub2 = superexpr]; MemberQ[Flatten@parentlist, Alternatives@@check]];

(******************************************************************************)



(******************************************************************************)
(* MakeThumbnail *)

Options[MakeThumbnail] = {Framed -> False, FrameMargins -> 10, Excerpt->{{0, 0}, 1}};

MakeThumbnail[expr_, OptionsPattern[]] :=
	Block[{
			thumbnailSize = 300, frameColor = LightGray, frameThickness = 2, 
			excerpt = OptionValue[Excerpt],
			size, nativeSize, thumbnailImage
		},

		size = thumbnailSize - If[OptionValue[Framed], 2 OptionValue[FrameMargins], 0];
		nativeSize = Max[ImageDimensions[If[Head[expr]=!=Image,Rasterize[expr],expr]]];
		thumbnailImage = ImageCrop[
			Rasterize[Cell[BoxData@ToBoxes@Style[expr, Magnification->N[size/nativeSize]/Last[excerpt]],"Output",PageWidth->1000],"Image"],
			{size, size}, If[excerpt[[2]]==1, {0, 0}, ((excerpt[[2]] + 2 # - 1)/(excerpt[[2]] - 1))& /@ excerpt[[1]]], Padding -> White];
		thumbnailImage = If[!OptionValue[Framed],
			thumbnailImage,
			ReplacePixelValue[
				ImageCrop[thumbnailImage, thumbnailSize {1, 1}, Padding -> White], 
				{{1;;thumbnailSize, 1;;frameThickness} -> frameColor,
				{1;;thumbnailSize, (thumbnailSize - frameThickness + 1);;thumbnailSize} -> frameColor,
				{1;;frameThickness, 1;;thumbnailSize} -> frameColor, 
				{(thumbnailSize - frameThickness + 1);;thumbnailSize, 1;;thumbnailSize} -> frameColor
			}]
		];
		CellPrint[Cell[BoxData[ToBoxes[Row[{Style[thumbnailImage, Magnification->0.5, ImageSizeMultipliers->{}], " (size preview)"}]]],
			"Output", "SuppressedMaterial"]];
		CellPrint[Cell[BoxData[ToBoxes[thumbnailImage]], "Output", "AlternateThumbnail"]]
	]
(******************************************************************************)

(******************************************************************************)
(* handle special links in the WSM docs *)

(* beware recursion: rhs of a rule should not contain the lhs of the rule *)
stringReplaceRepeated[s_String, rules___] := FixedPoint[StringReplace[#, rules]&, s];

(* this returns a valid http:// url for every known modelica:// link in WSM 4.1 *)
convertModelicaLink[link_String] := 
  If[StringMatchQ[link, StartOfString ~~ "modelica://" ~~ __],
    (* StringReplace[link, RegularExpression["modelica:///([^\\.]+)(.*\)"] -> "http://reference.wolfram.com/system-modeler/libraries/$1/$1$2.html"], *)
    StringReplace[link, RegularExpression["modelica:///([^\\.]+)(.*)"] -> "/system-modeler/libraries/$1/$1$2.html"],
    link
  ]

(* this returns a valid http:// url for every known wsm:// link in WSM 4.1 *)
convertWSMLink[link_String] := 
  If[StringMatchQ[link, StartOfString ~~ "wsm://" ~~ __],
    stringReplaceRepeated[link, {

      ".nb" ~~ EndOfString                   -> ".html",
      (* StartOfString ~~ "wsm://"              -> "http://reference.wolfram.com", *) (* absolute paths *)
      StartOfString ~~ "wsm://"              -> "", (* relative paths *)
      "/mathematica/ReferencePages/Symbols/" -> "/system-modeler/WSMLink/ref/",
      "/mathematica/Tutorials/"              -> "/system-modeler/WSMLink/tutorial/",
      "/ref/GettingStarted/"                 -> "/system-modeler/GettingStarted/",
      "/ref/UserGuide/"                      -> "/system-modeler/UserGuide/",
      "/ref/root.html"                       -> "/system-modeler/",
      "/ref/guide/WolframRoot.html"          -> "/language/guide/WolframRoot.html",
      "/ref/WhatsNew/"                       -> "/system-modeler/WhatsNew/",

      (* temporary kludge til the file is renamed *)
      "ReliabilityInSystemModeler.html"      -> "ReliabilityinSystemModeler.html"
    }],
    link
  ]

(******************************************************************************)

CheckSeeAlsos[nbexpr_, info_:DocumentationBuild`Export`$CoreInformation["English"]] :=
  Module[{seealsocell = First[Cases[nbexpr, Cell[___,"SeeAlso",___], Infinity]], uris, notOKuris, newcell},
    If[MatchQ[seealsocell, {}], Return[nbexpr]];
    uris = Cases[seealsocell, TemplateBox[{_, uri_}, __] :> uri, Infinity];
    If[MatchQ[uris, {}], Return[nbexpr]];
    notOKuris = Select[uris, Module[{uriinfo = (# /. info)}, MatchQ[uriinfo, {}] || 
      MatchQ["Flag" /. uriinfo, "FutureFlag"|"ExcisedFlag"]] &];
    If[Length[notOKuris] > 0, Global`AntLog["Bad URIs in See Alsos: ", notOKuris]];
    newcell = CollapseListings[DeleteCases[seealsocell, Cell[c_,___]/;
      MemberQ[c, Alternatives@@notOKuris, Infinity], Infinity], "Symbol"];
    ReplaceAll[nbexpr, seealsocell -> newcell]
  ]

CheckGuideInlineListings[nbexpr_, info_:DocumentationBuild`Export`$CoreInformation["English"]] :=
  Module[{listingcells},
  (* TODO: write this function *)
  nbexpr]

CollapseListings[cell_, type_:"Symbol"]:=
 Module[{newcell, buttons, delimiter = StyleBox["\[MediumSpace]\[FilledVerySmallSquare]\[MediumSpace]", "InlineSeparator"]},

  buttons = Cases[cell, Cell[__, "InlineFunctionSans" | "InlineFormula", ___], Infinity];
  newcell = 
     Cell[TextData[Flatten[Append[{#, "\[NonBreakingSpace]", delimiter, " "}& /@ Most[buttons], Last[buttons]]]],
       Sequence@@Rest[cell]]
  ]

End[]
EndPackage[]
