(* :Title: Info.m *)

(* :Authors:
	Andrew Hunt, Shaun McCance, Todd Gayley
	andy@wolfram.com, shaunm@wolfram.com, tgayley@wolfram.com
*)

(* :Package Version: 0.50 *)

(* :Mathematica Version: 6.0 *)

BeginPackage["DocumentationBuild`Info`", {
    "DocumentationBuild`Common`",
    "DocumentationBuild`Utils`",
    "Transmogrify`"
}];


(******************************************************************************)
(** Usage Statements **)

HasSourceInformation::usage = "";

GetSourceInformation::usage = "GetSourceInformation[uri, lang, keys___] looks up uri\
 in the source information for language lang.  If keys are given, it looks up information\
 for those keys.";

GetNotebookInformation::usage = "";
GetNotebookTitle::usage = "";
GetNotebookURI::usage = "";
GetNotebookHistoryData::usage = "";
GetNotebookCategorization::usage = "";
GetNotebookDescription::usage = "";
GetNotebookKeywords::usage = "";
GetNotebookSynonyms::usage = "";
GetNotebookFlag::usage = "";
GetNotebookSummary::usage = "";
GetNotebookKeywords::usage = "";
GetNotebookSynonyms::usage = "";
GetFunctionDevelopers::usage = "";
GetCoreInformationLinkTrail::usage = "";
GetCoreInformationTitle::usage = "";
GetUnderdevelopment::usage = "";
GetButtonTextAndURL::usage = "";
GetTutorialCollectionLinks::usage = "";
GetWebResourcesLinks::usage = "";

ExportSourceInformation::usage = "ExportSourceInformation[dir,langs] exports a list of information about source notebooks to a standard location.";
ExportOnlineInformation::usage = "ExportOnlineInformation[dir,langs] exports a list of information about online notebooks to a standard location.";
ExportAllOnlineInformation::usage = "";
CreateSourceInformation::usage = "CreateSourceInformation[dir,lang] creates a list of information about source notebooks to a standard location.";
CreateOnlineInformation::usage = "CreateOnlineInformation[dir,lang] creates a list of information about online notebooks to a standard location.";
CreateCommonInformation::usage = "";
MakePrimaryGuideList::usage = "";

(* Load foreign function list *)
MakeForeignFunctionKeyword::usage = "ForeignFunctionKeywords[] creates a map from the functions in darta/keywords/ForeignFunctions.";
GetForeignFunctionKeywords::usage = "";
ForeignFunctionKeywords::usage = "";

CreateGuidesLists::usage = "CreateGuidesList[dir,langs] creates a map from the functions in dir/lang/System/ReferencePages to the guides in dir/lang/System/Guides they appear in, where lang is each element in langs";
GetDemonstrationSymbolList::usage = "GetDemonstrationSymbolList[] returns a list of current symbols categories used in demonstrations.w.c.";
GetDemonstrationKeywordList::usage = "";
LoadCoreInformation::usage = "";


Begin["`Private`"];


(******************************************************************************)
(** LoadCoreInformation  **)
LoadCoreInformation::missing = "LoadCoreInformation could not find default CoreInformation file.";

LoadCoreInformation[info_List, language_String] := 
	Module[{data}, data = info; 
		If[Length[data]>0, 
			Dispatch[data], 
			LoadCoreInformation[Automatic, language]]];

LoadCoreInformation[file_String, language_String] := 
Module[{data}, data = Get[file]; 
		If[Length[data]>0, 
			Dispatch[data], 
			LoadCoreInformation[Automatic, language]]];
  
LoadCoreInformation[Automatic, language_String] := 
Module[{data}, data = Get[ToFileName[
    {$DocumentationBuildDirectory, "Internal", "data"},
    StringJoin["SourceInformation.",
               Switch[language, "Japanese", "ja", "ChineseSimplified", "zh", "Spanish", "es", _,"en"],
               ".m"]]]; 
		If[Length[data]>0, 
			Dispatch[data], 
			Message[LoadCoreInformation::missing]]];

(******************************************************************************)
(** HasSourceInformation **)
HasSourceInformation[u_, lang_] :=
  Module[{uri = u},
    If[!StringMatchQ[uri, "paclet:*"], uri = "paclet:"<>uri];
    (uri /. DocumentationBuild`Export`$CoreInformation[lang]) =!= uri
  ];


(******************************************************************************)
(** GetSourceInformation **)
GetSourceInformation::nouri = "GetSourceInformation could not find `1`.";
GetSourceInformation::nokey = "GetSourceInformation could not find the key `1` for `2`.";
GetSourceInformation[u_, lang_, keys___] :=
  Module[{uri = u, info},
    If[!StringMatchQ[uri, "paclet:*"], uri = "paclet:"<>uri];
    info = uri /. DocumentationBuild`Export`$CoreInformation[lang] /. {uri -> None};
    If[info === None,
(*
      Message[GetSourceInformation::nouri, uri];
*)
      Return[None]];
    If[Length[{keys}] === 0,
      info,
      Fold[
        Function[{l, k},
          If[MatchQ[l, {___, _[k, _], ___}],
            k /. l,
            (* Message[GetSourceInformation::nokey, {keys}, uri]; *)
            Return[None]
        ]],
        info, {keys}]]
  ];


(******************************************************************************)
(** GetNotebookInformation **)
GetNotebookInformation[file_String /; FileType[file] === File, opts___?OptionQ] := 
  GetNotebookInformation[ Get@file, opts]
GetNotebookInformation[expr_Notebook, opts___?OptionQ] := Module[{uri, type, title, contextval, contexttemp, pacletnameval},
  (* get EntityType *)
  type = GetNotebookCategorization[expr, "Entity Type"];
  (* was EntityType found? If not, return $Failed *)
  If[type === $Failed, Return[$Failed] ];
  (* continue processing nb *)
  uri = GetNotebookURI[expr, opts];
  title = GetNotebookTitle[expr, type];

  If[type === "Guide" && MatchQ[uri, (* XX Root guide rename change XX *)"guide/WolframRoot"|"guide/Mathematica"], 
    type = "Root Guide";
    title = "Wolfram Mathematica";
  ];
  
    pacletnameval=GetNotebookCategorization[expr, "Paclet Name"];
  (*This code handles where the context is blank but is part of Mathematica*)
  contexttemp=GetNotebookCategorization[expr, "Context"];
  contextval=If[FreeQ[pacletnameval, "Mathematica" | ""], contexttemp, If[FreeQ[contexttemp, ""], contexttemp, "System`"]];
  
  {
    "Title" -> title,
    "TitleModifier" -> GetNotebookCategorization[expr, "Title Modifier"],
    "HistoryData" -> GetNotebookHistoryData[expr],
    "EntityType" -> type,
    "PacletName" -> pacletnameval,
    "Context" -> contextval,
    "URI" -> uri,
    "Keywords" -> GetNotebookKeywords[expr],
    "Underdevelopment" -> GetUnderdevelopment[expr],
    "TutorialCollectionLinks" -> GetTutorialCollectionLinks[expr],
    "ForeignFunctionKeywords" -> {},
    "Synonyms" -> GetNotebookSynonyms[expr],
    "Summary" -> GetNotebookSummary[expr, type, title],
    "Flag" -> GetNotebookFlag[expr],
    "IndexQ" -> GetNotebookIndexQ[type]
  }];



GetNotebookTitle::notitle= "The title of the notebook could not be found. It should be in a cell of style, `1`."

(******************************************************************************)
(** GetNotebookTitle **)
GetNotebookTitle[expr_] :=
  GetNotebookTitle[expr, GetNotebookCategorization[expr, "Entity Type"]];
GetNotebookTitle[expr_, type_] :=
  Module[{cellStyle, title},
    cellStyle = Switch[type,
      ("Character" | "Character Name"),
        ("CharacterName"),
      ("Solution"),
        ("SolutionsTitle"),
      ("HowTo"),
        ("Title"),
      ("Guide"),
        ("GuideTitle" | "GuideTOCTitle"),
      ("Guide Overview"),
        ("TOCDocumentTitle"),
      ("Note"),
        ("DataSourceTitle"),
      ("MarketingPage" | "ComparisonTitle"),
        ("MarketingTitle"),
      ("Tutorial" | "Upgrade Information" | "Overview"),
        ("Title" | "Section" | "TOCDocumentTitle" | "DataSourceTitle" | "UpgradeObjectName"),
      ("Featured Example" | "FeaturedExample"),
        ("FeaturedExampleTitle"),
      ("Example Page" | "Example"),
        ("FeaturedExampleTitle"),
      _,
        ("ObjectName" | "ObjectNameSmall" | "ObjectNameAlt" | "UpgradeObjectName" | "DataSourceTitle")
    ];
    title = Cases[expr, Cell[c_, cellStyle, ___] :> c, Infinity];
    title = DeleteCases[title, StyleBox[_, "FullListing"], Infinity];
    If[ Length[ title] === 0, Message[GetNotebookTitle::notitle, cellStyle]];
    title = Transmogrify`ConvertToString[First[title]];
    title = If[StringFreeQ[title, "\n"], title, First@StringSplit[title, "\n"]];
    title = StringReplace[title, {
        "\[OpenCurlyDoubleQuote]" | "\[CloseCurlyDoubleQuote]" | "\[OpenCurlyQuote]" | "\[CloseCurlyQuote]" -> "'",
        "\[FilledRightTriangle]" -> ">",
        "\[Hyphen]" -> "-",
        "\"" -> "",
        " \[MediumSpace]" -> "",
        "\[MediumSpace]" -> "",
        " - Wolfram Mathematica" -> ""
    }];
    title
  ];


(******************************************************************************)
(** GetNotebookURI **)
GetNotebookURI[nb_Notebook, opts___?OptionQ] :=
  Module[{uri, pname, fname, type},
    uri = ToString[GetNotebookCategorization[nb, "URI"]];
    (* check to see if uri is fake paclet:dir/XXX *)
    uri = If[ StringMatchQ[uri, RegularExpression[".*XXX+"]], "$Failed", uri];
    If[uri === $Failed || uri === "$Failed",
      pname = ToString[GetNotebookCategorization[nb, "Paclet Name"]];
      type = ToString[GetNotebookCategorization[nb, "Entity Type"]];
      uri = If[FreeQ[{"", "Mathematica"}, pname], pname<>"/", ""];
      uri = uri <> Switch[type,
        "AppleScript",         "ref/applescript",
        "Character Name",      "ref/character",
        "Character",           "ref/character",
        "Example",             "example",
        "File",                "ref/file",
        "Format",              "ref/format",
        "Embedding Format",    "ref/embeddingformat",
        "Interpreter",         "ref/interpreter",
        "Front End Object",    "ref/frontendobject",
        "Guide Overview",      "guide",
        "Guide",               "guide",
        "HowTo",               "howto",
        "Indicator",           "indicator",
        "LibraryLink C Function", "ref/callback",
        "MathLink C Function", "ref/c",
        "Menu Item",           "ref/menuitem",
        "Message",             "ref/message",
        "Method",              "ref/method",
        "Note",                "note",
        "Overview",            "tutorial",
        "Root Guide",          "guide",
        "Symbol",              "ref",
        "Screencast",          "screencast",
        "System Program",      "ref/program",
        "Tutorial",            "tutorial",
        "Upgrade Information", "tutorial",
        _, ""
      ];
      fname = "FilePath" /. {opts}; /. "FilePath" -> "FIXME";
      fname = FileBaseName[fname];
      uri = uri <> "/" <> fname;
      uri = Switch[uri, "$Failed//FilePath", "FIXME", "guide/FilePath", "FIXME", "tutorial/FilePath", "FIXME", "ref/FilePath", "FIXME", _, uri];
    ];
    uri
  ];


(******************************************************************************)
(** GetNotebookHistoryData **)
GetNotebookHistoryData::notfound = "Could not find the history data for `1`";
GetNotebookHistoryData[nb_Notebook] := Map[
  Function[{type},
    type -> If[# === {},
      Message[GetNotebookHistoryData::notfound, type]; "",
      First[#]
    ]&[
      Cases[nb,
        Cell[c_String, "HistoryData", ___, CellTags -> type, ___]
          :> StringReplace[c, " " -> ""],
        Infinity
    ]]
  ],
  {"New", "Modified", "Obsolete", "Excised"}];

GetNotebookHistoryData[___] := $Failed


(******************************************************************************)
(** GetNotebookCategorization **)
GetNotebookCategorization::val = "`1` value not found in notebook.";
GetNotebookCategorization[nb_String, cat_String] :=
  GetNotebookCategorization[GetQuietly[nb], cat];
GetNotebookCategorization[expr_Notebook, cat_String] :=
  Module[{val},
    val = Cases[expr,
      Cell[t_, "Categorization", ___, CellLabel->cat, ___] :> t,
      Infinity];
    ServerLog[ separator <> "\nGetNotebookCategorization: " <> ToString[{cat, val}]  ];
    If[Length[val] > 0, 
    	If[cat === "Title Modifier", 
 DocumentationBuild`Utils`makeTitleModiferString[
  DeleteCases[Map[If[# === "XXXX", "", #] &, val], ""]], First[val]], 
    	If[cat==="Title Modifier", "", 
    		Message[GetNotebookCategorization::val, cat]; $Failed]]
  ];


(******************************************************************************)
(** GetNotebookKeywords **)
GetNotebookKeywords[expr_Notebook] :=
  Cases[expr, Cell[c_String /; !StringMatchQ[c, RegularExpression["X*"]], ___, "Keywords", ___] :> 
    StringReplace[c, "," -> ""], Infinity];


(******************************************************************************)
(** GetNotebookSynonyms **)
GetNotebookSynonyms[expr_Notebook] :=
  Cases[expr, Cell[c_String /; !StringMatchQ[c, RegularExpression["X*"]], ___, "Synonyms", ___]:> 
    StringReplace[c, "," -> ""], Infinity];


(******************************************************************************)
(** GetFunctionDevelopers **)
GetFunctionDevelopers[expr_Notebook] :=
  Cases[expr, Cell[d_String, "Details", CellLabel->"Developers", ___]:> d, Infinity];

(******************************************************************************)
(** GetUnderdevelopment**)
GetUnderdevelopment[expr_Notebook] :=
  If[(Length[Cases[expr, "U N D E R   D E V E L O P M E N T", Infinity]]>0) ||
  	  (ScreenStyleEnvironment /. Options[expr]) === "ExperimentalObject", True, False];
  
(******************************************************************************)
(** GetButtonTextAndURL **)

GetButtonTextAndURL[cell_Cell] := Module[{buttontext, buttonurl}, 
       buttontext = First[Cases[cell, ButtonBox[aa_, ___] :> aa, Infinity]]; 
       buttonurl = First[Cases[cell, URL[bb_, ___] :> bb, Infinity]]; 
                 Flatten[{buttontext, buttonurl}]];

(******************************************************************************)
(** GetTutorialCollectionLinks **)

GetTutorialCollectionLinks[expr_Notebook] :=
Module[{guidetutorialcollectioncells, tutorialcollectiontitleandlinks},
	    guidetutorialcollectioncells = 
	       Cases[expr, Cell[_, "GuideTutorialCollection"|"TutorialCollection", ___], Infinity];
        tutorialcollectiontitleandlinks=
           If[Length[guidetutorialcollectioncells]>0, 
    	      Map[GetButtonTextAndURL, guidetutorialcollectioncells], 
    	    	{}]
    ];
    
(******************************************************************************)
(** GetWebResourcesLinks **)

GetWebResourcesLinks[expr_Notebook] := 
 Module[{links}, 
  links = Flatten[(Cases[expr, 
        Cell[l_, ___, "WebResources", ___, 
          CellLabel -> #, ___] :> {l, #}, 
        Infinity] & /@ {"Training Courses", "Videos", 
       "Demonstrations", "Community"}), 1];
  DeleteCases[links, {"XXXX", ___} | {}]
  ];

(******************************************************************************)
(** GetCoreInformationLinkTrail **)
GetCoreInformationLinkTrail[u_String, language_String] :=
Module[{linktrails},
	linktrails=DocumentationBuild`Utils`GetValueFromOptions[DocumentationBuild`Export`$CoreInformation[language], u, "LinkTrails"];
	linktrails=Switch[linktrails, "$Failed", (* XX Root guide rename change XX *){{"paclet:guide/WolframRoot"}}, {{{"$Failed"->"$Failed"}}}, (* XX Root guide rename change XX *){{"paclet:guide/WolframRoot"}}, _, linktrails];
	linktrails
];

(******************************************************************************)
(** GetCoreInformationTitle **)
GetCoreInformationTitle[u_String, language_String] :=
Module[{title},
	title=DocumentationBuild`Utils`GetValueFromOptions[DocumentationBuild`Export`$CoreInformation[language], u, "Title"];
	title=If[title==="$Failed", "Mathematica", title];
	title
];


(******************************************************************************)
(** GetNotebookFlag **)

(* This is identical as DocumentationBuild`Utils`DocumentFlag, and used far less... *)
GetNotebookFlag = DocumentationBuild`Utils`DocumentFlag;
(*GetNotebookFlag[expr_Notebook]:=
Module[{f},
  f = Cases[expr, Cell[_, "ObsoleteFlag"|"InternalFlag"|"TemporaryFlag"|"PreviewFlag"|"FutureFlag"|"ExcisedFlag"|"AwaitingFutureDesignReviewFlag", ___], Infinity];
  If[Length@f > 0,
  	If[Length[Cases[expr, "U N D E R   D E V E L O P M E N T", Infinity]]>0, 
  		Switch[Part[List@@First@f,2], 
  			"ObsoleteFlag", None,
  			"InternalFlag", Part[List@@First@f,2],
  			"TemporaryFlag", None,
  			"PreviewFlag", None,
  			"FutureFlag", None,
  			"ExcisedFlag", Part[List@@First@f,2],
  			"AwaitingFutureDesignReviewFlag", None,  
  			_ , Part[List@@First@f,2]], 
  			Part[List@@First@f,2]
  			],  
  	None]
];*)


(******************************************************************************)
(** GetNotebookSummary **)
GetNotebookSummary[expr_Notebook, type_String, name___String] :=
Module[{text, summary},
  (* test to see if nb already has summery in TaggingRules *)
  summary = Cases[expr, Rule[TaggingRules, {___, Rule["Metadata", {___, Rule["summary", s_], ___}], ___}] :> s, Infinity];
  If[summary =!= {}, Return[summary] ];
  (* compute summary *)
  text = ExtractSummaryText[expr, type, name];
  (* Fix weird characters bug (bug 246625 and others) -- edit: turns out this was fixed in Make.m;
     apparently the ButtonBox->TemplateBox conversion was messing this up *)
  (*text = StringReplace[text, "\"\!\(" ~~ Shortest[___] ~~ "StyleBox[\"" ~~ Shortest[content__] ~~ 
  "\", \"TI\"]" ~~ Shortest[___] ~~ "\"" :> content];*)
  (* Check for content. *)
  If[text === "", Return[" "] ];
  text = FixedPoint[StringReplace[#, {
             RegularExpression["^\"(.*)\"$"] -> "$1", 
             "\[OpenCurlyDoubleQuote]" | "\[CloseCurlyDoubleQuote]" | "\[OpenCurlyQuote]" | "\[CloseCurlyQuote]"->"'",
             "\[AliasIndicator]" -> "ESC", 
             "\[Hyphen]" -> "-",
             "\[FilledRightTriangle]" -> ">",
             "\[RightGuillemet]" -> ">",
             "\n" | "\\[FilledSmallSquare]" | "\r" | "\[LineSeparator]" | "\[ParagraphSeparator]" -> " ",
             (* Linear Syntax strings *)
             RegularExpression["StyleBox\\[\\s*\\\\\"(\\w+)\\\\\",\\s*\\\\\"\\w+\\\\\"\\s*]"] -> "$1",
             (*RegularExpression["StyleBox\\[\\s*\\\\\"(.+)\\\\\",\\s*\\\\\"\\w+\\\\\"\\s*]"] -> "$1", *)
             "\\!\\(\\*" -> "", "\\(" -> "", "\\)" -> "", "\\!" -> "", "\\*" -> "",
             RegularExpression["^\\s+"] -> "", 
             RegularExpression["\\s+"] -> " ",
             RegularExpression["\\[(\")*\\s(\\w)"] -> "[$1$2",
             RegularExpression[",\\s{\"\\s(\\w)"] -> ", {\"$1"
             (*  Remove SubscriptBox  *)
             (* Shortest["\"SubscriptBox[" ~~ a___ ~~ ", " ~~ b___ ~~ "]" ~~ c___ ~~ "\""] :> "\"" <> a <> "_" <> b <> c <> "\"" *)
           }]&, text];
  (* safety fallthrough *)
    If[Head@text === String, StringReplace[StringReplace[text, {Shortest[
   "\"SubscriptBox[" ~~ aa___ ~~ ", " ~~ bb___ ~~ "]" ~~ cc___ ~~ 
    "\""] :> "\"" <> aa <> "_" <> bb <> cc <> "\"", 
 Shortest["SubsuperscriptBox[" ~~ aa___ ~~ ", " ~~ bb___ ~~ ", " ~~ 
    cc___ ~~ "]" ~~ dd___ ~~ ""] :> 
  "" <> aa <> "_" <> bb <> "^" <> cc <> "", 
 Shortest["DynamicBox[ToBoxes[If[$OperatingSystem === MacOSX, " ~~ \
___ ~~ ", " ~~ a___ ~~ "], StandardForm], ImageSizeCache -> " ~~ ___ ~~
     "}]" ] :> a, 
 Shortest["TagBox[" ~~ a___ ~~ 
    ", EventHandlerTag[{ \"MouseDown\" -> Null, PassEventsDown -> \
Automatic, PassEventsUp -> True}]]"] :> a, 
 "GraphicsBox[TagBox[RasterBox[CompressedData[" ~~ ___ :> "", 
 "StyleBox[context`]" :> "context`", 
 "[\"StyleBox[\\\"context`\\\", \\\"TI\\\"]\"]" :> "[\"context\"]", 
 "StyleBox[x, TI]" :> "x", "StyleBox[package`]" :> "package`", 
 "StyleBox[\[Theta], TR]" :> "\[Theta]", 
 "\" StyleBox[\\\"context`\\\", \\\"TI\\\"]\"" :> "\"context`\"", 
 "StyleBox[\"E\"_FontWeight->\"Bold\"^FontSlant->\"Plain\", \
\"\\[Nu]\", \"\\[Mu]\"](z)" :> "E_\[Nu]^\[Mu](z)", 
 "GridBox[{ { RowBox[{ RowBox[{a_, 1}], ,, ..., ,, a_p}]}, { RowBox[{ \
RowBox[{b_, 1}], ,, ..., ,, b_q}]} }]). " :> 
  "{{a_1, ..., a_p}, {b_1, ... , b_p}}", 
 "UnderscriptBox[lim, m \\[Rule] 0, LimitsPositioning->True]" :> 
  "lim_(m->0)", 
 "UnderoverscriptBox[\"\\[Product]\", RowBox[{\"i\", \"=\", \
\"i_min\"}], \"i_max\", LimitsPositioning->True]\\ f" :> 
  "\[Product]_(i=i_min)^(i_max) f", 
 "UnderoverscriptBox[\"\\[Sum]\", RowBox[{\"i\", \"=\", \"i_min\"}], \
\"i_max\", LimitsPositioning->True]\\ f" :> 
  "\[Sum]_(i=i_min)^(i_max) f", 
 "UnderoverscriptBox[\"\\[Product]\", RowBox[{\"i\", \"=\", \"1\"}], \
\"i_max\", LimitsPositioning->True]expr" :> 
  "\[Product]_(i=1)^(i_max) expr", 
 "UnderoverscriptBox[\"\\[Product]\", RowBox[{\"i\", \"=\", \
\"i_min\"}], \"i_max\", LimitsPositioning->True] \
UnderoverscriptBox[\"\\[Product]\", RowBox[{\"j\", \"=\", \
\"j_min\"}], \"j_max\", LimitsPositioning->True]...expr" :> 
  "\[Product]_(i=i_min)^(i_max) \[Product]_(j=j_min)^(j_max) \
...expr", 
 "UnderoverscriptBox[\"\\[Sum]\", RowBox[{\"i\", \"=\", \"1\"}], \
\"i_max\", LimitsPositioning->True]expr" :> 
  "\[Sum]_(i=1)^(i_max) expr", 
 "UnderoverscriptBox[\"\\[Sum]\", RowBox[{\"i\", \"=\", \"i_min\"}], \
\"i_max\", LimitsPositioning->True] UnderoverscriptBox[\"\\[Sum]\", \
RowBox[{\"j\", \"=\", \"j_min\"}], \"j_max\", \
LimitsPositioning->True]...expr" :> 
  "\[Sum]_(i=i_min)^(i_max) \[Sum]_(j=j_min)^(j_max) ...expr", 
 "UnderoverscriptBox[\"\\[Product]\", RowBox[{\"i\", \"=\", \"1\"}], \
\"i_max\", LimitsPositioning->True]\\ f" :> 
  "\[Product]_(i=1)^(i_max) f", 
 "UnderoverscriptBox[\"\\[Product]\", RowBox[{\"i\", \"=\", \
\"i_min\"}], \"i_max\", LimitsPositioning->True] \
UnderoverscriptBox[\"\\[Product]\", RowBox[{\"j\", \"=\", \
\"j_min\"}], \"j_max\", LimitsPositioning->True]...\\ f" :> 
  "\[Product]_(i=i_min)^(i_max) \[Product]_(j=j_min)^(j_max) f", 
 "UnderscriptBox[\\[Product], i]f" :> "\[Product]_i f", 
  "\\[LeftBracketingBar]" :> "|", 
  "\\[RightBracketingBar]" :> "|",
 "TagBox[RowBox[{\"(\", GridBox[{ { TagBox[\"n\", Identity, \
Editable->True, Selectable->True]}, { TagBox[\"m\", Identity, \
Editable->True, Selectable->True]} }], \")\"}], \
InterpretTemplate[Cell$$986`QBinomial[#, #2, #3]& ], Editable->False, \
Selectable->False] q" :> "(n; m)_q", 
 "( GraphicsBox[RasterBox[CompressedData[ ], {Offset[{0.5, -21.5}, \
{0.5, 0.5}], Offset[{21.5, -0.5}, {0.5, 0.5}]}], ImageSize -> {19., \
Automatic}] \
1:eJyt0tENgCAMBNA7ncEBXMkRWID9/4wJjRTa0kSP8Pe40MBZ6lV2AEfbG/4L \
mXbM2MchYVsfVa9xRvqge+nZuZOuHe5JBlbNQ4rtl1jM8o3Vb0nXDTJwSoau \
kwvXz75K1rlvZMmkM//St9y9xwKJ , GraphicsBox[RasterBox[CompressedData[ \
], {Offset[{0.5, -21.5}, {0.5, 0.5}], Offset[{21.5, -0.5}, {0.5, \
0.5}]}], ImageSize -> {20., Automatic}] \
1:eJxTTMoPSmJmYGAQhWImBvoDRjAgRh0DGBJSy8gINRJZLVY9OFRiUYtdJSNO \
tWjuhOrEphbN73BbUCB2x6MBHOowVOJUh6YSjzoUlXjVIakkoA7Z74QAsepw \
xhE2lUSqw5GWKAEAwE0Cig== , and GraphicsBox[RasterBox[CompressedData[ \
], {Offset[{0.5, -21.5}, {0.5, 0.5}], Offset[{21.5, -0.5}, {0.5, \
0.5}]}], ImageSize -> {20., Automatic}] \
1:eJxTTMoPSmJmYGAQhWImBnoCRiAgViUDZWpx6MWilpFotSAecWoZGfGqRVcJ \
VosMcWlDAbjMR1eJUx2aSjzqUFTiVYekkoA6ZL8TAsSqwxtH6CqJVIczLZEP AHfnAm0= \
). Double-click the background and press the r key to make the \
Reshape tool ( GraphicsBox[RasterBox[CompressedData[ ], {Offset[{0.5, \
-21.5}, {0.5, 0.5}], Offset[{21.5, -0.5}, {0.5, 0.5}]}], ImageSize -> \
{15., Automatic}] \
1:eJxTTMoPSmJmYGAQhWImhmEKGElQSaxaRkZi1TIyEquWkRGmFhniUYkAOM0n \
Vh2aSrzuJVYdkkqC/idWHYnhSWwcER/vRKojGgAAJPYCTA== )" :> "()"}], 
{"StyleBox[\"E\"_FontWeight->\"Bold\"^FontSlant->\"Plain\", \"\\[Nu]\", \
\"\\[Mu]\"](z).":>"E_\\[Nu]^\\[Mu](z).", "Unicode: 2211. Alias: Esc\[ThinSpace]sum\[ThinSpace]Esc. Compound \
operator with built-in evaluation rules. UnderoverscriptBox[\[Sum], \
i, SubscriptBox[i, max], LimitsPositioning -> False]f is by default \
interpreted as Sum[f,{i,i_max}]. UnderoverscriptBox[\[Sum], \
SubscriptBox[StyleBox[RowBox[{i, =, i}], TI], min], SubscriptBox[i, \
max], LimitsPositioning -> False]f is by default interpreted as \
Sum[f,{i,i_min,i_max}]. Not the same as the Greek letter \
\\[CapitalSigma]. " :> "Unicode: 2211. Alias: Esc\[ThinSpace]sum\
\[ThinSpace]Esc. Compound operator with built-in evaluation rules. \
\[Sum]_(i)^i_max(f) is by default interpreted as Sum[f,{i,i_max}]. \
\[Sum]_(i=i_min)^i_max(f)f is by default interpreted as \
Sum[f,{i,i_min,i_max}]. Not the same as the Greek letter \
\\[CapitalSigma]. ", "Unicode: 220F. Alias: Esc\[ThinSpace]prod\[ThinSpace]Esc. Compound \
operator with built-in evaluation rules. \
UnderoverscriptBox[\[Product], i, SubscriptBox[i, max], \
LimitsPositioning -> False]f is by default interpreted as \
Product[f,{i,i_max}]. UnderoverscriptBox[\[Product], \
SubscriptBox[StyleBox[RowBox[{i, =, i}], TI], min], SubscriptBox[i, \
max], LimitsPositioning -> False]f is by default interpreted as \
Product[f,{i,i_min,i_max}]. Not the same as the Greek letter \
\\[CapitalPi]. " :> "Unicode: 220F. Alias: Esc\[ThinSpace]prod\
\[ThinSpace]Esc. Compound operator with built-in evaluation rules. \
\[Product]_(i)^i_max(f) is by default interpreted as \
Product[f,{i,i_max}]. \[Product]_(i=i_min)^i_max(f) is by default \
interpreted as Product[f,{i,i_min,i_max}]. Not the same as the Greek \
letter \\[CapitalPi]. ", "Unicode: F361. Alias: Esc\[ThinSpace]pw\[ThinSpace]Esc. Prefix \
operator with built-in evaluation rules. \
\[Piecewise]GridBox[{{SubscriptBox[StyleBox[e, TI], StyleBox[1, TR]], \
SubscriptBox[StyleBox[c, TI], StyleBox[1, TR]]}, \
{SubscriptBox[StyleBox[e, TI], StyleBox[2, TR]], \
SubscriptBox[StyleBox[c, TI], StyleBox[2, TR]]}}] is by default \
interpreted as Piecewise[{{e_1,c_1},{e_2,c_2}}]. Extensible \
character. Not the same as keyboard {. " :> "Unicode: F361. Alias: \
Esc\[ThinSpace]pw\[ThinSpace]Esc. Prefix operator with built-in \
evaluation rules. \[Piecewise]Grid[{{e_1, c_1}, {e_2, c_2}}] is by \
default interpreted as Piecewise[{{e_1,c_1},{e_2,c_2}}]. Extensible \
character. Not the same as keyboard {. ", "This constructs the sum UnderoverscriptBox[\[Sum], RowBox[{i, =, \
1}], 7, LimitsPositioning -> True] ( x^i ) / ( i ) . You can leave \
out the lower limit if it is equal to 1. This makes i increase in \
steps of 2, so that only odd-numbered values are included. " :> "This \
constructs the sum \[Sum]_(i=1)^7 (x^i)/(i) . You can leave out the \
lower limit if it is equal to 1. This makes i increase in steps of 2, \
so that only odd-numbered values are included. ", "Basic descriptive statistics operations. Given a list with n \
elements x_i, the mean Mean[list] is defined to be \
\[Mu](x)=OverscriptBox[x, _]=\[Sum]x_i/n. The variance Variance[list] \
is defined to be var(x)=\[Sigma]^2(x)=\[Sum](x_i-\[Mu](x))^2/(n-1), \
for real data. (For complex data \
var(x)=\[Sigma]^2(x)=\[Sum](x_i-\[Mu](x))(OverscriptBox[RowBox[{\
SubscriptBox[x, i], -, RowBox[{\[Mu], (, x, )}]}], _])/(n-1).) " :> \
"Basic descriptive statistics operations. Given a list with n \
elements x_i, the mean Mean[list] is defined to be \
\[Mu](x)=x^=\[Sum]x_i/n. The variance Variance[list] is defined to be \
var(x)=\[Sigma]^2(x)=\[Sum](x_i-\[Mu](x))^2/(n-1), for real data. \
(For complex data \
var(x)=\[Sigma]^2(x)=\[Sum](x_i-\[Mu](x))(x_i-\[Mu](x))^/(n-1).) ", "SqrtBox[RowBox[{-, 1}]]" :> "Sqrt[-1]", "Mathematical constants. Euler's constant EulerGamma is given by the \
limit \[Gamma]=UnderscriptBox[lim, RowBox[{m, \[Rule], \[Infinity]}], \
LimitsPositioning -> True](UnderoverscriptBox[\[Sum], RowBox[{k, =, \
1}], m, LimitsPositioning -> True] ( 1 ) / ( k ) - logm). It appears \
in many integrals, and asymptotic formulas. It is sometimes known as \
the Euler-Mascheroni constant, and denoted C. Catalan's constant \
Catalan is given by the sum UnderoverscriptBox[\[Sum], RowBox[{k, =, \
0}], \[Infinity], LimitsPositioning -> True](-1)^k(2k+1)^-2. It often \
appears in asymptotic estimates of combinatorial functions. It is \
variously denoted by C, K, or G. " :> "Mathematical constants. \
Euler's constant EulerGamma is given by the limit \[Gamma]=lim_(m\
->\[Infinity])(\[Sum]_(k=1)^\[Infinity]((1)/(k)-log(m))). It appears \
in many integrals, and asymptotic formulas. It is sometimes known as \
the Euler-Mascheroni constant, and denoted C. Catalan's constant \
Catalan is given by the sum \
\[Sum]_(k=0)^\[Infinity]((-1)^k(2k+1)^-2). It often appears in \
asymptotic estimates of combinatorial functions. It is variously \
denoted by C, K, or G. ", "Numerical integration functions. This finds a numerical \
approximation to the integral \[Integral]_0^\[Infinity]e^-x^3\
\[DifferentialD]x. Here is the numerical value of the double integral \
\[Integral]_RowBox[{-^1}, 1]dx \[Integral]_RowBox[{-^1}, 1]dy \
(x^2+y^2). " :> "Numerical integration functions. This finds a \
numerical approximation to the integral \[Integral]_(0)^\[Infinity] \
e^-x^3 \[DifferentialD]x. Here is the numerical value of the double \
integral \[Integral]_(-1)^1 dx \[Integral]_(-1)^1 dy (x^2+y^2). ", "Numerical sums and products. This gives a numerical approximation to \
UnderoverscriptBox[\[Sum], RowBox[{i, =, 1}], \[Infinity], \
LimitsPositioning -> True] ( 1 ) / ( i^3+i! ) . There is no exact \
result for this sum, so the Wolfram Language leaves it in a symbolic form. " \
:> "Numerical sums and products. This gives a numerical approximation \
to \[Sum]_(i=1)^\[Infinity]((1)/(i^3+i!)). There is no exact result \
for this sum, so the Wolfram Language leaves it in a symbolic form. ", "Simplifying with assumptions. The Wolfram Language does not automatically \
simplify this, since it is only true for some values of x. \
SqrtBox[SuperscriptBox[x, 2]] is equal to x for x\[GreaterEqual]0, \
but not otherwise. " :> "Simplifying with assumptions. The Wolfram Language \
does not automatically simplify this, since it is only true for some \
values of x. Sqrt[x^2] is equal to x for x\[GreaterEqual]0, but not \
otherwise. ", "Numerical sums, products and integrals. Here is a numerical \
approximation to UnderoverscriptBox[\[Sum], RowBox[{i, =, 1}], \
\[Infinity], LimitsPositioning -> True] ( 1 ) / ( i^3 ) . NIntegrate \
can handle singularities in the integration region. " :> "Numerical \
sums, products and integrals. Here is a numerical approximation to \
\[Sum]_(i=1)^\[Infinity]((1)/(i^3)). NIntegrate can handle \
singularities in the integration region. ", "file StyleBox[\\\".\\\", \\\"TI\\\"] ext\", expr]" :> "file.ext\", expr]"}], ""]
];


(******************************************************************************)
(** ExtractSummaryText - Internal **)

ExtractSummaryText[nb_Notebook, "Symbol" | "File" | "Menu Item" | "System Program" | "LibraryLink C Function" | "MathLink C Function" | "Format" | "Front End Object" | "Widget", ___]:=
Module[{n, args, forms, maxArgs=2},
  n = Cases[nb, Cell[c_, "Usage"|"FormatUsage"|"ObsolescenceNote", ___], Infinity];
  If[n =!= {}, 
    (
    n = First@Flatten@n;
    n = n /. TextData[{a___, b_String, c___}] :> TextData[{a, StringReplace[b, "\[LineSeparator]" -> " "], c}];
    n = n /. ButtonBox[c_, ___] :> c;
    n = n /. StyleBox[a_, b_] :> a;
    n = n /. TemplateBox[{a_, b_}, ___] :> a;
    If[!FreeQ[n, "ObsolescenceNote"], 
      n = DeleteCases[n, Cell[BoxData[GraphicsBox[___]],___], Infinity]];
    n = DeleteCases[n, 
      Cell[c___] /; (MemberQ[{c}, TooltipBox, Infinity, Heads->True] && MemberQ[{c}, "ModInfo", Infinity]), Infinity];
    n = DeleteCases[n, Cell[_String, "ModInfo"], Infinity];
    n = n /. SuperscriptBox[a_, b_] :> 
      If[StringQ[a] && StringQ[b], 
        StringJoin[ StringReplace[a, RegularExpression["\\s*+"]->""], "^", StringReplace[b, RegularExpression["^\\s+"]->""] ], 
        RowBox@{a,b}
      ];
    n = n /. SubscriptBox[a_, b_] :> 
      If[StringQ[a] && StringQ[b], 
        StringJoin[ StringReplace[a, RegularExpression["\\s+$"]->""], (*"_"*)"", StringReplace[b, RegularExpression["^\\s+"]->""] ], 
        RowBox@{a,b}
      ];
          n = n /. UnderoverscriptBox[a_, b_, c_, ___] :> 
      If[StringQ[a] && StringQ[b] && StringQ[c], 
        StringJoin[ StringReplace[a, RegularExpression["\\s+$"]->""], (*"_"*)"", StringReplace[b, RegularExpression["^\\s+"]->""],"^", StringReplace[c, RegularExpression["^\\s+"]->""]], 
        RowBox@{a,b,c}
      ];
    (* take the first element *)
   If[ Cases[n, _GridBox, 3] =!= {},
       (* extact and flatten GridBox data *)
      n = Cases[n, GridBox[d_, ___] :> d, 2];
      n = Cell[TextData[Flatten@{n}]];
      (* attempt to fix usages running together without a space between them *)
      n = ReplaceAll[n, ""->" "];
    ];
   forms = CreateInputForm @ n;
   If[forms == $Failed, Message[CreateInputForm::err, n]; Return[""]];
   
   forms = StringReplace[forms, {
     RegularExpression["\\^\\s+"]->"^", RegularExpression["\\_\\s+"]->"_",
     "\\[Ellipsis]" -> "...", "\\[Cross]" -> "*"
     }]
    ),
    forms = "" ];
  forms
];

ExtractSummaryText[nb_Notebook, "Solutions Pages", ___]:=
Module[{abstract},
  abstract = Cases[nb, Cell[c_, "Categorization",CellLabel->"Description", ___]:> c, Infinity];
  str = Transmogrify`ConvertToString[abstract];
  str
];

ExtractSummaryText[nb_Notebook, "Tutorial" | "Overview" | "Upgrade Information" | "Notes" | "Note", ___]:=
Module[{abstract},
  abstract = Cases[nb, Cell[c_, "UpgradeSummary" | "DataSourceSummary" | "Text"|"Caption"|"MathCaption"|"TOCChapter"|"Reference", ___]:> c, Infinity];
  abstract = If[Length@abstract > 3, Take[abstract,3], abstract];
  abstract = Riffle[abstract, " "];
  str = Transmogrify`ConvertToString[abstract];
  str
];

ExtractSummaryText[nb_Notebook, "Guide" | "Alphabetical Listing", ___]:=
Module[{abstract},
  abstract = Cases[nb, Cell[c_, "GuideAbstract"|"GuideTOCAbstract", ___]:> c, Infinity];
  If[(Length@abstract > 0) && FreeQ[abstract, "XXXX", Infinity], 
    abstract = First@abstract, 
    (
      abstract = Cases[nb, Cell[c_, "GuideText" | "GuideTOCLink" | "GuideFunctionsSubsection" | "InlineGuideFunctionListing", ___]:> c, Infinity];
      abstract = abstract /. StyleBox[c_,"InlineSeparator",___] :> "";
      abstract = abstract /." \[EmptyVerySmallSquare] " :> ", ";
      abstract = Riffle[abstract, ", "];
    )
  ];
  str = Transmogrify`ConvertToString[abstract];
  str
];

ExtractSummaryText[nb_Notebook, "Character Name", ___]:=
Module[{abstract},
  abstract = Cases[nb, Cell[c_, "Notes", ___]:> c, Infinity];
  abstract = Riffle[abstract, " "];
  str = StringJoin @ Transmogrify`ConvertToString @ abstract;
  str
];

ExtractSummaryText[nb_Notebook, "HowTo", ___]:=
Module[{abstract},
  abstract = Cases[nb, Cell[c_, "HowToAbstract", ___]:> c, Infinity];
  If[Quiet[!StringQ[abstract[[1]]]],
    abstract = Cases[abstract, TextData[{c___}, ___] :> c, Infinity];
    abstract = Riffle[abstract, " "];
  ];
  str = StringJoin @ Transmogrify`ConvertToString @ abstract;
  str
];

ExtractSummaryText[nb_Notebook, "Service", ___]:=
Module[{abstract},
  abstract = Cases[nb, Cell[c_, "ServiceAbstract", ___]:> c, Infinity];
  If[(Length@abstract > 0) && FreeQ[abstract, "XXXX", Infinity], 
    abstract = First@abstract, 
    (
      abstract = Cases[nb, Cell[c_, "GuideText" | "GuideTOCLink" | "GuideFunctionsSubsection" | "InlineGuideFunctionListing", ___]:> c, Infinity];
      abstract = abstract /. StyleBox[c_,"InlineSeparator",___] :> "";
      abstract = abstract /." \[EmptyVerySmallSquare] " :> ", ";
      abstract = Riffle[abstract, ", "];
    )
  ];
  str = Transmogrify`ConvertToString[abstract];
  str
];

ExtractSummaryText[nb_Notebook, ___]:= 
Module[{abstract}, 
  abstract = Cases[nb, Cell[c_, "ObjectName" | "ObjectNameSmall" | "ObjectNameAlt" | "Usage" | "GuideAbstract" | "Title" | "Section" | "Text", ___]:> c, Infinity];
  str = StringJoin @ Transmogrify`ConvertToString @ abstract;
  str
];

(******************************************************************************)
(** GetNotebookDescription **)
GetNotebookDescription[nb_Notebook]:=Module[{dscrptn}, 
  dscrptn=Cases[nb,
    Cell[c_String /; !StringMatchQ[c, RegularExpression["X*"]], ___, "Description", ___]
      :> c,
    Infinity]; If[Length[dscrptn]===0, "", dscrptn[[1]]]];
GetNotebookDescription[s___] := (Message[GetNotebookDescription::args, s]; $Failed);
GetNotebookDescription::args = "Expected Notebook but found: `1`";

(******************************************************************************)
(** GetNotebookKeywords **)
GetNotebookKeywords[nb_Notebook]:=
  Cases[nb,
    Cell[c_String /; !StringMatchQ[c, RegularExpression["X*"]], ___, "Keywords", ___]
      :> StringReplace[c, "," -> ""],
    Infinity];
GetNotebookKeywords[s___] := (Message[GetNotebookKeywords::args, s]; $Failed);
GetNotebookKeywords::args = "Expected Notebook but found: `1`";


(******************************************************************************)
(** GetNotebookSynonyms **)
GetNotebookSynonyms[nb_Notebook]:=
  Cases[nb,
    Cell[c_String /; !StringMatchQ[c, RegularExpression["X*"]], ___, "Synonyms", ___]
      :> StringReplace[c, "," -> ""],
    Infinity];
GetNotebookSynonyms[s___]:= (Message[GetNotebookSynonyms::args, s]; $Failed);
GetNotebookSynonyms::args = "Expected Notebook but found: `1`";


(******************************************************************************)
(** GetDemonstrationKeywordList **)
GetDemonstrationKeywordList::failed = "Failed to import demonstration keyword list";
GetDemonstrationKeywordList[___] := (Message[GetDemonstrationKeywordList::failed]; $Failed);
GetDemonstrationKeywordList[] :=
Module[{lst, file},
  file = $TemporaryPrefix <> "demo-keyword-data.m";
  Export[file, Import["http://demonstrations.internal.wolfram.com/reports/keywords.jsp", "Byte"], "Byte"];
  lst = Import[file, "Package"];
  DeleteFile[file];
  If[lst =!= $Failed && Head[lst] =!= Import,
  	lst = lst /. Rule[a_, b_] :> {a, b};
  	DocumentationBuild`Info`$DemonstrationKeywordTitles = 
  		Union @ Flatten[{
  			{"http://demonstrations.wolfram.com/" -> {"Title" -> "Wolfram Demonstrations Project"}},
  			{Rule[#[[3]],{ Rule["Title", #[[2]] ]} ]} &/@ lst}, 1];
  	(* switch 2nd and 3rd columns for uniformity *)
  	lst = Union[{#[[1]], #[[3]]} & /@ lst];
    Return@lst,
    (Message[GetDemonstrationKeywordList::failed]; $Failed)
  ]
];

(******************************************************************************)
(** GetDemonstrationSymbolList **)
GetDemonstrationSymbolList::failed = "Failed to import demonstration list";
GetDemonstrationSymbolList[___] := (Message[GetDemonstrationSymbolList::failed]; $Failed);
GetDemonstrationSymbolList[] :=
Module[{lst},
  lst = Import["http://demonstrations.internal.wolfram.com/reports/symbols.jsp", "TEXT"];
  If[lst =!= $Failed && Head[lst] =!= Import,
    StringSplit[lst, "\n"],
    (Message[GetDemonstrationSymbolList::failed]; $Failed)
  ]
];

demonbs = {
  "CellularAutomaton",
  "ContourPlot3D",
  "Derivative",
  "DigitCount",
  "D",
  "Expand",
  "FactorInteger",
  "Factor",
  "FindFit",
  "FindRoot",
  "Fit",
  "Fourier",
  "FromCharacterCode",
  "IntegerPartitions",
  "NDSolve",
  "PadeApproximant",
  "ParametricPlot3D",
  "ParametricPlot",
  "Plot",
  "Plot3D",
  "PolarPlot",
  "PolyhedronData",
  "RiemannSiegelZ",
  "Series",
  "Solve",
  "TreePlot"
};

(* List of M- functions which appear in NKSOnline index from Oyvind 11/29/06 *)
nksIndexedFunctions = 
{"AiryAi", "And", "Apply", "ArcCos", "ArcSin", "ArcTan", "ArithmeticGeometricMean", "AspectRatio", "BernoulliB", "BesselI", "BesselJ", "BesselY", "Binomial", "BitAnd", "BitOr", "BitXor", "Block", "CarmichaelLambda", "Catalan", "Catch", "Ceiling", "CellularAutomaton", "ChebyshevT", "CoefficientList", "Collect", "Compile", "Complement", "ContinuedFraction", "Cos", "Cosh", "Cot", "Coth", "Csc", "Cuboid", "Cyclotomic", "DedekindEta", "Delete", "DeleteCases", "Depth", "Derivative", "Det", "DigitCount", "Distribute", "Divide", "Divisors", "DivisorSigma", "Do", "Dot", "Drop", "Eigenvalues", "Element", "EllipticK", "EllipticTheta", "Equal", "Erf", "EulerGamma", "EulerPhi", "Exists", "Exp", "Expand", "ExpIntegralEi", "Export", "ExtendedGCD", "Factor", "Factorial", "Factorial2", "FactorInteger", "False", "Fibonacci", "FindMinimum", "FindRoot", "Fit", "FixedPoint", "FixedPointList", "Flat", "Flatten", "Floor", "Fold", "FoldList", "For", "ForAll", "Fourier", "FractionalPart", "FresnelC", "FresnelS", "FromDigits", "FullForm", "FullSimplify", "Function", "FunctionExpand", "Gamma", "GCD", "GegenbauerC", "GoldenRatio", "Graphics3D", "GroebnerBasis", "HermiteH", "Hypergeometric1F1", "Hypergeometric2F1", "HypergeometricPFQ", "If", "Implies", "Inner", "InputForm", "Insert", "IntegerDigits", "IntegerExponent", "IntegerQ", "Integrate", "Intersection", "InverseFunction", "JacobiP", "JacobiSN", "JacobiSymbol", "Join", "Khinchin", "KleinInvariantJ", "KroneckerDelta", "LaguerreL", "LatticeReduce", "LCM", "LeafCount", "LegendreP", "Length", "Less", "LinearSolve", "Listable", "ListConvolve", "ListCorrelate", "ListPlay", "Log", "LogIntegral", "Map", "MapAt", "MapIndexed", "MatchQ", "MathieuC", "MatrixPower", "Max", "MeijerG", "MemberQ", "Mod", "ModularLambda", "MoebiusMu", "Multinomial", "MultiplicativeOrder", "Nand", "NDSolve", "Nest", "NestList", "NestWhile", "NestWhileList", "NIntegrate", "Nor", "Not", "NSolve", "NullSpace", "OddQ", "Or", "Orderless", "Outer", "PadLeft", "Partition", "PartitionsP", "Pi", "Play", "Plot3D", "Plus", "PolyGamma", "PolyLog", "PolynomialMod", "PolynomialReduce", "PolynomialRemainder", "Position", "Power", "PowerMod", "Prepend", "PrimePi", "PrimeQ", "Product", "ProductLog", "PseudoInverse", "Quotient", "Random", "Range", "Raster", "RealDigits", "RepeatedNull", "ReplaceAll", "ReplaceList", "ReplacePart", "ReplaceRepeated", "Rest", "Resultant", "RiemannSiegelZ", "Root", "RotateLeft", "Rule", "Sec", "Sech", "SeedRandom", "Select", "Series", "Simplify", "Sin", "Sinh", "SinIntegral", "SixJSymbol", "Slot", "Sort", "SphericalHarmonicY", "Split", "Sqrt", "StandardForm", "StieltjesGamma", "StirlingS1", "StirlingS2", "StringJoin", "StringLength", "StringPosition", "StringReplace", "StringReplacePart", "SubsuperscriptBox", "Sum", "Table", "Take", "Tan", "Tanh", "TensorRank", "Through", "Throw", "TimeConstraint", "Times", "ToExpression", "ToString", "Transpose", "TrigFactor", "True", "TrueQ", "Union", "WeierstrassP", "While", "Xor", "Zeta", "$IterationLimit"};



(******************************************************************************)
(** GetForeignFunctionKeywords **)

ForeignFunctionKeywordsComputed = False;
GetForeignFunctionKeywords[] = {};
GetForeignFunctionKeywords[uri_] :=
  Module[{in, files, data},
    If[ForeignFunctionKeywordsComputed, Return[{}]];
    ForeignFunctionKeywordsComputed = True;
    in = ToFileName[{$DocumentationBuildDirectory, "Internal", "data", "keywords", "ForeignFunctions"}];
    files = FileNames["*.m", in, 2];
    data = {};
    Scan[
      Function[{file},
        data = Join[data,
          ReadList[StringToStream[Import[file, "Text", CharacterEncoding -> "UTF-8"]]]]
      ],
      files];
    data = DeleteCases[data, {"", _}];
    data = data /. {a_, b_} :> {StringReplace[b, "paclet:" -> ""], a};

    Scan[
      Function[{datum}, Module[{keywords},
        keywords = GetForeignFunctionKeywords[First[datum]];
        If[!MatchQ[keywords, {___, Last[datum], ___}], 
          GetForeignFunctionKeywords[First[datum]] = Append[keywords, Last[datum]] ];
      ]],
      data];

    GetForeignFunctionKeywords[uri]
  ];
  
  
(******************************************************************************)
(** MakeForeignFunctionKeyword - Internal **)
MakeForeignFunctionKeyword[] :=
Module[{in, files, data},
	in = ToFileName[{$DocumentationToolsDirectory, "Internal", "data", 
    	"keywords", "ForeignFunctions"}];
	files = FileNames["*.m", in, 2];
	data = {};
	Scan[(data = Join[ data,
		ReadList[
			StringToStream[
				Import[#, "Text", CharacterEncoding -> "UTF-8"]]] ]) &, files];
	(*remove entries without keywords*)
	data = DeleteCases[data, {"", _}];
	data = data /. {a_, b_} :> Rule[StringReplace[b, "paclet:" -> ""], a];
 
	ForeignFunctionKeywords[] := {};
	ForeignFunctionKeywords[_] := {};

	Scan[If[! MatchQ[ForeignFunctionKeywords[First[#]], {___, Last[#], ___}], 
		ForeignFunctionKeywords[First[#]] = 
			Append[ForeignFunctionKeywords[First[#]], Last[#]]] &, data];
];


(******************************************************************************)
(** ExportSourceInformation / ExportOnlineInformation - Internal **)
ExportSourceInformation[dir_, langs_List, pkg_String:""] :=
  Map[ExportSourceInformation[dir, #, pkg]&, langs];

ExportOnlineInformation[dir_, langs_List, pkg_String:""] :=
  Map[ExportOnlineInformation[dir, #, pkg]&, langs];

ExportSourceInformation[dir_, lang_String:"English", pkg_String:""] :=
  Module[{outfile},
    outfile = ToFileName[{$DocumentationToolsDirectory,
                          "Internal", "data"},
        StringJoin["SourceInformation.",
            Switch[lang, "Japanese", "ja", "ChineseSimplified", "zh", "Spanish", "es", _, "en"],
            ".m"]];
    If[ListQ[dir],
      Export[outfile, dir],
      Export[outfile, CreateSourceInformation[dir, lang, pkg]] ];
  ];

ExportOnlineInformation[dir_, lang_String:"English", pkg_String:""] :=
  Module[{outfile},
    outfile = ToFileName[{$DocumentationToolsDirectory,
                          "Internal", "data"},
        StringJoin["OnlineInformation.",
            Switch[lang, "Japanese", "ja", "ChineseSimplified", "zh", "Spanish", "es", _, "en"],
            ".m"]];
    If[ListQ[dir],
      Export[outfile, dir],
      Export[outfile, CreateOnlineInformation[dir, lang, pkg]] ];
  ];


(******************************************************************************)
(** ExportAllOnlineInformation - Internal **)

(****************** This code will be refined later. ********)
  
CreateOnlineInformationForPackages[lang_String: "English"] := 
 Module[{PackagesDirectory = ToFileName[{$InstallationDirectory, "Documentation", lang, "Packages"}]}, 
  Flatten[CreateOnlineInformation[ToFileName[{PackagesDirectory, 
                                              StringReplace[#, DirectoryName[#] -> ""], "Documentation"}], 
                                  lang, StringReplace[#, DirectoryName[#] -> ""]] & /@ 
    Cases[FileNames["*", PackagesDirectory], x_String /; Not@StringMatchQ[x, __ ~~ ("GUIKit" | "Compatibility")]], 1]]
      
CreateOnlineInformationForMathematicaLinks[lang_String: "English"] := 
 Module[{LinksDirectory = ToFileName[{$InstallationDirectory, "SystemFiles", "Links"}]}, 
  Flatten[CreateOnlineInformation[ToFileName[{LinksDirectory, #, "Documentation"}], lang, #] & /@ 
   {"DatabaseLink", "JLink", "NETLink", "WebServices"}, 1]]
     
ExportAllOnlineInformation[lang_String: "English"] := 
  Module[{outfile}, outfile = ToFileName[{$DocumentationToolsDirectory, "Internal", "data"}, 
     StringJoin["OnlineInformation.", Switch[lang, "Japanese", "ja", "ChineseSimplified", "zh", "Spanish", "es", _, "en"], ".m"]];
   Export[outfile, Join[CreateOnlineInformation[ToFileName[{$InstallationDirectory, "Documentation"}], "English", ""], 
                        CreateOnlineInformationForMathematicaLinks["English"], 
                        CreateOnlineInformationForPackages["English"]]]]
      


(******************************************************************************)
(** CreateSourceInformation / CreateOnlineInformation **)

CreateSourceInformation[dir_, lang_String:"English", pkg_String:""] :=
  CreateCommonInformation[dir, lang, pkg, "Source"];

CreateOnlineInformation[dir_, lang_String:"English", pkg_String:""] :=
  CreateCommonInformation[dir, lang, pkg, "Online"];

CreateCommonInformation[indir_, lang_String:"English", pkg_String:"", type_String] :=
Module[{dir, sourceQ, symbols, refs, guides, tutorials, howtos, examples, info, getCommon, putCommon,
        systemDir, demoList, demonbs, title, mkuri, primaryGuideList, uridata = {}, examplePageDir = "",
        GetGuideRelevance, GetGuideWeight, GetWeightedRelatedGuides, OneLineListingQ,
        GetSectionPos, webguidevisits, guidepageexpr, GetGuideRanking, 
        examplePageData = {}, examplepageURLroot = "http://reference.wolfram.com/language/example/"},
  dir = StringReplace[indir, 
          RegularExpression["^(.+)(/|\\\\)"<>lang<>"(/|\\\\)*$"] -> "$1"];
  sourceQ = (type === "Source");
  
  webguidevisits = Get[FileNameJoin[{$DocumentationBuildDirectory, "Internal", "data", "GuidePageWebVisits.m"}]];
  
  OneLineListingQ[guideuri_, refuri_] := 
   Module[{
   	nbdata = guidepageexpr},
    TrueQ[MemberQ[
      Cases[nbdata, 
       Cell[c_, ___, "GuideText" | "InlineGuideFunctionListing", ___] /; 
       MemberQ[c, ButtonBox[Last[StringSplit[refuri, "/"]], ___], 
      Infinity], Infinity], "GuideText", Infinity]]];
      
  GetSectionPos[guideuri_, refuri_] := 
   Module[{nbdata = guidepageexpr, group, count, pos, refgroup},
   group = 
    First@Cases[nbdata, 
      Cell[CellGroupData[c_, ___]] /; 
       MemberQ[c, 
        Cell[cont_, ___, 
          "GuideText" | "InlineGuideFunctionListing", ___] /; 
         MemberQ[cont, ButtonBox[Last[StringSplit[refuri, "/"]], ___], 
          Infinity], Infinity], Infinity];
   refgroup = 
    Cases[group, 
     ButtonBox[_String, ___, 
      BaseStyle -> bs_ /; MemberQ[{bs}, "Link", Infinity], ___], 
     Infinity]; count = Max[Length[refgroup], 1];
   pos = Position[refgroup, Last[StringSplit[refuri, "/"]]];
   If[pos =!= {}, pos = pos[[1, 1]], pos = 1];
   (* larger is stronger *)
   N[(count - pos + 1)/count]
  ];
  
  GetGuideRelevance[uri_, guideuri_List] := 
   Module[{guideuris = guideuri}, 
    Map[(guidepageexpr = Get[FileNameJoin[{dir, lang, systemDir, "Guides", 
    	 Last[StringSplit[#, "/"]] <> ".nb"}]]; 
      {#, {"OneLine" -> OneLineListingQ[#, uri], 
      "PositionInSection" -> GetSectionPos[#, uri]}}) &, guideuris]];
      
  GetGuideRanking[guideuri_String] := 
   Module[{name = Last[StringSplit[guideuri, "/"]], pos},
    pos = Position[webguidevisits, name];
    If[pos =!= {}, pos = pos[[1, 1]]; 
     N[(Length[webguidevisits] - pos)/Length[webguidevisits]], 0]];
     
  GetGuideWeight[{uri_String, lis_List}] := 
  	{uri, .4 ("PositionInSection" /. lis) + .25 GetGuideRanking[uri] + 
     If[TrueQ[("OneLine" /. lis)], .25, 0]};
     
  GetWeightedRelatedGuides[refuri_, guideuris_List] := Module[{relev = GetGuideRelevance[refuri, Last/@guideuris], lis},
  	(lis = GetGuideWeight /@ relev);
  	First /@ Reverse[SortBy[lis, Last]]
  ];
  
  Print["Code A"];
  If[sourceQ && pkg === "",
    demoList = GetDemonstrationSymbolList[];
    If[demoList === $Failed, demoList = demonbs],
    demoList = demonbs
  ];
  systemDir = If[StringLength@pkg < 1, "System", ""];
  Print["Code B"];
  (* Populate uridata for use in tutorial navigation only if in System directory for now *)
  (* If in System directory, get Example Page data for backlinking to guides *)
  If[StringMatchQ[systemDir, "System"], 
  	examplePageDir = FileNameJoin[{dir, lang, systemDir, "ExamplePages"}];
  	Print["Num files: ", Length[FileNames["*.nb",examplePageDir]]];
  	examplePageData = DeleteCases[DocumentationBuild`Navigation`GetExamplePageData /@ 
  	  FileNames["*.nb", examplePageDir], $Failed];
  	Print["Length ExamplePage data: ", Length[examplePageData]];
    uridata = Select[Flatten[
    	Quiet[DocumentationBuild`SystemResourcesFromDocs`CreateTutorialBrowserData[{
    		ToFileName[{dir, lang, systemDir, "Tutorials"}]}]]],
    	StringMatchQ[ToString[#], "paclet:*tutorial/*"] && 
    	StringFreeQ[ToString[#], "#"~~DigitCharacter..]&]];
   Print["Code C"];
  mkuri[args__] := StringJoin[
    "paclet:",
    If[StringLength@pkg > 0, pkg<>"/", ""],
    Riffle[{args}, "/"]];
  (* Find "PrimaryObject" links in guide pages. This determines the link's primary guide page *)
  primaryGuideList = MakePrimaryGuideList[ToFileName[{dir, lang, systemDir, "Guides"}], pkg];
  Print["Code D"];
  (* Common *)
  Clear[linktypes, links, info];
  linktypes[_] := {};
  links[_, _] := {};
  info[_, "Title"] := "";
  info[_, "Categorization"] := "";
  info[_, "HistoryData"] := {" ", " ", " ", " "};
  info[_, "Flag"] := "";
  info[_, "Developers"] := {};
  info[_, "PrimaryGuide"] := "";
  info[_, "LinkTrails"] := {};
  info[_, "URI"] := "";
  info[_, "ForwardBackNavData"] := {};
  Print["Code E"];
  putCommon[uri_] := Join[
    If[sourceQ,
      Map[(# -> info[uri, #])&, {"Title", "Categorization", "HistoryData", "Flag", "Developers", "PrimaryGuide", "LinkTrails", "URI", "ForwardBackNavData"}],
      {}],
    {"Links" -> Map[
      (# -> If[MemberQ[{"Guides"}, #],
        (* These link types get sorted and uniqued *)
        Module[{originalguides = First /@ Split[Sort[links[uri, #],
                If[#1[[1]] === #2[[1]],
                OrderedQ[{#1[[2]], #2[[2]]}], 
                OrderedQ[{#1[[1]], #2[[1]]}] ]&]], newuris}, 
          newuris = GetWeightedRelatedGuides[uri, originalguides];
          SortBy[originalguides, Position[newuris, #[[2]]] &]
        ],
        (* But the rest don't *)
        links[uri, #]])&,
      Union[linktypes[uri]]
    ]}
  ];
Print["Code F"];
  getCommon[uri_, nb_] := Module[{cat,pglist},
  	Print["Running inside getCommon for the following uri: "<>uri];
    If[sourceQ,
      (* Source notebook *)
      Print["Getting Categorization"];
      cat = Cases[nb, Cell[s_, "Categorization", ___] :> s, Infinity, 1];
      If[Length[cat] =!= 0,
        info[uri, "Categorization"] = First[cat]];

      Print["Getting Title"];
      info[uri, "Title"] = title = GetNotebookTitle[nb];
      Print["Getting Developers"];
      info[uri, "Developers"] = GetFunctionDevelopers[nb];
      Print["Getting HistoryData"];
      info[uri, "HistoryData"] = {"New", "Modified", "Obsolete", "Excised"} /. GetNotebookHistoryData[nb] /. {"New"->"", "Modified"->"", "Obsolete"->"", "Excised"->""};
      Print["Getting Primary Guide"];
      info[uri, "PrimaryGuide"] = DocumentationBuild`Utils`GetPrimaryGuide[uri, lang, primaryGuideList];
      Print["Getting Primary Guide List"];
      pglist = Join[primaryGuideList, {uri -> {"PrimaryGuide" -> info[uri, "PrimaryGuide"]}}];
      Print["Getting LinkTrails"];
      info[uri, "LinkTrails"] = DocumentationBuild`Info`Private`GetLinkTrails[uri, pglist ];
      Print["Getting Notebook URI"];
      info[uri, "URI"] = GetNotebookURI[nb];
      
      (*If[MatchQ[nb[[1, 1]], Cell[_, s_/;StringMatchQ[s, "*Flag"], ___]],
        info[uri, "Flag"] = nb[[1, 1, 2]] ];*)
      Print["Getting Notebook Flag"];
      info[uri, "Flag"] = GetNotebookFlag[nb];

    ,
      (* Online notebooks *)
      Module[{bboxFilter, sections},
        bboxFilter[str_String | {URL[str_String], None}] := str;
        sections = Union@Cases[nb,
          Cell[__, sty_String /; StringMatchQ[sty, (___~~"Section") | "Subsection" | "Title"], ___] :> sty,
          Infinity];
        Scan[
          Function[{linktype},
            linktypes[uri] = Append[linktypes[uri], linktype];
            links[uri, linktype] = (Union[Cases[
              Cases[nb,
                Cell[CellGroupData[{Cell[__, linktype, __], ___}, ___], ___],
                Infinity],
              ButtonBox[bt_, ___, ButtonData->bd_, ___] :>
                If[StringQ[bd] || MatchQ[bd, {URL[_String], None}],
                  {bt, bboxFilter@bd}, Unevaluated[Sequence[]] ],
              Infinity]]
              //. {a___, {b_String, u_}, {c_String, u_}, d___}
                :> {a, {b<>c, u}, d})
          ],
          sections];
      ];
    ]; (* end If[sourceQ, ...] *)
  ];

  (* Symbols *)
  info[_, "Demonstrations"] := False;
  info[_, "HasOptions"] := False;

  symbols = Reap[Scan[
    Function[{file}, Module[{base, uri, nb, hasopts},
      Print["Kernel Memory: "<>ToString@MemoryInUse[]];
      Print["Loading File"];
      nb = DocumentationBuild`Utils`GetQuietly[file];
      Print["Getting BaseName"];
      base = FileBaseName[file];
      Print["BaseName: "<>ToString[base]];
      uri = Sow[mkuri["ref", base]];
      Print["uri: "<>ToString[uri]];
      If[MemberQ[{"GeodesicArrow", "GeodesicLine", "GeoDisk"},ToString[base]],
      hasopts = True,
      hasopts = DocumentationBuild`Utils`HasOptionsQ[nb, base]
      ];
      Print["Does this have Options: "<>ToString[hasopts]];
      Print["Running getCommon"];
      getCommon[uri, nb];

      If[sourceQ,
        If[MemberQ[demoList, base], info[uri, "Demonstrations"] = True]];
      If[sourceQ,
        If[hasopts, info[uri, "HasOptions"] = True]];
    ]],
    FileNames["*.nb", ToFileName[{dir, lang, systemDir, "ReferencePages", "Symbols"}]]
  ]];

  refs = Reap[Scan[
    Function[{pair}, Module[{file, nb, base, uri},
      file = Last[pair];
      Print["Inside AA"];
      nb = DocumentationBuild`Utils`GetQuietly[file];
      Print["Inside BB"];
      base = FileBaseName[file];
      uri = Sow[mkuri["ref", First[pair], base]];
      Print["Inside CC"];
      getCommon[uri, nb];
    ]],
    Join[
      {"applescript", #}& /@
        FileNames["*.nb", ToFileName[{dir, lang, systemDir, "ReferencePages", "AppleScript"}]],
      {"c", #}& /@
        FileNames["*.nb", ToFileName[{dir, lang, systemDir, "ReferencePages", "C"}]],
      {"character", #}& /@
        FileNames["*.nb", ToFileName[{dir, lang, systemDir, "ReferencePages", "Characters"}]],
      {"device", #}& /@
        FileNames["*.nb", ToFileName[{dir, lang, systemDir, "ReferencePages", "Devices"}]],
      {"embeddingformat", #}& /@
        FileNames["*.nb", ToFileName[{dir, lang, systemDir, "ReferencePages", "EmbeddingFormats"}]],
      {"file", #}& /@
        FileNames["*.nb", ToFileName[{dir, lang, systemDir, "ReferencePages", "Files"}]],
      {"format", #}& /@
        FileNames["*.nb", ToFileName[{dir, lang, systemDir, "ReferencePages", "Formats"}]],
      {"frontendobject", #}& /@
        FileNames["*.nb", ToFileName[{dir, lang, systemDir, "ReferencePages", "FrontEndObjects"}]],
      {"indicator", #}& /@
        FileNames["*.nb", ToFileName[{dir, lang, systemDir, "ReferencePages", "Indicators"}]],
      {"interpreter", #}& /@
        FileNames["*.nb", ToFileName[{dir, lang, systemDir, "ReferencePages", "Interpreters"}]],
      {"menuitem", #}& /@
        FileNames["*.nb", ToFileName[{dir, lang, systemDir, "ReferencePages", "MenuItems"}]],
      {"message", #}& /@
        FileNames["*.nb", ToFileName[{dir, lang, systemDir, "ReferencePages", "Messages"}]],
      {"method", #}& /@
        FileNames["*.nb", ToFileName[{dir, lang, systemDir, "ReferencePages", "Methods"}]],
      {"program", #}& /@
        FileNames["*.nb", ToFileName[{dir, lang, systemDir, "ReferencePages", "Programs"}]],
      {"service", #}& /@
        FileNames["*.nb", ToFileName[{dir, lang, systemDir, "ReferencePages", "Services"}]]
  ]]];

  (* guides *)
  
  info[_, "FeaturedExampleLinks"] := {};
  info[_, "WebResourcesLinks"] := {};

  guides = Reap[Scan[
    Function[{file}, Module[{nb, base, uri, exclude},
      nb = DocumentationBuild`Utils`GetQuietly[file];
      base = FileBaseName[file];
      uri = Sow[mkuri["guide", base]];
      getCommon[uri, nb];

      If[sourceQ && FreeQ[{"paclet:guide/AlphabeticalListing","paclet:guide/NewIn60AlphabeticalListing"}, uri],
        exclude = Reap[
          Scan[
            Function[{cell}, Scan[
              Function[{bbox}, Module[{buri},
                buri = Which[
                  MatchQ[bbox, ButtonBox[___, ButtonData -> _, ___]],
                    ButtonData /. (List@@Cases[bbox, _~(Rule|RuleDelayed)~_]),
                  StringQ[First[bbox]],
                    mkuri["ref", First[bbox]],
                  True, None
                ];
                If[buri =!= None, Sow[buri]];
              ]],
              Cases[cell,
                ButtonBox[___, _->("Link"|{___,"Link",___}), ___],
                Infinity]
            ]],
            Cases[nb, Cell[___, "ExcludedObject", ___],
              Infinity]
          ];
          Scan[
            Function[{cell}, Scan[
              Function[{bbox}, Module[{buri},
                buri = Which[
                  MatchQ[bbox, ButtonBox[___, ButtonData -> _, ___]],
                    ButtonData /. (List@@Cases[bbox, _~(Rule|RuleDelayed)~_]),
                  StringQ[First[bbox]],
                    mkuri["ref", First[bbox]],
                  True, None
                ];
                If[buri =!= None, Sow[buri]];
              ]],
              Cases[cell,
                ButtonBox[___, _->{___,"Link",___,"ExcludedObject",___}, ___],
                Infinity]
            ]],
            Cases[nb, Cell[_,
                "InlineFormula"|"InlineGuideFunction"|"InlineGuideFunctionListing"|"AlphabeticalListing"|"InlineCode",
                ___], Infinity]
          ];
        ];
        If[Length[exclude[[2]]] > 0, exclude = exclude[[2, 1]], exclude = {}];

        Scan[
          Function[{cell},
            Scan[
              Function[{bbox}, Module[{buri},
                buri = Which[
                  MatchQ[bbox, ButtonBox[___, ButtonData -> _, ___]],
                    ButtonData /. (List@@Cases[bbox, _~(Rule|RuleDelayed)~_]),
                  StringQ[First[bbox]],
                    mkuri["ref", First[bbox]],
                  True, None
                ];
                If[buri =!= None && FreeQ[exclude, buri],
                  linktypes[buri] = Append[linktypes[buri], "Guides"];
                  links[buri, "Guides"] = Append[links[buri, "Guides"], {info[uri, "Title"], uri}]];
              ]],
              Cases[cell, ButtonBox[___, _->("Link"|{___,"Link",___}), ___], Infinity]
          ]],
          Cases[nb, Cell[_,
              "InlineFormula"|"InlineGuideFunction"|"InlineGuideFunctionListing"|"AlphabeticalListing"|"InlineCode",
              ___], Infinity]
        ];
        
        info[uri, "FeaturedExampleLinks"] = 
           DocumentationBuild`Navigation`GetFeaturedExampleLinksToGuide[uri, examplePageData];
        info[uri, "WebResourcesLinks"] = 
           GetWebResourcesLinks[nb];
        Print[uri," Example Links: ", 
        	DocumentationBuild`Navigation`GetFeaturedExampleLinksToGuide[uri, examplePageData]];
        
      ]; (* end If[sourceQ, ...] *)
      
    ]],
    FileNames["*.nb", ToFileName[{dir, lang, systemDir, "Guides"}]]
  ]];

  (* tutorials *)

  info[_, "TutorialSections"] := {};
  
  tutorials = Reap[Scan[
    Function[{file}, Module[{base, uri, nb, cat},
      nb = DocumentationBuild`Utils`GetQuietly[file];
      base = FileBaseName[file];
      uri = Sow[mkuri["tutorial", base]];
      getCommon[uri, nb];

      If[sourceQ,
        If[info[uri, "Categorization"] === "Overview", Module[{getSects},
          getSects[expr_, types_] := Module[{poss},
            poss = Position[expr,
              Cell[_, First@types, ___] |
              Cell[CellGroupData[{Cell[_, First@types, ___], ___}, ___], ___],
              Infinity];
            poss = Last /@ Split[poss,
              (Length[#1] > Length[#2] && Take[#1, Length[#2]] === #2)&];
            Map[
              Function[{pos}, Module[{cell, bbox, buri},
                cell = First@Extract[expr, {pos}];
                If[MatchQ[cell, Cell[CellGroupData[___], ___]],
                  { cell[[1, 1, 1, 1]], getSects[Rest[cell[[1, 1]]], Rest[types]] },
                  ( bbox = First@Cases[cell, _ButtonBox, Infinity, 1];
                    buri = ButtonData /. (List@@Cases[bbox, _~(Rule|RuleDelayed)~_]);
                    linktypes[buri] = Append[linktypes[buri], "Overview"];
                    links[buri, "Overview"] = {info[uri, "Title"], uri};
                    {First[bbox], buri})
              ]]],
              poss
          ]];
          linktypes[uri] = Append[linktypes[uri], "Chapter"];
          links[uri, "Chapters"] = getSects[nb,
              {"TOCChapter", "TOCSection", "TOCSubsection", "TOCSubsubsection"}];
        ]];
        
        info[uri, "ForwardBackNavData"] = 
          DocumentationBuild`Navigation`GetNavDestinations[
          	uri, uridata, lang];
        info[uri, "TutorialSections"] = DocumentationBuild`Navigation`GetTutorialAnchors[
        	uri, nb, lang];
      ]; (* end If[sourceQ, ...] *)
    ]],
    FileNames["*.nb", ToFileName[{dir, lang, systemDir, "Tutorials"}]]
  ]];
  
  howtos = Reap[Scan[
    Function[{file}, Module[{base, uri, nb, cat},
      nb = DocumentationBuild`Utils`GetQuietly[file];
      base = FileBaseName[file];
      uri = Sow[mkuri["howto", base]];
      getCommon[uri, nb];

      If[sourceQ,
        If[info[uri, "Categorization"] === "HowTo", Module[{getSects},
          getSects[expr_, types_] := Module[{poss},
            poss = Position[expr,
              Cell[_, First@types, ___] |
              Cell[CellGroupData[{Cell[_, First@types, ___], ___}, ___], ___],
              Infinity];
            poss = Last /@ Split[poss,
              (Length[#1] > Length[#2] && Take[#1, Length[#2]] === #2)&];
            Map[
              Function[{pos}, Module[{cell, bbox, buri},
                cell = First@Extract[expr, {pos}];
                If[MatchQ[cell, Cell[CellGroupData[___], ___]],
                  { cell[[1, 1, 1, 1]], getSects[Rest[cell[[1, 1]]], Rest[types]] },
                  ( bbox = First@Cases[cell, _ButtonBox, Infinity, 1];
                    buri = ButtonData /. (List@@Cases[bbox, _~(Rule|RuleDelayed)~_]);
                    linktypes[buri] = Append[linktypes[buri], "HowTo"];
                    links[buri, "HowTo"] = {info[uri, "Title"], uri};
                    {First[bbox], buri})
              ]]],
              poss
          ]];
          linktypes[uri] = Append[linktypes[uri], "Chapter"];
          links[uri, "Chapters"] = getSects[nb,
              {"TOCChapter", "TOCSection", "TOCSubsection", "TOCSubsubsection"}];
        ]];
      ]; (* end If[sourceQ, ...] *)
    ]],
    FileNames["*.nb", ToFileName[{dir, lang, systemDir, "HowTos"}]]
  ]];
  
  (* featured examples --- only for System directory *)
  If[StringMatchQ[systemDir, "System"] && sourceQ,
  	examples = Reap[Scan[
  		Function[{file}, Module[{base, uri, nb},
  			nb = DocumentationBuild`Utils`GetQuietly[file];
            base = FileBaseName[file];
     		uri = Sow[mkuri["example", base]];
      		info[uri, "Thumbnail"] = Compress[DocumentationBuild`Navigation`GetExamplePageThumbnail[nb]];
      		info[uri, "Title"] = GetNotebookTitle[nb, "Featured Example"];
      		info[uri, "WebURL"] = StringJoin[examplepageURLroot, base, ".html"];
      		info[uri, "URI"] = GetNotebookURI[nb];
      		info[uri, "Flag"] = GetNotebookFlag[nb];
  		]],
  		FileNames["*.nb", ToFileName[{dir, lang, systemDir, "ExamplePages"}]]
    ]],
    examples = {Null, {}}];
  
  Join[
    If[Length[symbols[[2]]] > 0, Map[
      Function[{uri}, uri -> Join[
        putCommon[uri],
        If[sourceQ,
          {"Demonstrations" -> (info[uri, "Demonstrations"] === True)},
          {}],
        If[sourceQ,
          {"HasOptions" -> (info[uri, "HasOptions"] === True)},
          {}]
      ]],
      symbols[[2, 1]] ],
      {}],
    If[Length[refs[[2]]] > 0, Map[
      Function[{uri}, uri -> putCommon[uri]],
      refs[[2, 1]] ],
      {}],
    If[Length[guides[[2]]] > 0, Map[
      Function[{uri}, uri -> Join[
      	putCommon[uri],
      	If[sourceQ,
      		{"FeaturedExampleLinks" -> info[uri, "FeaturedExampleLinks"]},
      		{}],
        If[sourceQ,
      		{"WebResourcesLinks" -> info[uri, "WebResourcesLinks"]},
      		{}]
      ]],
      guides[[2, 1]] ],
      {}],
    If[Length[tutorials[[2]]] > 0, Map[
      Function[{uri}, uri -> Join[
      	putCommon[uri],
      	If[sourceQ,
      		{"TutorialSections" -> (info[uri, "TutorialSections"])},
      		{}]
      ]],
      tutorials[[2, 1]] ],
      {}],
    If[Length[howtos[[2]]] > 0, Map[
      Function[{uri}, uri -> putCommon[uri]],
      howtos[[2, 1]] ],
      {}],
      
    If[Length[examples[[2]]] > 0, Map[
      Function[{uri}, 
      	uri -> 
      	  Map[(# -> info[uri, #] &), {"Title", "Thumbnail", "WebURL", "URI", "Flag"}]],
      examples[[2, 1]] ],
      {}]
]];


(******************************************************************************)
(** MakePrimaryGuideList **)

(*
	MakePrimaryGuideList creates rule list of function->guide pages
	based on "PrimaryObject" links in guide pages 

	in: guide page dir
	out: {.., paclet:ref/Sin->{PrimaryGuide->paclet:guide/TrigonometricFunctions},...}
*)
MakePrimaryGuideList[guideDir_String, linkBase_String:""] := 
Module[{files, expr, pkg=linkBase,files2,pos,pos2,excisedFiles,newfiles,test},
  pkg = If[ MemberQ[{"Mathematica", ""}, pkg], "", pkg];
  files = FileNames["*.nb", guideDir];
  files2 = Map[
   Cases[Import[#, "NB"], Cell[_String, "ExcisedFlag", _], Infinity] &, 
   files];
   pos = Position[files2, Cell[_, "ExcisedFlag", _] ];
   pos2 = Map[{First[#]} &, pos];
   excisedFiles = Extract[files, pos2];
   newfiles = Complement[files, excisedFiles];
   
Flatten@{{expr = DocumentationBuild`Utils`GetQuietly[#]; 
       Function[{guide, functions}, 
         Flatten@{Rule[#, {Rule["PrimaryGuide",guide]}] & /@ functions}][
        "paclet:"<>DocumentationBuild`Common`FilepathToURI[#, pkg], 
        Union@Flatten@
          Cases[expr, 
           bb : ButtonBox[func_String, ___, BaseStyle -> {___, "PrimaryObject", ___}, ___] :> pacletFromButtonBox@bb, 
           Infinity]]} & /@ newfiles}];



(******************************************************************************)
(** pacletFromButtonBox - Internal **)
pacletFromButtonBox[ButtonBox[func_String, opts___?OptionQ]] :=
 Module[{bd},
  bd = ButtonData /. {opts};
  If[Head@bd =!= String, "paclet:ref/" <> func, bd]
  ];
pacletFromButtonBox[___] := {}


(******************************************************************************)
(** GetLinkTrails - Internal **)

GetLinkTrails[u_String /; !StringMatchQ[u, "paclet:*"], primaryGuides_]:= 
	GetLinkTrails["paclet:"<>u, primaryGuides];
GetLinkTrails[u_String /; StringMatchQ[u, "paclet:*"], primaryGuides_]:= 
Module[{trail, uri=u, info, primaryGuide, pg, lst, res},
	LogIt["-- LinkTrailFromPacletURI: "<>u<>", "<>ToString[Length@primaryGuides]];
	(* get the primary guide *)
	pg = DocumentationBuild`Utils`GetPrimaryGuide[uri, primaryGuides];
	(* If no primary guide, then return root guide *)
	If[uri==="paclet:tutorial/VirtualBookOverview", Return[{{"paclet:tutorial/VirtualBookOverview"}}]];
	If[pg === uri, (* XX Root guide rename change XX *)Return[{{"paclet:guide/WolframRoot"}}] ];
	LogIt["-- "];
	trail =
	If[DocumentationBuild`Utils`CoreMathematicaPacletQ[uri], 
		(* wri guide/ref/tutorial... *)
		If[StringMatchQ[uri, "paclet:*"], 
			(* compute link trail for wri guides *)
			If[StringMatchQ[uri, "paclet:tutorial/*"],
				res = DocumentationBuild`Info`GetLinkTrailFromFilename[uri, DocumentationBuild`Navigation`$TutPages, primaryGuides];
			If[res === {}, {{"paclet:tutorial/VirtualBookOverview"}}, res],
				res = DocumentationBuild`Info`GetLinkTrailFromFilename[uri, DocumentationBuild`Navigation`$Pages, primaryGuides];
			If[res === {}, (* XX Root guide rename change XX *){{"paclet:guide/WolframRoot"}}, res]]
		,
			(* uri is not a wri guide... *)
			(* check to see if paclet uri has a primary guide *)
			If[(pg === (* XX Root guide rename change XX *)"paclet:guide/WolframRoot") || (pg === ""),
				(* return link to root guide *)
				(* XX Root guide rename change XX *){ {"paclet:guide/WolframRoot"} }
			,
				(* return root + primary guide *)
				DocumentationBuild`Utils`UnsortedUnion @ { Flatten @ DocumentationBuild`Utils`UnsortedUnion @
					(* XX Root guide rename change XX *)DeleteCases[{"paclet:guide/WolframRoot", DocumentationBuild`Utils`GetPrimaryGuide[pg, primaryGuides], pg}, ""|Null|$Failed]  }
			]
		]
	,
		(* return pakage's primary guide *)
		{{pg }}

	]

];

(* error check *)
GetLinkTrails[u___]:= (Message[GetLinkTrails::arg, u]; $Failed)
GetLinkTrails::arg = "Paclet uri expected, but found `1`";

GetLinkTrailFromFilename[filename_String /; StringMatchQ[filename, "paclet:ref/" ~~ __], data_List, primaryGuides_List] :=
Module[{list, primaryguide, fn = filename},
    Which[

      (* normal Symbol ref page *)
      StringFreeQ[fn, "ref/message/"],
        primaryguide = Quiet["PrimaryGuide" /. (filename /. primaryGuides)];
        (* next line does a check to see if ref page has an entry in the $CoreInformation data set *)
        If[primaryguide === Quiet[("PrimaryGuide" /. filename)], Message[GetLinkTrailFromFilename::nodat,filename];Return[{}]];
        (* continue as normal *)
        If[primaryguide === (* XX Root guide rename change XX *)"paclet:guide/WolframRoot", {primaryguide}, 
        list = Cases[data,
            {StringReplace[primaryguide, {"paclet:guide/"->"", ".nb"->""}] /. DocumentationBuild`Navigation`$NameToNumberRules, trail_} :> trail, Infinity] /. DocumentationBuild`Navigation`$NumberToNameRules;
        Join[#, {primaryguide, filename}] & /@ Flatten[list //. {a___, b_String /; ! StringMatchQ[b, "paclet:*"], c___} :> {a, "paclet:guide/" <> b, c}, 1]],

      (* message page code goes here eventually... *)
      !StringFreeQ[fn, "ref/message/"],
        (*code here*)
        {},

      (* catch all default value to prevent docs breaks for now *)
      True, {}]
];    

GetLinkTrailFromFilename[filename_String /; StringMatchQ[filename, "paclet:guide/" ~~ __], data_List, primaryGuides_List] := 
Module[{list},
	list = Cases[data, 
			{StringReplace[filename, {"paclet:guide/"->"", ".nb"->""}] /. DocumentationBuild`Navigation`$NameToNumberRules, trail_} :> trail, Infinity] /. DocumentationBuild`Navigation`$NumberToNameRules;
(*Print["List: "<>ToString@list];*)
	Append[#, filename]& /@ Flatten[list //. {a___, b_String /; ! StringMatchQ[b, "paclet:*"], c___} :> {a, "paclet:guide/" <> b, c}, 1]
];

GetLinkTrailFromFilename[filename_String /; StringMatchQ[filename, "paclet:tutorial/" ~~ __], data_List, primaryGuides_List] :=
Module[{list},
	list = Cases[data, 
			{StringReplace[filename, {"paclet:tutorial/"->"", ".nb"->""}] 
				/. DocumentationBuild`Navigation`$TutNameToNumberRules, trail_} :> trail, Infinity] 
				/. DocumentationBuild`Navigation`$TutNumberToNameRules;
(*Print["List: "<>ToString@list];*)
	Append[#, filename]& /@ Flatten[list //. {a___, b_String /; ! StringMatchQ[b, "paclet:*"], c___} :> {a, "paclet:tutorial/" <> b, c}, 1]
];

GetLinkTrailFromFilename[filename_String /; StringMatchQ[filename, "paclet:howto/" ~~ __], data_List, primaryGuides_List] :=
{};

GetLinkTrailFromFilename::list = "Navigation\[RawQuote]$Pages not a list: `1`";
GetLinkTrailFromFilename::nodat = "The specified URI, `1`, is missing a Primary Guide.";


End[]
EndPackage[]
