(* :Title: Make.m *)

(* :Authors:
	Andrew Hunt, Shaun McCance, Todd Gayley
	andy@wolfram.com, shaunm@wolfram.com, tgayley@wolfram.com
*)

(* :Package Version: 0.50 *)

(* :Mathematica Version: 6.0 *)
                     
BeginPackage["DocumentationBuild`Export`", {
  "DocumentationBuild`Common`",
  "DocumentationBuild`Info`",
  "DocumentationBuild`Utils`",
  "DocumentationBuild`Make`",
  "Transmogrify`"
}];


(******************************************************************************)
(** Usage Statements **)

ExportDocumentation::usage = "ExportDocumentation[file, file.nb | Notebook, ExportFormats -> {...}] creates\
 documentation files from source notebook file.nb.";

(* Web *)
ExportWebPage::usage = "ExportWebPage[file.html, file.nb | Notebook] takes a production notebook and exports to xml.";
BuildLog;
ExportWebPageFromResultNB::usage="";
CreateFeaturedPage::usage="";
ExportNavigatorWebPage::usage = "ExportNavigatorWebPage[file.html, file.nb] converts a function navigator notebook into a web page.";
ExportLandingPage::usage = "ExportLandingPage[path] creates landing page for future site from Bill's reports.";
GetFutureLandingContent::usage = "Generate data for future landing page in http://reference.future.wolfram.com/mathematica.html.";
MakeLandingContent::usage="";
(* XMLTransform functions *)
htmlBoxesQ;
inputQueue;
ReplaceSpecialCharacters;
WrapInOpenCloseButton;
AddFirstAndLastToexpandWraps::usage="";
CreateLearningResources::usage="";
deleteNonASCII::usage="";

BuildDocPacletIndex::usage = "BuildDocPacletIndex builds the PcletDB.m file \
 files from a documentation directory. It is typically run after ExportOnlineNotebooks\
 and operates on the output directory of converted notebook files produced by that function.";
  
MakeMarketingThumbnailList::usage = "";

(* load external data *)

(* Load TreeBrowse` code from faisal until in System` *)
(* Unprotect per faisal *)
WriteLog[logfile, "Refresh TreeBrowse.m definitions."];
If[FindFile["TreeBrowse.m"] =!= $Failed,
  Quiet[Unprotect/@Names["TreeBrowse`*"]; Remove["TreeBrowse`*"]]; Get["TreeBrowse.m"],
  WriteLog[logfile, "FAILED -- Using existing definitions instead", LineSeparator->None, TimeStamp->None]];


(* load temporary compatibility package until CharacterReplacements is completely fixed *)
Get[ToFileName[{DocumentationBuild`Common`$DocumentationBuildDirectory, "Internal", "data"}, "CharacterReplacements.m"]];

(* Get walsh's url encode/decoder *)
Get[ToFileName[{DocumentationBuild`Common`$DocumentationBuildDirectory}, "ShortenIdentifier.m"]];

(*$NewIn6FunctionList = Get[ToFileName[{DocumentationBuild`Common`$DocumentationBuildDirectory, "Internal", "web", "data"}, "NewIn6FunctionList.m"]];
*)
(* Usage and function information files for packages *)
$PackageFunctionInformation = Get[ToFileName[{DocumentationBuild`Common`$DocumentationBuildDirectory, "Internal", "data"}, "PackageFunctionInformation.m"]];
$PackageUsage = Import[ToFileName[{DocumentationBuild`Common`$DocumentationBuildDirectory, "Internal", "data"}, "PackageUsage.m"], "Lines"];


Begin["`Private`"];

(******************************************************************************)
(** ExportDocumentation **)

Options[ExportDocumentation] = {
    "ExportFormats" -> {"Notebook" -> "OutputDirectory", "HTML"},
    "Cache" -> False,
    "CacheLocation"->"",
    "Language" -> "English",
    "BuildNumber" -> "0000000",
    "CoreInformation" -> Automatic
};

ExportDocumentation::support = "Format not supported: `1`";

ExportDocumentation[outFile_String, file_String, opts___?OptionQ]:=
  ExportDocumentation[outFile, GetQuietly[file], "FilePath" -> file, opts];

ExportDocumentation[file_String, nb_Notebook, opts___?OptionQ]:=
  Module[{dir, outdir, outfile, expr, info, formats, language, res},
    info = GetNotebookInformation[nb, opts];
    
    ServerLog[ separator <> "\nExportDocumentation:\n"];
    ServerLog[ nb ];
    ServerLog[ info ];

    {formats, language} = {"ExportFormats", "Language"} /. {opts} /. Options[ExportDocumentation];
    formats = Flatten@{formats};
    formats = Map[If[StringQ[#], # -> {}, #]&, formats];

    BuildLog["** Entering ExportDocumentation **"];
    BuildLog["   Language: "<>language ];
    BuildLog["   Formats: "<>ToString[formats] ];

    BuildLog[separator ];

    res = {};

    (* create in-product notebook *)
    expr = MakeNotebook[nb, info, opts];

	(* set default export directory *)
    dir = DirectoryName[file];
    BuildLog["   Default export directory: "<>dir ];

    (* create in-product notebook *)    
    If[MemberQ[formats, ("Notebook" -> _)],
      outfile = FileBaseName[file];
      If[Transmogrify`GetFileExtension[outfile] != "nb",
        outfile = FileBaseName[outfile] <> ".nb"];

      If[MemberQ["Notebook" /. formats, ("OutputDirectory" -> _)],
        outdir = "OutputDirectory" /. ("Notebook" /. formats),
        outdir = DirectoryName[file] (*ToFileName[{dir, "Files", "Notebooks"}]*) ];
      (* make sure directory exists *)
      CreateDirectory[ outdir ];
      (* export notebook and add notebook's path to result list *)
      BuildLog["   Notebook path: "<>ToFileName[{outdir}, outfile] ];
      res = Append[res,
        "Notebook" -> Export[ToFileName[{outdir}, outfile], expr]];
    ];

    BuildLog[separator ];

    (* create on-web page *)
    If[MemberQ[formats, ("HTML" -> _)],
      outfile = FileBaseName[file];
      If[Transmogrify`GetFileExtension[outfile] != "html",
        outfile = FileBaseName[outfile] <> ".html"];

      If[MemberQ["HTML" /. formats, ("OutputDirectory" -> _)],
        outdir = "OutputDirectory" /. ("HTML" /. formats),
        outdir = DirectoryName[file] (*ToFileName[{dir, "Files", "HTML"}]*) ];
      (* make sure directory exists *)
      CreateDirectory[ outdir ];
      (* export html and add mlht's path to result list *)
      BuildLog["   HTML path: "<>ToFileName[{outdir}, outfile] ];
      
      res = Append[res,
        "HTML" -> ExportWebPage[ToFileName[{outdir}, outfile], expr, info, opts]];
    ];

  res
];


(******************************************************************************)
(** ExportWebPage **)
(* 
    ExportWebPage[outfile, expr, opts] 
    this is the main Transmogrify[] wrapper.
    NOTE - outfile = None just returns the raw Transmogrify output
           ExportTest->True causes ExportWebPage to use TransmogrifyString[]
           the shorthand functions rawmogtest[] and mogtest[], respectively,
           are wrappers for these two functionalities.
*)

Options[ExportWebPage] = {
    "DefaultParameters" -> {},
    "BuildMode" -> Normal,
    "LogFile" -> None,
    "NeedPlatMsgIn" -> None,
    "ExportTest" -> False,
    "ModifiedNotesQ" -> "False",
    "Version" -> "6.0",
    "CompleteHTMLQ" -> False,
    "BuildNumber" -> "0000000",
    "HistoryData"->{"New"->"", "Modified"->"", "Obsolete"->"", "Excised"->""},
    "ShowDashboardQ" -> False,
    "IsManipulate" -> True,
    "RelatedSearches" -> False,
    "Cache" -> False,
    "CacheLocation"->"",
    "DataSource" -> Automatic,
    "CoreInformation"-> Automatic,
    "NotebookContent" -> "Source",
    "IncludeLinkTrails" -> True,
    "Language" -> "English",
    "FunctionPaclet" -> False,
    "SearchSSI" -> "<!--#include virtual=\"/searchlink.cgi?$QUERY_STRING\"-->",
    "FooterSSI" -> "<!--#include virtual=\"/includes/footer2010.cgi\"-->",
    "LanguageLinkSSI" -> "<!--#include virtual=\"/includes/langlink.cgi\"-->",
    "CopyrightSSI" -> "<!--#include virtual=\"/mathematicaIncludes/copyright.html\"-->",
    "URLBaseRemote" -> Automatic
};

ExportWebPageFromResultNB[outfile_String, nbfile_String, opts___?OptionQ] := 
Module[{lang,metainfo,nbInfo, nbobj, nb, nbget, nbobj2}, 
	metainfo = DocumentationBuild`Utils`GetSearchMetaDataList[nbfile];
	lang = Switch[("language" /. metainfo), "en", "English", "ja", "Japanese", "zh", "Chinese", 
		"es", "Spanish", _, "English"];

	DocumentationBuild`Export`$CoreInformation[lang_] := DocumentationBuild`Export`$CoreInformation[lang] = LoadCoreInformation["CoreInformation" /.  {opts} /. Options[ExportWebPage], lang];

nbInfo={"Title" -> ("windowtitle" /. metainfo), 
 "TitleModifier" -> ("titlemodifier" /. metainfo), 
 "HistoryData" -> Apply[{"New" -> #1, "Modified" -> #2, "Obsolete" -> #3, "Excised" -> #4} &, ("history" /. metainfo)], 
 "EntityType" -> ("type" /. metainfo), 
 "PacletName" -> ("paclet" /. metainfo), 
 "Context" -> ("context" /. metainfo), 
 "URI" -> ("uri" /. metainfo), 
 "Keywords" -> ("keywords" /. metainfo),
 "TutorialCollectionLinks" ->  ("tutorialcollectionlinks" /. metainfo),
 "ForeignFunctionKeywords" -> {}, 
 "Synonyms" -> ("synonyms" /. metainfo), 
 "Summary" -> ("summary" /. metainfo), 
 "TMJTest" -> False,
 "Flag" -> DocumentationBuild`Utils`GetValueFromOptions[DocumentationBuild`Export`$CoreInformation[lang], "paclet:" <> ("uri" /. metainfo), "Flag"], 
 "IndexQ" -> ("index" /. metainfo)};
 
 System`StrokeForm;

(* This is because of a problem seen with the version 10 build not making CellGroups right *)

nbobj = NotebookOpen[nbfile];
nbget = NotebookGet[nbobj];
NotebookClose[nbobj];
nbobj2 = NotebookPut[nbget];
nb=NotebookGet[nbobj2];
NotebookClose[nbobj2];
 
ExportWebPage[outfile, nb, nbInfo, opts]
    ];

ExportWebPage[_, $Failed, ___] = $Failed;

ExportWebPage[outfile_String, nbfile_String, nbInfo_List, opts___?OptionQ] :=
  ExportWebPage[outfile, GetQuietly[nbfile], nbInfo, opts];

(*********************************************************************
billw 20151112: this definition of ExportWebPage is our final chance
to munge a notebook expression before sending it off to Transmogrify.

  |------------------|
  | source notebooks |
  |------------------|
          ||
          || <= make.m
          ||
          \/
|----------------------|
| in-product notebooks |
|----------------------|
          ||
          || <= export.m: do anything needed to 
          ||    make html transformation easier
          ||    [you are here]
          \/
|- - - - - - - - - - - - -+
| <<intermediate massaged | <-- this is ephemeral; 
|    expression>>         | it's also what gets transmogrified
|- - - - - - - - - - - - -+
          ||
          || <= XMLTransforms
          ||
          \/
       |------|
       | html |
       |------|

*********************************************************************)

ExportWebPage[file_String, nb_?(!AtomQ[#]&), info_List, opts___?OptionQ] :=
  Block[{expr, outfile, language, langext, params, logfile, test, isManipulate, relatedSearches, cache, cachelocation,
      completeHTMLQ, buildNumber, history, showDashboardQ, dataSource, contextName, keywordList = {},
      transform, layoutfile, pageTitle = "None", pageTitleModifier = "", linkTrail = {}, includeLinkTrails, specialKeywords, tutorialcollectionlinks,
      tutorialFunctionList={}, tutorialMoreAboutList = {}, relatedTutorialsList = {}, 
      guideFunctionList={}, guideMoreAboutList = {}, guideTutorialsList = {}, 
      htmlHeader = "", htmlFooter = "", htmlBase = "", filePath, flag = None,
      urlBaseRemote, wsmlinkQ, wsmlibQ, uri, summaryText, searchSSI, footerSSI, languageLinkSSI, copyrightSSI,
      pacletQ, isTMJQ, isSolutionsQ, rootCaptions, posToInsertRef, fileName, nbobj, nbget, nbobj2,
(*
        synonymList = {}, r, entityType,pacletName,
       seeAlsoFunctionList, moreAboutGuideList, version, pageType=pagetype,
      number, documentFlag, convertFunc,
      outFile=outfile, 
      urlRules, exampleSectionList, tutorialList, uri, dashboard,  relativeBase,
*)
      layoutdir = ToFileName[{$DocumentationBuildDirectory, "XMLTransforms", "HTML", "Layouts"}],
      pacletName, entityType, synonymList, nbObj, anchorBarList, notfullysupported, featuredexampledata,
      seeAlsoFunctionList, moreAboutGuideList, exampleSectionList,  tutorialList, dashboard,
      relativeBase, properties={}, r, errMessageList={},
      buildmode, additionalstyles, defaultstylesheet, pacletname
    },
    
    (* Check to see if a WSMLink build *)
    
    uri = "URI" /. info;
    pacletname = "PacletName"/.info;
    wsmlinkQ = StringMatchQ[uri, "*WSMLink/*"]||StringMatchQ[pacletname, "*SystemModeler*"];
    wsmlibQ = StringMatchQ[pacletname, "Wolfram SystemModeler Library"];
    
    
    urlBaseRemote = "URLBaseRemote" /. {opts} /. Options[ExportWebPage] /. {Automatic :> "/language/"};

    Check[

    BuildLog["** Entering ExportWebPage **"];
  
    Off[CreateDirectory::filex];
    (* turn off error message limit *)
    Off[General::stop];

    nbobj2 = NotebookPut[nb];
    nbget=NotebookGet[nbobj2];
    NotebookClose[nbobj2];

    expr = nbget;
    
    SetOptions[$FrontEndSession, Rule[AutoStyleOptions, {Rule["HighlightUndefinedSymbols", False]}]];
    
    additionalstyles = {
      Cell[StyleData["InlineFormula"], FontFamily->"Source Sans Pro", FontSize->Inherited 1.08],
      Cell[StyleData["MR"], FontFamily->"Source Sans Pro"],
      Cell[StyleData["StandardForm"],FontFamily->"Source Sans Pro"],
      Cell[StyleData["InputForm"],FontFamily->"Source Sans Pro"],
      Cell[StyleData["Input"],  FontFamily->"Source Sans Pro", FontSize->13, LineSpacing->{1.2, 0}],
      Cell[StyleData["Output"], FontSize->13],
      Cell[StyleData["ServiceSubsection", StyleDefinitions -> StyleData["Section"]],Editable->False, CellFrame->False, CellGroupingRules->{"SectionGrouping", 40},FontSize->16, FontWeight->"Bold",FontColor->GrayLevel[0.305882]],
      (*********************************************************************
      https://jira.wolfram.com/jira/browse/WLDOCS-870?focusedCommentId=2358054&page=com.atlassian.jira.plugin.system.issuetabpanels:comment-tabpanel#comment-2358054
      *********************************************************************)
      Cell[StyleData["TextFrame"], Thickness -> .25]
    };
             
    defaultstylesheet= Cell[StyleData[StyleDefinitions -> FrontEnd`FileName[{"Wolfram"}, "Reference.nb"]]];
    
    expr = expr /. 
    {
      Rule[StyleDefinitions, FrontEnd`FileName[___]] :>
        Rule[StyleDefinitions, Notebook[Flatten[{defaultstylesheet, additionalstyles}], Rule[AutoStyleOptions,{Rule["HighlightUndefinedSymbols",False]}]]],
      Rule[StyleDefinitions, Notebook[{contents___}, options___]] :>
        Rule[StyleDefinitions, Notebook[Flatten[{contents, additionalstyles}], options, Rule[AutoStyleOptions,{Rule["HighlightUndefinedSymbols",False]}]]]
    };
   
    linkTrail = Cases[Options[expr],Rule["LinkTrails",a__]:>a,Infinity];
    linkTrail = If[MemberQ[{List[""], $Failed, ""}, linkTrail], {},
    Apply[CellGroupData, Cases[linkTrail, _RowBox, Infinity]] /. CellGroupData[a__] :> Cell[CellGroupData[{a}]]
    ];

    linkTrail = If[MemberQ[{List[""], $Failed, ""}, linkTrail], {},
    linkTrail /. RowBox[{a__}] :> Cell[TextData[{a}], "LinkTrail"] /. RowBox[{b__}] :> Cell[TextData[{b}]] /. 
     ButtonBox[c1__String, ButtonData -> c2_String, BaseStyle -> {"Link", "DockedLinkTrail"}] :> 
     ButtonBox[c1, BaseStyle -> {"Link", "LinkTrail"}, ButtonData -> c2] /. 
     Cell[">", "DockedLinkTrailSeparator"] :> StyleBox[" > ", "LinkTrailSeparator"]
    ];
     
     rootCaptions = Cases[Options[expr],Rule["RootCaptions",a__]:>a,Infinity];
     rootCaptions = If[Length[Map[Cell[#, "GuideText"] &, rootCaptions[[1]]]] >= 8, Take[Map[Cell[#, "GuideText"] &, rootCaptions[[1]]], 8]];

    (* Add DocumentationBuild\XMLTransforms if it isn't already on path *)
    If[ !MemberQ[ToFileName[{$DocumentationBuildDirectory, "XMLTransforms"}], $XMLTransformPath], 
      PrependTo[$XMLTransformPath, ToFileName[{$DocumentationBuildDirectory, "XMLTransforms"}] ];
    ];

    outfile = file;

    (* parse user opts *)
    params = "DefaultParameters" /. {opts} /. Options[ExportWebPage];
    logfile = "LogFile" /. {opts} /. Options[ExportWebPage];
    test = "ExportTest" /. {opts} /. Options[ExportWebPage];
    isManipulate = "IsManipulate" /. {opts} /. Options[ExportWebPage];
    relatedSearches = "RelatedSearches" /. {opts} /. Options[ExportWebPage];
    cache = "Cache" /. {opts} /. Options[ExportWebPage];
    cachelocation = "CacheLocation" /. {opts} /. Options[ExportWebPage];
    completeHTMLQ = "CompleteHTMLQ" /. {opts} /. Options[ExportWebPage];
    buildNumber = "BuildNumber" /. {opts} /. Options[ExportWebPage];
    showDashboardQ = "ShowDashboardQ" /. {opts} /. Options[ExportWebPage];
    dataSource = "DataSource" /. {opts} /. Options[ExportWebPage];
    includeLinkTrails = "IncludeLinkTrails" /. {opts} /. Options[ExportWebPage];
    searchSSI = "SearchSSI" /. {opts} /. Options[ExportWebPage];
    footerSSI = "FooterSSI" /. {opts} /. Options[ExportWebPage];
    languageLinkSSI = "LanguageLinkSSI" /. {opts} /. Options[ExportWebPage];
    copyrightSSI = "CopyrightSSI" /. {opts} /. Options[ExportWebPage];
    pacletQ = "FunctionPaclet" /. {opts} /. Options[ExportWebPage];
    buildmode = "BuildMode" /. {opts} /. Options[ExportWebPage];
	specialKeywords = If[Length[Cases[expr, Rule["specialkeywords",a_]:>a, Infinity]]>0, Cases[expr, Rule["specialkeywords",a_]:>a, Infinity][[1]], {}];
    isTMJQ = "TMJTest" /. {opts} /. Options[ExportTMJWebPage];
    isSolutionsQ = "SolutionsTest" /. {opts} /. Options[ExportSolutionsWebPage];
    
    notfullysupported = Cases[expr, Rule["NeedPlatMsgIn",a__]:>a, Infinity];
    
    notfullysupported = Which[
    	   notfullysupported == {}, 
    	      "", 
    	   StringQ[notfullysupported[[1]]],
    	      Uncompress[notfullysupported[[1]]],
    	   True,
    	      ""
    	   ];
    
    notfullysupported = 
    If[ListQ[notfullysupported], 
    	Which[Length[notfullysupported] == 0,
    		  "",
    		  Length[notfullysupported] == 1,
    		  notfullysupported[[1]],
    		  Length[notfullysupported] > 1,
    		  notfullysupported[[1]],
    		  True, 
    		  ""],
    	""];    
      
(* FIXME
    {titlename, version, filePath, notebookContent} =
      { TitleName, Version, FilePath, "NotebookContent"} 
        /. {opts} /. Options[ExportWebPage];
*)

    (* Which language? *)
    language = "Language" /. {opts} /. Options[ExportWebPage];
    langext = Switch[language,
      "Japanese", "ja",
      "ChineseSimplified", "zh",
      "Spanish", "es",
      "English", "en",
      _, "en"
    ];
    
	(* get paclet name *)
    pacletName = "PacletName" /. info;
	(* get entity type *)
    entityType = "EntityType" /. info;
	(* get notebook flag *)
	flag = "Flag" /. info;
	(* get notebook history *)
	history = "HistoryData" /. info;
	(* get notebook summary *)
    summaryText = "Summary" /. info;
    (* get tutorial collection *)
    tutorialcollectionlinks = "TutorialCollectionLinks" /. info;
    
    pageTitle =
      If[entityType =!= "RootGuidePage",
        "Title" /. info, {"Wolfram Mathematica: Documentation Center"} ];

    pageTitleModifier ="TitleModifier" /. info;

    If[dataSource === "DataPaclet", (
       metadataList = DocumentationBuild`Utils`GetSearchMetaDataList@expr;
       uri = If[OptionQ[metadataList], "uri" /. metadataList, $Failed];
       If[MemberQ[{$Failed, "uri"}, uri],  
          BuildLog["UNABLE TO DETERMINE URI -- CONVERSION FAILED"];
          Return[$Failed]
        ];
      (* capture keywords *)
      keywordList = InternalUtilities`DocumentationData["paclet:"<>uri, "Keywords"];
      synonymList = InternalUtilities`DocumentationData["paclet:"<>uri, "Synonyms"]; 
    ), (
      (* capture keywords *)

      keywordList = "Keywords" /. info;
      synonymList = "Synonyms" /. info;
      (* page uri *)
      uri = "URI" /. info;
    )];

	(* get classification *)
    contextName = GetClassificationBoxes[info, language];
    
	(*contextName = If[(entityType === "Guide") && (pacletName === "Mathematica"), "", 
      AddSpaceToCamelCase[contextName]];*)
    contextName = AddSpaceToCamelCase[contextName];

	(* add in foreign keywords *)
	fileName = Last[Drop[StringSplit[outfile, {"/","\\", "."}], -1]];
	keywordList = Flatten[{Append[keywordList, fileName]}];
	keywordList = If[Length[keywordList] > 0,
		StringDrop[StringDrop[ToString[keywordList], 1], -1],
		keywordList];
	
(*********************************************************************
here beginneth our last-chance expression munging before running
Transmogrify
*********************************************************************)

	(* Filter out the bad cells in function page examples causing double traingles. *)
	expr = expr /. Cell[BoxData[InterpretationBox[Cell[TextData[
		Cell[BoxData[InterpretationBox[Cell[TextData[{a1__, Cell[a2_,"ExampleCount"]}],"ExampleSection"],b__]]]],"ExampleSection"],c__]],
		"ExampleSection",d__]:>Cell[BoxData[InterpretationBox[Cell[TextData[{a1, Cell[a2,"ExampleCount"]}],"ExampleSection"],b]],"ExampleSection"];
	
    (* Remove InterpretationBox[] for example section cells *)
        
    If[entityType==="Root Guide", 
    	nbbox = expr /. {Notebook[{a___}, b___]:>Notebook[{"XXXX"}, b]};
	    tagboxes = Cases[expr, TagBox[ActionMenuBox[a___], b___] :> TagBox[ActionMenuBox[a], b], Infinity];
	    newnbcontent = Cell[CellGroupData[
			  Map[Cell[CellGroupData[#]] &, 
			   Table[expr2 = (tagboxes[[iterator]] /. {TagBox[
			           ActionMenuBox[
			            PaneSelectorBox[{False -> 
			               ButtonBox[
			                OverlayBox[{___, 
			                  TagBox[GridBox[{{___, 
			                    StyleBox[w_, 
			                    "RootSectionTitle", ___], ___}}, ___], 
			                   "Grid"], ___}, ___], ___], ___}, ___], {b___}, ___], ___] :> {{Cell[ToExpression[w], "RootItemSectionTitle"]}, {b}}}) /.
			      {
			       RuleDelayed[aaaa_, 
			         FEPrivate`FrontEndExecute[
			          FrontEndToken["OpenHelpLink", {bbbb_, ___}]]] :> 
			        Cell[BoxData[ButtonBox[ToExpression[aaaa], Rule[BaseStyle,"Link"], Rule[ButtonData,bbbb]]], "RootItem"],
			       Delimiter -> Cell["----", "RootItemDelimiter"]
			       }; Flatten[{expr2[[1]], 
			     Cell[CellGroupData[
			       Map[Cell[CellGroupData[#]] &, 
			        SplitBy[expr2[[2]], 
			         SameQ[#, Cell["----", "RootItemDelimiter"]] &]]]]}], {iterator, 1, 
			     Length[tagboxes]}]]]];
		 expr = nbbox /. {Notebook[{"XXXX"}, b___] :> Notebook[{newnbcontent}, b]};
      ];
     
    expr = expr /. {Cell[BoxData[
   GridBox[{{a___, 
      GridBox[b___, 
       Rule[GridBoxItemSize, {"Columns" -> {{Scaled[0.75]}}, 
         "Rows" -> {{0.4}}}], c___]}}]], "CharacterImage"] :> 
 Cell[BoxData[
   GridBox[{{a, 
      GridBox[b, 
       Rule[GridBoxItemSize, {"Columns" -> {{}}, "Rows" -> {{0.4}}}], 
       c]}}]], "CharacterImage"] };
        
    expr = expr /.
      Cell[a_,"ExampleSection", "ExampleSection", b___]
        :> Cell[a, "ExampleSection", b];
     expr = expr /.
      Cell[a_,"ExampleSubsection", "ExampleSubsection", b___]
        :> Cell[a, "ExampleSubsection", b];
    expr = expr /.
      Cell[BoxData[InterpretationBox[stuff:Cell[a_,"ExampleSection",b___], $Line = 0;]], "ExampleSection", c___]
        :> Cell[a, "ExampleSection", b, c];
    expr = expr /.
      Cell[BoxData[InterpretationBox[stuff:Cell[a_,"ExampleSubsection",b___], $Line = 0;]], "ExampleSubsection", c___]
        :> Cell[a, "ExampleSubsection", b, c];
    expr = expr /.
      Cell[BoxData[InterpretationBox[stuff:Cell[a_,"ExampleSubsubsection",b___], $Line = 0;]], "ExampleSubsubsection", c___]
        :> Cell[a, "ExampleSubsubsection", b, c];
    expr = expr /.
      Cell[BoxData[InterpretationBox[stuff_, $Line = 0;]], "PrimaryExamplesSection", a___]
        :> Cell[Localized["PrimaryExamplesSectionText", language], "PrimaryExamplesSection", a];
    expr = expr /. Cell[BoxData[InterpretationBox[stuff_, $Line = 0;]], "ExampleDelimiter", s___]
        :> Cell["\t", "ExampleDelimiter", s];
    expr = expr /.
      Cell[cellcontent_, "InlineFormula", "TemplateInclusion", cellOptions___]
        :> Cell[cellcontent, "InlineFormula", cellOptions];
    expr = expr /.
      Cell[cellcontent_, "InlineFormula", "TemplateInclusion", cellOptions___]
        :> Cell[cellcontent, "InlineFormula", cellOptions];
    expr = expr /.
      Cell[cellcontent_, "InlineFormula", "TemplateInclusion", cellOptions___]
        :> Cell[cellcontent, "InlineFormula", cellOptions];
        
    (* Symbols/AutoIndent.nb revision 1.70, Symbols/Piecewise.nb revision 1.91 *)
    expr = expr /. Cell[BoxData[FormBox[DynamicBox[ToBoxes[If[$OperatingSystem === "MacOSX", a_, b_], c__], d___], TraditionalForm]], "KeyEvent", e___] :>
      StyleBox[b, "KeyEvent"];

    expr = expr /. Cell[BoxData[FormBox[StyleBox[DynamicBox[ToBoxes[If[$OperatingSystem === "MacOSX", a_, b_], 
      c__], d___], "KeyEvent"],TraditionalForm]], "KeyEvent", e___]:>
   		StyleBox[b, "KeyEvent"];
        
    expr = expr /. StyleBox[DynamicBox[ToBoxes[If[$OperatingSystem === "MacOSX", a_, 
    "\[NegativeMediumSpace]"], b___], c___], "KeyEvent"]:>" ";
 
    expr = expr /. Cell[BoxData[FormBox[RowBox[{a___, StyleBox[DynamicBox[ToBoxes[If[$OperatingSystem === "MacOSX", aa_, bb_], c__], ___], "KeyEvent"]} ], Style_  ]], NotebookDefault, cellOptions___]:> Sequence[a, StyleBox[bb, "KeyEvent"]  ];

    expr = expr /.
      Cell[BoxData[FormBox[RowBox[{a___, StyleBox[DynamicBox[ToBoxes[b_, _], ___], "KeyEvent"]} ], 
          Style_  ]], NotebookDefault, cellOptions___]:> Sequence[a, StyleBox[b, "KeyEvent"]  ];
        
    expr = expr /. Cell[BoxData[FormBox[DynamicBox[If[$OperatingSystem === "MacOSX", a_String, b_String],
    __], TextForm]], "KeyEvent"]:>Sequence[StyleBox[b, "KeyEvent"], " (", StyleBox[a, "KeyEvent"], " on MacOSX)"];
    
    expr = expr /. StyleBox[DynamicBox[ToBoxes[If[$OperatingSystem === "MacOSX", a_, b_], StandardForm],c___], "KeyEvent"] :> 
 StyleBox[ToBoxes[b, StandardForm], "KeyEvent"];
 
     expr = expr /. DynamicBox[If[$OperatingSystem === "MacOSX", a_, b_], c___]:> b;
     
     expr = expr /. DynamicBox[ToBoxes[If[$OperatingSystem === "Windows", a_, b_], c___], d___]:> a;
     
     (* System/Piecewise.nb revision 1.91 *)
     expr = expr /. DynamicBox[ToBoxes[If[$OperatingSystem === "MacOSX", a_, b_], c___], d___] :> a;

     expr = expr /. StyleBox[DynamicBox[ToBoxes[If[$OperatingSystem === "MacOSX", a__, 
    "\[NegativeMediumSpace]"], StandardForm], c___], "TableText"]:>" ";
    
     expr = expr /. StyleBox[" ", "KeyEvent"] :> " ";
     
     expr = expr /. {
     Cell[BoxData[FormBox[a_, b_]], "InlineFormula"]:>
          	Cell[BoxData[FormBox[a, TraditionalForm]], "InlineFormula"],
     Cell[BoxData[FormBox[a_, b_]], "InlineMath"]:>
     		Cell[BoxData[FormBox[a, TraditionalForm]], "InlineMath"]};
     
     
     expr = expr /. {		
     Cell[BoxData[RowBox[{a__ButtonBox, b__}]], d_String/;MemberQ[{"InlineFormula","InlineCode","InlineOutput","InlineMath","FunctionTemplate"}, d]]:>
     	Sequence[Cell[BoxData[a], d],Cell[BoxData[RowBox[{b}]], d]],
     Cell[BoxData[RowBox[{a__, b__ButtonBox, c__}]], d_String/;MemberQ[{"InlineFormula","InlineCode","InlineOutput","InlineMath","FunctionTemplate"}, d]]:>
     	Sequence[Cell[BoxData[RowBox[{a}]], d], Cell[BoxData[b], d],
     		Cell[BoxData[RowBox[{c}]], d]],
     Cell[BoxData[RowBox[{a__, b__ButtonBox}]], d_String/;MemberQ[{"InlineFormula","InlineCode","InlineOutput","InlineMath","FunctionTemplate"}, d]]:>
     	Sequence[Cell[BoxData[RowBox[{a}]], d], Cell[BoxData[b], d]]
     };
     
    
    expr = expr /.
    Cell[TextData[{a___,Cell[aa__, "KeyEvent"],b___}], "Text", c___]:>
    Cell[TextData[{a,StyleBox[aa, "KeyEvent"],b}], "Text", c];
    expr = expr /.
      Cell[cellcontent__, "NumberedEquation", cellOpts__]
        :>  Cell[cellcontent, "DisplayFormula", cellOpts];
    expr = expr /. 
      Cell[BoxData[GridBox[{{a___, 
          Cell[TextData[{b___, Cell[c__, "InlineMath", d___], e___}]], 
            f___}}]], "Usage", g___]
        :> Cell[BoxData[GridBox[{{a, 
            Cell[TextData[{b, Cell[c, "InlineMath", Rule["GridBoxOptions", {Rule[BaseStyle, {Rule[FontFamily, "Times"]}], 
              Rule[GridBoxSpacings, {}], Rule[GridBoxDividers, {}]}], d], e}]], 
                f}}]], "Usage", g];
   
   expr = expr /. 
      Cell[BoxData[GridBox[{{a___, 
          Cell[TextData[{b___}]], 
            c___}}]], "FormatUsage", d___]
        :> Cell[TextData[{b}], "FormatUsage"];
          
    expr = expr //. 
      Cell[a_String, "DeviceDoubleUsage", d___]
        :> Cell[a, "DeviceUsage", d];
   
       expr = expr /. Cell[a___, "DeviceUsage", b___] :> Cell[MapAll[If[StringQ[#], StringReplace[#, "\n" -> "\[LineSeparator]"], #] &, a], "DeviceUsage", b];
    
    
    expr = FixedPoint[
 Replace[#, 
   Notebook[{aaa___, Cell[CellGroupData[{a:Cell[___, "ObjectNameAlt", ___], b___, c:Longest@PatternSequence[Cell[___, "DeviceNotes", ___] ..], d___}, aa___], bb___], bbb___}, ccc___] :> 
    Notebook[{aaa, Cell[CellGroupData[{a, b, Cell[CellGroupData[{Cell["temp", "DeviceList"], c}]], d}, aa], bb], bbb}, ccc]] &, expr];
   
    expr = FixedPoint[
 Replace[#, 
   Notebook[{aaa___, Cell[CellGroupData[{a:Cell[___, "ObjectNameGrid"|"ObjectNameAlt", ___], b___, c:Longest@PatternSequence[Cell[___, "FormatNotes", ___] ..], d___}, aa___], bb___], bbb___}, ccc___] :> 
    Notebook[{aaa, Cell[CellGroupData[{a, b, Cell[CellGroupData[{Cell["temp", "FormatList"], c}]], d}, aa], bb], bbb}, ccc]] &, expr];

    expr = expr /. Cell[a___, "FormatUsage", b___] :> Cell[MapAll[If[StringQ[#], StringReplace[#, "\n" -> "\[LineSeparator]"], #] &, a], "FormatUsage", b];
    
expr = expr /. 
   Cell[BoxData[GridBox[{{"", Cell[a___]}}]], "IndicatorUsage", 
     d___] :> Cell[BoxData[Cell[a]], "IndicatorUsage"];
     
expr = expr /. 
   Cell[BoxData[GridBox[{{"", Cell[a___]}, {"", Cell[b___]}}]], 
     "IndicatorUsage", d___] :> 
    Cell[BoxData[{Cell[a],"\n", Cell[b]}], "IndicatorUsage"];
   
    expr = expr /. Cell[a___, "IndicatorUsage", b___] :> Cell[MapAll[If[StringQ[#], StringReplace[#, "\n" -> "\[LineSeparator]"], #] &, a], "IndicatorUsage", b];
    
    expr = expr /. Cell[a___, "UpgradeSummary", b___] :> Cell[MapAll[If[StringQ[#], StringReplace[#, "\n" -> "\[LineSeparator]"], #] &, a], "UpgradeSummary", b];
    
    expr = expr /.  Cell[a_,"TutorialCollectionSection",aa___]:> 
    Cell[Localized["TutorialCollectionSectionText", language],"TutorialCollectionSection",aa];
  		
    expr = expr /.
  Cell[CellGroupData[{Cell[a_,"TOCChapter",aa__],
  	Cell[CellGroupData[{Cell[TextData[b_ButtonBox],"TOCSection",bb__],
  		c__},Open]],d__,
  		Cell[CellGroupData[{Cell[e_,"TutorialCollectionSection"],ee__},Open]]},Open]]:>
  	{Cell[CellGroupData[{Cell[a,"TOCChapter",aa],
  	Cell[CellGroupData[{Cell[TextData[b],"TOCSection",bb],
  		c},Open]],d},Open]],
  		Cell[CellGroupData[{Cell[e,"TutorialCollectionSection"],ee},Open]]};
  
     expr = expr /. Cell[
    CellGroupData[{Cell[a_String, "TOCDocumentTitle", b___], x___, 
      Cell[CellGroupData[{Cell[aa__, "TOCChapter", bb___], 
         Cell[CellGroupData[{Cell["TUTORIAL COLLECTION", 
             "TutorialCollectionSection", c1___], c2___}, c3___]]}, 
        c4___]]}, c5___]] :> {Cell[
     CellGroupData[{Cell[a, "TOCDocumentTitle", b], x, 
       Cell[CellGroupData[{Cell[aa, "TOCChapter", bb]}, c4]]}, 
      c5]], Cell[
     CellGroupData[{Cell["TUTORIAL COLLECTION", 
        "TutorialCollectionSection", c1], c2}, c3]]};
                
    expr = expr /. 
   Cell[BoxData[StyleBox[RowBox[{"L", StyleBox[
           AdjustmentBox["A", cellOptions___],
           moreOptions___], "T", AdjustmentBox["E", otherOpts___], "X"}]]]] :> 
   Cell[BoxData[StyleBox[RowBox[{"L", StyleBox[
           AdjustmentBox["A", cellOptions],
           moreOptions], "T", AdjustmentBox["E", otherOpts], "X"}]]], "LaTex"];
         
     expr = expr /. 
   Cell[BoxData[StyleBox[RowBox[{"T", 
         AdjustmentBox["E", cellOptions___], "X"}]]]] :>
   Cell[BoxData[StyleBox[RowBox[{"T", 
         AdjustmentBox["E", cellOptions], "X"}]]], "LaTex"];
         
      expr = expr /.  PaneSelectorBox[{Rule[a_,Cell[TextData[StyleBox[aa_String,"RootNewInLink"]],"RootLinkText"]],
      	Rule[b_,Cell[TextData[StyleBox[bb_String,"RootLink"]],"RootLinkText"]]},c_]:>
      	PaneSelectorBox[{Rule[b,Cell[TextData[StyleBox[bb,"RootLink"]],"RootLinkText"]],
      		Rule[a,Cell[TextData[StyleBox[aa,"RootNewInLink"]],"RootLinkText"]]},c];
                      
     (*expr = expr /.  Cell[TextData[{x_String,
                      ButtonBox[btext_, BaseStyle -> "RootNewInLink", opts1___],
                      ButtonBox[" \[RightGuillemet]", BaseStyle -> "RootLink", ___] }], "RootLinkText"] :>
         Cell[TextData[{x,
                      ButtonBox[{StyleBox[btext, "RootNewInLink"]," \[RightGuillemet]"}, BaseStyle -> "RootLink", opts1] }], "RootLinkText"] ;
     expr = expr /.  PaneSelectorBox[{rl1:Rule[False, c1:Cell[TextData[{x__}]]], rl2:Rule[True, c2:Cell[TextData[{x__}],"RootDarkLink"]]},_] :>c2;
     expr = expr /.  PaneSelectorBox[{rl1:Rule[False, c1:Cell[TextData[{StyleBox[x_,"RootNewInLink"]}],"RootLinkText"]], rl2__},_] :>c1;
     
     expr = expr /.  PaneSelectorBox[{rl1:Rule[False, c1:Cell[x__]],
     	rl2:Rule[True,c2:Cell[TextData[{x__}],"RootDarkLink"]]},__]:>rl2;*)
     	
     	     (* fix complex subscript boxes with italic numbers *)
     expr = expr /. Cell[BoxData[SubscriptBox[StyleBox[a_, "TI"], StyleBox[StyleBox[RowBox[{b_, StyleBox[c_, FontSlant->"Plain"]}]], "TI"]]], "InlineFormula"] :>
      Cell[BoxData[SubscriptBox[StyleBox[a, "TI"], RowBox[{StyleBox[b, "TI"], StyleBox[c,FontSlant->"Plain"]}]]], "InlineFormula"];

     expr = expr /. Cell[a_String,"TableRowIcon","BrighterFlag",b___]:>
     Cell[a,"BrighterFlag",b];
     
    expr = expr /. Cell[BoxData[
  FormBox["m", Cell$$38228`TraditonalForm]], "InlineMath"]:>
  Cell[StyleBox["m", FontSlant->"Italic"], "InlineMath"];
  
    expr = expr /. ButtonBox[a1_, Rule[BaseStyle,{"Hyperlink", Rule[FontWeight,a3_]}], b_Rule ]:>
  	  StyleBox[ButtonBox[a1,BaseStyle->"Hyperlink", b], Rule[FontWeight,a3]];
  	  
  	expr = expr /.  Cell[a__, "Usage", GridBoxOptions -> {GridBoxBackground -> {b__,"Rows" -> {c___,
       Dynamic[If[CurrentValue[{TaggingRules, "ModificationHighlight"}] === True, RGBColor[0.92, 1, 0.59], None]], d___}, 
     e___}}, f___] :> Cell[a, "Usage", GridBoxOptions -> {GridBoxBackground -> {b,"Rows" -> {c, RGBColor[0.92, 1, 0.59], d}, e}}, f];
    
    expr = expr /. {Cell[TextData[Cell[BoxData[GridBox[a___]]]], "TableText"] :> Cell[TextData[Cell[BoxData[GridBox[a]], "TableText"]], "TableText"]};
    
    expr = expr/. Cell[BoxData[RowBox[{a___String, StyleBox[b_String, c_String, d___], 
   		 e___String}]], "InlineFormula", f___]:>
    Cell[BoxData[RowBox[{a, StyleBox[b, c, d], e}]], "InlineFormulaTableText", f];
    
    expr = expr/. Cell[BoxData[StyleBox[a_String,b_String,c___]],"InlineFormula",d___]:>
    Cell[BoxData[StyleBox[a,b,c]],"InlineFormulaTableText",d];
    
    expr = expr/. Cell[BoxData[StyleBox[a_String, b_String]], "InlineFormula"]:>
    Cell[BoxData[StyleBox[a, b]], "InlineFormulaTableText"];
     
    expr = expr/. Cell[BoxData[a_GridBox], "DefinitionBox", "TableHeaderLine", b___]:>
 Cell[BoxData[a], "DefinitionBox", "TableHeaderLine", GridBoxOptions->{
 GridBoxDividers->{
  "Columns" -> {{False}}, "ColumnsIndexed" -> {}, "Rows" -> {False, True, {False}, False}, 
   "RowsIndexed" -> {}}}, b];
    
    expr = expr/. Cell[TextData[Cell[BoxData[a_String]]], "TableText"]:>Cell[a, "InlineFormulaTableText"];
     
    expr = expr /. a_TextData :> ConvertTextData[a];
    expr = expr /. ConvertTextData[a_] :> a;
    
    If[entityType==="Guide" && !StringFreeQ[ "URI"/.info, {
    	   "ComputableDataOverview", 
    		"CoreLanguageOverview", 
    		"DataManipulationOverview", 
    		"DynamicInteractivityOverview", 
    		"MathematicsAndAlgorithmsOverview", 
    		"NotebooksAndDocumentsOverview",
    		"SystemsInterfacesAndDeploymentOverview", 
    		"VisualizationAndGraphicsOverview"}], entityType="Guide Overview"];
    		
        If[entityType==="Guide" && !StringFreeQ[ "URI"/.info, {
    	   "StandardExtraPackages"}], entityType="Standard Extra Packages"];

   If[entityType==="Guide" && !StringFreeQ[ "PacletName"/.info, {
    	   "Compatibility Package"}], entityType="Guide Overview"];

    (* indentify parameters to send to Transmogrify *)
    Switch[entityType,
        ("AppleScript" 
        | "File"
        | "LibraryLink C Function" 
        | "MathLink C Function" 
        | "LibraryLink C Function" 
        | "Method" 
        | "Symbol" | "Message" | "System Program" | "Widget" | "Screencast"),
            transform = If[buildmode==="Future", {"HTML","FunctionsInternalBuild.m"}, {"HTML","Functions.m"}];
            layoutfile = ToFileName[{layoutdir},"Functions.html"];
            (* Temporary so that CreateInputForm can show in error log *)
            Global`PageTitleForCreateInputForm = pageTitle;
        ,
        ("Program"), (* ref/program/wolframscript *)
            transform = {"HTML", "Functions.m"};
            layoutfile = ToFileName[{layoutdir}, "ManPage.html"];
        ,
        ("Format"),
            transform = {"HTML","Functions.m"};
            layoutfile = ToFileName[{layoutdir},"Formats.html"];
        ,
        ("Embedding Format"),
            transform = {"HTML","Functions.m"};
            layoutfile = ToFileName[{layoutdir},"Formats.html"];
        ,
        ("Device Connection"),
            transform = {"HTML","Functions.m"};
            layoutfile = ToFileName[{layoutdir},"Formats.html"];
        ,
        ("Interpreter"),
            transform = {"HTML","Functions.m"};
            layoutfile = ToFileName[{layoutdir},"Formats.html"];
        ,
        ("Service Connection"),
            transform = {"HTML","Services.m"};
            layoutfile = ToFileName[{layoutdir},"Services.html"];
        ,
        ("Example"),
            transform = {"HTML","ExamplePages.m"};
            layoutfile = ToFileName[{layoutdir},"ExamplePages.html"];
        ,
        ("HowTo"),
            transform = {"HTML","Functions.m"};
            layoutfile = ToFileName[{layoutdir},"HowTos.html"];
        ,
         ("Front End Object"),
            transform = {"HTML","Functions.m"};
            layoutfile = ToFileName[{layoutdir},"FrontEndObjects.html"];
        ,
        ("Indicator" ),
            transform = {"HTML","Functions.m"};
            layoutfile = ToFileName[{layoutdir},"Indicators.html"];
        ,
        ("Menu Item" ),
            transform = {"HTML","Functions.m"};
            layoutfile = ToFileName[{layoutdir},"MenuItems.html"];
        ,
        ("Character" | "Character Name"),
            transform = {"HTML","Functions.m"};
            layoutfile = ToFileName[{layoutdir},"Characters.html"];
        ,

        ("Application"),
            transform = {"HTML","Applications.m"};
            layoutfile = ToFileName[{layoutdir},"Applications.html"];
            pageTitle = Cases[nb, Cell[c_, "ObjectName"|"ObjectNameSmall", ___]:> c, Infinity];
            linkTrail = Join[linkTrail, {{"/language/Symbols/", "Symbols"}, {"/language/Symbols/System/", "System"}}];
            contextName = "Built-in <i>Mathematica</i> Object"; 
            (* add routine to literalize CounterBoxes before exporting to html*)
            nbObj = NotebookPut[nb];
            FrontEndExecute[{FrontEnd`NotebookDynamicToLiteral[nbObj]}];
            nb = NotebookGet@nbObj;
            NotebookClose[nbObj];
            (* end routine to literalize CounterBoxes *)
        ,
        ("Guide"),
            transform = {"HTML","Guides.m"};
            layoutfile = ToFileName[{layoutdir},"Guides.html"];
        ,
        ("Root Guide"),
            transform = {"HTML","Guides.m"};
            flag = "RootGuidePage";
            layoutfile = ToFileName[{layoutdir}, "RootGuide.html"];
        ,
        ("Guide Overview"),
            transform = {"HTML","Guides.m"};
            layoutfile = ToFileName[{layoutdir},"GuideOverview.html"];
            ,
        ("Standard Extra Packages"),
            transform = {"HTML","SpecialGuides.m"};
            layoutfile = ToFileName[{layoutdir},"SpecialGuides.html"];
        ,
        ("Note" | "Tutorial" | "Upgrade Information"), 
            layoutfile = ToFileName[{layoutdir},"Tutorials.html"];
            transform = {"HTML","Tutorials.m"};
        ,
        ("Overview"),
            transform = {"HTML","TutorialOverviews.m"};
            layoutfile = ToFileName[{layoutdir},"TutorialOverviews.html"];
        ,
        ("Guide"),
            transform = {"HTML","Guides.m"};
            layoutfile = ToFileName[{layoutdir},"Guides.html"];
        , 
        ("Note" | "Tutorial"),
            layoutfile = ToFileName[{layoutdir},"Tutorials.html"];
            transform = {"HTML","Tutorials.m"};
        ,

        "generic",
            transform = {"HTML","Generic.m"};
            layoutfile = ToFileName[{layoutdir},"Generic.html"];
            (* pageTitle = Cases[nb, Cell[c_, "ObjectName"|"ObjectNameSmall", ___]:> c, Infinity];  *)
            (* linkTrail = Join[linkTrail, {{"/language/Symbols/", "Symbols"}, {"/language/Symbols/System/", "System"}}];
            contextName = "Built-in <i>Mathematica</i> Object";  *)
            (* add routine to literalize CounterBoxes before exporting to html*)
            nbObj = NotebookPut[nb];
            FrontEndExecute[{FrontEnd`NotebookDynamicToLiteral[nbObj]}];
            nb = NotebookGet@nbObj;
            NotebookClose[nbObj];
            (* end routine to literalize CounterBoxes *)
        ,
        "alphabeticalpages",
            transform = {"HTML","Guides.m"};
            layoutfile = ToFileName[{layoutdir},"Guides.html"];
        ,

        _XMLTransform | _String | {__String},
            (* assume we're being passed a valid transform file *)
            (* all parameters will need to be passed via DefaultParameters *)
            transform = pageType
    ];

    BuildLog["   Replace various patterns"];
    
    (* Combine {Cell, Input, Deemphasis} into one style
       TODO: Allow Transmogrify templates to take multiple matcher arguments
    *)
    expr = expr /. Cell[c_, "Input", "Deemphasis", o___] :> Cell[ c, "InputDeemphasis", o];
    expr = expr /. Cell[c_, "Program", "InlineCode", o___] :> Cell[ c, "Program", o];

	expr = expr//. Cell[TextData[a_, b:(ButtonBox[__]|{__})], "RootMoreAbout", c___] :> 
Cell[b, "RootMoreAbout", c]; 
	expr = expr//. Cell[TextData[{a_, b:(ButtonBox[__]|{__})}], "RootMoreAbout", c___] :> 
Cell[b, "RootMoreAbout", c];

    (*   *)
    expr = expr /.
      Cell[c_, "NotesSection", o___]
        :> Cell[
        	If[Length[Cases[c, Cell[TextData[{a_, b_String}], "NotesFrameText"] :> b, Infinity]]===1,
        	Cases[c, Cell[TextData[{a_, b_String}], "NotesFrameText"] :> b, Infinity][[1]], Localized["NotesSectionText", language]] , "NotesSection", o];

    (* add output cell for Input followed by ExampleDelimiter --next 2 rules  *)
    expr = expr //.
      {pre___, c:Cell[_, "Input", ___],
          Cell[CellGroupData[{Cell[cc1_, "ExampleDelimiter", cc2___], cgd___}, oc_], cc___], post___}
        :> {pre, c, Cell[" ", "Spacer"],
            Cell[CellGroupData[{Cell[cc1, "ExampleDelimiter", cc2], cgd}, oc], cc], post};  

    expr = expr //.
      {pre___, 
          Cell[CellGroupData[{Cell[cc1_, "ExampleDelimiter", cc2___],
          c:Cell[_, "ExampleText", ___], Cell[a_, "Input", b___]}, oc_], cc___], 
          post___}
        :> {pre,
            Cell[CellGroupData[{Cell[cc1, "ExampleDelimiter", cc2], c, Cell[a, "Input", b], Cell[" ", "Spacer"]}, oc], cc], 
            post};  

    expr = expr //.
      {pre___, 
          Cell[CellGroupData[{Cell[cc1_, "ExampleDelimiter", cc2___], c:Cell[_, "ExampleText", ___]}, oc_], cc___], 
          post___}
        :> {pre,
            Cell[CellGroupData[{Cell[cc1, "ExampleDelimiter", cc2], c, Cell[" ", "Spacer"]}, oc], cc], 
            post};  

    (* Fix local page linking *)
    expr = expr /. 
      (*ButtonBox[con_, BaseStyle->"Hyperlink", Appearance ->__List, ButtonData->t_String/;StringTake[t, 5]=!="http:", o___] :> *)
      ButtonBox[con_, BaseStyle->"Hyperlink", Appearance ->__List, ButtonData->t_String/;StringFreeQ[t, "http://"], o___] :> 
      ButtonBox[con, BaseStyle->"Hyperlink", ButtonData->{URL["#"<>t], None}, o];

    expr = expr /. 
      (*ButtonBox[con_, BaseStyle->"Hyperlink", ButtonData->t_String/;StringTake[t, 5]=!="http:", o___]*)
      ButtonBox[con_, BaseStyle->"Hyperlink", ButtonData->t_String/;StringFreeQ[t, "http://"], o___]:> 
      ButtonBox[con, BaseStyle->"Hyperlink", ButtonData->{URL["#"<>t], None}, o];

    expr = expr /. 
      StyleBox[ButtonBox[str_String, o___], "MenuName"]
        :> StyleBox[ButtonBox[
            StringSplit[str,
              RegularExpression["\\s*\[FilledRightTriangle]\\s*"] -> 
                StyleBox[" \[FilledRightTriangle] ", "MenuNameDelimiter" ]],
            o], "MenuName"];

    (* Wrap FilledRightTriangle in StyleBox in order to decrease size on web *)
    expr = expr //.
      Cell[TextData[{a___, " \[EmptyVerySmallSquare] ", b___}], "RelatedLinks", o___]
        :> Cell[TextData[{a, StyleBox["*", "SeeAlsoBullet"], b}], "RelatedLinks", o];
    expr = expr //.
      Cell[TextData[{a___, " \[EmptyVerySmallSquare] ", b___}], "SeeAlso", o___]
        :> Cell[TextData[{a, StyleBox["*", "SeeAlsoBullet"], b}], "SeeAlso", o];

    expr = expr //.
      Cell[TextData[{a___, " \[EmptyVerySmallSquare] ", b___}], o___]
        :> Cell[TextData[{a, "\[NonBreakingSpace]",
            StyleBox["\[FilledVerySmallSquare]", "InlineSeparator"], " ", b}], o];

    (* Remove MSG second style in Messages -- TODO: need to recognize multiple styles *)
    expr = expr //.
      Cell[a_, "Message", b___, "MSG", c___]
        :> Cell[a, "Message", b, FontFamily->CurrentValue["PanelFontFamily"]];

   expr = expr //. 
     Cell[CellGroupData[{Cell[a___, "NotesSection", b___], c___, 
         Cell[d___, "Message", e___], f___}, g___], h___] :> 
         Cell[CellGroupData[{Cell[a, "NotesSection", b], c, Cell[{Cell[d, "Message", e]}, "MessageNote"], f}, g], h];

   expr = expr //. 
     Cell[CellGroupData[{Cell[a___, "ImportExportSection", b___], c___, 
         Cell[d___, "WeakDivider", e___], f___}, g___], h___] :> 
         Cell[CellGroupData[{Cell[a, "ImportExportSection", b], c, Cell[{Cell[d, "WeakDivider", e]}, "FormatNotes"], f}, g], h];

   expr = expr //. 
     Cell[CellGroupData[{Cell[a___, "NotebookInterfaceSection", b___], c___, 
         Cell[d___, "WeakDivider", e___], f___}, g___], h___] :> 
         Cell[CellGroupData[{Cell[a, "NotebookInterfaceSection", b], c, Cell[{Cell[d, "WeakDivider", e]}, "FormatNotes"], f}, g], h];

   expr = expr //. 
     Cell[CellGroupData[{Cell[a___, "ElementsSection", b___], c___, 
         Cell[d___, "WeakDivider", e___], f___}, g___], h___] :> 
         Cell[CellGroupData[{Cell[a, "ElementsSection", b], c, Cell[{Cell[d, "WeakDivider", e]}, "FormatNotes"], f}, g], h];

   expr = expr //. 
     Cell[CellGroupData[{Cell[a___, "OptionsSection", b___], c___, 
         Cell[d___, "WeakDivider", e___], f___}, g___], h___] :> 
         Cell[CellGroupData[{Cell[a, "OptionsSection", b], c, Cell[{Cell[d, "WeakDivider", e]}, "FormatNotes"], f}, g], h];



    expr = expr //.
      Cell[a_, "Print", b___, "PrintUsage", c___]
        :> Cell[a, "Print", c, 
          PageWidth -> 550,
          CellFrame->{{0, 0}, {0, 3}},
          CellFrameColor->RGBColor[0.947448, 0.91873, 0.880354], 
          Background->RGBColor[0.972732, 0.966506, 0.947112]
        ];

    expr = expr //.
      Cell[a_, "Input", b___, "Deemphasis", c___]
        :> Cell[a, "Deemphasis", c,  FontColor->GrayLevel[0.75]];

    expr = expr //.
      Cell[a_, "DefinitionBox", b___, "TableHeaderLine", c___]
        :> Cell[a, "DefinitionBox", c];
    expr = expr //.
      Cell[TextData[a_, b:(ButtonBox[__]|{__})], "RootMoreAbout", c___]
        :> Cell[b, "RootMoreAbout", c]; 

    expr = expr //.
      Cell[a_, "Input", b___, "ConversionLockdown", c___] :> Cell[a, "Input", c];
    expr = expr //.
      Cell[a_, "Output", b___, "ConversionLockdown", c___] :> Cell[a, "Output", c];
    expr = expr //.
      Cell[a_, "MathCaption", b___, "ConversionLockdown", c___] :> Cell[a, "MathCaption", c];
    expr = expr //.
      Cell[a_, "Print", b___, "ConversionLockdown", c___] :> Cell[a, "Print", c];
      
    expr = expr //.
      Cell[__, CellTags->"UnderDevelopment", ___] :> Cell["U N D E R  D E V E L O P M E N T", "UnderDevelopment"];
    
    expr = expr //.
      Cell[TextData[{
        Cell[a___, "ObjectName", b___],
        Cell[BoxData[InterpretationBox[___,Spacer[10]]]],c___}]] :> 
             Cell[TextData[{Cell[a, "ObjectName", b],Cell["[[UNFINISHED DRAFT]]", "UnfinishedDraft"]}]];

    expr = expr //.
      Cell[TextData[{
        Cell[a___, "ObjectName", b___],
        ___, Cell[BoxData[InterpretationBox[___,Spacer[10]]]], ___
      }], c___] :> Cell[TextData[{Cell[a, "ObjectName", b]}], c];

    expr = expr //. 
      Cell[TextData[{
        Cell[a_, "ObjectName", b___],
        Cell[BoxData[InterpretationBox[___,Spacer[8]]]],
        Cell[__, "ObjectNameTranslation"], 
        c___
      }], d___] :> Cell[TextData[{Cell[a, "ObjectName", b], c}], d];

    expr = expr //.
      Cell[TextData[{
    Cell[a___, "GuideTitle", b___],
    Cell[BoxData[
       InterpretationBox[
         c___,
         Spacer[10]]]], d___
   }], "GuideTitleGrid"] :> 
             Cell[BoxData[GridBox[{{Cell[TextData[{Cell[a, "GuideTitle", b],Cell["[[UNFINISHED DRAFT]]", "UnfinishedDraft"]}]]}}]], "GuideTitleGrid"];
    
    expr = expr //.
      Cell[a__, CellTags->"TextAnnotation", b___] :> Cell[a, b];
      
    expr = expr /. Cell[TextData[{a_String, StyleBox[b_String/;StringMatchQ["("~~_~~")", b], "OperatorCharacter"]}], "ObjectName",c___]:>
      Cell[TextData[{a, "(", StyleBox[b, "OperatorCharacter"], ")"}], "ObjectName", c];
    
    expr = expr //.  StyleBox[FormBox[x_String/;StringMatchQ[x, RegularExpression["[0-9]+"]], TraditionalForm], "TR"]:>StyleBox["1", "TR"];
 
   (* expr = expr /. Cell[TextData[{a_String, (*StyleBox[b_, "OperatorCharacter"]*)StyleBox[b__/;StringMatchQ[b,"("~~__~~")"], "OperatorCharacter"]}], "ObjectName",c___]:>
      Cell[TextData[{a, StyleBox["(", "OperatorCharacterText"], StyleBox[StringReplace[b, {"("->"",")"->""}], "OperatorCharacter"], StyleBox[")", "OperatorCharacterText"]}], "ObjectName", c];*)

    (*Deal with Input/LegacyInput in Compatibility pages *)
    expr = expr //.
      Cell[a_, "Input", b___, "LegacyInput", c___]
        :> Cell[a, "LegacyInput", b,
          PageWidth->550,
          CellFrame->True,
          CellMargins->{{28, 4}, {3, 8}},
          CellFrame->{{1, 1}, {1, 1}},
          CellFrameMargins->{{12, 12}, {12, 12}},
          CellFrameColor->RGBColor[0.55, 0.650000000000003, 0.55],
          CellFrameLabels->{{
            Cell[TextData[{
              StyleBox["Version", FontSize -> 9], "\n  5.2"}], "LegacyInputLabel"], None}, {
            None, None}},
          FontColor->GrayLevel[0.5]];
          
     expr = expr //. 
       Cell[a_, "Input", b___, "LegacyInputV7", c___] 
         :> Cell[a, "LegacyInputV7", b,
           PageWidth->550,
           CellFrame->True,
           CellMargins->{{28, 4}, {3, 8}},
           CellFrame->{{1, 1}, {1, 1}},
           CellFrameMargins->{{12, 12}, {12, 12}},
           CellFrameColor->RGBColor[0.55, 0.650000000000003, 0.55],
           CellFrameLabels->{{
             Cell[
               TextData[{
                StyleBox["Version", FontSize -> 9], "\n  6.0"}], "LegacyInputLabel"], None}, {
             None, None}}, 
           FontColor->GrayLevel[0.5]]; 

     expr = expr //. 
       Cell[a_, "Input", b___, "LegacyInputV8", c___] 
         :> Cell[a, "LegacyInputV8", b,
           PageWidth->550,
           CellFrame->True,
           CellMargins->{{28, 4}, {3, 8}},
           CellFrame->{{1, 1}, {1, 1}},
           CellFrameMargins->{{12, 12}, {12, 12}},
           CellFrameColor->RGBColor[0.55, 0.650000000000003, 0.55],
           CellFrameLabels->{{
             Cell[
               TextData[{
                StyleBox["Version", FontSize -> 9], "\n  7.0"}], "LegacyInputLabel"], None}, {
             None, None}}, 
           FontColor->GrayLevel[0.5]]; 
           
       expr = expr //. 
       Cell[a_, "Input", b___, "LegacyInputV9", c___] 
         :> Cell[a, "LegacyInputV9", b,
           PageWidth->550,
           CellFrame->True,
           CellMargins->{{28, 4}, {3, 8}},
           CellFrame->{{1, 1}, {1, 1}},
           CellFrameMargins->{{12, 12}, {12, 12}},
           CellFrameColor->RGBColor[0.55, 0.650000000000003, 0.55],
           CellFrameLabels->{{
             Cell[
               TextData[{
                StyleBox["Version", FontSize -> 9], "\n  8.0"}], "LegacyInputLabel"], None}, {
             None, None}}, 
           FontColor->GrayLevel[0.5]]; 

    expr = expr //.
      Cell[a_, "UpgradeDetail" | "Input", b___, "LegacyInputError", c___]
        :> Cell[a, "LegacyInputError", b,
          PageWidth-> 500,
          CellFrame->True,
          CellMargins->{{28, 4}, {3, 8}},
          Evaluatable->False,
          GeneratedCellStyles->{"Output"->"LegacyOutput"},
          ShowCellLabel->False,
          CellFrameMargins->{{12, 12}, {12, 12}},
          CellFrameColor->RGBColor[0.55, 0.650000000000003, 0.55],
          CellFrameLabels->{{
            Cell[TextData[{
              StyleBox["Version", FontSize -> 9], "\n  5.2"}], "LegacyInputLabel"], 
              Cell["Error", "LegacyInputLabel"]}, {None, None}},
          ShowAutoStyles->False,
          MenuPosition->800,
          MenuCommandKey->"8",
          FontColor->GrayLevel[0.5],
          Background->RGBColor[1, 0.9, 0.9]]; 

    expr = expr //.
      Cell[a_, "UpgradeDetailLabel", b___, "LegacyInputError", c___]
        :> Cell[a, "LegacyInputError", b,
          PageWidth-> 500,
          CellFrame->True,
          CellMargins->{{28, 4}, {3, 8}},
          Evaluatable->False,
          GeneratedCellStyles->{"Output"->"LegacyOutput"},
          ShowCellLabel->False,
          CellFrameMargins->{{12, 12}, {12, 12}},
          CellFrameColor->RGBColor[0.55, 0.650000000000003, 0.55],
          CellFrameLabels->{{
            Cell[TextData[{
              StyleBox["Version", FontSize -> 9], "\n  5.2"}], "LegacyInputLabel"], 
              Cell["Error", "LegacyInputLabel"]}, {None, None}},
          ShowAutoStyles->False,
          MenuPosition->800,
          MenuCommandKey->"8",
          FontColor->GrayLevel[0.5],
          Background->RGBColor[1, 0.9, 0.9]]; 
          
     expr = expr//. 
       Cell[a_, "UpgradeDetail" | "UpgradeDetailLabel" | "Input", b___, "LegacyInputErrorV7", c___] 
         :> Cell[a, "LegacyInputErrorV7", b,
          PageWidth-> 500,
          CellFrame->True,
          CellMargins->{{28, 4}, {3, 8}},
          Evaluatable->False,
          GeneratedCellStyles->{"Output"->"LegacyOutput"},
          ShowCellLabel->False,
          CellFrameMargins->{{12, 12}, {12, 12}},
          CellFrameColor->RGBColor[0.55, 0.650000000000003, 0.55],
          CellFrameLabels->{{
          Cell[
          TextData[{
                 StyleBox["Version", FontSize -> 9], "\n  6.0"}], "LegacyInputLabel", FontColor->RGBColor[0.3000000000000003, 0.6000000000000006, 0.5], Background->None], 
              Cell["returns\nunevaluated", "LegacyInputLabel", FontColor->RGBColor[0.3000000000000003, 0.6000000000000006, 0.5], Background->None]}, {None, None}},
           ShowAutoStyles->False,
           MenuPosition->800,
           MenuCommandKey->"8",
           FontColor->GrayLevel[0.5],
           Background->RGBColor[1, 0.9, 0.91]]; 

     expr = expr//. 
       Cell[a_, "UpgradeDetail" | "UpgradeDetailLabel" | "Input", b___, "LegacyInputErrorV8", c___] 
         :> Cell[a, "LegacyInputErrorV8", b,
          PageWidth-> 500,
          CellFrame->True,
          CellMargins->{{28, 4}, {3, 8}},
          Evaluatable->False,
          GeneratedCellStyles->{"Output"->"LegacyOutput"},
          ShowCellLabel->False,
          CellFrameMargins->{{12, 12}, {12, 12}},
          CellFrameColor->RGBColor[0.55, 0.650000000000003, 0.55],
          CellFrameLabels->{{
          Cell[
          TextData[{
                 StyleBox["Version", FontSize -> 9], "\n  7.0"}], "LegacyInputLabel", FontColor->RGBColor[0.3000000000000003, 0.6000000000000006, 0.5], Background->None], 
              Cell["returns\nunevaluated", "LegacyInputLabel", FontColor->RGBColor[0.3000000000000003, 0.6000000000000006, 0.5], Background->None]}, {None, None}},
           ShowAutoStyles->False,
           MenuPosition->800,
           MenuCommandKey->"8",
           FontColor->GrayLevel[0.5],
           Background->RGBColor[1, 0.9, 0.91]]; 
           
      expr = expr//. 
       Cell[a_, "UpgradeDetail" | "UpgradeDetailLabel" | "Input", b___, "LegacyInputErrorV9", c___] 
         :> Cell[a, "LegacyInputErrorV9", b,
          PageWidth-> 500,
          CellFrame->True,
          CellMargins->{{28, 4}, {3, 8}},
          Evaluatable->False,
          GeneratedCellStyles->{"Output"->"LegacyOutput"},
          ShowCellLabel->False,
          CellFrameMargins->{{12, 12}, {12, 12}},
          CellFrameColor->RGBColor[0.55, 0.650000000000003, 0.55],
          CellFrameLabels->{{
          Cell[
          TextData[{
                 StyleBox["Version", FontSize -> 9], "\n  8.0"}], "LegacyInputLabel", FontColor->RGBColor[0.3000000000000003, 0.6000000000000006, 0.5], Background->None], 
              Cell["returns\nunevaluated", "LegacyInputLabel", FontColor->RGBColor[0.3000000000000003, 0.6000000000000006, 0.5], Background->None]}, {None, None}},
           ShowAutoStyles->False,
           MenuPosition->800,
           MenuCommandKey->"8",
           FontColor->GrayLevel[0.5],
           Background->RGBColor[1, 0.9, 0.91]]; 

    (*  *)
    (*expr = expr /.
      ButtonBox[con_, a___, ButtonFunction~(Rule|RuleDelayed)~f_, b___]
        :> ButtonBox[con, a, ButtonFunction->"DEADFUNCTION", b];*)
    expr = expr /.
      ButtonBox[con_, a___, ButtonFunction~(Rule|RuleDelayed)~f_, b___]
        :> ButtonBox[con, a, b];

    expr = expr /.
      ButtonBox[con_,
        BaseStyle ~(Rule|RuleDelayed)~ None,
        ButtonFunction ~(Rule|RuleDelayed)~ "DEADFUNCTION",
        ButtonFrame ~(Rule|RuleDelayed)~ "None",
        Evaluator ~(Rule|RuleDelayed)~ None,
        Method ~(Rule|RuleDelayed)~ "Preemptive"]
      :>con; 
      
    expr = expr /.
      ButtonBox[con_,
        BaseStyle ~(Rule|RuleDelayed)~ None,
        _Rule,
        Evaluator ~(Rule|RuleDelayed)~ None,
        Method ~(Rule|RuleDelayed)~ "Preemptive",
        ButtonFunction ~(Rule|RuleDelayed)~ "DEADFUNCTION"]
      :>con;
      
    expr=expr/.ButtonBox[aa___, BaseStyle->{"Link", "GuideFunctionsSubsection" | "MenuName" | "FullListing"}, bb___]
      :> ButtonBox[aa, BaseStyle -> "Link", bb];
      
     expr = expr/.{Cell[BoxData[GridBox[{
         {a___, Cell[b___]}
        }]], "ObsolescenceNote", c___] :> Cell[b, "ObsolescenceNote"]};
     
     expr = expr/.{Cell[BoxData[GridBox[{
         {a___, Cell[b___]}
        }]], "AwaitingReviewNote", c___] :> Cell[b, "AwaitingReviewNote"]};

     expr = expr /. {Cell[TextData[{
      Cell[BoxData[
        InterpretationBox[
          StyleBox[
            GraphicsBox[{},
              BaselinePosition -> Baseline,
              ImageSize -> {6, 0}],
            CacheGraphics -> False],
          Spacer[6]]]],
          w___
     }], "FunctionEssaySection",
      WholeCellGroupOpener -> True] 
      :> Cell[TextData[{w}], "FunctionEssaySection", WholeCellGroupOpener -> True]};

    (* bug 309600 *)
    expr = expr /. Cell[BoxData[""], "Input"] -> Sequence[];

    (* remove a large empty image from ref/format pages *)
    expr = expr /. Cell[TextData[{
      a:Cell[_, "ObjectNameAlt", ___],
      Cell[BoxData[InterpretationBox[_, Spacer[8]]]],
      b:Cell[_, "ObjectNameTranslation"]
    }]] :> Cell[TextData[{a, b}]];
    
    (* https://jira.wolfram.com/jira/browse/WLDOCS-936 *)
    expr = Block[{pos, usage},
      pos = Position[expr, Cell[_, "Usage", ___]];
      usage = Extract[expr, pos];
      usage = usage /. Cell[BoxData[""], "Input", ___] -> Sequence[];
      ReplacePart[expr, pos -> usage]
    ];

    expr = expr /. Cell[BoxData[RowBox[list_List]], "Echo", rest___] :> Cell[BoxData[RowBox[Flatten@{
      TagBox["\[RightGuillemet]", "EchoLabel", TagBoxOptions -> {BaseStyle -> Large}],
       " ",
       list
     }]], "Echo", rest];

    expr = expr /. Cell[BoxData[s_String], "Echo", rest___] :> Cell[BoxData[RowBox[{
      TagBox["\[RightGuillemet]", "EchoLabel", TagBoxOptions -> {BaseStyle -> Large}],
       " ",
       s
     }]], "Echo", rest];

    (* wolframscript, the new Program entity - v11, July 2016 *)

    expr = Block[{pos, synopsis},
      pos = Position[expr, Cell[CellGroupData[{Cell["SYNOPSIS", "ProgramSection", ___], Longest[Cell[_, "ProgramCodeText", ___]..], ___}, ___], ___], Infinity];
      synopsis = Extract[expr, pos];
      synopsis = synopsis /. a:Cell[_, "ProgramCodeText", ___] :> Cell[TextData[{a}], "ProgramNotes"];
      ReplacePart[expr, pos -> First[synopsis]]
    ];

    expr = Block[{pos, target},
      pos = Position[expr, Cell[CellGroupData[{Cell["DESCRIPTION", "ProgramSection", ___], ___}, ___], ___], Infinity];
      target = Extract[expr, pos];
      target = target /. Cell[first:_, "ProgramNotes", rest:___] :> Cell[first, "ProgramNotesDescriptionSection", rest];
      ReplacePart[expr, pos -> First[target]]
    ];

    expr = Block[{pos, target},
      pos = Position[expr, Cell[CellGroupData[{Cell["EXAMPLES", "ProgramSection", ___], ___}, ___], ___], Infinity];
      target = Extract[expr, pos];
      target = target /. Cell[first:_, "ProgramNotes", rest:___] :> Cell[first, "ProgramNotesExamplesSection", rest];
      ReplacePart[expr, pos -> First[target]]
    ];

    expr = Block[{pos, target},
      pos = Position[expr, Cell[CellGroupData[{Cell["DETAILS", "ProgramSection", ___], ___}, ___], ___], Infinity];
      target = Extract[expr, pos];
      target = target /. Cell[first:_, "ProgramNotes", rest:___] :> Cell[first, "ProgramNotesDetailsSection", rest];
      ReplacePart[expr, pos -> First[target]]
    ];

    expr = expr /. Cell[s_String, "ProgramCodeText", rest___] :> StyleBox[s, "ProgramCodeText", rest];

    (* the *Standalone transforms insert the right <ul> element around these
    tables because wolframscript's tables are not already contained in an
    environment that inserts the <ul>.  i.e., these are standalone tables,
    outside the usual table-containing structures. *)

    If[Positive @ Count[expr, "uri" -> "ref/program/wolframscript", Infinity],
      expr = expr /. {
        Cell[a_, "2ColumnTable", rest___] :> Cell[a, "2ColumnTableStandalone", rest],
        Cell[a_, "3ColumnTable", rest___] :> Cell[a, "3ColumnTableStandalone", rest],
        Cell[a_, "1ColumnTableMod", rest___] :> Cell[a, "1ColumnTableModStandalone", rest],
        Cell[a_, "2ColumnTableMod", rest___] :> Cell[a, "2ColumnTableModStandalone", rest],
        Cell[a_, "3ColumnTableMod", rest___] :> Cell[a, "3ColumnTableModStandalone", rest]
        Cell[a_, "3ColumnIllustratedTableMod", rest___] :> Cell[a, "3ColumnIllustratedTableModStandalone", rest]
      }
    ];

(*********************************************************************
here endeth our last-chance expression munging before running
Transmogrify
*********************************************************************)

     modifiedNotesQ = If[Length[Flatten[Cases[expr, Cell[CellGroupData[{Cell[___, "NotesSection", ___], b___}, ___], ___] :> Cases[{b}, "ModificationHighlight", Infinity], Infinity]]] > 0, "True", "False"];
     
    (* Get list of anchor bar elements *)
    anchorBarList = {};
   
    featuredexampledata = 
    If[entityType === "Guide",
    	Cases[Cases[Cases[expr, TaggingRules~(Rule | RuleDelayed)~_], "ExamplePageLinks"~(Rule | RuleDelayed)~a_ :> a, Infinity], {_String, _String, _String}, Infinity],
    {}];
   
    BuildLog["   Assemble property list"];

     (* If[entityType === "Guide",
      (* capture Tutorial function list *)
      guideFunctionList = GetGuideFunctionList[expr];
      guideMoreAboutList = GetGuideMoreAboutSectionList[expr];
      guideTutorialsList = GetGuideTutorialsSectionList[expr];
    ];*)

    (* If[entityType === "Tutorial",
      (* capture Tutorial function list *)
      tutorialFunctionList = GetTutorialFunctionList[expr];
      tutorialMoreAboutList = GetTutorialMoreAboutSectionList[expr];
      relatedTutorialsList = GetRelatedTutorialsSectionList[expr];
    ];*)
    
    (*seeAlsoFunctionList = GetSeeAlsoSectionList[expr];
    moreAboutGuideList = GetMoreAboutSectionList[expr];
    exampleSectionList = {};
    tutorialList = GetTutorialsSectionList[expr];*)

    exampleSectionList = {};

	dashboard = 
	  If[showDashboardQ, 
		"
		<div><!--#include virtual=\"/mathematicaIncludes/feedback.html\"--></div>
		
  <!--#if expr='\${is_development} = 1' -->
  <!--#else -->
  <!--#endif -->
		",
		"<div class=\"dashboard\" />"];

    (* determine relative location of css to html page *)
    (* this changes depending on page (e.g ref/Sin v. ref/messages/func/tag) *)
    relativeBase = StringJoin @@ Flatten@{ Table["../", { (Length@StringCases[uri, "/"])-1 }] };
    BuildLog["relativeBase: "<>relativeBase];
    expr = DeleteCases[expr, Rule["LinkTrails", ___], Infinity];
    (*linkTrail = If[TrueQ@includeLinkTrails, If[Length[Cases[expr, Cell[___,"LinkTrail",___], Infinity]]>0, Cell[CellGroupData[Cases[expr, Cell[___,"LinkTrail",___], Infinity]]], None], None];*)
    linkTrail = If[TrueQ@includeLinkTrails, linkTrail, None];
    BuildLog["linkTrail: "<>ToString[linkTrail]];
    
    htmlFooter = "" (* "<script language='javascript' type='text/javascript' src='" <> relativeBase <> "javascript/common.js' />\n"<>
        "</body>\n</html>"*);

    If[completeHTMLQ===True,
      htmlBase = relativeBase <> "images";
      urlBaseRemote = If[TrueQ[wsmlinkQ], If[TrueQ[wsmlibQ], "wsm:///sma/", "wsm:///ref/"], "http://reference.wolfram.com/language/"];
      
      htmlHeader = "<!DOCTYPE html PUBLIC \"-//W3C//DTD XHTML 1.0 Transitional//EN\"
        \"http://www.w3.org/TR/xhtml1/DTD/xhtml1-transitional.dtd\">\n"<>
        "<html xmlns='http://www.w3.org/1999/xhtml' xml:lang='"<>langext<>"' lang='"<>langext<>"'>\n"<>
        "<head>\n<title>"<>
        Which[pageTitleModifier==="", 
        	pageTitle, 
        	pageTitleModifier==="XXXX", 
        	pageTitle, 
        	True,
        	pageTitle<>" "<>pageTitleModifier]<>"</title>\n"<>
        "<meta name='keywords' content='"<>keywordList<>"' />\n" <>
        "<meta name='description' content='" <>CleanSummaryForWeb[summaryText]<> "' />\n" <>
        "<link rel='stylesheet' type='text/css' href='" <> relativeBase <> "css/functions.css' />\n"<>
If[pacletQ,
  "<link rel='stylesheet' type='text/css' href='" <> relativeBase <> "css/paclets.css' />\n",
  ""]<>
"<script src='" <> relativeBase <> "javascript/menu.js' type='text/javascript'></script>\n"<>
"<script src='" <> relativeBase <> "javascript/common.js' type='text/javascript'></script>\n"<>
"<script src='" <> relativeBase <> "javascript/prototype-1.4.0.js' type='text/javascript'></script>\n"<>
"<script src='" <> relativeBase <> "javascript/search.js' type='text/javascript'></script>\n"<>
"<script src='" <> relativeBase <> "javascript/jquery.js' type='text/javascript'></script>\n"<>
"<script src='" <> relativeBase <> "javascript/jquery-query.js' type='text/javascript'></script>\n"<>
"<script src='" <> relativeBase <> "javascript/jquery.dimensions.min.js' type='text/javascript'></script>\n"<>
"<script src='" <> relativeBase <> "javascript/jquery.mousewheel.min.js' type='text/javascript'></script>\n"<>
"<script src='" <> relativeBase <> "javascript/jScrollPane.js' type='text/javascript'></script>\n"<>
"<script src='" <> relativeBase <> "javascript/FunctionBrowser.js' type='text/javascript'></script>\n"<>
"<script src='" <> relativeBase <> "javascript/VirtualBook.js' type='text/javascript'></script>\n"<>
"<script src='" <> relativeBase <> "javascript/Navigator.js' type='text/javascript'></script>\n"<>
"<script language='JavaScript' type='text/javascript'>baselang='"<>StringReplace[FileBaseName[file], ".html"->""]<>"';</script>\n"<>
        
        (*"<script src='" <> relativeBase <> "javascript/common.js' type='text/javascript'> </script>"<>"\n"<>*)
        "</head>\n<body style='margin:35px 0px 0px 0px; background:#ffffff'>\n"<>
        "<!-- saved from url=(0014)about:internet -->";
      (*htmlFooter = "</body>\n</html>";*)
      
      langext = "";
     ,
      htmlBase = "";
      langext = "." <> langext;
    ];

    properties = 
      Join[DeleteCases[
        {"layoutfile"->layoutfile, 
         "title" -> pageTitle, 
         "modifiednotesQ"->modifiedNotesQ,
         "titlemodifier"->pageTitleModifier, 
         "contextName" -> contextName, 
         "buildmode" -> buildmode,
         "url" -> uri, 
         "linkTrail" -> linkTrail, 
         "keywords" -> keywordList, 
         "flag" -> flag,
         "language" -> language, 
         "seeAlsoFunctions" -> seeAlsoFunctionList, 
         "rootCaptions" -> rootCaptions,
         "moreAboutGuides" -> moreAboutGuideList, 
         "pacletName" -> pacletName,
         "exampleSections" -> exampleSectionList, 
         "tutorials" -> tutorialList,
         "urlBaseRemote" -> urlBaseRemote, 
         "entityType" -> entityType, 
         "tutorialcollection"->tutorialcollectionlinks, 
         "specialKeywords" -> specialKeywords, 
         "tutorialFunctions" -> tutorialFunctionList, 
         "synonyms" -> synonymList, "guideFunctions" -> guideFunctionList,
         "tutorialMoreAbout" -> tutorialMoreAboutList, 
         "CompleteHTMLQ" -> completeHTMLQ, 
         "guideMoreAbout" -> guideMoreAboutList,
         "BuildNumber" -> buildNumber, "HistoryData"->({"New", "Modified", "Obsolete", "Excised"} /. history),
         "IsManipulate" -> isManipulate, "RelatedSearches" -> relatedSearches, "Cache" -> cache, "CacheLocation" -> cachelocation, "htmlBase" -> htmlBase, "summary" -> summaryText,
         "relatedTutorials" -> relatedTutorialsList, "guideTutorials" -> guideTutorialsList, "htmlHeader" -> htmlHeader, 
         "htmlFooter" -> htmlFooter, "anchorBar" -> anchorBarList, "NotFullySupported" -> notfullysupported, "featuredexampledata"->featuredexampledata, "dashboard" -> dashboard, "IsTMJQ"->isTMJQ, "IsSolutionsQ"->isSolutionsQ, 
         "searchSSI" -> searchSSI, "footerSSI" -> footerSSI, "languageLinkSSI" -> languageLinkSSI, "copyrightSSI" -> copyrightSSI},
        (Alternatives @@ First /@ params) -> _, Infinity], params];

    (* if we just want to see how the conversions _would_ happen,
       just go ahead and do it before notebook creation and manipulation.
       this allows us to do tests like mogtest[sometroublingExpression]
       to debug the XMLTransform/Transmogrify[] *)
    If[ TrueQ[test] || outfile===None,
      Return[If[ test,
        TransmogrifyString[##],
        Transmogrify[##]
      ]&[expr, transform, DefaultParameters->properties, opts]]
    ];
    
    (* Change WindowSize to a width for nicer line wrapping in gifs *)
    (* if windowsize with is changed here, also change it in the local variables in Transmogrify.m:transmogrifyImage *)
    (* https://jira.wolfram.com/jira/browse/WLDOCS-1023 - 550 is the largest WindowSize that still forces a linebreak in NestList Out[2] *)
    expr = expr /. Rule[WindowSize, {_, _}] :> WindowSize -> {550, 700};
   
    BuildLog["***** linkTrail: "<>ToString[linkTrail]];

    BuildLog["   Send expr off to Transmogrify"];

    CacheStylesheet[nb];

    (* now call Transmogrify with the appropriate parameters set.
       note that language extension by default goes before the .html *)
       
    r = PubsEvaluateWithFE@Transmogrify[
      If[StringQ[langext],
        StringReplace[outfile,RegularExpression["(?(?<!\\.\\w\\w)|(?<=\\.tm))(\\.html)"]->langext<>"$1"],
        outfile
      ],
      expr, transform, 
      DefaultParameters -> properties,
      Sequence@@DeleteCases[{opts}, DefaultParameters~(Rule|RuleDelayed)~_]
    ];
    BuildLog["** Leaving ExportWebPage **"];

    If[Head@r =!= String, First@r, r]

  ,
  (* fail expression for Check *)
  
  errMessageList = DocumentationBuild`Utils`Private`formatMessageList@$MessageList;
  BuildLog[StringJoin["   MESSAGES: ", errMessageList]];
  If[Head@r =!= String, First@r, r]]
]
(* End: ExportWebPage *)


(******************************************************************************)
(** ExportNavigatorWebPage **)

Options[ExportNavigatorWebPage] := {
  "BaseURL" -> "http://reference.wolfram.com/language/",
  "ImageBaseURL" -> "http://reference.devel.wolfram.com/mathematicaImages/"
};

ExportNavigatorWebPage[outfile_String, nb_String, opts___?OptionQ] :=
  ExportNavigatorWebPage[outfile, Import[nb], opts];

ExportNavigatorWebPage[outfile_String, nb_Notebook, opts___?OptionQ] :=
  Module[{base, imgbase}, Block[{groupid = 0},
    base = "BaseURL" /. {opts} /. Options[ExportNavigatorWebPage];
    imgbase = "ImageBaseURL" /. {opts} /. Options[ExportNavigatorWebPage];
    Export[outfile,
      XMLElement["html", {}, {
        XMLElement["head", {}, {
          XMLElement["link", {
            "rel" -> "stylesheet", "type" -> "text/css",
            "href" -> "/css/Navigator.css"
          }, {}],
          XMLElement["script", {
            "type" -> "text/javascript",
            "language" -> "javascript",
            "src" -> "/javascript/jquery.js"
          }, {" "}],
          XMLElement["script", {
            "type" -> "text/javascript",
            "language" -> "javascript",
            "src" -> "/javascript/Navigator.js"
          }, {" "}]
        }],
        XMLElement["body", {}, {
          XMLElement["div", {"id" -> "TabBar"}, {
            XMLElement["table", {}, {XMLElement["tr", {}, {
              XMLElement["td", {}, {
                XMLElement["div",
                  {"id" -> "FunctionNavigatorTab", "class" -> "Tab TabActive"},
                  {
                    XMLElement["a", {
                        "id" -> "FunctionNavigatorLink",
                        "href" -> "javascript:nav_tab(\"FunctionNavigator\")"
                      },
                      {"Function Navigator"}]
              }] }],
              XMLElement["td", {}, {
                XMLElement["div",
                  {"id" -> "VirtualBookTab", "class" -> "Tab"},
                  {
                    XMLElement["a", {
                        "id" -> "VirtualBookLink",
                        "href" -> "javascript:nav_tab(\"VirtualBook\")"
                      },
                      {"Virtual Book"}]
              }] }]
            }] }],
            XMLElement["div", {"id" -> "TabBottom"}, {
              XML`RawXML["&nbsp;"]
            }]
          }],
          XMLElement["div", {"id" -> "FunctionNavigator"},
            Map[ExportNavigatorWebPage[outfile, #, opts]&,
              Cases[nb,
                Cell[CellGroupData[{Cell[_, "Level 1", ___], ___}, ___], ___],
                {2}]
          ]],
          XMLElement["div", {"id" -> "VirtualBook"},
            Map[ExportNavigatorWebPage[outfile, #, opts]&,
              Cases[nb,
                Cell[CellGroupData[{Cell[_, "TutorialLevel 1", ___], ___}, ___], ___],
                {2}]
          ]]
        }]
      }],
    "XML"];
  ]];

ExportNavigatorWebPage[
    outfile_String,
    Cell[CellGroupData[{h_Cell, r___Cell}, ___], ___],
    opts___?OptionQ] :=
  Module[{base, imgbase, bbox, str, tag, class},
    base = "BaseURL" /. {opts} /. Options[ExportNavigatorWebPage];
    imgbase = "ImageBaseURL" /. {opts} /. Options[ExportNavigatorWebPage];
    groupid++;
    bbox = First@Cases[h, _ButtonBox, Infinity, 1];
    tag = First@Cases[h, (CellTags -> t_) :> t];
    If[StringLength[tag] > 7 && StringTake[tag, 7] == "paclet:",
      tag = StringDrop[tag, 7]];
    If[tag == "guide/NewIn60AlphabeticalListing",
      str = "New In 6.0",
    (* Else *)
      str = Transmogrify`ConvertToString[bbox];
      str = StringReplace[str, {
        RegularExpression["^\\s*\"(.*)\"\\s*$"] :> "$1"
      }];
    ];
    class = StringReplace[h[[2]], {" " -> "", "Tutorial" -> ""}];
    Module[{id = groupid},
      XMLElement["div", {
          "class" -> class,
          "id" -> ToString[id]
        }, Flatten@{
          XMLElement["div", {"class" -> "Head"}, {
            XMLElement["span", {"class" -> "Dingbat"}, {
              XMLElement["a", {"href" ->
                  "javascript:nav_toggle(\"" <> ToString[id] <> "\")"}, {
                XMLElement["img", {"src" -> (imgbase <> "closedGroup.gif")}, {}]
              }]
            }],
            XMLElement["a", {"href" ->
                ("javascript:nav_open(\"" <> base <> tag <> ".html\")") },
              {str}]
          }],
          If[class == "Level1",
            (Export[
              ToFileName[{DirectoryName[outfile], "HTMLFiles"}, ToString[id] <> ".txt"],
              #, "XML"];
            ); {},
            #          
          ]&[
            XMLElement["div", {"class" -> "Content"},
              Map[ExportNavigatorWebPage[outfile, #, opts]&, {r}]]
          ]
      }]
    ]
  ];

ExportNavigatorWebPage[
    outfile_String,
    Cell[_, _String, "Delimiter", ___],
    opts___?OptionQ] :=
  XMLElement["div", {"class" -> "Delimiter"}, {
    XMLElement["hr", {"class" -> "Delimiter"}, {}]
  }];

ExportNavigatorWebPage[
    outfile_String,
    expr:Cell[_, _String, ___],
    opts___?OptionQ] :=
  Module[{base, imgbase,bbox, str, tag, class},
    base = "BaseURL" /. {opts} /. Options[ExportNavigatorWebPage];
    imgbase = "ImageBaseURL" /. {opts} /. Options[ExportNavigatorWebPage];
    bbox = First@Cases[expr, _ButtonBox, Infinity, 1];
    str = Transmogrify`ConvertToString[bbox];
    str = StringReplace[str, {
      RegularExpression["^\\s*\"(.*)\"\\s*$"] :> "$1"
    }];
    tag = First@Cases[expr, (CellTags -> t_) :> t];
    If[StringLength[tag] > 7 && StringTake[tag, 7] == "paclet:",
      tag = StringDrop[tag, 7]];
    class = StringReplace[expr[[2]], " " -> ""];
    XMLElement["div", {"class" -> class}, {
      XMLElement["span", {"class" -> "Dingbat"}, {XML`RawXML["&bull; "]}],
      XMLElement["a", {"href" ->
          ("javascript:nav_open(\"" <> base <> tag <> ".html\")") },
        {str}]
    }]
  ];

ExportNavigatorWebPage[c__] := {};


GetFutureLandingContent[dir_String] :=
  Module[{manifest, res, outExpr1 = outExpr2 = {}, logstream, increment, count = 0},
  	
    Print["Finding the manifest"];
    Print[dir];
    manifest = FileNames["*.nb", dir, Infinity];
    increment = Floor[N[Length@manifest / 100]];

    Print["Processing the manifest, which involves finding the flag of each notebook"];
     res = Reap[Module[{nb, nbExpr, flag},
      (* poor man's progress meter *)
      count++;
      If[increment > 1 && Mod[count, increment] === 0,
        Print[ToString@count<> " of "<> ToString@Length@manifest<> ": "<>ToString[Round[100*N[count/Length@manifest]]]<>"%"]];
      (* end poor man's progress meter *)
      nb = #;
      nbExpr = Quiet@Get@nb;
      flag = DocumentFlag[nbExpr];
      Which[
        (* test1 *)
        MemberQ[{None, "AwaitingFutureDesignReviewFlag"}, flag],
        (* expr1 *)
        Which[
          Count[nbExpr, Cell[s_String /; StringMatchQ[s, ___~~"7"~~___], "HistoryData", CellTags -> "New"], Infinity] > 0,
          Switch[nb,
            s_String /; StringMatchQ[s, __~~FileNameJoin[{"System", "Guides"}]~~__], Sow[nb, "Newin7Guides"],
            s_String /; StringMatchQ[s, __~~FileNameJoin[{"System", "HowTos"}]~~__], Sow[nb, "Newin7HowTos"],
            s_String /; StringMatchQ[s, __~~FileNameJoin[{"System", "ReferencePages", "AppleScript"}]~~__], Sow[nb, "Newin7AppleScript"],
            s_String /; StringMatchQ[s, __~~FileNameJoin[{"System", "ReferencePages", "Characters"}]~~__], Sow[nb, "Newin7Characters"],
            s_String /; StringMatchQ[s, __~~FileNameJoin[{"System", "ReferencePages", "C"}]~~__], Sow[nb, "Newin7C"],
            s_String /; StringMatchQ[s, __~~FileNameJoin[{"System", "ReferencePages", "Files"}]~~__], Sow[nb, "Newin7Files"],
            s_String /; StringMatchQ[s, __~~FileNameJoin[{"System", "ReferencePages", "Formats"}]~~__], Sow[nb, "Newin7Formats"],
            s_String /; StringMatchQ[s, __~~FileNameJoin[{"System", "ReferencePages", "FrontEndObject"}]~~__], Sow[nb, "Newin7FrontEndObject"],
            s_String /; StringMatchQ[s, __~~FileNameJoin[{"System", "ReferencePages", "MenuItems"}]~~__], Sow[nb, "Newin7MenuItems"],
(*            s_String /; StringMatchQ[s, __~~FileNameJoin[{"System", "ReferencePages", "Messages"}]~~__], Sow[nb, "Newin7Messages"],*)
            s_String /; StringMatchQ[s, __~~FileNameJoin[{"System", "ReferencePages", "Methods"}]~~__], Sow[nb, "Newin7Methods"],
            s_String /; StringMatchQ[s, __~~FileNameJoin[{"System", "ReferencePages", "Programs"}]~~__], Sow[nb, "Newin7Programs"],
            s_String /; StringMatchQ[s, __~~FileNameJoin[{"System", "ReferencePages", "Symbols"}]~~__], Sow[nb, "Newin7Symbols"],
            s_String /; StringMatchQ[s, __~~FileNameJoin[{"System", "Tutorials"}]~~__], Sow[nb, "Newin7Tutorials"]],
          Count[nbExpr, Cell[s_String /; StringMatchQ[s, ___~~"7"~~___], "HistoryData", CellTags -> "Modified"], Infinity] > 0,
          Switch[nb,
            s_String /; StringMatchQ[s, __~~FileNameJoin[{"System", "Guides"}]~~__], Sow[nb, "Modin7Guides"],
            s_String /; StringMatchQ[s, __~~FileNameJoin[{"System", "HowTos"}]~~__], Sow[nb, "Modin7HowTos"],
            s_String /; StringMatchQ[s, __~~FileNameJoin[{"System", "ReferencePages", "AppleScript"}]~~__], Sow[nb, "Modin7AppleScript"],
            s_String /; StringMatchQ[s, __~~FileNameJoin[{"System", "ReferencePages", "Characters"}]~~__], Sow[nb, "Modin7Characters"],
            s_String /; StringMatchQ[s, __~~FileNameJoin[{"System", "ReferencePages", "C"}]~~__], Sow[nb, "Modin7C"],
            s_String /; StringMatchQ[s, __~~FileNameJoin[{"System", "ReferencePages", "Files"}]~~__], Sow[nb, "Modin7Files"],
            s_String /; StringMatchQ[s, __~~FileNameJoin[{"System", "ReferencePages", "Formats"}]~~__], Sow[nb, "Modin7Formats"],
            s_String /; StringMatchQ[s, __~~FileNameJoin[{"System", "ReferencePages", "FrontEndObject"}]~~__], Sow[nb, "Modin7FrontEndObject"],
            s_String /; StringMatchQ[s, __~~FileNameJoin[{"System", "ReferencePages", "MenuItems"}]~~__], Sow[nb, "Modin7MenuItems"],
(*            s_String /; StringMatchQ[s, __~~FileNameJoin[{"System", "ReferencePages", "Messages"}]~~__], Sow[nb, "Modin7Messages"],*)
            s_String /; StringMatchQ[s, __~~FileNameJoin[{"System", "ReferencePages", "Methods"}]~~__], Sow[nb, "Modin7Methods"],
            s_String /; StringMatchQ[s, __~~FileNameJoin[{"System", "ReferencePages", "Programs"}]~~__], Sow[nb, "Modin7Programs"],
            s_String /; StringMatchQ[s, __~~FileNameJoin[{"System", "ReferencePages", "Symbols"}]~~__], Sow[nb, "Modin7Symbols"],
            s_String /; StringMatchQ[s, __~~FileNameJoin[{"System", "Tutorials"}]~~__], Sow[nb, "Modin7Tutorials"]],
        True, ""];
        Which[
          Count[nbExpr, Cell[s_String /; StringMatchQ[s, ___~~"8"~~___], "HistoryData", CellTags -> "New"], Infinity] > 0,
          Switch[nb,
            s_String /; StringMatchQ[s, __~~FileNameJoin[{"System", "Guides"}]~~__], Sow[nb, "Newin8Guides"],
            s_String /; StringMatchQ[s, __~~FileNameJoin[{"System", "HowTos"}]~~__], Sow[nb, "Newin8HowTos"],
            s_String /; StringMatchQ[s, __~~FileNameJoin[{"System", "ReferencePages", "AppleScript"}]~~__], Sow[nb, "Newin8AppleScript"],
            s_String /; StringMatchQ[s, __~~FileNameJoin[{"System", "ReferencePages", "Characters"}]~~__], Sow[nb, "Newin8Characters"],
            s_String /; StringMatchQ[s, __~~FileNameJoin[{"System", "ReferencePages", "C"}]~~__], Sow[nb, "Newin8C"],
            s_String /; StringMatchQ[s, __~~FileNameJoin[{"System", "ReferencePages", "Files"}]~~__], Sow[nb, "Newin8Files"],
            s_String /; StringMatchQ[s, __~~FileNameJoin[{"System", "ReferencePages", "Formats"}]~~__], Sow[nb, "Newin8Formats"],
            s_String /; StringMatchQ[s, __~~FileNameJoin[{"System", "ReferencePages", "FrontEndObject"}]~~__], Sow[nb, "Newin8FrontEndObject"],
            s_String /; StringMatchQ[s, __~~FileNameJoin[{"System", "ReferencePages", "MenuItems"}]~~__], Sow[nb, "Newin8MenuItems"],
            (*s_String /; StringMatchQ[s, __~~FileNameJoin[{"System", "ReferencePages", "Messages"}]~~__], Sow[nb, "Newin8Messages"],*)
            s_String /; StringMatchQ[s, __~~FileNameJoin[{"System", "ReferencePages", "Methods"}]~~__], Sow[nb, "Newin8Methods"],
            s_String /; StringMatchQ[s, __~~FileNameJoin[{"System", "ReferencePages", "Programs"}]~~__], Sow[nb, "Newin8Programs"],
            s_String /; StringMatchQ[s, __~~FileNameJoin[{"System", "ReferencePages", "Symbols"}]~~__], Sow[nb, "Newin8Symbols"],
            s_String /; StringMatchQ[s, __~~FileNameJoin[{"System", "Tutorials"}]~~__], Sow[nb, "Newin8Tutorials"]],
          Count[nbExpr, Cell[s_String /; StringMatchQ[s, ___~~"8"~~___], "HistoryData", CellTags -> "Modified"], Infinity] > 0,
          Switch[nb,
            s_String /; StringMatchQ[s, __~~FileNameJoin[{"System", "Guides"}]~~__], Sow[nb, "Modin8Guides"],
            s_String /; StringMatchQ[s, __~~FileNameJoin[{"System", "HowTos"}]~~__], Sow[nb, "Modin8HowTos"],
            s_String /; StringMatchQ[s, __~~FileNameJoin[{"System", "ReferencePages", "AppleScript"}]~~__], Sow[nb, "Modin8AppleScript"],
            s_String /; StringMatchQ[s, __~~FileNameJoin[{"System", "ReferencePages", "Characters"}]~~__], Sow[nb, "Modin8Characters"],
            s_String /; StringMatchQ[s, __~~FileNameJoin[{"System", "ReferencePages", "C"}]~~__], Sow[nb, "Modin8C"],
            s_String /; StringMatchQ[s, __~~FileNameJoin[{"System", "ReferencePages", "Files"}]~~__], Sow[nb, "Modin8Files"],
            s_String /; StringMatchQ[s, __~~FileNameJoin[{"System", "ReferencePages", "Formats"}]~~__], Sow[nb, "Modin8Formats"],
            s_String /; StringMatchQ[s, __~~FileNameJoin[{"System", "ReferencePages", "FrontEndObject"}]~~__], Sow[nb, "Modin8FrontEndObject"],
            s_String /; StringMatchQ[s, __~~FileNameJoin[{"System", "ReferencePages", "MenuItems"}]~~__], Sow[nb, "Modin8MenuItems"],
(*            s_String /; StringMatchQ[s, __~~FileNameJoin[{"System", "ReferencePages", "Messages"}]~~__], Sow[nb, "Modin8Messages"],*)
            s_String /; StringMatchQ[s, __~~FileNameJoin[{"System", "ReferencePages", "Methods"}]~~__], Sow[nb, "Modin8Methods"],
            s_String /; StringMatchQ[s, __~~FileNameJoin[{"System", "ReferencePages", "Programs"}]~~__], Sow[nb, "Modin8Programs"],
            s_String /; StringMatchQ[s, __~~FileNameJoin[{"System", "ReferencePages", "Symbols"}]~~__], Sow[nb, "Modin8Symbols"],
            s_String /; StringMatchQ[s, __~~FileNameJoin[{"System", "Tutorials"}]~~__], Sow[nb, "Modin8Tutorials"]],
        True, ""]
        ,
        (* test2 *)
        flag === "FutureFlag",
        (* expr2 *)
        Switch[nb,
            s_String /; StringMatchQ[s, __~~FileNameJoin[{"System", "Guides"}]~~__], Sow[nb, "FutureGuides"],
            s_String /; StringMatchQ[s, __~~FileNameJoin[{"System", "HowTos"}]~~__], Sow[nb, "FutureHowTos"],
            s_String /; StringMatchQ[s, __~~FileNameJoin[{"System", "ReferencePages", "AppleScript"}]~~__], Sow[nb, "FutureAppleScript"],
            s_String /; StringMatchQ[s, __~~FileNameJoin[{"System", "ReferencePages", "Characters"}]~~__], Sow[nb, "FutureCharacters"],
                        s_String /; StringMatchQ[s, __~~FileNameJoin[{"System", "ReferencePages", "C"}]~~__], Sow[nb, "FutureC"],
            s_String /; StringMatchQ[s, __~~FileNameJoin[{"System", "ReferencePages", "Files"}]~~__], Sow[nb, "FutureFiles"],
            s_String /; StringMatchQ[s, __~~FileNameJoin[{"System", "ReferencePages", "Formats"}]~~__], Sow[nb, "FutureFormats"],
            s_String /; StringMatchQ[s, __~~FileNameJoin[{"System", "ReferencePages", "FrontEndObject"}]~~__], Sow[nb, "FutureFrontEndObject"],
            s_String /; StringMatchQ[s, __~~FileNameJoin[{"System", "ReferencePages", "MenuItems"}]~~__], Sow[nb, "FutureMenuItems"],
            (* s_String /; StringMatchQ[s, __~~FileNameJoin[{"System", "ReferencePages", "Messages"}]~~__], Sow[nb, "FutureMessages"], *)
            s_String /; StringMatchQ[s, __~~FileNameJoin[{"System", "ReferencePages", "Methods"}]~~__], Sow[nb, "FutureMethods"],
            s_String /; StringMatchQ[s, __~~FileNameJoin[{"System", "ReferencePages", "Programs"}]~~__], Sow[nb, "FuturePrograms"],
            s_String /; StringMatchQ[s, __~~FileNameJoin[{"System", "ReferencePages", "Symbols"}]~~__], Sow[nb, "FutureSymbols"],
            s_String /; StringMatchQ[s, __~~FileNameJoin[{"System", "Tutorials"}]~~__], Sow[nb, "FutureTutorials"]];
      ];
    ]& /@ manifest, _, Rule][[2]]
  ] ;


landinggrid[rr_, col_] := 
  Block[{ls = rr, mm, griddata}, mm = Ceiling[Length[rr]/col];
   ls = PadRight[rr, col*mm, ""];
   griddata = 
    If[Length[rr] > 0, 
     Transpose[Partition[ls, mm, mm, 1, ""]], {{""}}];
   griddata];
   
landinggridDIV[rr_, col_, machinename_] := 
 Module[{lst}, lst = landinggrid[rr, col]; 
  MapIndexed[
   XMLElement[
     "div", {"class" -> 
       Switch[#2, {1}, "columnLeft", {2}, "columnCenter", {3}, 
        "columnRight"]}, Flatten[#1]]&, 
   Transpose[
 Map[If[# =!= "", 
    {XMLElement[
     "a", {"href" -> 
       "http://"<>machinename<>"/language/" <>StringReplace[
 Take[FileNameSplit[#], {-2}][[1]], {
 	"Guides" -> "guide", 
 	"HowTos" -> "howto", 
 	"AppleScript" -> "ref/applescript", 
 	"Characters" -> "ref/character", 
 	"C" -> "ref/c", 
 	"Files" -> "ref/file", 
 	"Formats" -> "ref/format", 
 	"Devices" -> "ref/device",
 	"Interpreters" -> "ref/interpreter",  
 	"Services" -> "ref/service", 
 	"Indicators" -> "ref/indicator",
 	"Example" -> "example",  
 	"FrontEndObject" -> "ref/frontendobject", 
 	"MenuItems" -> "ref/menuitem",
    "Messages" -> "ref/message", 
(*    "Methods" -> "ref/method", *)
    "Programs" -> "ref/program", 
    "Symbols" -> "ref", 
    "Tutorials" -> "tutorial"
    }]<>"/"<> FileBaseName[#]<>".html"}, {If[StringLength[FileBaseName[#]]>=33, StringTake[FileBaseName[#], 30]<>"...", FileBaseName[#]]}], XML`RawXML["&nbsp;"]},
     ""] &, lst, {2}]]]];

grouplandinggrid[type_, rr_, col_, machinename_] := 
 Module[{}, 
  If[rr =!= {}, XMLElement["div", {"class" -> "ItemizedTextParagraph"}, 
   Join[{XMLElement["div", {}, {XMLElement["b", {}, {type}]}]}, 
    landinggridDIV[rr, col, machinename]]], {}]];

MakeLandingContent[nbdir_String, exportfile_String]:=Module[{layout, filename, headerContent, flagged, pagetypes, content, 
	pageText, theYear, theMonth, theDate, landingPage, ensuredata, newin7, newin8, newin8marketing, modifiedin7, modifiedin8, 
	future, confContent, iFrameLink, iFrame},
   theYear=ToString@Extract[Take[DateList[], {1, 3}], 1];
   theMonth=ToString@Extract[Take[DateList[], {1, 3}], 2];
   theDate=ToString@Extract[Take[DateList[], {1, 3}], 3];
   landingPage = theMonth<>"/"<>theDate<>"/"<>theYear;
   layout = ToFileName[{DocumentationBuild`Common`$DocumentationBuildDirectory, "XMLTransforms", "HTML", "Layouts"}, "LandingPage.html"];
   content=GetFutureLandingContent[nbdir];
   flagged = {"Newin7", "Modin7", "Newin8", "Modin8", "Future"};
   pagetypes = {"Guides", "HowTos", "AppleScript", "C", "Characters", 
   "Files", "Formats", "FrontEndObject", "MenuItems", (*"Messages", *)
   "Methods", "Programs", "Symbols", "Tutorials", "Examples"};
   (*this ensures that we get an empty list if the Sow above does find any items*)
   ensuredata= Flatten[Table[flagged[[i]] <> pagetypes[[j]] -> {}, {i, 1, Length[flagged]}, {j, 1, Length[pagetypes]}]];
   iFrameLink ="http://www.devel2.wolfram.com/language/new-in-8/embeddedlisting.html";
   iFrame=XMLElement["IFRAME",{"src"->iFrameLink, "width"->"500px;", "height"->"200px;"}, {"If you can see this your browser does not understand IFRAME. 
   You can link directly to the page here: ",
   XMLElement["a",{"url"->"iFrameLink"}, {iFrameLink}]
   }];
   
    newin7=XMLElement["div",{"id"->"div1","style"->"display:none"}, Flatten[{
   	Map[grouplandinggrid[StringReplace[#<>" Pages", "s "->" "],("Newin7"<>#)/.content/.ensuredata, 3, "reference.wolfram.com"]&, pagetypes]
    }]];
  modifiedin7=XMLElement["div",{"id"->"div2","style"->"display:none"}, Flatten[{
  	Map[grouplandinggrid[StringReplace[#<>" Pages", "s "->" "],("Modin7"<>#)/.content/.ensuredata, 3, "reference.wolfram.com"]&, pagetypes]
  	}]];
   newin8=XMLElement["div",{"id"->"div3","style"->"display:none"}, Flatten[{
   	Map[grouplandinggrid[StringReplace[#<>" Pages", "s "->" "],("Newin8"<>#)/.content/.ensuredata, 3, "reference.devel.wolfram.com"]&, pagetypes]
    }]];
  newin8marketing=XMLElement["div",{"id"->"div6","style"->"display:none"}, 
  	{iFrame}
  ];
  modifiedin8=XMLElement["div",{"id"->"div4","style"->"display:none"}, Flatten[{
  	Map[grouplandinggrid[StringReplace[#<>" Pages", "s "->" "],("Modin8"<>#)/.content/.ensuredata, 3, "reference.devel.wolfram.com"]&, pagetypes]
  	}]];
  future=XMLElement["div",{"id"->"div5","style"->"display:none"}, Flatten[{
  	Map[grouplandinggrid[StringReplace[#<>" Pages", "s "->" "],("Future"<>#)/.content/.ensuredata, 3, "reference.future.wolfram.com"]&, pagetypes]
  	}]];
    
   pageText = Transmogrify`FillInLayout[layout, "Text", 
   	"landingPage"->landingPage,
   	"NEWIN7"->ExportString[newin7,"XML"], 
   	"MODIFIEDIN7"->ExportString[modifiedin7,"XML"],
   	"NEWIN8MARKETING"->ExportString[newin8marketing,"XML"], 
   	"NEWIN8"->ExportString[newin8,"XML"], 
   	"MODIFIEDIN8"->ExportString[modifiedin8,"XML"],
   	"FUTURE"->ExportString[future,"XML"]];
   	
   	confContent="SetEnv STYLE_CustomHeaderFile 0
SetEnv STYLE_InternalPageWidth 100%
SetEnv STYLE_Body '<body style=\"margin:0px;\" onLoad=\"javascript:set_domain()\">'";
   	   headerContent="
   <title>Mathematica Documentation Development</title>
   
   <link rel=\"stylesheet\" type=\"text/css\" href=\"/common/includes/CSS.cgi?b=menustyles;p=css2008/reference.wolfram.com/;v=en\">
   <link rel=\"shortcut icon\" href=\"/common/images2003/favicon.ico\" type=\"image/x-icon\">

   <script src=\"/javascript/search.js\" type=\"text/javascript\"></script>
<script type=\"text/javascript\" src=\"/common/javascript/jquery/plugins/droppy/0.1.2rm/jquery.droppy.js\"></script>
<!--Initializes the header menu-->
<script type='text/javascript'>
  $(function() {
    $('#nav').droppy();
  });
  
function togglediv(thediv) {$(thediv).slideToggle();}
</script>

   <style class=\"text/css\">
ul.indentednote a.graylink:active,ul.indentednote a.graylink:link,ul.indentednote a.graylink:visited {
	font-weight:normal;
	color:gray}
div.GuideSection {
	margin-left:110px;
	font-size:14px;
	font-weight:normal;
	border-top-color: #aa0000;
	width:820px;}
a:link,a:visited {color:#aa0000;}
ul.indentednote a,ul.indentednote a:link,ul.indentednote a:visited {font-weight:bold;color:#aa0000;}
h3 {
	font-weight:normal;
	font-size:26px;
	font-family:Georgia,Arial,sanserif}
.ItemizedTextParagraph a:link,.ItemizedTextParagraph a:visited {color:#666;}
.ItemizedTextParagraph a:hover,.ItemizedTextParagraph a:active {color:#cc0000;}
.ItemizedTextParagraph div {padding-bottom:4px;}
div.columnLeft,div.columnCenter,div.columnRight {
	line-height:.85em;
	width:220;
	float: left;
	padding-rigth: 10px;	
	position: relative; 		
	text-align: justify;
	margin: 0;}
div.columnRight {float:right;}
.ItemizedTextParagraph a {display:block}
div.ItemizedTextParagraph {clear:both}
</style>";

   Export[exportfile<>"-conf", confContent, "Text"];  
   Export[exportfile<>"-head", headerContent, "Text"];   	
   Export[exportfile, pageText,"Text"]
   ];



(* Make Future landing page from Bill's report *)
ExportLandingPage[path_String] := 
  Module[{layout, filename, sevenContent, futureContent, parsedFutureContent,
    landingPageVariables, pageText, theYear, theMonth, theDate, landingPage, headerContent},
   theYear=ToString@Extract[Take[DateList[], {1, 3}], 1];
   theMonth=ToString@Extract[Take[DateList[], {1, 3}], 2];
   theDate=ToString@Extract[Take[DateList[], {1, 3}], 3];
   landingPage = theMonth<>"/"<>theDate<>"/"<>theYear;
	
   layout = ToFileName[{DocumentationBuild`Common`$DocumentationBuildDirectory, "XMLTransforms", "HTML", "Layouts"}, "LandingPage.html"];
   filename = ToFileName[{path}, "mathematica.en.html"];
   sevenContent = Import["https://internal.wolfram.com/wpt/reports/mathematica/gregm/NewModified.txt"];
   futureContent = Import["https://internal.wolfram.com/wpt/reports/mathematica/gregm/Future.txt"];
   parsedFutureContent = StringReplace[futureContent, 
   {
   	"<div class=\"GuideSection\">Quick links to future <i>Mathematica</i> pages:</div>"->"", 
   (*"<div class=\"ItemizedTextParagraph\"><b>"~~Shortest[a__]~~"</b></div>"->a<>"&nbsp;",*)
   "<div class=\"ItemizedTextParagraph\"><b>"~~Shortest[a__]~~"</b></div>"->"<div class=\"NewItemizedTextParagraph\"><b>"<>a<>"</b></Newdiv>&nbsp;",
   "<div class=\"ItemizedTextParagraph\">"->"",
   "</div>"->""
   }];
   
 
   
   (*Print@StringSplit[parsedFutureContent, "<div class=\"NewItemizedTextParagraph\">"->"<div class=\"ItemizedTextParagraph\">"];*)
   (*Print@StringSplit[parsedFutureContent,"<div class=\"NewItemizedTextParagraph\"><b>"~~Shortest[a__]~~"</b></div>"];
   landingPageVariables = {"Body1" -> sevenContent,
     						"Body2" -> futureContent, 
     						"landingPage" ->landingPage};*)
   pageText = Transmogrify`FillInLayout[layout, "Text", landingPageVariables ];
   headerContent="
   <title>Wolfram Research, Inc.</title>
   
   <link rel=\"stylesheet\" type=\"text/css\" href=\"/common/includes/CSS.cgi?b=menustyles;p=css2008/reference.wolfram.com/;v=en\">
   <link rel=\"shortcut icon\" href=\"/common/images2003/favicon.ico\" type=\"image/x-icon\">

   <script src=\"/javascript/search.js\" type=\"text/javascript\"></script>
<script type=\"text/javascript\" src=\"/common/javascript/jquery/plugins/droppy/0.1.2rm/jquery.droppy.js\"></script>
<!--Initializes the header menu-->
<script type='text/javascript'>
  $(function() {
    $('#nav').droppy();
  });
  
function togglediv(thediv) {$(thediv).slideToggle();}
</script>

   <style class=\"text/css\">
ul.indentednote a.graylink:active,ul.indentednote a.graylink:link,ul.indentednote a.graylink:visited {
	font-weight:normal;
	color:gray}
div.GuideSection {
	margin-left:110px;
	font-size:14px;
	font-weight:normal;
	border-top-color: #aa0000;
	width:820px;}
a:link,a:visited {color:#aa0000;}
ul.indentednote a,ul.indentednote a:link,ul.indentednote a:visited {font-weight:bold;color:#aa0000;}
h3 {
	font-weight:normal;
	font-size:26px;
	font-family:Georgia,Arial,sanserif}
.ItemizedTextParagraph a:link,.ItemizedTextParagraph a:visited {color:#666;}
.ItemizedTextParagraph a:hover,.ItemizedTextParagraph a:active {color:#cc0000;}
.ItemizedTextParagraph div {padding-bottom:4px;}
div.columnLeft,div.columnCenter,div.columnRight {
	line-height:.85em;
	width:220;
	float: left;
	padding-rigth: 10px;	
	position: relative; 		
	text-align: justify;
	margin: 0;}
div.columnRight {float:right;}
.ItemizedTextParagraph a {display:block}
div.ItemizedTextParagraph {clear:both}
</style>";

   Export[filename<>"-head", headerContent, "Text"];
   Export[filename, pageText, "Text"]
   
   ];

(******************************************************************************

	Wrapper functions for html builds

******************************************************************************)

(******************************************************************************)
(** htmlPreBuild **)
htmlPreBuild[inputDir_String ]:=
Module[{},

	(* User variables *)
	(* if creating local html, add 'html' dir to output path *)
	outputDir = 
		If[ TrueQ@ToExpression@AntProperty["completeHTMLQ"] && TrueQ@ToExpression@AntProperty["local"],
			AppendHTMLDirectory@AntProperty["outputDir"],
			AntProperty["outputDir"]
		];

	max = {0, "None"};
	AntSetProperty["max", ToString@max];
	lang = AntProperty["language"];
	linkBase = AntProperty["app.name"];
	guidesDir = DocumentationBuild`Utils`GetGuidesDirectory[inputDir, lang]; 
	completeHTMLQ = ToExpression@AntProperty["completeHTMLQ"];
	Global`$AntBuildQ = True;

	DocumentationBuild`Transform`debug = False;
	DocumentationBuild`Export`$DocumentationExportDebug = ToExpression @ AntProperty["debug"];
	        
	feMemory = InputForm @ 
		DocumentationBuild`Common`PubsEvaluateWithFE[MathLink`CallFrontEndHeld[FrontEnd`Value[FEPrivate`MemoryInUse]]];
	AntSetProperty["FEMemory", ToString@feMemory];

	BuildLog @ StringJoin[
	   "M- Version: ", ToString[Internal`$CreationID], " ", ToString@$Version, "\n", 
	   "Current FE memory: ", ToString @ feMemory, "\n",
	   "Installation Directory: ", ToString@$InstallationDirectory, "\n",
	   "DocumentationBuild Directory: ", DocumentationBuild`Common`$DocumentationBuildDirectory
	 ];

	 (* Test Import/Export convertors *)
	If[MatchQ[
		ImportString[ExportString["<tag>1</tag>", "XML"], "XML"], 
	    	$Failed | Null],
	AntFail["Export convertor failure. Unable to Import/Export XML!"] ];

	includeLinkTrails = If[MemberQ[{"false", "False"}, AntProperty["includeLinkTrails"]], False, True];
	BuildLog["IncludeLinkTrails: "<>ToString[includeLinkTrails]];

	buildType = AntProperty["buildType"];
	 
	If[ (buildType==="Full") || (buildType==="HeadFiles"),
		files = DocumentationBuild`Common`notebookFileNames @ inputDir,
		files = DocumentationBuild`Common`notebookFileNames[ inputDir, HoursAgo->21, FileList->{}]
	 ];

	(* remove files which aren't under 'language' *)
	files = Select[files, StringMatchQ[#, "*" ~~ ($PathnameSeparator | "/") ~~ lang ~~ "*"]& ];
  
   	(* remove files which aren't notebooks *)
   	files = Select[files, StringMatchQ[#, "*.nb"]& ];

   	(* remove "Messages" from web builds *)
	(*TODO: make this an option *)
	(* sw recanted and wants Messages to appear on the web 
    	files = Select[files, ! StringMatchQ[#, "*" ~~ ($PathnameSeparator | "/") ~~ "Messages" ~~ "*"]& ];
	*)

	AntSetProperty["numberTotalNotebooks", ToString@Length@files];
	BuildLog["Guides directory:"<>guidesDir];
	BuildLog["_"];
	BuildLog["Notebooks to process: "<>ToString@Length[files]];
	BuildLog["_"];

	files = ToString@Sequence[files];
	files = StringTake[StringReplace[files, RegularExpression["\\s*,\\s*"]->"," ], {2,-2}];
	AntSetProperty["notebooks", files ];

	If[ !MemberQ[{"Mathematica", ""}, linkBase],
		BuildLog["Computing $CoreInformation data"];
		DocumentationBuild`Export`$CoreInformation[lang] =
		DocumentationBuild`CreateSourceInformation[DirectoryName@inputDir, lang, linkBase]
	];

	BuildLog["_"];
	BuildLog["Computing guide link trails"];
	DocumentationBuild`Navigation`$Pages = DocumentationBuild`Navigation`ComputeLinkTrails[guidesDir];
	BuildLog["Number of pages: "<>ToString@Length@DocumentationBuild`Navigation`$Pages];
	BuildLog["_"];

];


(******************************************************************************)
(** htmlBuildIterator **)
htmlBuildIterator[inFile_String, opts___?OptionQ ]:=
Module[{outFile, inputDir, outputDir, language, localQ, completeHTMLQ, 
	debug, exitCodePath, totalNumberNotebooks, requestLimit, max, version, info, 
	entityType, time, flag, res, tnn, nsf, nff, npf},

	{"Language", "LocalQ", "CompleteHTMLQ", "Debug", "ExitCodePath", "RequestLimit", 
		"Max", "Version"} = 
		{language, localQ, completeHTMLQ, debug, exitCodePath, requestLimit, 
			max, version} /. {opts} /. Options[htmlBuildIterator];
	
	tnn = DocumentationBuild`Build`$TotalNumberNotebooks;
	nsf = DocumentationBuild`Build`$NumSuccessfulFiles;
	nff = DocumentationBuild`Build`$NumFailedFiles;
	npf = DocumentationBuild`Build`$NumProcessedFiles;
	
	DocumentationBuild`Transform`debug = 
	DocumentationBuild`Export`$DocumentationExportDebug = debug;


	(* if creating local html, add 'html' dir to output path *)
	outputDir = 
		If[ completeHTMLQ && localQ,
			AppendHTMLDirectory@outputDir,
			outputDir
		];

		(* TODO Fix hack for Preview pages *)
		(* The following should work but doesn't for Preview pages *)
		outFile = StringReplace[inFile, {".nb" -> "", ToFileName[{inputDir, language}] -> $PathnameSeparator}];
		outFile = StringReplace[outFile, {".nb" -> "", ToFileName[{inputDir}] -> $PathnameSeparator}];

		outFile = 
			DocumentationBuild`Common`FilePathToDocURI[
				StringJoin[outputDir, StringReplace[outFile, {".nb" -> ""}], ".html"], $PathnameSeparator];

		(* TODO clean up this path name section *)
		outFile =
			FixedPoint[StringReplace[#, {
				"System\\" -> "",
				"Packages\\" -> "",
				"Documentation\\" <> language <> "\\" -> "",
				language <> "\\" -> "",
				"\\\\" -> "\\",
				"//" -> "/"
				}] &, outFile];

          npf++; DocumentationBuild`Build`$NumProcessedFiles++;
          max = If[Head@max =!= String, "{0, None}", max];
          max = StringSplit[StringReplace[max, "{" | "}" -> ""], RegularExpression[",\\s+"]];
          max = {ToExpression@max[[1]], max[[2]]};

          BuildLog["Processing: " <> 
			FixedPoint[StringReplace[#, {
				outputDir -> "", "\\" -> "/", 
				RegularExpression["^\\s*\\/(.+)"] -> "$1"}] &, outFile]];

		  info = GetNotebookInformation[Get @ inFile];
		  If[ info =!= $Failed, 
			entityType = ("EntityType" /. info);
		  (* check type *)
		  entityType = If[ StringMatchQ[entityType, "EntityType"], $Failed, entityType];
			,
				entityType = $Failed;
			];
					
          (* Optional build settings *)
          opts = {"Version" -> version};

          Which[
            (* ignore files and CopyFile to output dir without converting *)
            !DocumentationBuild`Utils`ConvertFileQ[ inFile],
        	  BuildLog["            File marked to ignore -- file not included in build."];
              time = {0,""},
            (* Notebook to convert doesn't contain proper EntityType *)
            MemberQ[{entityType}, $Failed],
              BuildLog["            File's Entity Type not found -- file not included in build."];
              time = {0,""},
            (* pass over flagged nbs which shouldn't be included within on-web distribution *)
            MatchQ[ flag = DocumentFlag[ inFile] , "ExcisedFlag" | "FutureFlag" | "InternalFlag" | "TemporaryFlag"],
              BuildLog["            " <> flag <> " -- file not included in build."];
              time = {0,""},
			(* Convert everything else *)   
            True, (

				If[buildType === "HeadFiles",
				   (* Only export -head files *)
				   time = AbsoluteTiming[ 
				     res = DocumentationBuild`Common`PubsEvaluateWithFE @ 
				       ExportWebHeadFile[ 
				         outFile, 
				         inFile, 
				         pageType, 
				         language,
					     "CompleteHTMLQ" -> completeHTMLQ ] 
				      ],
				   (* Convert notebook to html *)
				   time = AbsoluteTiming[ 
				     res = DocumentationBuild`Common`PubsEvaluateWithFE @ 
				       ExportDocumentation[ 
				         outFile, 
				         inFile, 
				        "ExportFormats"->"HTML",
						"Language"->language,
					    "CompleteHTMLQ" -> completeHTMLQ,
					    "IncludeLinkTrails" -> includeLinkTrails,
						If[ localQ, 
							Rule["SearchSSI",""],
						    Rule["Foo", a]
						   ],
					    If[ localQ, 
						    Rule["LanguageLinkSSI",""],
					        Rule["Foo", a]
					       ],
					    If[ localQ, 
						    Rule["CopyrightSSI",""],
					        Rule["Foo", a]
					       ],						
					    If[ localQ, 
					        Rule["FooterSSI",""],
				            Rule["Foo", a]
				          ]
						] 
				      ];
					(*BuildLog[res]*)
				];

              (* Output file test *)
			outFile = "HTML" /. res;
			  If[ (FileType[outFile]===None || FileByteCount[outFile]<1), 
				BuildLog["*** Output file test failed *** "]; 
				BuildLog[outFile]; 
				res = $Failed]; 
            )];


          If[res =!= $Failed, 
            (
              nsf++; DocumentationBuild`Build`$NumSuccessfulFiles++;
              If[First@time > max[[1]], max = {First@time, StringReplace[inFile, inputDir->""]}];
              AntSetProperty["max", ToString@max];

              BuildLog[StringJoin[ "Success!    ", ToString[First @ time], " secs" ] ];
            )
            , 
            (
              nff++; DocumentationBuild`Build`$NumFailedFiles++;
              BuildLog["*** Failed *** "];
            )
          ];

          BuildLog @ StringJoin[ "[Processed  ", ToString@npf,
            "/", ToString@tnn, "; ", ToString@nsf,
            " successful, ", ToString@nff, " failed]"];

          (* Force quit the kernel after n requests *)
          If[ Mod[ToExpression@npf, ToExpression@requestLimit]===0, 
            (BuildLog["Quiting MathKernel after "<>requestLimit<>" requests."]; Quit[];)
            , 
            BuildLog["_"];
          ]; 
    
 
];


(******************************************************************************)
(** htmlPostBuild **)
htmlPostBuild[ ]:=
Module[{},
	pf = AntProperty["processedFiles"];
	sf = AntProperty["successfulFiles"];
	ff = AntProperty["failedFiles"];
	max = AntProperty["max"];
	max = StringSplit[StringReplace[max, "{" | "}" -> ""], RegularExpression[",\\s+"]];

	If[ TrueQ@ToExpression@AntProperty["completeHTMLQ"], (
		outputDir = AppendHTMLDirectory@AntProperty["outputDir"];
		overviewFile = ToFileName[{inputDir,"Tutorials"},"Overview.nb"];
		If[ FileType@overviewFile===File,
			overviewFile = DocumentationBuild`Export`MakeOnlineNotebook[overviewFile, 
							DocumentationBuild`Export`NotebookEntityType @ overviewFile, language]];
		If[FileType@overviewFile===File, (
			BuildLog["Creating Eclipse xml resource files."];
			BuildLog["Using "<>overviewFile];
			ret = DocumentationBuild`Utils`ExportEclipseResourceFiles[
					AntProperty["outputDir"],
					overviewFile, linkBase];
			If[Head@ret =!= String, ret];),
			BuildLog["Overview tutorial not found."]
		];

		BuildLog["_"];
		BuildLog["Copying html files for local viewing."];
	htmlTemplateDir = ToFileName[{DocumentationBuild`Common`$DocumentationBuildDirectory, "Internal", "web", "html"}];

		  (* Copy ExampleData files/dir ignoring 'bad' files *)
		  htmlTemplateFiles = FileNames["*", htmlTemplateDir, Infinity];
		  htmlTemplateFiles = Select[htmlTemplateFiles, (
		    StringFreeQ[#, $PathnameSeparator~~"CVS"~~$PathnameSeparator] && 
		    FileType[#] === File) &];

		  Map[
		    Function[
		      outFile = ToFileName[{outputDir},StringReplace[#, {htmlTemplateDir->""}]]; 
		      CreateDirectory[DirectoryName[ outFile ]]; 
		      CopyFile[#, outFile ]; 
		    ], htmlTemplateFiles]; ),
		outputDir = AntProperty["outputDir"];		    
	];
		  successRate = .9;
	
	BuildLog["_"];

	(* Wrap up... *)
	BuildLog["Finished Processing!"];
	BuildLog["Success Rate: "<>sf<>"/"<>pf<>" ("<> 
	  ToString@
	If[(ToExpression@pf === 0), 100, N[100 ToExpression@sf/ToExpression@pf]] <> "%)"];

	(* Which file took the longest to process? *)
	BuildLog["Maximum file time: "<>ToString@max[[1]]<>" secs: "<>max[[2]]];

	(* How many failed files? *)
	If[ToExpression@ff > 0,
	  BuildLog["Number of failed files: "<>ff (*<>"\n"<>
	   "Failed files: \n"<>ToString[failedFiles]*) ],
	  ""];

	BuildLog["_"];

	(* Return stuff *)
	BuildLog @
	If[
	  (* the current default successRate is 90% of all notebooks for a successful build *)
	  (* For final release, successRate must be 100% *)
	  (ToExpression@pf === 0) || (ToExpression@sf > (ToExpression@pf * successRate) ), 
    
	  (* Successful build *)
	  BuildLog["***** Build acceptable for internal/Alpha/Beta release *****" ]
	  ,
	  (* Failed build *)
	  BuildLog["***** BUILD FAILED! *****"];
	  (* Return ANTFAIL *)
	  AntFail["build failed"]
	];

];



(****************************************************************
            X M L T r a n s f o r m   F U N C T I O N S

    The following helper functions are used only in the 
    Transmogrify XMLTransforms.
 ***************************************************************)
(**
  htmlBoxesQ
  check to see if *Box's need to be images 
**)

htmlBoxesQ[expr_]:=
	Module[{pos}, 
	Length[Select[
		Position[expr,
			(_FractionBox | _RadicalBox | _SqrtBox | _OverscriptBox | _UnderscriptBox | _SubsuperscriptBox | _GridBox | _SliderBox | _Slider2DBox | _CheckboxBox | _RadioButtonBox | _OpenerBox | _ActionMenuBox | _PopupMenuBox | _ProgressIndicatorBox | _InputFieldBox | _ColorSetterBox ),
			{1, Infinity}],
		Function[{pos},
			FreeQ[
			Map[Head[Extract[expr, #]] &,
				Map[Take[pos, #] &, Range[2, Length[pos]]]
			],
			Cell
		]]
	]]=== 0]



(**
    inputQueue[] - get the appropriate list of Inputs up to and including an
    inputline containing a "%".

    Expects "s" to be the result of CreateInputForm[]. Also, use RenumberCells[]
    to make sure each example section starts at In[1] for best results. 
*)
inputQueue[s_String,label_] := Module[{out,refs},
  Catch[

    (* reset the queue if we've started a new section. we're assuming that the
    first input cell does not contain a "%", as that would just be silly... *)
    If[
      !StringFreeQ[label/.None->"","In[1]"]
    ,
      ClearAll[$InputQueue];
      $InputQueue[$InputCount=1]=s;
      Throw[s]
    ];
                                                                                               
    (* now add to the queue ... note that $IntegerCount _should_ already be
    initialized above *)
    $InputQueue[++$InputCount] = s;

    Which[
      StringFreeQ[s,"%"], s (* return the string if there's no "%" or ... *)
    ,
      !StringFreeQ[$InputQueue[1],"%"], s
    ,
      (* ... otherwise, return the specified input requested by "%". join all
      the % references in the queue, separated by ";\n" *)
      True, (StringJoin@@((#<>";\n"&)/@Most[#])<>Last[#])&[findInputRefs[$InputQueue[$InputCount],{$InputCount}]]
    ]
  ]
]

inputQueue[f___] := "$Failed";


(*  fun recursion to get all %-referenced inputs.
    i'm sure there's a non-Recursive-MapIndexed way to do this
    but this seemed easiest. :) *)
findInputRefs[s_String,p_]:=Module[{spl, pos = p[[1]], refs},
  spl = percentSplit[s];
  (* FIXME: this doesn't handle %1 digitized refs, although
     percentSplit does.  I don't think we use those in our
     documentation though.
  *)
  (* FIXME: this is actually broken when refs contains % refs,
     as it gets them all out of order and doesn't uniqueify
     them in any meaningful way.  Try this:
       $InputQueue[++$InputCount] = "One[1]";
       $InputQueue[++$InputCount] = "Two[%]";
       $InputQueue[++$InputCount] = "Three[%]";
       $InputQueue[++$InputCount] = "Four[%, %%]";
       findInputRefs[$InputQueue[$InputCount], {$InputCount}]
     This bug has existed for a while, but I don't think we
     actually trigger it anywhere.
  *)
  refs = Flatten[
    If[# > 0 && pos =!= #, $InputQueue[pos - #], $InputQueue[pos]] & /@ 
      Reverse[Sort[StringLength /@
        UnsortedUnion[Cases[spl, pc_/;(StringTake[pc, 1] === "%")]]
    ]] ];
  Flatten@Append[MapIndexed[findInputRefs, refs], s]
];

percentSplit[s_String] := percentSplit[Characters[s]];
percentSplit[chars_List] :=
Module[{res = "", c, i = 1, esc = False, str = False},
  While[i <= Length[chars],
    c = chars[[i]];
    Switch[c,
      "\\", (esc = !esc; res = res<>c),
      "\"", (If[esc, esc = False, str = !str]; res = res<>c),
      "%", If[esc || str, res = res<>c, Break[]],
      _, (esc = False; res = res<>c)];
    i++;
  ];
  Flatten[{res,
    If[i > Length[chars], {},
      Module[{dig = "%", c},
        i++;
        While[i < Length[chars],
          c = chars[[i]];
          If[DigitQ[c] || c === "%", dig = dig<>c, Break[]];
          i++];
        {dig,
          If[i > Length[chars], {},
            percentSplit[Drop[chars, i - 1]] ]}
    ]]
  }]
];

inlineModified[Cell[TextData[lst_List], o___]] :=
 Module[{new = lst},
  new = new /. Cell[BoxData["modified"], "InlineFormula"] :> StyleBox["modified", "GuideText", FontWeight->"Plain"];
  new = new /. {" (modified) \[EmptyVerySmallSquare] "} :> {" ", StyleBox["(modified)", "GuideText", FontWeight->"Plain"], " \[EmptyVerySmallSquare] "};
  Cell[TextData[Flatten@new], o]
  ];

inlineSeparator[Cell[TextData[lst_List], o___]] :=
 Module[{new},
  new = lst /. {s_String :> inlineSeparatorString@s};
  Cell[TextData[Flatten@new], o]
  ]
inlineSeparator[c___]:= c;  

inlineSeparatorString[str_String] :=
 Sequence @@ 
  StringReplace[
   str, {RegularExpression[
      "(.*)\\s+\[EmptyVerySmallSquare]\\s+(.*)"] :> {"$1", " ", 
      StyleBox["\[FilledVerySmallSquare]", "InlineSeparator"], " ", 
      "$2"}}]

inlineSeparatorString[c___] := c;


(** ReplaceSpecialCharacters[] 
    this takes care of turning unicode longname \[Character]'s into an
    HTML image representation like <img src="/some/location/Character.gif" />
    
    note that simply calling StringSubstitute (=StringReplace[]) is too slow
    for such a large list of rules (currently there are about 652).
    Therefore, thanks to the wonderful magic of oyvind, we employ a hybrid
    use of string replacement + mathematica replacements.
**)

(*  The following code is a one time setup of some pattern variables
    that are actually used in the ReplaceSpecialCharacters[] implementation
    below. *)

(* go ahead and dispatchify two char rules now, though this might not
   be necessary *)

$TwoCharRules = Dispatch @
    DocumentationBuild`DocumentationConversion`CharacterReplacements`twoCharacterReplacementRules;


(* all single characters we're trying to match are NOT standard printable ASCII. *)

$OneCharPattern = RegularExpression[
  StringPattern`PatternConvert[
    Except[
      CharacterRange[FromCharacterCode[32],FromCharacterCode[126]]
    ]
  ][[1]]
];


(* take advantage of the fact that all 2 character strings end in "=". 
   uncomment the second definition if this is no longer the case *)

$TwoCharPattern = RegularExpression["[<>!=]="];


(* $TwoCharPattern = RegularExpression["[<>!=]=|[-:]>"]; *)
(* $TwoCharPattern = 
  RegularExpression[StringPattern`PatternConvert[$TwoCharRules[[All,1]]][[1]]] *)

(* apply rule replacements on string *)
ReplaceSpecialCharacters[ text:(_String|{__String}) ] := Module[{rawCharList, oneCharRules},
	
	rawCharList=DocumentationBuild`DocumentationConversion`CharacterReplacements`characterReplacementRules;
	
	ModifyRawCharacterList[list_List]:= Module[{pos1,pos2,newlist},
		pos1=Position[list, "\[OpenCurlyQuote]" -> XML`RawXML["&#39;"], 2];
		pos2=Position[list, "\[CloseCurlyQuote]" -> XML`RawXML["&#39;"], 2];
		newlist=ReplacePart[list, {pos1->"\[OpenCurlyQuote]" -> "'", pos2->"\[CloseCurlyQuote]" -> "'"}];
		newlist
	];
	
	ModifyRawCharacterListForSolutions[list_List]:= Module[{pos1,pos2,newlist},
		pos1=Position[list, "\[Dash]" -> "-", 2];
		newlist=ReplacePart[list, {pos1->"\[Dash]" -> "\[Dash]"}];
		newlist
	];
		
	rawCharList=Which[GetParameter["IsTMJQ"]===True,
		ModifyRawCharacterList[rawCharList], 
			GetParameter["IsSolutionsQ"]===True,
		ModifyRawCharacterListForSolutions[rawCharList], 
			True, rawCharList];

  oneCharRules = Dispatch @ Prepend[
  	
   rawCharList,
    "\n" -> Which[
        GetParameter["newline2br"], XMLElement["br", {}, {}],
        GetParameter["newline2div"], XML`RawXML[ GetParameter["styleUsageMessageBreak"] ],
        True, "\n"
    ]
  ];

  Sequence @@ StringReplace[ text, { 

    "\t" -> XML`RawXML["&nbsp;&nbsp;&nbsp;&nbsp;"],
(*    x:$TwoCharPattern :> ( x /. $TwoCharRules ),*)
    x:$OneCharPattern :> ( x /. oneCharRules )
(*
    x:RegularExpression["[!<>=]=|[-:]>"] :> (XMLElement["img", {"src" -> ("URL" /. #),
        "width" -> ("Width" /. #), "height" -> ("Height" /. #), "alt" -> " "}, {}]&[
            BoxToImage[ToFileName[{"HTMLImages", GetFileNameBase[]},
                GetFileNameBase[] <> "_str_" <> IncrementCounter["SubstringImages"] <> ".gif"],
                StringContent -> Switch[x,
                    "!=", "\[NotEqual]",
                    "<=", "\[LessEqual]",
                    ">=", "\[GreaterEqual]",
                    "==", "\[Equal]",
                    ":>", "\[RuleDelayed]",
                    "->", "\[Rule]",
                    _, x
                ] ]]),
    c:RegularExpression["[^\:0020-\:007E]"] :> Switch[c,
        "\[NonBreakingSpace]",      "\[NonBreakingSpace]",
        "\[AutoOperand]",           "",
        "\[Rule]",                  "->",
        "\[Equal]",                 "=",
        "\[Ellipsis]",              "...", 
        "\[OpenCurlyDoubleQuote]",  "\"", 
        "\[CloseCurlyDoubleQuote]", "\"", 
        "\[OpenCurlyQuote]",        XML`RawXML["&#39;"],
        "\[CloseCurlyQuote]",       XML`RawXML["&#39;"],
        "'",                        XML`RawXML["&#39;"],
        "\[LineSeparator]",         XMLElement["br", {}, {}],
        "\:f35f",                   "",
        "\[IndentingNewLine]",      XMLElement["br", {}, {}],
        "\[ParagraphSeparator]",    XMLElement["p", {}, {}], 
        "\[Hyphen]",                "-", 
        "\[Conjugate]",             "*",
        "\[Ellipsis]",              "...", 
        "\[Colon]",                 ":", 
        "\[VerticalBar]",           "|", 
        "\[Precedes]",              "<", 
        "\[Succeeds]",              ">", 
        "\[Backslash]",             "\\", 
        "\[Dash]",                  "\[Dash]", 
        "\[LongDash]",              "\[LongDash]", 
        "\[InvisibleApplication]",  "", 
        "\[InvisibleSpace]",        "", 
        "\[VeryThinSpace]",         "", 
        "\[ThinSpace]",             " ", 
        "\[MediumSpace]",           " ", 
        "\[ThickSpace]",            " ", 
        "\[NegativeVeryThinSpace]", "", 
        "\[NegativeThinSpace]",     "", 
        "\[NegativeMediumSpace]",   "", 
        "\[NegativeThickSpace]",    "",  
        "\[Null]",                  "", 
        "\[NoBreak]",               "",
        "\t", "\[NonBreakingSpace]\[NonBreakingSpace]\[NonBreakingSpace]\[NonBreakingSpace]", 
        "\n", Which[
        	GetParameter["newline2br"], XMLElement["br", {}, {}],
        	GetParameter["newline2div"], XML`RawXML[ GetParameter["styleUsageMessageBreak"] ],
    	    True, "\n"
        ],
        _, XMLElement["img", {"src" -> ("URL" /. #),
            "width" -> ("Width" /. #), "height" -> ("Height" /. #), "alt" -> " "}, {}]&[
                BoxToImage[ToFileName[{"HTMLImages", GetFileNameBase[]},
                    GetFileNameBase[] <> "_str_" <> IncrementCounter["SubstringImages"] <> ".gif"],
                    StringContent -> c ]]
    ] 
    *)
}]
(*
    oneCharRules
  }] /.
        (LongName[c_] :> (XMLElement["img", {"src" -> ("URL" /. #),
        "width" -> ("Width" /. #), "height" -> ("Height" /. #), "alt" -> " "}, {}]&[
            BoxToImage[ToFileName[{"HTMLImages", GetFileNameBase[]},
                GetFileNameBase[] <> "_str_" <> IncrementCounter["SubstringImages"] <> ".gif"],
                StringContent -> c ]]))
  )
*)
];

RelativeFileNames[pat_String:"*", dir_, lev_:1] :=
    Module[{dirWithTrailingSlash},
        dirWithTrailingSlash = If[StringMatchQ[dir, "*" <> $PathnameSeparator], dir, dir <> $PathnameSeparator];
        StringReplace[#, dirWithTrailingSlash -> ""]& /@ FileNames[pat, dir, lev]
    ];


(* Builds the PacletDB.m file that describes the docs that ship in the layout. This file goes into the layout as a
   pre-cached descriptor that can be read quickly by the PacletManager. The topDir directory should have a
   Documentation/lang/System hierarchy beneatch it, and also a Packages dir beneath it.
   The top-level subdirs within topDir will be the standard set Symbols, Guides, tutorials, etc. 

   The resulting PacletDB.m file is placed into outputDir.
*)
Options[BuildDocPacletIndex] = Options[ExportOnlineNotebooks]

BuildDocPacletIndex[topDir_String, outputDir_String, language_String, OptionsPattern[]]:= 
    Module[{logfile, append, files},
    
        Print["** Entering BuildDocPacletIndex **"];
                
        {logfile, append} = OptionValue[{LogFile, LogAppend}];
        
        (* quick logfile error checking *)
        If[!MatchQ[logfile, _String | None],
            Message[DocumentationExport::logfile, logfile];
            logfile = None;
        ];
        
        (* Create outdir *) 
        If[Transmogrify`Utilities`Private`createDirectory[ToFileName[{outputDir}]] === $Failed,
            Print[
                "Couldn't create output directory " <> ToFileName[{outputDir}],
                CloseLog->True
            ];
            Return[$Failed]
        ];
        
        Print[
            "BuildDocPacletIndex[] Started", 
            "Source: " <> topDir,
            "PacletDB.m file destination: " <> outputDir,
            "M- Version: " <> ToString[Internal`$CreationID] <> " " <> ToString[$Version],
            "PacletManager CreationID: " <> ToString[PacletManager`Information`$CreationID],
            "Installation Directory: " <> ToString[$InstallationDirectory],
            LogAppend->append
        ];
        
        files = FileNames["*.nb", topDir, Infinity];
        Print[ ToString[Length[files]] <> " Notebooks to Process."];
    
        (* Version 2 of the PM debuts a new format for the PacletInfo descriptor file for docs in the layout.
           We create a PacletDB.m file instead of the PacletInfo.db file used in earlier versions.
        *)
        If[StringQ[PacletManager`Package`BuildDocDBFile[outputDir, topDir, language]],
            Print[ "Successfully built PacletDB.m file", CloseLog->True],
        (* else *)
            Print[ "PacletDB.m file FAILED", CloseLog->True];
            $Failed
        ]
    ]


    

GetTutorialsFromGuide[nbexpr_] := 
  Module[{}, Cases[nbexpr, Cell[___, "GuideTutorial", ___], Infinity]];
  
GetTrainingLink[nbexpr_] := 
  Module[{urls, videourl}, urls = Cases[nbexpr, TemplateBox[{Cell[Localized["LearningResourcesTrainingCoursesText", GetParameter["language"]]], a_, ___}, ___]:>a, Infinity];
  videourl = If[Length[urls] > 0, urls[[1]], ""]
  ];
  
GetVideoLink[nbexpr_] := 
  Module[{urls, videourl}, urls = Cases[nbexpr, TemplateBox[{Cell[Localized["LearningResourcesVideosText", GetParameter["language"]]], a_, ___}, ___]:>a, Infinity];
  videourl = If[Length[urls] > 0, urls[[1]], ""]
  ];
  
GetCommunityLink[nbexpr_] := 
  Module[{urls, videourl}, urls = Cases[nbexpr, TemplateBox[{Cell[Localized["LearningResourcesCommunityText", GetParameter["language"]]], a_, ___}, ___]:>a, Infinity];
  videourl = If[Length[urls] > 0, urls[[1]], ""]
  ];  

GetDemonstrationLinkFromGuide[nbexpr_] := 
  Module[{urls, videourl}, urls = Cases[nbexpr, TemplateBox[{Cell[Localized["LearningResourcesDemonstrationsText", GetParameter["language"]]], a_, ___}, ___]:>a, Infinity];
  videourl = If[Length[urls] > 0, urls[[1]], ""]
  ];

GetHowTosFromGuide[nbexpr_] := 
  Module[{}, 
   Cases[nbexpr, Cell[___, "GuideHowTo", ___], Infinity]
   ];



CreateLearningResources[nbexpr_, uri_] := 
  Module[{tutorials, howtos, trainingcourse, video, community, 
    demonstration, includetutorials, includehowtos, 
    includedemonstration, includetrainingcourse, includevideo, 
    includecommunity, leftcellgroup, rightcellgroup}, 
   tutorials = GetTutorialsFromGuide[nbexpr] /. Cell[a___, "GuideTutorial", b___]:>Cell[a, "LearningResourceTutorial", b];
   howtos = GetHowTosFromGuide[nbexpr] /. Cell[a___, "GuideHowTo", b___]:>Cell[a, "LearningResourceHowTo", b];
   trainingcourse = ConvertToString[GetTrainingLink[nbexpr]];
   video = ConvertToString[GetVideoLink[nbexpr]];
   demonstration = ConvertToString[GetDemonstrationLinkFromGuide[nbexpr]];
   community = ConvertToString[GetCommunityLink[nbexpr]];
   includetutorials = If[Length[tutorials] > 0, True, False];
   includehowtos = If[Length[howtos] > 0, True, False];
   includetrainingcourse = 
    If[StringLength[trainingcourse] > 0, True, False];
   includevideo = If[StringLength[video] > 0, True, False];
   includedemonstration = 
    If[StringLength[demonstration] > 0, True, False];
   includecommunity = If[StringLength[community] > 0, True, False];
   leftcellgroup = 
    If[includetutorials || includehowtos, 
     Cell[CellGroupData[
       Join[{Cell["Left", "GuideLearningResourcesLeftSection", 
          CellGroupingRules -> {"SectionGrouping", 31}]}, 
        Flatten[{If[includetutorials, 
           Cell[CellGroupData[
             Join[{Cell[Localized["TutorialsSectionText", GetParameter["language"]], 
                "GuideLearningResourcesTutorialSection"]}, 
              tutorials]]], {}],
          If[includehowtos, 
           Cell[CellGroupData[
             Join[{Cell[Localized["GuideHowTosSectionText", GetParameter["language"]], 
                "GuideLearningResourcesHowToSection"]}, 
              howtos]]], {}]}]]]], {}];
   rightcellgroup = 
    If[includedemonstration || includetrainingcourse || includevideo || includecommunity,
     Cell[CellGroupData[
       Join[{Cell["Right", "GuideLearningResourcesRightSection", 
          CellGroupingRules -> {"SectionGrouping", 31}]}, Flatten[{
          Cell[Localized["LearningResourcesRelatedWebText", GetParameter["language"]], "SmallTitle"],
          
          If[includetrainingcourse, 
           Cell[TextData[
             ButtonBox[Localized["LearningResourcesTrainingCoursesText", GetParameter["language"]], BaseStyle -> "ResourceTraining", 
              ButtonData -> URL[trainingcourse]]], 
            "ResourceTraining"], {}],
          
          If[includevideo, 
           Cell[TextData[
             ButtonBox[Localized["LearningResourcesVideosText", GetParameter["language"]], BaseStyle -> "ResourceVideos", 
              ButtonData -> URL[video]]], "ResourceVideos"], {}],
          
          If[includedemonstration, 
           Cell[TextData[
             ButtonBox[Localized["LearningResourcesDemonstrationsText", GetParameter["language"]], BaseStyle -> "ResourceDemonstrations", 
              ButtonData -> URL[demonstration]]], 
            "ResourceDemonstrations"], {}],
          
          If[includecommunity, 
           Cell[TextData[
             ButtonBox[Localized["LearningResourcesCommunityText", GetParameter["language"]], BaseStyle -> "ResourceCommunity", 
              ButtonData -> URL[community]]], 
            "ResourceCommunity"], {}]}]]]], {}];
   If[includetutorials || includehowtos || includedemonstration || 
     includetrainingcourse || includevideo || includecommunity,
    Flatten[{leftcellgroup,
         rightcellgroup}], {}]
   ];
   
 CreateFeaturedPage[featuredexampledata_List] := 
 Module[{featuredexampledata2}, 
  If[Length[featuredexampledata] > 0,
  	featuredexampledata2 = Partition[Flatten[featuredexampledata], 3];
   Map[Cell[
         CellGroupData[{Cell["/language/example/"<>FileBaseName[#[[3]]] <> ".html", "FeaturedExampleLink", 
            CellGroupingRules -> {"SectionGrouping", 31}], 
           Cell[BoxData[ToBoxes[Uncompress[#[[1]]]]], 
            "FeaturedExampleThumbnailImage"], 
           Cell[#[[2]], "FeaturedExampleText"]}]] &, 
       featuredexampledata2], {}]];
       
AddFirstAndLastToexpandWraps[expr2_] := Module[{pos},
  pos = Position[expr2, "expandWrap"];
  Which[Length[pos] === 1, 
   ReplacePart[expr2, First[pos] -> "expandWrap first last"],
   Length[pos] > 1, 
   ReplacePart[
    expr2, {First[pos] -> "expandWrap first", 
     Last[pos] -> "expandWrap last"}],
   True, expr2]
  ];

deleteNonASCII[str_String] := Module[{}, 
	FromCharacterCode[DeleteCases[DeleteCases[ToCharacterCode[str], chrctr_ /; chrctr > 126], 
		33 | 35 | 36 | 37 | 38 | 40 | 41 | 42 | 44 | 47 | 58 | 60 | 61 | 62 | 63 | 64 | 91 | 93 | 94 | 96 | 123 | 124 | 125 | 126]]];
  
MakeMarketingThumbnailList[expr_, language_] := Module[{tds},
  (*expr is a list of pairs {thumbail,caption}*)
  tds = {
  	XMLElement[
       "div", {},
       {XMLElement[
       "a", {"href" -> 
         Cases[{#[[1]]}, 
           XMLElement["a", {___, Rule["href", b___], ___}, {___}] :> 
            b, Infinity][[1]]}, 
       Flatten[{Flatten[{Flatten[{Cases[{#[[1]]}, 
              w : XMLElement["img", {___}, {___}] :> w, 
              Infinity]}]}]}]]}], 
      XMLElement[
       "div", {}, {XMLElement[
       "a", {"href" -> 
         Cases[{#[[1]]}, 
           XMLElement["a", {___, Rule["href", b___], ___}, {___}] :> 
            b, Infinity][[1]]}, 
       Flatten[{Flatten[
          Cases[{#[[2]]}, 
           XMLElement[
             "a", {___}, {x___}] :> {XMLElement[
              "span", {"class" -> "chevron-after"}, {x}]}, 
           Infinity]]}]]}]} & /@ expr;
  XMLElement["section", {"class" -> "examples"}, 
   Flatten[{XMLElement[
      "h3", {}, {Localized["MarketingRelatedExamples", language]}], 
     XMLElement["div", {"class" -> "table"}, 
      Map[XMLElement["div", {"class" -> "heirs-width-25per"}, #] &, 
       Flatten[Map[Transpose, Partition[tds, 4, 4, 1, {}]], 1]]]}]]];

   
End[]
EndPackage[]
