(* ---------------------------------------------------------------------------- 

   Title: Utilities.m                                                       

   Authors: Stephen Layland <layland@wolfram.com>

   Version: $Id: Utilities.m,v 1.29 2016/02/19 20:30:10 billw Exp $ 

   Discussion:                                                                 

   These are useful utility functions that are often used in Transmogrify`
   that were split out of Transmogrify.m for easier maintenance.

 ---------------------------------------------------------------------------- *)
(* <!-- Public 
InlineFormattedQ::usage = "Tests if a string has inline formatted characters"
FromInlineString::usage = "Turns an inline formatted string into its StandardForm box\
 structure"
FromInlineStringTests::usage = "FromInlineStringTests is a set of rules to test changes\
 to FromInlineString.  Evaluating the LHS should give the RHS."
--> *)

Begin["`Utilities`Private`"]

PrependOptions::usage = "PrependOptions[expr, newopts] - adds newopts to the beginning\
 of the options in expr."
DeleteOptions::usage = "DeleteOptions[expr, optname] - removes all instances of optname\
 from the options in expr.\nDeleteOptions[expr, opt->val] - removes only the instance\
 of opt-val from the options in expr."
ResetOptions::usage = "ResetOptions[expr, newopts] - replaces (or adds) newopts in the\
 options of expr."

mkdirRecursive::usage = "Create a directory along with parent directories if they don't\
 exist.  This is the same as the unix command \"mkdir -p\""

createDirectory::usage = "Mimic pre-10.4 behavior of CreateDirectory: if the requested
directory cannot be created, return the name of the directory.";

BaseName::usage = "BaseName[file] returns the base file name of a full path\
 filename.  For example, /path/to/file.nb -> file.nb.  BaseName[file,\
 Extension->True] causes the file extension to not be included,\
 /path/to/file.nb -> file"
Extension::usage = "Option for BaseName[] that removes the file extension\
 when set to True"


(* ------------------------------------------------------------- *)

(** PrependOptions | ResetOptions | DeleteOptions 
    option manipulation helpers
    these are the fastest implementations i could come up with! 

    As you can see, PrependOptions is by far the fastest as it doesn't
    try to delete the original option.  this is fine for most cases where
    they are applied via Replace*.
**)
PrependOptions[h_[stuff___,opts___?OptionQ],newopts:{__?OptionQ}]:=
  h[stuff,Sequence@@Flatten@newopts,opts]
PrependOptions[h_[stuff___,opts___?OptionQ], newopts__?OptionQ]:=
  h[stuff,newopts,opts]

DeleteOptions[h_[stuff___,opts___?OptionQ], newopts__?OptionQ]:=
  DeleteOptions[h[stuff,opts],{newopts}]

(* delete the exact option,value pair *)
DeleteOptions[h_[stuff___,opts___?OptionQ], newopts:{__?OptionQ}]:=
  h[stuff, Sequence @@ DeleteCases[{opts},Alternatives@@newopts]]

(* delete all options with the optioname specified in newopts *)
DeleteOptions[h_[stuff___,opts___?OptionQ], newopts:{__}]:=
  h[stuff, Sequence @@ DeleteCases[{opts},_[Alternatives@@newopts,_]]]

If[ $VersionNumber >= 6,
  (* make use of v6's System`Utilities`FilterOutOptions which is slightly faster
     than DeleteCases[].  much of this should be made obsolete with Oyvind's new 
     OptionPattern[] functionality. *)
  ResetOptions[h_[stuff___,opts___?OptionQ], newopts__?OptionQ]:=
    h[stuff, Sequence@@Flatten@{newopts}, Sequence @@ System`Utilities`FilterOutOptions[
      {newopts},{opts}]],

  (* actually reset options. it's faster to define this in terms of 
   delete cases rather than have another call to DeleteOptions *)
  ResetOptions[h_[stuff___,opts___?OptionQ], newopts__?OptionQ]:=
    h[stuff, Sequence@@Flatten@{newopts}, Sequence @@ DeleteCases[{opts}, _[Alternatives @@ (Flatten[{newopts}][[All, 1]]), _]]]
]

ResetOptions[h_[stuff___],newopts__?OptionQ]:=h[stuff,newopts]

(** mkdirRecursive : create directory and all parent directories **)
(* CreateDirectory has done the right thing since 6.  Maintain this
   for a while for compatibility. *)
mkdirRecursive = CreateDirectory;

(* As of 10.4, the possible return values of CreateDirectory[dir] have changed.  

CreateDirectory now behaves as documented since V2: it returns $Failed if it fails to
create the requested directory.  See the extensive discussion starting at
https://mail-archive.wolfram.com/archive/l-kernel/2016/Feb00/

Previously, it would return the name of the requested directory if it couldn't be
created, which is the behavior relied upon by Transmogrify and DocumentationBuild.  Note
that CreateDirectory[] still behaves as advertised: it creates a dir in
$TemporaryDirectory.

Also note that pre-10.4, insufficient permissions did not provoke $Failed:

In[3]:= CreateDirectory["test"]
CreateDirectory::privv: Privilege violation for file or directory /private/var/folders/bq/52z56ksx2j765fb78zw6sczr000_9z/T/m00000276111/test/.
Out[3]= CreateDirectory[test]

In[4]:= $Version
Out[4]= 10.3.0 for Mac OS X x86 (64-bit) (October 9, 2015)

So we provide createDirectory[dir] for use where needed, following Bob Sandheinrich in
https://mail-archive.wolfram.com/archive/l-kernel/2016/Feb00/0184.html

Where is such a thing needed?  *Anywhere we used the value returned by CreateDirectory.*

 - anywhere 'dir = CreateDirectory[<existing dir>]' might happen.  Many but not all
   instances of CreateDirectory are expected to return a name that's used later, even
   when the requested directory cannot be created.

 - anywhere we rely on CreateDirectory to return $Failed.  I guess someone read the docs
   and assumed it could return $Failed as documented.  It hasn't been able to return
   $Failed for years, and we need to retain that lack of $Failed with createDirectory.
   Ideally we would test all those old clauses that might now be exercised post-10.4
   when CreateDirectory might return $Failed, but ain't nobody got time for that.

*)

createDirectory[dir_] := dir /; DirectoryQ[dir];
createDirectory[dir_] := CreateDirectory[dir, CreateIntermediateDirectories->True];

(** filename utilities **)
(** 
    BaseName[].  Get the basename of a file.  The added option
    Extension->False tells BaseName to not include the file
    extension.
**)
Options[BaseName] = {Extension->True}
BaseName[f_String,opts___?OptionQ]:= (
  If[TrueQ[$VersionNumber>=7.0], If[(`Extension /. {opts} /. Options[BaseName])===True,
    FileBaseName[f]<>FileExtension[f],
    FileBaseName[f]
  ],
  If[(`Extension /. {opts} /. Options[BaseName])===True,
    #,
    StringReplace[#,"."~~Except["."]..~~EndOfString->""]
  ]&@StringReplace[f,DirectoryName[f]->""]
  ]
)


(** inline formatted string functions **)
(** 
  InlineFormattedQ 
  quick function that tries to infer if a string has inline formatting rules
  like "\!\(\*StyleBox[\"blah\",\"IT\"]\)".
**)
InlineFormattedQ[s_String]:= Or[
  Not[StringFreeQ[ToString[FullForm[s]],
    RegularExpression["\\\\!\\\\(.*?\\\\)"]]],
  $ShowStringCharacters===False && StringMatchQ[s, "\"*\""]
];

(** FromInlineString **)
(* TODO - use jah's new ReparseBoxes packet? *)
FromInlineString[s_, showstring_:Automatic] := 
  Block[{nb = NotebookCreate[Visible -> False],
         str, res, qbef = False, qaft = False, show},
    str = s;
    If[showstring === Automatic, show=($ShowStringCharacters=!=False), show=showstring];
    If[show===False(*StringMatchQ[str, RegularExpression[".*\\\\\\!\\\\\\(.*\\\\\\).*"]]*),
      str = Cell[BoxData[str], ShowStringCharacters->False],
      (* Else *)
      If[StringMatchQ[str, "\"*"],
        qbef = True,
        str = "\"" <> str];
      If[StringMatchQ[str, "*\""],
        qaft = True,
        str = str <> "\""];
    ];
    NotebookWrite[nb, str];
    SelectionMove[nb, All, Notebook];
    SelectionMove[nb, Before, CellContents];
    FrontEndTokenExecute[nb, "DeleteNext"];
    SelectionMove[nb, After, CellContents];
    FrontEndTokenExecute[nb, "DeletePrevious"];
    SelectionMove[nb, All, CellContents];
    res = NotebookRead[nb];
    NotebookClose[nb];
    If[qbef || qaft,
      TextData[Join[
        If[qbef, {"\""}, {}],
        {res},
        If[qaft, {"\""}, {}]
      ]],
      res]];


FromInlineStringTests = {
{"\"\<\!\(\*StyleBox[\"file\", \"TI\"]\)\!\(\*StyleBox[\".\", \"MR\"]\)\!\(\*StyleBox[\"gif\", \"MR\"]\)\>\"",
True}
-> TextData[{"\"",
   RowBox[{StyleBox["file", "TI"], StyleBox[".", "MR"], StyleBox["gif", "MR"]}],
   "\""}],

{"\<\"\\!\\(\\* StyleBox[\"c\", \"TI\"]\\)\"\>",
False}
-> StyleBox["c", "TI"],

{"\"\<\!\(\*\
StyleBox[\"elem\", \"TI\"]\)\!\(\*\
StyleBox[\"\\\"\<\>\", \"TI\"]\)\>",
True}
-> TextData[{"\"", StyleBox[RowBox[{"elem", "\""}], "TI"]}],

{"\<\"[\!\(\*SubscriptBox[StyleBox[\"c\", \"TI\"],StyleBox[\"1\", \"TR\"]]\)\!\(\*\
SubscriptBox[StyleBox[\"c\", \"TI\"],StyleBox[\"2\", \"TR\"]]\)\!\(\*\
StyleBox[\"\[Ellipsis]\", \"TR\"]\)]\"\>",
False}
-> RowBox[{"[", 
   RowBox[{SubscriptBox[StyleBox["c", "TI"], StyleBox["1", "TR"]], 
   SubscriptBox[StyleBox["c", "TI"], StyleBox["2", "TR"]], 
   StyleBox["\[Ellipsis]", "TR"]}], "]"}]
};


(* in need a stringifyer... *)
ConvertToString[c_String] := c;
ConvertToString[c_Cell] := ConvertToString@c[[1]];
ConvertToString[c_List] := StringJoin[ConvertToString /@ c];
ConvertToString[c_ButtonBox] := ConvertToString@c[[1]];
ConvertToString[c_BoxData] := ConvertToString@c[[1]];
ConvertToString[c_StyleBox] := ConvertToString@c[[1]];
ConvertToString[c_RowBox] := StringJoin[ConvertToString /@ c[[1]]];
ConvertToString[c_TextData] := ConvertToString@c[[1]];
ConvertToString[c_FormBox] := ConvertToString@c[[1]];
ConvertToString[c_AdjustmentBox] := ConvertToString@c[[1]];
ConvertToString[c_SuperscriptBox] := StringJoin[ConvertToString@c[[1]] <> "^"<>ConvertToString@c[[2]]];
ConvertToString[c_SubscriptBox] := StringJoin[ConvertToString@c[[1]] <> "_"<>ConvertToString@c[[2]]];
ConvertToString[f_FractionBox] := 
	StringJoin[" ( ", ConvertToString@f[[1]], " ) / ( ", ConvertToString@f[[2]] , " ) "];
ConvertToString[c : TemplateBox[_, "RefLink"|"OrangeLink"|"BlueLink", ___]] := ConvertToString[c[[1,1]]];
ConvertToString[c_] := ToString@c;
(* warn if a sequence *)
ConvertToString[c__] := (Message[ConvertToString::sequence]; StringJoin[ConvertToString /@ {c}]);
ConvertToString::sequence = "Warning: Sequence used as an argument, should be List";
(* legacy support *)
stringifyContent[c_]:= (Message[ConvertToString::legacy]; ConvertToString[c]);
ConvertToString::legacy = "stringifyContent[] has been deprecated. Use ConvertToString[] instead.";

End[] (* Utilities`Private` *)
