(* 
    Title: Debug.m 

    Author: Stephen Layland <layland@wolfram.com>

    Version: $Id: init.m,v 1.5 2006/01/19 23:20:18 shaunm Exp $

    Discussion: 

    This package, similar to Parallel`Debug`, is a debug tool
    for Transmogrify developers.  It should be loaded prior to loading
    the regular Transmogrify package. If not, it will attempt to 
    clear all definitions from the Transmogrify` context first.
*)

If[ NameQ[ "Transmogrify`Transmogrify" ],
  Unprotect["Transmogrify`Debug`*"];
  Unprotect["Transmogrify`*"];
  ClearAll@@Release[
    Hold[Sequence[#<>"*",#<>"Private`*"]]& /@ 
    Select[$ContextPath, StringMatchQ[#,"Transmogrify*"]&]
  ];
]

BeginPackage["Transmogrify`Debug`"]


(* -----------------------------------------------
        P U B L I C   D E C L A R A T I O N S
   ----------------------------------------------- *)

Transmogrify`$DebugMode = True

$Debug::usage = "Object used to save current state of Transmogrify debugging."
$TraceOptions::usage = "List of available TraceOptions."
$TraceHandlers::usage = "List of available TraceHandlers."
TraceFile::usage = "When set, this $Debug option indicates the file to be used when saving debug output via\
 TraceHandler->Save.  This file will be appended to."
$TraceList::usage = "List of debug sections currently being debugged."
$TraceHandler::usage = "Current TraceHandler."
$TraceFile::usage = "Current TraceFile value"

$TraceOptions = {
    Expressions, NBProfile, ImageCreation, 
    Recursion, TransformParsing, TransformRules 
  }

Expressions::usage = ""
NBProfile::usage = ""
ImageCreation::usage = ""
Recursion::usage = ""
TransformParsing::usage = ""
TransformRules::usage = ""

$TraceHandlers = {Print,Save}

TraceHandler::usage = "Option for $Debug which controls how debug results are handled.  See $TraceHandlers for valid options."
Verbosity::usage = "Option for $Debug which controls how much info is printed.  Should be positive integer or 0."

(* -----------------------------------------------
       B E G I N   I M P L E M E N T A T I O N
   ----------------------------------------------- *)
Begin["`Private`"]

$defaultVerbosity = 1

If[Length@Options[$Debug] == 0,
  Options[$Debug] = {
    TraceHandler->Print, Trace->{},
    Verbosity->$defaultVerbosity,TraceFile->None
  };
  Options[NBProfile] = {"SaveProfile"->False,"ProfileFile"->"%f-Profile.m"}
]

$Debug /: SetOptions[$Debug, opts__?OptionQ] :=
  Block[{t,th, goodopts=Append[$TraceOptions,{}], $Trap=True},
    {t,th} = {Trace,TraceHandler} /. {opts} /. Options[$Debug];
    Catch[
      Which[
        !MemberQ[$TraceHandlers, th],
          Message[SetOptions::badval, th, TraceHandler];
          Throw[$Failed],
        MatchQ[t,_?AtomQ|{_?AtomQ,__?OptionQ}],
          t = {t}
      ];
      
      (If[!(MatchQ[#,{_?AtomQ,___?OptionQ}]&&MemberQ[goodopts,First[#]]),
        Message[SetOptions::badval, First[#], Trace];
        Throw[$Failed]
      ])&/@(Flatten[{#},1]&/@t);
      SetOptions[$Debug, opts, Trace->t,TraceHandler->th]
    ]
  ]/;!TrueQ[$Trap]
SetOptions::badval="`1` is not a valid value for `2`."

$TraceList := Trace /. Options[$Debug]
$TraceHandler := TraceHandler /. Options[$Debug]
$Verbosity := Verbosity /. Options[$Debug]
$TraceFile := StringReplace[ToString[TraceFile /. Options[$Debug]],
  StartOfString~~"~"->$HomeDirectory]

(*
traceLevel[section_] := Module[
  {v},
  If[Length[v=Cases[$TraceList,section|{section,_}]]==0,
    Return[0]];

  If[ListQ[v[[1]]],
    v[[1,-1]],
    Verbosity /. Options[$Debug]
  ]
]
*)
(** 
  trace[]
**)

(*
trace[ImageCreation,"i am some stuff"]
trace[{ImageCreation,lev}, "more stuff"]
trace[{ImageCreation,lev}, "this is pic: ", expr]
*)

(* quick helper to get current sub options of a given function *)
traceOpts[x_]:= Cases[$TraceList, {x, opts___?OptionQ} :> opts]

trace[section_, stuff__]:= trace[{section,$defaultVerbosity},stuff]
trace[{section_,lev_},stuff__]:= Module[
  {},
  If[!MemberQ[$TraceList,section|{section,___?OptionQ}],
    Return[]
  ];
  If[ lev <= $Verbosity,
    traceHandler[$TraceHandler][section,stuff]
  ]
]

traceHandler[Print][section_, stuff___]:=
  Print[(*ToString@section<>":\n",*)
    Sequence@@(("\t"<>ToString[#]<>"\n")&/@{stuff}),line]
  
shortline = "------------"
line = "-----------------------------------------\n"

traceHandler[Save][section_, stuff___]:=
  Module[{s,close},

    s = Streams[$TraceFile];
    If[ s === {},
      If[(s=OpenAppend[$TraceFile])===$Failed,
        Message[traceHandler::nofile];
        Abort[]
      ]
    ];

(*    WriteString[s,shortline<>ToString@section<>shortline<>"\n"];*)
    WriteString[s, ToString[First@{stuff}]<>"\n"];
    If[Length@{stuff}>1,
      WriteString[s, "\t"<>ToString@#<>"\n"]&/@Rest@{stuff}
    ];
    WriteString[s,line];
  ]/;StringQ[$TraceFile]

traceHandler[Save][___]:=(Message[$Debug::file])/;!StringQ[$TraceFile]

$Debug::file = "TraceFile is not set or does not point to a filename.  No debug information will be saved."
traceHandlerCleanUp[Save]:=If[StringQ[$TraceFile] && Streams[$TraceFile] =!= {},
  Close[$TraceFile]]

traceHandler::nofile="Could not open the file specified by $TraceFile.\
 Please make sure this variable is set or check that you have appropriate\
 permissions to access the file."

Protect["Transmogrify`Debug`*"]
Unprotect[$TraceFile];

End[] (* `Private` *)

EndPackage[]

(* go ahead and load Transmogrify now... *)
Get["Transmogrify`"]
