(* :Title: SystemResourcesFromDocs.m *)

(* :Authors:
	Jerry Walsh
	walsh@wolfram.com

	Jay Warendorff
	jayw@wolfram.com

	Oyvind Tafjord
	tafjord@wolfram.com
*)

(* :Package Version: 0.10 *)

(* :Mathematica Version: 6.0 *)

(* :Requirements: *)
(* :Discussion:
*)

BeginPackage["DocumentationBuild`SystemResourcesFromDocs`", {
  "DocumentationBuild`Common`",
  "DocumentationBuild`Info`",
  "DocumentationBuild`Utils`",
  "DocumentationBuild`Make`",
  "Transmogrify`"
}];


(************************************************************************
    Public methods; for usage messages, see DocumentationConversion.m
**************************************************************************)

CreateFunctionInformationFile

CreateUsageMessageFile

CollectTemplates::usage = "\
CollectTemplates[ dir] scans all notebooks in the directory
dir for function syntax templates, and returns a list of
rules of the form {\"functionName\" -> {tmpl1, tmpl2, ... }}."

CreateFunctionTemplatesFile::usage = "\
CreateFunctionTemplatesFile[{dir1, dir2, ...}, outputfile] scans all
notebooks in the directories {dir1, dir2, ...} creating argument patterns
for each relevant function. The list of argument patterns are saved
in outputfile (typically FunctionTemplates.m). The function tries to
read the context from the directory, and if the context is not System,
a context` is prepended to each function name in the output file. If
errors occur during conversion, a message is Print'ed and no argument
pattern is generated for that function."

$SuppressFunctionOptions::usage = "$SuppressFunctionOptions gives a list of
symbols whose options should be omitted from the file put out by
CreateFunctionOptionsFile."

InsertPackageSetting::usage =
"Inserts the context as specified by the Context cell in a function page
type notebook into the notebook level tagging rule
\"NeededPackages\"->{context} for each function page type notebook in the
directory dir which may be nested.";

AddQuotesToSubscriptedVariables::usage = "Option of CreateUsageMessageFile. The default is None. Otherwise it is specified by AddQuotesToSubscriptedVariables -> {{\"function1\", {\"var11\", \"var12\", ...}}, {\"function2\", {\"var21\", \"var22\", ...}}, ...}. With this setting quotes are added to the subscripted variables \"var11\", \"var12\", ... in the usage message for \"function1\", etc. for cases that are not handled by the general rules of CreateUsageMessageFile.";

CreateTextUsageMessageFile::usage =
"CreateTextUsageMessageFile[functionPages, fileoutpath] creates a usage message file with path fileoutpath consisting of string usage messages from a set of function pages functionPages.";

CreateUsageMessageString::usage = "CreateUsageMessageString[name, usagecell] where name is a function name and usagecell is a usage cell in a function page returns a simple text usage message or $Failed.";

AddPackageSyntaxInformation::usage=
"AddPackageSyntaxInformation[functionPages, packageFile] will find syntax information in the functionPages files and replace the $SyntaxInformationData statement in packageFile with the updated information.";
ExplicitOptionNames  (* Option for AddPackageSyntaxInformation *)

ExportOnlineResourceFiles::usage= "Exports Usage.m and other resource files"

GetJLinkEtcGuidePageDirs::usage = "";
GetJLinkEtcTutorialPageDirs::usage = "";
CreateFunctionBrowserData::usage = "";
CreateTutorialBrowserData::usage = "";
CreateFunctionNavigator::usage = "";
GetTemplateCellsWithUsage::usage = "";
CreateFunctionInformationFileWithUsage::usage="";
GetUsageFromUsageCell::usage = "";
GetUsageFromNoteCell::usage = "";
ExtractTemplatesAndUsagesFromUsageCell::usage = "";
ExtractTemplatesAndUsagesFromNoteCell::usage = "";
ExtractTemplatesAndUsagesFromNotebooksNoteCell::usage = "";
ExtractTemplatesAndUsagesFromNotebooksAdditionalFunctionTemplateCell::usage = "";
GetTemplateCellsWithUsage2::usage = "";

(***************************************
	Begin function definitions
***************************************)

Begin["`Private`"];

`escape = DocumentationConversion`Private`escape

`$CVSVersion = "$Id: SystemResourcesFromDocs.m,v 1.104.2.1 2016/08/16 20:48:12 brianv Exp $"

(***************************************
	Messages
***************************************)



(***************************************
	Methods
***************************************)

GetTemplateCells::nusage = "No usage cell with one or more ModInfo cells found for `1`."

GetTemplateCells[ nbex_Notebook, name_] :=
Module[{
	nbex2=DeleteCases[nbex, c:Cell[___, "FutureExample" | "TentativeExample"| "ExcludedMaterial", ___] /; FreeQ[c, "ModInfo", Infinity], Infinity],
   preusagecell1,
   usagecell, 
   ff
},


preusagecell1 = ReplaceAll[Cases[ nbex2, Cell[a_/;Not@FreeQ[a, Cell[_, "ModInfo",___]], "Usage", ___], Infinity], 
 Cell[BoxData[FormBox[StyleBox[DynamicBox[ToBoxes[If[$OperatingSystem === "MacOSX", "Return", "Enter"], StandardForm], 
                                          ImageSizeCache -> _], "KeyEvent"], TraditionalForm]], NotebookDefault, 
      ShowStringCharacters -> False] -> StyleBox["Enter", "KeyEvent"]];

 preusagecell1 = preusagecell1 /. Cell[BoxData[TooltipBox[Cell[v___, "ModInfo",vv___],___]]] :> Cell[v, "ModInfo", vv];

   name
      ->
   (Flatten[
      Join[
         If[ preusagecell1 =!= {} && ((usagecell = ReduceCell[preusagecell1[[1]]]) =!= Cell[TextData[{}], "Usage"]),
            Cases[
               Extract[ usagecell, 
                  Union[
                     Position[
                        usagecell,
                        x_String /; StringMatchQ[x, "*\[LineSeparator]*"]
                     ]
                        /. {a__, b_Integer} :> {a, b - 1},
                     Position[
                        usagecell,
                        Cell[ _, "ModInfo" | "UsageModInfo", ___]
                     ]
                        /. {a__, b_Integer} :> {a, b + 1}
                  ]
               ],
                  Cell[
                     BoxData[ RowBox[{name | ButtonBox[name, __], "[", __}] ],
                     "InlineFormula",
                     a___ /; Not @ MemberQ[{a}, "TemplateExclusion"]
                  ],
               Infinity
            ]
         , 
            Print["No usage cell for "<>ToString[name]]; {}
         ], 
         Cases[ nbex2,
            a : Cell[_, "Notes", ___] :>
               Cases[ a,
                  Cell[BoxData[RowBox[{name | ButtonBox[name, __], "[", __}]],
                     "FunctionTemplate" | "InlineFormula", 
                     b___ /; MemberQ[{b}, "TemplateInclusion"]
                  ],
                  Infinity
               ],
            Infinity
         ],
         Cases[ nbex2,
            Cell[BoxData[RowBox[{name | ButtonBox[name, __], "[", __}]],
               __,
               CellLabel -> "Additional Function Template",
               ___
            ],
            Infinity
         ]
      ]
   ] /. ButtonBox[n_, __] :> n)
]

GetTemplateCells[ path_String] :=
Module[{
},
   GetTemplateCells[
      Quiet[Get[path]],
      StringReplace[ path, {DirectoryName[ path] -> "", ".nb" -> ""}]
   ]
];

(* Clean *)
GetTemplateCells[___]:= $Failed;

GetTemplateCellsWithUsage::nusage = "No usage cell with one or more ModInfo cells found for `1`."

GetTemplateCellsWithUsage[ nbex_Notebook, name_] :=
Module[{
	nbex2=DeleteCases[nbex, c:Cell[___, "FutureExample" | "TentativeExample"| "ExcludedMaterial", ___] /; FreeQ[c, "ModInfo", Infinity], Infinity],
   preusagecell1,
   postusagecell,
   usagecell, 
   ff
},

preusagecell1 = ReplaceAll[Cases[ nbex2, Cell[a_/;Not@FreeQ[a, Cell[_, "ModInfo",___]], "Usage", ___], Infinity], 
 Cell[BoxData[FormBox[StyleBox[DynamicBox[ToBoxes[If[$OperatingSystem === "MacOSX", "Return", "Enter"], StandardForm], 
                                          ImageSizeCache -> _], "KeyEvent"], TraditionalForm]], NotebookDefault, 
      ShowStringCharacters -> False] -> StyleBox["Enter", "KeyEvent"]];
      
preusagecell1 = preusagecell1 /. Cell[BoxData[TooltipBox[Cell[v___, "ModInfo",vv___],___]]] :> Cell[v, "ModInfo", vv];

   name
      ->
   (Flatten[
      Join[
         If[ preusagecell1 =!= {} && ((usagecell = ReduceCell[preusagecell1[[1]]]) =!= Cell[TextData[{}], "Usage"]),
            postusagecell=Cases[usagecell, _TextData][[1]] /. TextData -> Identity;
            postusagecell=Map[Cell[BoxData[RowBox[#]], "FunctionTemplateUsage"]&,Split[postusagecell, Not[MatchQ[#2, Cell[___, "ModInfo" | "UsageModInfo", ___]]] &]];
            postusagecell=Delete[ postusagecell, 
                  Union[
                     Position[
                        postusagecell,
                        x_String /; StringMatchQ[x, "*\[LineSeparator]*"]
                     ]
                        /. {a__, b_Integer} :> {a, b - 1},
                     Position[
                        postusagecell,
                        Cell[ _, "ModInfo" | "UsageModInfo", ___]
                     ]
                        /. {a__, b_Integer} :> {a, b + 1}
                  ]
               ];
             postusagecell=Delete[ postusagecell, 
                  Position[
                        postusagecell,
                        Cell[ _, "ModInfo" | "UsageModInfo", ___]
                     ]
                        /. {a__, b_Integer} :> {a, b}
               ];
             postusagecell = postusagecell/. x_String /; StringMatchQ[x, "*\n*" | "*\[LineSeparator]*"] :> 
 StringReplace[x, {"\n" -> "", "\[LineSeparator]" -> ""}]
         , 
            Print["No usage cell for "<>ToString[name]]; {}
         ], 
         DeleteCases[
Cases[nbex2, 
  a : Cell[_, "Notes", ___] :> 
   Cases[a, 
    w : TextData[{ww___, 
        Cell[BoxData[
          RowBox[{name | ButtonBox[name, __], 
            "[", r__}]], "FunctionTemplate" | "InlineFormula", 
         b___ /; MemberQ[{b}, "TemplateInclusion"]], zz___}] :> 
     Cell[BoxData[RowBox[{ww, Cell[BoxData[RowBox[{name, "[", r}]], "InlineFormula"], zz}]], "FunctionTemplateUsage"], Infinity], Infinity],
 ButtonBox[___, Rule[BaseStyle, "ExampleLink"], ___], Infinity],
         Cases[ nbex2,
            Cell[BoxData[RowBox[{name | ButtonBox[name, __], "[", __}]],
               __,
               CellLabel -> "Additional Function Template",
               ___
            ]:>Cell[BoxData[RowBox[{"no usage information for this particular functionality."}]], "FunctionTemplateUsage"],
            Infinity
         ]
      ]
   ] /. ButtonBox[n_, __] :> n)
]

GetTemplateCellsWithUsage[ path_String] :=
Module[{
},
   GetTemplateCellsWithUsage[
      Quiet[Get[path]],
      StringReplace[ path, {DirectoryName[ path] -> "", ".nb" -> ""}]
   ]
];

(* Clean *)
GetTemplateCellsWithUsage[___]:= $Failed;


GetUsageFromUsageCell[expr_] := 
  Module[{}, 
   ReplaceAll[
    expr, {a___, Cell[_, "ModInfo" | "UsageModInfo", ___], c___, 
      x_String /; StringMatchQ[x, "*\[LineSeparator]*"], f___} :> {x, 
      f}]];

GetUsageFromNoteCell[expr_, name_] := 
  Module[{}, 
   First[Cases[expr, 
     w : TextData[{ww___, 
         Cell[BoxData[RowBox[{name | ButtonBox[name, __], "[", r__}]],
           "FunctionTemplate" | "InlineFormula", 
          b___ /; MemberQ[{b}, "TemplateInclusion"]], zz___}] :> 
      Cell[BoxData[
        RowBox[{ww, 
          Cell[BoxData[RowBox[{name, "[", r}]], "InlineFormula"], 
          zz}]], "FunctionTemplateUsage"], Infinity]]];

ExtractTemplatesAndUsagesFromUsageCell[expr_, name_] := 
  Module[{pos}, 
   pos = Position[expr, 
     Cell[BoxData[RowBox[{name | ButtonBox[name, __], "[", __}]], 
      "InlineFormula", 
      a___ /; Not@MemberQ[{a}, "XXXXTemplateExclusionXXXX"]]]; 
   If[Length[pos] > 0, 
    Table[{Extract[expr, pos[[i]]], 
      GetUsageFromUsageCell[Extract[expr, First[pos[[i]]]]]}, {i, 1, 
      Length[pos]}], {}]];

ExtractTemplatesAndUsagesFromNoteCell[expr_, name_] := 
  Module[{pos}, 
   pos = Position[expr, 
     Cell[BoxData[RowBox[{name | ButtonBox[name, __], "[", r__}]], 
      "FunctionTemplate" | "InlineFormula", 
      b___ /; MemberQ[{b}, "TemplateInclusion"]]]; 
   If[Length[pos] > 0, 
    Table[{Extract[expr, pos[[i]]], 
      GetUsageFromNoteCell[expr, name]}, {i, 1, Length[pos]}]]];

ExtractTemplatesAndUsagesFromNotebooksNoteCell[nbexpr_, name_] := 
  Module[{}, 
   DeleteCases[
    Cases[Cases[nbexpr, 
      a : Cell[_, "Notes", ___] :> 
       Cases[a, 
        w : TextData[{ww___, 
            Cell[BoxData[
              RowBox[{name | ButtonBox[name, __], "[", r__}]], 
             "FunctionTemplate" | "InlineFormula", 
             b___ /; MemberQ[{b}, "TemplateInclusion"]], zz___}] :> 
         ExtractTemplatesAndUsagesFromNoteCell[a, name], Infinity], 
      Infinity], {aa_, bb_} :> {aa, bb}, 2], 
    ButtonBox[___, Rule[BaseStyle, "ExampleLink"], ___], Infinity]];

ExtractTemplatesAndUsagesFromNotebooksAdditionalFunctionTemplateCell[
   nbexpr_, name_] := 
  Module[{}, 
   Cases[nbexpr, 
    a : Cell[
       BoxData[RowBox[{name | ButtonBox[name, __], "[", __}]], __, 
       CellLabel -> "Additional Function Template", ___] :> {a, 
      Cell[BoxData[
        RowBox[{"no usage information for this particular \
functionality."}]], "FunctionTemplateUsage"]}, Infinity]];


GetTemplateCellsWithUsage2::nusage = "No usage cell with one or more ModInfo cells found for `1`."

GetTemplateCellsWithUsage2[ nbex_Notebook, name_] :=
Module[{
	nbex2=DeleteCases[nbex, c:Cell[___, "FutureExample" | "TentativeExample"| "ExcludedMaterial", ___] /; FreeQ[c, "ModInfo", Infinity], Infinity],
   preusagecell1,
   postusagecell,
   usagecell, 
   ff
},

preusagecell1 = ReplaceAll[Cases[ nbex2, Cell[a_/;Not@FreeQ[a, Cell[_, "ModInfo",___]], "Usage", ___], Infinity], 
 Cell[BoxData[FormBox[StyleBox[DynamicBox[ToBoxes[If[$OperatingSystem === "MacOSX", "Return", "Enter"], StandardForm], 
                                          ImageSizeCache -> _], "KeyEvent"], TraditionalForm]], NotebookDefault, 
      ShowStringCharacters -> False] -> StyleBox["Enter", "KeyEvent"]];
      
preusagecell1 = preusagecell1 /. Cell[BoxData[TooltipBox[Cell[v___, "ModInfo",vv___],___]], ___] :> Cell[v, "ModInfo", vv];

   name
      ->
   Partition[(Flatten[
      If[ preusagecell1 =!= {} && ((usagecell = ReduceCell[preusagecell1[[1]]]) =!= Cell[TextData[{}], "Usage"]),
            postusagecell=Cases[usagecell, _TextData][[1]] /. TextData -> Identity;
            postusagecell=Map[Cell[BoxData[RowBox[#]], "FunctionTemplateUsage"]&,Split[postusagecell, Not[MatchQ[#2, Cell[___, "ModInfo" | "UsageModInfo", ___]]] &]];
             postusagecell = ExtractTemplatesAndUsagesFromUsageCell[postusagecell, name];
             postusagecell = postusagecell/. x_String /; StringMatchQ[x, "*\n*" | "*\[LineSeparator]*"] :> 
 StringReplace[x, {"\n" -> "", "\[LineSeparator]" -> ""}]
         , 
            Print["No usage cell for "<>ToString[name]]; {}
         ]
   ] /. ButtonBox[n_, __] :> n), 2]
];

GetTemplateCellsWithUsage2[ path_String] :=
Module[{
},
   GetTemplateCellsWithUsage2[
      Quiet[Get[path]],
      StringReplace[ path, {DirectoryName[ path] -> "", ".nb" -> ""}]
   ]
];

(* Clean *)
GetTemplateCellsWithUsage2[___]:= $Failed;


CollectTemplates[dir_String /; FileType[dir] === Directory] := GetTemplateCells /@ FileNames["*.nb", {dir}]


(********************************************
 Oyvind's code for generating Argument patterns from Syntax Templates
*********************************************)

(* Trick to avoid annoying contexts for pattern names being introduced below *)

placeholder = Global`placeholder;
options = Global`options;

(* Various utility functions for doing the conversion and merging of templates *)

CleanUpTemplate[s_]:=StringReplace[s,{":>"->":>","->"->"->",Characters["\n \"`:\\+=->/$"]->""}]

ConvertTemplate[s_] /; $JerrysWay := ConvertTemplateAlaJerry[s]

ConvertTemplate[s_] :=
  StringReplace[
    s, {RegularExpression["(\\w+),((\\w+),)*\[Ellipsis]+"] :> "$1..",
      RegularExpression[",([^{},]+),([^{}]*,)*\[Ellipsis]+"] :> ",($1)...",
(* comma, then one or more nonlists separated by commas, then ellipsis *)
      RegularExpression[",([^{}]*?)\[Ellipsis]+"] :> ",placeholder...",
      "\[Ellipsis]" -> "placeholder...", "..."->"placeholder...","." -> ""}]

ConvertTemplateAlaJerry[s_] :=
  StringReplace[
    s, {RegularExpression["((\\w+),)*(\\w+),\[Ellipsis]+"] :> "$1$3..",
      RegularExpression[",([^{},]+),([^{}]*,)*\[Ellipsis]+"] :> ",($1)...",
(* comma, then one or more nonlists separated by commas, then ellipsis *)
      RegularExpression[",([^{}]*?)\[Ellipsis]+"] :> ",placeholder...",
      "\[Ellipsis]" -> "placeholder...", "..."->"placeholder...","." -> ""}]

GetArgs[s_] :=
  StringCases[s,
      StringExpression[title : (WordCharacter ..),
          args : StringExpression["[", ___, "]"]] :> args][[1]]

TranslateArgs[args_String] :=
ToExpression[
   StringJoin["{", StringReplace[ args, Characters["[]"] -> ""], "}"]
] /.
{(Rule | RuleDelayed)[a_, b_] :> Global`rule}/.
{
   List -> List,
   HoldPattern[Repeated][x_] :> Hold[Repeated][x],
   HoldPattern[RepeatedNull][x_] :> Hold[RepeatedNull][x],
   x_Symbol :> Hold[Pattern][x, Blank[]],
   x_Integer :> Hold[Pattern][placeholder, Blank[]]
}

MergeArgs[args_List]:=
If[ Length[args]==1
, (* then *)
   args[[1]]
, (* else *)
   Module[{len=Length/@args, listq, rep, lists, el, opt, $$j,n,repat=-1},
      Take[Table[
         If[repat>0,Null  (* if a Repeated has been found, no point in any more patterns *),
         listq = True;
            rep = False;
            lists = {};
            el = Null;
            opt = False;
            n = 0;
            Do[
               If[ Length[ args[[$$j]] ] >= $$i
               , (* then *)
                  arg = args[[$$j,$$i]];
                  n++;
                  If[ !ListQ[arg] && listq && !rep,
                     listq = False;
                     el = arg
                  ];
               If[ Head[ arg] === Repeated || Head[ arg] === RepeatedNull,
                     listq = False;
                     rep = True;
                     repat = $$i;
                     el = arg
                  ];
                  If[ ListQ[ arg] && listq, AppendTo[ lists, arg]];
                  If[ el === Null, el = arg]
               , (* else *)
                  opt = True
               ],
               {$$j, Length[args]}
            ];
            If[ listq, el = MergeArgs[lists]];
            If[ rep && n > 1, el[[1]] = placeholder];
            If[ opt,
                Switch[Head[el],
                    Repeated, el[[0]]=RepeatedNull,
                    RepeatedNull, el = el,
                    _, el = Optional[el]
                ]
            ];
            el],
         {$$i, Max[ len]}
      ], {1,repat}]
   ]
]

AddOptions[fun_, patt_, contextprefix_, opts_:Automatic]:= Module[{iOpts},
   iOpts = If[opts === Automatic, getoptionnames[fun, contextprefix], opts];
   If[!MatchQ[iOpts, {} | None],
      Append[patt,OptionsPattern[]]
   , (* else *)
      ToExpression[fun<>";Print[Options[Unevaluated["<>fun<>"]]]"];patt
   ] /.
          {HoldPattern[Optional][
               HoldPattern[Pattern][options,Blank[]]]:>
            OptionsPattern[],
         HoldPattern[Pattern][options,Blank[]]:>
            OptionsPattern[]}
]

FinalCleanUp[patt_] := patt//.
   {HoldPattern[Repeated][a_Symbol] :> xPattern[a,__],
    HoldPattern[RepeatedNull][a_Symbol] :> xPattern[a,___],
    HoldPattern[Pattern][placeholder, p_] :> p,
    HoldPattern[Pattern][_, p_] :> p (* this should clear all Patterns *),
    {a___,HoldPattern[Blank][], HoldPattern[BlankNullSequence][],b___}:>{a,__,b},
    HoldPattern[Pattern][Element, _][__] :> Blank[],
    HoldPattern[Pattern][Alternatives, _][__] :> Blank[],
    HoldPattern[Pattern][Association, _][__] :> Blank[],
    xPattern -> Pattern}



(* The following function cleans up the box structure of a cell to generate
a simple string, it might be fragile and need modification if more general
boxes start appearing in the templates *)

TemplateToString[tcell_Cell]:=
 StringJoin[(tcell[[1]]//.
 	{BoxData->List,TextData->List,RowBox->List,SubscriptBox->List,SuperscriptBox->List,
      StyleBox[s_,__]:>s, ButtonBox[s_,___]:>s})
      /.s_String:>ToString[s]  (*To get rid of linear syntax *)]

(* This function takes the output from GetTemplates and returns the
corresponding entry for FunctionTemplates.m. If a problem occurs,
a message is Print'ed and no template is generated for that function
(this indicates a problem in the original documentation notebook) *)

SyntaxTemplatesToSyntaxPattern[function_String->cells_, opts_, contextprefix_:""]:=
 TimeConstrained[Check[Module[{$$data}, 
 	Print["Processing "<> function];
 	$$data = cells;
    If[cells==={},Return[{}]];
    $$data = ConvertTemplate/@CleanUpTemplate/@TemplateToString/@cells;
    $$data=ReleaseHold[TranslateArgs/@GetArgs/@$$data];
    $$data=MergeArgs[$$data];
    $$data = AddOptions[function, $$data, contextprefix, opts];
    $$data = FinalCleanUp[$$data];
    {function,$$data}],
   Print["Messages generated when processing "<>function<>". This function will be skipped."];{}],
   30,Print["Time-out error in "<>function];{}]

(*
SyntaxTemplatesToSyntaxPattern[$Failed, r__]:=
  (Print["SyntaxTemplatesToSyntaxPattern failure: "<>ToString[r] ]; {});

SyntaxTemplatesToSyntaxPattern[r___]:=
  (Print["SyntaxTemplatesToSyntaxPattern failure: "<>ToString[r] ]; {});
*)
(* This function takes as input a list of directories containing documentation notebooks,
and outputs a FunctionTemplates.m file to fileout.
It tries to read the context from the directory, and if the context is not System,
a context` is prepended to each function name.
See comments for previous functions for what happens in cases of errors in the source *)

CreateFunctionTemplatesFile[dirs_List,fileout_String]:=
Module[{context,prefix,res},
  res=Flatten[(
    context=
      StringCases[#,$PathnameSeparator~~(c:WordCharacter..)~~($PathnameSeparator|"")~~EndOfString:>c];
    If[context==={},Print["No context found!"];context="System", context=context[[1]]];
       prefix=If[context==="System","",context<>"`"];
       {prefix<>#[[1]],#[[2]]}&/@
        DeleteCases[
         Function[file,
           SyntaxTemplatesToSyntaxPattern[
           TimeConstrained[GetTemplateCells[file],30,
           Print["Time-out error in reading "<>file];file->{}], Automatic]]/@
          FileNames["*.nb",{#}],{}
      ]
    )&/@dirs,1];
  Put[res,fileout]]

(* This signature added by tgayley for building the FunctionInformation.m file for a function paclet
   built from the author template notebook.
*)
CreateFunctionTemplatesFile[context_String, symbolNameAndNotebookExprPairs:{{_String, _Notebook}...}, fileout_String] :=
    Module[{syntaxInfo},
        syntaxInfo =
            Function[{symbolName, nbExpr},
                SyntaxTemplatesToSyntaxPattern[GetTemplateCells[nbExpr, symbolName], Automatic]
            ] @@@ symbolNameAndNotebookExprPairs;
        syntaxInfo = DeleteCases[syntaxInfo, {}];
        If[MatchQ[syntaxInfo, {{_String, _List}...}],
            Put[{{context, syntaxInfo}}, fileout],
        (* else *)
            $Failed
        ]
    ]


(* Code for generating FunctionInformation.m *)

GetSyntaxTemplates[nbex_Notebook]:=
  Replace[ToExpression[
    If[# === {}, None, #[[1]]] &[
      Cases[nbex, Cell[data_, __, CellLabel -> #, ___] :> data,Infinity, 1]]],
    Null -> None] & /@
      {"Arguments Pattern", "Option Names", "Local Variables", "Color Equal Signs"}


contextreplacementexceptions={"BoxForm`ActionFunction" -> "ActionFunction", 
 "BoxForm`AllowDebugging" -> "AllowDebugging", 
 "BoxForm`CellChangeTimeMergeInterval" -> 
  "CellChangeTimeMergeInterval", 
 "BoxForm`CellEditDuplicateMakesCopy" -> "CellEditDuplicateMakesCopy",
  "BoxForm`CellSplitCopiesTags" -> "CellSplitCopiesTags", 
 "BoxForm`ClickRadius" -> "ClickRadius", 
 "BoxForm`ClosingSaveDialog" -> "ClosingSaveDialog", 
 "BoxForm`CreateCellID" -> "CreateCellID", 
 "BoxForm`DefaultNewGraphics" -> "DefaultNewGraphics", 
 "BoxForm`DefaultReturnCreatedCellStyle" -> 
  "DefaultReturnCreatedCellStyle", 
 "BoxForm`GenerateImageCachesOnPlacement" -> 
  "GenerateImageCachesOnPlacement", 
 "BoxForm`GreekStyle" -> "GreekStyle", 
 "BoxForm`GroupName" -> "GroupName", 
 "BoxForm`LegacyGraphicsCompatibility" -> 
  "LegacyGraphicsCompatibility", 
 "BoxForm`LocatorAutoReselect" -> "LocatorAutoReselect", 
 "BoxForm`LowerCaseStyle" -> "LowerCaseStyle", 
 "BoxForm`MultiscriptBoxOptions" -> "MultiscriptBoxOptions", 
 "BoxForm`Name" -> "Name", 
 "BoxForm`PreserveOldOutputGraphicsAttributes" -> 
  "PreserveOldOutputGraphicsAttributes", 
 "BoxForm`RecursionLimit" -> "RecursionLimit", 
 "BoxForm`RenderingTarget" -> "RenderingTarget", 
 "BoxForm`ReturnCreatesNewCell" -> "ReturnCreatesNewCell", 
 "BoxForm`SpeechNavigation" -> "SpeechNavigation", 
 "BoxForm`SpeedOfTime" -> "SpeedOfTime", 
 "BoxForm`StripStyleOnPaste" -> "StripStyleOnPaste", 
 "BoxForm`StyleEnvironment" -> "StyleEnvironment", 
 "BoxForm`TemporaryControlActiveInterval" -> 
  "TemporaryControlActiveInterval", 
 "BoxForm`TrackCellChangeTimes" -> "TrackCellChangeTimes", 
 "BoxForm`WindowPersistentStyles" -> "WindowPersistentStyles", 
 "Charting`DateFormat" -> "DateFormat", 
 "Combinatorica`Type" -> "Type", 
 "Control`CommonDump`DescriptorDynamics" -> "DescriptorDynamics", 
 "Control`FeedbackSector" -> "FeedbackSector", 
 "Control`FeedbackSectorStyle" -> "FeedbackSectorStyle", 
 "Geodesy`Radius" -> "Radius", 
 "GraphComputation`LayerSizeFunction" -> "LayerSizeFunction", 
 "GraphComputation`MultiedgeStyle" -> "MultiedgeStyle", 
 "GraphComputation`PackingMethod" -> "PackingMethod", 
 "GraphComputation`SelfLoopStyle" -> "SelfLoopStyle", 
 "Internal`ColorRules" -> "ColorRules", 
 "Internal`LineNumbers" -> "LineNumbers", 
 "JLink`AllowShortContext" -> "AllowShortContext", 
 "JLink`ForceLaunch" -> "ForceLaunch", 
 "JLink`StaticsVisible" -> "StaticsVisible", 
 "Parallel`Developer`Scheduling" -> "Scheduling", 
 "SymbolicC`Private`Embedded" -> "Embedded", 
 "System`BarSpacing" -> "BarSpacing", 
 "System`ConfidenceLevel" -> "ConfidenceLevel", 
 "System`EdgeStyle" -> "EdgeStyle", 
 "System`EdgeWeight" -> "EdgeWeight", 
 "System`PlaceholderReplace" -> "PlaceholderReplace", 
 "System`ReferenceLineStyle" -> "ReferenceLineStyle", 
 "System`SignificanceLevel" -> "SignificanceLevel", 
 "System`VertexStyle" -> "VertexStyle", 
 "System`VertexWeight" -> "VertexWeight", 
 "System`Weights" -> "Weights"};

getoptionnames[
   "PalettePut" | "NotebookPut" | "System`PalettePut" | "System`NotebookPut", ___
] :=
getoptionnames["System`Notebook"]
(* Options does not work appropriately for these, but according
to the function pages, they can take any Notebook option. *)

getoptionnames[ "NotebookRead" | "System`NotebookRead", ___ ] =
{"\"WrapBoxesWithBoxData\""}
(* Options[NotebookRead] gives an empty list, but NotebookRead does
accept this undocumented string-named option.
This special-case definition does not mean that the string "WrapBoxesWithBoxData"
will be colored like a legal option name _symbol_ would be.  As a
string, it is currently colored gray rather than black.
The important thing is to get NotebookRead's function information
entry to show that it may take an option.
*)

getoptionnames[ "KillProcess" | "System`KillProcess" | "ProcessConnection" | "System`ProcessConnection" | "Processes" |  "System`Processes" |  "ProcessInformation" | "System`ProcessInformation" |  "ProcessObject" | "System`ProcessObject" |  "ProcessReadBuffer" | "System`ProcessReadBuffer" |  "ProcessStatus" | "System`ProcessStatus" |  "RunProcess" |  "System`RunProcess" | "StartProcess" |  "System`StartProcess", ___ ] = {}
(* This was added because it acts unpredictably on Windows *)


getoptionnames[str_String, contextprefix_:""]:=
(Print[contextprefix<>str];ToString[First[#], InputForm]&/@If[MatchQ[#,{OptionsPattern[]}],#,{}]&@
  ToExpression[contextprefix<>str<>";Options[Unevaluated["<>contextprefix<>str<>"]]"])


CategorizationFromFunctionPage[ nbex_Notebook, categLabel_String] :=
Replace[
  Cases[ nbex,
    Cell[ categString_,
      "Categorization",
      opts___ /; Replace[ CellLabel, {opts}] === categLabel
    ] :> categString,
    -1,
    1
  ],
{
  {found_String /; StringLength @ found > 0} :> found,
  _ -> None
}]

ContextFromFunctionPage[ nbex_Notebook] :=
Replace[ CategorizationFromFunctionPage[ nbex, "Context"], {
  s_String /; StringMatchQ[ s, "*`"] :> s,
  _ -> "System`"
}]

FunctionInformationEntry[ nbex_Notebook, name_String, contextprefix_String:""] :=
Module[{argpatt, locvar, coleqsign, opts, ft, res},
  {argpatt, opts, locvar, coleqsign} = GetSyntaxTemplates @ nbex;
  Print[{name, opts}];
  If[opts===None || !FreeQ[opts, Automatic],
    opts=Flatten[opts/.(None|Automatic)->StringReplace[getoptionnames[contextprefix<>name], contextreplacementexceptions]]];
  opts = Replace[opts, {}->None];

  If[ argpatt === None,
    argpatt =
      Replace[
        SyntaxTemplatesToSyntaxPattern[ GetTemplateCells[ nbex, name], opts, contextprefix ],
        {} -> {name,None}
      ][[2]]
  ];
 
  res={name,argpatt, opts,locvar,coleqsign};
  While[res[[-1]]===None,res=Drop[res,-1]];
  If[Length[res]<2,{},res]
]


FunctionInformationEntryWithUsage[ nbex_Notebook, name_String, contextprefix_String:""] :=
Module[{argpatt, locvar, coleqsign, opts, ft, res, ftwu, functiontemplatecellswithusage, functiontemplatecells, usagecells},
  {argpatt, opts, locvar, coleqsign} = GetSyntaxTemplates @ nbex;
  If[ argpatt === None,
    argpatt =
      Replace[
        SyntaxTemplatesToSyntaxPattern[ GetTemplateCells[ nbex, name], Automatic, contextprefix ],
        {} -> {name,None}
      ][[2]]
  ];
  
  functiontemplatecellswithusage = GetTemplateCellsWithUsage2[ nbex, name][[2]];
  
  If[functiontemplatecellswithusage === {}, 
  	functiontemplatecells = None;
  	usagecells = None, 
  	functiontemplatecells = functiontemplatecellswithusage[[All, 1]];
  	usagecells = functiontemplatecellswithusage[[All, 2]];
  	];
  
  If[ functiontemplatecells =!= None && ListQ[functiontemplatecells], 
  	If[Length[functiontemplatecells]>0, 
  		functiontemplatecells=Table[MapAt[GridBox[{{#}}] &, 
  			functiontemplatecells[[i]], {{1, 1}}], {i, 1, Length[functiontemplatecells]}]]];
  
  If[ usagecells =!= None && ListQ[usagecells], 
  	If[Length[usagecells]>0, 
  		usagecells = usagecells]];

  If[opts===None || !FreeQ[opts, Automatic],
    opts=Flatten[opts/.(None|Automatic)->StringReplace[getoptionnames[contextprefix<>name], contextreplacementexceptions]]];
  opts = Replace[opts, {}->None];
  
  res={name,argpatt,functiontemplatecells, usagecells, opts,locvar,coleqsign};
  While[res[[-1]]===None,res=Drop[res,-1]];
  If[Length[res]<2,{},res]
]


(* Special exception cases not handled in documentation notebooks *)
AddFISpecialCases[ context_String, data_List] :=
Replace[ context, {
  "System`" :>
    Join[
      data,
      {
        {"\[Integral]", None,None,{"IntegralSign", {2}}},
        {"\[Sum]", None, None,{"SumSign", {1}}},
        {"\[Product]", None, None, {"SumSign", {1}}}
      }
    ],
  _ :> data
}]


AddFISpecialCasesWithUsage[ context_String, data_List] :=
Replace[ context, {
  "System`" :>
    Join[
      data,
      {
        {"\[Integral]", None, {}, {}, None,{"IntegralSign", {2}}},
        {"\[Sum]", None, {}, {}, None,{"SumSign", {1}}},
        {"\[Product]", None, {}, {}, None, {"SumSign", {1}}}
      }
    ],
  _ :> data
}]

(* Explicit list of functions for which to suppress options, because an
   optional argument can also be a list.
   Now defunct.  *)
(*$SuppressFunctionOptions = {"Trace","StringSplit"};  *)


CreateFunctionInformationFile[ dirAndFilesPairs:{___List}, fileout_String]:=
Put[
  Apply[
    {#1, Union[AddFISpecialCases[ #1, #2]]} &,
    Last[
      Reap[
        Scan[
          Function[{file},
            Replace[
              TimeConstrained[
                DocumentationBuild`Utils`BlockMessageOff[{
                  Syntax::newl
                },
                  Get @ file
                ],
                30,
                Print["Time-out error in reading " <> file]; $Failed
              ],
            {
              gotfile_Notebook :>
                Block[{
                  contextFromFunctionPageResult =
                    ContextFromFunctionPage @ gotfile,
                  pacletFromFunctionPageResult =
                    CategorizationFromFunctionPage[ gotfile, "Paclet Name"]
                }, 
(* Do not use package function pages for System` function information: *)
                  If[
                    pacletFromFunctionPageResult === "Mathematica" ||
                    contextFromFunctionPageResult =!= "System`"
                  ,
                    Needs[ contextFromFunctionPageResult ];
                    Replace[
                      FunctionInformationEntry[
                        gotfile,
                        StringReplace[ file, {
                          DirectoryName @ file -> "",
                          ".nb" -> ""
                        }], contextFromFunctionPageResult
                      ],
                    {
                      l:{__} :>
                        Sow[ l,
                          Null[
                            CreateFunctionInformationFile,
                            contextFromFunctionPageResult
                          ]
                        ]
                    }]
                  ]
                ]
            }]
          ],
          Flatten @ dirAndFilesPairs[[ All, 2]]
        ],
        Null[ CreateFunctionInformationFile, _],
        {Last @ #1, #2}&
      ]
    ],
    1
  ],
  fileout
]

(* Function that takes the output of CreateFunctionInformationFileWithUsage and  *)
MakeTemplatesNotebook[data_List] :=
  Module[{newdata, cells = {}, name, templates, usages},
  	newdata = SelectFirst[Rest[data], StringMatchQ[First[#], "System`"]& ][[2]];
  	(name = #[[1]]; templates = #[[3]]; usages = #[[4]]; 
     templates = 
     Append[#, CellTags -> name <> "_templates"] & /@ templates;
     usages = Append[#, CellTags -> name <> "_usages"] & /@ usages;
     AppendTo[cells, {templates, usages}])& /@ newdata;
     
     Notebook[Flatten[cells]]  	
  ];

CreateFunctionInformationFile[dirs:{___String}, fileout_String]:=
CreateFunctionInformationFile[{#, FileNames["*.nb", {#}]}& /@ dirs, fileout]


(* 7/17/14: Edited below to call MakeTemplatesNotebook and Put the resulting notebook *)
CreateFunctionInformationFileWithUsage[ dirAndFilesPairs:{___List}, fileout_String]:=
Module[{data, dir = DirectoryName[fileout]}, 
 Put[
  Compress[data = Join[{9}, Apply[
    {#1, Union[AddFISpecialCasesWithUsage[ #1, #2]]} &,
    Last[
      Reap[
        Scan[
          Function[{file},
            Replace[
              TimeConstrained[
                DocumentationBuild`Utils`BlockMessageOff[{
                  Syntax::newl
                },
                  Get @ file
                ],
                30,
                Print["Time-out error in reading " <> file]; $Failed
              ],
            {
              gotfile_Notebook :>
                Block[{
                  contextFromFunctionPageResult =
                    ContextFromFunctionPage @ gotfile,
                  pacletFromFunctionPageResult =
                    CategorizationFromFunctionPage[ gotfile, "Paclet Name"]
                },
              
(* Do not use package function pages for System` function information: *)
                  If[
                    pacletFromFunctionPageResult === "Mathematica" ||
                    contextFromFunctionPageResult =!= "System`"
                  ,
                    Needs[ contextFromFunctionPageResult ];
                    Replace[
                      FunctionInformationEntryWithUsage[
                        gotfile,
                        StringReplace[ file, {
                          DirectoryName @ file -> "",
                          ".nb" -> ""
                        }], contextFromFunctionPageResult
                      ],
                    {
                      l:{__} :>
                        Sow[ l,
                          Null[
                            CreateFunctionInformationFileWithUsage,
                            contextFromFunctionPageResult
                          ]
                        ]
                    }]
                  ]
                ]
            }]
          ],
          Flatten @ dirAndFilesPairs[[ All, 2]]
        ],
        Null[ CreateFunctionInformationFileWithUsage, _],
        {Last @ #1, #2}&
      ]
    ],
    1
  ]]],
  fileout
];

Export[FileNameJoin[{dir, "SystemFunctionTemplates.nb"}], MakeTemplatesNotebook[data]]
]

CreateFunctionInformationFileWithUsage[dirs:{___String}, fileout_String]:=
CreateFunctionInformationFileWithUsage[{#, FileNames["*.nb", {#}]}& /@ dirs, fileout]


(* Code for adding syntax information to packages *)
SyntaxInformationDataEntry[nbex_Notebook, name_String, contextprefix_:"", optnames_:True]:=
 Module[{argpatt,locvar,coleqsign,opts,opts1,res},
  {argpatt,opts1, locvar,coleqsign}=GetSyntaxTemplates[nbex];
  If[argpatt===None,
   argpatt=Replace[
      SyntaxTemplatesToSyntaxPattern[
       GetTemplateCells[nbex,name], Automatic],{}->{name,None}][[2]]];
  opts = opts1;
  If[(optnames && opts===None) || !FreeQ[opts, Automatic],
    defopts = getoptionnames[contextprefix<>name];
    opts=Flatten[opts/.(None|Automatic)->defopts]];
  If[opts1 =!= {}, opts = Replace[opts, {}->None]];
  {ToExpression[name],
  DeleteCases[{"ArgumentsPattern"->argpatt,"OptionNames"->opts,
               "LocalVariables"->locvar,"ColorEqualSigns"->coleqsign},
     _->None]}]

Options[AddPackageSyntaxInformation] = {ExplicitOptionNames -> True};
AddPackageSyntaxInformation[functionfiles_List,packagefile_String, context_:"", OptionsPattern[]]:=
 Catch[Module[{res,file, pos,pos2,out},
   res=DeleteCases[
     Function[file,
       TimeConstrained[
        SyntaxInformationDataEntry[DocumentationBuild`Utils`BlockMessageOff[{Syntax::newl},Get@file],
         StringReplace[file,{DirectoryName[file]\[Rule]"",".nb"\[Rule]""}],
         context, OptionValue[ExplicitOptionNames]],
        30,Print["Time-out error in reading "<>file];{}]]/@
      functionfiles,{_,{}}];
   If[FileType[packagefile]=!=File,
    Print["Package file not found, returning raw $SyntaxInformationData"];
    Throw[res]];
   file=ReadList[packagefile,String];
   pos=Position[
     StringCases[file,
      StartOfString~~"System`Utilities`SetSyntaxInformation["],Except[{}],{1},1,Heads\[Rule]False];
   If[pos==={},
    Print["Couldn't find System`Utilities`SetSyntaxInformation declaration in file! Returning raw $data"];
    Throw[res]];
   pos=pos[[1,1]];
   pos2=pos+1;
   While[pos2<Length[file]&&file[[pos2]]=!="",pos2++];
   If[pos2>Length[file],
    Print["Couldn't find empty line after System`Utilities`SetSyntaxInformation declaration! Returning raw data"];
    Throw[res]];
   out=OpenWrite[packagefile,CharacterEncoding\[Rule]{}];
   Do[WriteString[out,file[[i]]<>"\n"],{i,pos-1}];
   WriteString[out,"System`Utilities`SetSyntaxInformation["<>ToString[res,InputForm]<>"];\n"];
   Do[WriteString[out,file[[i]]<>"\n"],{i,pos2,Length[file]}];
   Close[out]
   ]]




(********************************************
End of Oyvind's code
*********************************************)

(* Code for CreateUsageMessageFile: *)

transf1=x_String/;
    StringMatchQ[x,"\"\!\(\*\nStyleBox[\""~~a__~~"\",\n"~~b__~~"]\)\""]:>
   StringReplace[x,
    "\"\!\(\*\nStyleBox[\""~~a__~~"\",\n"~~b__~~"]\)\"":>
     "StyleBox[\""~~"\\\""~~a~~"\\\""~~"\","~~b~~
      ",ShowStringCharacters->True]"];

transf11=x_String\[RuleDelayed]StringReplace[x,{"\*"\[Rule]"","\n"\[Rule]""}];

transf12=x_String/;
    StringMatchQ[x,
     "\"\!\(SubscriptBox["~~__~~"["~~__~~"],"~~ __~~
      "]\)\""]\[RuleDelayed]StringReplace[x,
    "\"\!\(SubscriptBox["~~a__~~"["~~b__~~"],"~~c__~~"]\)\"":>
     "SubscriptBox["~~a~~"["~~b~~"],"~~ c~~"]"];

transf2=x_String/;Not@StringMatchQ[x,"*Box*"]\[RuleDelayed]"\""<>x<>"\"";

transf3=ShortestMatch[
    "SubscriptBox[StyleBox[\""~~a__~~"\",\"TI\"], \""~~b__~~
     "\"]"]\[RuleDelayed]"StyleBox[\"\\\"\\!\\(\\*StyleBox[\\\""~~a~~
    "\\\",\\\"TI\\\"]\\_"~~b~~"\\)\\\"\",ShowStringCharacters->True]";

transf4=ShortestMatch[
    "StyleBox[\"\\\""~~a__~~"\",\"TI\"]\).\!\(StyleBox[\""~~b__~~
     "\\\"\",\"TI\",ShowStringCharacters->True]"]:>
   "StyleBox[\"\\\""~~a~~"."~~b~~"\\\"\",\"TI\",ShowStringCharacters->True]";

transf5=ShortestMatch["\"StyleBox["~~a__~~", TI]."~~b__~~"\""]:>
   "StyleBox[\"\\\"\\!\\(\\*StyleBox[\\\""~~a~~"\\\",\\\"TI\\\"]\\)."~~b~~
    "\\\"\",ShowStringCharacters->True]";

transf6=ShortestMatch[
    "\"StyleBox[StyleBox["~~a__~~
     ", TI] `]\""]\[RuleDelayed]"StyleBox[\"\\\"\\!\\(\\*StyleBox[\\\""~~a~~
    "\\\",\\\"TI\\\"]\\)`\\\"\", ShowStringCharacters->True]";

transf7="\"" ~~ a__?(# =!= "\"" &) ~~ "://StyleBox[" ~~ b__ ~~ ", TI]\"" :>  
   "StyleBox[\"\\\"" ~~ a ~~ "://" ~~ "\\!\\(\\*StyleBox[\\\"" ~~ b ~~   
    "\\\",\\\"TI\\\"]\\)" ~~ "\\\"\",ShowStringCharacters->True]";
    
specialrule1=
  x_String/;StringMatchQ[x,
     "\!\(\*RowBox[{\"Export\", \"[\", RowBox[{StyleBox[\"\[Ellipsis]\", \"TR\"], \
\",\", \"\""~~v:("HTML"|"TeX")~~"\"\"}], \"]\"}]\)"]:>
   StringReplace[x,
    "\!\(\*RowBox[{\"Export\", \"[\", RowBox[{StyleBox[\"\[Ellipsis]\", \"TR\"], \",\", \
\"\""~~a:("HTML"|"TeX")~~"\"\"}], \"]\"}]\)":>
     "\!\(\*RowBox[{\"Export\", \"[\", RowBox[{StyleBox[\"\[Ellipsis]\", \"TR\"], \
\",\", "~~"StyleBox[\""~~"\\\""~~a~~"\\\""~~"\","~~
      ",ShowStringCharacters->True]"~~"}], \"]\"}]\)"];

transf8=ShortestMatch[
    "\"SubscriptBox[StyleBox["~~a__?(# =!= "\"" &)~~", TI], "~~
     b__?(# =!= "\"" &)~~"]`\""]:>
   "StyleBox[\"\\\"\\!\\(\\*StyleBox[\\\""~~a~~"\\\",\\\"TI\\\"]\\_"~~b~~
    "\\)`\\\"\",ShowStringCharacters->True]";
    
transf9=ShortestMatch["\"StyleBox[" ~~ a : (Except["\""] ..) ~~ ", TI]\""] :> 
 "StyleBox[\"\\\"\\!\\(\\*StyleBox[\\\"" ~~ a ~~ 
  "\\\",\\\"TI\\\"]\\)\\\"\", ShowStringCharacters->True]"
  
transf10=ShortestMatch[".StyleBox[" ~~ a : (Except["\""] ..) ~~ ", TI]"] :> 
 ".\\!\\(\\*StyleBox[\\\"" ~~ a ~~ "\\\",\\\"TI\\\"]\\)"
 
specialrule2=
x_String/;StringMatchQ[x, "*\"\"" ~~ (LetterCharacter ..) ~~ "\"\"*"] :> 
   StringReplace[x, "\"\"" ~~ l : (LetterCharacter ..) ~~ "\"\"" :> 
                    "StyleBox[\"\\\"" ~~ l ~~ "\\\"\",ShowStringCharacters->True]"]
                    
(* For bug 77933: *)

specialrule3 = "\!\(\*RowBox[{\"Delimiters\", \"->\", RowBox[{\"{\", RowBox[{\"\"<*\"\", \",\", \"\"*>\"\"}], \"}\"}]}]\)" -> "\!\(\*RowBox[{\"Delimiters\", \" \", \"->\", \" \", RowBox[{\"{\", StyleBox[RowBox[{\"\\\"<*\\\"\", \",\", \" \", \"\\\"*>\\\"\"}], ShowStringCharacters -> True], \"}\"}]}]\)"
                    
(* ReduceCell removes future parts of a usage cell as indicated by "ModInfo". *)

ReduceCell[cell_] := 
 Module[{positions, pairs, cs, app, rules, rep}, 
  positions = First/@Position[cell[[1, 1]], 
                               Cell[b_String /; StringMatchQ[b, (a : NumberString /; ToExpression@a > $VersionNumber) ~~ __], 
                                    "ModInfo", ___] | Cell[" F", "ModInfo", "BrighterFlag"] |
                               Cell[BoxData[TooltipBox[Cell[" F", "ModInfo", "BrighterFlag"], _]], ___] | 
                               Cell[BoxData[TooltipBox[Cell[s_String /; StringMatchQ[s, LetterCharacter..], 
                               	    "ModInfo", "FutureExample", ___], _]], ___] |
                               Cell[___, "FutureExample", ___]]; 
  pairs = Partition[Append[Flatten[Cases[Position[cell[[1, 1]], Cell[_, "ModInfo", ___] | 
                                                          Cell[BoxData[TooltipBox[Cell[_, "ModInfo", ___], _]], ___]],{_}]], 
                           Length@cell[[1, 1]]], 2, 1];
  cs = Cases[Take[pairs, Length@pairs - 1] /. {a_Integer, b_} :> {a, b - 1}, {a_ /; MemberQ[positions, a], _}];
  app = If[MemberQ[positions, pairs[[-1, 1]]], Append[cs, pairs[[-1]]], cs]; 
  rules = # -> Sequence[] & /@ Flatten[Range[Sequence @@ #] & /@ app]; 
  rep = ReplacePart[cell[[1, 1]], rules]; 
  rep = If[MatchQ[rep, {__, x_String /; StringMatchQ[x, __ ~~ "\n"]}], 
           MapAt[StringDrop[#, -1] &, rep, -1], 
           rep]; 
  Cell[TextData[rep], "Usage"]]
  
  
GetTemplateBoxDisplayFunction[stylesheetexpr_, style_] := 
 If[# === {}, {}, #[[1]]] &@Flatten[Cases[stylesheetexpr, 
    Cell[StyleData[style], b : (TemplateBoxOptions -> a_)] :> Cases[a, c : (DisplayFunction -> d_) :> d, 2], Infinity]]

Options[CreateUsageMessageFile] = {AddQuotesToSubscriptedVariables -> None};
    
CreateUsageMessageFile[functionPages:{___String?AtomQ}, internals_:{}, fileout_String, opts___] := 
Module[{addQuotesToSubscriptedVariables, letters, capitalletters, romanletters, CharactersWithStylesNeededForPotentialProducts, core, stmp, nbexpr, context,
        paclet, cs, name, result, cell, templateboxes, keystrings, im, quoterules},
        
   addQuotesToSubscriptedVariables = AddQuotesToSubscriptedVariables /. {opts} /. Options[CreateUsageMessageFile];
   
   letters = CharacterRange["a", "z"]; capitalletters = CharacterRange["A", "Z"]; romanletters = Union[letters, capitalletters];
   
   CharactersWithStylesNeededForPotentialProducts = {"Backslash", "Cap", "CenterDot", "CircleDot", "CirclePlus", "CircleTimes", "Colon", 
      "Congruent", "Coproduct", "Cup", "CupCap", "Diamond", "DotEqual", "DoubleDownArrow", "DoubleLeftArrow", "DoubleLeftRightArrow", 
      "DoubleLeftTee", "DoubleLongLeftArrow", "DoubleLongLeftRightArrow","DoubleLongRightArrow", "DoubleRightArrow", "DoubleRightTee", 
      "DoubleUpArrow", "DoubleUpDownArrow", "DoubleVerticalBar", "DownArrow", "DownArrowBar", "DownArrowUpArrow", 
      "DownLeftRightVector", "DownLeftTeeVector", "DownLeftVector", "DownLeftVectorBar", "DownRightTeeVector", "DownRightVector", 
      "DownRightVectorBar", "DownTee", "DownTeeArrow", "EqualTilde", "Equilibrium", "GreaterEqualLess", "GreaterFullEqual", 
      "GreaterGreater", "GreaterLess", "GreaterSlantEqual", "GreaterTilde", "HumpDownHump", "HumpEqual", "LeftArrow", 
      "LeftArrowBar", "LeftArrowRightArrow", "LeftDownTeeVector", "LeftDownVector", "LeftDownVectorBar", "LeftRightArrow", 
      "LeftRightVector", "LeftTee", "LeftTeeArrow", "LeftTeeVector", "LeftTriangle", "LeftTriangleBar", "LeftTriangleEqual", 
      "LeftUpDownVector", "LeftUpTeeVector", "LeftUpVector", "LeftUpVectorBar", "LeftVector", "LeftVectorBar", 
      "LessEqualGreater", "LessFullEqual", "LessGreater", "LessLess", "LessSlantEqual", "LessTilde", "LongEqual", "LongLeftArrow", 
      "LongLeftRightArrow", "LongRightArrow", "LowerLeftArrow", "LowerRightArrow", "MinusPlus", "NestedGreaterGreater", 
      "NestedLessLess", "NotCongruent", "NotCupCap", "NotDoubleVerticalBar", "NotEqualTilde", "NotGreater", 
      "NotGreaterEqual", "NotGreaterFullEqual", "NotGreaterGreater", "NotGreaterLess", "NotGreaterSlantEqual", "NotGreaterTilde", 
      "NotHumpDownHump", "NotHumpEqual", "NotLeftTriangle", "NotLeftTriangleBar", "NotLeftTriangleEqual", "NotLess", 
      "NotLessEqual", "NotLessFullEqual", "NotLessGreater", "NotLessLess", "NotLessSlantEqual", "NotLessTilde", 
      "NotNestedGreaterGreater", "NotNestedLessLess", "NotPrecedes", "NotPrecedesEqual", "NotPrecedesSlantEqual", "NotPrecedesTilde", 
      "NotReverseElement", "NotRightTriangle", "NotRightTriangleBar", "NotRightTriangleEqual", "NotSquareSubset", "NotSquareSubsetEqual",
      "NotSquareSuperset", "NotSquareSupersetEqual", "NotSubset", "NotSubsetEqual", "NotSucceeds", "NotSucceedsEqual", 
      "NotSucceedsSlantEqual", "NotSucceedsTilde", "NotSuperset", "NotSupersetEqual", "NotTilde", "NotTildeEqual", 
      "NotTildeFullEqual", "NotTildeTilde", "NotVerticalBar", "PlusMinus", "Precedes", "PrecedesEqual", "PrecedesSlantEqual", 
      "PrecedesTilde", "Proportion", "Proportional", "ReverseElement", "ReverseEquilibrium", "ReverseUpEquilibrium", "RightArrow", 
      "RightArrowBar", "RightArrowLeftArrow", "RightDownTeeVector", "RightDownVector", "RightDownVectorBar", "RightTee", 
      "RightTeeArrow", "RightTeeVector", "RightTriangle", "RightTriangleBar", "RightTriangleEqual", "RightUpDownVector", 
      "RightUpTeeVector", "RightUpVector", "RightUpVectorBar", "RightVector", "RightVectorBar", "RoundImplies", "ShortLeftArrow", 
      "ShortRightArrow", "SmallCircle", "SquareIntersection", "SquareSubset", "SquareSubsetEqual", "SquareSuperset", 
      "SquareSupersetEqual", "SquareUnion", "Star", "Subset", "SubsetEqual", "Succeeds", "SucceedsEqual", "SucceedsSlantEqual", 
      "SucceedsTilde", "Superset", "SupersetEqual", "Therefore", "Tilde","TildeEqual", "TildeFullEqual", "TildeTilde", "UnionPlus", 
      "UpArrow", "UpArrowBar", "UpArrowDownArrow", "UpDownArrow", "UpEquilibrium", "UpperLeftArrow", "UpperRightArrow", "UpTee", 
      "UpTeeArrow", "Vee", "VerticalBar", "VerticalSeparator", "VerticalTilde", "Wedge"};
      
     Quiet[core = Get[ToFileName[{$InstallationDirectory, "SystemFiles", "FrontEnd", "StyleSheets"}, "Core.nb"]];
 
   stmp = OpenWrite[fileout, PageWidth -> Infinity];
   Scan[
      Function[{functionFile},
         Catch[
            Quiet[nbexpr = Get[functionFile]];
            paclet =
              Replace[
                CategorizationFromFunctionPage[ nbexpr, "Paclet Name"],
              { 
                p_String :> p,
                _ -> "Mathematica"
              }];
            context = ContextFromFunctionPage[ nbexpr];
(* Do not use package function pages for System` context usage messages: *)
            If[ paclet =!= "Mathematica" && context === "System`",
              Throw[ result = $Failed]];
            (* Remove Usage cells which are marked as Future, Tentative, etc. *)
            nbexpr = DeleteCases[nbexpr, Cell[_, c___] /; MemberQ[{c}, "Usage", Infinity] && 
            	MemberQ[{c}, "FutureExample" | "TentativeExample" | "ExcludedMaterial", Infinity],
            	Infinity];
            cs = ReplaceAll[Cases[nbexpr, Cell[_, "Usage", ___], Infinity], 
 Cell[BoxData[FormBox[StyleBox[DynamicBox[ToBoxes[If[$OperatingSystem === "MacOSX", "Return", "Enter"], StandardForm], 
                                          ImageSizeCache -> _], "KeyEvent"], TraditionalForm]], NotebookDefault, 
      ShowStringCharacters -> False] -> StyleBox["Enter", "KeyEvent"]]; 
            If[cs === {}, Throw[result = $Failed]];
      (* The function name. *) 
            name = StringReplace[functionFile, {DirectoryName[functionFile] -> "", ".nb" -> ""}];
            
            (* To italicize some variables. *)
            
            cell = cs[[1]] /. Cell[TextData[{a__}], "Usage", ___] :> 
 Cell[TextData[Switch[#,
 
                      Cell[BoxData[FormBox[RowBox[{___, "(", c_String /; MemberQ[letters, c], ")"}], TraditionalForm]], "InlineMath"],
                      
                      # /. Cell[BoxData[FormBox[RowBox[{b___, "(", c_String /; MemberQ[letters, c], ")"}], TraditionalForm]], "InlineMath"] :> 
		      Cell[BoxData[FormBox[RowBox[{b, "(", StyleBox[c, "TI"], ")"}], TraditionalForm]], "InlineMath"],
 
                      Cell[BoxData[FormBox[RowBox[{SubscriptBox[s_String /; MemberQ[letters, s], "1"], 
                                                   "\[Cross]", SubscriptBox[s_String, "2"], "\[Cross]", "\[Ellipsis]"}], 
                                           TraditionalForm]], "InlineMath"], 
                      # /. Cell[BoxData[FormBox[RowBox[{SubscriptBox[s_String /; MemberQ[letters, s], "1"], 
                                                   "\[Cross]", SubscriptBox[s_String, "2"], "\[Cross]", "\[Ellipsis]"}],
                                           TraditionalForm]], "InlineMath"] :> 
                      Cell[BoxData[RowBox[{SubscriptBox[StyleBox[s, "TI"], StyleBox["1", "TR"]], 
                                           "\[Cross]", SubscriptBox[StyleBox[s, "TI"], StyleBox["2", "TR"]], "\[Cross]", StyleBox["\[Ellipsis]", "TR"]}]], 
                           "InlineFormula"],
                           
                      Cell[BoxData[FormBox[s_String /; MemberQ[letters, s], TraditionalForm]], "InlineMath"], 
                      # /. Cell[BoxData[FormBox[s_String /; MemberQ[letters, s], TraditionalForm]], "InlineMath"] :> 
                      Cell[BoxData[StyleBox[s, "TI"]], "InlineFormula"],
                      
                      Cell[BoxData[FormBox[RowBox[{c_String /; MemberQ[letters, c], d_String, e_String /; MemberQ[letters, e], d_String, "\[Ellipsis]"}], 
                                           TraditionalForm]], "InlineMath"],
                                           
                      # /. Cell[BoxData[FormBox[RowBox[{c_String /; MemberQ[letters, c], 
		                                        d_String /; Quiet@MemberQ[CharactersWithStylesNeededForPotentialProducts, 
		                                                                  StringTake[ToString[d, InputForm, CharacterEncoding -> None], {4, -3}]], 
		                                        e_String /; MemberQ[letters, e], 
		                                        d_String /; Quiet@MemberQ[CharactersWithStylesNeededForPotentialProducts, 
		                                                                  StringTake[ToString[d, InputForm, CharacterEncoding -> None], {4, -3}]], 
		                                        "\[Ellipsis]"}], 
		                                TraditionalForm]], "InlineMath"] :> 
		      Cell[BoxData[
		          FormBox[RowBox[{StyleBox[c, "TI"], d, StyleBox[e, "TI"], d, "\[Ellipsis]"}], TraditionalForm]], "InlineMath"],
                      
                      _, 
                      
                      #] & /@ {a}], "Usage"];
                     
            cell = cell /. ce:Cell[_, "InlineMath"] :> (ReplaceRepeated[ce /. FormBox[a_, _] :> a, 
   b_[c_, d__] /; And[Not[MemberQ[{StyleBox, Cell}, b]], 
                      Length[{c, d}] < 4, 
                      MemberQ[{c, d}, z_String /; MemberQ[romanletters, z]]] :> b @@ (If[StringQ[#] && MemberQ[romanletters, #], 
                  StyleBox[#, "TI"], #] & /@ {c, d})] //. {RowBox[{e___, f_String /; MemberQ[romanletters, f], g__}] :> RowBox[{e, StyleBox[f, "TI"], g}], 
                                                          RowBox[{e__, f_String /; MemberQ[romanletters, f], g___}] :> RowBox[{e, StyleBox[f, "TI"], g}]});
                      
            cell = If[FreeQ[cell,TemplateBox], 
                      cell,
                      templateboxes = Cases[cell, TemplateBox[a_, b_, ___], Infinity];
                      If[templateboxes === {},
                         cell,
                         cell /. (RuleDelayed @@ (# -> GetTemplateBoxDisplayFunction[core, #[[2]]][Sequence @@ #[[1]]]) & /@ templateboxes)]];qrk=cell;
                      
            keystrings =
               If[ MatchQ[ cell, Cell[_String, "Usage", ___]], 
                  {
                  StringReplace[ cell[[1]], {
                     "\[LineSeparator]" -> " ",
                     "  " -> " "
                  }]
                  }
               , (* else *)
                  (Map[
                  	If[MemberQ[#, s_String /; !StringFreeQ[s,"http"], Infinity],
                  		Transmogrify`ConvertToString[#],
                     If[
                        MatchQ[#,
                           RowBox[{__}] |
                           SubscriptBox[__] |
                           StyleBox[__] |
                           FractionBox[__] |
                           SuperscriptBox[__] |
                           SubsuperscriptBox[__] |
                           OverscriptBox[__] |
                           RadicalBox[__] |
                           SqrtBox[_] | 
                           UnderoverscriptBox[__] |
                           UnderscriptBox[__] |
                           RadioButtonBox[__] |
                           CheckboxBox[__] | 
                           TogglerBox[__] |
                           SetterBox[__] |
                           OpenerBox[__] ], 
                        "\!\(\*" <>
                          StringReplace[
                           StringReplace[
                              ToString[
                                 # /. transf1 /. transf11 /. transf12 /. transf2
                              ],
                                 {transf7, transf3, transf4, transf5, transf6, transf8, transf9}
                           ], transf10] <>
                           "\)"
                     , (* else *)
                        #
                     ]] &,
                     ((
                     If[FreeQ[cell, 
		              Cell[b_String /; StringMatchQ[b, (a : NumberString /; ToExpression@a > $VersionNumber) ~~ __], 
                                   "ModInfo", ___] |
                              Cell[BoxData[TooltipBox[Cell[b_String /; StringMatchQ[b, (a : NumberString /; ToExpression@a > $VersionNumber) ~~ __],
                                   "ModInfo", ___], _]], ___] |
                              Cell[BoxData[TooltipBox[Cell[" F", "ModInfo", ___], _]], ___] | 
                              Cell[BoxData[TooltipBox[Cell[s_String /; StringMatchQ[s, LetterCharacter..], "ModInfo", 
                              	"FutureExample", ___], _]], ___]],
                        cell, 
                        ReduceCell[cell]]
                        /. Cell[a_BoxData, b_, ___] :> Cell[a, b]
                        /. {
                           Cell[BoxData[TooltipBox[Cell[_, "ModInfo", ___], _]]] -> Sequence[], 
                           Cell[_, "ModInfo" | "UsageModInfo", ___] -> Sequence[], 
                           ButtonBox[__, (BaseStyle | DefaultBaseStyle) -> "ExampleLink", ___] -> Sequence[],
                           ButtonBox[a_, ___] :> a, 
                           FormBox[a_, TraditionalForm] :> a, 
                           TagBox[a_, ___] :> a
                           }
                        /. Cell[BoxData[box_], style___] :> box
                     )[[1, 1]]
                        /.  x_String :>
                           StringReplace[x, {
                              " \[LineSeparator]"->" ",
                              "\[LineSeparator]"->" "}]) /. 
                  {StyleBox[RowBox[List["\"", "string", "\""]], "TI"] -> "\"\!\(\*\nStyleBox[\"string\",\n\"TI\"]\)\"", 
                   "\"Save\"" -> "StyleBox[\"\\\"Save\\\"\",ShowStringCharacters->True]"}
                  ]
                     /. specialrule1) /. specialrule2
               ];
            result =
               If[And @@ (StringQ /@ (# /. specialrule3)),
                  StringReplace[StringReplace[StringJoin @@ (# /. specialrule3), "  " -> " "], "_StyleBox" -> "_\*StyleBox"],
                  $Failed]&[Switch[name,
                                   "GeoPosition",
                                   DeleteCases[keystrings, TooltipBox[Cell[_, "ModInfo", "BrightFlag"], _]|TooltipBox[_]],
                                   "SlotSequence",
                                   keystrings/.Cell[TextData[{"##", StyleBox["n", "TI"]}], "InlineFormula"] -> Unevaluated@Sequence["##", "\!\(\*StyleBox[RowBox[{StyleBox[\"n\", \"TI\"], \"\[Null]\"}]]\)"],
                                   _,
                                   (* fix for Area, Volume not being included in Usage.m due to ModInfo *)
                                   (* further fix for any ref page with ModInfo in usage *)
                                   DeleteCases[keystrings, 
                                   	TooltipBox[_String] | Cell[BoxData[TooltipBox[_String]], _String]]]]
         ];
         If[ StringQ[result] && Not[ StringMatchQ[ result, ___ ~~ "XXXX" ~~ ___] ],
            If[addQuotesToSubscriptedVariables =!= None && MemberQ[First/@addQuotesToSubscriptedVariables, name],
               quoterules = (Shortest["SubscriptBox[StyleBox[\"" ~~ # ~~ "\", \"TI\"], StyleBox[\"" ~~ d : DigitCharacter ~~ "\", \"TR\"]]"] :> 
"StyleBox[\"\\\"\\!\\(\\*StyleBox[\\\"" ~~ # ~~ "\\\",\\\"TI\\\"]\\_" ~~ d ~~ "\\)\\\"\", ShowStringCharacters->True]") & /@ 
                             Cases[addQuotesToSubscriptedVariables, {name, _}][[1, 2]];
               result = StringReplace[result, quoterules]];
            WriteString[ stmp, 
              If[ context === "System`", name, context <> name] <>
              "::usage = " <>
              ToString[ result, InputForm, CharacterEncoding -> "ASCII"] <>
              "\n"
            ]
         ]
      ],
      functionPages
   ];
   Scan[
      With[{
         symbolName = StringReplace[ #, {DirectoryName[#] -> "", ".nb" -> ""}]
      },
         WriteString[ stmp,
            symbolName <> "::usage = \"" <> symbolName <> " is an internal symbol.\"\n"
         ]
      ]&
      ,
      internals
   ];
   Close[stmp]; 
(*
   im = Import[fileout, "Lines"]; 
   Export[ fileout, 
      Map[
         StringReplace[#,
            LongestMatch["\"" ~~ b__ ~~ "::usage = " ~~ a__ ~~ "\""] -> 
               b ~~ "::usage = " ~~ "\"" ~~ a ~~ "\""] &,
         im],
      "Lines"]
*)
,Syntax::sntufn]]

CreateTextUsageMessageFile[
  functionPagesDir_String /; FileType[functionPagesDir] === Directory,
  fileoutpath_String
] :=
CreateTextUsageMessageFile[ FileNames["*.nb", {functionPagesDir}], fileoutpath]

CreateTextUsageMessageFile[ functionPages_List, fileoutpath_String] :=
 Module[{stmp, nbexpr, cs, name, attemptedusage, result, cell, im},
        stmp = OpenWrite[fileoutpath, PageWidth -> Infinity];
        DocumentationBuild`Utils`BlockMessageOff[{Syntax::"newl"},
         ((Catch[nbexpr = Get[#];
                 cs = Cases[nbexpr, Cell[_, "Usage", ___], Infinity];
                 If[cs === {}, Throw[result = $Failed]];
                 name = Cases[nbexpr,
			Cell[n_String | TextData[{n_String,___}], "ObjectName"|"ObjectNameSmall", ___] :> n,
			Infinity];
                 If[name === {}, Throw[result = $Failed]];
                 cell = cs[[1]];
                 name = StringSplit[name[[1]]][[1]];
                 attemptedusage = CreateUsageMessageString[name, cell];
                 result = If[StringQ[attemptedusage], attemptedusage, $Failed]];
           If[ StringQ[result] && Not[ StringMatchQ[ result, ___ ~~ "XXXX" ~~ ___] ],
              PutAppend[name <> "::usage = " <> result, stmp]]) &) /@ functionPages];
        Close[stmp];
        im = Import[fileoutpath, "Lines"];
        Export[fileoutpath, (StringReplace[#, LongestMatch["\"" ~~ b__ ~~ "::usage = " ~~ a__ ~~ "\""] ->
                                                                b ~~ "::usage = " ~~ "\"" ~~ a ~~ "\""] &) /@ im, "Lines"]]

(* Create a UsageMessages string from authored notebook *)
(* Insanely by jayw *)
CreateUsageMessageString[name_String, usagecell_Cell]:=
 Module[{newcell, stcs, sipf, $UsageMessagesSectionFound},
   Catch[
   If[Not@MatchQ[usagecell /. {Cell[_, "ModInfo" | "UsageModInfo", ___] -> Sequence[],
 ButtonBox[__, DefaultStyle -> "ExampleLink", ___] -> Sequence[]},
           Cell[TextData[{(Cell[BoxData[_] | _String, "InlineFormula" | "InlineMath", ___] |
                           Cell[BoxData[FormBox[_, TraditionalForm]]] | StyleBox[a_String, __] | _String) ..}],
                "Usage", ___]],
      Throw[$Failed]];
   newcell = ((((((((((((((((((((((((((((((usagecell /. {Cell[_, "ModInfo" | "UsageModInfo", ___] -> Sequence[],
 ButtonBox[__, DefaultStyle -> "ExampleLink", ___] -> Sequence[], (CellChangeTimes -> _) -> Sequence[]}) /.
 Cell[TextData[{a__, Cell[BoxData[StyleBox[b_String /; StringLength[b] === 1, "TI"]], "InlineFormula"],
    Cell[BoxData[FormBox[SuperscriptBox["\[Null]", "th"], TraditionalForm]], "InlineMath"], c__}], "Usage", ___] :>
 Cell[TextData[{a, Cell[BoxData[SuperscriptBox[StyleBox[RowBox[{StyleBox[b, "TI"], "\[Null]"}]], "th"]],
      "InlineFormula"], c}], "Usage"]) /.
   {Cell[BoxData[SuperscriptBox[StyleBox[RowBox[{StyleBox[a_String /; MemberQ[thlist, a], "TI"],
        "\[Null]"}]], "th"]], "InlineFormula"] :> a <> "th",
 Cell[BoxData[RowBox[{SubscriptBox[StyleBox[a_String /; MemberQ[thlist, a], "TI"],
       StyleBox[b_String /; MemberQ[thlist, b], "TI"]],
                SuperscriptBox["\[Null]", "th"]}]], "InlineFormula"] :> a <> b <> "th",
 Cell[BoxData[FormBox[SuperscriptBox[StyleBox[a_String /; MemberQ[thlist, a], "TI"], "th"],
                      TraditionalForm]]] :> a <> "th",
 Cell[BoxData[RowBox[{a_String, "[", StyleBox[b_, "TI"], "]"}]], "InlineFormula"] :> StringJoin[a, "[", b, "]"],
 Cell[BoxData[RowBox[{SubsuperscriptBox[StyleBox[_String, "TI"], StyleBox[_String, "TI"],RowBox[{"(", _String, ")"}]],
     RowBox[{"(", StyleBox[_String, "TI"], ")"}]}]], "InlineFormula"] -> Sequence[],
 Cell[BoxData[SubsuperscriptBox[StyleBox[_String, "TI"], StyleBox[_String, "TI"],
    RowBox[{"(", StyleBox[_String, "TI"], ")"}]]], "InlineFormula"] -> Sequence[],
 Cell[BoxData[RowBox[{SubscriptBox[StyleBox[_String, "TI"], StyleBox[_String, "TI"]],
     RowBox[{"(", StyleBox[_String, "TI"], ")"}]}]], "InlineFormula"] -> Sequence[],
 Cell[BoxData[RowBox[{SubsuperscriptBox[StyleBox[_String, "TI"], StyleBox[_String, "TI"], StyleBox[_String, "TI"]],
     RowBox[{"(", StyleBox[_String, "TI"], ")"}]}]], "InlineFormula"] -> Sequence[],
 Cell[BoxData[FormBox[RowBox[{SubscriptBox[_String, _String], RowBox[{"(", _String, ")"}]}], TraditionalForm]],
      "InlineMath"] -> Sequence[],
Cell[BoxData[FormBox[RowBox[{SubsuperscriptBox[_String, _String, _String], RowBox[{"(", _String, ")"}]}], TraditionalForm]],
     "InlineMath"] -> Sequence[],
Cell[BoxData[RowBox[{SubsuperscriptBox[StyleBox[_String, "TI"], StyleBox[RowBox[{_String, _String}], "TI"],
     StyleBox[RowBox[{_String, _String}], "TI"]], RowBox[{"(", RowBox[{StyleBox[_String, "TI"], "\[VerticalSeparator]",
     GridBox[_]}], ")"}]}]], "InlineFormula"] -> Sequence[],
Cell[BoxData[RowBox[{"(", GridBox[{{StyleBox["n", "TI"]}, {StyleBox["m", "TI"]}}], ")"}]], "InlineFormula"] -> Sequence[],
Cell[BoxData[FormBox[RowBox[{SuperscriptBox["cosh", RowBox[{"-", "1"}]], RowBox[{"(", "z", ")"}]}], TraditionalForm]],
     "InlineMath", ___] -> Sequence[],
Cell[BoxData[FormBox[RowBox[{SuperscriptBox["cos", RowBox[{"-", "1"}]], "(", "z", ")"}], TraditionalForm]],
     "InlineMath", ___] -> Sequence[],
Cell[BoxData[RowBox[{SuperscriptBox["coth" | "cot" | "csch" | "csc" | "sech" | "sec" | "sinh" | "sin" | "tanh" | "tan",
     RowBox[{"-", "1"}]], RowBox[{"(", StyleBox["z", "TI"], ")"}]}]], "InlineFormula", ___] -> Sequence[],
Cell[BoxData[FractionBox[StyleBox["y", "TI"], StyleBox["x", "TI"]]], "InlineFormula"] -> "y/x",
Cell[BoxData[RowBox[{"\[Phi]", "=", RowBox[{"(", RowBox[{"1", "+", SqrtBox["5"]}], ")"}], "/", "2"}]],
     "InlineFormula"] -> "(1 + Sqrt[5])/2", Cell[BoxData[SqrtBox[RowBox[{"-", "1"}]]], "InlineFormula"] -> "Sqrt[-1]",
StyleBox["a", "TB"] -> RowBox[{"{", RowBox[{SubscriptBox[StyleBox["a", "TI"], "1"], ",", StyleBox["\[Ellipsis]", "TR"],
     ",", SubscriptBox[StyleBox["a", "TI"], StyleBox["p", "TI"]]}], "}"}],
StyleBox["b", "TB"] -> RowBox[{"{", RowBox[{SubscriptBox[StyleBox["b", "TI"], "1"], ",", StyleBox["\[Ellipsis]", "TR"],
     ",", SubscriptBox[StyleBox["b", "TI"], StyleBox["q", "TI"]]}], "}"}]}) /.
  x_String :> StringReplace[x,
     {"\"\!\(\*\nStyleBox[\"" ~~ a__ ~~ "\",\n\"TI\"]\).\!\(\*\nStyleBox[\"" ~~ b__ ~~ "\",\n\"TI\"]\)\"" :>
 "\"" <> a <> "." <> b <> "\"",
ShortestMatch["\"" ~~ a__ ~~ "://\!\(\*\nStyleBox[\"" ~~ b__ ~~ "\",\n\"TI\"]\)\""] :> "\"" <> a <> "://" <> b <> "\"",
"\"\!\(\*\nStyleBox[\"" ~~ a__ ~~ "\",\n\"TI\"]\)." ~~ b__ ~~ "\"" :> "\"" <> a <> "." <> b <> "\"",
  "\nStyleBox" -> "StyleBox", "\n\"" ~~ "TI" ~~ "\"" -> "\"" ~~ "TI" ~~ "\"",
   "\n\"" ~~ "TR" ~~ "\"" -> "\"" ~~ "TR" ~~ "\"", "\nSubscriptBox" -> "SubscriptBox",
   "* SubscriptBox[ StyleBox[" -> "*SubscriptBox[StyleBox[", "  StyleBox" -> " StyleBox"}]) /. x_String :>
 StringReplace[x,
  y : ShortestMatch["\"\\!\\" ~~ (a__ /; Not@StringFreeQ[a, "(\\*SubscriptBox[StyleBox[\\\""]) ~~ __ ~~ (b__ /;
    Not@StringFreeQ[b, ", StyleBox[\\\""]) ~~ "]]\\)\""] :> (stcs = StringCases[y,
                                                                         ShortestMatch["StyleBox[\\\"" ~~ c__ ~~ "\\\","]];
 If[MatchQ[stcs, {_String, _String}],
    StringJoin @@ (StringReplace[#, {"StyleBox[\\\"" -> "", "\\\"," -> ""}] & /@ stcs), y])]) /.
                              Cell[BoxData[ButtonBox[a_,___,DefaultStyle\[Rule]_,___]],__]:>a) /.
                         ButtonBox[a_, ___, DefaultStyle -> _, ___] :> a ) /.
                                   SubscriptBox[StyleBox[a_String, _], StyleBox[b_String, _]] :> a <> b) /.
                   {StyleBox[RowBox[{a_String, "`"}]] :> a <> "`",
                    SuperscriptBox[a_String, "\[Prime]"] :> a <> "'"}) /. StyleBox[a_, _] :> a) /.
                               SubscriptBox[a_String,b_String] :> (sipf = ToString[a, InputForm, CharacterEncoding -> None];
      If[StringTake[sipf, 3] === "\"\\[" && StringTake[sipf, {-2, -1}] === "]\"" && MemberQ[$GreekLowerCase, sipf],
          ToLowerCase[StringTake[sipf, {4, -3}]], a]) <> b) /.
       Cell[BoxData[a_], "InlineFormula", ___] :> a)) /.
               x_String :> StringReplace[x, {"\[LineSeparator]" -> " ", "\n" -> "", "\[NonBreakingSpace]"-> " ",
                                             "\[Ellipsis]" -> "..."}]) /. x_String :>
 StringReplace[x, {"\"\!\(\*StyleBox[\(\*StyleBox[" ~~ "\"" ~~ a__ ~~ "\"" ~~ ",\"TI\"]`\)]\)\"" :>
                   "\"" <> a <> "`" <> "\"",
  "\"\!\(\*StyleBox[" ~~ "\"" ~~ a__ ~~ "\"" ~~ ",\"TI\"]\)" ~~ b : (".nb" | ".mx") ~~ "\"" :> "\"" <> a <> b <> "\"",
              "\"\!\(\*SubscriptBox[StyleBox[" ~~ "\"" ~~ a__ ~~ "\"" ~~ ",\"TI\"], " ~~ "\"" ~~ b__ ~~ "\"" ~~ "]\)`\"" :>
   "\"" <> a <> b <> "`" <> "\"", "\"\!\(\*StyleBox[" ~~ "\"" ~~ a__ ~~ "\"" ~~ ",\"TI\"]\)\"" :> "\"" <> a <> "\""}]) /.
   a : RowBox[{__}] :>
    ((a /. "," -> ", ") //. RowBox[{b__}] :> {b})) //. {a__String} :> StringJoin[a]) /. StyleBox[a_String] :> a) /.
    {Cell[BoxData[FormBox[a_String, TraditionalForm]], "InlineMath"] :> a,
     Cell[BoxData[FormBox[SuperscriptBox["\[Null]", "th"], TraditionalForm]]] :> "th"}) //.
    {SubscriptBox[a_String, b_String] :> a <> b, BoxData[a_String] :> a, Cell[a_, "InlineFormula"] :> a,
    StyleBox[a_String, FontSlant -> _] :> a}) //.
    List[a__String] :> StringJoin[a]) /. Cell[a_, "InlineFormula"] :> a) /. SuperscriptBox[x_String, y_String] /;
  StringLength[x] === 1 && StringLength[y] === 1 :> x <> "^" <> y) /.
  List["y^2", "=", "x^3", "+", "a", "x^2", "+", "b", "x"] ->
  StringJoin @@ Riffle[List["y^2", "=", "x^3", "+", "a", "x^2", "+", "b", "x"], " "]) /.
    {TextData[{a__String}] :> StringJoin[a], TextData[a_String] :> a}) /.
   x_String :> StringReplace[x, (a_ /; a =!= Whitespace) ~~ "\[Cross]" ~~ (b_ /; b =!= Whitespace) -> a ~~ " by " ~~ b]) /.
   x_String :> StringReplace[x,
 ShortestMatch["\"\!\(\*SubscriptBox[StyleBox[" ~~ "\"" ~~ c__ ~~ "\"" ~~ ",\"TI\"]," ~~ " \"" ~~ b_ ~~ "\"" ~~ "]\)\""] :>
   "\"" ~~ c ~~ b ~~ "\""]) /.
 x_String :> StringReplace[x, ShortestMatch[
"\"\!\(\*SubscriptBox[StyleBox[" ~~ "\"" ~~ a__ ~~
                                          "\",\"TI\"], StyleBox[" ~~ "\"" ~~ b__ ~~ "\",\"T" ~~ "I" | "R" ~~ "\"]]\)\""] :>
   "\"" <> a <> b <> "\""]) /. x_String :> StringReplace[x, "\[Hyphen]"->"-"]) /.
   x_String :> StringReplace[x, "." ~~ (a_ /; (LetterQ[a] && UpperCaseQ[a])) :> ". " <> a]) /.
    x_String :> StringReplace[x, "  " -> " "];
    If[MatchQ[newcell, Cell[_String, ___]],
       newcell = newcell /. Cell[a_, "Usage", ___]:> Cell[TrimEnds[a], "UsageMessages"];
       If[Not@FreeQ[usagecell, Alternatives @@ formsToEliminate],
          (newcell /. x_String :> StringReplace[x, " ." -> "."])[[1]],
          newcell[[1]]],
       $Failed]]];

TrimEnds[a_String]:= StringReplace[a, (StartOfString ~~ Whitespace) | (Whitespace ~~ EndOfString) -> ""]

formsToEliminate =
 {Cell[BoxData[RowBox[{SubsuperscriptBox[StyleBox[_String, "TI"], StyleBox[_String, "TI"], RowBox[{"(", _String, ")"}]],
        RowBox[{"(", StyleBox[_String, "TI"], ")"}]}]], "InlineFormula"],
  Cell[BoxData[SubsuperscriptBox[StyleBox[_String, "TI"], StyleBox[_String, "TI"],
       RowBox[{"(", StyleBox[_String, "TI"], ")"}]]], "InlineFormula"],
  Cell[BoxData[RowBox[{SubscriptBox[StyleBox[_String, "TI"], StyleBox[_String, "TI"]],
       RowBox[{"(", StyleBox[_String, "TI"], ")"}]}]], "InlineFormula"],
  Cell[BoxData[RowBox[{SubsuperscriptBox[StyleBox[_String, "TI"], StyleBox[_String, "TI"], StyleBox[_String, "TI"]],
       RowBox[{"(", StyleBox[_String, "TI"], ")"}]}]], "InlineFormula"],
  Cell[BoxData[FormBox[RowBox[{SubscriptBox[_String, _String], RowBox[{"(", _String, ")"}]}], TraditionalForm]],
       "InlineMath"],
  Cell[BoxData[FormBox[RowBox[{SubsuperscriptBox[_String, _String, _String], RowBox[{"(", _String, ")"}]}],
                       TraditionalForm]], "InlineMath"],
  Cell[BoxData[RowBox[{SubsuperscriptBox[StyleBox[_String, "TI"], StyleBox[RowBox[{_String, _String}], "TI"],
       StyleBox[RowBox[{_String, _String}], "TI"]],
       RowBox[{"(", RowBox[{StyleBox[_String, "TI"], "\[VerticalSeparator]", GridBox[_]}], ")"}]}]], "InlineFormula"],
  Cell[BoxData[RowBox[{"(", GridBox[{{StyleBox["n", "TI"]}, {StyleBox["m", "TI"]}}], ")"}]], "InlineFormula"],
  Cell[BoxData[FormBox[RowBox[{SuperscriptBox["cosh", RowBox[{"-", "1"}]],
       RowBox[{"(", "z", ")"}]}], TraditionalForm]], "InlineMath", ___],
  Cell[BoxData[FormBox[RowBox[{SuperscriptBox["cos", RowBox[{"-", "1"}]], "(", "z", ")"}], TraditionalForm]],
       "InlineMath", ___],
  Cell[BoxData[RowBox[{SuperscriptBox["coth" | "cot" | "csch" | "csc" | "sech" | "sec" | "sinh" | "sin" |
      "tanh" | "tan", RowBox[{"-", "1"}]], RowBox[{"(", StyleBox["z", "TI"], ")"}]}]], "InlineFormula", ___]};

$GreekLowerCase = {"\[Alpha]","\[Beta]","\[Gamma]","\[Delta]","\[Epsilon]",
   "\[Zeta]","\[Xi]","\[Eta]","\[Theta]","\[Iota]","\[Kappa]","\[Lambda]",
   "\[Mu]","\[Nu]","\[Omicron]","\[Pi]","\[Rho]","\[Sigma]","\[Tau]",
   "\[Upsilon]","\[Phi]","\[Psi]","\[Chi]","\[Omega]"};

thlist = {"i", "j", "k", "m", "n", "p", "q", "r", "s"};

$PackageFunctionInformation = 
  Get[ToFileName[{DocumentationBuild`Common`$DocumentationBuildDirectory, "Internal", "data"}, 
    "PackageFunctionInformation.m"]];
$PackageUsage = 
  Import[ToFileName[{DocumentationBuild`Common`$DocumentationBuildDirectory, "Internal", "data"}, "PackageUsage.m"], 
   "Lines"];

$NoUsageExtract = $NoFunctionInfoExtract = 
   StringJoin[#, ".nb"] & /@ {"AlgebraicRules", "Alias", "CellArray", 
     "Compose", "ConstrainedMax", "ConstrainedMin", "ContourLevels", 
     "ContourSpacing", "Debug", "DefaultFont", "DSolveConstants", 
     "Dump", "EvaluationEpilog", "EvaluationFunction", 
     "EvaluationProlog", "FontForm", "FromASCII", "FullOptions", 
     "HeldPart", "HomeDirectory", "LegendreType", "Literal", 
     "LUBackSubstitution", "MeshRange", "Plot3Matrix", "Release", 
     "Renderings", "ReplaceHeldPart", "ResetMedium", "StartProcess", 
     "Subscripted", "ToASCII", "ToHeldExpression", 
     "$TopDirectory", "$$Media", "$CUDADeviceCount", "$CUDADevice",
     "$CUDADeviceCount", "$CUDALinkExampleDataPath", "$CUDALinkLibraryPath", 
     "$CUDALinkPath", "$CUDAResourcesVersion"};

ExportOnlineResourceFiles[indir_String, outdir_String:"", lang_String]:= 
PubsEvaluateWithFE@Module[{inputDir=indir, outputDir=outdir, logfile, files, result, usagedata, usagesymbols, badusages, goodusages, packageResult, stmp, res, symbolDir, selectedfiles},

  (* Print full messages *)
  $MessagePrePrint = .;
  (* turn off error message limit *)
  Off[General::stop];

  Print["** Entering ExportOnlineResourceFiles **"];

  symbolDir = ToFileName[{inputDir, "Documentation", lang, "System", "ReferencePages", "Symbols"}];
 
 symbolDir=If[MemberQ[{"Unix", "MacOSX"}, $OperatingSystem], Union[Map[DirectoryName, 
  Select[Select[
    FileNames["*.nb", inputDir, 
     Infinity], ! StringFreeQ[#, "/ReferencePages/Symbols/"] &], 
     !StringFreeQ[#, "/"<>lang<>"/"] &]]],
     Union[Map[DirectoryName, 
  Select[Select[
    FileNames["*.nb", inputDir, 
     Infinity], ! StringFreeQ[#, "\\ReferencePages\\Symbols\\"] &], 
     !StringFreeQ[#, "\\"<>lang<>"\\"] &]]]];
 
  (* Create syntax coloring file *)
  Print["Creating function template and options file."];
  CreateDirectory[ToFileName[{outputDir}]];

  Print["Identifying Symbol nbs to use from: "<> ToString@ToFileName[{inputDir}]];
  
    Print["Starting with "<>ToString[Length[FileNames["*.nb", symbolDir]]]<>" files"];

  selectedfiles = 
  Select[Select[FileNames["*.nb", symbolDir], 
    FreeQ[FileBaseName /@ DocumentationBuild`SystemResourcesFromDocs`Private`$NoUsageExtract,
      FileBaseName[#]] &], 
   Not[MatchQ[DocumentFlag[#], 
      "ExcisedFlag" | "FutureFlag" | "PreviewFlag"]] &];

Print["Actually going to process "<>ToString[Length[selectedfiles]]<>" files"];

  Print["Export function information file to: "<>ToString@ToFileName[{outputDir}, "FunctionInformation.m"]];
  result = PubsEvaluateWithFE @
    DocumentationBuild`SystemResourcesFromDocs`CreateFunctionInformationFile[ 
      {
        {  (* provide {dirname, {list of files}} *)
          symbolDir,
          selectedfiles
        }
      },
      ToFileName[{outputDir}, "FunctionInformation.m"]
    ];
  (* Bail *)
  If[(result === $Failed ) || (FileType[ ToFileName[{outputDir}, "FunctionInformation.m"] ] === None), 
    Print["-- Failed!"];
    (* Failed build *)
    Print["***** BUILD FAILED! *****"];
    (* Quit with a non-zero exit code *)
    Quit[1],
    Print["-- Success!"];
    (* If successful, add in Packages *)

    (* Print["-- Adding Package function information"];
    packageResult = Get @ ToFileName[{outputDir}, "FunctionInformation.m"];
    Put[ Join[packageResult, DocumentationBuild`SystemResourcesFromDocs`Private`$PackageFunctionInformation], ToFileName[{outputDir}, "FunctionInformation.m"]];
    If[(packageResult === $Failed ) || (FileType[ ToFileName[{outputDir}, "FunctionInformation.m"] ] === None), 
      Put[ result, ToFileName[{outputDir}, "FunctionInformation.m"]];
      Print["-- Failed!"];
      ];*)

  ];
  
    Print["Export new function information file to: "<>ToString@ToFileName[{outputDir}, "FunctionInformation2.m"]];
  
  (* Now that there's no need for a file with the templates and usages, changing this from
     CreateFunctionInformationFileWithUsage to CreateFunctionInformationFile 
     UPDATE: reverted to fix breakages with CoursewareTools *)
  result = PubsEvaluateWithFE @
    DocumentationBuild`SystemResourcesFromDocs`CreateFunctionInformationFileWithUsage[ 
      {
        {  (* provide {dirname, {list of files}} *)
          symbolDir,
          selectedfiles
        }
      },
      ToFileName[{outputDir}, "FunctionInformation2.m"]
    ];
  (* Bail *)
  If[(result === $Failed ) || (FileType[ ToFileName[{outputDir}, "FunctionInformation2.m"] ] === None), 
    Print["-- Failed!"];
    (* Failed build *)
    Print["***** BUILD FAILED! *****"];
    (* Quit with a non-zero exit code *)
    Quit[1],
    Print["-- Success!"];
    (* If successful, add in Packages *)

    (* Print["-- Adding Package function information"];
    packageResult = Get @ ToFileName[{outputDir}, "FunctionInformation.m"];
    Put[ Join[packageResult, DocumentationBuild`SystemResourcesFromDocs`Private`$PackageFunctionInformation], ToFileName[{outputDir}, "FunctionInformation.m"]];
    If[(packageResult === $Failed ) || (FileType[ ToFileName[{outputDir}, "FunctionInformation.m"] ] === None), 
      Put[ result, ToFileName[{outputDir}, "FunctionInformation.m"]];
      Print["-- Failed!"];
      ];*)

  ];

  (* Create Usage.m *)
  Print["Creating usage messages file."];
  Print["Export usage message file to: "<>
    ToString@ToFileName[{outputDir}, "Usage.m"]];
  result =
    PubsEvaluateWithFE @ DocumentationBuild`SystemResourcesFromDocs`CreateUsageMessageFile[
      selectedfiles,
      ToFileName[{outputDir}, "Usage.m"],
      DocumentationBuild`SystemResourcesFromDocs`AddQuotesToSubscriptedVariables -> {{"AbsoluteTime", {"e"}}, {"ClearAll", {"form"}}, {"Clear", {"form"}},
        {"ControllerState", {"c"}}, {"DateDifference", {"unit"}}, {"DateList", {"e"}}, {"DatePattern", {"e"}},
        {"DateString", {"elem"}}, {"DeclarePackage", {"name"}}, {"DeleteFile", {"file"}}, {"FindList", 
        {"text", "file"}}, {"Find", {"text"}}, {"Protect", {"form"}}, {"Remove", {"form"}}, {"StringJoin", {"s"}},
        {"StringReplacePart", {"snew"}}, {"Unprotect", {"form"}}}];
  (* Bail *)
  If[(result === $Failed ) || (FileType[ ToFileName[{outputDir}, "Usage.m"] ] === None), 
    Print["-- Failed!"];
    (* Failed build *)
    Print["***** BUILD FAILED! *****"];
    (* Quit with a non-zero exit code *)
    Quit[1],
    Print["-- Success!"];
    (* If successful, add in Packages *)
    
    usagedata = Union[Import[FileNameJoin[{outputDir, "Usage.m"}], "Lines"]];
    usagesymbols = StringCases[usagedata, z__ ~~ "::usage" :> z];
    badusages = Cases[Map[Reverse, 
    Tally[Flatten[
      Table[Last[Flatten[StringSplit[usagesymbols[[i]], "`"]]], {i, 1, 
        Length[usagesymbols]}]]]], {2, a_} :> "`" <> a <> "::usage"];
    goodusages = Flatten[Cases[StringSplit[usagedata, badusages], {_}]];
    Export[FileNameJoin[{outputDir, "Usage.m"}], goodusages, "Lines"];
    
(*    Print["-- Adding Package usage messages"];
    stmp = OpenAppend[ToFileName[{outputDir}, "Usage.m"], PageWidth -> Infinity];
    WriteString[stmp, "\n", #] & /@ DocumentationBuild`SystemResourcesFromDocs`Private`$PackageUsage;
    Close[stmp]; 
    If[(stmp === $Failed ) || (FileType[ ToFileName[{outputDir}, "Usage.m"] ] === None), 
      Print["-- Failed!"];
      ];*)

  ];

  (* Create UsageText.m *)
  (* Print["Creating UsageText messages file."];
  Print["Export UsageText message file to: "<> ToString@@ToFileName[{outputDir}, "UsageText.m"]];
  result =
    Developer`UseFrontEnd @@ DocumentationBuild`SystemResourcesFromDocs`CreateTextUsageMessageFile[
      selectedfiles,
      ToFileName[{outputDir}, "UsageText.m"]
    ];
    (* Bail *)
  If[(result = $Failed ) || (FileType[ ToFileName[{outputDir}, "UsageText.m"] ] === None),
    Print["-- Failed!"];
    (* Failed build *)
    Print["***** BUILD FAILED! *****"];
    (* Quit with a non-zero exit code *)
    Quit[1]
    ];  *)

  Print["Creating Navigator file."];
  Print["Export Navigator file to: "<> ToString@@ToFileName[{outputDir}, "DocumentationNavigator.nb"]];
  
  result=CreateFunctionNavigator[inputDir, outputDir, lang];
  
    If[(result = $Failed ) || (FileType[ ToFileName[{outputDir}, "DocumentationNavigator.nb"] ] === None),
    Print["-- Failed!"];
    (* Failed build *)
    Print["***** BUILD FAILED! *****"];
    (* Quit with a non-zero exit code *)
    Quit[1]
    ];  
  

];

(******************************************************************************)
(** GetJLinkEtcGuidePageDirs **)

GetJLinkEtcGuidePageDirs[indir_String /; FileType@indir === Directory, language_String:"English"]:=
Module[{paths={}, appDirs},
	appDirs = Flatten@{
	ToFileName[Flatten@{StringSplit[DirectoryName@indir, "/|\\"], #}] & /@ {
(*	{"Applications", "DatabaseLink", "Documentation", language, "Tutorials"},
	{"WebDevelopment", "JLink", "Documentation", language, "Tutorials"},
	{"WebDevelopment", "Applications", "WebServices", "Documentation", language, "Tutorials"},
	{"Java", "Applications", "GUIKit", "docs", "Documentation", language, "Tutorials"},
	{"NET", "NETLink", "Documentation", language, "Tutorials"}
	,*) 
	{"Mathematica", "Packages", "ANOVA", "Documentation", language, "Guides"},
	{"Mathematica", "Packages", "Audio", "Documentation", language, "Guides"},
	{"Mathematica", "Packages", "BarCharts", "Documentation", language, "Guides"},
	{"Mathematica", "Packages", "Benchmarking","Documentation", language, "Guides"},
	{"Mathematica", "Packages", "BlackBodyRadiation", "Documentation", language, "Guides"},
	{"Mathematica", "Packages", "Calendar", "Documentation", language, "Guides"},
	(* {"Mathematica", "Documentation", language, "PackagesMirror", "CCodeGenerator", "Guides"}, *)
	Which[
		language==="Japanese", 
			{"Mathematica", "Paclets", "CCodeGeneratorJ", "CCodeGeneratorJ", "Documentation", language, "Guides"}, 
		language==="ChineseSimplified", 
			{"Mathematica", "Paclets", "CCodeGeneratorZH", "CCodeGeneratorZH", "Documentation", language, "Guides"}, 
		True,
			{"Mathematica", "Paclets", "CCodeGenerator", "CCodeGenerator", "Documentation", language, "Guides"}],
	(* {"Mathematica", "Documentation", language, "PackagesMirror", "CCompilerDriver", "Guides"}, *)
	Which[
		language==="Japanese", 
			{"Mathematica", "Paclets", "CCompilerDriverJ", "CCompilerDriverJ", "Documentation", language, "Guides"}, 
		language==="ChineseSimplified", 
			{"Mathematica", "Paclets", "CCompilerDriverZH", "CCompilerDriverZH", "Documentation", language, "Guides"}, 
		True,
			{"Mathematica", "Paclets", "CCompilerDriver", "CCompilerDriver", "Documentation", language, "Guides"}],
	{"Mathematica", "Packages", "Combinatorica", "Documentation", language, "Guides"},
	(* {"Mathematica", "Documentation", language, "PackagesMirror", "CompiledFunctionTools", "Guides"}, *)
	Which[
		language==="Japanese", 
			{"Mathematica", "Paclets", "CompiledFunctionToolsJ", "CompiledFunctionToolsJ", "Documentation", language, "Guides"}, 
		language==="ChineseSimplified", 
			{"Mathematica", "Paclets", "CompiledFunctionToolsZH", "CompiledFunctionToolsZH", "Documentation", language, "Guides"}, 
		True,
			{"Mathematica", "Paclets", "CompiledFunctionTools", "CompiledFunctionTools", "Documentation", language, "Guides"}],
	{"Mathematica", "Packages", "ComputationalGeometry", "Documentation", language, "Guides"},
	{"Mathematica", "Packages", "ComputerArithmetic", "Documentation", language, "Guides"},
	(*{"Mathematica", "Documentation", language, "PackagesMirror", "ControlSystems", "Guides"},*)
	(*{"Pubs", "Documentation", "RawMaterial", "AddOns", "ControlSystems", "Tutorials"},*)
	Which[
		language==="Japanese", 
			{"Mathematica", "Paclets", "CUDALinkJ", "CUDALinkJ", "Documentation", language, "Guides"}, 
		language==="ChineseSimplified", 
			{"Mathematica", "Paclets", "CUDALinkZH", "CUDALinkZH", "Documentation", language, "Guides"}, 
		True,
			{"Mathematica", "Paclets", "CUDALink", "CUDALink", "Documentation", language, "Guides"}],
	(*{"Mathematica", "PackagesMirror", "DatabaseLink", "Documentation", language, "Guides"},*)
	{"Applications", "DatabaseLink", "Documentation", language, "Guides"},
	{"Mathematica", "Packages", "Developer", "Documentation", language, "Guides"},
	{"Mathematica", "Packages", "EquationTrekker", "Documentation", language, "Guides"},
	{"Mathematica", "Packages", "ErrorBarPlots", "Documentation", language, "Guides"},
	{"Mathematica", "Packages", "Experimental", "Documentation", language, "Guides"},
	Which[
		language==="Japanese", 
			{"Mathematica", "Paclets", "FEMDocumentationJ", "FEMDocumentationJ", "Documentation", language, "Guides"}, 
		language==="ChineseSimplified", 
			{"Mathematica", "Paclets", "FEMDocumentationZH", "FEMDocumentationZH", "Documentation", language, "Guides"}, 
		True,
			{"Mathematica", "Paclets", "FEMDocumentation", "FEMDocumentation", "Documentation", language, "Guides"}],
	{"Mathematica", "Packages", "FiniteFields", "Documentation", language, "Guides"},
	{"Mathematica", "Packages", "FourierSeries", "Documentation", language, "Guides"}, 
	{"Mathematica", "Packages", "FunctionApproximations", "Documentation", language, "Guides"},
	{"Mathematica", "Packages", "Geodesy", "Documentation", language, "Guides"},
	{"Mathematica", "Packages", "GraphUtilities", "Documentation", language, "Guides"},
	(*{"Mathematica", "PackagesMirror", "GUIKit", "Documentation", language, "Guides"},*)
	{"Java", "Applications", "GUIKit", "docs", "Documentation", language, "Guides"},
	{"Mathematica", "Packages", "HierarchicalClustering", "Documentation", language, "Guides"},
	{"Mathematica", "Packages", "Histograms", "Documentation", language, "Guides"},
	{"Mathematica", "Packages", "HypothesisTesting", "Documentation", language, "Guides"},
	(*{"Mathematica", "Documentation", language, "PackagesMirror", "InternalUtilities", "Guides"},*)
	(*{"Mathematica", "Documentation", language, "PackagesMirror", "JLink", "Guides"},*)
	{"WebDevelopment", "JLink", "Documentation", language, "Guides"},
	(* {"Mathematica", "Documentation", language, "PackagesMirror", "LibraryLink", "Guides"}, *)
	Which[
		language==="Japanese", 
			{"Mathematica", "Paclets", "LibraryLinkJ", "LibraryLinkJ", "Documentation", language, "Guides"}, 
		language==="ChineseSimplified", 
			{"Mathematica", "Paclets", "LibraryLinkZH", "LibraryLinkZH", "Documentation", language, "Guides"}, 
		True,
			{"Mathematica", "Paclets", "LibraryLink", "LibraryLink", "Documentation", language, "Guides"}],
	{"Mathematica", "Packages", "LinearRegression", "Documentation", language, "Guides"},
	{"Mathematica", "Packages", "MultivariateStatistics", "Documentation", language, "Guides"},
	{"Mathematica", "Packages", "Music", "Documentation", language, "Guides"},
	{"Mathematica", "LanguageBindings","NETLink", "Documentation", language, "Guides"},
	{"Mathematica", "Packages", "NonlinearRegression", "Documentation", language, "Guides"},
	{"Mathematica", "Packages", "Notation","Documentation", language, "Guides"},
	{"Mathematica", "Packages", "NumericalCalculus", "Documentation", language, "Guides"},
	{"Mathematica", "Packages", "NumericalDifferentialEquationAnalysis", "Documentation", language, "Guides"},
	(* {"Mathematica", "Documentation", language, "PackagesMirror", "OpenCLLink", "Guides"}, *)
	Which[
		language==="Japanese", 
			{"Mathematica", "Paclets", "OpenCLLinkJ", "OpenCLLinkJ", "Documentation", language, "Guides"}, 
		language==="ChineseSimplified", 
			{"Mathematica", "Paclets", "OpenCLLinkZH", "OpenCLLinkZH", "Documentation", language, "Guides"}, 
		True,
			{"Mathematica", "Paclets", "OpenCLLink", "OpenCLLink", "Documentation", language, "Guides"}],
	{"Applications", "ParallelTools", "Parallel", "Documentation", language, "Guides"},
	{"Mathematica", "Packages", "PhysicalConstants", "Documentation", language, "Guides"},
	{"Mathematica", "Packages", "PieCharts", "Documentation", language, "Guides"},
	{"Mathematica", "Packages", "PlotLegends", "Documentation", language, "Guides"},
	{"Mathematica", "Packages", "PolyhedronOperations", "Documentation", language, "Guides"},
	{"Mathematica", "Packages", "Polytopes", "Documentation", language, "Guides"},
	{"Mathematica", "Packages", "PrimalityProving", "Documentation", language, "Guides"},
	{"Mathematica", "Packages", "Quaternions", "Documentation", language, "Guides"},
	{"Mathematica", "Packages", "RegressionCommon", "Documentation", language, "Guides"},
	{"Mathematica", "Packages", "ResonanceAbsorptionLines", "Documentation", language, "Guides"},
	Which[
		language==="Japanese", 
			{"Mathematica", "Paclets", "RLinkJ", "RLinkJ", "Documentation", language, "Guides"}, 
		language==="ChineseSimplified", 
			{"Mathematica", "Paclets", "RLinkZH", "RLinkZH", "Documentation", language, "Guides"}, 
		True,
			{"Mathematica", "Paclets", "RLink", "RLink", "Documentation", language, "Guides"}],
	{"Mathematica", "Packages", "Splines", "Documentation", language, "Guides"},
	{"Mathematica", "Packages", "StandardAtmosphere", "Documentation", language, "Guides"},
	{"Mathematica", "Packages", "StatisticalPlots", "Documentation", language, "Guides"},
	(*{"Mathematica", "Documentation", language, "PackagesMirror", "SymbolicC", "Guides"},*)
	Which[
		language==="Japanese", 
			{"Mathematica", "Paclets", "SymbolicCJ", "SymbolicCJ", "Documentation", language, "Guides"}, 
		language==="ChineseSimplified", 
			{"Mathematica", "Paclets", "SymbolicCZH", "SymbolicCZH", "Documentation", language, "Guides"}, 
		True,
			{"Mathematica", "Paclets", "SymbolicC", "SymbolicC", "Documentation", language, "Guides"}],
	{"Mathematica", "Packages", "Units", "Documentation", language, "Guides"},
	{"Mathematica", "Packages", "VariationalMethods", "Documentation", language, "Guides"},
	{"Mathematica", "Packages", "VectorAnalysis", "Documentation", language, "Guides"},
	{"Mathematica", "Packages", "VectorFieldPlots", "Documentation", language, "Guides"},
	(*{"Mathematica", "Documentation", language, "PackagesMirror", "WebServices", "Guides"},*)
	{"WebDevelopment", "Applications", "WebServices", "Documentation", language, "Guides"},
	{"Mathematica", "Packages", "WorldPlot", "Documentation", language, "Guides"},
	{"Mathematica", "Packages", "XML", "Documentation", language, "Guides"}
	}
	};
	If[ FileType[#] === Directory,
		AppendTo[paths, #],
		Message[GetJLinkEtcGuidePageDirs::notfound, #];
	]& /@ appDirs;
	paths
];
GetJLinkEtcGuidePageDirs[con___]:= (Message[GetJLinkEtcGuidePageDirs::nodir, con]; $Failed);
GetJLinkEtcGuidePageDirs::notfound = "Path not found: `1`";
GetJLinkEtcGuidePageDirs::nodir = "Expected path to directory";


(******************************************************************************)
(** GetJLinkEtcTutorialPageDirs **)

GetJLinkEtcTutorialPageDirs[indir_String /; FileType@indir === Directory, language_String:"English"]:=
Module[{paths={}, appDirs},
	appDirs = Flatten@{
	ToFileName[Flatten@{StringSplit[DirectoryName@indir, "/|\\"], #}] & /@ {
(*	{"Applications", "DatabaseLink", "Documentation", language, "Guides"},
	{"WebDevelopment", "JLink", "Documentation", language, "Guides"},
	{"WebDevelopment", "Applications", "WebServices", "Documentation", language, "Guides"},
	{"Java", "Applications", "GUIKit", "docs", "Documentation", language, "Guides"},
	{"NET", "NETLink", "Documentation", language, "Guides"}
	,*) 
	{"Mathematica", "Packages", "ANOVA", "Documentation", language, "Tutorials"},
	{"Mathematica", "Packages", "Audio", "Documentation", language, "Tutorials"},
	{"Mathematica", "Packages", "BarCharts", "Documentation", language, "Tutorials"},
	{"Mathematica", "Packages", "Benchmarking", "Documentation", language, "Tutorials"},
	{"Mathematica", "Packages", "BlackBodyRadiation", "Documentation", language, "Tutorials"},
	{"Mathematica", "Packages", "Calendar", "Documentation", language, "Tutorials"},
	(* {"Mathematica", "Documentation", language, "PackagesMirror", "CCodeGenerator", "Tutorials"}, *)
	Which[
		language==="Japanese", 
			{"Mathematica", "Paclets", "CCodeGeneratorJ", "CCodeGeneratorJ", "Documentation", language, "Tutorials"}, 
		language==="ChineseSimplified", 
			{"Mathematica", "Paclets", "CCodeGeneratorZH", "CCodeGeneratorZH", "Documentation", language, "Tutorials"}, 
		True,
			{"Mathematica", "Paclets", "CCodeGenerator", "CCodeGenerator", "Documentation", language, "Tutorials"}],
	(*{"Mathematica", "Documentation", language, "PackagesMirror", "CCompilerDriver", "Tutorials"},*)
	Which[
		language==="Japanese", 
			{"Mathematica", "Paclets", "CCompilerDriverJ", "CCompilerDriverJ", "Documentation", language, "Tutorials"}, 
		language==="ChineseSimplified", 
			{"Mathematica", "Paclets", "CCompilerDriverZH", "CCompilerDriverZH", "Documentation", language, "Tutorials"}, 
		True,
			{"Mathematica", "Paclets", "CCompilerDriver", "CCompilerDriver", "Documentation", language, "Tutorials"}],
	(*{"Mathematica", "Documentation", language, "PackagesMirror", "CompiledFunctionTools", "Tutorials"},*)
	Which[
		language==="Japanese", 
			{"Mathematica", "Paclets", "CompiledFunctionToolsJ", "CompiledFunctionToolsJ", "Documentation", language, "Tutorials"}, 
		language==="ChineseSimplified", 
			{"Mathematica", "Paclets", "CompiledFunctionToolsZH", "CompiledFunctionToolsZH", "Documentation", language, "Tutorials"}, 
		True,
			{"Mathematica", "Paclets", "CompiledFunctionTools", "CompiledFunctionTools", "Documentation", language, "Tutorials"}],
	{"Mathematica", "Packages", "Combinatorica","Documentation", language, "Tutorials"},
	{"Mathematica", "Packages", "CompiledFunctionTools", "Documentation", language, "Tutorials"},
	{"Mathematica", "Packages", "ComputationalGeometry", "Documentation", language, "Tutorials"},
	{"Mathematica", "Packages", "ComputerArithmetic", "Documentation", language, "Tutorials"},
	(*{"Mathematica", "Documentation", language, "PackagesMirror", "ControlSystems", "Tutorials"},*)
	(*{"Pubs", "Documentation", "RawMaterial", "AddOns", "ControlSystems", "Tutorials"},*)
	Which[
		language==="Japanese", 
			{"Mathematica", "Paclets", "CUDALinkJ", "CUDALinkJ", "Documentation", language, "Tutorials"}, 
		language==="ChineseSimplified", 
			{"Mathematica", "Paclets", "CUDALinkZH", "CUDALinkZH", "Documentation", language, "Tutorials"}, 
		True,
			{"Mathematica", "Paclets", "CUDALink", "CUDALink", "Documentation", language, "Tutorials"}],
	(*{"Mathematica", "Documentation", language, "PackagesMirror", "DatabaseLink", "Tutorials"},*)
	{"Applications", "DatabaseLink", "Documentation", language, "Tutorials"},
	{"Mathematica", "Packages", "Developer", "Documentation", language, "Tutorials"},
	{"Mathematica", "Packages", "EquationTrekker", "Documentation", language, "Tutorials"},
	{"Mathematica", "Packages", "ErrorBarPlots", "Documentation", language, "Tutorials"},
	{"Mathematica", "Packages", "Experimental", "Documentation", language, "Tutorials"},
	Which[
		language==="Japanese", 
			{"Mathematica", "Paclets", "FEMDocumentationJ", "FEMDocumentationJ", "Documentation", language, "Tutorials"}, 
		language==="ChineseSimplified", 
			{"Mathematica", "Paclets", "FEMDocumentationZH", "FEMDocumentationZH", "Documentation", language, "Tutorials"}, 
		True,
			{"Mathematica", "Paclets", "FEMDocumentation", "FEMDocumentation", "Documentation", language, "Tutorials"}],
	{"Mathematica", "Packages", "FiniteFields", "Documentation", language, "Tutorials"},
	{"Mathematica", "Packages", "FourierSeries", "Documentation", language, "Tutorials"}, 
	{"Mathematica", "Packages", "FunctionApproximations", "Documentation", language, "Tutorials"},
	{"Mathematica", "Packages", "Geodesy", "Documentation", language, "Tutorials"},
	{"Mathematica", "Packages", "GraphUtilities", "Documentation", language, "Tutorials"},
	(*{"Mathematica", "Documentation", language, "PackagesMirror", "GUIKit", "Tutorials"},*)
	{"Java", "Applications", "GUIKit", "docs", "Documentation", language, "Tutorials"},
	{"Mathematica", "Packages", "HierarchicalClustering", "Documentation", language, "Tutorials"},
	{"Mathematica", "Packages", "Histograms", "Documentation", language, "Tutorials"},
	{"Mathematica", "Packages", "HypothesisTesting", "Documentation", language, "Tutorials"},
	(*{"Mathematica", "Documentation", language, "PackagesMirror", "InternalUtilities", "Tutorials"},*)
	(*{"Mathematica", "Documentation", language, "PackagesMirror", "JLink", "Tutorials"},*)
	{"WebDevelopment", "JLink", "Documentation", language, "Tutorials"},
	(* {"Mathematica", "Documentation", language, "PackagesMirror", "LibraryLink", "Tutorials"}, *)
	Which[
		language==="Japanese", 
			{"Mathematica", "Paclets", "LibraryLinkJ", "LibraryLinkJ", "Documentation", language, "Tutorials"}, 
		language==="ChineseSimplified", 
			{"Mathematica", "Paclets", "LibraryLinkZH", "LibraryLinkZH", "Documentation", language, "Tutorials"}, 
		True,
			{"Mathematica", "Paclets", "LibraryLink", "LibraryLink", "Documentation", language, "Tutorials"}],
	{"Mathematica", "Packages", "LinearRegression", "Documentation", language, "Tutorials"},
	{"Mathematica", "Packages", "MultivariateStatistics", "Documentation", language, "Tutorials"},
	{"Mathematica", "Packages", "Music", "Documentation", language, "Tutorials"},
	{"Mathematica", "LanguageBindings","NETLink", "Documentation", language, "Tutorials"},
	{"Applications", "ParallelTools", "Parallel", "Documentation", language, "Tutorials"},
	{"Mathematica", "Packages", "NonlinearRegression", "Documentation", language, "Tutorials"},
	{"Mathematica", "Packages", "Notation", "Documentation", language, "Tutorials"},
	{"Mathematica", "Packages", "NumericalCalculus", "Documentation", language, "Tutorials"},
	{"Mathematica", "Packages", "NumericalDifferentialEquationAnalysis", "Documentation", language, "Tutorials"},
	(* {"Mathematica", "Documentation", language, "PackagesMirror", "OpenCLLink", "Tutorials"}, *)
	Which[
		language==="Japanese", 
			{"Mathematica", "Paclets", "OpenCLLinkJ", "OpenCLLinkJ", "Documentation", language, "Tutorials"}, 
		language==="ChineseSimplified", 
			{"Mathematica", "Paclets", "OpenCLLinkZH", "OpenCLLinkZH", "Documentation", language, "Tutorials"}, 
		True,
			{"Mathematica", "Paclets", "OpenCLLink", "OpenCLLink", "Documentation", language, "Tutorials"}],
	{"Mathematica", "Packages", "PhysicalConstants", "Documentation", language, "Tutorials"},
	{"Mathematica", "Packages", "PieCharts", "Documentation", language, "Tutorials"},
	{"Mathematica", "Packages", "PlotLegends", "Documentation", language, "Tutorials"},
	{"Mathematica", "Packages", "PolyhedronOperations", "Documentation", language, "Tutorials"},
	{"Mathematica", "Packages", "Polytopes", "Documentation", language, "Tutorials"},
	{"Mathematica", "Packages", "PrimalityProving", "Documentation", language, "Tutorials"},
	{"Mathematica", "Packages", "Quaternions", "Documentation", language, "Tutorials"},
	{"Mathematica", "Packages", "RegressionCommon", "Documentation", language, "Tutorials"},
	{"Mathematica", "Packages", "ResonanceAbsorptionLines", "Documentation", language, "Tutorials"},
	Which[
		language==="Japanese", 
			{"Mathematica", "Paclets", "RLinkJ", "RLinkJ", "Documentation", language, "Tutorials"}, 
		language==="ChineseSimplified", 
			{"Mathematica", "Paclets", "RLinkZH", "RLinkZH", "Documentation", language, "Tutorials"}, 
		True,
			{"Mathematica", "Paclets", "RLink", "RLink", "Documentation", language, "Tutorials"}],
	{"Mathematica", "Packages", "Splines", "Documentation", language, "Tutorials"},
	{"Mathematica", "Packages", "StandardAtmosphere", "Documentation", language, "Tutorials"},
	{"Mathematica", "Packages", "StatisticalPlots", "Documentation", language, "Tutorials"},
	(* {"Mathematica", "Documentation", language, "PackagesMirror", "SymbolicC", "Tutorials"}, *)
	Which[
		language==="Japanese", 
			{"Mathematica", "Paclets", "SymbolicCJ", "SymbolicCJ", "Documentation", language, "Tutorials"}, 
		language==="ChineseSimplified", 
			{"Mathematica", "Paclets", "SymbolicCZH", "SymbolicCZH", "Documentation", language, "Tutorials"}, 
		True,
			{"Mathematica", "Paclets", "SymbolicC", "SymbolicC", "Documentation", language, "Tutorials"}],
	{"Mathematica", "Packages", "Units", "Documentation", language, "Tutorials"},
	{"Mathematica", "Packages", "VariationalMethods", "Documentation", language, "Tutorials"},
	{"Mathematica", "Packages", "VectorAnalysis", "Documentation", language, "Tutorials"},
	{"Mathematica", "Packages", "VectorFieldPlots", "Documentation", language, "Tutorials"},
	(*{"Mathematica", "Documentation", language, "PackagesMirror", "WebServices", "Tutorials"},*)
	{"WebDevelopment", "Applications", "WebServices", "Documentation", language, "Tutorials"},
	{"Mathematica", "Packages", "WorldPlot", "Documentation", language, "Tutorials"},
	{"Mathematica", "Packages", "XML", "Documentation", language, "Tutorials"}
	}
	};
	If[ FileType[#] === Directory,
		AppendTo[paths, #],
		Message[GetJLinkEtcTutorialPageDirs::notfound, #];
	]& /@ appDirs;
	paths
];
GetJLinkEtcTutorialPageDirs[con___]:= (Message[GetJLinkEtcTutorialPageDirs::nodir, con]; $Failed);
GetJLinkEtcTutorialPageDirs::notfound = "Path not found: `1`";
GetJLinkEtcTutorialPageDirs::nodir = "Expected path to directory";

(*******************************************

	Code to create function browser data 
		from brianv (Brian Van Vertloo)

********************************************)
"VirtualBook.nb"

(* Since the internal function uses a list of filenames to mine the guide page data, this function, which takes 
   a list of filenames, simply passes it along to the internal function. *)

CreateFunctionBrowserData[filepaths_List] := 
 CreateFunctionBrowserDataInternal[filepaths]


(* This one, however, takes a directory that contains the guide pages to mine, and therefore needs to transform
   that input into a list of filenames to pass to the internal function. *)

CreateFunctionBrowserData[directory_String] := 
 Module[{files}, 
  files = Complement[
    FileNames["*.nb", directory, 
     Infinity], {"SummaryOfNewFeaturesIn60.nb", 
     "AlphabeticalListing.nb", 
     "AlphabeticalListingOfMathLinkCFunctions.nb"}];
  CreateFunctionBrowserDataInternal[files]
  ]


(* This is the internal data collection function. The heavy lifting is done by the recursive function getlinkslist, but there are other
   functions that facilitate assigning depth data to specific pages and providing the necessary cleanup required to get the data into shape. *)

CreateFunctionBrowserDataInternal[fullPaths_List] :=
    Module[ {files=fullPaths},

        ToList[x___] := If[ !ListQ[x], {x}, x];
        unsortedUnion[x___] :=
            (First/@Tally[ToList[x]]);
 
 		$HandlePrimaryObject = False;
 
        $Root = "Mathematica.nb"; 

        LinkTypeQ[x___] := MemberQ[{x},ButtonData,Infinity] && StringMatchQ[ButtonData/.FilterRules[{x},ButtonData],"paclet:guide"~~___]; 
        CleanupName[s_] := 
        	StringReplace[s,
        		{" \[RightGuillemet]"->""," \[FilledRightTriangle]"->"","\[FilledRightTriangle]"->""}];
        CleanupItalics[s_] :=
            s/.{StyleBox["Mathematica",FontSlant->"Italic"]-> "Mathematica",
				StyleBox["Mathematica ",FontSlant->"Italic"]-> "Mathematica ",
				StyleBox["Mathematica\[LongDash]",FontSlant->"Italic"]-> "Mathematica\[LongDash]",
            	StyleBox["DatabaseLink",FontSlant->"Italic"]->"DatabaseLink",
            	StyleBox["MathLink",FontSlant->"Italic"]->"MathLink",
            	StyleBox[" MathLink", FontSlant->"Italic"]->" MathLink",
            	StyleBox["Combinatorica",FontSlant->"Italic"]->"Combinatorica",
            	StyleBox["GUIKit",FontSlant->"Italic"]->"GUIKit",
            	StyleBox["Mathematica\[LongDash][Root Guide Page]", FontSlant->"Italic"]-> "Mathematica-[Root Guide Page]",
				StyleBox["q",FontSlant->"Italic"]->"q",
				StyleBox["LibraryLink",FontSlant->"Italic"]->"LibraryLink",
				StyleBox["CUDALink",FontSlant->"Italic"]->"CUDALink",
				StyleBox[str_String, FontSlant->"Italic"]:>str,
				StyleBox[" ",__]->" ",
            		AdjustmentBox["Mathematica",___]->"Mathematica"
            	}/.{TextData[x___]:> StringJoin[x]};
        link[name_String,uri_String] :=
            Hyperlink[name,"paclet:"<>uri];
        
		(* This function essentially finds all guide -> guide links to create the NearestFunction by which they will be sorted. *)
		grabguideswlink[nb_] :=
            ReplaceAll[Select[(
            			{Transmogrify`ConvertToString[#],
            					unsortedUnion[Cases[#,ButtonBox[y_,x___]:>(ButtonData/.FilterRules[{x},ButtonData]),Infinity]]}&/@
            						(DeleteCases[Cases[nb,Cell[___,"GuideFunctionsSubsection"|"TOCChapter"|"GuideTOCLink",___],Infinity],
            										Cell[___,"FutureExample",___],Infinity])
            				),StringQ[#[[1]]]&&(#[[1]]!="...")&]
            				,
            					{x_,{StringReplace[___]}}:>{x,{x}}];

		(* guidelinks is the embodiment of the guide->guide links *)
        guidelinks = 
        	ReplaceAll[DeleteCases[
    			Module[ {title,newtitle,s1,s2,x,uri},
               		With[ {nbdata = Quiet[Get[#]],name = Last[StringSplit[StringReplace[#, ".nb" -> ""], "\\" | "/"]]},
                     	uri = First@Cases[nbdata, Cell[str_String, "Categorization", ___, CellLabel->"URI",___] :> str, Infinity];
                     	If[ !MemberQ[{DocumentationBuild`Utils`DocumentFlag[nbdata]},"ExcisedFlag",Infinity] && 
							 (!MemberQ[{DocumentationBuild`Utils`DocumentFlag[nbdata]},"FutureFlag",Infinity] || MemberQ[nbdata,"UnderDevelopment",Infinity]),
                                	    title = Cases[nbdata,Cell[t_,"GuideTitle"|"GuideTOCTitle",___]:>t,Infinity];
                                	    newtitle = CleanupItalics[title];
                                    	x = unsortedUnion[Quiet@Map[CleanupName,grabguideswlink[nbdata],{2}]];
                                   		{link[name,uri],x}
                      		]
               			]
                    ]&/@files,Null],{x_,{{y_,{}}}}:>{x,{}}];

        guidelinks = DeleteCases[guidelinks,{_,{}},Infinity];
        
		(* Here the NearestFunction is created, with t serving as the graph on which it's built. *)
		t = Table[guidelinks[[i,1,2]]->guidelinks[[i,2,j,2,1]],{i,Length[guidelinks]},{j,Length[guidelinks[[i,2]]]}];
        t2 = Flatten[t];
        f = Nearest[t2];
        
		(* This function is the run through of the actual authoring notebooks. It strips out all link data and grouping data (such as delimiters). *)
		graballwdelims[nb_] :=
            Module[ {lis = {}},
                With[ {nbdata = 
                	ReplaceAll[ReplaceAll[nb, {
                		Cell[___,"FutureExample",___]->Cell[Null,"FutureExample"],
						(Cell[___,"TentativeExample",___]|Cell[___,CellDingbat->Cell[___,"TentativeID",___],___])->Cell[Null,"TentativeExample"],
						Cell[___,CellDingbat->db_/;MemberQ[db,"BrighterFlag",Infinity],___]|Cell[BoxData[ButtonBox[___,BaseStyle->{___,"BrighterFlag",___},___],___],___]->Cell[Null,"FutureExample"],
                		Cell[TextData[f:{___,Cell[___,"InlineFormula"|"InlineCode"|"InlineCharacterDisplay"|"InlineCharacterName",___]..,___}],
                								___,"InlineGuideFunctionListing",___]:>f,
                		Cell[BoxData[RowBox[{x_," "}]],"InlineGuideFunction"]:>Cell[BoxData[x],"InlineGuideFunction"],
	                	Cell[___,"GuideDelimiter",___]->Cell[BoxData[BoxData["DELIMITERHERE"]],"GuideDelimiter"],
    	            	Cell[TextData[g:{___,Cell[___,"InlineCode"],___}],___,"GuideText",___]:>g,
        	        	Cell[TextData[{other___,Cell[___,"InlineFormula"|"InlineCode",___],___}],___,"GuideText"|"GuideAbstract",___]:>
            	    		Cell[TextData[{other,Cell[Null,"InlineFormulaKEEPOUT"]}]],
            	    	FormBox[Cell[___,"InlineFormula",___],TraditionalForm]:>
            	    		Cell[Null,"InlineFormulaKEEPOUT"],
                		Cell[___,"TOCExcludedObject",___]|Cell[TextData[ButtonBox[__,BaseStyle->{___,"TOCExcludedObject",___},___],___],___]->Cell[BoxData[BoxData["EXCLUDED"]],"GuideDelimiter"]}],
                	{Cell[(BoxData | TextData)[ButtonBox[___,BaseStyle->{___,"BrighterFlag",___},___],___],___]->Cell[Null,"FutureExample"],
					 Cell[(BoxData | TextData)[StyleBox[___,"FutureExample",___],___],___]->Cell[Null,"FutureExample"]}]},                	
                    
                    	ReplaceAll[Select[If[ StringQ[#[[1]]], #, {#[[1,1]],Flatten@Rest[#]} ]&/@
                    		({Transmogrify`ConvertToString[#],unsortedUnion[Cases[#,ButtonBox[y_,x___]:>(ButtonData/.FilterRules[{x},ButtonData]),Infinity]]} &/@ 
                    			DeleteCases[Cases[nbdata,
                    				Cell[BoxData[ButtonBox[___]],___,"InlineGuideFunction"|"InlineCharacterDisplay"|"InlineCharacterName",___]|
                    				Cell[___,"InlineFormula"|"InlineCode"|"InlineCharacterDisplay",___]|
                    				Cell[___,"GuideDelimiter",___]|StyleBox[ButtonBox[___],___,"MenuName",___]|
                    				Cell[___,"GuideFunctionsSubsection"|"TOCChapter"|"GuideTOCLink",___]|
                    				Cell[TextData[{ButtonBox[___],___,StyleBox[ButtonBox[___],___],___}],___,"GuideFunctionsSubsection",___],Infinity]
                    			,
                    				Cell[___,"FutureExample",___]|Cell[___,"TentativeExample",___]|
                    				Cell[BoxData[_String|FractionBox[___]|SubscriptBox[___]|SuperscriptBox[___]|SubsuperscriptBox[___]|
                    					OverscriptBox[___]|UnderscriptBox[___]|UnderoverscriptBox[___]],___,"InlineFormula"|"InlineCode",___],Infinity] /. 
                    						{StyleBox["Mathematica",FontSlant->"Italic"]->"Mathematica",TextData[x___]:>StringJoin[x]}
                    		),(#[[1]]!="...")&], If[$HandlePrimaryObject,
							{{x_,{ButtonData,"Link"}}|{x_,{ButtonData,{"Link"}}}:>{x,{"paclet:ref/"<>x}},
								{x_,{ButtonData,{"Link","PrimaryObject"}}}:>{x,{"paclet:ref/"<>x,"PrimaryObject"}},
								{x_String,{ButtonData,{"Link","ExcludedObject"}}}:>{x,{"paclet:ref/"<>x}},{x_String,{str_String,"Link"}}:>{x,{str}},{x_String,{str_String,{"Link","ExcludedObject"}}}:>{x,{str}}},
							{x_,{ButtonData}}:>{x,{"paclet:ref/"<>x}}]]
                ]
            ];

		(* New In 6.0 guides are to be treated differently, with no headers (except other New in 6 pages) being treated as links. 
			This function makes it happen. *)

		newin6fix[lis_List] := Quiet[
			If[StringQ[#[[2,1]]]&&StringMatchQ[#[[2,1]],"paclet:guide/"~~__]&&!StringMatchQ[#[[2,1]],"paclet:guide/NewIn60"~~__]&&!StringMatchQ[#[[2,1]],"paclet:guide/"~~__~~"Menu"],
				{#[[1]],{}},{#[[1]],#[[2]]}]&/@lis
			];

        finallinks = 
        	DeleteCases[Module[ {x,title,newtitle, uri},
    				With[ {nbdata = Quiet[Get[#]],name = Last[StringSplit[StringReplace[#, ".nb" -> ""], "\\" | "/"]]},
						uri = First@Cases[nbdata, Cell[str_String, "Categorization", ___, CellLabel->"URI",___] :> str, Infinity];
						If[ !MemberQ[{DocumentationBuild`Utils`DocumentFlag[nbdata]},"ExcisedFlag",Infinity] && 
							 (!MemberQ[{DocumentationBuild`Utils`DocumentFlag[nbdata]},"FutureFlag",Infinity] || MemberQ[nbdata,"UnderDevelopment",Infinity]),
                        	title = Cases[nbdata,Cell[t_,"GuideTitle"|"GuideTOCTitle",___]:>t,Infinity];
                            newtitle = CleanupItalics[title];
                            x = Quiet@Map[StringReplace[#," \[RightGuillemet]"->""]&,
									If[StringMatchQ[name,"NewIn60"~~__],newin6fix[graballwdelims[nbdata]],graballwdelims[nbdata]],
								{2}];
                            {link[name,uri],newtitle,x}
                         ]
                    ]
				]&/@files,Null];

        finallinks2 = DeleteCases[finallinks,{_,_,{}}|{_,_,{{"EXCLUDED",{}}..}}];
        finallinks2 = DeleteCases[finallinks2,{_,{"New in 6.0: Alphabetical Listing"},___}];
        finallinks2 = DeleteCases[finallinks2,{_String,{path_String}}/;StringMatchQ[path,"paclet:guide/"~~__~~"Menu"],Infinity];

        finallinks3 = Append[#,{}]&/@finallinks2;

		(* This function will assign a depth to each guide page in finallinks3. It's calculated below using the NearestFunction created above (f). *)
        assigndepth[str_String,i_Integer] :=
            AppendTo[finallinks3[[Cases[Position[finallinks3,str],{_,1,2}][[1,1]],4]],i];

        Map[assigndepth[#,0]&,{"paclet:guide/CoreLanguageOverview","paclet:guide/MathematicsAndAlgorithmsOverview","paclet:guide/DataManipulationOverview","paclet:guide/ComputableDataOverview","paclet:guide/SystemsInterfacesAndDeploymentOverview","paclet:guide/DynamicInteractivityOverview","paclet:guide/VisualizationAndGraphicsOverview","paclet:guide/NotebooksAndDocumentsOverview"}];
		assigndepth["paclet:guide/StandardExtraPackages",1];
        Quiet@Map[assigndepth[#,1]&,f[#]]&/@{"paclet:guide/CoreLanguageOverview","paclet:guide/MathematicsAndAlgorithmsOverview","paclet:guide/DataManipulationOverview","paclet:guide/ComputableDataOverview","paclet:guide/SystemsInterfacesAndDeploymentOverview","paclet:guide/DynamicInteractivityOverview","paclet:guide/VisualizationAndGraphicsOverview","paclet:guide/NotebooksAndDocumentsOverview"};
        Quiet@Map[assigndepth[#,2]&,Union@Flatten[f/@f[#]]]&/@{"paclet:guide/CoreLanguageOverview","paclet:guide/MathematicsAndAlgorithmsOverview","paclet:guide/DataManipulationOverview","paclet:guide/ComputableDataOverview","paclet:guide/SystemsInterfacesAndDeploymentOverview","paclet:guide/DynamicInteractivityOverview","paclet:guide/VisualizationAndGraphicsOverview","paclet:guide/NotebooksAndDocumentsOverview","paclet:guide/StandardExtraPackages"};
        Quiet@Map[assigndepth[#,3]&,Union@Flatten[f/@Union@Flatten[f/@f[#]]]]&/@{"paclet:guide/CoreLanguageOverview","paclet:guide/MathematicsAndAlgorithmsOverview","paclet:guide/DataManipulationOverview","paclet:guide/ComputableDataOverview","paclet:guide/SystemsInterfacesAndDeploymentOverview","paclet:guide/DynamicInteractivityOverview","paclet:guide/VisualizationAndGraphicsOverview","paclet:guide/NotebooksAndDocumentsOverview","paclet:guide/StandardExtraPackages"};
        Quiet@Map[assigndepth[#,4]&,Union@Flatten[f/@Union@Flatten[f/@Union@Flatten[f/@f["paclet:guide/SystemsInterfacesAndDeploymentOverview"]]]]];
        finallinks4 = Map[{#[[1]],#[[2]],#[[3]],Min[#[[4]]]}&,finallinks3];
        
        (* Begin new code to handle PrimaryObject tagging *)

		If[$HandlePrimaryObject,
		(duplicatecount = Select[({#[[1]],Length[#]})&/@
			Gather[First/@Cases[finallinks4,{s_String,{path_String,rest___}}/;StringMatchQ[path,"paclet:"~~___~~"ref/"~~__],Infinity]],
			#[[2]]>1&];
		finallinks4 = DeleteCases[finallinks4,{s_String,{path_String,rest___}}/;
			(StringMatchQ[path,"paclet:"~~___~~"ref/"~~__]&&MemberQ[First/@duplicatecount,s]&&!MemberQ[{rest},"PrimaryObject"]),Infinity];
		finallinks4 = ReplaceAll[finallinks4,{s_String,{path_String,rest__}}:>{s,{path}}])];

		(* End PO Code *)        
        
        finallinkstest = finallinks4;

        (* This Do loop assigns depths to all guide links INSIDE of a guide entry in finallinkstest. This will be used to determine whether to
		   follow a guide branch. *)
        Do[finallinkstest[[j]] = 
        	With[ {depth = #[[4]]+1},
            	{#[[1]],#[[2]],ReplaceAll[#[[3]],{x_,{s_}}:>{x,{s},depth}/;StringMatchQ[s,"paclet:"~~___~~"guide/"~~__]],#[[4]]}
                ]&[finallinkstest[[j]]],{j,Length[finallinkstest]}];
        finallinks4 = finallinkstest;
        Do[finallinks4[[j]] = ReplaceAll[finallinks4[[j]],{link_,{path___},depth___Integer}:>{link,{},{path},depth}],{j,Length[finallinks4]}];
        Do[finallinks4[[j]] = ReplaceAll[finallinks4[[j]],{link_,{},{path_String},depth_}:>{link,finallinks4[[Cases[Position[finallinks4,Last[StringSplit[path,"/"]]],{n_,1,1}:>n][[1]]]][[2,1]],{path},depth} /; StringMatchQ[path,("paclet:"~~___~~"guide/"~~__)]],{j,Length[finallinks4]}];
        Do[finallinks4[[j]] = ReplaceAll[finallinks4[[j]],{x_String,l_List,{path_String}}:>{Last[StringSplit[path,"/"]],l,{path}}/;(!StringMatchQ[x,Last[StringSplit[path,"/"]]|"\""~~Last[StringSplit[path,"/"]]~~"\""]&&!StringMatchQ[path,("paclet:ref/program/"~~__)|("paclet:ref/menuitem/"~~__)])],{j,Length[finallinks4]}];
		
		(* This function removes duplicate guide links from the list of links *)
		cleanguidelinks[l_List]:=Module[{lis={}},
			Do[If[MatchQ[l[[i]],{_String,_String,{_String},_Integer}],
				If[!MemberQ[lis[[All,3]],l[[i,3]]],
				AppendTo[lis,l[[i]]]],
				AppendTo[lis,l[[i]]]],
			{i,Length[l]}];
			lis];

		(* Make all menu item references be on guide/MenuItems *)
		finallinks4 = Join[Take[finallinks4,Position[finallinks4,"MenuItems"][[1,1]]-1],
			{ReplaceAll[finallinks4[[Position[finallinks4,"MenuItems"][[1,1]]]],{s_String,{},{path_String}}:>{s,{Hold},{path}}]},
			Take[finallinks4,-(Length[finallinks4]-Position[finallinks4,"MenuItems"][[1,1]])]];
		finallinks4 = DeleteCases[finallinks4,{_String,{},{path_String}}/;StringMatchQ[path,"paclet:ref/menuitem/"~~__],Infinity];
		finallinks4 = ReplaceAll[finallinks4,{s_String,{Hold},{path_String}}:>{s,{},{path}}];
		
		finallinks4={#[[1]],#[[2]],cleanguidelinks[#[[3]]],#[[4]]}& /@ finallinks4;		

		(* These global variables will be used to determine how far the recursive function has gone (iteration), as well as an alterable 
		   depth limit (maxdepth). *)
        maxdepth = 3;
        iteration = 0;

		(* This function removes duplicate symbol entries except Delimiters. *)
        delimsafeUnsortedUnion[lis_List]:= Module[{newlis={}},
			Do[
				If[lis[[i,1]]==="DELIMITERHERE",AppendTo[newlis,lis[[i]]],
					If[!MemberQ[newlis,lis[[i]]],AppendTo[newlis,lis[[i]]]]],
			{i,Length[lis]}];
			newlis];
        
		(* This function strips off delimiters on the end of a list of symbols. *)
		stripenddelimiters[lis_List] :=
            Module[ {l = lis},
                Do[If[ First[l][[1]]=="DELIMITERHERE",
                       l = Delete[l,1]
                   ];
                   If[ Last[l][[1]]=="DELIMITERHERE",
                       l = Delete[l,Length[l]]
                   ];,{2}];
                l
            ];

		(* This is the recursive function that goes through and creates the data. Not recommended to debug without a stack view, although the
		   commented-out Print statement at the beginning of the Do loop can help. *)
        getlinkslist[index_Integer] := 
 			Module[{info = finallinks4[[index]], lis = {}, lispostdelim = {}, postdelim = False, afterguide = False,
		  	 guide = {"PLACEHOLDER", "PLACEHOLDER", {"paclet:guide/PLACEHOLDER"}, 0}, ret = {}}, 
  				If[info[[4]] <= maxdepth, 
				   With[{guts = info[[3]], name = info[[1, 1]], depth = info[[4]]}, 
				    Do[(*Print["------------------"];Print/@{i,guide,lis,lispostdelim,postdelim,afterguide,ret};*)
					If[afterguide,lis={}];
					If[StringMatchQ[ToString@guts[[i, 3, 1]], "paclet:"~~___~~"guide/"~~__], 
				    (If[guide[[1]] == "EXCLUDED", lis = {}, 
       		 		If[(Union[lis] != {} && 
       	    		Union[lis] != {{"DELIMITERHERE", {}, {}}}), 
         			If[guide[[3]] == {}, 
          				AppendTo[ret, {StringReplace[guide[[1]],"-"->"\[Hyphen]"], {}, delimsafeUnsortedUnion[stripenddelimiters[lis]]}]; lis = {}, 
          				If[lis != {} && guide[[1]] == "PLACEHOLDER", 
           					AppendTo[ret, delimsafeUnsortedUnion[stripenddelimiters[lis]]]; 
           					lis = {}]]; lis = {}]]);
					If[postdelim && lispostdelim!={},AppendTo[ret,If[lispostdelim!={{"DELIMITERHERE",{},{}}},Prepend[delimsafeUnsortedUnion[stripenddelimiters[lispostdelim]],{"DELIMITERHERE",{},{}}],delimsafeUnsortedUnion[stripenddelimiters[lispostdelim]]]];
						ret=ReplaceAll[ret,Except[{_String,{},{({_String,{},{_String}}|{"DELIMITERHERE",{},{}})..}},{f:(({_String,{},{_String}}|{"DELIMITERHERE",{},{}})..)}]:>f];
						lispostdelim={}];
						
						guide = guts[[i]]; afterguide = False; postdelim = False;
      				If[guide[[4]] <= maxdepth && iteration <= maxdepth, iteration++;
        				AppendTo[
       					 ret, {StringReplace[guide[[2]],"-"->"\[Hyphen]"], guide[[3, 1]], getlinkslist[Select[(First /@ Cases[Position[finallinks4, 
               				StringReplace[guts[[i, 3, 1]], 
                			"paclet:"~~___~~"guide/" -> ""]], {_, 1, 1}]), MatchQ[ finallinks4[[#, 1]], Hyperlink[_, guts[[i, 3, 1]]] ] &][[1]]]}]; iteration--; lis = {}; afterguide = True, guide[[3]] = {}],
      				If[(! StringMatchQ[guts[[i, 1]], "DELIMITERHERE"]) && MatchQ[guts[[i, 3]], {}], 
      				(If[guide[[1]] == "EXCLUDED", lis = {}, 
         				If[(Union[lis] != {} && Union[lis] != {{"DELIMITERHERE", {}, {}}}) && guide[[3]] == {}, 
          					AppendTo[ret, {StringReplace[guide[[1]],"-"->"\[Hyphen]"], {}, delimsafeUnsortedUnion[stripenddelimiters[lis]]}]; lis = {}, 
          					If[lis != {} && guide[[1]] == "PLACEHOLDER", 
           						AppendTo[ret, delimsafeUnsortedUnion[stripenddelimiters[lis]]]; 
           						ret = ret[[1]]; lis = {}]]]);
						If[postdelim && lispostdelim!={},AppendTo[ret,If[lispostdelim!={{"DELIMITERHERE",{},{}}},Prepend[delimsafeUnsortedUnion[stripenddelimiters[lispostdelim]],{"DELIMITERHERE",{},{}}],delimsafeUnsortedUnion[stripenddelimiters[lispostdelim]]]];
							ret=ReplaceAll[ret,Except[{_String,{},{({_String,{},{_String}}|{"DELIMITERHERE",{},{}})..}},{f:(({_String,{},{_String}}|{"DELIMITERHERE",{},{}})..)}]:>f];
							lispostdelim={}];
           			guide = guts[[i]]; afterguide = False; postdelim = False, 
						If[postdelim==True,If[(StringMatchQ[guts[[i, 1]], "DELIMITERHERE"] || 
          			StringMatchQ[ToString@guts[[i, 3, 1]], "paclet:"~~___~~"ref/"~~__]), 
        				AppendTo[lispostdelim, guts[[i]]]],If[(StringMatchQ[guts[[i, 1]], "DELIMITERHERE"] || 
          			StringMatchQ[ToString@guts[[i, 3, 1]], "paclet:"~~___~~"ref/"~~__]), 
        				AppendTo[lis, guts[[i]]]];
						If[StringMatchQ[guts[[i,1]],"DELIMITERHERE"]&&guide[[1]]!="PLACEHOLDER",postdelim=True;afterguide=False];]]], {i, Length[guts]}]; 
    		If[guide[[1]] == "EXCLUDED", lis = {}, 
     			If[(Union[lis] != {} && Union[lis] != {{"DELIMITERHERE", {}, {}}}) && guide[[3]] == {}, 
      				AppendTo[ret, {StringReplace[guide[[1]],"-"->"\[Hyphen]"], {}, delimsafeUnsortedUnion[stripenddelimiters[lis]]}]; lis = {}, 
      				If[lis != {} && guide[[1]] == "PLACEHOLDER", 
       					AppendTo[ret, delimsafeUnsortedUnion[stripenddelimiters[lis]]]; 
       					ret = ret[[1]]; lis = {}]]];
			If[lispostdelim != {} && lis != {} && guide[[1]] != "PLACEHOLDER", AppendTo[ret,{StringReplace[guide[[1]],"-"->"\[Hyphen]"], guide[[2]], delimsafeUnsortedUnion[stripenddelimiters[lis]]}];lis={};guide = {"PLACEHOLDER", "PLACEHOLDER", {"paclet:guide/PLACEHOLDER"}, 0}];
			If[postdelim==True && lis=={},AppendTo[ret,If[lispostdelim!={{"DELIMITERHERE",{},{}}},Prepend[delimsafeUnsortedUnion[stripenddelimiters[lispostdelim]],{"DELIMITERHERE",{},{}}],delimsafeUnsortedUnion[stripenddelimiters[lispostdelim]]]];
				ret=ReplaceAll[ret,Except[{_String,{},{({_String,{},{_String}}|{"DELIMITERHERE",{},{}})..}},{f:(({_String,{},{_String}}|{"DELIMITERHERE",{},{}})..)}]:>f];lispostdelim={};postdelim=False;]]; 
       	ret]];

		getlinkspkg[index_Integer]:=Module[{info=finallinks4[[index]],ret={},current,guidestoget={},header},
			With[{guts=info[[3]]},
			Do[
			(* Get the next item to look at *)
			current=guts[[i]];
			(* For this page, Delimiters are useless *)
			If[current[[1]]!="DELIMITERHERE",
			(* If it's a non-linking header, dump the list into the return *)
			If[current[[3]]=={}&&guidestoget!={},AppendTo[ret,{header[[1]],None,{StringReplace[#[[2]],"-"->"\[Hyphen]"],#[[3,1]],getlinkslist[Position[finallinks4,Last[StringSplit[#[[3,1]],"/"]]][[1,1]]]}&/@guidestoget}];guidestoget={}];	    
			(* Get the opening unlinked header and store it for later *)
			If[current[[3]]=={}&&guidestoget=={},header = current];
			(* If it's a link, drop it into the list of guides to get *)
			If[StringMatchQ[current[[3,1]],"paclet:"~~__~~"guide/"~~__],AppendTo[guidestoget,current]]]
			,{i,Length[guts]}]];
			ret];

		pkgs = Quiet@getlinkspkg[Position[finallinks4,"StandardExtraPackages"][[1,1]]];

		(* Reads in the Root Guide page and strips out data for where to start *)
		rootlinks = Module[{nbdata=Quiet[Get[Select[fullPaths,(Last[StringSplit[#,"\\"|"/"]]=="Mathematica.nb")&][[1]]]]},
						ReplaceRepeated[{Transmogrify`ConvertToString[#],
							unsortedUnion[Cases[#,ButtonBox[y_,x___]:>(ButtonData/.FilterRules[{x},ButtonData]),Infinity]]}&/@
							Cases[nbdata, c:Cell[___, "GuideFunctionsSection", ___] /; 
							  !MatchQ[c, Cell[___, "FutureExample", ___]], Infinity],
					      {{a_,{b_}}:>{a,b},
						  {s1_,s2_}:>{s1,s2,Position[finallinks4,Last[StringSplit[s2,"\\"|"/"]]][[1,1]]}}]];
		
		(* Creates the data from the above-assigned starting places. *)
        Quiet[fulllist = DeleteCases[ReplaceAll[rootlinks,{s1_,s2_,index_}:>{s1,s2,getlinkslist[index]}],
					{{},_String,DocumentationTools`Utilities`Private`getlinkslist[___]},Infinity]];

		(* Make some touchups/replacements to the data *)
        {fulllist2, pkgfull} = 
        		DeleteCases[ReplaceRepeated[#, {
					{n_String,s_,(f:(({_String,{},{_String}}|{"DELIMITERHERE",{},{}})..))}:>{n,s,{f}},
					{n_String,path_String,{lis___List,{}}}:>{n,path,{lis}},
        			{n_String,{},{path_String}}:>{n,path}/;StringMatchQ[path,("paclet:"~~___~~"ref/"~~__)],
        			{n_String,path_String}:>{"\""<>n<>"\"",path}/;(StringMatchQ[path,"paclet:ref/format/"~~__]&&!StringMatchQ[n,"\""~~__~~"\""]),
        			{n_String,{},{refs__}}:>{n,None,{refs}},{{},x__}:>{x},
        			{"DELIMITERHERE",{},{}}->Delimiter,"DELIMITERHERE"->Delimiter,
					{one___,{x_String,path_String},Delimiter,two___}:>{one,{x,path,Delimiter},two},
					
					(* Commented out this replacement because it was removing valid delimiters *)
					(*{s_String,path_String/;StringMatchQ[path,"paclet:"~~___~~"guide/"~~__],{first___,Delimiter,rest___}}:>{s,path,{first,rest}},*)
					
					{refs:{({_String,_String}|{_String,_String,Delimiter})..},gbranches:{_String,_String|None,_List}..}:>{Sequence@@refs,gbranches}}],
					{_String,{},{}}|{_String,_String,{}},Infinity]& /@ {fulllist, pkgs};

		pkgfull = {"navigatorPackages","paclet:guide/StandardExtraPackages", 
			ReplaceAll[ReplaceAll[
				ReplaceRepeated[pkgfull,
					{stuff___,f_List,Delimiter,patt:({s_String,path_String}|{s_String,path_String,Delimiter}),r___}:>
						{stuff,Sequence@@ReplaceAll[{f},patt2:Last[Cases[{f},{st_String,path2_String},Infinity]]:>Append[patt2,Delimiter]],patt,r}],{Delimiter,rest___}:>{rest}],
							Except[{_String,_String,Delimiter},{f__,Delimiter}]:>{f}]};

		(* Extract the "New in 6.0" sections *)
		newin6 =
			{"navigatorLabelNewIn6","paclet:guide/NewIn60AlphabeticalListing",unsortedUnion[Cases[fulllist2,{x_String, y_String /; StringMatchQ[y,"paclet:guide/NewIn60"~~("CoreLanguage"|"MathematicsAndAlgorithms"|"DataHandlingAndDataSources"|"SystemInterfacesAndDeployment"|"DynamicInteractivity"|"VisualizationAndGraphics"|"NotebooksAndDocuments")],{___}},Infinity]]};

		newin7 =
		    {"navigatorLabelNewIn7","paclet:guide/NewIn70AlphabeticalListing",Cases[fulllist2,{x_String,y_String /; StringMatchQ[y,"paclet:guide/NewIn70"~~("CoreLanguage"|"MathematicsAndAlgorithms"|"DataHandlingAndDataSources"|"DataManipulation"|"ComputableData"|"SystemsInterfacesAndDeployment"|"DynamicInteractivity"|"VisualizationAndGraphics"|"NotebooksAndDocuments")],{___}},Infinity]};
		    
		newin8 =
		    {"navigatorLabelNewIn8","paclet:guide/NewIn80AlphabeticalListing",unsortedUnion[Cases[fulllist2,{x_String,y_String /; StringMatchQ[y,"paclet:guide/NewIn80"~~("CoreLanguage"|"MathematicsAndAlgorithms"|"DataHandlingAndDataSources"|"DataManipulation"|"ComputableData"|"SystemsInterfacesAndDeployment"|"DynamicInteractivityAndComputableData"|"DynamicInteractivity"|"VisualizationAndGraphics"|"NotebooksAndDocuments")],{___}},Infinity]]};
		    
		newin9 =
		    {"navigatorLabelNewIn9","paclet:guide/NewIn90AlphabeticalListing",unsortedUnion[Cases[fulllist2,{x_String,y_String /; StringMatchQ[y,"paclet:guide/NewIn90"~~("CoreLanguage"|"MathematicsAndAlgorithms"|"DataHandlingAndDataSources"|"DataManipulation"|"ComputableData"|"SystemsInterfacesAndDeployment"|"DynamicInteractivityAndComputableData"|"DynamicInteractivity"|"VisualizationAndGraphics"|"NotebooksAndDocuments")],{___}},Infinity]]};

		addonsandpkgs = {"navigatorAddOnsAndPackages",None,{pkgfull}};

		(* Make unlinked headers point to their respective parent pages *)
		fulllist2 =
			ReplaceRepeated[fulllist2,{n_String,path_String,{a___,{g_String,None,{b___}},c___}}:>{n,path,{a,{g,path,{b}},c}}];

		{newin6,newin7,newin8,newin9} =
			ReplaceRepeated[#,{n_String,path_String,{a___,{g_String,None,{b___}},c___}}:>{n,path,{a,{g,path,{b}},c}}]& /@ {newin6,newin7,newin8,newin9};
		
		whatsnew =
			{"navigatorLabelNew", None, {newin9, newin8, newin7, newin6}};

        subfinaldata = {"Root Guide Page","paclet:guide/Mathematica",Append[Append[DeleteCases[fulllist2,{x_String,y_String/;StringMatchQ[y,("paclet:guide/NewIn60"|"paclet:guide/NewIn70"|"paclet:guide/NewIn80"|"paclet:guide/NewIn90")~~("CoreLanguage"|"MathematicsAndAlgorithms"|"DataHandlingAndDataSources"|"DataManipulation"|"ComputableData"|"SystemInterfacesAndDeployment"|"SystemsInterfacesAndDeployment"|"DynamicInteractivityAndComputableData"|"DynamicInteractivity"|"VisualizationAndGraphics"|"NotebooksAndDocuments")],{___}},Infinity],whatsnew],addonsandpkgs]};

		Quiet[subfinaldata2 = ReplaceRepeated[subfinaldata,{stuff___,f_List,Delimiter,patt:({s_String,path_String}|{s_String,path_String,Delimiter}),r___}:>{stuff,Sequence@@ReplaceAll[{f},patt2:Last[Cases[{f},{st_String,path2_String},Infinity]]:>Append[patt2,Delimiter]],patt,r}];];
		subfinaldata2 = ReplaceAll[subfinaldata2, {Delimiter,rest___}:>{rest}];
		DeleteCases[ReplaceAll[subfinaldata2,Except[{_String,_String,Delimiter},{f__,Delimiter}]:>{f}],{},Infinity]
    ];




(*******************************************

	Code to create Virtual Book 
	(ie tutorial navigator) 

********************************************)

treeAdjust[arg : {title_String, link : (_String | None), data___}] := 
 Switch[{data}, {_List ..}, {title, link, treeAdjust /@ {data}}, {}, arg]; 
treeAdjust[arg_List] := Sequence[] /; Union[Flatten@arg] === {None};

getRawCellContents[cells_] := 
 DeleteCases[cells /. StyleBox[cont_, __] :> cont, 
   Cell[_, _, _String?(StringMatchQ[#, 
         "Tentative*" | "Future*"] &), ___] | (BaseStyle -> "Link") | 
    ButtonBox[Except[_String], __], Infinity] /. 
  Cell[cont_, style_, ___] :> 
   Cell[cont /. TextData[s : {_String ..}] :> StringJoin[s] /. 
     TextData -> Identity];

convertButtonToNodes = {b : {(_ButtonBox | _String) ..} :> 
    If[MemberQ[b, _ButtonBox], 
     ButtonBox[
      StringJoin[(If[MatchQ[#1, _ButtonBox], First[#1], #1] &) /@ b], 
      ButtonData -> 
       First[Cases[b, ButtonBox[_, ButtonData -> uri_] :> uri, 1, 
         1]]], b], 
   ButtonBox[title_, ButtonData -> uri_] :> {title, uri}};

(* OLD VERSION

CreateTutorialBrowserData[file_String /; FileType[file] === File] := 
  treeAdjust[
   getRawCellContents[Part[Import[file, "Notebook"], 1, -1]] //. 
      convertButtonToNodes //. {Cell[s_String] :> {s, None}, 
      c_Cell :> First[c]} //. 
    CellGroupData[l_, _] :> 
     If[l === {}, 
      Unevaluated@
       Sequence[], {Sequence @@ 
          If[ListQ[#1], #1, {#1, None}], ##2} & @@ l]];*)

CreateTutorialBrowserData[file_String /; FileType[file] === File, lang_: "English"]:=
 (CreateTutorialBrowserData[{file}, lang]);

(* in: paths is a list of directories where tutorials can be found *)
          
CreateTutorialBrowserData[paths_List, lang_: "English"] := 
 Module[{IncludeCellTagURILinks = True, 
   PossibleStyles = 
    Alternatives["TOCPart", "TOCChapter", "TOCSection", "TOCSubsection", 
     "TOCSubsubsection", "TOCSubsubsubsection", 
     "TOCSubsubsubsubsection","TOCSubsubsubsubsubsection"], currentrecursionLevel = 0, recurse, 
   data, datafull, files}, 
   
   recurse[roottitleanduri_] :=
   Module[{tmplist, cells, data2, 
     nbdata = 
      Quiet[Get[
        First[Select[files, 
          Function[{x}, (And @@ 
          	(! StringFreeQ[x, $PathnameSeparator <> #] & /@ 
          	StringSplit[StringJoin[StringReplace[
             roottitleanduri[[2]], {"tutorial" -> "Tutorials", 
             "paclet:" -> ""}], ".nb"], "/"]))]]]]], list = {}}, 
    data2 = Select[
       ReplaceRepeated[First[nbdata], 
        Cell[CellGroupData[l_, ___], ___] :> {l}], 
       MemberQ[#, "TOCDocumentTitle" | "Title", Infinity] &][[1]];
       
    (* Remove Future/Tentative cells *)
    data2 = ReplaceRepeated[data2, 
       Cell[___, ft : "FutureExample" | "TentativeExample", ___] :> 
        Cell[Null, ft]];
        
    cells = 
     ReplaceRepeated[#, l_List /; Length[l] == 1 :> Sequence @@ l] & /@
       ReplaceRepeated[
       DeleteCases[
        ReplaceRepeated[data2, 
         Cell[expr_, ___, PossibleStyles, ___] :> 
          
          If[FreeQ[expr, ButtonBox[___], Infinity], 
           Transmogrify`ConvertToString[expr], 
           Cases[expr, ButtonBox[___], Infinity]]], 
        (Cell[e___] /; FreeQ[{e}, PossibleStyles, Infinity]) | 
        (ButtonBox[contents___] /; !FreeQ[{contents}, GraphicsBox]), 
        Infinity], (bb : {___, 
           ButtonBox[___], ___}) :> {Transmogrify`ConvertToString[bb],
          ButtonData /. FilterRules[Options /@ bb, ButtonData]}];
          
    (* Handle ref links in Overviews (I'm looking at you, LibraryLink) *)
    cells = ReplaceRepeated[cells,
       {name_String,ButtonData}:>{name,"paclet:ref/"<>name}];
          
    If[MatchQ[cells, List[List[___]]], cells = cells[[1]]];
    cells = DeleteCases[cells, {}, Infinity];
    cells = ReplaceRepeated[cells, 
       {{first : ({_String, _String} ..), {{n1_String, n2_String}},
        last : ({_String, _String} ...)} :> {first, {n1, n2}, last},
       {first : ({_String, _String} ...), {{n1_String, n2_String}},
        last : ({_String, _String} ..)} :> {first, {n1, n2}, last}}];
          
    If[TrueQ[
         Quiet@StringMatchQ[#[[2]], "paclet:*tutorial/*Overview"]] && 
        Or[IncludeCellTagURILinks, !StringMatchQ[#[[2]], "paclet:*tutorial/*#*"]], 
       AppendTo[
        list, {#[[1]], #[[2]], currentrecursionLevel++; 
         tmplist = recurse[#]; currentrecursionLevel--; tmplist}], 
       (* Catch cases like Advanced Topics under Mathematics & Algorithms overview *)
       If[MatchQ[#, {_String, {_String, _String}..}] && 
       	(And@@(Function[subl, 
       		MatchQ[subl, {_String, s_String} /; StringMatchQ[s, "paclet:*tutorial/*Overview"]]] /@ 
       		Rest[#])), 
       	AppendTo[
       		list, {#[[1]], None, 
       		  Function[rec, 
       			(currentrecursionLevel++; 
       			 tmplist=recurse[rec]; 
       			 currentrecursionLevel--; 
       			 {rec[[1]], rec[[2]], tmplist})] /@ Rest[#]}],
       If[Or[IncludeCellTagURILinks, !StringMatchQ[#[[2]], "paclet:*tutorial/*#*"]], 
        AppendTo[list, #]]]] & /@ cells;
    list];
    
  files = 
   Select[Flatten[
     FileNames["*Overview.nb" | "VirtualBookContents.nb", #, 
        Infinity] & /@ paths], StringMatchQ[#, ___ ~~ lang ~~ ___] &];
  
  data = {"Virtual Book", None, 
    recurse[{"Virtual Book", "paclet:tutorial/VirtualBookOverview"}]};  
  
  datafull = 
   ReplaceRepeated[
   	ReplaceRepeated[ReplaceRepeated[
    DeleteCases[data, {}, Infinity], {{first___, (pre : ({_String, _String} | {_String, _String |
              None, _List})), {{f1_String, 
         f2_String}, (rest : (({_String, _String} | {_String, _String | None, _List}) ..))}, last___} :> {first, pre, {f1, f2, {rest}}, 
       last}, {{{f1_String, 
         f2_String}, (rest : (({_String, _String} | {_String, _String | None, _List}) ..))}, last___} :> {{f1, f2, {rest}}, 
       last}, {s_String, l : (_List ..)} :> {s, None, {l}}, {n_String,
        u_String, {n2_String, rest___}} :> {n, u, {{n2, rest}}}}], {{n1_String,n2_String}}:>{n1,n2}],
         {p_String, u : (_String | None), {n1_String, n2_String}} :> {p, u, {{n1, n2}}}];
  
  datafull];

(******************************************************************************)
(** CreateFunctionNavigator **)

CreateFunctionNavigator[inputDir_, outputDir_, language_] := 
 Module[{funcNavDataFiles, funcNavData, tutNavDataFiles, virtualBookData, 
   docsNavigatorObj, docsNavigatorPath}, 
  funcNavDataFiles = 
   FileNames["*.nb", 
    Flatten@{DocumentationBuild`Utils`GetGuidesDirectory[inputDir, 
       language], 
      DocumentationBuild`SystemResourcesFromDocs`GetJLinkEtcGuidePageDirs[inputDir, language]}];
      
  tutNavDataFiles = 
   Flatten@{DocumentationBuild`Utils`GetTutorialsDirectory[inputDir, 
       language], 
      DocumentationBuild`SystemResourcesFromDocs`GetJLinkEtcTutorialPageDirs[inputDir, language]};
      
  Print["Gathering Function Navigator data from:\n" <> 
    StringJoin @@ Riffle[funcNavDataFiles, "\n"]];
  Print["Gathering Function Navigator data."];
  funcNavData = 
   DocumentationBuild`SystemResourcesFromDocs`CreateFunctionBrowserData@funcNavDataFiles;
  
  (*Is data a tree?If not,use modified data*)
  
  If[Not@TreeBrowse`TreeQ@funcNavData, 
   Print["FAILED Creating perfect Function Data tree-- running fixedtree to prune bad parts"];
   (*CreateDirectory[
    FileNameJoin[
     Flatten[{FileNameSplit[outputDir], "Failed"}]]];
   (*debug*)
   Put[funcNavData, 
    FileNameJoin[
     Flatten[{FileNameSplit[outputDir], "Failed",
       "FunctionNavigator.m"}]]];*)
   funcNavData = TreeBrowse`Dump`$fixedTree];
  
  (*Create virtual book data*)
  Print["Gathering Virtual Book data from:\n" <> StringJoin @@ Riffle[tutNavDataFiles, "\n"]];
  Print["Gathering Virtual Book data."];
  (*Import virtual book data file*)
  
  (*virtualBookData = 
   FileNameJoin[
    Flatten[{FileNameSplit[inputDir], "Documentation", language, 
      "System", "Tutorials" , "VirtualBookContents.nb"}]];*)
      
  virtualBookData = 
   If[ListQ@tutNavDataFiles, 
    DocumentationBuild`SystemResourcesFromDocs`CreateTutorialBrowserData[tutNavDataFiles, language],
    DocumentationBuild`SystemResourcesFromDocs`CreateTutorialBrowserData[{tutNavDataFiles}, language]
   ];
   
  (*Is data a tree?If not,use modified data*)
  
  If[Not@TreeBrowse`TreeQ@virtualBookData, 
   Print["FAILED Creating Virtual Data -- Using modified data set"];
   CreateDirectory[
    FileNameJoin[
     Flatten[{FileNameSplit[outputDir], "Failed"}]]];
   (*debug*)
   Put[virtualBookData, 
    FileNameJoin[
     Flatten[{FileNameSplit[outputDir], "Failed" , "VirtualBook.m"}]]];
   virtualBookData = TreeBrowse`Dump`$fixedTree];
  
  Print["Creating docs navigator notebook."];
  docsNavigatorObj = 
   PubsEvaluateWithFE@
    TreeBrowse`CreateDocsNavigator[funcNavData, virtualBookData];
  Print["Head@docsNavigatorObj: " <> 
    ToString[Head@docsNavigatorObj]];
  docsNavigatorPath = 
   FileNameJoin[
    Flatten[{FileNameSplit[outputDir] , "DocumentationNavigator.nb"}]];
  
  If[(Head[docsNavigatorObj] === NotebookObject),
   CreateDirectory[
    FileNameJoin[
     Flatten[{FileNameSplit[outputDir]}]]];
   PubsEvaluateWithFE@
    NotebookSave[docsNavigatorObj, docsNavigatorPath];
   Print[docsNavigatorPath];
   PubsEvaluateWithFE@NotebookClose[docsNavigatorObj];
   (*Check to make sure outfile was created*)
   If[(FileType[docsNavigatorPath] === 
       None) || (FileByteCount[docsNavigatorPath] < 1), 
    Print["FAILED! -- No notebook created!"], 
    Print["Notebook created."]], 
   Print["   Return value not NotebookObject!"]];];


End[]

EndPackage[]

