(* ::Package:: *)

(* :Title: Make.m *)

(* :Authors:
	Andrew Hunt, Shaun McCance, Todd Gayley
	andy@wolfram.com, shaunm@wolfram.com, tgayley@wolfram.com
*)

(* :Package Version: 0.50 *)

(* :Mathematica Version: 7.0 *)
                     
BeginPackage["DocumentationBuild`Make`", {
    "DocumentationBuild`Common`",
    "DocumentationBuild`Utils`",
    "DocumentationBuild`Info`",
    "Transmogrify`"
}]


(******************************************************************************)
(** Usage Statements **)

MakeNotebook::usage = "MakeNotebook[nb] create an in-product notebook\
 from an authoring notebook.";
 

Options[MakeNotebook] = {
  "ReleaseVersion" -> "11",
  "ThresholdVersion" -> Automatic,
  "BuildMode"->"Normal",
  "Language" -> "English",
  (* WebPage *)
  "Web" -> False,
  "CoreInformation"->Automatic,
  (* IncludeHistoryCell *)
  IncludeHistory -> True,
  "FunctionPaclet" -> False,
  "Debug"->"False",
  "outfile"->False,
  "SafeRaster"->False,
  (* V9 EDIT BELOW *)
  "CapitalizeHeadingStyles" -> {
    (*"AllowableOptionsSection",*) "ApplicationNotesSection", 
    "AttributesSection", "CategorizationSection", "TestCasesSection",
    "DesignDiscussionSection", "DocumentStatusSection", 
    (*"ElementsSection",*) "ErrorMessagesSection", (*"ExampleSection",*) 
    "ExtendedExamplesSection", "ExtensionsSection", 
    (*"GuideFunctionsSection",*) "GuideMoreAboutSection", 
    "GuideTutorialsSection", (*"ImportExportSection",*) "KeywordsSection", 
    "LegacyMaterialSection", (*"MoreAboutSection",*) 
    "MoreToExploreSection", "MoreInformationSection",
    (*"NotebookInterfaceSection",*) (*"NotesSection",*) (*"OptionsSection", *)
    (*"PrimaryExamplesSection",*) (*"RelatedDemonstrationsSection", *)
    (*"RelatedLinksSection",*) (*"Section",*) (*"SeeAlsoSection",*) 
    "TemplatesSection", (*"TOCSection",*) (*"TutorialsSection",*) 
    (*"RelatedTutorialsSection",*) (*"TutorialMoreAboutSection",*) (*"TutorialCollectionSection",*)
    (*"TutorialRelatedLinksSection",*) (*"TutorialsTutorialCollectionSection",*)
    "UsageMessagesSection", "UsageSection"(*, "IndicatorExampleSection", "IndicatorDescriptionSection", "IndicatorCategorizationSection", "IndicatorAbbreviationSection", "IndicatorFormulaSection"*)
  }
};

$MarketingPageStyleSheet = Null;
$ComparisonPageStyleSheet = Null;
$CoreStyleSheet = Null;
$NewIn6FunctionList = Get[ToFileName[{DocumentationBuild`Common`$DocumentationBuildDirectory, "Internal", "web", "data"}, "NewIn6FunctionList.m"]];


MakeMarketingPageNotebook::usage = "MakeMarketingPageNotebook[nb] takes a production notebook and returns a massaged notebook ready for inclusion in product/on web.";
ExportMarketingWebPage::usage = "ExportMarketingWebPage[file.html, nb, language] takes a production notebook and exports to xml.";
ExportMarketingCategoryWebPages::usage = "";
ExportCapabilitiesWebPage::usage ="";
ExportComparisonWebPage::usage="";
MakeWhatsNewSidebar::usage="";
ExportWhatsNewSidebar::usage="";
MakeMarketingSubpage::usage = "";
MakeMarketingThumbnailTable::usage = "";
MakeSolutionsPageNotebook::usage = "MakeSolutionsPageNotebook[nb] takes a production notebook and returns a massaged notebook ready for Web export.";
ExportSolutionsWebPage::usage =  "ExportSolutionsWebPage[file.html, nb, language] exports Solutions page for Web from output of MakeSolutionsPageNotebook.";
MakeUpdatedSolutionsNotebook::usage = "MakeUpdatedSolutionsNotebook[nb] takes a production notebook and returns a massaged notebook ready for Web export.";
ExportUpdatedSolutionsPage::usage =  "ExportUpdatedSolutionsPage[file.html, nb, language] exports Solutions page for Web from output of MakeUpdatedSolutionsNotebook.";
MakeTMJNotebook::usage= "MakeTMJNotebook[nb] takes a production notebook and returns a massaged notebook ready for Web export.";
ExportTMJWebPage::usage =  "ExportTMJWebPage[file.html, nb, language] exports TMJ expression for Web from output of MakeTMJNotebook.";
MakeInsiderNotebook::usage = "MakeInsiderNotebook[nb] takes a production notebook and returns a massaged notebook ready for Web export.";
ExportInsiderWebPage::usage =  "ExportInsiderWebPage[file.html, nb, language] exports newsletter html or email from output of MakeInsiderNotebook.";
MakeAlphaAPINotebook::usage= "MakeAlphaAPINotebook[nb] takes a production notebook and returns a massaged notebook ready for Web export.";
ExportAlphaAPIWebPage::usage =  "ExportAlphaAPIWebPage[file.html, nb, language] creates html output from MakeAlphaAPINotebook.";


Begin["`Private`"];


(******************************************************************************)
(** MakeNotebook **)

MakeNotebook[nb_String, opts___?OptionQ] :=
  MakeNotebook[GetQuietly[nb], "FilePath" -> nb, opts];

MakeNotebook[nb_Notebook, opts___?((OptionQ[#] && !ListQ[#])&)] :=
  MakeNotebook[nb, GetNotebookInformation[nb, opts], opts];

MakeNotebook[nb_Notebook, info_List, opts___?OptionQ] :=
  Module[{pacletName, type, lang, uri, expr, poss, webpage, functionPaclet, rootCaptionList,
          stylesheet, styledefs,  indicatorstyles, versionYears, historyCell, overviewLinks, footerCell, classLabel,
          includeHistoryCell, nbContext, outfilelocation, buildmode, moreAboutCells, moreAboutCellPos, newMoreAboutCells,
          specialKeywords, guidetutorialcollectioncells, tutorialcollectiontitleandlinks, linktrailexprs, colortype,
          debug, time1,time2,time2a,time3,time4,time5,time6,time7,time8,time9,time10,time11,time12,time13,time14,time15,time16,time17,makenboutput, autoitalics,
          examplepagedata, hasoptions, warnicon, $Rebranded, saferaster, summary, fnessay, GetTableTaggingRules, tabletaggingrules},

    Off[CreateDirectory::filex];

    (* StrokeForm (and others) not loaded, so load it hack *)
    System`StrokeForm;
    System`Absolute;

    DocumentationBuild`URIsWithoutCloudSupport = {"ref/format/CDF", "howto/CreateAComputableDocumentFormatFile", "ref/CDFExport", 
  "ref/CDFDeploy", "ref/CDFInformation", "ref/DeviceOpen", "ref/DeviceObject", "ref/DeviceConfigure", "ref/DeviceClose", 
  "ref/Devices", "ref/FindDevices", "ref/DeviceRead", "ref/DeviceReadList", "ref/DeviceReadTimeSeries", 
  "ref/DeviceReadLatest", "ref/DeviceReadBuffer", "ref/DeviceWrite", "ref/DeviceWriteBuffer", "ref/DeviceExecute", 
  "ref/DeviceExecuteAsynchronous", "ref/DeviceStreams", "ref/ImageCapture", "ref/$DefaultImagingDevice",
  "ref/$ImagingDevices", "ref/$ImagingDevice", "ref/device/*", "ref/menuitem/*", 
  "ref/Beep", "ref/Speak", 
  "ref/SpokenString", "ref/SystemDialogInput", "ref/Locator", "ref/Texture", "ref/Annotation", 
  "ref/EventHandler", "ref/Hyperlink", "ref/PopupWindow", "ref/StatusArea", 
  "ref/AxesStyle", "ref/BaselinePosition", "ref/ContentSelectable", "ref/DisplayFunction", "ref/FormatType", 
  "ref/FrameStyle", "ref/FrameTicksStyle", "ref/GridLinesStyle", "ref/ImagePadding", "ref/LabelStyle", "ref/PreserveImageOptions", 
  "ref/RotateLabel", "ref/TicksStyle", "ref/InteractiveTradingChart",(*  Graphics3D next  *)"ref/Arrow", 
  "ref/BezierCurve", "ref/BSplineSurface", "ref/AbsoluteDashing", 
  "ref/AbsolutePointSize", "ref/AbsoluteThickness", "ref/Arrowheads", "ref/CapForm", "ref/Dashing", "ref/JoinForm", "ref/PointSize", 
  "ref/Texture", "ref/Thickness", "ref/Annotation", "ref/EventHandler", "ref/Hyperlink", "ref/PopupWindow", 
  "ref/StatusArea", "ref/Run", "ref/RunThrough", "ref/c/*", "ref/SystemInformation", "CCompilerDriver/*", "CCodeGenerator/*", 
  "ref/CompilationTarget", "CUDALink/*", "OpenCLLink/*", "guide/MenuItems", "guide/GPUComputing", "guide/WSTPAPI", "guide/WSTPCLanguageFunctions"};
  
    DocumentationBuild`URIsWithoutPublicCloudSupport = {"DatabaseLink/*", "ParallelTools/*",
  "ref/Parallelize", "ref/ParallelTry", "ref/ParallelEvaluate", "ref/DistributeDefinitions", 
  "ref/ParallelNeeds", "ref/ParallelMap", "ref/ParallelTable", "ref/ParallelSum", "ref/ParallelCombine", 
  "ref/ParallelSubmit", "ref/WaitAll", "ref/WaitNext", "ref/SetSharedVariable", "ref/SetSharedFunction", 
  "ref/$SharedVariables", "ref/$SharedFunctions", "ref/UnsetShared", "ref/CriticalSection", "ref/LaunchKernels",
  "ref/$KernelCount", "ref/$KernelID", "ref/Kernels", "ref/AbortKernels", "ref/CloseKernels", "ref/$ProcessorCount",
  "ref/Parallelization", "ref/$ConfiguredKernels", "ref/ParallelDo", "ref/ParallelProduct", "ref/ParallelArray",
  "ref/EvaluationObject", "guide/ParallelComputing"};
  
    DocumentationBuild`URIsWithoutDesktopSupport = 
    	{"ref/CloudBase", "ref/CloudConnect", "ref/CloudDeploy", 
    		"ref/CloudDirectory", "ref/CloudDisconnect", "ref/CloudEvaluate", "ref/CloudExport", 
    		"ref/CloudFunction", "ref/CloudGet", 
    		"ref/CloudImport", "ref/CloudObject", "ref/CloudObjectInformation", 
    		"ref/CloudObjectInformationData", "ref/CloudObjects", "ref/CloudPut", 
    		"ref/CloudSave", "ref/CloudSymbol", "ref/SetCloudDirectory", "ref/CloudPublish",
    		"ref/$CloudBase", "ref/$CloudConnected", "ref/$CloudEvaluation", 
    		"ref/$CloudRootDirectory", "ref/$CloudSymbolBase", "ref/APIFunction", "ref/FormFunction",
    		"ref/ExternalBundle", "ref/ExportForm","ref/ResponseForm",
    		"ref/Delayed", "ref/HTTPResponse", "ref/HTTPRedirect",
    		"ref/Permissions", "ref/$Permissions", "ref/CloudAccountData", "ref/$EvaluationEnvironment",
    		"ref/HTTPRequestData", "ref/$RequesterAddress", "ref/EmbeddedHTML", "ref/EmbeddedService",
    		"ref/CloudExpression", "ref/CreateCloudExpression", "ref/CloudExpressions",
    		"ref/CloudExpressionObject", "ref/DeleteCloudExpression"};

    $Rebranded = True;
    saferaster = "SafeRaster" /. {opts} /. Options[MakeNotebook];

    pacletName = "PacletName" /. info;
    pacletName = If[pacletName === "Mathematica", "", pacletName];
    type = "EntityType" /. info;
    lang = "Language" /. {opts} /. Options[MakeNotebook];
    uri = "URI" /. info;
    webpage = "Web" /. {opts} /. Options[MakeNotebook];
    buildmode = "BuildMode" /. {opts} /. Options[MakeNotebook];
    outfilelocation = "outfile" /. {opts} /. Options[MakeNotebook];
    functionPaclet = "FunctionPaclet" /.  {opts} /. Options[MakeNotebook];
    debug = "Debug" /.  {opts} /. Options[MakeNotebook];
    
    debug = "True";
    
    Global`AntLog["Starting time1"];time1 = ToString@AbsoluteTiming[
    DocumentationBuild`Export`$CoreInformation[lang_] := DocumentationBuild`Export`$CoreInformation[lang] = LoadCoreInformation["CoreInformation" /.  {opts} /. Options[MakeNotebook], lang];
    ][[1]];
    If[debug==="True",Global`AntLog["Loading CoreInformation: "<>time1]];
	(* check for explicit CellContext setting - present in some/most M- Tutorials *)
	nbContext = (CellContext /. Rest[List @@ nb]);
	specialKeywords = Cases[nb, Cell[a_,"Keywords",Rule[CellLabel,b_]]:>a, Infinity];

    BuildLog["** Entering MakeNotebook **"];
    BuildLog["   Language: "<>lang ];

    expr = nb;
    
    GetRootCaptionList[nbexpression_]:= Module[{SectionCaptions},
    SectionCaptions = Cases[nbexpression, Cell[__, "GuideText", ___], Infinity]; 
    SectionCaptions = DeleteCases[SectionCaptions, Cell[__, "GuideText","TOCExcludedObject", ___], Infinity];
    SectionCaptions = Cases[SectionCaptions, Cell[a__, "GuideText", ___]:>a, Infinity];
    SectionCaptions
    ];
    
    rootCaptionList=If[type==="Root Guide",
    	GetRootCaptionList[expr],
    	""];
    

indicatorstyles={
Cell[StyleData["IndicatorExampleSection"],
 Editable->False,
 CellFrame->{{0, 0}, {0, 2}},
 ShowCellBracket->Automatic,
 ShowGroupOpener->True,
 CellMargins->{{24, 24}, {6, 18}},
 CellElementSpacings->{"ClosedGroupTopMargin"->28},
 CellGroupingRules->{"SectionGrouping", 50},
 CellFrameMargins->4,
 CellFrameColor->RGBColor[
  0.48235300000000003`, 0.5176470000000001, 0.8196080000000001],
 StyleMenuListing->None,
 FontFamily->"Verdana",
 FontSize->9,
 FontTracking->"Wide",
 FontColor->GrayLevel[0]],

Cell[StyleData["IndicatorExample"],
 CellMargins->{{52, 12}, {1, 1}},
 ReturnCreatesNewCell->True,
 CellChangeTimes->{3.475262460013097*^9},
 FontFamily->"Verdana",
 FontSize->11],

Cell[StyleData["IndicatorDescriptionSection"],
 Editable->False,
 CellFrame->{{0, 0}, {0, 2}},
 ShowCellBracket->Automatic,
 ShowGroupOpener->True,
 CellMargins->{{24, 24}, {6, 18}},
 CellElementSpacings->{"ClosedGroupTopMargin"->28},
 CellGroupingRules->{"SectionGrouping", 50},
 CellFrameMargins->4,
 CellFrameColor->RGBColor[
  0.48235300000000003`, 0.5176470000000001, 0.8196080000000001],
 StyleMenuListing->None,
 FontFamily->"Verdana",
 FontSize->9,
 FontTracking->"Wide",
 FontColor->GrayLevel[0]],

Cell[StyleData["IndicatorDescription"],
 CellMargins->{{52, 12}, {2, 2}},
 ReturnCreatesNewCell->True,
 CellChangeTimes->{3.475262460013297*^9},
 FontFamily->"Verdana",
 FontSize->11],

Cell[StyleData["IndicatorCategorizationSection"],
Editable->False,
 CellFrame->{{0, 0}, {0, 2}},
 ShowCellBracket->Automatic,
 ShowGroupOpener->True,
 CellMargins->{{24, 24}, {6, 18}},
 CellElementSpacings->{"ClosedGroupTopMargin"->28},
 CellGroupingRules->{"SectionGrouping", 50},
 CellFrameMargins->4,
 CellFrameColor->RGBColor[
  0.48235300000000003`, 0.5176470000000001, 0.8196080000000001],
 StyleMenuListing->None,
 FontFamily->"Verdana",
 FontSize->9,
 FontTracking->"Wide",
 FontColor->GrayLevel[0]],

Cell[StyleData["IndicatorCategorization"],
 CellMargins->{{52, 12}, {2, 2}},
 ReturnCreatesNewCell->True,
 CellChangeTimes->{3.475262460013495*^9},
 FontFamily->"Verdana",
 FontSize->11],

Cell[StyleData["IndicatorAbbreviationSection"],
 Editable->False,
 CellFrame->{{0, 0}, {0, 2}},
 ShowCellBracket->Automatic,
 ShowGroupOpener->True,
 CellMargins->{{24, 24}, {6, 18}},
 CellElementSpacings->{"ClosedGroupTopMargin"->28},
 CellGroupingRules->{"SectionGrouping", 50},
 CellFrameMargins->4,
 CellFrameColor->RGBColor[
  0.48235300000000003`, 0.5176470000000001, 0.8196080000000001],
 StyleMenuListing->None,
 FontFamily->"Verdana",
 FontSize->9,
 FontTracking->"Wide",
 FontColor->GrayLevel[0]],

Cell[StyleData["IndicatorAbbreviation"],
 CellMargins->{{52, 12}, {2, 2}},
 ReturnCreatesNewCell->True,
 CellChangeTimes->{3.475262460013693*^9},
 FontFamily->"Verdana",
 FontSize->11],

Cell[StyleData["IndicatorFormulaSection"],
Editable->False,
 CellFrame->{{0, 0}, {0, 2}},
 ShowCellBracket->Automatic,
 ShowGroupOpener->True,
 CellMargins->{{24, 24}, {6, 18}},
 CellElementSpacings->{"ClosedGroupTopMargin"->28},
 CellGroupingRules->{"SectionGrouping", 50},
 CellFrameMargins->4,
 CellFrameColor->RGBColor[
  0.48235300000000003`, 0.5176470000000001, 0.8196080000000001],
 StyleMenuListing->None,
 FontFamily->"Verdana",
 FontSize->9,
 FontTracking->"Wide",
 FontColor->GrayLevel[0]],

Cell[StyleData["IndicatorFormula"],
 CellMargins->{{52, 12}, {2, 2}},
 ReturnCreatesNewCell->True,
 CellChangeTimes->{3.475262460013907*^9},
 FontFamily->"Verdana",
 FontSize->11],
 
 Cell[StyleData["IndicatorUsage"],
 Editable->False,
 CellFrame->{{0, 0}, {0.5, 0.5}},
 ShowCellBracket->Automatic,
 CellMargins->{{24, 8}, {6, 6}},
 CellFrameColor->RGBColor[0.7490200000000001, 0.694118, 0.5529410000000001],
 Hyphenation->False,
 ParagraphSpacing->{0.5, 0},
 StyleMenuListing->None,
 FontFamily->"Verdana",
 FontSize->11,
 GridBoxOptions->{BaseStyle->{FontFamily -> "Verdana"},
 GridBoxAlignment->{
  "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}}, 
   "RowsIndexed" -> {}},
 GridBoxItemSize->{"Columns" -> {
     Scaled[0.003], {
      Scaled[0.996]}}, "ColumnsIndexed" -> {}, "Rows" -> {{1.}}, 
   "RowsIndexed" -> {}}}]
};

    (* This is inserted into the notebook at the end. *)
    stylesheet = Switch[lang,
      "Japanese", FrontEnd`FileName[{"Wolfram"}, "Reference-ja.nb"],
      "ChineseSimplified", FrontEnd`FileName[{"Wolfram"}, "Reference-zh.nb"],
      _, FrontEnd`FileName[{"Wolfram"}, "Reference.nb"] ];

    styledefs = {};
    If[type === "Program",
      styledefs = Join[styledefs, {
        Cell[StyleData["InlineFormula"], FontSize->
          FEPrivate`Switch[FEPrivate`$OperatingSystem, 
            "MacOSX", 14,
            "Windows", Inherited*1.025,
            "Unix", 13,
            _, Inherited]],
        Cell[StyleData["ProgramCodeText"], FontSize->
          FEPrivate`Switch[FEPrivate`$OperatingSystem, 
            "MacOSX", 14,
            "Windows", 14,
            "Unix", 13,
            _, Inherited]]}]];
    If[type === "Tutorial",
      styledefs = Join[styledefs, {    
        Cell[StyleData["Input"], CellContext -> Notebook],
        Cell[StyleData["Output"], CellContext -> Notebook],
        Cell[StyleData["TutorialMoreAboutSection"], CellGroupingRules -> {"SectionGrouping", 30}],
        Cell[StyleData["RelatedTutorialsSection"], CellGroupingRules -> {"SectionGrouping", 30}],
        Cell[StyleData["TutorialRelatedLinksSection"], CellGroupingRules -> {"SectionGrouping", 30}]
    }] ];
     If[type === "Overview",
      styledefs = Join[styledefs, {    
        Cell[StyleData["Input"], CellContext -> Notebook],
        Cell[StyleData["Output"], CellContext -> Notebook],
        Cell[StyleData["TOCSubsubsection"], CellGroupingRules -> {"SectionGrouping", 69}],
        Cell[StyleData["TutorialMoreAboutSection"], CellGroupingRules -> {"SectionGrouping", 30}],
        Cell[StyleData["RelatedTutorialsSection"], CellGroupingRules -> {"SectionGrouping", 30}],
        Cell[StyleData["TutorialRelatedLinksSection"], CellGroupingRules -> {"SectionGrouping", 30}]
    }] ];
(*    If[type === "HowTo",
      styledefs = Join[styledefs, {    
        Cell[StyleData["Input"], CellContext -> Notebook],
        Cell[StyleData["Output"], CellContext -> Notebook]
    }] ]; *)
        (*If[type === "Indicator",
      styledefs = Join[styledefs, indicatorstyles] ]; *)
    If[nbContext === "Global`",
      styledefs = Join[styledefs, {    
        Cell[StyleData["Input"], CellContext -> "Global`"],
        Cell[StyleData["Output"], CellContext -> "Global`"]
    }] ];
If[nbContext === "Notebook",
      styledefs = Join[styledefs, {    
        Cell[StyleData["Input"], CellContext -> Notebook],
        Cell[StyleData["Output"], CellContext -> Notebook]
    }] ];
If[nbContext === Notebook,
      styledefs = Join[styledefs, {    
        Cell[StyleData["Input"], CellContext -> Notebook],
        Cell[StyleData["Output"], CellContext -> Notebook]
    }] ];
    If[TrueQ[functionPaclet],
      styledefs = Join[styledefs, {
        Cell[StyleData["AnchorBarActionMenu"],
          Background -> RGBColor[0.968627, 0.960784, 0.964706],
          FontColor -> RGBColor[0.0352941, 0.596078, 1.]
        ],
        Cell[StyleData["Usage"],
          Background -> GrayLevel[0.972549],
          GridBoxOptions -> {
            GridBoxDividers -> {"RowsIndexed" -> {
              1 -> Directive[GrayLevel[0.576471], Thickness[3]],
              -1 -> Thickness[0]
          }}}
        ]
    }]];
    If[Length[styledefs] > 0,
      stylesheet = Notebook[Join[
        {Cell[StyleData[StyleDefinitions -> stylesheet]]},
        styledefs
    ]] ];
    BuildLog["   Stylesheet: "<>ToString[stylesheet]];

    (* These are inserted into the notebook at the end. *)
(* FIXME *)
Global`AntLog["Starting time2"];time2 = ToString@AbsoluteTiming[
   	overviewLinks = DocumentationBuild`Utils`GetOverviewButtonLinks["paclet:"<>uri, lang];
][[1]];
If[debug==="True",Global`AntLog["Running GetOverviewButtonLinks: "<>time2]];
    (* FIXME: should this be in info? *)
time2a = ToString@AbsoluteTiming[
    classLabel = Transmogrify`ConvertToString[GetClassificationBoxes[info, lang, "Rebranded"->$Rebranded]];
    ][[1]];
    
    If[debug==="True",Global`AntLog["Running ConvertToString: "<>time2a]];
    
    classLabel = If[functionPaclet, "Mathematica Function Paclet Symbol", classLabel];

    (* List of version numbers -> years list for history cell: {ver->year, ...} *)
    versionYears = {"1"->"1988", "1.2"->"1989", "2"->"1991", "2.1"->"1992", "2.2"->"1993", "3"->"1996", "4"->"1999", 
    	"4.1"->"2000", "4.2"->"2002", "5"->"2003", "5.1"->"2004", "5.2"->"2005", "6"->"2007", 
    	"6.0.1"->"2007", "6.0.2"->"2008", "6.0.3"->"2008", "7"->"2008", "7.0.1"->"2009", "8"->"2010", 
    	"8.0.1"->"2011", "8.0.4"->"2011", "9"->"2012", "9.0.1"->"2013", "10"->"2014", "10.0.1"->"2014",
    	"10.0.2"->"2014", "10.1"->"2015", "10.1.1"->"2015", "10.2"->"2015", "10.3"->"2015",
    	"10.4"->"2016", "11"->"2016"};
    
    (* Store the history cell for later insertion *)
    Module[{newIn, modIn, newVers, modVers, inc},
      inc = IncludeHistory /. {opts} /. Options[MakeNotebook];
      {newIn, modIn} = {"New", "Modified"} /. ("HistoryData" /. info);
      newIn = If[(StringQ[newIn] && !StringMatchQ[newIn, RegularExpression["\\s+|\\?+|XX+"]]), newIn, ""];
      (* Handle future/UnderDev pages *)
      If[!StringMatchQ[newIn, (DigitCharacter | ".")..] && TrueQ["Underdevelopment" /. info], 
      	newIn = StringReplace[ToString[$VersionNumber], "."~~EndOfString -> ""]]; 
      modIn = If[(StringQ[modIn] && !StringMatchQ[modIn, RegularExpression["\\s+|\\?+|XX+"]]), modIn, Null];
      modIn = If[!StringFreeQ[modIn, ","], Last[StringSplit[modIn, ","]], modIn];
      newVers = If[StringFreeQ[newIn, "."], newIn <> ".0", newIn];
      modVers = If[StringFreeQ[modIn, "."], modIn <> ".0", modIn];
      (* Substitute in years for version numbers *)
      newIn = ReplaceAll[StringReplace[newIn, ".0"~~EndOfString -> ""], versionYears];
      modIn = ReplaceAll[StringReplace[modIn, ".0"~~EndOfString -> ""], versionYears];
      If[!MatchQ[newIn, $Failed | Null | ""], 
         If[! MatchQ[modIn, $Failed | Null | ""], 
              historyCell = Cell[TextData[{Localized["HistoryNewIn", lang, newIn], Cell[" ("<> ToString[newVers] <>")", "HistoryVersion"], " | ", Localized["HistoryModIn", lang, modIn], Cell[" ("<> ToString[modVers] <>")", "HistoryVersion"]}], "History"], 
              historyCell = Cell[TextData[{Localized["HistoryNewIn", lang, newIn], Cell[" ("<> ToString[newVers] <>")", "HistoryVersion"]}], "History"]], 
         historyCell = {}
         ];
      
      historyCell = If[(inc === True) && (("PacletName" /. info) === "Mathematica") && (type === "Symbol" || type ==="Format" || type === "Character Name"),
        {historyCell},
        {}];
      BuildLog["-- includeHistoryCell -> "<>ToString[includeHistoryCell] ];
    ];

    footerCell = Cell[" ", "FooterCell"];
 
    (*expr = DeleteCases[expr, (Cell[CellGroupData[{#, ___}, ___], ___] | #)&[
        Cell[__,
          "GuideTutorialCollection" | "GuideTutorialCollectionSection" | "TutorialCollectionSection" | "TutorialCollection",
        ___]
      ], Infinity];*)
 
    (* change MoreAbout stub "Autogenerated" with XXX so it's removed natuarlly *)
    expr = expr /. Cell["Autogenerated", "MoreAbout", ___] :> Cell["XXXX", "MoreAbout"];

    (* Convert *newer* style of ModInfo first *)
    expr = expr /. Cell[BoxData[TooltipBox[Cell[v_, "ModInfo", vv___], tt_String]], "ModInfoWrapper"] :>
          Cell[tt, "ModInfo", vv];

    (* Remove new style of ModInfo *)   
    expr = expr /. Cell[BoxData[TooltipBox[Cell[v_, "ModInfo", vv___], ___], ___], ___] :> Cell[v, "ModInfo", vv];
    expr = expr /. TooltipBox[Cell[v_, "ModInfo", vv___], _] :> Cell[v, "ModInfo", vv];

    BuildLog["-- Delete table rows with XXXX"];
    poss = Position[expr,
        Cell[BoxData[GridBox[___]],
          ("1ColumnTableMod" | "2ColumnTableMod" | "3ColumnTableMod" | "3ColumnIllustratedTableMod" | 
           "DefinitionBox"), ___],
        Infinity];
    Scan[Function[{cellpos}, Module[{gridpos, grid},
        gridpos= Join[cellpos, {1, 1}];
        grid = Extract[expr, gridpos];
        If[buildmode==="Future",
        expr,
        expr = Delete[expr,
          Map[Join[gridpos, {1}, #]&,
            Position[First[grid],
              {___, 
              (s_String /; StringMatchQ[s, 
              	RegularExpression["\\s*XX+"] | RegularExpression["\\s*xxx+"]]) | 
              (Cell[s_String /; StringMatchQ[s, RegularExpression["\\s*XX+"] | 
              	RegularExpression["\\s*xxx+"]], ___]), ___}
              ]
        ]]];
      ]],
      poss];

    (* Get tagged tables' TaggingRules (autocomplete data) before they're stripped out in the next block of code *)
    GetTableTaggingRules[expr_] :=
      Cases[expr, _["TemplateArgumentValues", c_] :> c, Infinity];
    If[MatchQ[type, "Symbol"], tabletaggingrules = GetTableTaggingRules[expr]];


    BuildLog["-- Delete various cells"];
    If[buildmode==="Future",
      expr = DeleteCases[expr, Alternatives[
      (* Flags and other things we use for internal purposes *)
      (Cell[CellGroupData[{#, ___}, ___], ___] | #)&[
        Cell[__,
          "FutureFlag" | "ExcisedFlag" | "InternalFlag" | "ObsoleteFlag" |
          "TemporaryFlag" | "AwaitingFutureDesignReviewFlag" | "PreviewFlag" | 
          "AllowableOptions" | "AllowableOptionsSection" | "ApplicationNotesSection" | 
          "Attributes" | "AttributesSection" | "ExampleStatusSection" | 
          "Categorization" | "CategorizationSection" | "DesignDiscussionSection" | 
          "DocumentStatus" | "DocumentStatusSection" | "Details" | "DetailsSection" | 
          "ErrorMessages" | "ErrorMessagesSection" | "ExtensionsSection" | 
          "History" | "Keywords" | "KeywordsSection" | "TestCasesSection" |
          "LegacyMaterialSection" | "Synonyms" | "SynonymsSection" | 
          "Template" | "TemplatesSection" | 
          "UsageMessages" | "UsageMessagesSection" |
          "WebResources" | "WebResourcesSection" | "AuthorDate" | "SecuritySection" | "SecurityDetails",
        ___]
      ],
      (* Cell groups that don't contain much of anything *)
      Cell[CellGroupData[{_, Cell["" | " ", ___]}, ___], ___],
      (* Cell labels used for modification version *)
      _[CellLabel, s_String] /; StringMatchQ[s, RegularExpression["^\\d[+~Xx]"] ],
      (* All CellChangeTimes and TaggingRules *)
      _[System`CellChangeTimes | TaggingRules, _]
    ], Infinity],
    expr = DeleteCases[expr, Alternatives[
      (* Flags and other things we use for internal purposes *)
      (Cell[CellGroupData[{#, ___}, ___], ___] | #)&[
        Cell[__,
          "FutureFlag" | "ExcisedFlag" | "InternalFlag" | "ObsoleteFlag" |
          "TemporaryFlag" | "AwaitingFutureDesignReviewFlag" | "PreviewFlag" | 
          "AllowableOptions" | "AllowableOptionsSection" | "ApplicationNotesSection" | 
          "Attributes" | "AttributesSection" | "ExampleStatusSection" | 
          "Categorization" | "CategorizationSection" | "DesignDiscussionSection" | 
          "DocumentStatus" | "DocumentStatusSection" | "Details" | "DetailsSection" | 
          "ErrorMessages" | "ErrorMessagesSection" | "ExtensionsSection" | 
          "History" | "Keywords" | "KeywordsSection" | "TestCasesSection" |
          "LegacyMaterialSection" | "Synonyms" | "SynonymsSection" | 
          "Template" | "TemplatesSection" | 
          "UsageMessages" | "UsageMessagesSection" |
          "WebResources" | "WebResourcesSection" | "AuthorDate" | "SecuritySection" | "SecurityDetails",
        ___]
      ],
      (* Cell groups that don't contain much of anything *)
      Cell[CellGroupData[{_, Cell["" | " ", ___]}, ___], ___],
      (* Cell labels used for modification version *)
      _[CellLabel, s_String] /; StringMatchQ[s, RegularExpression["^\\d[+~Xx]"] ],
      (* All CellChangeTimes and TaggingRules *)
      _[System`CellChangeTimes | TaggingRules, _],
      (* FutureExample cells that aren't ModInfo cells *)
      Cell[pre___, "FutureExample" | "TentativeExample", post___, _[CellID, _], ___] /; FreeQ[{pre, post}, "ModInfo"],
      (* XX... cells, just dummy placeholders *)
      Cell[c_String /; StringMatchQ[c, RegularExpression["\\s*XX+"]], ___]
    ], Infinity]
    ];

    BuildLog["-- Remove Future tagged table "];
    If[buildmode==="Future",
    expr,
    poss = Position[expr, _GridBox, Infinity];
    expr = ReplacePart[expr, Flatten[Map[
      Function[{pos}, Module[{grid},
        grid = Extract[expr, pos];
        If[MatchQ[First[grid],
            {___, {___, (Cell|StyleBox)[___, "FutureExample" | "TentativeExample", ___], ___}, ___}],
          (pos -> ReplacePart[grid, {1} ->
            DeleteCases[First[grid],
              {___, (Cell|StyleBox)[___, "FutureExample" | "TentativeExample", ___], ___} ]]),
          {}]
      ]],
      poss ]]]];

    (* Remove ConversionSpecial and corresponding CellTag *)
    expr = expr /. {
      Cell[c__, "ConversionSpecial", o1___,
          CellTags -> t_List /; {a___, "InProductOnly", b___}, o2___]
        :> Cell[c, o1, CellTags->{a,b}, o2],
      Cell[c__, "ConversionSpecial", o1___,
          CellTags -> t_String /; StringMatchQ[t, "InProductOnly"], o2___]
        :> Cell[c, o1, o2]
    };

    expr = expr /. {
      Cell[c__, "ConversionSpecial", o1___,
          CellTags -> t_List /; {a___, "OnWebOnly", b___}, o2___]
        :> Cell[c, o1, CellTags->{a,b}, o2,  ShowCellBracket->False,
 CellElementSpacings->{"ClosedCellHeight"->0},
 CellOpen->False],
      Cell[c__, "ConversionSpecial", o1___,
          CellTags -> t_String /; StringMatchQ[t, "OnWebOnly"], o2___]
        :> Cell[c, o1, o2, ShowCellBracket->False,
 CellElementSpacings->{"ClosedCellHeight"->0},
 CellOpen->False]
    };

    (* Remove XX+ tables and preceding notes cells *)
    If[buildmode==="Future",
    expr,
    poss = Position[expr,
        Cell[BoxData[GridBox[{
            {_, s_String /; StringMatchQ[s, RegularExpression["\\s*XX+"]], ___}..
          }, ___]],
          ("2ColumnTableMod" | "3ColumnTableMod" | "3ColumnIllustratedTableMod"), ___],
        Infinity];
    expr = Delete[expr,
        Join@@Map[
          If[MatchQ[Extract[expr, Append[Most[#], Last[#]-1]],
                Cell[_, "Notes", ___]],
            {Append[Most[#], Last[#]-1], #},
            {#}]&,
          poss]]
    ];

Global`AntLog["Starting time3"];time3 = ToString@AbsoluteTiming[
    expr = AdjustModInfo[expr, info, opts] ;
    ][[1]];
    If[debug==="True",Global`AntLog["Running AdjustModInfo: "<>time3]];
    (* More About section needs to be populated before anchor bar is created. Only rewrite for Symbols. *) 
Global`AntLog["Starting time4"];time4 = ToString@AbsoluteTiming[
    expr = EnsureCellGroupData[expr, "MoreAboutSection", lang];
    ][[1]];
    If[debug==="True",Global`AntLog["Running EnsureCellGroupData: "<>time4]];
  poss = Position[expr,
    CellGroupData[{Cell[_, "MoreAboutSection", ___], ___}, ___],
    Infinity, 1];
  BuildLog["   Position of section(s):"<>ToString@moreAboutPos];
Global`AntLog["Starting time5"];time5 = ToString@AbsoluteTiming[
  If[type==="Symbol", 
  If[Length[poss] > 0,
    poss = First@poss;
    Module[{maCell, maNew, maManual},
      (* maManual allows authors to put in More Abouts that aren't autogenerated *)
      maManual = Cases[expr, (Cell[c_, ___, "MoreAbout", ___] /; !MatchQ[c, "XXXX"|"Autogenerated"]) :> 
      	{Transmogrify`ConvertToString[c], First[Cases[c, _[ButtonData,ur_String] :> ur, Infinity]]}, Infinity];
      maCell = Extract[expr, Join[poss, {1, 1}]];
      maNew = DocumentationBuild`Info`GetSourceInformation["paclet:" <> uri, lang, "Links", "Guides"];
      If[maNew == None, maNew = {}];
      maNew = DeleteDuplicates[Join[maNew, Select[maManual, FreeQ[Last/@maNew, #[[2]]]&]]];
      If[buildmode==="Future",
      	maNew,
      	maNew = Select[maNew,
        FreeQ[{"ExcisedFlag", "FutureFlag", "InternalFlag", "TemporaryFlag"},
          DocumentationBuild`Info`GetSourceInformation[#[[2]], lang, "Flag"] ]& ]];
      (* 1/15/15: edit below to remove all SummaryOfNewFeatures* and NewIn* guides *)
      maNew = DeleteCases[maNew, {_, s_String /; 
      	StringMatchQ[s, "paclet:guide/AlphabeticalListing"|"paclet:guide/SummaryOfNewFeaturesIn*"|
      	  ("paclet:guide/NewIn" ~~ DigitCharacter.. ~~ __)]}];
      
      (* 1/13/15: Commenting out the below to allow heuristic sorting of Related Guides *)
      (*maNew = Sort[maNew,
        Module[{s1, s2, m1, m2},
          {s1, s2} = StringDrop[#, StringPosition[#, "guide/"][[-1, -1]]]& /@ {#1[[2]], #2[[2]]};
          {m1, m2} = StringMatchQ[#, "NewIn*"] & /@ {s1, s2};
          Which[
            m1 && m2, OrderedQ[{s1, s2}],
            m1, False,
            m2, True,
            True, OrderedQ[{s1, s2}] ]]& ];*)
      If[Length[maNew] > 0,
        expr = ReplacePart[expr,
          poss -> CellGroupData[Join[{maCell},
            Map[
              If[Head[#] === Cell, DeleteCases[#, r_[CellChangeTimes, _], Infinity],
              	Cell[TextData[ButtonBox[
                If[(Head@First[#] =!= String && Length@First@First[#] > 0),
                  First@First[#],
                  First[#]
                ], BaseStyle->"Link",
                ButtonData -> Last[#] ]], "MoreAbout", CellID->1]]&,
              maNew
          ]], Open]
      ]];
  ]]
  ];
  ][[1]];
If[debug==="True",Global`AntLog["Create MoreAbout: "<>time5]];
  
  

    (* delete GridBox Options on nColumnTableMod styled cells *)
    expr = expr /.
      Cell[a_, n:("2ColumnTableMod"|"3ColumnTableMod"|"3ColumnIllustratedTableMod"|"4ColumnTableMod"), d___, GridBoxOptions->_, y___]
        :> Cell[a, n, d, y];
        
    expr = expr /. Cell[a___, Rule[CellFrameLabels, {{cf1_, cf2_}, {cf3_, Cell[___, "TextAnnotation",___]}}], b___] :> Cell[a, b];
        
Global`AntLog["Starting time6"];time6 = ToString@AbsoluteTiming[
    expr = ReplacePart[expr, (# -> DocumentationBuild`Utils`MakePacletButton[Extract[expr, #]]) & /@ Position[expr, ButtonBox[_String, ___]]];
][[1]];
If[debug==="True",Global`AntLog["Running MakePacletButton: "<>time6]];
	
    expr = expr /.
           StyleBox[con_, sty:("MenuName"|"FullListing"), opt___] :>
             StyleBox[ (con /. ButtonBox[bb__] :>
               Module[{bs = Cases[{bb}, (BaseStyle->s_):>s]},
                 Apply[ButtonBox,
                   Append[DeleteCases[{bb}, _[BaseStyle, _]],
              BaseStyle -> Append[If[ListQ[bs], bs, {bs}], sty]
        ]]]
      ), sty, opt];

Global`AntLog["Starting time7"];time7 = ToString@AbsoluteTiming[
    (* Add label *)
    expr = AddNotebookLabel[expr, info, lang, "BuildMode"->buildmode, "PacletName" -> pacletName, "Rebranded" -> $Rebranded];
][[1]];

If[debug==="True", Global`AntLog["Running AddNotebookLabel: "<>time7]];

(* time8 = ToString@AbsoluteTiming[
    (* Add link trail *)
    expr = AddHeaderLinkTrail[expr, uri, lang];
][[1]];

If[debug==="True", Global`AntLog["Running AddHeaderLinkTrail: "<>time8]];

time9 = ToString@AbsoluteTiming[
    (* Add color bar *)
    expr = AddHeaderColorBar[expr, info];
 ][[1]];   
 
If[debug==="True",Global`AntLog["Running AddHeaderColorBar: "<>time9]];
 *)
    expr = expr /. {
      (* >> and >> -> \[RightGuillemet] *)
      ButtonBox[c_String /; StringMatchQ[c, RegularExpression["\\s*\\\\\\[RightSkeleton\\]\\s*"]], Rule[BaseStyle,{"Link","PreserveRightSkeletonLink"}], op___] :> 
        ButtonBox["PRESERVERIGHTSKELETONLINK", Rule[BaseStyle,"PreserveRightSkeletonLink"],op],
        
      ButtonBox[c_String /; StringMatchQ[c, RegularExpression["\\s*\\\\\\[RightSkeleton\\]\\s*"]], o___] :> 
        ButtonBox["\[RightGuillemet]", o],

      (* ExcludedObject is only for authoring nb *)
      ButtonBox[f__, BaseStyle->{"Link", "ExcludedObject"}, l___] :>
        ButtonBox[f, BaseStyle->"Link", l],
      Cell[f__, "ExcludedObject", l___] :> Cell[f, l],

      (* PrimaryObject is only for authoring nb *)
      ButtonBox[f__, BaseStyle->{"Link", "PrimaryObject"}, l___] :>
        ButtonBox[f, BaseStyle->"Link", l],

      (* Remove second style for Input/Output *)
      Cell[c_, "Input", "HiddenMaterial", o___] :> Cell[c, "Input", o],
      Cell[c_, "Output", "HiddenMaterial", o___] :> Cell[c, "Output", o]
    };
    
      (* Remove second style for Input, Deemphasis and place in Stylebox *)
    Cell[BoxData[a_], "Input", "Deemphasis", b__]:>
    Cell[BoxData[StyleBox[a, "InputDeemphasis"]], "Input", b];
    
    expr = expr /.  ButtonBox["PRESERVERIGHTSKELETONLINK", Rule[BaseStyle,"PreserveRightSkeletonLink"],opt___]:>
      ButtonBox["\[RightSkeleton]", Rule[BaseStyle,"Link"], opt];

       (* Change SeeAlso strings to links *)
      (* If 'SeeAlso' list is a string, make into a list of Buttons *) 
      expr = expr /. {
      Cell[c_String, "SeeAlso", o___] :> InlineFunctionButton[Cell[c, "SeeAlso", o]]};
     
       (* Fix links which are InlineFormula cells and should be Strings *)
       expr = expr /. {
      Cell[TextData[Cell[BoxData[ButtonBox[bb__]], "InlineFormula"]],
          sty:("Tutorials"|"RelatedLinks"|"MoreAbout"|"Extensions"),
          o___] :> 
        Cell[TextData[ButtonBox[bb]], sty, o]};

Global`AntLog["Starting time10"];time10=ToString@AbsoluteTiming[
    (* Apply the specific conversions for this type *)
    expr = ConvertNotebook[expr, info, type, opts, "Rebranding" -> $Rebranded];
][[1]];

If[debug==="True",Global`AntLog["Running ConvertNotebook: "<>time10]];

    BuildLog["-- Buttons into Paclet uris"];
    (* manipulate some button boxen *)


(*    (* Add label *)
    expr = AddNotebookLabel[expr, info, lang];
 
    (* Add link trail *)
    expr = AddHeaderLinkTrail[expr, uri, lang];

    (* Add color bar *)
    expr = AddHeaderColorBar[expr, info];*)

    BuildLog["-- Renumber In/Out labels"];
    (* Renumber In/Out Cell labels *)
 Global`AntLog["Starting time11"];time11 = ToString@AbsoluteTiming[
    expr = RenumberCells[expr,
      {"PrimaryExamplesSection","ExtendedExamplesSection","ExampleSection","ExampleSubsection",
       "ExampleDelimiter", "Title"}];
    ][[1]];
    
    If[debug==="True",Global`AntLog["Running RenumberCells: "<>time11]];

    BuildLog["-- Remove backlinks"];
    expr = expr //.
      Cell[c__, CellFrameLabels ->
          {{x_,  Cell[TextData[ ButtonBox[___, Rule[_, "ExampleBacklink"] , ___]]]}, {y_,  z_}}, o___]
      :>  Cell[c, CellFrameLabels -> {{x, None}, {y, z}}, o];

    (* Convert the old Example link syntax *)
	expr = expr /.
      Cell[BoxData[ButtonBox["Example", o__]], "Input"] :>
        ButtonBox["\[NonBreakingSpace]\[RightGuillemet]", o];
	expr = expr /. {
      (* Remove 'More Examples' link for now... *)
(*      Cell[_, "PrimaryExamplesSection", o___] :>
        Cell[BoxData[
            InterpretationBox[StyleBox[#,
                    "PrimaryExamplesSection"], ($Line = 0; Null)] &[Localized["PrimaryExamplesSectionText", lang]]], "PrimaryExamplesSection", o],*)
      Cell[_, "ExtendedExamplesSection", o___] :>
        Cell[Localized["ExtendedExamplesSectionText", lang], "ExtendedExamplesSection", o],

      (* remove TOCExcludedObject style *)
      Cell[c__, "TOCExcludedObject", o___] :> Cell[c, o],
      
      {"Link", "TOCExcludedObject"} :> "Link"    
    };

Global`AntLog["Starting time12"];time12 = ToString@AbsoluteTiming[
Module[{pacletBase = StringReplace[pacletName, " Package"->""]},If[MemberQ[{"", "ANOVA", "Audio", "BarCharts", "Benchmarking", "BlackBodyRadiation", "Calendar", "CCompilerDriver", "ClusterIntegration", "Combinatorica", "Compatibility", "ComputationalGeometry", "ComputerArithmetic", "DatabaseLink", "Developer", "DLLTools", "EquationTrekker", "ErrorBarPlots", "Experimental", "FiniteFields", "FourierSeries", "FunctionApproximations", "Geodesy", "GraphUtilities", "GUIKit", "HierarchicalClustering", "Histograms", "HypothesisTesting", "JLink", "LightweightGridClient", "LinearRegression", "Mathematica", "Messages", "MultivariateStatistics", "Music", "NETLink", "NonlinearRegression", "Notation", "NumericalCalculus", "NumericalDifferentialEquationAnalysis", "Parallel", "ParallelTools", "PhysicalConstants", "PieCharts", "PlotLegends", "PolyhedronOperations", "Polytopes", "PrimalityProving", "Quaternions", "RawGuides", "RegressionCommon","RemoteSLightweightGridCliental", "ResonanceAbsorptionLines","Splines", "StandardAtmosphere", "StatisticalPlots", "Symbols","Units", "VariationalMethods", "VectorAnalysis", "VectorFieldPlots","WebServices", "WorldPlot", "XML"}
, pacletBase],
expr = expr /. {      (* create self links *)
      Cell[BoxData[c_String /; ShippingFunctionQ[c, pacletBase] ], "InlineFormula", o___] :> 
        Cell[BoxData[ButtonBox[c, BaseStyle->"Link", ButtonData->"paclet:"<>pacletBase<>If[StringLength[pacletBase]>0,"/",""]<>"ref/"<>c]], "InlineFormula", o],
      
      Cell[BoxData[RowBox[{c_String /; ShippingFunctionQ[c, pacletBase], "[", stuff___},
          o1___], o2___], "InlineFormula", o3___] :> 
        Cell[BoxData[RowBox[{ButtonBox[c, BaseStyle->"Link", ButtonData->"paclet:"<>pacletBase<>If[StringLength[pacletBase]>0,"/",""]<>"ref/"<>c], "[", stuff},
          o1], o2], "InlineFormula", o3],
          
        Cell[BoxData[RowBox[{c_String /; ShippingFunctionQ[c, pacletBase], "->", stuff___},
          o1___], o2___], "InlineFormula", o3___] :> 
        Cell[BoxData[RowBox[{ButtonBox[c, BaseStyle->"Link", ButtonData->"paclet:"<>pacletBase<>If[StringLength[pacletBase]>0,"/",""]<>"ref/"<>c], "->", stuff},
          o1], o2], "InlineFormula", o3],
          
      Cell[BoxData[RowBox[{RowBox[{
        c_String /; ShippingFunctionQ[c, pacletBase], "[", stuff1___},
        opts1___], stuff2___}, opts2___], opts3___], "InlineFormula", opts4___] :> 
      Cell[BoxData[RowBox[{RowBox[{
        ButtonBox[c, BaseStyle->"Link", ButtonData->"paclet:"<>pacletBase<>If[StringLength[pacletBase]>0,"/",""]<>"ref/"<>c], "[", stuff1},
        opts1], stuff2}, opts2] opts3], "InlineFormula", opts4]
        }
]];
][[1]];
  
  If[debug==="True",Global`AntLog["Running ButtonBox Creation: "<>time12]];
    
    (* change \[EmptyVerySmallSquare] to "something else" *)
    expr = expr //. Cell[TextData[{a___, s_String /; StringMatchQ[s, RegularExpression["\\s*\[EmptyVerySmallSquare]\\s*"] ], b___}], 
      sty_String /; StringMatchQ[sty, "InlineGuideFunctionListing" | "GuideText" | "RelatedLinks"|"SeeAlso"|"Extensions"|"Tutorials"], o___] :> 
      Cell[TextData[{a, "\[NonBreakingSpace]", StyleBox["\[MediumSpace]\[FilledVerySmallSquare]\[MediumSpace]", "InlineSeparator"], " ", b}], sty, o];
      (* change \[EmptyVerySmallSquare] to "something else" for even more complex strings *)
    expr = expr //. Cell[TextData[{a___, s_String /; StringMatchQ[s, RegularExpression["\"\\w+\"\\s+\[EmptyVerySmallSquare]\\s+\\D+"] ], b___}], 
      sty_String /; StringMatchQ[sty, "InlineGuideFunctionListing" | "GuideText" | "RelatedLinks"|"SeeAlso"|"Extensions"|"Tutorials"], o___] :> 
      Cell[TextData[{a, "\[NonBreakingSpace]", StringSplit[s, RegularExpression["\\s+\[EmptyVerySmallSquare]\\s+"]->
      	Sequence[" ",StyleBox["\[MediumSpace]\[FilledVerySmallSquare]\[MediumSpace]", "InlineSeparator"], " "]], b}], sty, o];	
    expr = expr //. Cell[TextData[{a___, s_String /; StringMatchQ[s, RegularExpression["\\s+\[EmptyVerySmallSquare]\\s+"]~~__ ], b___}], 
      sty_String /; StringMatchQ[sty, "InlineGuideFunctionListing" | "GuideText" | "RelatedLinks"|"SeeAlso"|"Extensions"|"Tutorials"], o___] :> 
      Cell[TextData[{a, "\[NonBreakingSpace]", StringSplit[s, RegularExpression["\\s+\[EmptyVerySmallSquare]\\s+"]->
      	Sequence[" ",StyleBox["\[MediumSpace]\[FilledVerySmallSquare]\[MediumSpace]", "InlineSeparator"], " "]], b}], sty, o];	  	
    expr = expr //. Cell[TextData[{a___, s_String /; StringMatchQ[s, RegularExpression["\"\\w+\"\\s+\[EmptyVerySmallSquare]\\s+"]~~__ ], b___}], 
      sty_String /; StringMatchQ[sty, "InlineGuideFunctionListing" | "GuideText" | "RelatedLinks"|"SeeAlso"|"Extensions"|"Tutorials"], o___] :> 
      Cell[TextData[{a, "\[NonBreakingSpace]", StringSplit[s, RegularExpression["\\s+\[EmptyVerySmallSquare]\\s+"]->
      	Sequence[" ",StyleBox["\[MediumSpace]\[FilledVerySmallSquare]\[MediumSpace]", "InlineSeparator"], " "]], b}], sty, o];	
    expr = expr //. 
      	Cell[TextData[{a___, s_String /; StringMatchQ[s, ")"~~RegularExpression["\\s+\[EmptyVerySmallSquare]\\s+\\D+"]~~__ ], b___}], 
      sty_String /; StringMatchQ[sty, "InlineGuideFunctionListing" | "GuideText" | "RelatedLinks"|"SeeAlso"|"Extensions"|"Tutorials"], o___] :> 
      Cell[TextData[{a, (*"\[NonBreakingSpace]",*) StringSplit[s, RegularExpression["\\s+\[EmptyVerySmallSquare]\\s+"]->
      	Sequence[StyleBox["\[MediumSpace]\[FilledVerySmallSquare]\[MediumSpace]", "InlineSeparator"], " "]], b}], sty, o];	
    expr = expr //. Cell[TextData[{a___, s_String /; StringMatchQ[s, RegularExpression[" \\D+\\s+\[EmptyVerySmallSquare]\\s+"]~~__ ], b___}], 
      sty_String /; StringMatchQ[sty, "InlineGuideFunctionListing" | "GuideText" | "RelatedLinks"|"SeeAlso"|"Extensions"|"Tutorials"], o___] :> 
      Cell[TextData[{a, "\[NonBreakingSpace]", StringSplit[s, RegularExpression["\\s+\[EmptyVerySmallSquare]\\s+"]->
      	Sequence[" ",StyleBox["\[MediumSpace]\[FilledVerySmallSquare]\[MediumSpace]", "InlineSeparator"], " "]], b}], sty, o];	
      	
      	
    expr = expr //. Cell[TextData[{a___, s_String /; StringMatchQ[s, ") \[EmptyVerySmallSquare] " ], b___}], 
      sty_String /; StringMatchQ[sty, "InlineGuideFunctionListing" | "GuideText" | "RelatedLinks"|"SeeAlso"|"Extensions"|"Tutorials"], o___] :> 
      Cell[TextData[{a, ")\[NonBreakingSpace]", StyleBox["\[MediumSpace]\[FilledVerySmallSquare]\[MediumSpace]", "InlineSeparator"], " ", b}], sty, o];
      
    (* GuideTOCLinks need a RightGuillemet *)
    expr = addGuideTOCRightGuillemet[expr];

(*    expr = expr /. {
      (* >> and >> -> \[RightGuillemet] *)
      ButtonBox[c_String /; StringMatchQ[c, RegularExpression["\\s*\\\\\\[RightSkeleton\\]\\s*"]], o___] :> 
        ButtonBox["\[RightGuillemet]", o],

      (* ExcludedObject is only for authoring nb *)
      ButtonBox[f__, BaseStyle->{"Link", "ExcludedObject"}, l___] :>
        ButtonBox[f, BaseStyle->"Link", l],
      Cell[f__, "ExcludedObject", l___] :> Cell[f, l],

      (* PrimaryObject is only for authoring nb *)
      ButtonBox[f__, BaseStyle->{"Link", "PrimaryObject"}, l___] :>
        ButtonBox[f, BaseStyle->"Link", l],

      (* Remove second style for Input/Output *)
      Cell[c_, "Input", "HiddenMaterial", o___] :> Cell[c, "Input", o],
      Cell[c_, "Output", "HiddenMaterial", o___] :> Cell[c, "Output", o]
    };*)
Global`AntLog["Starting time13"];time13 = ToString@AbsoluteTiming[
expr = ReplacePart[expr, (# -> DocumentationBuild`Utils`MakePacletButton[Extract[expr, #]]) & /@ Position[expr, ButtonBox[_String, ___]]];
][[1]];

If[debug==="True",Global`AntLog["Running MakePacletButton2: "<>time13]];

    expr = expr /.
      ButtonBox[c__, BaseStyle->"Link", o1___, ButtonData->p_String/;StringMatchQ[p, "paclet:note/*"], o2___] :>
        ButtonBox[c, BaseStyle->"Hyperlink", o1, 
          ButtonData -> {URL[ DotHTML[StringReplace[p, "paclet:"->"http://reference.wolfram.com/language/"]]],
            None}, o2];

    (* fix split hyperlinks *)
Global`AntLog["Starting time14"];time14 = ToString@AbsoluteTiming[
    expr = DocumentationBuild`Utils`Private`fixSplitHyperlinks[expr];
    ][[1]];
    
    If[debug==="True",Global`AntLog["Running fixSplitHyperlinks: "<>time14]];
    
    (* Have to do this late, because MakeCommon has a rule to kill GridBoxOptions *)
    BuildLog["-- Highlight ModInfo rows"];
    poss = Position[expr,
        Cell[_, "2ColumnTableMod"|"3ColumnTableMod"|"3ColumnIllustratedTableMod"|"4ColumnTableMod", ___],
        Infinity];
    BuildLog["   Number of tables to check: "<> ToString@Length@poss];
    Scan[
      Module[{cexpr, grid, options},
        cexpr = Extract[expr, #];
        grid = First@Cases[cexpr, _GridBox, Infinity];
        options = Cases[cexpr, _[GridBoxOptions, s_]:>s];
        If[Length[options]>0, options=First@options];
        BuildLog["   Length opts: "<>ToString@Length@options];
        expr = ReplacePart[expr, # -> Append[
          DeleteCases[cexpr, _[GridBoxOptions, _]],
          GridBoxOptions->Join[options, {
            RowBackgrounds->Map[
              If[MatchQ[First[#],
                  Cell["\[FilledVerySmallSquare]", "TableRowIcon", ___]],
                Dynamic[If[CurrentValue[{TaggingRules, "ModificationHighlight"}]===True,
                  RGBColor[1, 0.988235, 0.721569], None]],
                None]&,
              First@grid],
            GridBoxDividers->{"Rows" -> {Join[{True}, Map[
              (Last[#] =!= "\[SpanFromLeft]")&,
              First@grid]
            ]}}
          }]
        ]]
      ]&,
      poss];

    (* Replace any remaining ModInfo with a space TODO: Comment out for internal documentation variant *)
    expr = expr /. Cell[_, "ModInfo", o___] :> Cell[" ", "ModInfo", o];
    expr = If[buildmode==="Future",
    expr,
    DeleteCases[expr,
      (Rule|RuleDelayed)[CellDingbat, Cell[s_String /; StringMatchQ[s, RegularExpression["[\\s]*"]], "ModInfo", ___]],
      Infinity]
    ];
    expr =If[buildmode==="Future",
    expr,
    DeleteCases[expr,
      Cell[s_String /; StringMatchQ[s, RegularExpression["[\\s]*"]], "ModInfo", ___],
      Infinity]
    ];

    (* remove empty cell groups *)
    BuildLog["-- Remove empty cell groups"];
Global`AntLog["Starting time15"];time15 = ToString@AbsoluteTiming[
    (* V9 EDIT BELOW *)
    expr = 
    	RemoveEmptyCellGroups[expr, 
			Flatten[{
				"CapitalizeHeadingStyles" /. Options[MakeNotebook],
				{"ExampleSection", "PrimaryExamplesSection", "ExampleSubsection", "GuideFunctionsSection", "GuideFunctionsSubsection", 
					"GuideMoreAboutDelimiter", "GuideRelatedLinksSection", "GuideDelimiter", "GuideTutorialsSection", "GuideMoreAboutSection",
					"IndicatorExampleSection", "IndicatorDescriptionSection", "IndicatorCategorizationSection", "IndicatorAbbreviationSection", "IndicatorFormulaSection", "SeeAlsoSection",
                     "RelatedLinksSection", "MoreAboutSection", "TutorialsSection", "TutorialRelatedLinksSection", "TutorialMoreAboutSection",
                     "RelatedTutorialsSection", "TutorialsTutorialCollectionSection", "RelatedDemonstrationsSection",
                     "FeaturedExampleMoreAboutSection", "FeaturedExampleSeeAlsoSection", "FeaturedExampleRelatedFunctionsSection", 
                     "FeaturedExampleRelatedLinksSection", "DeviceSubsection", "ServiceSubsection", "FunctionEssaySection",
                     "EmbeddingFormatSection", "InterpreterSection", "AlternativeNamesSection", "DesignJustificationSection",
                     "InternalLinksSection", "DesignChoicesSection", "FormatBackground", "ProgramSubsection", "ProgramSection"}
			}] ];
    ][[1]];

    If[debug==="True",Global`AntLog["Running RemoveEmptyCellGroups2: "<>time15]];
    
 (* Remove short forms, etc. 'Map (/@)' *) 
    ShortFormTitle[pageTitle_]:=
      Module[{s, o = pageTitle},
        s = StringSplit[pageTitle, " "];
        If[(Length@s > 1) && StringMatchQ[s[[2]], "(*"],
          First@s, o]];
			
	(*Italicize AutoItalicWords in MoreAbout cells *)
   moreAboutCells=Cases[expr, 
  	Cell[TextData[ButtonBox[_String,__]],"MoreAbout",___],
  	Infinity ];
  	
   autoitalics = AutoItalicWords /. DocumentationBuild`Common`PubsEvaluateWithFE[Options[Cell, AutoItalicWords]];

   newMoreAboutCells=Map[ItalicizeMoreAboutKeyWords[#, autoitalics]&, moreAboutCells];

   moreAboutCellPos=Position[expr, 
  	Cell[TextData[ButtonBox[_String,__]],"MoreAbout",___],
  	Infinity ];

   expr = ReplacePart[expr, Thread[Rule[moreAboutCellPos, newMoreAboutCells]]    ];

   (* V9 EDIT (NEW) BELOW *)
   (* Adding spacers to beginnings and ends of sections (but NOT See Also sections) *)
   Module[{sections="MoreAboutSection"|"TutorialsSection"|"GuideMoreAboutSection"|"RelatedLinksSection"|
         "GuideMoreAboutSection"|"GuideRelatedLinksSection"|"GuideTutorialCollectionSection"|"GuideTutorialsSection"|
         "TutorialMoreAboutSection"|"NotesSection"|"TutorialRelatedLinksSection"|
         "RelatedTutorialsSection"|"TutorialsTutorialCollectionSection"|"TutorialCollectionSection"|
         "ImportExportSection"|"NotebookInterfaceSection"|"OptionsSection"|"ElementsSection"|
         "IndicatorDescriptionSection"|"IndicatorCategorizationSection"|"IndicatorAbbreviationSection"|
         "IndicatorFormulaSection"|"GuideFunctionsSection"|
         "FeaturedExampleMoreAboutSection"|"FeaturedExampleSeeAlsoSection"|"FeaturedExampleRelatedFunctionsSection"|
         "FeaturedExampleRelatedLinksSection"|"DeviceSubsection"|"ServiceSubsection"|"EmbeddingFormatSection"|
         "InterpreterSection"|"FunctionEssaySection"|"AlternativeNamesSection"|"DesignJustificationSection"|
         "InternalLinksSection"|"DesignChoicesSection",
         numsects,cells, pos, endpos},
     cells = Cases[expr, Cell[CellGroupData[{Cell[___,
         sections, ___],___}, ___], ___], Infinity];
     numsects = 
       Length[cells];
     (* Every section gets a header *)
     expr = expr /. MapThread[Rule, {cells, Insert[#, Cell["","SectionHeaderSpacer"], {1,1,2}]&/@cells}];
     (* The last section does NOT get a footer *)
     cells = Cases[expr, Cell[CellGroupData[{Cell[___,
         sections, ___],___}, ___], ___], Infinity];
     (* special-cased for Message pages, that only have a notes section and examples -- 
        but need a footer in the notes *)
     Do[expr = expr /. cells[[i]]->Insert[cells[[i]],Cell["","SectionFooterSpacer"],{1,1,-1}],
       {i,If[MatchQ[type, "Message"], numsects, numsects-1]}];
     (* Hack for GuideReferenceSection cell since it hasn't grouped yet *)
     pos = Position[expr, Cell[___, "GuideReferenceSection", ___]];
     If[pos =!= {},
        pos = MapAt[#+1&, pos, {1,-1}];
        expr = Insert[expr, Cell["", "SectionHeaderSpacer"], pos];
        If[cells =!= {},
          endpos = {Append[Most[pos[[1]]], Length[expr[[Sequence@@Most[pos[[1]]]]]] + 1]};
          expr = Insert[expr, Cell["", "SectionFooterSpacer"], endpos];
        ];
     ];
     (* Hack for spacer after example cells since loading Manipulate examples breaks everything *)
     If[MatchQ[type, "Symbol"|"Indicator"|"Format"] && 
        MemberQ[expr, Cell[___, "PrimaryExamplesSection"|"IndicatorExampleSection", ___], Infinity],
       Which[
         (pos = Position[expr, Cell[___, "SeeAlsoSection", ___]]) =!= {},
         expr = Insert[expr, Cell["", "SectionFooterSpacer"], pos],
         (pos = Position[expr, Cell[___, "TutorialsSection", ___]]) =!= {},
         expr = Insert[expr, Cell["", "SectionFooterSpacer"], pos],
         (pos = Position[expr, Cell[___, "MoreAboutSection", ___]]) =!= {},
         expr = Insert[expr, Cell["", "SectionFooterSpacer"], pos],
         (pos = Position[expr, Cell[___, "RelatedLinksSection", ___]]) =!= {},
         expr = Insert[expr, Cell["", "SectionFooterSpacer"], pos]
       ]
     ]; 
     (* Guess what?  Another hack.  This time for guides without a Reference section (menu guides) *)
     If[MatchQ[type, "Guide"] && FreeQ[expr, Cell[___, "GuideReferenceSection", ___], Infinity],
       Which[
         (pos = Position[expr, Cell[___, "GuideTutorialCollectionSection", ___]]) =!= {},
         expr = Insert[expr, Cell["", "SectionFooterSpacer"], pos],
         (pos = Position[expr, Cell[___, "GuideMoreAboutSection", ___]]) =!= {},
         expr = Insert[expr, Cell["", "SectionFooterSpacer"], pos]
       ]
     ];        
   ];

   (* Removing Tutorial Collections cells *)
   expr = DeleteCases[expr, Cell[CellGroupData[{
   	Cell[___, 
   		"TutorialCollectionSection"|"GuideTutorialCollectionSection"|
   		  "TutorialsTutorialCollectionSection", 
   		 ___], ___}, ___]], Infinity];

   expr = expr/. Cell[a___, b:(
   	"ElementsSection" | 
   	"ExampleSection"|
   	"ExampleSubsection"|
   	"ExampleSubsubsection"|
   	"ExtendedExamplesSection" |
   	"GuideRelatedLinksSection"| 
   	"GuideTutorialsSection"| 
   	"GuideMoreAboutSection"|
    "ImportExportSection" | 
    "NotebookInterfaceSection" | 
    "MoreAboutSection" |
    "MoreToExploreSection" | 
    "MoreInformationSection" | 
    "OptionsSection" | 
    "PrimaryExamplesSection" |  
    "RelatedLinksSection"|
    "RelatedTutorialsSection"|
    "SeeAlsoSection" | 
    "TutorialsSection" | 
    "TutorialCollectionSection" |
    "TutorialsTutorialCollectionSection" |
    "GuideTutorialCollectionSection" |
    "TutorialRelatedLinksSection" |
    "UsageDetailsSection" | "NotesSection" |
    "TutorialMoreAboutSection" | "GuideReferenceSection" |
    "GuideLearningResourcesSection" | "FeaturedExampleMoreAboutSection" | 
    "FeaturedExampleSeeAlsoSection" | "FeaturedExampleRelatedFunctionsSection" |
    "FeaturedExampleRelatedLinksSection" | "DeviceSubsection" | "ServiceSubsection" | "FunctionEssaySection" | 
    "InterpreterSection" | "FormatBackground"
   	), c___]:>Cell[a, b, c, System`WholeCellGroupOpener->True];

colortype = Switch["EntityType" /. info,
      "Guide" | "Guide Overview" | "Alphabetical Listing", "Guide",
      "Tutorial" | "Overview", "Tutorial", 
      "HowTo", "Tutorial",
      "Example", "Tutorial",
      "Root Guide", "Root",
      _ , "Symbol"
    ];
colortype = colortype<>"Color";


(* FIXME (someday): hacked in "Mathematica" -> "Wolfram Language" replacement here to not break 
   link trail gen code (2/3/14) *)
linktrailexprs = Cases[Flatten[{
	ReplaceAll[MakeLinkTrailExpr[uri, lang], 
		StyleBox[ButtonBox["Mathematica", bb___], FontSlant->"Italic"] :> 
		ButtonBox[Localized["WolframLanguageText", lang], bb]]}], Cell[___, "LinkTrail", ___],Infinity];
linktrailexprs = linktrailexprs //. {
  Cell[a___, "LinkTrail", b___] :> Cell[a, "DockedLinkTrail", b], 
  StyleBox[a___, "LinkTrailSeparator", b___] :> StyleBox[a, "DockedLinkTrailSeparator", b], 
  Rule[BaseStyle, {"Link", "LinkTrail"}] :> Rule[BaseStyle, {"Link", "DockedLinkTrail"}]
  };
examplepagedata = If[type === "Guide", 
	DocumentationBuild`Info`GetSourceInformation["paclet:" <> uri, lang, "FeaturedExampleLinks"],
	{}];
examplepagedata = Map[
	{DocumentationBuild`Info`GetSourceInformation["paclet:example/" <> #[[1]], lang, "Thumbnail"],
	DocumentationBuild`Info`GetSourceInformation["paclet:example/" <> #[[1]], lang, "Title"],
	DocumentationBuild`Info`GetSourceInformation["paclet:example/" <> #[[1]], lang, "URI"]}&, 
	examplepagedata];
	
hasoptions = DocumentationBuild`Info`GetSourceInformation["paclet:"<>uri, lang, "HasOptions"];
	
    (* add spacer to pass-through sections (to keep text from crashing into openers) *)
    expr = ReplaceAll[expr,
      Cell[s_String, some___, 
        sty:("SeeAlsoSection"|"MoreAboutSection"|"TutorialsSection"|"TutorialCollectionSection"|
         "GuideMoreAboutSection"|"RelatedLinksSection"|
         "GuideRelatedLinksSection"|"GuideTutorialCollectionSection"|"GuideLearningResourcesSection"|"GuideTutorialsSection"|
         "TutorialMoreAboutSection"|"TutorialRelatedLinksSection"|
         "RelatedTutorialsSection"|"TutorialsTutorialCollectionSection"|"GuideReferenceSection"|
         "ImportExportSection"|"NotebookInterfaceSection"|"ElementsSection"|"OptionsSection"|
         "IndicatorExampleSection"|"IndicatorDescriptionSection"|"IndicatorCategorizationSection"|
         "IndicatorAbbreviationSection"|"IndicatorFormulaSection"|"PrimaryExamplesSection"|
         "GuideFunctionsSection"|
         "FeaturedExampleMoreAboutSection"|"FeaturedExampleSeeAlsoSection"|"FeaturedExampleRelatedFunctionsSection"|
         "FeaturedExampleRelatedLinksSection"|"DeviceSubsection"|"ServiceSubsection"|"FunctionEssaySection"|
         "EmbeddingFormatSection"|"InterpreterSection"|"RelatedInterpreterSection"|"AlternativeNamesSection"|
         "DesignJustificationSection"|"InternalLinksSection"|"DesignChoicesSection"|"FormatUsageSection"|"FormatBackground"),
      rest___] :>
      Cell[TextData[{Cell[BoxData[ToBoxes[Spacer[6]]]],s}], some, sty, rest]
    ];

    (* Handle rebranding terms (change them back to "Mathematica") if $Rebranded is False *)
    (* Also remove rebranding tagging information if $Rebranded is true *)
    If[!TrueQ[$Rebranded], 
    	expr = ReplaceRepeated[expr, StyleBox[_, "RebrandingTerm"] -> StyleBox["Mathematica", FontSlant->"Italic"]];
    	expr = ReplaceRepeated[expr, StyleBox[_, "RebrandingTerm", "ReplacedText"->text_] :> text],
    	
    	expr = ReplaceRepeated[expr, StyleBox[c_, "RebrandingTerm", ___] :> c];
    ];

    (* Remove Under Development banner (replaced by UNFINISHED DRAFT banner, added during 
       ConvertNotebook[] *)
    expr = DeleteCases[expr, 
      Cell["U N D E R   D E V E L O P M E N T", ___], Infinity];

    (* Moving Unfinished Draft banner on Interpreter pages because hack hack hack *)
    If[MemberQ[{"Interpreter"}, type] && MemberQ[expr, "DeviceAbstract", Infinity],
      expr = Module[{cell = First[Cases[expr, Cell[___, "DeviceAbstract", ___], Infinity]], pos, bannerpos},
        pos = Position[expr, Cell[___, "ObjectNameAlt", ___]];
        bannerpos = MapAt[(# + 1 &), pos, {1, -1}];
        Insert[DeleteCases[expr, cell, Infinity], cell, bannerpos]]];

    (* Fix bug 246625; apparently ButtonBox -> TemplateBox conversion was messing up summary generation *) 
    summary = (DocumentationBuild`Info`GetNotebookSummary[expr, ("EntityType" /. info),
                ("Title" /. info)]);

    (* Convert ButtonBoxes to TemplateBoxes *)
    expr = If[type =!= "Root Guide", DocumentationBuild`Utils`ConvertButtonBoxesToTemplateBoxes[expr], expr];

    (* Replace Function Essay bit with something more Dynamic *)
    (* Since font changed to Verdana, let's turn off the function name resizing *)
    fnessay = (*ReplaceAll[*)Flatten[Cases[expr, 
    	Cell[CellGroupData[{Cell[__, "FunctionEssaySection", ___], ___, 
    		cell : (Cell[__, "FunctionEssay", ___] ..), 
    		Cell[c__ /; FreeQ[{c}, "FunctionEssay"]] ...}, ___]] :> {cell}, Infinity]](*, 
      Cell[some___,"InlineFormula",rest___]:>
      Cell[some,"InlineFormula",rest, FontSize->12]]*);
    expr = If[fnessay =!= {}, (* comments below reflect openness of section, per SW (4/21/14) *)
       (*With[{
       	showmoreimg = ToBoxes@Image[Import[FileNameJoin[
       		{$DocumentationBuildDirectory, "Internal", "online", "Images", "FunctionEssay-showmore-arrow.png"}]],
             ImageSize->Magnification[1], BaselinePosition->Baseline],
       	showlessimg = ToBoxes@Image[Import[FileNameJoin[
       		{$DocumentationBuildDirectory, "Internal", "online", "Images", "FunctionEssay-showless-arrow.png"}]],
             ImageSize->Magnification[1], BaselinePosition->Baseline]},*)
       	ReplaceAll[expr, Cell[CellGroupData[{Cell[some___, "FunctionEssaySection", other___], ___},___]] :>
         Cell[CellGroupData[{Cell[some, "FunctionEssaySection", other], Cell["", "SectionHeaderSpacer"],
            With[{full=Cell[BoxData[fnessay],"FunctionEssay"],first=First[fnessay]},
            Cell[BoxData[DynamicBox[
              If[(*CurrentValue[EvaluationNotebook[],{TaggingRules,"FunctionEssayOpen"}]*)True,full,
              first]]],"FunctionEssay",Selectable->True]],
            (*Cell[BoxData[DynamicBox[If[
              CurrentValue[EvaluationNotebook[],{TaggingRules,"FunctionEssayOpen"}],
               TagBox[ButtonBox[StyleBox[RowBox[{"Show Less ", showlessimg}],"FunctionEssayShowMore"],BaseStyle->"Link",Evaluator->Automatic,Method->"Queued",
                 ButtonFunction:>(CurrentValue[EvaluationNotebook[],{TaggingRules,"FunctionEssayOpen"}]=False)],MouseAppearanceTag["LinkHand"]],
               TagBox[ButtonBox[StyleBox[RowBox[{"Show More ", showmoreimg}],"FunctionEssayShowMore"],BaseStyle->"Link",Evaluator->Automatic,Method->"Queued",
                 ButtonFunction:>(CurrentValue[EvaluationNotebook[],{TaggingRules,"FunctionEssayOpen"}]=True)],MouseAppearanceTag["LinkHand"]]]]],"FunctionEssayShowMore"],*)
            Cell["", "SectionFooterSpacer"]}, 
          Closed]]](*]*),
       expr];

    (* Frame KeyEvents *)
    expr = ReplaceAll[expr, StyleBox[s_String, some___, "KeyEvent", rest___]:>
      Cell[s, "KeyEvent", some, rest]];

    (* Check SeeAlsos and guide inline listing links for bad (future, nonexistent) stuff and strip it *)
    expr = Which[MatchQ[type, "Symbol" | "Interpreter"], DocumentationBuild`Utils`CheckSeeAlsos[expr, DocumentationBuild`Export`$CoreInformation[lang]],
             MatchQ[type, "Guide"], (* TODO *)DocumentationBuild`Utils`CheckGuideInlineListings[expr, DocumentationBuild`Export`$CoreInformation[lang]],
             True, expr];


    BuildLog["   Return Notebook expression."];
    Global`AntLog["Starting time16"];time16 = ToString@AbsoluteTiming[
    expr = Notebook[Flatten[{ First@expr, historyCell, footerCell }], 
      Sequence@@Flatten@{
        ScreenStyleEnvironment -> "Working", 
        WindowSize -> {725, 750}, 
        Saveable -> False,
        (* FIXME: we may need to do something different for packages *)
        WindowTitle -> If[type === "Root Guide", Localized["RootGuideWindowTitle", lang],
        	(StringReplace[("Title" /. info), {" (\[Conditioned])"->""," (\[Distributed])"->"",  " (<->,\[UndirectedEdge])"->"",  
             " (\[DirectedEdge])"->"", "Wolfram Mathematica"->"Mathematica", 
             StartOfString~~"Automatic"~~EndOfString -> " Automatic", StartOfString~~"None"~~EndOfString -> " None"}] (*<> " - Wolfram Mathematica"*))
             ],
        WindowMargins -> {{0, Automatic}, {Automatic, 0}}, 
        StyleDefinitions -> stylesheet,
        If[type === "Root Guide",
          WindowElements -> {"StatusArea", "MemoryMonitor", "MagnificationPopUp",
                             "VerticalScrollBar", "MenuBar"},
          {} 
        ],
		(* explicit CellContext setting - present in some/most M- Tutorials *)
        If[nbContext =!= CellContext, 
        	CellContext->nbContext, 
        	{}
        ],
        TaggingRules -> Flatten@{
          "ModificationHighlight" -> False,
          (*"ColorType"->If[type==="Root Guide", "", colortype],*)
          "ColorType"->colortype,
          "LinkTrails"->If[type === "Root Guide" || type === "HowTo", "", 
          	If[Length[linktrailexprs]>0, (GridBox[Map[List,linktrailexprs], ColumnAlignments -> Left]/. Cell[TextData[{a___}], b___] :> RowBox[{a}])/.StyleBox[" > ", "DockedLinkTrailSeparator"] :> Cell[">", "DockedLinkTrailSeparator"], ""]],
          If[MatchQ[hasoptions, {} | None | "None"], {}, "HasOptions" -> hasoptions],
          "ExampleCounter" -> 1,
          (* Commented out in removal of Example Page links per SW: 4/6/15 *)
          (*"ExamplePageLinks" -> examplepagedata,*)
          "NeedPlatMsgIn" -> 
            Module[{plats = {}}, 
            	If[
            		StringMatchQ[uri,DocumentationBuild`URIsWithoutDesktopSupport],
            	    AppendTo[plats, "Desktop"]
            	    ];
            	If[
            		StringMatchQ[uri,DocumentationBuild`URIsWithoutCloudSupport],
            		AppendTo[plats, "Cloud"]
            	    ];
            	If[
            		StringMatchQ[uri, DocumentationBuild`URIsWithoutPublicCloudSupport],
            		AppendTo[plats, "PublicCloud"]
            	];
            	If[plats === {}, None, Compress[plats]]
            	],
          (*"FunctionEssayOpen" -> False,*)
          "RootCaptions"->rootCaptionList,
          If[overviewLinks === {}, {}, "Overview" -> overviewLinks],
          "Metadata"-> ConvertValuesForSearch[{
            "built" -> ToString@Date[],
            "history" -> {"New", "Modified", "Obsolete", "Excised"} /. ("HistoryData" /. info),
            "context" -> ("Context" /. info),
            "keywords" -> ("Keywords" /. info),
            "specialkeywords" -> specialKeywords,
            "tutorialcollectionlinks" ->("TutorialCollectionLinks" /. info),
            "index" -> "IndexQ" /. info,
            "label" -> classLabel,
            "language" -> GetLanguageExtension[lang],
            "paclet" -> ("PacletName" /. info),
            "status" -> ("Flag" /. info /. None -> "None"),
            (* This was replaced to fix bug 235409, but was added to fix bug 226210, so
               check and make sure that both bugs are fixed later.
            "summary" -> If[!StringFreeQ[("Summary" /. info), "XXXX"], "", ("Summary" /. info)],*)
            "summary" -> summary,
            "synonyms" -> ("Synonyms" /. info),
            "tabletags" -> tabletaggingrules,
            (* FIXME: this needs to be special-cased for root guide *)
            "title" -> ShortFormTitle[("Title" /. info)],
            "titlemodifier"->("TitleModifier" /. info),
            "windowtitle" ->("Title" /. info),
            "type" -> ("EntityType" /. info),
            "uri" -> uri
          }, lang] }
    }]; 
    ][[1]];
    
    If[debug==="True",Global`AntLog["Assemble Notebook: "<>time16]];
    
    expr = If[type==="Root Guide", 
    	Join[
    DeleteCases[expr,
      ((CellGrouping|Selectable|ShowGroupOpener|Background)~(Rule|RuleDelayed)~_)],
    Notebook[
      (*CellGrouping->Manual,*)
      ShowGroupOpener->False,
      Selectable->False,
      Background->GrayLevel[0.266667]
    ]]
    ,expr];
    
Global`AntLog["Starting time17"];time17 = ToString@AbsoluteTiming[
	CacheStylesheet[expr];
    makenboutput= If[outfilelocation===False,
        PubsEvaluateWithFE[ApplyConvertGraphicsToBitmapsToNotebook[expr, type, uri]],
        {PubsEvaluateWithFE[SaveConvertGraphicsToBitmapsToNotebook[outfilelocation, expr, type, uri,"Debug"->debug, "SafeRaster"->saferaster]], DocumentationBuild`Utils`GetSearchMetaDataList[expr]}
        ];
][[1]];
        
    If[debug==="True",Global`AntLog["Use Frontend to build final notebook: "<>time17]];
       
    makenboutput
  ];


(******************************************************************************)
(** ConvertNotebook - Alphabetical Listings **)
ConvertNotebook[expr_, info_,
    ("Alphabetical Listing"),
    opts___?OptionQ] := 
  Module[{chars, linkcell, pos, nb=expr},
    chars = Cases[nb, Cell[c_String, ___, "GuideFunctionsSubsection", ___] :> c, Infinity];
    linkcell = Cell[BoxData[GridBox[{TagBox[
       ButtonBox[
         PaneSelectorBox[{False->#, True->StyleBox[#,FontColor->RGBColor[0.854902, 0.396078, 0.145098]]}, Dynamic[CurrentValue["MouseOver"]]], 
       BaseStyle->{"Hyperlink", FontFamily->"Verdana"}, 
       ButtonFunction:>NotebookLocate[#], ButtonData->"", ButtonNote->""], MouseAppearanceTag["LinkHand"]]& /@ chars}]], 
       "AlphabeticalListingAnchor"];
    pos = Position[nb, Cell[___, "GuideAbstract", ___]];
    If[pos === {}, pos = Position[nb, Cell[___, "GuideTitle", ___]]];
    pos = MapAt[#+1&, pos, {1,-1}];
    If[chars =!= {}, nb = Insert[nb, linkcell, pos]];
    nb
  ];


(******************************************************************************)
(** ConvertNotebook - Example pages **)
ConvertNotebook[expr_, info_,
    ("Example" | "Featured Example"),
    opts___?OptionQ] := 
  Module[{nb=expr, lang},
    lang = "Language" /. {opts} /. Options[MakeNotebook];

    nb = DeleteCases[nb, Cell[___, "ContributorSection" | "Contributor" | "WeightingSection" | "Weight" | "AlternateThumbnail", ___], Infinity];

    BuildLog["-- Localize MoreAbout on example pages"];
    nb = nb /. Cell[c_, "FeaturedExampleMoreAboutSection", o___] :> 
      Cell[Localized["MoreAboutSectionText", lang], "FeaturedExampleMoreAboutSection", o];
      
    BuildLog["-- Localize see also section on example pages"];
    nb = nb /. Cell[c_, "FeaturedExampleSeeAlsoSection", o___] :> 
      Cell[Localized["SeeAlsoSectionText", lang], "FeaturedExampleSeeAlsoSection", o];

    BuildLog["-- Localize RelatedLinks on example pages"];
    nb = nb /. Cell[c_, "FeaturedExampleRelatedLinksSection", o___] :> 
      Cell[Localized["RelatedLinksSectionText", lang], "FeaturedExampleRelatedLinksSection", o];
   
   nb = RenumberCells[nb, {"FeaturedExampleTitle", "GuideDelimiter"}];
   
    nb
  ];


(******************************************************************************)
(** ConvertNotebook - new-style Program (man) pages **)
ConvertNotebook[nb_, info_,
    ("Program"),
    opts___?OptionQ] := 
  Module[{expr=nb, lang, type, uri, seealsos, newseealsos},
    lang = "Language" /. {opts} /. Options[MakeNotebook];
    type = "EntityType" /. info;
    uri = "URI" /. info;

    BuildLog["-- Localize MoreAbout on program pages"];
    expr = expr /. Cell[c_, "MoreAboutSection", o___] :> 
      Cell[Localized["MoreAboutSectionText", lang], "MoreAboutSection", o];
      
    BuildLog["-- Localize see also section on program pages"];
    expr = expr /. Cell[c_, "SeeAlsoSection", o___] :> 
      Cell[Localized["SeeAlsoSectionText", lang], "SeeAlsoSection", o];

    BuildLog["-- Localize RelatedLinks on program pages"];
    expr = expr /. Cell[c_, "RelatedLinksSection", o___] :> 
      Cell[Localized["RelatedLinksSectionText", lang], "RelatedLinksSection", o];

    BuildLog["-- Delete various cells"];
    If[buildmode==="Future",
    	expr,
    	expr = DeleteCases[expr, Alternatives[
        Cell[s_String /;
          StringMatchQ[s, RegularExpression[".*XX+"]], ___],
        Cell[TextData[s_List /; Apply[ Or,
        	StringMatchQ[Transmogrify`ConvertToString[#], 
        		RegularExpression[".*XX+.*"]] & /@ s]], "Notes", ___],
        Cell[cc_ /; (And[
            Length[#] > 0,
            StringMatchQ[Transmogrify`ConvertToString[First[#]], RegularExpression["XX+"]]
          ]&[Cases[cc,
            Cell[_, ("InlineFormula"|"InlineGuideFunctionListing"), ___], Infinity]]),
          "InlineGuideFunctionListing" | "GuideText" ___],
        Cell[s_String /;
          StringMatchQ[s, RegularExpression["(XX+( \\. )?)+"]],
          "InlineFormula", ___],
        Cell[s_String /;
          StringMatchQ[s, RegularExpression["(XX+( \\. )?)+"]],
          "InlineGuideFunctionListing", ___],
        Cell[cc_ /;
          (Count[cc, Cell[BoxData[s_String /;
              StringMatchQ[s, RegularExpression["XX+"]]
            ], "InlineGuideFunction", ___], Infinity] > 0),
          "Notes", ___]
    ], Infinity]];

    seealsos = First[Cases[expr, Cell[__, "SeeAlso", ___], Infinity]];
    newseealsos = ReplaceAll[seealsos, 
      ButtonBox[c___]:>StyleBox[ButtonBox[c], FontFamily->"Verdana"]];
    expr = expr /. seealsos :> newseealsos;

    (* Convert Notes cells to ProgramNotes cells *)
    expr = expr /. Cell[first__, "Notes", rest___] :> Cell[first, "ProgramNotes", rest];
    expr = expr /. Cell[first__, "ObjectName"|"ObjectNameAlt", rest___] :> Cell[first, "ProgramName", rest];
   
    expr
  ];


(******************************************************************************)
(** ConvertNotebook - Symbols **)
ConvertNotebook[nb_, info_,
    ("AppleScript" | "Character" |  "Character Name" | "File" | "Front End Object" | "LibraryLink C Function" | "MathLink C Function" | "Menu Item" |
     "Message" | "Method" | "Symbol" | "System Program" | "Widget" | "Format" | "Screencast" | "Indicator" | "Device Connection" | "Service Connection" | 
     "Interpreter" | "Embedding Format"),
    opts___?OptionQ] := 
  Module[{expr, lang, poss, uri, pacletName, type,
      awaitingReviewNote, obsolescenceNote, objectNameCell, objectName, underDevelopmentCell, characterNameCell, characterName, characterImageCell, chargridOpts, characterImage, characterImageGrid, usageNameCell, examplesGroup, notesGroup, seeAlsoGroup, relatedInterpretersGroup,
      importExportGroup,   indicatorExampleGroup, indicatorDescriptionGroup, indicatorFormulaGroup, elementGroup, notebookInterfaceGroup, optionsGroup, tutorialsGroup, relatedLinksGroup, moreAboutGroup, 
      elucidationGroup, tutorialCollectionGroup, otherTypeGroup, headerCells, nbOrder, closedGroups, CloseGroups, nksIndexedFunctions, warnicon, fnEssayGroup, fnessay
    },
  BuildLog["** Entering ConvertNotebook - Symbols **"];

  type = "EntityType" /. info;
  pacletName = "PacletName" /. info;
  uri = "URI" /. info;
  lang = "Language" /. {opts} /. Options[MakeNotebook];
  expr = nb;
  	
  (* handle special Device notebook styles *)
  expr = If[MemberQ[{"Device Connection"}, type],
  	DeleteCases[expr,
  		Cell["General statement of device compatibility/support..."|
  			 "Listing of some devices in this class..."|"Additional notes ..."|
  			 "These examples were run on XXXX with XXXX.", ___, "Notes"|"DeviceNotes"|"DeviceExamplePrologue", ___],
  	Infinity], expr];
  expr = If[MemberQ[{"Device Connection"}, type],expr/.Cell[a__,"Notes",b___]:>Cell[a,"DeviceNotes",b], expr];
  expr = If[MemberQ[{"Device Connection"}, type],expr/.Cell[a__, "Usage", b___]:>Cell[a,"DeviceDoubleUsage",b], expr];
  expr = If[MemberQ[{"Device Connection"}, type], 
  	ReplaceAll[
  		DeleteCases[expr, Cell[c___, "AlternateThumbnail", ___] /; 
  			!FreeQ[{c}, s_String /; !StringFreeQ[s, "Thumbnail Placeholder"]], Infinity], 
  		Cell[s___, "AlternateThumbnail", r___] :> Cell[s, "DeviceImage", r]], expr];
  expr = If[MemberQ[{"Device Connection"}, type], 
  	If[FreeQ[expr, Cell[___, "DeviceImage"|"AlternateThumbnail", ___], Infinity], 
  		DeleteCases[expr, Cell[___, "DeviceImageCaption", ___], Infinity], expr], expr];
  (* Clean up "DeviceUsage" cell (remove lines from template with XXXX) *)
  expr = If[MemberQ[{"Device Connection"}, type], ReplaceAll[expr, cell:Cell[___, "DeviceUsage", ___] :> 
  	ReplaceAll[cell, s_String /; !StringFreeQ[s, "XXXX"] :>
  	  StringJoin[Riffle[Select[StringSplit[s, "\n"], StringFreeQ[#, "XXXX"]& ], "\n"]]]], expr];
  
  (* In Service pages, convert "InlineGuideFunction" to "InlineFunctionSans" *)	  
  expr = If[MemberQ[{"Service Connection"}, type], 
  	ReplaceAll[expr, Cell[BoxData[c_] | TextData[c_], some___, "InlineGuideFunction", rest___] :> 
  	 Cell[TextData[c], some, "InlineFunctionSans", rest]], 
  	expr];
  (* Again, in service pages, convert "Notes" cells to "ServiceNotes" -- which drops the dingbat
     and aligns the text *)
  expr = If[MemberQ[{"Service Connection"}, type], 
  	ReplaceAll[expr, Cell[some___, "Notes", rest___] :> 
  	 Cell[some, "ServiceNotes", rest]], 
  	expr];
  (* Another for service pages: convert xxx . xxx InlineGuideFunctionListings to something better *)
  expr = If[MemberQ[{"Service Connection"}, type], 
  	ReplaceAll[expr, Cell[c_String /; StringMatchQ[c, ___~~" . "~~___], 
  		some___, "InlineGuideFunctionListing", rest___] :> 
  	 Cell[TextData[Riffle[(StyleBox[#, "InlineFunctionSans"]) & /@ StringSplit[c, " . "], 
  	 	StyleBox[" \[FilledVerySmallSquare] ", "InlineSeparator"]]], some, 
  	 	"InlineGuideFunctionListing", "InlineGuideFunctionListingGray", rest]], 
  	expr];
 
  (* Remove ILD-specific cells -- commented out section types that will stay in the built nb*)
  expr = Module[{ildstyles = 
  	  {"ILDBanner", (*"AlternativeNames", "DesignJustification", *)
  	   "InternalLinks", "DesignChoicesNumbered", "DesignChoices"}}, 
  	DeleteCases[expr, Cell[___, Alternatives@@ildstyles, ____]]
  ];
  	 
  (* Gather then remove Function Essay, to be re-added later *)
  fnessay = Flatten[Cases[expr, 
    	Cell[CellGroupData[{Cell[__, "FunctionEssaySection", ___], ___, 
    		cell : (Cell[__, "FunctionEssay", ___] ..), 
    		Cell[c__ /; FreeQ[{c}, "FunctionEssay"]] ...}, ___]] :> {cell}, Infinity]];
  expr = DeleteCases[expr, Cell[___, "FunctionEssay", ___], Infinity];
  	  
  (* For non-symbol ref pages, add Unfinished Draft banner *)
  expr = OtherRefAddUnfinishedDraftBanner[expr, info, type, opts];

  (* 'AwaitingReviewNote' *)
  awaitingReviewNote = Cases[expr, Cell[_, "AwaitingReviewNote", ___], Infinity];
  awaitingReviewNote = DeleteCases[awaitingReviewNote, 
    ButtonBox[__,
      ButtonData->"paclet:tutorial/ObjectsInMathematicaAwaitingFutureDesignReview", ___], Infinity];
  expr = DeleteCases[expr, Cell[_, "AwaitingReviewNote", ___], Infinity];

  obsolescenceNote = Cases[expr, Cell[_, "ObsolescenceNote", ___], Infinity];
  expr = DeleteCases[expr, Cell[_, "ObsolescenceNote", ___], Infinity];
  
  (* V9 EDIT BELOW *)
  headerCells = Cases[expr, Cell[_, "SymbolColorBar" | "LinkTrail" | "AnchorBarGrid", ___], Infinity];
  expr = DeleteCases[expr, Cell[_, "SymbolColorBar" | "LinkTrail" | "AnchorBarGrid", ___], Infinity];
  
  (* save ObjectName cell *)
  objectNameCell = Cases[expr, Cell[cont_, "ObjectName"|"ObjectNameSmall"|"ObjectNameAlt",___], Infinity, 1];
  objectName = Cases[objectNameCell, Cell[cont_String, "ObjectName"|"ObjectNameSmall"|"ObjectNameAlt",___]:>cont, Infinity];
  If[Length[objectName] == 0,
    objectName = Cases[objectNameCell,
      Cell[cont:TextData[{ ___}, ___], "ObjectName"|"ObjectNameSmall"|"ObjectNameAlt",___]
        :> Transmogrify`ConvertToString[cont],
      Infinity]];
  objectName = ToString[ If[Length@objectName > 0, First@objectName, " "] ];
  objectName = StringReplace[objectName, RegularExpression[" .*"] -> ""];

  characterNameCell = Cases[expr, Cell[cont_, "CharacterName",___], Infinity, 1];
  characterName = Cases[characterNameCell, Cell[cont_String, "CharacterName",___]:>cont, Infinity];
  characterName = ToString[ If[Length@characterName > 0, First@characterName, " "] ];

  chargridOpts = Sequence[
    GridBoxBackground->{"Columns" -> {{
       GrayLevel[1]}}},
    GridBoxDividers->{"Columns" -> {{Directive[AbsoluteThickness[1], GrayLevel[0.596078]]}}, 
       "Rows" -> {{Directive[AbsoluteThickness[1], GrayLevel[0.596078]]}}},
    GridBoxSpacings->{"Columns" -> {{0.4}}, "Rows" -> {{0.2}}}];
  Quiet[characterImageCell = First[Cases[expr, Cell[cont_, "CharacterImage", rest___]:>
    {First[Cases[cont, GridBox[l_,___]:>l, Infinity]], {rest}}, Infinity, 1]]];
  If[Head[characterImageCell] =!= First,
  characterImageCell = Cell[BoxData[GridBox[characterImageCell[[1]], chargridOpts]], Sequence@@characterImageCell[[2]]];
  characterImage = Cases[characterImageCell, Cell[cont_String, "CharacterImage",___]:>cont, Infinity];
  characterImage = ToString[ If[Length@characterImage > 0, First@characterImage, " "] ]];

  expr = If[MemberQ[{"Indicator"}, type],expr/.Cell[a__,"Usage",b___]:>Cell[a,"IndicatorUsage",b], expr];

  (* save Usage cell, delete and add it back later *)
  usageNameCell = Cases[expr, Cell[cont_, "Usage",___], Infinity, 1];
  If[Length[usageNameCell] > 0, usageNameCell = First@usageNameCell];

  underDevelopmentCell = Cases[expr, Cell["U N D E R   D E V E L O P M E N T",___], Infinity, 1];
  If[Length[underDevelopmentCell] > 0, underDevelopmentCell = First@underDevelopmentCell];

  (* Add NKS Online links *)
  nksIndexedFunctions = 
{"AiryAi", "And", "Apply", "ArcCos", "ArcSin", "ArcTan", "ArithmeticGeometricMean", "AspectRatio", "BernoulliB", "BesselI", "BesselJ", "BesselY", "Binomial", "BitAnd", "BitOr", "BitXor", "Block", "CarmichaelLambda", "Catalan", "Catch", "Ceiling", "CellularAutomaton", "ChebyshevT", "CoefficientList", "Collect", "Compile", "Complement", "ContinuedFraction", "Cos", "Cosh", "Cot", "Coth", "Csc", "Cuboid", "Cyclotomic", "DedekindEta", "Delete", "DeleteCases", "Depth", "Derivative", "Det", "DigitCount", "Distribute", "Divide", "Divisors", "DivisorSigma", "Do", "Dot", "Drop", "Eigenvalues", "Element", "EllipticK", "EllipticTheta", "Equal", "Erf", "EulerGamma", "EulerPhi", "Exists", "Exp", "Expand", "ExpIntegralEi", "Export", "ExtendedGCD", "Factor", "Factorial", "Factorial2", "FactorInteger", "False", "Fibonacci", "FindMinimum", "FindRoot", "Fit", "FixedPoint", "FixedPointList", "Flat", "Flatten", "Floor", "Fold", "FoldList", "For", "ForAll", "Fourier", "FractionalPart", "FresnelC", "FresnelS", "FromDigits", "FullForm", "FullSimplify", "Function", "FunctionExpand", "Gamma", "GCD", "GegenbauerC", "GoldenRatio", "Graphics3D", "GroebnerBasis", "HermiteH", "Hypergeometric1F1", "Hypergeometric2F1", "HypergeometricPFQ", "If", "Implies", "Inner", "InputForm", "Insert", "IntegerDigits", "IntegerExponent", "IntegerQ", "Integrate", "Intersection", "InverseFunction", "JacobiP", "JacobiSN", "JacobiSymbol", "Join", "Khinchin", "KleinInvariantJ", "KroneckerDelta", "LaguerreL", "LatticeReduce", "LCM", "LeafCount", "LegendreP", "Length", "Less", "LinearSolve", "Listable", "ListConvolve", "ListCorrelate", "ListPlay", "Log", "LogIntegral", "Map", "MapAt", "MapIndexed", "MatchQ", "MathieuC", "MatrixPower", "Max", "MeijerG", "MemberQ", "Mod", "ModularLambda", "MoebiusMu", "Multinomial", "MultiplicativeOrder", "Nand", "NDSolve", "Nest", "NestList", "NestWhile", "NestWhileList", "NIntegrate", "Nor", "Not", "NSolve", "NullSpace", "OddQ", "Or", "Orderless", "Outer", "PadLeft", "Partition", "PartitionsP", "Pi", "Play", "Plot3D", "Plus", "PolyGamma", "PolyLog", "PolynomialMod", "PolynomialReduce", "PolynomialRemainder", "Position", "Power", "PowerMod", "Prepend", "PrimePi", "PrimeQ", "Product", "ProductLog", "PseudoInverse", "Quotient", "Random", "Range", "Raster", "RealDigits", "RepeatedNull", "ReplaceAll", "ReplaceList", "ReplacePart", "ReplaceRepeated", "Rest", "Resultant", "RiemannSiegelZ", "Root", "RotateLeft", "Rule", "Sec", "Sech", "SeedRandom", "Select", "Series", "Simplify", "Sin", "Sinh", "SinIntegral", "SixJSymbol", "Slot", "Sort", "SphericalHarmonicY", "Split", "Sqrt", "StandardForm", "StieltjesGamma", "StirlingS1", "StirlingS2", "StringJoin", "StringLength", "StringPosition", "StringReplace", "StringReplacePart", "SubsuperscriptBox", "Sum", "Table", "Take", "Tan", "Tanh", "TensorRank", "Through", "Throw", "TimeConstraint", "Times", "ToExpression", "ToString", "Transpose", "TrigFactor", "True", "TrueQ", "Union", "WeierstrassP", "While", "Xor", "Zeta", "$IterationLimit"};

  If[MemberQ[{"Symbol", "Menu Item"},type], 
  BuildLog["-- Add nksonline links"];
  If[MemberQ[nksIndexedFunctions, objectName], 
    Module[{rcell, pos, rlink},
      rcell = Cell[TextData[{
        ButtonBox["NKS|Online",
          BaseStyle->"Hyperlink",
          ButtonData->{URL["http://www.wolframscience.com/nksonline/index/search.cgi?SearchIndex="<>objectName], None}],
        " (",
        StyleBox[
          ButtonBox["A New Kind of Science",
            BaseStyle->"Hyperlink",
            ButtonData->{URL["http://www.wolframscience.com/nksonline/"], None}],
          FontSlant->"Italic"],
        ")"
      }], "RelatedLinks"];
      expr = EnsureCellGroupData[expr, "RelatedLinksSection", lang];
      pos = Position[expr,
        Cell[CellGroupData[{Cell[_, "RelatedLinksSection", ___], ___}, ___], ___],
        Infinity];
      pos = First[pos];
      rlinks = Extract[expr, pos];
      rlinks[[1, 1]] = Append[rlinks[[1, 1]], rcell];
      expr = ReplacePart[expr, pos->rlinks];
  ]]   ];

  (* Add Demonstration links *)
  If[MemberQ[{"Symbol"},type],
  If[$DocumentationExportDebug, BuildLog["-- Add demonstration site links"]];
  If[DocumentationBuild`Info`GetSourceInformation["paclet:" <> uri, lang, "Demonstrations"] === True,
    Module[{pos, rlinks, rcell},
      (* FIXME: get right text, localize *)
      rcell = Cell[TextData[Join[
        {Localized["DemonstrationsWith", lang][objectName]},
        Localized["WolframDemonstrationsProject", lang]
      ]], "RelatedLinks"];
      pos = Position[expr,
        Cell[CellGroupData[{Cell[_, "RelatedLinksSection", ___], ___}, ___], ___],
        Infinity];
      If[Length[pos] != 0,
        pos = First[pos];
        rlinks = Extract[expr, pos];
        rlinks[[1, 1]] = Insert[rlinks[[1, 1]], rcell, 2];
        expr = ReplacePart[expr, pos->rlinks];
        , (* Else *)
        expr = Insert[expr,
          Cell[CellGroupData[{
            Cell[StringReplace[Localized["RelatedLinksSectionText", lang], {" \[RightGuillemet]" | "\[RightGuillemet]"->""}], "RelatedLinksSection"],
            rcell
          }, Open]],
          First[Position[expr,
            Cell[CellGroupData[{Cell[_, "SeeAlsoSection", ___], ___}, ___], ___],
            Infinity]] ];
      ];
  ]]   ];
  

  BuildLog["-- Rearrange Notes and Examples"];
  (* Move Examples groups directly below Notes per sw *)
  notesGroup = GetNotesGroup[expr, type, lang, opts, "HasOptions" -> DocumentationBuild`Info`GetSourceInformation[uri, lang, "HasOptions"]];
  fnEssayGroup = Cell[CellGroupData[{Cell[Localized["GuideBackgroundSectionText", lang], "FunctionEssaySection"], Sequence @@ Flatten[{fnessay}]}, Open]];
  seeAlsoGroup = GetSeeAlsoGroup[expr, lang];
  examplesGroup = GetExamplesGroup[expr, lang];
  moreAboutGroup = GetMoreAboutGroup[expr, lang];
  relatedLinksGroup = GetRelatedLinksGroup[expr, lang];
  tutorialsGroup = GetTutorialsGroup[expr, lang];
  elucidationGroup = Cases[expr, Cell[_,"Elucidation",___], Infinity];
  indicatorExampleGroup = GetIndicatorExampleGroup[expr, lang];
  indicatorDescriptionGroup = GetIndicatorDescriptionGroup[expr, lang];
  indicatorFormulaGroup = GetIndicatorFormulaGroup[expr, lang];
  tutorialCollectionGroup = GetTutorialCollectionGroup[expr, lang];
  
  importExportGroup = If[MemberQ[{"Format"}, type], GetImportExportGroup[expr, lang], {}];
  elementGroup = If[MemberQ[{"Format"}, type], GetElementGroup[expr, lang], {}]; 
  notebookInterfaceGroup = If[MemberQ[{"Format"}, type], GetNotebookInterfaceGroup[expr, lang], {}];
  optionsGroup = If[MemberQ[{"Format"}, type], GetOptionsGroup[expr, lang], {}];
  otherTypeGroup = If[MemberQ[{"Service Connection", 
  	"Device Connection", "Interpreter", "Embedding Format"}, type], 
  	GetOtherSectionGroup[expr, lang], {}];

(* CHARACTER PAGE: make image / notes grid *)
Quiet[characterImageGrid = DocumentationBuild`Utils`MakeCharacterNotesGrid[characterImageCell, notesGroup]];

expr = DeleteCases[expr, (Cell[CellGroupData[{#, ___}, ___], ___] | #)&[
        Cell[__,
          "GuideTutorialCollection" | "GuideTutorialCollectionSection" | "TutorialCollectionSection" | "TutorialCollection",
        ___]
      ], Infinity];
      
  BuildLog["-- Delete various cells"];
  (* Delete cell groups, to be explicitly assembled later *)
  expr = DeleteCases[expr, Alternatives[
    (* Remove unwanted groups *)
    (Cell[CellGroupData[{#, ___}, ___], ___] | #)&[
      Cell[_, 
        "ObjectName" | "ObjectNameSmall" | "ObjectNameAlt" | "CharacterName" | "CharacterImage" | "PrimaryExamplesSection" | "ExtendedExamplesSection" | "SeeAlsoSection" | 
        "ImportExportSection" | "ElementsSection" | "NotebookInterfaceSection" | "OptionsSection" | "TutorialsSection" | "MoreAboutSection" |
        "MoreToExploreSection" | "MoreInformationSection" | "RelatedLinksSection" | "IndicatorExampleSection"| "IndicatorDescriptionSection"| "IndicatorFormulaSection" | 
        "ServiceSubsection" | "DeviceSubsection" | "InterpreterSection" | "EmbeddingFormatSection",
        ___]],
    Cell[_, "PrimaryExamplesSection", ___],
    Cell["U N D E R   D E V E L O P M E N T",___],
    Cell[_, "SeeAlsoSection", ___],
    Cell[_, "Elucidation", ___]
  ], Infinity];
  
  BuildLog["-- Remove SeeAlso links to flagged notebooks"];
  Scan[
    Function[{pos}, Module[{cell, origcells, newcells},
      cell = Extract[seeAlsoGroup, pos];
      origcells = Cases[cell,
        Cell[BoxData[_ButtonBox, ___], ___]
          | Cell[BoxData[StyleBox[_ButtonBox, ___], ___], ___],
        Infinity];
      (* TODO-andy If ButtonData exists, it has to be used! *)
      If[buildmode==="Future",
      	newcells = origcells,
      newcells = Select[origcells,
        Function[{bbcell}, Module[{sym},
          If[MatchQ[bbcell,
                Cell[___, "FutureExample" | "TentativeExample", ___]
                  | Cell[BoxData[StyleBox[___, "FutureExample" | "TentativeExample", ___], ___], ___]],
              False,
            (* Else *)
            sym = Cases[bbcell, _ButtonBox, Infinity, 1];
            uri = Cases[sym, ButtonData~(Rule|RuleDelayed)~uri_String :> uri, Infinity, 1];
            If[ (Length @ uri > 0) && StringMatchQ[First@uri, "paclet:*"],
               (* If Button contains ButtonData, use it *)
               ShippingFunctionQ[First@uri, pacletName],
               (* no ButtonData so need to create it *)
               sym = Transmogrify`ConvertToString[sym[[1, 1]]];
               sym = StringReplace[sym, "\""->""];
               sym = "paclet:"<>If[ FreeQ[pacletName, "Mathematica" | ""], pacletName<>"/", ""]<>"ref/" <> sym;
               ShippingFunctionQ[sym, pacletName]
            ]
      ]]]]];
      If[Length[origcells] =!= Length[newcells],
        seeAlsoGroup = ReplacePart[seeAlsoGroup, pos -> Join[
          Cell[TextData[
            Flatten[Riffle[newcells, {{
              "\[NonBreakingSpace]",
              StyleBox["\[MediumSpace]\[FilledVerySmallSquare]\[MediumSpace]", "InlineSeparator"],
              " "
            }}, 2 ]]
          ]],
          Rest[cell]
      ]]];
    ]],
    Position[seeAlsoGroup, Cell[_TextData, "SeeAlso", ___], Infinity]
  ];

  (* Add [[Not Ready Yet]] for those symbols that have no examples *)
  (* Removing this per Andre -- 1/22/14 *)
  (*examplesGroup = 
   If[FreeQ[examplesGroup, Cell[___, "Input", ___], Infinity],
     Insert[examplesGroup, Cell["[[ Not Ready Yet ]]", "ExamplesUnderConstruction"], MapAt[(#+1)&, 
       Position[examplesGroup, Cell[___, "PrimaryExamplesSection", ___]],{1,-1}]],
     examplesGroup];*)
  

  (* Do example counts *)
  Module[{poss, repl, cexpr, cpos, excnt, totalcnt},
    poss = Position[examplesGroup,
      CellGroupData[{Cell[_, "ExampleSection" | "ExampleSubsection" | "ExampleSubsubsection", ___], ___}, ___],
      Infinity];
    excnt = Function[{exex},
      Apply[Plus, Flatten[{
        (* We don't always add one, because that gives us an off-by-one
           when the section only has subsections.  But we do want to add
           one if there's an actual example directly in the section.
           Note that the {2} level specification excludes anything after
           a delimiter, because ExampleDelimiter groups. *)
        If[Count[exex,
          Cell[_, "ExampleText" | "Input" | "Code" | "Program", ___]
            | Cell[CellGroupData[{Cell[_, "ExampleText" | "Input" | "Code" | "Program", ___], ___}, ___], ___],
          {2}] > 0, 1, 0],
        (* The number of ExampleDelimiters *)
        Count[exex,
          Cell[CellGroupData[{Cell[_, "ExampleDelimiter", ___], ___}, ___], ___],
          {2}],
        (* The total count of the contained subsections *)
        Map[excnt,
          Cases[exex, CellGroupData[{Cell[_, "ExampleSubsection" | "ExampleSubsubsection", ___], ___}, ___], {3}]]
    }]]];
    repl = Map[Function[{pos},
      cexpr = Extract[examplesGroup, Join[pos, {1, 1}]];
      (* Position is post-order depth-first, and we want the
         inner-most if there's an InterpretationBox going on. *)
      cpos = First[Position[cexpr, Cell[_, cexpr[[2]], ___], {0, Infinity}]];
      cpos = Join[pos, {1, 1}, cpos, {1}];
      cpos -> TextData[Join[
        If[Head[#] === TextData, First[#], {#}]&[Extract[examplesGroup, cpos]],
        {
          "\[NonBreakingSpace]\[NonBreakingSpace]",
          Cell["(" <>
            ToString[excnt[First[Extract[examplesGroup, {pos}]]]]
            <> ")", "ExampleCount"]
      }]]
    ], poss];
  examplesGroup = ReplacePart[examplesGroup, repl];
  totalcnt = Total[Flatten@Cases[examplesGroup,
    (Cell[c___,"ExampleSection",___] /; FreeQ[{c}, InterpretationBox, Infinity]):>
     Cases[{c},Cell[n_String,___,"ExampleCount",___]:>ToExpression[n],
    Infinity],Infinity]];
  examplesGroup = ReplaceAll[examplesGroup, 
    Cell[s_String, some___, "PrimaryExamplesSection", rest___]:>
    Cell[TextData[{Cell[BoxData[ToBoxes[Spacer[6]]]], 
      s, "\[NonBreakingSpace]\[NonBreakingSpace]", Cell["("<>ToString[totalcnt]<>")", "ExampleCount"]}], some, "PrimaryExamplesSection", rest]];
  ];

  (* ExampleSection cells are a mess.  It's high time they're cleaned up... *)  
  examplesGroup = Quiet[ReplaceRepeated[examplesGroup,
    Cell[BoxData[InterpretationBox[cell_, ___]], ((sty:("ExampleSection"|"ExampleSubsection"|"ExampleSubsubsection"))..), rest___] :>
      Cell[cell[[1]], sty, rest]]];
      
  (* Examples that require the "ExampleFreeze" DocuTools style may have "outputs" that are not in the
     "Output" style, but "PrintTemporary" instead.  These are to become "Output". See ref/FormFunction
     for examples of such examples *)
  examplesGroup = ReplaceRepeated[examplesGroup, 
  	Cell[c___, "PrintTemporary", more___, "ExampleFreeze", rest___] :>
  	Cell[c, "Output", more, "ExampleFreeze", rest, CellLabel->""]];

  BuildLog["-- Change SeeAlso Links"];
  seeAlsoGroup = If[MemberQ[{"Format","Character Name"}, type], seeAlsoGroup, 
  	seeAlsoGroup /. ButtonBox[c___] :> StyleBox[ButtonBox[c], FontFamily->"Verdana"]];
  	
  (* Take out interpreter links from See Alsos, put in own section; build that section from scratch *)
  If[MemberQ[{"Interpreter"}, type], relatedInterpretersGroup = Cell[CellGroupData[{
    Cell[Localized["RelatedInterpretersSectionText", lang], "RelatedInterpreterSection"],
    Cell[TextData[Flatten[
       Riffle[Cases[seeAlsoGroup, Cell[c_, ___, "InlineFormula", ___] /; 
       MemberQ[c, s_String/;StringMatchQ[s,"paclet:ref/interpreter/*"], Infinity], Infinity], 
          {{"\[NonBreakingSpace]",StyleBox["\[MediumSpace]\[FilledVerySmallSquare]\[MediumSpace]","InlineSeparator"]," "}}]]], "RelatedInterpreters"]
    }, Open]];
    seeAlsoGroup = DeleteCases[seeAlsoGroup, Cell[c_, ___, "InlineFormula", ___] /; 
       MemberQ[c, s_String/;StringMatchQ[s,"paclet:ref/interpreter/*"], Infinity], Infinity];
    seeAlsoGroup = ReplaceRepeated[seeAlsoGroup, {
      {first___, " \[EmptyVerySmallSquare] ", " \[EmptyVerySmallSquare] ", last___} :> {first, " \[EmptyVerySmallSquare] ", last},
      {" \[EmptyVerySmallSquare] ", last__} :> {last},
      {first__, " \[EmptyVerySmallSquare] "} :> {first}
      }],
    relatedInterpretersGroup = {}];

  (* Arrange the rearranged notebook *)
  (*expr = MapAt[Flatten@Insert[#, 
    {objectNameCell, usageNameCell, notesGroup, examplesGroup, seeAlsoGroup}, 1
    ] &, expr, 1];*)
  tutorialCollection=True;
  
  (* Add "(interpretation type)" to the top of ref/interpreter/* pages *)
  If[MemberQ[{"Interpreter"}, type],
   If[MemberQ[underDevelopmentCell, "UnderDevelopment", Infinity],
    With[{cell = underDevelopmentCell},
      objectNameCell = Append[ReplaceAll[objectNameCell, objectName -> 
      TextData[{objectName, " ", Cell[Localized["InterpreterInterpretationType", lang], "InterpreterSubtitle"]}]], 
      ReplaceAll[cell, "UnderDevelopment" -> "DeviceAbstract"]]],
    objectNameCell = ReplaceAll[objectNameCell, objectName -> 
      TextData[{objectName, " ", Cell[Localized["InterpreterInterpretationType", lang], "InterpreterSubtitle"]}]]]
  ];
  
  (* Transform usage cell in Format pages to new inline-bulleted style *)
  If[MemberQ[{"Format"}, type],
    Module[{usagetext, sep = StyleBox["\[FilledSmallSquare] ","FormatUsageSeparator"], col1, col2}, usagetext = Cases[usageNameCell, Cell[TextData[c_], ___] :> c, Infinity];
      If[MatchQ[usagetext, {}], usagetext = Cases[{usageNameCell}, Cell[s_String, ___] :> s, Infinity]];
      usagetext = ReplaceRepeated[usagetext, {some___, first___String, StyleBox[s_, "RebrandingTerm", ___], second___String, Shortest[rest___]} :> 
        {some, StringJoin[first, s, second], rest}];
      usagetext = ReplaceRepeated[usagetext,StyleBox["\n",__]:>"\n"];
      usagetext = Module[{brokentext = Append[Prepend[
        Flatten@ReplaceAll[usagetext, 
          s_String /; !StringFreeQ[s,"\n"] :> StringSplit[s, "\n" -> StyleBox["BREAK"]]], StyleBox["BREAK"]], StyleBox["BREAK"]]},
        DeleteCases[ReplaceRepeated[brokentext, 
         {Shortest[some___],StyleBox["BREAK"],Shortest[cont__] /; FreeQ[{cont},StyleBox["BREAK",Infinity]], 
            StyleBox["BREAK"], Shortest[rest___]}:>{some, RowBox[{cont}], StyleBox["BREAK"], rest}], StyleBox["BREAK"], Infinity]];
      usagetext = DeleteCases[usagetext, " "|""];
      usagetext = Multicolumn[usagetext, 2][[1]];
      {col1, col2} = Transpose[Map[Cell[TextData[If[MatchQ[#,""|" "|Null], #, Flatten[{sep, #}]]], "FormatUsage", ParagraphIndent -> -10]&, usagetext, {2}]];
      col1 = List /@ col1; col2 = List /@ col2;
      If[!MatchQ[Flatten[DeleteCases[usagetext, Null|"Null", Infinity]], {} | {""}], usageNameCell = Cell[CellGroupData[{Cell[Localized["GuideBackgroundSectionText", lang], "FormatBackground"], 
        Cell[BoxData[GridBox[{{
            GridBox[col1,GridBoxAlignment->{"Columns"->{{Left}},"Rows"->{{Top}}},
              GridBoxItemSize->{"Columns"->{Scaled[1.]}},
              GridBoxSpacings->{"Rows"->{{1.2}}}],
            GridBox[col2,GridBoxAlignment->{"Columns"->{{Left}},"Rows"->{{Top}}},
              GridBoxItemSize->{"Columns"->{Scaled[1.]}},
              GridBoxSpacings->{"Rows"->{{1.2}}}]}},
          GridBoxAlignment->{"Columns"->{{Left}},"Rows"->{{Top}}},
          GridBoxItemSize->{"Columns"->{Scaled[0.5],{Scaled[0.5]}}},
          GridBoxSpacings->{"Columns"->{{1.3}}}]], "FormatUsage",
          CellMargins->{{24,24},{24,18}}]}, Closed]],
        usageNameCell = Cell[Localized["GuideBackgroundSectionText", lang], "FormatBackground"]];
    ]
  ];
    
  nbOrder = Which[
  	type==="Format",
  	{underDevelopmentCell, objectNameCell,notesGroup,usageNameCell,importExportGroup,notebookInterfaceGroup,elementGroup,optionsGroup,
  		examplesGroup, seeAlsoGroup, tutorialsGroup, moreAboutGroup, relatedLinksGroup},
  	  	type==="Indicator",
  	{underDevelopmentCell, objectNameCell, usageNameCell, elucidationGroup, notesGroup, indicatorExampleGroup, indicatorDescriptionGroup, indicatorFormulaGroup, seeAlsoGroup, 
  		tutorialsGroup, If[tutorialCollection===True,tutorialCollectionGroup,""], moreAboutGroup, relatedLinksGroup},
  		type==="Character Name", 
  			{underDevelopmentCell, characterNameCell, characterImageGrid,importExportGroup,notebookInterfaceGroup,elementGroup,optionsGroup,
  		seeAlsoGroup, tutorialsGroup, moreAboutGroup, examplesGroup, relatedLinksGroup},
  True, 
  	{underDevelopmentCell, objectNameCell, usageNameCell, elucidationGroup, notesGroup, otherTypeGroup, fnEssayGroup, examplesGroup, relatedInterpretersGroup, seeAlsoGroup,
  		tutorialsGroup, If[tutorialCollection===True,tutorialCollectionGroup,""], moreAboutGroup, relatedLinksGroup}];
  	
  expr = 
  	MapAt[Flatten@Insert[#, nbOrder, 1] &, expr, 1];
  
  (* add special Format/Interpreter notebook styles *)
  expr = If[MemberQ[{"Format"}, type],expr/.Cell[a__,"Usage",b___]:>Cell[a,"FormatUsage",b], expr];
  expr = If[MemberQ[{"Interpreter"}, type],expr/.Cell[a__,"Usage",b___]:>Cell[a,"InterpreterUsage",b], expr];
  expr = If[MemberQ[{"Format"}, type],expr/.Cell[a__,"Notes",b___]:>Cell[a,"FormatNotes",b], expr];
  expr = If[MemberQ[{"Interpreter"}, type],expr/.Cell[a__,"Notes",b___]:>Cell[a,"InterpreterNotes",b], expr];
  

  (* Remove SyntaxTemplateExclusion *)
  expr = expr //. {
    StyleBox[foo_, "SyntaxTemplateExclusion", ___] :> foo,
    Cell[foo_, "InlineFormula", "TemplateExclusion", o___] :> Cell[foo, "InlineFormula", o]
  };

  (* Add ObsolescenceNote *)
  warnicon = ToBoxes[Rasterize[
    Import[FileNameJoin[{$DocumentationBuildDirectory, "Internal", "online", "Images", "ObsoleteIcon.png"}], "PNG"], Background->GrayLevel[0.941176]]];
  obsolescenceNote = If[obsolescenceNote === {}, obsolescenceNote, {Cell[BoxData[GridBox[{{Cell[BoxData[warnicon]], Cell[obsolescenceNote[[1, 1]], FontFamily->"Verdana"]}}]], Sequence@@Rest[obsolescenceNote[[1]]]]}];
  expr = expr /. 
    Cell[c_, sty_String/;StringMatchQ[sty, "ObjectName"|"ObjectNameSmall"|"ObjectNameAlt"], o___] :>
      Cell[CellGroupData[Flatten[{
        Cell[c, sty, o],
        obsolescenceNote
      }, 1], Open]];

  (* Add Awaiting Review Note *)
  awaitingReviewNote = If[awaitingReviewNote === {}, awaitingReviewNote, {Cell[BoxData[GridBox[{{Cell[BoxData[warnicon]], Cell[awaitingReviewNote[[1, 1]], FontFamily->"Verdana"]}}]], Sequence@@Rest[awaitingReviewNote[[1]]]]}];
  expr = expr /. 
    Cell[c_, sty_String /; StringMatchQ[sty, "ObjectName"|"ObjectNameSmall"|"ObjectNameAlt"], o___] :> 
      Cell[CellGroupData[Flatten[{
        Cell[c, sty, o],
        awaitingReviewNote
      }], Open]];

  If[(*(("PacletName" /. info) === "Mathematica")
      &&*) (Length@Cases[expr, Cell[c_, ___, "ObjectName"|"ObjectNameSmall"|"ObjectNameAlt", ___]:>c, Infinity] > 0),
    BuildLog["-- Add ObjectName Graphic"];\

    expr = AddObjectNameGraphic[expr, info, opts];
  ];

  BuildLog["-- Close groups"];

  (*  Close groups and set Dynamic *)
  (* 4/26/14: Sadly, we have to iterate through closedGroups individually, because it seems that
     since these are nested styles, the copies used to replace are being made before the deeper 
     replacements occur, so only the highest-level style -- ExampleSection -- is being Closed *)
  
  closedGroups = Which[MemberQ[{"Format", "Interpreter", "Service Connection", "Device Connection", "Embedding Format"}, type], 
  	"ExampleSection"|"ExampleSubsection"|"ExampleSubsubsection"|"ImportExportSection"|"NotebookInterfaceSection"|"OptionsSection"|
  	"InterpreterSection"|"EmbeddingFormatSection"|"ServiceSubsection"|"DeviceSubsection"(*|"TutorialsSection"*), 
  True, "ExampleSection"|"ExampleSubsection"|"ExampleSubsubsection"];
  
  CloseGroups[nbexpr_Notebook, style_String] := 
    nbexpr /. {
    Cell[CellGroupData[{
      Cell[con_, sty_String /; StringMatchQ[sty, style], o___],
      (*Cell[con_, sty_String /; StringMatchQ[sty, "ExampleSection"|"ExampleSubsection"], o___],*)
        cs___
      }, ___], g___] 
    :> Cell[CellGroupData[{
      Cell[con, sty, o],
        cs
      }, If[(Length[Cases[con, Localized["BasicExamplesSectionText", lang], Infinity]] > 0), Open, Closed]], g],
    Cell[CellGroupData[{
      Cell[con_, "NotesSection", o___],
        cs___
      }, ___], g___]
    :> Cell[CellGroupData[{
      Cell[con, "NotesSection", o],
        cs
      }, (* Details section to be open by default on Message pages: brianv--05/12/16*) 
      If[MatchQ[type, "Message"], Open, Closed]], 
    g]
  };
  
  expr = Fold[CloseGroups, expr, closedGroups];

  (* make sure Input and delimiter groups are open *)
  expr = expr /. 
  	Cell[CellGroupData[{
	  Cell[a__, sty_String /; StringMatchQ[sty, "Input"],b___], c___}, Closed],e___] :>
  	Cell[CellGroupData[{
	  Cell[a,sty,b], c}, Open], e];
  expr = expr /. 
  	Cell[CellGroupData[{
	  Cell[a__, sty_String /; StringMatchQ[sty, "ExampleDelimiter"],b___], c___}, Closed],e___] :>
  	Cell[CellGroupData[{
	  Cell[a,sty,b], c}, Open], e];
	  
  (* Delete special conversion cells *)
  expr = expr /. 
  	Cell[CellGroupData[{
	  Cell[a__, sty_String /; StringMatchQ[sty, "ExampleDelimiter"],b___], c___}, Closed],e___] :>
  	Cell[CellGroupData[{
	  Cell[a,sty,b], c}, Open], e];
  expr = DeleteCases[expr, Cell[__, "SuppressedMaterial", ___], Infinity];

  (*  Change headings to have Open/close buttons *)
  expr = expr /. Cell[con_,
      sty:Alternatives@@("CapitalizeHeadingStyles" /. {opts} /. Options[MakeNotebook]),
      o___] :>
    Cell[TextData@WrapInOpenCloseButton@CapitalizeString[con], sty, o];  
    
    (*  Special hack for ref/Underlined failure at 100% magnification in Windows. *)
  expr = expr /. Cell[BoxData[StyleBox["\<\"Text\"\>",StripOnInput->False,
  		FontVariations->{"Underline"->True}]], "Output", op___]:>
  Cell[BoxData[StyleBox["\<\"Text\"\>",StripOnInput->False,
  FontVariations->{"Underline"->True}]], "Output", Magnification->1.1, op];
 
  
  (* Add header cells back in *) 
  expr = MapAt[Flatten@Insert[#, {headerCells}, 1] &, expr, 1];

  BuildLog["-- CellChangeTimes still exist? ",
    ToString @ Cases[expr, Cell[___, System`CellChangeTimes~(Rule|RuleDelayed)~__, ___], Infinity] ];

  BuildLog["** Leaving MakeNotebook (Symbol) **"];

  expr 

  ];



(******************************************************************************)
(** ConvertNotebook - Formats **)
(*ConvertNotebook[nb_, info_,
    ("Format"),
    opts___?OptionQ] := 
  Module[{expr},
    expr = nb;
    expr
  ];*)


(******************************************************************************)
(** ConvertNotebook - Guides **)
ConvertNotebook[nb_, info_,
    ("Guide"),
    opts___?OptionQ] := 
  Module[{uri, lang, expr, labelCell, unfindraftimg, langext, obsolescenceNote, warnicon},
    BuildLog["** Entering ConvertNotebook - Guides **"];

    uri = "URI" /. info;
    lang = "Language" /. {opts} /. Options[MakeNotebook];
    expr = nb;
    
    (* Convert Alphabetical listings separately *)
    If[!StringFreeQ[uri,"AlphabeticalListing"], expr = ConvertNotebook[nb, info, "Alphabetical Listing", opts]];

    PlaceTutorialCollectionGroup[notebook_]:= Module[{tcg, pos, nbexpr},	 
                 tcg = GetGuideTutorialCollectionGroup[notebook];	 
                 pos = Position[notebook,	 
             Cell[CellGroupData[{	 
                 Cell[_ , "GuideTutorialsSection", ___],	 
                 ___}, ___], ___] ];	 
             pos = MapAt[# + 1 &, pos, {1, -1}];	 
             nbexpr = DeleteCases[notebook, Cell[CellGroupData[{	 
                 Cell[_ , "GuideTutorialCollectionSection", ___],	 
                 ___}, ___], ___], Infinity];	 
 	 
             Insert[nbexpr, tcg, pos]	 
          ];	 
 	 
    expr = If[IncludeTutorialCollectionGroup===False, expr, 
    If[Cases[expr,Cell[CellGroupData[{	 
                 Cell[_ , "GuideTutorialsSection", ___],	 
                 ___}, ___], ___],Infinity]=!={} &&	 
              Cases[expr,Cell[CellGroupData[{	 
                 Cell[_ , "GuideTutorialCollectionSection", ___],	 
                 ___}, ___], ___],Infinity]=!={},	 
                 PlaceTutorialCollectionGroup[expr], 	 
                 expr]
    ];	 

    (* V9 CHANGE BELOW *)
    (* Insert Reference section header *)
    expr = Module[{pos,
       override = {"guide/WolframPredictiveInterface", "guide/NotebookShortcuts", "guide/CellGroupsAndOutlining"}},
       If[
         (Cases[expr, 
           Cell[___,"InlineGuideFunction"|"InlineGuideFunctionListing",___],Infinity] === {}) && FreeQ[override, uri], expr,
         pos = Position[expr, Cell["", ___, "GuideFunctionsSection", ___]];
         Insert[expr, Cell[Localized["GuideReferenceSectionText", lang], "GuideReferenceSection"], pos]
    ]];

    (* Insert Featured Examples section *)
    (* Commented out to remove per SW: 4/6/15 *)
    (* If re-adding, make sure to reinstate the "ExamplePageLinks" TaggingRule *)
    (*expr = Module[{pos, links = 
         DocumentationBuild`Info`GetSourceInformation[
           uri, lang, "FeaturedExampleLinks"], headercell, next, nextover, prev, prevover, nextdis, prevdis},
       If[MatchQ[links, {} | None | "None"], expr,
       pos = Position[expr, Cell[_, "GuideReferenceSection", ___]];
       {next, nextdis, nextover} = ToBoxes[Import[FileNameJoin[{$DocumentationBuildDirectory, "Internal", "online",
         "Images", #}]]]& /@ {"NextSlideButton.png", "NextSlideButton-disabled.png", "NextSlideButton-over.png"};
       {prev, prevdis, prevover} = ToBoxes[Import[FileNameJoin[{$DocumentationBuildDirectory, "Internal", "online",
         "Images", #}]]]& /@ {"PreviousSlideButton.png", "PreviousSlideButton-disabled.png", 
         "PreviousSlideButton-over.png"};

       headercell = With[{
          prevdis=prevdis,prevover=prevover,prev=prev,
          next=next,nextover=nextover,nextdis=nextdis,
          length=Length[links]},
        If[length > 4, Cell[BoxData[GridBox[{{
        Cell[TextData[{Cell[BoxData[ToBoxes[Spacer[6]]]], Cell[Localized["GuideFeaturedExamplesSectionText", lang], "FeaturedExamplesText"]}]],
        Cell[BoxData[PaneBox[RowBox[{
          ButtonBox[DynamicBox[
            If[CurrentValue[EvaluationNotebook[],{TaggingRules, "ExampleCounter"}]===1, prevdis,
              If[CurrentValue["MouseOver"], 
                TagBox[prevover, MouseAppearanceTag["LinkHand"]], 
                TagBox[prev, MouseAppearanceTag["LinkHand"]]]]], 
            Enabled->Dynamic[CurrentValue[EvaluationNotebook[],{TaggingRules, "ExampleCounter"}] > 1],
            ButtonFunction:>(CurrentValue[EvaluationNotebook[],{TaggingRules, "ExampleCounter"}] = 
              Max[1, CurrentValue[EvaluationNotebook[],{TaggingRules, "ExampleCounter"}] - 1])
          ], "\[NonBreakingSpace]\[NonBreakingSpace]",
          DynamicBox[StyleBox[ToString[CurrentValue[EvaluationNotebook[],{TaggingRules, "ExampleCounter"}]],"ExampleControls"]],
          Cell["\[NonBreakingSpace]\[VerticalSeparator]\[NonBreakingSpace]", "ExamplePositionDivider", FontSize->12],
          StyleBox[ToString[Ceiling[length/4]], "ExampleControls"], "\[NonBreakingSpace]\[NonBreakingSpace]",

          ButtonBox[DynamicBox[
            If[CurrentValue[EvaluationNotebook[],{TaggingRules, "ExampleCounter"}]===Ceiling[length/4], nextdis,
              If[CurrentValue["MouseOver"], 
                TagBox[nextover, MouseAppearanceTag["LinkHand"]], 
                TagBox[next, MouseAppearanceTag["LinkHand"]]]]],
            Enabled->Dynamic[CurrentValue[EvaluationNotebook[],{TaggingRules, "ExampleCounter"}] < Ceiling[length/4]],
            ButtonFunction:>(CurrentValue[EvaluationNotebook[],{TaggingRules, "ExampleCounter"}] = 
              Min[Ceiling[length/4], CurrentValue[EvaluationNotebook[],{TaggingRules, "ExampleCounter"}] + 1])
          ]
        }], Alignment->{Left, Center}]], "ExamplePosition"](*,
         TODO: Add "show all" button for Featured Examples 
        ""*)

       }}]], "GuideFeaturedExamplesSection"],
       Cell[TextData[{Cell[BoxData[ToBoxes[Spacer[6]]]], 
         Cell[Localized["GuideFeaturedExamplesSectionText", lang], "FeaturedExamplesText"]}], "GuideFeaturedExamplesSection", CellFrameMargins->{{20, Inherited}, {6, 6}}]
       ]];

       Insert[expr, Cell[CellGroupData[{
       headercell,
       Cell[BoxData[
          DynamicBox[TemplateBox[{
            CurrentValue[EvaluationNotebook[],{TaggingRules,"ExamplePageLinks"}]},"FeaturedExamplesGrid"]]],
          "FeaturedExamplesGridCell"]
       }]], pos]
    ]];*)

    (* Insert Learning Resources section *)
    (* Commented out to remove Learning Resources per SW: 4/6/15 *)
    (*expr = If[DocumentationBuild`Utils`GetGuideTutorialsSectionList[expr]==={} && 
              DocumentationBuild`Utils`GetGuideHowTosList[expr]==={} &&
              DocumentationBuild`Info`GetSourceInformation[uri, lang, "WebResourcesLinks"]==={}, 
      expr,

      Module[{tempexpr=expr, pos, tclink, vidlink, demolink, commlink, headercell, lrcell, left, right, tutorials, howtos,
                   train,trainover, video,videoover,demo,demoover,comm,commover, demos, tuthowtolist, tuticon, howtoicon},
       {train,trainover} = ToBoxes[Rasterize[
          Import[FileNameJoin[{$DocumentationBuildDirectory, "Internal", "online", "Images", #}], "PNG"], 
          Background->GrayLevel[0.952941]]]& /@
          {"GuideTrainingCourses.png", "GuideTrainingCourses-over.png"};
       {video,videoover} = ToBoxes[Rasterize[
          Import[FileNameJoin[{$DocumentationBuildDirectory, "Internal", "online", "Images", #}], "PNG"], 
          Background->GrayLevel[0.952941]]]& /@
          {"GuideVideos.png", "GuideVideos-over.png"};
       {demo,demoover} = ToBoxes[Rasterize[
          Import[FileNameJoin[{$DocumentationBuildDirectory, "Internal", "online", "Images", #}], "PNG"], 
          Background->GrayLevel[0.952941]]]& /@
          {"GuideDemonstrations.png", "GuideDemonstrations-over.png"};
       {comm,commover} = ToBoxes[Rasterize[
          Import[FileNameJoin[{$DocumentationBuildDirectory, "Internal", "online", "Images", #}], "PNG"], 
          Background->GrayLevel[0.952941]]]& /@
          {"GuideCommunity.png", "GuideCommunity-over.png"};
       {tuticon, howtoicon} = ToBoxes[Rasterize[
          Import[FileNameJoin[{$DocumentationBuildDirectory, "Internal", "online", "Images", #}], "PNG"], 
          Background->GrayLevel[0.952941]]]& /@
          {"LearningResourcesTutorial.png", "LearningResourcesHowTo.png"};
       pos = Position[expr, Cell[_, "GuideFeaturedExamplesSection", ___]];
       If[pos === {}, pos = Position[expr, Cell[_, "GuideReferenceSection", ___]]];
       
       headercell = Cell[Localized["GuideLearningResourcesSectionText", lang], "GuideLearningResourcesSection"];
       tutorials = DocumentationBuild`Utils`GetGuideTutorialsSectionList[tempexpr];
       howtos = DocumentationBuild`Utils`GetGuideHowTosList[tempexpr];

       (* Delete old Tutorials section and remove how to links *)
       tempexpr = DeleteCases[tempexpr, 
         (Cell[___, "GuideTutorial", ___]) | 
         (Cell[c___] /; (MemberQ[{c}, "GuideRelatedLinks", Infinity] && 
            MemberQ[{c}, s_String/;StringMatchQ[s, "paclet:*howto/*"],Infinity] &&
            !MemberQ[{c}, "GuideRelatedLinksSection", Infinity])), Infinity];

       tuthowtolist = {
         {If[Length[tutorials] === 0, "", Cell[
           BoxData[
            GridBox[{{tuticon,GridBox[{{""},{Cell["Tutorials", "GuideTutorialsSection"]},
              Sequence@@({Cell[BoxData[TemplateBox[{Cell[#[[1]]],
                If[StringMatchQ[#[[2]], "paclet:*"],#[[2]],"paclet:"<>#[[2]]]},
                "RefLink", BaseStyle->{"GuideTutorial"}]], "GuideTutorial"]}& /@ tutorials)},
            GridBoxDividers->{"Rows" -> {{False}}}]}}, GridBoxSpacings->{"Columns"->{{0.3}}}, 
            GridBoxAlignment->{"Rows" -> {{Top}}}]], "GuideWebResource"]]},
         {If[Length[howtos] === 0, "", Cell[BoxData[GridBox[{{howtoicon,GridBox[{{""},{Cell[Localized["GuideHowTosSectionText", lang], "GuideHowToSection"]},
              Sequence@@({Cell[BoxData[TemplateBox[{Cell[#[[1]]],
                If[StringMatchQ[#[[2]], "paclet:*"],#[[2]],"paclet:"<>#[[2]]]},
                "RefLink", BaseStyle->{"GuideHowTo"}]], "GuideHowTo"]}& /@ howtos)},
                GridBoxDividers->{"Rows" -> {{False}}}]}}, GridBoxSpacings->{"Columns"->{{0.3}}},
                GridBoxAlignment->{"Rows" -> {{Top}}}]], "GuideWebResource"]]}
         };
       tuthowtolist = If[Length[tutorials] === 0, Reverse[tuthowtolist], tuthowtolist];

       {tclink, vidlink, demolink, commlink} = ReplaceAll[(If[Length[#]>0, First[#]]&/@Function[{type}, 
         Flatten[Select[DocumentationBuild`Info`GetSourceInformation[uri, lang, "WebResourcesLinks"],
           MatchQ[#[[2]], type]&]]] /@ {"Training Courses", "Videos", "Demonstrations", "Community"}), None->Null];
       If[demolink === Null,
         Quiet[demolink = First[Flatten[Cases[tempexpr, Cell[c___, "GuideRelatedLinks", ___]:>
           Cases[{c},URL[s_String]/;!StringFreeQ[s, "demonstrations.wolfram.com/search.html?"]:>s, Infinity], Infinity]]]];
         tempexpr = DeleteCases[tempexpr, Cell[c___, "GuideRelatedLinks", ___] /; 
           MemberQ[{c}, s_String/;!StringFreeQ[s, "demonstrations.wolfram.com/search.html?"], Infinity], Infinity]];
       If[!StringQ[demolink], demolink = Null];
       lrcell = Cell[BoxData[GridBox[{{
         left = Cell[BoxData[GridBox[tuthowtolist, GridBoxSpacings -> {"Rows" -> {{2}}}]], 
         "GuideWebResource"],
         right = DeleteCases[Cell[BoxData[GridBox[{
         {Cell[CapitalizeString[Localized["LearningResourcesRelatedWebText", lang]], "GuideWebResourcesSection"], 
          "\[SpanFromLeft]", ToBoxes[Spacer[80]]},
         {Cell[BoxData[TemplateBox[{Cell[Localized["LearningResourcesTrainingCoursesText", lang]], tclink, train, trainover}, "GrayWebLinkWithIcon", BaseStyle->{"GuideWebResourceText"}]]], 
          "\[SpanFromLeft]", ""},
         {Cell[BoxData[TemplateBox[{Cell[Localized["LearningResourcesVideosText", lang]], vidlink, video, videoover}, "GrayWebLinkWithIcon", BaseStyle->{"GuideWebResourceText"}]]], 
          "\[SpanFromLeft]", ""},
         {Cell[BoxData[TemplateBox[{Cell[Localized["LearningResourcesDemonstrationsText", lang]], demolink, demo, demoover}, "GrayWebLinkWithIcon", BaseStyle->{"GuideWebResourceText"}]]], 
          "\[SpanFromLeft]", ""},
         {Cell[BoxData[TemplateBox[{Cell[Localized["LearningResourcesCommunityText", lang]], commlink, comm, commover}, "GrayWebLinkWithIcon", BaseStyle->{"GuideWebResourceText"}]]], 
          "\[SpanFromLeft]", ""}
         }, 
        GridBoxDividers -> {"Rows" -> {False, False, {True}, 
            False}},GridBoxItemSize->{"Columns"->{{Full},Scaled[0.01]}}]], "GuideWebResource"],
         {Cell[BoxData[TemplateBox[c___]/;MemberQ[{c}, {} | Null, Infinity]],___],__}, Infinity]
         }}, GridBoxAlignment -> {"Columns" -> {{Left, Center}}, 
         "Rows" -> {{Top}}},
        GridBoxDividers -> {"Columns" -> {{False, True, False}, False}}]], "GuideLearningResourcesGrid"];
        
        (* If there are no Related Web Resources, don't include the RHS of the grid *)
        (*lrcell = If[MemberQ[lrcell[[1, 1, 1, 1, 2]], TemplateBox[___], Infinity], lrcell, ReplacePart[lrcell,{1,1,1,1,2}->""]];*)
        lrcell = If[MemberQ[lrcell[[1, 1, 1, 1, 2]], TemplateBox[___], Infinity], lrcell, 
          Cell[BoxData[GridBox[{{left}}, GridBoxAlignment -> {"Columns" -> {{Left, Center}}, 
            "Rows" -> {{Top}}}]], "GuideLearningResourcesGrid"]];
        lrcell = If[Length[tutorials] === 0 && Length[howtos] === 0, 
          Cell[BoxData[GridBox[{{right}}, GridBoxAlignment -> {"Columns" -> {{Left, Center}}, 
            "Rows" -> {{Top}}}]], "GuideLearningResourcesGrid"], lrcell];

        Insert[tempexpr, Cell[CellGroupData[{headercell, lrcell}, Closed]], pos]
   ]];*)

    (* 5/20/15: Copied this from the above commented-out code block to make sure these sections
       still get removed *)
    (* Delete old Tutorials section and remove how to links *)
    (* -- only delete tutorials links for System guides *)
    (* Overriding to True only for bugfix build 11.0.1 (and possibly 11.0.2);
       this forces tutorials to be stripped everywhere because of Reference ss not updated *)
       expr = Module[{systemQ = True(*StringMatchQ[uri, "guide/*" | "paclet:guide/*"]*)},
       	DeleteCases[expr, 
         If[TrueQ[systemQ], (Cell[___, "GuideTutorial", ___]) | 
         (Cell[c___] /; (MemberQ[{c}, "GuideRelatedLinks", Infinity] && 
            MemberQ[{c}, s_String/;StringMatchQ[s, "paclet:*howto/*"],Infinity] &&
            !MemberQ[{c}, "GuideRelatedLinksSection", Infinity])),
            (Cell[c___] /; (MemberQ[{c}, "GuideRelatedLinks", Infinity] && 
            MemberQ[{c}, s_String/;StringMatchQ[s, "paclet:*howto/*"],Infinity] &&
            !MemberQ[{c}, "GuideRelatedLinksSection", Infinity]))], Infinity]];

    (* Change "..." to Ellipsis *)
    BuildLog["-- Change '...' to Ellipsis"];
    expr = expr /. d_String /; StringMatchQ[d, " ..."] :> " \[Ellipsis]";   

    BuildLog["-- GuideText LongDash to GuideEmDash style"];
    expr = expr /.
      Cell[TextData[{a___,
          b_String /; StringMatchQ[b, RegularExpression["\\s*\[LongDash]\\s*(.*)"]],
          c___}], "GuideText", d___] :> 
        ReplaceCharacters[
          Cell[TextData[{a, b, c}], "GuideText", d],
          "\[LongDash]",
          StyleBox["\[LongDash]", "GuideEmDash"]];

	obsolescenceNote = Cases[expr, Cell[_, "ObsolescenceNote", ___], Infinity];
    expr = DeleteCases[expr, Cell[_, "ObsolescenceNote", ___], Infinity];

    BuildLog["-- Delete various cells"];
    If[buildmode==="Future",
    	expr,
    	expr = DeleteCases[expr, Alternatives[
        Cell[s_String /;
          StringMatchQ[s, RegularExpression["XX+"]], ___],
        Cell[TextData[s_List /; Apply[ Or,
        	StringMatchQ[Transmogrify`ConvertToString[#], 
        		RegularExpression[".*XX+.*"]] & /@ s]], "GuideText", ___],
        Cell[cc_ /; (And[
            Length[#] > 0,
            StringMatchQ[Transmogrify`ConvertToString[First[#]], RegularExpression["XX+"]]
          ]&[Cases[cc,
            Cell[_, ("InlineFormula"|"InlineGuideFunctionListing"), ___], Infinity]]),
          "InlineGuideFunctionListing" | "GuideText" ___],
        Cell[s_String /;
          StringMatchQ[s, RegularExpression["(XX+( \\. )?)+"]],
          "InlineFormula", ___],
        Cell[s_String /;
          StringMatchQ[s, RegularExpression["(XX+( \\. )?)+"]],
          "InlineGuideFunctionListing", ___],
        Cell[cc_ /;
          (Count[cc, Cell[BoxData[s_String /;
              StringMatchQ[s, RegularExpression["XX+"]]
            ], "InlineGuideFunction", ___], Infinity] > 0),
          "GuideText", ___]
    ], Infinity]];
    
            expr = expr /. {Cell[c_String, "InlineGuideFunctionListing", o___] :> 
        InlineFunctionButton[Cell[c, "InlineGuideFunctionListing", o]]};

    BuildLog["-- Remove links to flagged symbols"];
    Module[{delpos},
    delpos = Reap[Scan[
      Function[{pos}, Module[{cell, links, newlinks, text},
        cell = Extract[expr, pos];
        links = Cases[cell,
          Cell[BoxData[_ButtonBox | StyleBox[_ButtonBox, ___]], "InlineGuideFunction", ___],
          Infinity];
        If[Length[links] > 0,
          If[buildmode==="Future", 
          	newlinks=links,
          	newlinks = Select[links,
            Function[{link}, Module[{buri},
              buri = Cases[link, _[ButtonData, u_] :> u, Infinity, 1];
              If[Length[buri] == 0,
                buri = "paclet:ref/" <> Transmogrify`ConvertToString[link[[1, 1, 1]]],
                buri = First[buri]];
              If[
                MatchQ[link,
                  Cell[___, "FutureExample" | "TentativeExample", ___] |
                  Cell[BoxData[StyleBox[___, "FutureExample" | "TentativeExample", ___], ___], ___] ],
                False,
                If[DocumentationBuild`Info`GetSourceInformation[uri, lang, "Flag"]==="ObsoleteFlag",
                FreeQ[{"InternalFlag", "TemporaryFlag", "FutureFlag", "ExcisedFlag"},
                  DocumentationBuild`Info`GetSourceInformation[buri, lang, "Flag"]],
                FreeQ[{"ObsoleteFlag", "InternalFlag", "TemporaryFlag", "FutureFlag", "ExcisedFlag"},
                  DocumentationBuild`Info`GetSourceInformation[buri, lang, "Flag"]]]
              ]
          ]]]];
          If[Length[links] =!= Length[newlinks],
            If[Length[newlinks] > 0,
              text = Cases[cell, _TextData, Infinity, 1][[1, 1]];
              text = Drop[text,
                Position[text, Cell[BoxData[_ButtonBox], "InlineGuideFunction", ___]][[-1, 1]]
              ];
              expr = ReplacePart[expr, pos -> Join[
                Cell[TextData[Join[Riffle[newlinks, ", "], text]]],
                Rest[cell] ]];
              , (* Else - removing all links *)
              Sow[pos];
            ];
        ]];
      ]],
      Position[expr, Cell[_, "GuideText", ___]]
    ]];
    If[Length[delpos[[2]]] > 0,
      expr = Delete[expr, delpos[[2, 1]] ]]; ];
    Scan[
      Function[{pos}, Module[{cell, links, newlinks, text},
        cell = Extract[expr, pos];
        links = Cases[cell,
          Cell[BoxData[_ButtonBox | StyleBox[_ButtonBox, ___]], "InlineFormula", ___],
          Infinity];
        If[buildmode==="Future",
        newlinks = links,
        newlinks = Select[links,
          Function[{link}, Module[{buri},
            buri = Cases[link, _[ButtonData, u_] :> u, Infinity, 1];
            If[Length[buri] == 0,
              buri = "paclet:ref/" <> Transmogrify`ConvertToString[link[[1, 1, 1]]],
              buri = First[buri]];
            If[
              MatchQ[link,
              Cell[___, "FutureExample" | "TentativeExample", ___] |
              Cell[_[StyleBox[___, "FutureExample" | "TentativeExample", ___], ___], ___] |
              Cell[_[ButtonBox[___, BaseStyle->{___, "BrighterFlag", ___}, ___], ___], ___] ],
              False,
              If[DocumentationBuild`Info`GetSourceInformation[uri, lang, "Flag"]==="ObsoleteFlag",
                FreeQ[{"InternalFlag", "TemporaryFlag", "FutureFlag", "ExcisedFlag"},
                  DocumentationBuild`Info`GetSourceInformation[buri, lang, "Flag"]],
                FreeQ[{"ObsoleteFlag", "InternalFlag", "TemporaryFlag", "FutureFlag", "ExcisedFlag"},
                  DocumentationBuild`Info`GetSourceInformation[buri, lang, "Flag"]]]
            ]
        ]]]];
        If[Length[links] =!= Length[newlinks],
          expr = ReplacePart[expr, pos -> Join[
            Cell[TextData[
              Flatten[Riffle[newlinks, {{
                  "\[NonBreakingSpace]",
                  StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
                  " "
                }}, 2]]
            ]],
            Rest[cell]
          ]];
        ]]],
      Position[expr, Cell[_, "InlineGuideFunctionListing", ___]]
    ];
    
    (* Make sure string boxes in an InlineGuideFunction don't get put in a TextData -- 
       they'll show up as raw strings instead of box interpretation *)
    expr = ReplaceAll[expr, Cell[BoxData[s_String /; !StringFreeQ[s, "\!"]], some___, "InlineGuideFunction", rest___] :> 
       Cell[BoxData[s], some, "InlineFunctionSans", rest]];

    (* change unlinked InlineGuideFunction to InlineFunctionSans *)
    expr = ReplaceAll[expr, Cell[BoxData[c_ /; FreeQ[c,GraphicsBox[___]|Graphics3DBox[___]|StyleBox[___,"TI"]]] | 
    	TextData[c_ /; FreeQ[c,GraphicsBox[___]|Graphics3DBox[___]|StyleBox[___,"TI"]]], some___, "InlineGuideFunction", rest___] :> 
  	 Cell[TextData[c], some, "InlineFunctionSans", rest]];

    BuildLog["-- Remove empty cells"];
    (* Remove empty cells *)
    expr = expr //. {
      Cell["", s_String /; !StringMatchQ[s, "GuideFunctionsSection"], ___] :> Sequence[],
      Cell[TextData[{}], s_String /; !StringMatchQ[s, "GuideFunctionsSection"], ___] :> Sequence[],
      BoxData[] :> ""};

    (* Need two types of GuideDelimiter for equal spacing around headings *)
    expr = expr //. 
      CellGroupData[{ce___, Cell[a_, "GuideDelimiter", b___],
          Cell[CellGroupData[{Cell[c_, "GuideFunctionsSubsection", d___], e___}, f___]], g___ },h___] :>
        CellGroupData[{ce, Cell[a, "GuideDelimiterSubsection", b],
          Cell[CellGroupData[{Cell[c, "GuideFunctionsSubsection", d], e}, f]],g},h];
          
    BuildLog["-- Localize MoreAbout on guide pages"];
    expr = expr /. Cell[c_, "GuideMoreAboutSection", o___] :> 
      Cell[Localized["GuideMoreAboutSectionText", lang], "GuideMoreAboutSection", o];
      
    BuildLog["-- Localize Tutorial Collection on guide pages"];
    expr = expr /. Cell[c_, "GuideTutorialCollectionSection", o___] :> 
      Cell[Localized["TutorialCollectionSectionText", lang], "GuideTutorialCollectionSection", o];

    BuildLog["-- Localize RelatedLinks on guide pages"];
    expr = expr /. Cell[c_, "GuideRelatedLinksSection", o___] :> 
      Cell[Localized["RelatedLinksSectionText", lang], "GuideRelatedLinksSection", o];

    BuildLog["-- Localize RelatedTutorials on guide pages"];
    expr = expr /. Cell[c_, "GuideTutorialsSection", o___] :> 
      Cell[Which[c==="Characters", Localized["CharactersSectionText", lang], True, Localized["TutorialsSectionText", lang]], "GuideTutorialsSection", o];


    (* Keep empty GuideAbstract for vertical spacing *)
    If[ Length@Cases[expr, Cell[_, "GuideAbstract", ___], Infinity, 1] > 0,
      expr = expr /. 
        Cell[str_String /; StringMatchQ[str, RegularExpression["\\s+|\\?+|XX+"]], "GuideAbstract", ___] :>
          Cell["\[NonBreakingSpace]", "GuideAbstract", CellID->1],
      expr = expr /. Cell[con_, "GuideTitle", o___] :> 
        Cell[CellGroupData[{
          Cell[con, "GuideTitle", o], Cell["\[NonBreakingSpace]", "GuideAbstract", CellID->1] }, Open]]
    ];

    (* Removing leading GuideDelimiters *)
    BuildLog["-- Removing leading GuideDelimiters on guide pages"];
    (* Clear a path by removing some empty cells *)
    expr = RemoveEmptyCellGroups[expr, {"GuideFunctionsSection"}];    
    expr = expr //.
      Cell[CellGroupData[{first___, gfs : Cell[___, "GuideFunctionsSection", ___], 
        Cell[CellGroupData[{Cell[___, "GuideDelimiter", ___], more___},___],___], rest___}, oc___], final___] :> 
        Cell[CellGroupData[{first, gfs, more, rest}, oc], final];

    (* Remove "Functions" per sw *)
    Module[{badheads},
      badheads = Alternatives[
        Cell[c_String/; StringMatchQ[c, "Functions:" | ""], "GuideFunctionsSection", ___],
        Cell[TextData[ButtonBox[s_String, BaseStyle->"GuideTestDrive"]], "GuideTestDrive", ___],
        Cell[TextData[ButtonBox[s_String, BaseStyle->"GuideMainTutorial"]], "GuideMainTutorial", ___]
      ];
      expr = expr /. {xa___, Cell[CellGroupData[{badheads, xb___}, ___], ___], xc___}
        :> {xa, xb, xc};
      expr = DeleteCases[expr, badheads, Infinity];
    ];

    (* change " . " to "something else" *)
    BuildLog["-- Change ' . ' to something else"];
    expr = expr //. { 
      Cell[TextData[{a___, s_String, b___}], 
        sty:("InlineGuideFunctionListing" | "GuideText"), o___] :> 
      Cell[TextData[{a, 
        StringSplit[s, RegularExpression["\\s+\\.\\s+"] -> 
          {"\[NonBreakingSpace]", StyleBox["\[FilledVerySmallSquare]", "InlineSeparator"], " "}], b}
        ], sty, o] };

    (* change " \[EmptyVerySmallSquare] " to InlineSeparator *)
    BuildLog["-- Add MakeInlineSeparator[] "];

    expr = expr //. Cell[c_TextData, sty:("GuideText"|"InlineGuideFunctionListing"), o___] :>
      MakeInlineSeparator[Cell[c, sty, o]];
    BuildLog["-- Change FilledSmallSquare to InlineSeparator"];
    expr = expr //. {StyleBox["\[FilledSmallSquare]", "InlineSeparator"] :> 
      StyleBox["\[FilledVerySmallSquare]", "InlineSeparator"]};
  
    (* Need color StyleBox around GuideFunctionsSubsection buttons *)
    BuildLog["-- Add color StyleBox around GuideFunctionsSubsection buttons"];
        expr = expr /. 
  Cell[c_, "GuideFunctionsSubsection", o___] :> 
   Cell[(c /. 
      ButtonBox[b__] :> 
       Module[{bs = Cases[{b}, (BaseStyle -> s_) :> s]}, 
        Apply[ButtonBox, 
         Append[DeleteCases[{b}, _[BaseStyle, _]], 
          BaseStyle -> 
           Append[If[ListQ[bs], bs, {bs}], 
            "GuideFunctionsSubsection"]]]]), 
    "GuideFunctionsSubsection", o];

    (* Change New In 6 links to a specific style *)
    BuildLog["-- Change NewIn6 links to special link"];
    expr = expr /. ButtonBox[f_String/;MemberQ[$NewIn6FunctionList, f], s___, BaseStyle->"Link", r___] :> 
      ButtonBox[f, s, BaseStyle->"NewInVersionLink", r];

    (* Add Dynamic for New In Version Links *)
    expr = expr /. 
      Cell[BoxData[ButtonBox[c__, BaseStyle->"NewInVersionLink", o1___], o2___], o3___] :>
        Cell[BoxData[ButtonBox[c, BaseStyle->"NewInVersionLink", o1], o2], o3,
          Background->Dynamic[
            If[CurrentValue[{TaggingRules, "ModificationHighlight"}] === True, 
            RGBColor[1, 0.988235, 0.721569], None]]];

    (* Change function name links on guide pages to be sans *)
    expr = expr /.
      Cell[BoxData[ButtonBox[c__]] | TextData[ButtonBox[c__]] ,
          "InlineGuideFunction" | "InlineFormula" | "InlineCode", o___] :>
        Cell[TextData[{ButtonBox[c]}], "InlineFunctionSans", o];

    (* If any InlineFunctionSans already exist in BoxData[] change to TextData[] *)
    expr = expr /. Cell[BoxData[ButtonBox[c__]], "InlineFunctionSans", o___] :>
      Cell[TextData[{ButtonBox[c]}], "InlineFunctionSans", o];

    If[StringMatchQ["Title" /. info, RegularExpression["\\s*New\\s*in\\s*6\\.0.*"]];
      BuildLog["-- 'New in 6.0' do not add UpdatedIn Graphic"];
      ,
      BuildLog["-- Add UpdatedIn Graphic"];
      expr = AddGuideNewInGraphic[expr, info, opts];
    ];

    (* Fix appearance of unlinked inline function listings (e.g., guide/DateAndTime) *)
    expr = ReplaceAll[expr, Cell[BoxData[text_String], some___, "InlineFormula", rest___] :>
      Cell[TextData[text], some, "InlineFunctionSans", rest]];

    (* If "Under Development", add [[UNFINISHED DRAFT]] banner *)
    expr = If[MemberQ[expr, "UnderDevelopment", Infinity] || TrueQ["Underdevelopment" /. info],
     langext = lang /. {"English"->"", "Japanese"->"_jp", "ChineseSimplified"->"_zh", "Spanish"->"_es" };
     unfindraftimg = Import[ToFileName[{$DocumentationBuildDirectory,
        "Internal", "online", "Images"}, "RefUnfinishedDraft"<>langext<>".m"]];
     ReplaceAll[expr, c : Cell[___, "GuideTitle", ___] :> 
       Cell[TextData[{c, Cell[BoxData[ToBoxes[Spacer[10]]]], Cell[BoxData[ToBoxes[unfindraftimg]]]}], "GuideTitleGrid"]],
     expr];
     
     (* Turn some TI guide listings into TI2, which ups their font size (per andre/SW, 1/30/14) *)
     expr = ReplaceAll[expr, c:Cell[___, "InlineFunctionSans"|"InlineGuideFunction"|"InlineGuideFunctionListing", ___] :> 
     	ReplaceAll[c, {StyleBox[s_, some___, "TI", rest___] :> StyleBox[s, some, "TI2", rest],
     	    s_String /; !StringFreeQ[s,"\"TI\""] :> StringReplace[s,"\"TI\""->"\"TI2\""]}]];
     	    
    (* Add ObsolescenceNote *)
    warnicon = ToBoxes[Rasterize[
    Import[FileNameJoin[{$DocumentationBuildDirectory, "Internal", "online", "Images", "ObsoleteIcon.png"}], "PNG"], Background->GrayLevel[0.941176]]];
    obsolescenceNote = If[obsolescenceNote === {}, obsolescenceNote, {Cell[BoxData[GridBox[{{Cell[BoxData[warnicon]], Cell[obsolescenceNote[[1, 1]], FontFamily->"Verdana"]}}]], Sequence@@Rest[obsolescenceNote[[1]]]]}];
    expr = expr /. 
    Cell[CellGroupData[{Cell[c_, sty_String/;StringMatchQ[sty, "GuideTitle"], o___], rest___},___]] :>
      Cell[CellGroupData[Flatten[{
        Cell[c, sty, o],
        obsolescenceNote,
        rest
      }, 1], Open]];

    expr
  ];


(******************************************************************************)
(** ConvertNotebook - Overviews **)
ConvertNotebook[expr_, info_,
    ("Guide Overview"),
    opts___?OptionQ] := 
  Module[{},
    expr
  ];



(******************************************************************************)
(** ConvertNotebook - Notes **)
ConvertNotebook[expr_, info_,
    ("Note"),
    opts___?OptionQ] := 
  Module[{},
    expr
  ];














(******************************************************************************)
(** ConvertNotebook - Root Guide **)
ConvertNotebook[nb_, info_,
    ("Root Guide"),
    opts___?OptionQ] := 
  Module[{expr, language, rootSectionCaptions, $Rebranded,
    data, MakeRootActionMenu, rootred, rootorange, rootblue, rootgreen, rootteal, rootgrey,
    brightrootred, brightrootorange, brightrootblue, brightrootgreen, brightrootteal, brightrootgrey, wfpRootQ,
    footerlinks},
   (* expr = ConvertNotebook[nb, info, "Guide", opts];*)
    expr=nb;
    language = "Language" /. {opts} /. Options[MakeNotebook];
    $Rebranded = "Rebranding" /. {opts} /. {"Rebranding" -> False};
    wfpRootQ = TrueQ[MemberQ["Keywords" /. info, "wfp"]];
    footerlinks = Cases[First[
    	Cases[expr, Cell[CellGroupData[{Cell[___, "Section", ___], 
    		rest__}, ___]] :> {rest}, Infinity]], 
    		ButtonBox[text_, ___, ButtonData -> bd_, ___] :> 
    			{text, bd}, Infinity];
    
    (
    LogIt["-- Root guide page"];
    expr = DeleteCases[expr, Cell[_, "LinkTrail", ___], Infinity];
    (
    (* Import images *)
    rasterHeader =  
      Import[ToFileName[{DocumentationBuild`Common`$DocumentationBuildDirectory, "Internal", "online", "Images"}, 
             If[TrueQ[$Rebranded],"RootTitleV10-WL","RootTitleV10"]<>Switch[language, "Japanese", "_jp", 
      		"ChineseSimplified", "_zh", "Spanish", "_es", 
      		_, ""]<>".png"], "PNG", Background->None];
    
    data=Cases[nb,Cell[CellGroupData[{Cell[c_,___,"GuideFunctionsSection",___],rest__},___],___]:>{ConvertToString[c],Cases[{rest},Cell[co_,___,sty:("GuideText"|"GuideDelimiter"),___]:>If[StringMatchQ[sty,"GuideDelimiter"],"Delimiter",{ConvertToString[co],Quiet@Check[First[Cases[co,_[ButtonData,uri_String]:>uri,Infinity]],{}]}],Infinity]},Infinity];
    data=ReplaceAll[data,{"Delimiter",rest__}:>{rest}];
    data=ReplaceAll[data,{some__,"Delimiter"}:>{some}];
 
    rootred=RGBColor[{165,0,0}/255.];
    rootorange=RGBColor[{153,67,0}/255.];
    rootblue=RGBColor[{0,81,112}/255.];
    rootgreen=RGBColor[{76,112,0}/255.];
    rootteal=RGBColor[{1,97,74}/255.];
    rootgrey=RGBColor[{68,68,68}/255.];

    brightrootred=RGBColor[.7,0,0];
    brightrootorange=RGBColor[.75,.41,0];
    brightrootblue=RGBColor[0,.44,.58];
    brightrootgreen=RGBColor[.36,.64,0];
    brightrootteal=RGBColor[.1,.62,.52];

    Options[MakeRootActionMenu]={Appearance->None,ImageSize->All,BaseStyle->{LineSpacing->{2,0}}};
    
    MakeRootActionMenu[header_,mouseover_,contents_List,col_:Automatic,options:OptionsPattern[]]:=
      Module[{opt=Join[options,Options[MakeRootActionMenu]]},
        With[{pairs=If[MatchQ[#,"Delimiter"],Delimiter,#[[1]]:>
          FEPrivate`FrontEndExecute[FrontEndToken["OpenHelpLink",{#[[2]],FEPrivate`If[CurrentValue["ShiftKey"],Null,FEPrivate`ButtonNotebook[]]}]]]&
         /@contents},
       ActionMenu[Mouseover[header,mouseover],pairs,opt,MenuStyle->{FontSize->14,FontFamily->"Arial",FontWeight->"Bold",FontColor->col}]]];

    makeRootCell = Function[{ex, st},
      Cell[BoxData[
        ex
      ], st]
    ];

    (* Extract information and delete cells *)
    abstract = Cases[expr, Cell[c_, "GuideAbstract", ___] :> c, Infinity, 1];
    expr = DeleteCases[expr, Cell[_, "GuideAbstract" | "FooterCell", ___], Infinity];
    (* Construct new cells *)
    c = If[Length@abstract > 0, First@abstract, " "];
    guideTitleCell = 
        Cell[BoxData[FrameBox[
          ToBoxes@Deploy@
            MapAt[{#, Text[c, ImageScaled[{0.05, .5}], {-1, 1}]} &, rasterHeader, {1}]
        ]], "RootTitle"];
    (* Insert new cells *)
    expr = expr /. {a___, Cell[_, "GuideTitle", ___], b___} :> {
      a,
      
      
      (*makeRootCell[
      Cell[BoxData[
        GridBox[{{
          TagBox[ButtonBox[
            Cell[TextData[Localized["RootGSVideos", language]]],
            BaseStyle -> "RootFooterMainLink",
            Background->Dynamic[
              If[CurrentValue["MouseOver"],
               RGBColor[0.596078,0.015686,0.039215], RGBColor[0.686274,0.05098,0.094117]]],
            ButtonData->"http://www.wolfram.com/support/learn/get-started-with-mathematica",
            ButtonNote -> None
          ],MouseAppearanceTag["LinkHand"]],
          TagBox[ButtonBox[
            Cell[TextData[Localized["RootWhatsNew", language]]],
            BaseStyle -> {"Link", "RootFooterMainLink"},
            Background->Dynamic[
              If[CurrentValue["MouseOver"],
               RGBColor[0.596078,0.015686,0.039215], RGBColor[0.686274,0.05098,0.094117]]],
            ButtonData -> "paclet:guide/SummaryOfNewFeaturesIn100",
            ButtonNote -> None
          ], MouseAppearanceTag["LinkHand"]],
          TagBox[ButtonBox[
            Cell[TextData[Localized["RootHowTo", language]]],
            BaseStyle -> {"Link", "RootFooterMainLink"},
            Background->Dynamic[
              If[CurrentValue["MouseOver"],
               RGBColor[0.596078,0.015686,0.039215], RGBColor[0.686274,0.05098,0.094117]]],
            Evaluator -> Automatic,
            ButtonData -> "paclet:guide/HowToTopics",
            ButtonNote -> None
          ], MouseAppearanceTag["LinkHand"]]
        }}]
      ], "RootHeaderGrid"],
    "RootHeader"],*)
      b};

    $i = 0;

    (* If 'GuideText' list is a string, make into a list of Buttons *) 
(* Commenting out 070226 -- andy
    expr = expr /. Cell[c_String, "GuideText", opts___] :> 
      inlineFunctionStringToButtonBox[Cell[c, "GuideText", opts]];
*)
    (* Center Root Guide page *)
    expr = expr /. Rule[WindowMargins,_] :> WindowMargins->Automatic;
    );

    expr = expr /. {
      Cell[a___, "GuideMainTutorial", z___] :> Cell[a, "RootGuideMainTutorial", z],
      Cell[a___, "GuideMoreAbout", z___] :> 
      Cell[TextData[
  Join[{"\[ThickSpace]\[FilledVerySmallSquare]\[ThickSpace]\[ThickSpace]\[ThickSpace]"},
    If[MatrixQ[List @@ a], a[[1]],  List @@ a]]], "RootMoreAbout", z, 
 FontColor -> 
  RGBColor[0.729411764705882, 0.21176470588235305`, 
   0.0627450980392157]],
      (*Cell[a___, "GuideMoreAbout", z___] :> 
      Cell[ TextData@@Join[{"\[ThickSpace]\[FilledVerySmallSquare]\[ThickSpace]\[ThickSpace]\[ThickSpace]"},List@@a] , 
      	"RootMoreAbout", z, FontColor->RGBColor[0.729411764705882, 0.21176470588235305`, 0.0627450980392157]],*)
      Cell[a___, "GuideMoreAboutDelimiter", z___] :> Cell[a, "RootMoreAboutDelimiter", z],
      Cell[a___, "GuideMoreAboutSub", z___] :> Cell[TextData@@Join[{"\[ThickSpace]\[FilledVerySmallSquare]\[ThickSpace]\[ThickSpace]\[ThickSpace]"},List@@a], "RootMoreAboutSub", z, FontColor->RGBColor[0.729411764705882, 0.21176470588235305`, 0.0627450980392157]]
    };
  (* Root guide page -> multi column links *)
  (* BETA:: Some changes in the following that were hacked in for the M9 beta 1 *)
  iconiter = 0;
  expr = DeleteCases[expr, Cell[CellGroupData[{Cell[___,"GuideFunctionsSection"|"Section",___],___},___],___], Infinity];
  expr[[1]] = With[{rootredtext=rootred,rootorangetext=rootorange,rootbluetext=rootblue,rootgreentext=rootgreen,roottealtext=rootteal,
                    rootredicon=brightrootred,rootorangeicon=brightrootorange,rootblueicon=brightrootblue,
                    rootgreenicon=brightrootgreen,roottealicon=brightrootteal}, Join[expr[[1]], 
   {Cell[BoxData[ToBoxes[
     Grid[Join[{{Item[Style[Image[rasterHeader,ImageSize->All], Magnification->(Inherited * 1)],Alignment->Left]}}, 
      
      {{If[!TrueQ[wfpRootQ], 
       Row[
       Table[Module[{
       	textcol=Which[i<=4,rootredtext,
         i<=9,rootorangetext,
         i<=16,rootbluetext,
         i<=18,rootgreentext,
         i>18,roottealtext],
         iconcol=Which[i<=4,rootredicon,
         i<=9,rootorangeicon,
         i<=16,rootblueicon,
         i<=18,rootgreenicon,
         i>18,roottealicon],pad,iconover,
       	iconimg=Import[FileNameJoin[{$DocumentationBuildDirectory,"Internal","online","Images","RootTileIcon"<>ToString[i]<>".png"}]]},
           pad = 254 - First[ImageDimensions[iconimg]];
           iconover = Rasterize@ImagePad[Rasterize[ImagePad[iconimg, {{pad,0},{0,0}}, None], Background->((5 #/6)& /@ iconcol)],{{0,5},{0,0}}, rootgrey];
           iconimg = Rasterize@ImagePad[Rasterize[ImagePad[iconimg, {{pad,0},{0,0}}, None], Background->iconcol],{{0,5},{0,0}}, rootgrey]; 
           
       	MouseAppearance[MakeRootActionMenu[
         Button[Overlay[{iconimg, 
           Grid[{{Spacer[15], Style[data[[i,1]], "RootSectionTitle"]}}, Alignment->{Left, Center},
             ItemSize->{{Scaled[.03],Scaled[.70]},Automatic}]}, Alignment->{Left, Center}],
           ImageSize->{259, 74}, Appearance->None],
         Button[Overlay[{iconover,
           Grid[{{Spacer[15], Style[data[[i,1]], "RootSectionTitle"]}}, 
            Alignment->{Left, Center},
             ItemSize->{{Scaled[.03],Scaled[.70]},Automatic}]}, Alignment->{Left, Center}],
           ImageSize->{259, 74}, Appearance->None],
         data[[i,2]],
         textcol
       ],"LinkHand"]],
       {i,Length[data]}]],
       (*else --- WFP Wolfram Finance Platform-specific code below --- had to start with 0 index since icons are numbered*)
       Row[
       Table[With[{i = j+1},Module[{
       	textcol=Which[i<=3,rootbluetext,
         i<=7,rootredtext,
         i<=12,rootorangetext,
         i<=17,rootbluetext,
         i<=19,rootgreentext,
         i>19,roottealtext],
         iconcol=Which[i<=3,rootblueicon,
         i<=7,rootredicon,
         i<=12,rootorangeicon,
         i<=17,rootblueicon,
         i<=19,rootgreenicon,
         i>19,roottealicon],pad,iconover,
       	iconimg=Import[FileNameJoin[{$DocumentationBuildDirectory,"Internal","online","Images","RootTileIcon"<>
              Which[i>3 && i<12,ToString[i-3],
              i>=12 && i<16, ToString[i-2],
              i>=16, ToString[i-1],
              i===1,"WFP",
              i===2,"14",
              i===3,"10"]
              <>".png"}]]},
           pad = 254 - First[ImageDimensions[iconimg]];
           iconover = Rasterize@ImagePad[Rasterize[ImagePad[iconimg, {{pad,0},{0,0}}, None], Background->((5 #/6)& /@ iconcol)],{{0,5},{0,0}}, rootgrey];
           iconimg = Rasterize@ImagePad[Rasterize[ImagePad[iconimg, {{pad,0},{0,0}}, None], Background->iconcol],{{0,5},{0,0}}, rootgrey]; 
           
       	MouseAppearance[MakeRootActionMenu[
         Button[Overlay[{iconimg, 
           Grid[{{Spacer[15], Style[data[[i,1]], "RootSectionTitle"]}}, Alignment->{Left, Center},
             ItemSize->{{Scaled[.03],Scaled[.70]},Automatic}]}, Alignment->{Left, Center}],
           ImageSize->{259, 74}, Appearance->None],
         Button[Overlay[{iconover,
           Grid[{{Spacer[15], Style[data[[i,1]], "RootSectionTitle"]}}, 
            Alignment->{Left, Center},
             ItemSize->{{Scaled[.03],Scaled[.70]},Automatic}]}, Alignment->{Left, Center}],
           ImageSize->{259, 74}, Appearance->None],
         data[[i,2]],
         textcol
       ],"LinkHand"]]],
       {j,0,Length[data]-1}]]]
      }},

        {{Item[Style[Row[With[{lctext = StringJoin[#[[1]], " \[RightGuillemet]"], hyperlinkQ = MemberQ[#[[2]], URL[__], Infinity]}, 
        	MouseAppearance[
        	Button[
        		PaneSelector[{
        			False -> Style[lctext,FontColor->GrayLevel[.85]],
        			True -> Style[lctext,FontColor->RGBColor[0.890196, 0., 0.0392157]]
            }, Dynamic[CurrentValue["MouseOver"]]],
            Evaluate[If[TrueQ[hyperlinkQ],
            	FrontEndExecute[{NotebookLocate[#2,"OpenInNewWindow"->CurrentValue["HyperlinkModifierKey"]]}]&,
            	FEPrivate`FrontEndExecute[FrontEndToken["OpenHelpLink",{#1,
            		FEPrivate`If[CurrentValue["ShiftKey"],Null,FEPrivate`ButtonNotebook[]]}]]&
            ]],
            BaseStyle -> "RootFooterLink",
            ButtonData -> #[[2]],
            ButtonNote -> None,Appearance->None], "LinkHand"]]& /@ footerlinks
        	(* commented out 082616 to use footer link info from source guide page, replaced with above code --brianv {
          MouseAppearance[Button[
            lctext = Localized["RootIntroProg", language];
            PaneSelector[{
              False -> Style[lctext,FontColor->GrayLevel[.85]],
              True -> Style[lctext,FontColor->RGBColor[0.890196, 0., 0.0392157]]
            }, Dynamic[CurrentValue["MouseOver"]]],
            FrontEndExecute[{NotebookLocate[#2,"OpenInNewWindow"->CurrentValue["HyperlinkModifierKey"]]}]&,
            BaseStyle -> "RootFooterLink",
            ButtonData -> {URL["http://www.wolfram.com/language/fast-introduction-for-programmers/"],None},
            ButtonNote -> None,Appearance->None], "LinkHand"],
          MouseAppearance[Button[
            lctext = Localized["RootEIWL", language];
            PaneSelector[{
              False -> Style[lctext,FontColor->GrayLevel[.85]],
              True -> Style[lctext,FontColor->RGBColor[0.890196, 0., 0.0392157]]
            }, Dynamic[CurrentValue["MouseOver"]]],
            FrontEndExecute[{NotebookLocate[#2,"OpenInNewWindow"->CurrentValue["HyperlinkModifierKey"]]}]&,
            BaseStyle -> "RootFooterLink",
            ButtonData -> {URL["http://www.wolfram.com/language/elementary-introduction/?source=desktop"],None},
            ButtonNote -> None,Appearance->None], "LinkHand"],
          MouseAppearance[Button[
            lctext = Localized["RootWhatsNew", language];
            PaneSelector[{
              False -> Style[lctext,FontColor->GrayLevel[.85]],
              True -> Style[lctext,FontColor->RGBColor[0.890196, 0., 0.0392157]]
            }, Dynamic[CurrentValue["MouseOver"]]],
            FEPrivate`FrontEndExecute[FrontEndToken["OpenHelpLink",{#1,
              FEPrivate`If[CurrentValue["ShiftKey"],Null,FEPrivate`ButtonNotebook[]]}]]&,
            BaseStyle -> "RootFooterLink",
            ButtonData -> "paclet:guide/SummaryOfNewFeaturesIn11",
            ButtonNote -> None,Appearance->None], "LinkHand"],
          MouseAppearance[Button[
            lctext = Localized["RootIndex", language];
            PaneSelector[{
              False -> Style[lctext,FontColor->GrayLevel[.85]],
              True -> Style[lctext,FontColor->RGBColor[0.890196, 0., 0.0392157]]
            }, Dynamic[CurrentValue["MouseOver"]]],
            FEPrivate`FrontEndExecute[FrontEndToken["OpenHelpLink",{#1,
              FEPrivate`If[CurrentValue["ShiftKey"],Null,FEPrivate`ButtonNotebook[]]}]]&,
            BaseStyle -> "RootFooterLink",
            ButtonData -> "paclet:guide/AlphabeticalListing",
            ButtonNote -> None,Appearance->None], "LinkHand"],
          (*MouseAppearance[Button[
            lctext = Localized["RootStandardPackages", language];
            PaneSelector[{
              False -> Style[lctext,FontColor->GrayLevel[.85]],
              True -> Style[lctext,FontColor->RGBColor[0.890196, 0., 0.0392157]]
            }, Dynamic[CurrentValue["MouseOver"]]],
            FEPrivate`FrontEndExecute[FrontEndToken["OpenHelpLink",{#1,
              FEPrivate`If[CurrentValue["ShiftKey"],Null,FEPrivate`ButtonNotebook[]]}]]&,
            BaseStyle -> "RootFooterLink",
            ButtonData -> "paclet:guide/StandardExtraPackages",
            ButtonNote -> None,Appearance->None], "LinkHand"],*)
          MouseAppearance[Button[
            lctext = Localized["RootAddOns", language];
            PaneSelector[{
              False -> Style[lctext,FontColor->GrayLevel[.85]],
              True -> Style[lctext,FontColor->RGBColor[0.890196, 0., 0.0392157]]
            }, Dynamic[CurrentValue["MouseOver"]]],
            FEPrivate`FrontEndExecute[FrontEndToken["OpenHelpLink",{#1,
              FEPrivate`If[CurrentValue["ShiftKey"],Null,FEPrivate`ButtonNotebook[]]}]]&,
            BaseStyle -> "RootFooterLink",
            ButtonData -> "paclet:guide/InstalledAddOns",
            ButtonNote -> None,Appearance->None], "LinkHand"]}*), Spacer[22]], Magnification->(Inherited * 1)], Alignment->Left]}}], 
      Alignment->Center,Spacings->{Automatic,{Automatic,2.2,{Automatic},2.2,Automatic}}]]], 
    "RootCell"]}]];

  expr = DeleteCases[expr, Cell[_, "RootMoreAbout", ___], Infinity];
  
      
  expr = With[{rootgrey=rootgrey},Join[
    DeleteCases[expr,
      ((CellGrouping|Selectable|ShowGroupOpener|Background)~(Rule|RuleDelayed)~_)],
    Notebook[
      (*CellGrouping->Manual,*)
      ShowGroupOpener->False,
      Selectable->False,
      Background->rootgrey
    ]]];
  );
  
(* Special case for stylebox merging/wrapping on root guide page *)	
expr = expr/. Cell[BoxData[ButtonBox[PaneSelectorBox[{Rule[b1__,Cell[TextData[{"MathLink"}]]],Rule[b3__,Cell[TextData[{"MathLink"}], "RootDarkLink"]]},psopt2_], 
		bbopts2__]], cellsty2__] :> 
		Cell[BoxData[ButtonBox[PaneSelectorBox[{Rule[b1,Cell[TextData[{StyleBox["MathLink", FontSlant->Italic]}]]],Rule[b3,Cell[TextData[{StyleBox["MathLink", FontSlant->Italic]}], "RootDarkLink"]]},psopt2], 
		bbopts2]], cellsty2];

expr = expr /. TextData[{ding_,StyleBox[Cell[BoxData[ButtonBox[PaneSelectorBox[{Rule[a1__,Cell[TextData[{a2__}]]],
		Rule[a3__,Cell[TextData[{a4__}], "RootDarkLink"]] },psopt1_], bbopts1__]], cellsty1__], sty__],
	Cell[BoxData[ButtonBox[PaneSelectorBox[{Rule[b1__,Cell[TextData[{b2__}]]],Rule[b3__,Cell[TextData[{b4__}], "RootDarkLink"]]},psopt2_], 
		bbopts2__]], cellsty2__]}] :>
		TextData[{ding, Cell[BoxData[ButtonBox[PaneSelectorBox[{Rule[a1,Cell[TextData[{StyleBox[a2, sty], b2}]]],
		Rule[a3,Cell[TextData[{StyleBox[a4, sty], b4}], "RootDarkLink"]] },psopt1], bbopts1]], cellsty1]}];
		
  expr
  ];


(******************************************************************************)
(** ConvertNotebook - Workflows **)
ConvertNotebook[nb_, info_,
    ("Workflow"),
    opts___?OptionQ] := 
  Module[{expr, uri, lang, stylerules},
    expr = nb;
    uri = "URI" /. info;
    uri = If[StringMatchQ[uri,"paclet:*"], uri, "paclet:"<>uri];
    lang = "Language" /. {opts} /. Options[MakeNotebook];
    
    (* Convert various cell styles *)
    stylerules = (Cell[s__, #, r___] :> Cell[s, StringJoin["Workflow",#], r])& /@ 
     {"Explanation", "ThingsToKnow", "Details", "RelatedWorkflows", "Text", "Item", "RelatedWorkflowsSection",
      "QAndASection", "Question", "Answer", "TechNotesSection", "TechNote"};
      
    expr = ReplaceAll[expr, stylerules];
    expr]


(******************************************************************************)
(** ConvertNotebook - Tutorials **)
ConvertNotebook[nb_, info_,
    ("Tutorial" | "Upgrade Information" | "Overview"),
    opts___?OptionQ] := 
  Module[{expr, sections, uri, lang, pos, titlepos, jumpboxcell, jumparrow, jumparrowover},
	expr = nb;
    uri = "URI" /. info;
    uri = If[StringMatchQ[uri,"paclet:*"], uri, "paclet:"<>uri];
    lang = "Language" /. {opts} /. Options[MakeNotebook];
    sections = DocumentationBuild`Info`GetSourceInformation[uri, lang, "TutorialSections"];
    If[Length[sections] > 1,
      jumparrow = ToBoxes[Image[Import[
        FileNameJoin[{$DocumentationBuildDirectory, "Internal", "online", "Images", 
          "JumpDownAnchorLinkArrows.png"}]], ImageSize->{8,9}]];
      jumparrowover = ToBoxes[Image[Import[
        FileNameJoin[{$DocumentationBuildDirectory, "Internal", "online", "Images", 
          "JumpDownAnchorLinkArrows-over.png"}]], ImageSize->{8,9}]];
      sections = 
        TemplateBox[{Cell[#[[1]]],uri<>"#"<>ToString[#[[2]]],jumparrow,jumparrowover}, "GrayLinkWithIcon", BaseStyle->{"TutorialJumpBoxLink"}]& /@ sections;
      sections = 
        Transpose[Partition[sections, Ceiling[Length[sections]/2], Ceiling[Length[sections]/2], {1,1}, ""]];
      titlepos = Position[expr, Cell[__, "Title", ___]];
      pos = MapAt[#+1&, titlepos, {1,-1}];
      jumpboxcell = Cell[BoxData[GridBox[sections]], "TutorialJumpBox"];
      expr = Insert[expr, jumpboxcell, pos];
    ];
    expr = expr /. Cell[a__, "TutorialCollectionSection", b___]:>Cell[a, "TutorialsTutorialCollectionSection", b];
    expr = expr /. Cell[con_,
      sty:Alternatives@@("CapitalizeHeadingStyles" /. {opts} /. Options[MakeNotebook]),
      o___] :>
    Cell[TextData@CapitalizeString[con], sty, o];
    
    BuildLog["-- Localize MoreAbout on tutorials"];
    expr = expr /. Cell[c_, "TutorialMoreAboutSection", o___] :> 
      Cell[Localized["TutorialMoreAboutSectionText", lang], "TutorialMoreAboutSection", o];
      
    BuildLog["-- Localize tutorial collection section on tutorials"];
    expr = expr /. Cell[c_, "TutorialsTutorialCollectionSection", o___] :> 
      Cell[Localized["TutorialCollectionSectionText", lang], "TutorialsTutorialCollectionSection", o];

    BuildLog["-- Localize Related Tutorials on tutorials"];
    expr = expr /. Cell[c_, "RelatedTutorialsSection", o___] :> 
      Cell[Localized["RelatedTutorialsSectionText", lang], "RelatedTutorialsSection", o];
 
   expr
  ];
  
(******************************************************************************)
(** ConvertNotebook - HowTo **)
ConvertNotebook[nb_, info_,
    ("HowTo"),
    opts___?OptionQ] := 
  Module[{expr, language, obsolescenceNote, screenshot, screenuri, warnicon},
	expr = nb;
	language = "Language" /. {opts} /. Options[MakeNotebook];
	
  obsolescenceNote = Cases[expr, Cell[_, "ObsolescenceNote", ___], Infinity];
  expr = DeleteCases[expr, Cell[_, "ObsolescenceNote", ___], Infinity];

  expr = expr /. Cell[p_, "Title", o___] :>
    Cell[
      TextData[Join[
        {
          Cell[Localized["HowToPrefix", language], "HowToTitlePrefix"],
          " "
        },
        If[Head[p] === TextData, First[p], {p}]
      ]],
      "HowToTitle", o];
  expr = expr /. Cell[p_, "Text", o___] :> Cell[p, "HowToText", o];

  (* Add ObsolescenceNote *)
  warnicon = ToBoxes[Rasterize[
    Import[FileNameJoin[{$DocumentationBuildDirectory, "Internal", "online", "Images", "ObsoleteIcon.png"}], "PNG"], Background->GrayLevel[0.941176]]];
  obsolescenceNote = If[obsolescenceNote === {}, obsolescenceNote, {Cell[BoxData[GridBox[{{Cell[BoxData[warnicon]], Cell[obsolescenceNote[[1, 1]], FontFamily->"Verdana"]}}]], Sequence@@Rest[obsolescenceNote[[1]]]]}];
  expr = expr /. 
    Cell[c_, sty_String/;StringMatchQ[sty, "Title"|"HowToTitle"], o___] :>
      Cell[CellGroupData[Flatten[{
        Cell[c, sty, o],
        obsolescenceNote
      }, 1], Open]];

  screenshot = Flatten[
   Cases[expr, Cell[img_, ___, "AlternateThumbnail", ___] :> Cases[img, GraphicsBox[___], Infinity], Infinity]];
  If[Length[screenshot] > 0, screenshot = First[screenshot]];

  expr = DeleteCases[expr, Alternatives[
      Cell[TextData[Cell[BoxData[""]]], "ScreencastMarker", ___],
      Cell["paclet:screencast/XXXX", "ScreencastURI", ___],
      Cell[___, "AlternateThumbnail", ___]
    ],
    Infinity];

  expr = expr /. Cell[cont_, "SeeAlso", o___] :>
    Cell[
      cont /. {ButtonBox[c___] :> StyleBox[ButtonBox[c], FontFamily->"Verdana"],
        StyleBox[ButtonBox[c___], "MenuName"] :> StyleBox[ButtonBox[c], "MenuName", FontSize->12]},
      "SeeAlso", o];


  (*  NEW V9 CODE -- old code is commented out just below *)
  Module[{abspos,abstract,htxt, screencell},
    screenuri = Cases[expr, Cell[uri_, "ScreencastURI", ___] :> uri, Infinity, 1];
    If[Length[screenuri] > 0,
      screenuri = StringReplace[First[screenuri],
        "paclet:screencast/" -> "http://reference.wolfram.com/mathematica/screencasts/"
      ];
      abspos = First@Position[expr, Cell[_, "HowToAbstract", ___], Infinity, 1];
      abstract = Extract[expr, abspos];
      expr = Delete[expr, abspos];
      htxt = Localized["VideoHowTo", language];
      screencell = Cell[BoxData[TagBox[ButtonBox[
        With[{screenoff = ToBoxes[ImagePad[ImageTake[ToExpression@screenshot, {3, -3}, {3, -3}], 2, GrayLevel[.8]]], 
              screenover = ToBoxes[ImagePad[ImageTake[ImagePad[
                ImageTake[ToExpression@screenshot, {3, -3}, {3, -3}], 2, GrayLevel[.8]], 
                {2, -2}, {2, -2}], 1, RGBColor[.854902, .396078, .145098]]]},
        DynamicBox[If[CurrentValue["MouseOver"], screenover, screenoff]]], 
        BaseStyle->"Hyperlink", 
        ButtonData->{URL[screenuri], None}],MouseAppearanceTag["LinkHand"]]], "HowToScreencastLink"];
      expr = Insert[expr, screencell, abspos];
      expr = Insert[expr, abstract, abspos];
      
  ]
  ];

  (*Module[{screenimg, abspos, abstract, htxt},
    screenuri = Cases[expr, Cell[uri_, "ScreencastURI", ___] :> uri, Infinity, 1];
    If[Length[screenuri] > 0,
      screenuri = StringReplace[First[screenuri],
        "paclet:screencast/" -> "http://reference.wolfram.com/mathematica/screencasts/"
      ];
      screenimg = First@Cases[
        Cases[expr, Cell[_, "ScreencastMarker", ___], Infinity, 1],
        _GraphicsBox, Infinity, 1];
      screenimg = ToBoxes[Import[ToFileName[
          {$DocumentationBuildDirectory, "Internal", "online", "Images"},
          "ScreencastLink.png"], {"PNG", "Graphics"}]];
      abspos = First@Position[expr, Cell[_, "HowToAbstract", ___], Infinity, 1];
      abstract = Extract[expr, abspos];
      htxt = Localized["VideoHowTo", language];
      abstract = Cell[
        BoxData[GridBox[{{
            Cell[First[abstract]],
            Cell[BoxData[
              ButtonBox[
                PaneSelectorBox[{
                  False->GridBox[{{screenimg},
                    {Cell[TextData[StyleBox[htxt, "HowToScreencastLink"]]]}}],
                  True->GridBox[{{screenimg},
                    {Cell[TextData[StyleBox[htxt, "HowToScreencastRedLink"]]]}}]
                },  Dynamic[CurrentValue["MouseOver"]]],
                (*ButtonStyle->"HowToScreencastLink",*)
                BaseStyle->"HowToScreencastLink",
                ButtonData->{URL[screenuri], None}]
            ], "HowToScreencastLink"]
          }}
        ]],
        Sequence@@Rest[abstract]];
      expr = ReplacePart[expr, abspos -> abstract];
  ]];*)
  expr = DeleteCases[expr, Cell[_, "ScreencastURI"|"ScreencastMarker"|"Time", ___], Infinity];

  expr = expr /. Cell[c_, "ScreencastPointerText", o___] :>
    Cell[
      c /. ButtonBox[s_, ___] :>
        ButtonBox[s, BaseStyle->{"Hyperlink", FontWeight->"Bold"},
          ButtonData->{URL[screenuri], None}],
      "ScreencastPointerText", o];

(*  expr = KillEmptyCellGroups[expr, {"MoreToExploreSection", "MoreInformationSection"}];

  (*  Change headings to have Open/close buttons *)
  expr = expr /. Cell[con_, sty_String /; StringMatchQ[sty, Alternatives@@Flatten@{"Styles" /. Options[DocumentationTools`Utilities`CapitalizeHeadings] } ], o___] :>
                   Cell[TextData@WrapInOpenCloseButton@con, sty, o];*)

  (* FIXME: cells not being renumbered *)
  expr = RenumberCells[expr, {"HowToAbstract", "HowToDelimiter"}];
  
  (* Replace Under Development with [[Unfinished Draft]] banner *)
  expr = Module[{langext, unfindraftimg},If[MemberQ[expr, "UnderDevelopment", Infinity] || 
  	      TrueQ["Underdevelopment"/.info],
     langext = language /. {"English"->"", "Japanese"->"_jp", "ChineseSimplified"->"_zh", "Spanish"->"_es" };
     unfindraftimg = Import[ToFileName[{$DocumentationBuildDirectory,
        "Internal", "online", "Images"}, "RefUnfinishedDraft"<>langext<>".m"]];
     ReplaceAll[expr, c : Cell[___, "HowToTitle", ___] :> 
       Cell[TextData[{c, Cell[BoxData[ToBoxes[Spacer[10]]]], Cell[BoxData[ToBoxes[unfindraftimg]]]}], "HowToTitleGrid"]],
     expr]];
  
    (*expr = expr /. Cell[con_,
      sty:Alternatives@@("CapitalizeHeadingStyles" /. {opts} /. Options[MakeNotebook]),
      o___] :>
    Cell[TextData@WrapInOpenCloseButton@CapitalizeString[con], sty, o];*)
    
    expr = expr /. Cell[con_,
      sty:Alternatives@@("CapitalizeHeadingStyles" /. {opts} /. Options[MakeNotebook]),
      o___] :>
    Cell[TextData@WrapInOpenCloseButton@CapitalizeString[con], sty, o];
    
    BuildLog["-- Localize MoreAbout on howtos"];
    expr = expr /. Cell[c_, "MoreAboutSection", o___] :>
      Cell[Localized["MoreAboutSectionText", language], "MoreAboutSection", o];

    BuildLog["-- Localize Related Tutorials on howtos"];
    expr = expr /. Cell[c_, "TutorialsSection", o___] :> 
      Cell[Localized["TutorialsSectionText", language], "TutorialsSection", o];
      
    BuildLog["-- Localize Related Links on howtos"];
    expr = expr /. Cell[c_, "RelatedLinksSection", o___] :> 
      Cell[Localized["RelatedLinksSectionText", language], "RelatedLinksSection", o];
      
    BuildLog["-- Localize SeeAlso on howtos"];
    expr = expr /. Cell[c_, "SeeAlsoSection", o___] :> 
      Cell[Localized["SeeAlsoSectionText", language], "SeeAlsoSection", o];

expr = DeleteCases[expr, Cell[_,  "LinkTrail", ___], Infinity];
  

  expr
  ];


(******************************************************************************)
(** ConvertNotebook - default **)
ConvertNotebook[expr_, info_, _, opts___?OptionQ] := expr;


(******************************************************************************)
(** EnsureCellGroupData - Internal **)
EnsureCellGroupData[expr_, type_, language_String:"English"] :=
Module[{allsects, sects, pos, p, nbexpr, newcell, presects, postsects},
  allsects = {"ObjectName", "ObjectNameSmall", "NotesSection", "PrimaryExamplesSection", "SeeAlsoSection",
              "TutorialSection", "RelatedDemonstrationsSection", "RelatedLinksSection",
              "MoreAboutSection", "History", "MoreToExploreSection", "MoreInformationSection"};
  If[Count[expr, Cell[_, type, ___], Infinity] =!= 0, expr,
    p = Position[allsects, type];
    If[p === {},
      presects = postsects = allsects,
      presects = Take[allsects, p[[1,1]] ];
      postsects = Drop[allsects, p[[1,1]] ] ];
    newcell = Cell[CellGroupData[{
      Cell[DocumentationBuild`Utils`CapitalizeString[
        Localized[type<>"Text", language]], type]}, Open]];

    (* We allow the styles not to be in groups, but this will get use both
       the group and the leading cells.  Position is post-order, so we can
       safely take the Last one. *)
    pos = Position[expr,
      Cell[CellGroupData[{Cell[_, Alternatives@@presects, ___], ___}, ___], ___]
       | Cell[_, Alternatives@@presects, ___],
      Infinity];

    If[pos =!= {},
      (* Take the Last and change it so we insert after *)
      pos = Last[pos];
      pos = Append[Array[0&, Length[pos] - 1], 1] + pos,

      (* Else *)
      pos = Position[expr,
        Cell[CellGroupData[{Cell[_, Alternatives@@postsects, ___], ___}, ___], ___]
         | Cell[_, Alternatives@@presects, ___],
        Infinity];
      (* Check if the first position is "inside" the second position.  If
         so, then the first is just the lead cell of the group that is the
         second.  Take the second. *)
      If[pos === {},
        pos = Position[expr, Cell[___, "FooterCell", ___], Infinity],
        If[Length[pos] > 1
            && (Length[pos[[1]]] > Length[pos[[2]]])
            && (Take[pos[[1]], Length[pos[[2]]]] === pos[[2]]),
          pos = pos[[2]],
          pos = pos[[1]]
      ]];
    ];
    If[pos === {},
      nb = ReplacePart[expr, {1} -> Append[First[expr], newcell]],
      (* Else *)
      nb = Insert[expr, newcell, pos]
    ];
    nb]
];


(******************************************************************************)
(** GetNotesGroup - Internal **)
(* get the notes cell group and make necessary adjustments *)
GetNotesGroup[expr_, type_, language_String:"English", opts___?OptionQ] :=
Module[{notesGroup, functionPaclet, hasopts},
  functionPaclet = "FunctionPaclet" /.  {opts} /. Options[MakeNotebook];
  hasopts = "HasOptions" /. Flatten[{opts}] /. ("HasOptions" -> False);
  notesGroup = 
    Cases[expr, 
      Cell[CellGroupData[{
        Cell[_, "ObjectName"|"ObjectNameSmall"|"ObjectNameAlt"|"CharacterImage", ___],
        ___}, ___], ___],
      Infinity, 1];
  notesGroup = DeleteCases[notesGroup, Cell[_, "ObjectName" | "ObjectNameSmall" | "ObjectNameAlt" | "Usage" | "CharacterImage"| "CharacterName" | "Elucidation", ___], Infinity];
  notesGroup = If[notesGroup === {Cell[CellGroupData[{}, Open]]}, {}, notesGroup];
  (* stripping out CellGroupData (we were relying on the style sheet to do this in the past, but
     it seems to have gotten unreliable of late) *)
  notesGroup = Replace[notesGroup, {Cell[CellGroupData[cells_, ___], ___]} :> cells];
  (* add 'more information' heading for notes *)
  notesGroup = Which[MemberQ[{"Character Name", "Format", "Indicator", "Device Connection", "Service Connection"},type], notesGroup,
  	True,
  notesGroup = If[Length[notesGroup] > 0,
    Cell[CellGroupData[Flatten[{Cell[TextData[{ 
      Cell@BoxData@WrapInOpenCloseButton[
        If[functionPaclet,
          ToBoxes[Import[ToFileName[
            {$DocumentationBuildDirectory, "Internal", "web", "html", "images", "mathematicaImages"},
            "MoreInformation.jpg"], "JPG"]],
          (* V9 EDIT BELOW *)
          
            Cell[TextData[{Cell[BoxData[ToBoxes[Spacer[6]]]],
              If[TrueQ[hasopts],Localized["NotesSectionText", language],
                Localized["NotesSectionNoOptsText", language]]}], "NotesFrameText"]
            
        ]]
      }], "NotesSection",
      CellGroupingRules->{"SectionGrouping", 50}, CellID->1 ], notesGroup},1], Open]],
    {}];
  notesGroup]
];


(******************************************************************************)
(** GetImportExportGroup - Internal **)
(* get the Import/Export cell group and make necessary adjustments *)
GetImportExportGroup[expr_, language_String:"English"] := Module[{importExportGroup},
  importExportGroup = 
    Cases[expr, 
      Cell[CellGroupData[{
        Cell[_, "ImportExportSection", ___],
        ___}, ___], ___],
      Infinity, 1];
  importExportGroup= importExportGroup /. Cell[a_,"ImportExportSection",b_] :> 
  		Cell[Localized[Switch[a, "Import", "ImportSectionText", "Export", "ExportSectionText", _, "ImportExportSectionText"], language],"ImportExportSection",b];
  importExportGroup
];


(******************************************************************************)
(** GetElementGroup - Internal **)
(* get the ElementsSection cell group and make necessary adjustments *)
GetElementGroup[expr_, language_String:"English"] := Module[{elementGroup},
  elementGroup = 
    Cases[expr, 
      Cell[CellGroupData[{
        Cell[_, "ElementsSection", ___],
        ___}, ___], ___],
      Infinity, 1];
  elementGroup = elementGroup /. Cell[_,"ElementsSection",b_] :> 
  		Cell[Localized["ElementsSectionText", language],"ElementsSection",b];
  elementGroup
];


(******************************************************************************)
(** GetNotebookInterfaceGroup - Internal **)
(* get the NotebookInterfaceSection cell group and make necessary adjustments *)
GetNotebookInterfaceGroup[expr_, language_String:"English"] := Module[{notebookInterfaceGroup},
  notebookInterfaceGroup = 
    Cases[expr, 
      Cell[CellGroupData[{
        Cell[_, "NotebookInterfaceSection", ___],
        ___}, ___], ___],
      Infinity, 1];
  notebookInterfaceGroup = notebookInterfaceGroup /. Cell[_,"NotebookInterfaceSection",b_] :> 
  		Cell[Localized["NotebookInterfaceSectionText", language],"NotebookInterfaceSection",b];
  notebookInterfaceGroup
];


(******************************************************************************)
(** GetOptionsGroup - Internal **)
(* get the OptionsSection cell group and make necessary adjustments *)
GetOptionsGroup[expr_, language_String:"English"] := Module[{optionsGroup},
  optionsGroup = 
    Cases[expr, 
      Cell[CellGroupData[{
        Cell[_, "OptionsSection", ___],
        ___}, ___], ___],
      Infinity, 1];
  optionsGroup = optionsGroup /. Cell[_,"OptionsSection",b_] :> 
  		Cell[Localized["OptionsSectionText", language],"OptionsSection",b];
  optionsGroup
];

(******************************************************************************)
(** GetOtherSectionGroup - Internal **)
(* get the OptionsSection cell group and make necessary adjustments *)
GetOtherSectionGroup[expr_, language_String:"English"] := Module[{sectionGroup,styles},
  styles = {"ServiceSubsection","DeviceSubsection","InterpreterSection","EmbeddingFormatSection"};
  sectionGroup = 
    Cases[expr, 
      Cell[CellGroupData[{
        Cell[_, Alternatives@@styles, ___],
        ___}, ___], ___],
      Infinity];
  sectionGroup
];

(******************************************************************************)
(** GetOptionsGroup - Internal **)
(* get the OptionsSection cell group and make necessary adjustments *)
GetTutorialsGroup[expr_, language_String:"English"] := Module[{tutorialsGroup},
  tutorialsGroup = 
    Cases[expr, 
      Cell[CellGroupData[{
        Cell[_, "TutorialsSection", ___],
        ___}, ___], ___],
      Infinity, 1];
  tutorialsGroup = tutorialsGroup /. Cell[_,"TutorialsSection",b_] :> 
          (* V9 EDIT BELOW *)
  		Cell[Localized["TutorialsSectionText", language],"TutorialsSection",b];
  tutorialsGroup
];

(******************************************************************************)
(** GetRelatedLinksGroup - Internal **)
(* get the OptionsSection cell group and make necessary adjustments *)
GetRelatedLinksGroup[expr_, language_String:"English"] := Module[{relatedLinksGroup},

  relatedLinksGroup = 
    Cases[expr, 
      Cell[CellGroupData[{
        Cell[_, "RelatedLinksSection", ___],
        ___}, ___], ___],
      Infinity, 1];
      
  relatedLinksGroup = relatedLinksGroup /. Cell[_,"RelatedLinksSection",b_] :> 
          (* V9 EDIT BELOW *)
  		Cell[Localized["RelatedLinksSectionText", language],"RelatedLinksSection",b];
  relatedLinksGroup
];


(******************************************************************************)
(** GetOptionsGroup - Internal **)
(* get the OptionsSection cell group and make necessary adjustments *)
GetMoreAboutGroup[expr_, language_String:"English"] := Module[{moreAboutGroup},

  moreAboutGroup = 
    Cases[expr, 
      Cell[CellGroupData[{
        Cell[_, "MoreAboutSection", ___],
        ___}, ___], ___],
      Infinity, 1];
      
  moreAboutGroup = moreAboutGroup /. Cell[_,"MoreAboutSection",b___] :> 
          (* V9 EDIT BELOW *)
  		Cell[StringReplace[Localized["MoreAboutSectionText", language], {" \[RightGuillemet]" | "\[RightGuillemet]"->""}],"MoreAboutSection",b];
  moreAboutGroup
];


(******************************************************************************)
(** GetSeeAlsoGroup - Internal **)
(* get the see Also cell group and make necessary adjustments *)
GetSeeAlsoGroup[expr_, language_String:"English"] := Module[{seeAlsoGroup},
  seeAlsoGroup = 
    Cases[expr, 
      Cell[CellGroupData[{
        Cell[_ , "SeeAlsoSection", ___],
        ___}, ___], ___],
      Infinity, 1];
  seeAlsoGroup = If[seeAlsoGroup === {Cell[CellGroupData[{}, Open]]}, {}, seeAlsoGroup];
  seeAlsoGroup = seeAlsoGroup /. Cell[_, "SeeAlsoSection", o___] :> 
    (* V9 EDIT BELOW *)
    Cell[StringReplace[Localized["SeeAlsoSectionText", language], {" \[RightGuillemet]" | "\[RightGuillemet]"->""}], "SeeAlsoSection", o];
  seeAlsoGroup
];


 (**	 
     GetTutorialCollectionGroup	 
     get the TutorialCollection group and make necessary adjustments	 
 *)	 
 GetTutorialCollectionGroup[expr_, language_String:"English"] := Module[{tutorialCollectionGroup},	 
     tutorialCollectionGroup =	 
         Cases[expr,	 
             Cell[CellGroupData[{	 
                 Cell[_ , "TutorialCollectionSection", ___],	 
                 ___}, ___], ___], Infinity, 1];	 
     tutorialCollectionGroup = If[tutorialCollectionGroup === {Cell[CellGroupData[{}, Open]]}, {}, tutorialCollectionGroup];	 
     tutorialCollectionGroup = tutorialCollectionGroup /. Cell[c_String, "TutorialCollectionSection", o___] :>	 
       Cell[c, "TutorialCollectionSection", o];	 
 tutorialCollectionGroup];	 
 
  	 	 
 (**	 
     GetGuideCollectionGroup	 
     get the TutorialCollection group and make necessary adjustments	 
 *)	 
 GetGuideTutorialCollectionGroup[expr_, language_String:"English"] := Module[{guideTutorialCollectionGroup},	 
     guideTutorialCollectionGroup =	 
         Cases[expr,	 
             Cell[CellGroupData[{	 
                 Cell[_ , "GuideTutorialCollectionSection", ___],	 
                 ___}, ___], ___], Infinity, 1];	 
     guideTutorialCollectionGroup = If[guideTutorialCollectionGroup === {Cell[CellGroupData[{}, Open]]}, {}, guideTutorialCollectionGroup];	 
     guideTutorialCollectionGroup = guideTutorialCollectionGroup /. Cell[c_String, "GuideTutorialCollectionSection", o___] :>	 
       Cell[ c, "GuideTutorialCollectionSection", o];	 
 guideTutorialCollectionGroup];
 
 (******************************************************************************)
(** GetIndicatorExampleGroup - Internal **)
(* get the GetIndicatorExampleGroup cell group and make necessary adjustments *)
GetIndicatorExampleGroup[expr_, language_String:"English"] := Module[{indicatorexampleGroup},
  indicatorexampleGroup = 
    Cases[expr, 
      Cell[CellGroupData[{
        Cell[_, "IndicatorExampleSection", ___],
        ___}, ___], ___],
      Infinity, 1];
  indicatorexampleGroup = indicatorexampleGroup /. Cell[_,"IndicatorExampleSection",b_] :> 
  		Cell[Localized["IndicatorExampleSectionText", language],"IndicatorExampleSection",b];
  indicatorexampleGroup
];

 (******************************************************************************)
(** GetIndicatorDescriptionGroup - Internal **)
(* get the GetIndicatorDescriptionGroup cell group and make necessary adjustments *)
GetIndicatorDescriptionGroup[expr_, language_String:"English"] := Module[{indicatordescriptionGroup},
  indicatordescriptionGroup = 
    Cases[expr, 
      Cell[CellGroupData[{
        Cell[_, "IndicatorDescriptionSection", ___],
        ___}, ___], ___],
      Infinity, 1];
  indicatordescriptionGroup = indicatordescriptionGroup /. Cell[_,"IndicatorDescriptionSection",b_] :> 
  		Cell[Localized["IndicatorDescriptionSectionText", language],"IndicatorDescriptionSection",b];
  indicatordescriptionGroup
];

 (******************************************************************************)
(** GetIndicatorFormulaGroup - Internal **)
(* get the GetIndicatorFormulaGroup cell group and make necessary adjustments *)
GetIndicatorFormulaGroup[expr_, language_String:"English"] := Module[{indicatorformulaGroup},
  indicatorformulaGroup = 
    Cases[expr, 
      Cell[CellGroupData[{
        Cell[_, "IndicatorFormulaSection", ___],
        ___}, ___], ___],
      Infinity, 1];
  indicatorformulaGroup = indicatorformulaGroup /. Cell[_,"IndicatorFormulaSection",b_] :> 
  		Cell[Localized["IndicatorFormulaSectionText", language],"IndicatorFormulaSection",b];
  indicatorformulaGroup
];


(******************************************************************************)
(** GetExamplesGroup - Internal **)
(* get the examples cell group and make necessary adjustments *)
GetExamplesGroup[expr_, language_String:"English"] := Module[
  {primaryExamplesGroup, extendedExamplesGroup, examplesGroup},

  (* Extract primary and extended groups *)
  primaryExamplesGroup = 
    Cases[expr,
      Cell[CellGroupData[{
        Cell[_, "PrimaryExamplesSection", ___],
        ___}, ___], ___],
      Infinity, 1];
  extendedExamplesGroup = 
    Cases[expr,
      Cell[CellGroupData[{
        Cell[_, "ExtendedExamplesSection", ___],
        ___}, ___], ___],
      Infinity, 1];

  (* Remove leading cells *)
  primaryExamplesGroup = Replace[primaryExamplesGroup,
    Cell[CellGroupData[l:{Cell[_, "PrimaryExamplesSection", ___], ___}, _], ___]
      :> Rest[l], Infinity];
  extendedExamplesGroup = Replace[extendedExamplesGroup,
    Cell[CellGroupData[l:{Cell[_, "ExtendedExamplesSection", ___], ___}, _], ___]
      :> Rest[l], Infinity];

  (* Assemble a single "Examples" group *)
  examplesGroup = 
    Cell[CellGroupData[Flatten[{
      Cell[Localized["PrimaryExamplesSectionText", language], "PrimaryExamplesSection",
        CellTags->"PrimaryExamplesSection", CellID->1],
      Cell[CellGroupData[Flatten@{
        Cell[Localized["BasicExamplesSectionText", language], "ExampleSection",CellID->1],
        primaryExamplesGroup}, Open]],
      extendedExamplesGroup
    }], Open]];

  (* Remove 'Interactive Examples' *)
  (* If[buildmode==="Future",
  examplesGroup,
  examplesGroup = DeleteCases[examplesGroup,
    Cell[CellGroupData[{
        Cell[BoxData[InterpretationBox[
          Cell["Interactive Examples", "ExampleSection", ___],___]], "ExampleSection",___],
        ___},___],___],
      Infinity]]; *)
      

  (* Remove ExampleDelimiter which follow *Section headings *)
  examplesGroup = DeleteCases[examplesGroup, Cell[CellGroupData[{},Open]], Infinity];
  examplesGroup = DeleteCases[examplesGroup, Cell[CellGroupData[{},Open]], Infinity];
  examplesGroup = DeleteCases[examplesGroup, Cell[CellGroupData[{},Open]], Infinity];
      examplesGroup = examplesGroup //. (
    Cell[CellGroupData[{aa___,
      Cell[_, "ExampleDelimiter", ___],
      Cell[CellGroupData[{
        Cell[a_, "ExampleDelimiter", b___], c___}, d___], e___], f___}, g___], h___]
    :>
    Cell[CellGroupData[{aa,
      Cell[CellGroupData[{
        Cell[a, "ExampleDelimiter", b], c}, d], e], f}, g], h] ); 
        
    examplesGroup = examplesGroup //. (
    Cell[CellGroupData[{aa___,
      Cell[CellGroupData[{Cell[_, "ExampleDelimiter", ___]}, ___],___],
      Cell[CellGroupData[{
        Cell[a_, "ExampleDelimiter", b___], c___}, d___], e___], f___}, g___], h___]
    :>
    Cell[CellGroupData[{aa,
      Cell[CellGroupData[{
        Cell[a, "ExampleDelimiter", b], c}, d], e], f}, g], h] ); 
  
    examplesGroup = examplesGroup //. (
    Cell[CellGroupData[{
      Cell[con_, sty:("ExampleSection" | "ExampleSubsection" | "ExtendedExamplesSection"), o___],
      Cell[_, "ExampleDelimiter", ___], d___}, e___], f___]
    :>
    Cell[CellGroupData[{
        Cell[con, sty, o], d}, e], f] ); 
        
     examplesGroup = examplesGroup //. (
    Cell[CellGroupData[{
      Cell[con_, sty:("ExampleSection" | "ExampleSubsection" | "ExtendedExamplesSection"), o___],
      Cell[CellGroupData[{
        Cell[_, "ExampleDelimiter", ___]}, b___], c___], d___}, e___], f___]
    :>
    Cell[CellGroupData[{
        Cell[con, sty, o], d}, e], f] ); 
        
   examplesGroup = examplesGroup //. (
    Cell[CellGroupData[{
      Cell[con_, sty:("ExampleSection" | "ExampleSubsection" | "ExtendedExamplesSection"), o___],
      Cell[CellGroupData[{
        Cell[_, "ExampleDelimiter", ___], a___}, b___], c___], d___}, e___], f___]
    :>
    Cell[CellGroupData[{
        Cell[con, sty, o],
        Cell[CellGroupData[{ a}, b], c], d}, e], f] );

  (* Close ExamplesSection groups *)
  examplesGroup = examplesGroup /. (
    Cell[CellGroupData[{
      Cell[con_, "ExampleSection", o___],
      cs___}, ___], g___]
    :> 
    Cell[CellGroupData[{
      Cell[con, "ExampleSection", "ExampleSection", o], cs },
      If[(StringQ[con] && StringMatchQ[con, Localized["ExtendedExamplesSectionText", language]]),
        Open, Closed]], g] );

  (* Close ExamplesSubsection groups *)
  examplesGroup = examplesGroup /. (
    Cell[CellGroupData[{
      Cell[con_, "ExampleSubsection", o___],
      cs___}, ___], g___]
    :> 
    Cell[CellGroupData[{
      Cell[con, "ExampleSubsection", "ExampleSubsection", o], cs }, Closed], g] );

  (* If no examples, do not include groups *)
  examplesGroup = 
    If[MatchQ[examplesGroup,
      Cell[CellGroupData[{
        Cell[Localized["PrimaryExamplesSectionText", language],
          "PrimaryExamplesSection", CellTags->"PrimaryExamplesSection"]
      }, ___], ___]],
      {},
      examplesGroup
    ];

  examplesGroup];


(******************************************************************************)
(** MakeAnchorBarCell - Internal **)
Options[MakeAnchorBarCell] = {
    "NewIn"->False,
    "IncludeSections"->Automatic,
    "BuildMode" -> "Normal",
    "PacletName" -> "Mathematica"
};
MakeAnchorBarCell[expr_, info_, language_String:"English", opts___?OptionQ] :=
Module[{
    func, styles, list, am, uri, buttonList = {}, buildmode, stylesOrder, pacletname, lclstr, wsmlinkQ,
    spacer = "\[ThickSpace]\[ThickSpace]\[ThickSpace]\[ThickSpace]\[ThickSpace]\[ThickSpace]", translationglobeimg
  },

  BuildLog["** Entering MakeAnchorBarCell **"];
  styles = "IncludeSections" /. {opts} /. Options[MakeAnchorBarCell]; 
  buildmode = "BuildMode" /. {opts} /. Options[MakeAnchorBarCell];
  pacletname = "PacletName" /. {opts} /. Options[MakeAnchorBarCell];
  styles = If[ styles === Automatic, {"MoreAbout", "SeeAlso", "Tutorials" }, styles];
  stylesOrder=If[("EntityType" /. info)==="Tutorial",styles,Reverse@styles];
  translationglobeimg = Import[FileNameJoin[{$DocumentationBuildDirectory, "Internal", "online", "Images", "TranslationGlobe.m"}]];

  BuildLog["-- buttonList: "<>ToString[buttonList] ];
  (* FIXME: localization *)
  Scan[Function[{style},
    BuildLog["-- "<>style];
      lclstr = Switch[style,
          "PrimaryExamplesSection", "ExamplesSection",
          "GuideFunction", "GuideFunction",
          "GuideMoreAbout", "GuideMoreAboutSection",
          "GuideTutorial", "GuideTutorialsSection",
          "MoreAbout", "MoreAboutSection",
          "SeeAlso", "SeeAlsoSection",
          "Tutorials", "TutorialsSection",
          _, style
          ];
      func = ToExpression@StringJoin["Get", lclstr, "List"];
      list = func[expr, "IncludePaclets"->True, "PacletName" -> ("PacletName" /. info), "Language"->language, "BuildMode"->buildmode, "PacletName"->pacletname ];
      BuildLog["-- "<>ToString@{func, list}];
      If[ !MemberQ[{ {{}}, {}}, list],
        am = Cell[BoxData@ToBoxes@MakeActionMenu[
          TextCell[Row[{Localized[lclstr<>"Text", language], Style[" \[FilledDownTriangle]", "AnchorBarArrow"]}]], style,
          list], LineSpacing->{1.4, 0}];
        buttonList = Join[buttonList, {am}]
    ]], stylesOrder];

  (* add function browser button *)
  buttonList =
    If[False,
      BuildLog["-- Adding function browser button"];
      Append[buttonList, Cell[TextData["function browser"]] ],
      BuildLog["-- No function browser button"];
      buttonList
    ];

  (* add url menu *)
  uri = "URI" /. info;
  wsmlinkQ = StringMatchQ[uri, "*WSMLink/*"];
  buttonList = Append[buttonList,
    Cell[BoxData[ToBoxes@ActionMenu[Framed[TextCell[Row[{Localized["url", language], Style[" \[FilledDownTriangle]", "AnchorBarArrow"]}]]], Flatten@{
      StringReplace[uri, "paclet:" -> ""] :> None,
      With[{uri=uri},
        Localized["Copy Mathematica url", language] :>
          Module[{DocumentationSearch`Private`nb}, 
            DocumentationSearch`Private`nb = 
              NotebookPut[Notebook[{Cell[uri]}, Visible -> False]];
            SelectionMove[DocumentationSearch`Private`nb, All, Notebook];
            FrontEndTokenExecute[DocumentationSearch`Private`nb, "Copy"];
            NotebookClose[DocumentationSearch`Private`nb];
      ]],
      If[ !StringMatchQ[uri, "*ref/message/*"], 
        {
        Delimiter,
        With[{uri=StringJoin[If[TrueQ[wsmlinkQ], "http://reference.wolfram.com/system-modeler/", 
        	"http://reference.wolfram.com/language/"], uri, ".html"]},
          Localized["Copy web url", language] :>
            Module[{DocumentationSearch`Private`nb}, 
              DocumentationSearch`Private`nb = NotebookPut[
                Notebook[{Cell[BoxData@MakeBoxes[Hyperlink[uri], StandardForm], 
                  "Input", System`TextClipboardType -> "PlainText"]}, Visible -> False]];
              SelectionMove[DocumentationSearch`Private`nb, All, Notebook];
              FrontEndTokenExecute[DocumentationSearch`Private`nb, "Copy"];
              NotebookClose[DocumentationSearch`Private`nb];
        ]],
        With[{uri=uri},
          Localized["Go to web url", language] :>
            FrontEndExecute[{NotebookLocate[{URL[
              StringJoin[If[TrueQ[wsmlinkQ], "http://reference.wolfram.com/system-modeler/", 
        	"http://reference.wolfram.com/language/"], uri, ".html"]
            ], None}]}] ]
        },
        (* Else *)
        {} ]
    }, Appearance -> None, MenuAppearance -> Automatic, MenuStyle -> "URLMenu"]], LineSpacing->{1.4, 0}]];
  
  (* Add translation language menu *)
  (* Removed for 10.1 release -- 2/24/15 *)
  (*If[MatchQ["EntityType" /. info, "Symbol"],
    buttonList = Append[buttonList,
    Cell[BoxData[ToBoxes@ActionMenu[Framed[TextCell[Row[{RawBoxes[AdjustmentBox[translationglobeimg, BoxBaselineShift -> -0.25]], Style[" \[FilledDownTriangle]", "AnchorBarArrow"]}]]], Flatten@{
      Map[Row[{Checkbox[Dynamic[CurrentValue[EvaluationNotebook[], {TaggingRules, "TranslationLanguage"}] === #]], "  "<>#}] :>
        (CurrentValue[EvaluationNotebook[], {TaggingRules, "TranslationLanguage"}] = #)&, translationLanguageList]
    }, Appearance -> None, MenuAppearance -> Automatic, MenuStyle -> "TranslationMenu"]], LineSpacing->{1.4, 0}]]
  ];*)

  buttonList = Flatten[{ Riffle[buttonList, spacer] }];
  BuildLog["-- "<>ToString[Cases[buttonList, ButtonBox[c_,___]:>c,Infinity]] ];
  BuildLog["** Leaving MakeAnchorBarCell **"];

  If[(Length@buttonList > 0) && (First[buttonList] =!= " "),
    Cell[TextData[buttonList], "AnchorBar"], " "]
];

(******************************************************************************)
(** MakeInlineSeparator - Internal **)
MakeInlineSeparator[Cell[TextData[lst_List], o___]] :=
  Module[{new},
    new = Map[MakeInlineSeparatorString, lst];
    Cell[TextData[Flatten@new], o]
  ];
MakeInlineSeparator[c___] := c;  

MakeInlineSeparatorString[str_String] :=
 Sequence @@ 
  StringReplace[
   str, {RegularExpression[
      "(.*)\\s+\[EmptyVerySmallSquare]\\s+(.*)"] :> {"$1", "\[NonBreakingSpace]", 
      StyleBox["\[MediumSpace]\[FilledVerySmallSquare]\[MediumSpace]", "InlineSeparator"], " ", 
      "$2"}}]
MakeInlineSeparatorString[c___] := c;


(******************************************************************************)
(** AdjustModInfo - Internal **)

AdjustModInfo[nb_, info_, opts___?OptionQ] :=
  Module[{expr = nb, poss, buildmode},
  	buildmode = "BuildMode" /. {opts} /. Options[MakeNotebook]; 
    BuildLog["-- Apply ModInfo across tables"];
    poss = Position[expr, Cell[_, "2ColumnTableMod"|"3ColumnTableMod"|"3ColumnIllustratedTableMod", ___], Infinity];
    (*Global`AntLog["Before Scan[]"];*)
    Scan[Function[{pos}, Module[{note, cell, mod, gridp, grid, sp},
      (*Global`AntLog["Running..."];*)
      note = Extract[expr, Append[Most[pos], Last[pos] - 1]];
      If[MatchQ[note, Cell[___,
          CellDingbat~(Rule|RuleDelayed)~Cell[s_String /; (CheckVersion[s, opts] > 0),
          "ModInfo", ___], ___]],
        mod = CellDingbat /. List@@Cases[note, _~(Rule|RuleDelayed)~_];
        cell = Extract[expr, pos];
        gridp = Position[cell, _GridBox, Infinity];
        If[Length[gridp] > 0,
          gridp = First[gridp];
          grid = Extract[cell, gridp];
          sp = False;
          expr = ReplacePart[expr, Join[pos, gridp] ->
              GridBox[Map[
                  Function[{row}, Module[{nrow},
                    nrow = Join[
                      If[sp && (row[[2]] === ""),
                        {First[row]},
                        {mod}],
                      Rest[row]];
                    If[Last[nrow] === "\[SpanFromLeft]", sp = True, sp = False];
                    nrow
                  ]],
                  First[grid]],
                Sequence@@Rest[grid]
          ]];
      ]];
    ]], Select[poss, (Last[#] > 1)&]];
    (*Global`AntLog["After Scan[]"];*)
    (* remove ModInfo prior to threshhold *)
    expr = If[buildmode==="Future", expr,
    	expr /.
      Cell[c_String /; (CheckVersion[c, opts] === 0), "ModInfo", ___]
        :> Cell["      ", "ModInfo"]
    ];
    expr = expr /.
      Cell[con___,
          CellDingbat~(Rule|RuleDelayed)~Cell[c_String /; (CheckVersion[c, opts] === 1),
          "ModInfo", ___], o___]
      :> Cell[con, o,
        Background->Dynamic[If[CurrentValue[{TaggingRules, "ModificationHighlight"}]===True,
          RGBColor[1, 0.988235, 0.721569], None]]];

    (*Global`AntLog["Usage cell"];*)
    BuildLog["-- Remove ModInfo from Usage cell"];
    expr = AdjustModInfoUsageCells[expr, info, opts];

    (*Global`AntLog["Tables cells"];*)
    BuildLog["-- Remove ModInfo from tables"];
    expr = AdjustModInfoTables[expr, info, opts];
    expr = DeleteCases[expr, Cell[BoxData[GridBox[{},___]], ___], Infinity];

    (*Global`AntLog["Remove tentatives"];*)
    (* Remove "FutureExample" | "TentativeExample" tagged cells *)
    BuildLog["-- Remove TentativeExample tagged cells"];
    Which[
    buildmode==="Future",
    expr,
    ("Guide"===("EntityType"/.info))&&("Underdevelopment"/.info),
    expr = DeleteCases[expr,
      Cell[CellGroupData[{Cell[___, "FutureExample" | "TentativeExample"(*|"ExcludedMaterial"*), ___],
        ___}, ___], ___],
      Infinity];
    expr = DeleteCases[expr, Cell[___, "ExcludedMaterial", ___],Infinity];
    expr = DeleteCases[expr, Alternatives[
        Cell[___, "FutureExample" | "TentativeExample" | "ExcludedMaterial", ___],
        Cell[__, CellDingbat->Cell[__, "BrighterFlag", ___], ___]
      ],
      Infinity];
      expr=expr/.Cell[a__, CellDingbat->Cell[__, "TentativeID", b___], c___]:>Cell[a, c],
    True,
    expr = DeleteCases[expr,
      Cell[CellGroupData[{Cell[___, "FutureExample" | "TentativeExample" (*| "ExcludedMaterial"*), ___],
        ___}, ___], ___],
      Infinity];
    expr = DeleteCases[expr, Alternatives[
        Cell[___, "FutureExample" | "TentativeExample" | "ExcludedMaterial", ___],
        Cell[__, CellDingbat->Cell[__, "TentativeID" | "BrighterFlag", ___], ___],
        Cell[__, CellDingbat->Cell[s_String /; (CheckVersion[s, opts] === Infinity), "ModInfo", ___], ___]
      ],
      Infinity]
    ];

    expr = expr /.
      Cell[c_String /; (CheckVersion[c, opts] > 0), "TableRowIcon", o___] :> 
        Cell["\[FilledVerySmallSquare]", "TableRowIcon", FontColor->None, o];

    (*Global`AntLog["Remove empty ModInfo"];*)
    (* remove empty ModInfo *)
    BuildLog["-- Remove empty ModInfo"];
    expr = expr /.
      Cell[a___, Rule[CellDingbat, Cell["", "ModInfo", ___]], h___] :> Cell[a,h];

    expr];


(******************************************************************************)
(** AdjustModInfoUsageCells **)

AdjustModInfoUsageCells[nb_, info_, opts___?OptionQ] :=
  Module[{expr = nb, poss, repl, usageNameCell},
    BuildLog["** Entering AdjustModInfoUsageCells **"];
    poss = Position[expr, Cell[_, "Usage"|"DeviceDoubleUsage", ___], Infinity];
    repl = Reap[Scan[
      Function[{pos}, Module[{cexpr, spl},
        cexpr = Extract[expr, pos];
        If[Head[First[cexpr]] === TextData,
          spl = Split[cexpr[[1, 1]],
            (!StringQ[#1] || (StringCount[#1, "\n"] == 0))&];
          spl = Map[Function[{sp}, If[
            MatchQ[First[sp],
              Cell[c_String /;
                  !(StringMatchQ[c, RegularExpression["\\s*"]] || CheckVersion[c, opts] =!= Infinity),
                "ModInfo", ___] ],
            Module[{str = Last[sp],
                nl = (StringQ[str] && StringPosition[str, "\n"])},
              If[StringQ[str] && (Length[nl] > 0) &&
                  (nl[[1, 1]] =!= StringLength[str]),
                {StringDrop[str, nl[[1, 1]]]},
                {}]],
            sp]],
            spl];
          spl = Flatten[spl];
          If[StringQ[Last[spl]], Module[{str, nl},
            str = Last[spl];
            nl = StringPosition[str, "\n"];
            If[(Length[nl] != 0) && (nl[[-1, 1]] == StringLength[str]),
              spl[[-1]] = StringDrop[str, -1]]
          ]];
          cexpr = ReplacePart[cexpr, {1, 1} -> spl];
          Sow[pos -> cexpr];
          (*,
          cexpr = Cell[BoxData[GridBox[{ {"", Cell[ First@cexpr ]} }]], "FormatUsage", CellID->1 ];
          Sow[pos -> cexpr];*)
        ]
    ]], poss]];

    If[Length[repl[[2]]] != 0,
      repl = repl[[2, 1]];
      expr = ReplacePart[expr, repl] ];

    poss = Position[expr, Cell[_, "Usage"|"DeviceDoubleUsage",___], Infinity];
    usageNameCell = Extract[expr, poss];
    
    (* Map over potential lists instead of assuming singleton *)
    (*If[Length[usageNameCell] > 0, usageNameCell = First@usageNameCell];*)
    If[!ListQ[usageNameCell], usageNameCell = {usageNameCell}];
    (* turn Usage cell into GridBox for easier ModInfo managemet *)
    Do[If[Head@usageNameCell[[i]] === Cell && Head@First@usageNameCell[[i]] === TextData,
      Module[{rows},
        rows = Split[usageNameCell[[i, 1, 1]],
          (!StringQ[#1] || (StringTake[#1, -1] =!= "\n"))&];
        rows = Map[
          If[StringQ[Last[#]] && StringTake[Last[#], -1] === "\n",
            Join[Most[#], {StringDrop[Last[#], -1]}],
            #]&,
          rows];
        usageNameCell[[i]] = ReplacePart[usageNameCell[[i]], {1} :> BoxData[GridBox[
          Map[
            If[MatchQ[First[#], Cell[_, "ModInfo", ___]],
              {First[#], Cell[TextData[Rest[#]]]},
              {"", Cell[TextData[#]]}
            ]&, rows]
        ]]];
        usageNameCell[[i]] = Append[usageNameCell[[i]], GridBoxOptions->{RowBackgrounds->
          Map[
            If[MatchQ[First[#], Cell[c_String /; (CheckVersion[c, opts] > 0), "ModInfo", ___]],
              Dynamic[If[CurrentValue[{TaggingRules, "ModificationHighlight"}]===True,
                RGBColor[1, 0.988235, 0.721569], None]],
              None]&,
            rows]
        }];
    ]];
    usageNameCell[[i]] = usageNameCell[[i]] /. Cell[_, "ModInfo", ___] :> "", {i, Length[usageNameCell]}];

    ReplacePart[expr, MapThread[Rule, {poss, usageNameCell}]]
  ];


(******************************************************************************)
(** AdjustModInfoTable **)
AdjustModInfoTables[nb_, info_, options___?OptionQ] :=
  Module[{expr = nb, poss, repl, buildmode},
  buildmode = "BuildMode" /. {options} /. Options[MakeNotebook]; 
  BuildLog["** Entering AdjustModInfoTables **"];
    poss = Position[expr,
      GridBox[{___, {Cell[_, "ModInfo", ___], ___}, ___}, ___],
      Infinity];
    repl = Reap[Scan[Function[{pos},
      Module[{drows, cpos, cexpr, cepos, gexpr, optpos},
        cpos = Last @ Select[(Take[pos, #]& /@ Range[Length[pos]]),
            (Head[First @ Extract[expr, {#}]] === Cell)& ];
        cepos = Drop[pos, Length[cpos]];
        gexpr = Extract[expr, pos];
        If[buildmode==="Future",
        drows={},
        drows = Position[gexpr,
            {Cell[_, "ModInfo", ___,  "BrighterFlag" | "TentativeExample" | "FutureExample", ___]
            |Cell[_, "ModInfo", ___, CellDingbat->Cell[___, "TentativeID" | "BrighterFlag", ___], ___],
            ___},
            Infinity]
        ];
        
        gexpr = If[
        	buildmode==="Future",
        	gexpr,
        	Delete[gexpr, drows]
        ];

        Scan[
            If[gexpr[[1, #, 1, 2]] === "ModInfo",
                gexpr[[1, #, 1, 2]] = "TableRowIcon"]&,
            Range @ Length @ First @ gexpr ];
        drows = Last /@ drows;

        cexpr = Extract[expr, cpos];
        cexpr = ReplacePart[cexpr, {cepos -> gexpr}];

        optpos = Position[cexpr, _[GridBoxOptions, _]];
        If[Length[optpos] != 0, Module[{opts, rowlp, rowl, spacp, spac},
            optpos = First[optpos];
            opts = Extract[cexpr, {optpos}][[1, 2]];

            rowlp = Position[opts, _[RowLines, _]];
            If[Length[rowlp] != 0,
                rowlp = First[rowlp];
                rowl = Extract[opts, {rowlp}][[1, 2]];
                If[ListQ[rowl],
                    rowl = Delete[
                        ReplacePart[rowl,
                            Map[(# -> Max[Extract[rowl, #-1], Extract[rowl, #]])&,
                                drows]],
                        {# - 1}& /@ drows];
                    opts = ReplacePart[opts, rowlp -> (RowLines -> rowl)];
            ]];

            spacp = Position[opts, _[RowSpacings, _]];
            If[Length[spacp] != 0,
                spacp = First[spacp];
                spac = Extract[opts, {spacp}][[1, 2]];
                If[ListQ[spac],
                    spac = Delete[
                        ReplacePart[spac,
                            Map[(# -> Max[Extract[spac, #-1], Extract[spac, #]])&,
                                drows]],
                        {# - 1}& /@ drows];
                    opts = ReplacePart[opts, spacp -> (RowSpacings -> spac)];
            ]];

            cexpr = ReplacePart[cexpr,
                optpos -> (GridBoxOptions -> opts)];
        ]];
        Sow[cpos -> cexpr];
    ]], poss]];
    BuildLog["** Leaving AdjustModInfoTables **"];

    If[Length[repl[[2]]] != 0,
        repl = repl[[2, 1]];
        ReplacePart[expr, repl],
        (* Else *)
        expr
  ]];


(******************************************************************************)
(** RenumberCells **)
(** 
    RenumberCells[nbexpr, {"section1", ...}]
    This function attempts to snag all the In[]/Out[] cell labels
    and renumber each "section" starting from 1.  In order to do 
    this, it must have the in/out cells within a CellGroupData[]
    expression where the first cell style is "section"

    in:         Notebook expression | notebook filename
    out:        modified Notebook expression
    options:    DelimiterStyles

**)

Options[RenumberCells] = {DelimiterStyles->{"ExampleDelimiter"}, InputStyles->{"Input"}}

RenumberCells[$Failed,___] = $Failed

RenumberCells[file_String, sections__String, opts___?OptionQ] :=
  RenumberCells[file,{sections}, opts]

RenumberCells[file_String, sections:{__String}, opts___?OptionQ] :=
  RenumberCells[Import[file,"NB"],sections, opts]

RenumberCells[nb_Notebook, sections__String, opts___?OptionQ] :=
  RenumberCells[nb, {sections}, opts]

RenumberCells[nb_Notebook, sections:{__String}, opts___?OptionQ] := 
  Module[{expr=nb,delStyles,chkDelimiter=False, inStyles},

    (* parse opts *)
    delStyles = DelimiterStyles /. {opts} /. Options[RenumberCells];
    inStyles = InputStyles /. {opts} /. Options[RenumberCells];

    If[MatchQ[delStyles, { __String }],
      chkDelimiter=True; delStyles=Alternatives@@delStyles];

    (* Map over each section style to renumber *)
    Scan[
      Function[{style}, Module[{pos,sectionCells},
        pos = Position[expr, CellGroupData[{Cell[___,style,___],___},___]];

        (* only process if we got something to do *)
        If[pos=!={},
          (* get sections to process *)
          sectionCells = Extract[expr, pos];
          (* loop over each cell group we found *)
          newSections = Map[
            Function[{scell}, Block[
                {stuff = scell[[1]], open = scell[[2]], count = 0, looper},
              looper = Function[{this},
                Which[
                  MatchQ[this,
                    Cell[CellGroupData[{Cell[___, "MathCaption", ___], ___}, ___], ___]],
                  ReplacePart[this, {1, 1} -> (looper /@ this[[1, 1]])],

                  (* match input output cell groups *)
                  MatchQ[this,
                    Cell[CellGroupData[{Cell[___, Alternatives@@inStyles, ___], ___}, ___], ___]],
                  SetCellInOutNumbers[this,++count,InputStyles->inStyles],

                  (* match lone innie / outties *)
                  MatchQ[this,
                    Cell[___,  Alternatives @@ Join[inStyles, {"Output"}],___]],
                  SetCellInOutNumbers[this,++count,InputStyles->inStyles],

                  (* reset @ specified delimiters if we need to *)
                  chkDelimiter && MatchQ[this,
                    Cell[___,delStyles,___]],
                  count = 0; this,

                  (* otherwise return unmodified *)
                  True,
                  this
              ]];
            CellGroupData[looper /@ stuff, open]
          ]], sectionCells];
         
          (* replace old sections with new *) 
          Scan[(expr=ReplacePart[expr,#[[1]]->#[[2]]])&, Transpose[{pos,newSections}]]
        ]
      ]],
      sections];

    (* return processed nb expr *)
    expr
  ];


(******************************************************************************)
(** SetCellInOutNumbers - Internal **)
SetCellInOutNumbers[expr_,num_Integer,funcopts___?OptionQ] :=
  Module[{old, inStyles},
    inStyles = InputStyles /. {funcopts} /. Options[RenumberCells];
    expr /. 
    {
      (* rename all Input cell labels *)
      Cell[stuff___,st:(Alternatives@@inStyles),opts___]:> (
        Cell[stuff,st,
          Sequence@@DeleteCases[{opts},_[CellLabel,_]],
          CellLabel->"In["<>ToString[num]<>"]:="
        ]),

      (* don't forget any errors... for now this is hardcoded.*)
      Cell[stuff___,"Message",opts___]:>(
        Cell[stuff,"Message",
          Sequence@@DeleteCases[{opts},_[CellLabel,_]],
          CellLabel->("During evaluation of In["<>ToString[num]<>"]:=")
        ]),

      (* 
         output cells _need _ to have a cell label otherwise
         we have to guess if it should have //*Form or whatnot
         if, for some reason, there is no celllabel already,
        just hope "Out[num]=" is ok.
      *)
      Cell[stuff___,"Output",opts___]:>(
        old=(Cases[{opts},_[CellLabel,l_]:>l]/.{a_}->a);
        Switch[old,
          _String, 
            CellLabel->StringReplace[old,
              RegularExpression["Out\\[\\d+\\]"]->(
                "Out["<>ToString[num]<>"]")],
          _, 
            CellLabel->"Out["<>ToString[num]<>"]="
        ]//
        Cell[stuff,"Output",
          Sequence@@DeleteCases[{opts},_[CellLabel,_]], #]&
      )
    }
  ];


(******************************************************************************)
(** WrapInOpenCloseButton - Internal **)

(* WrapInOpenCloseButton[expr_]:=
  ButtonBox[expr, 
    BaseStyle->None,
    ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
    FrontEnd`SelectedNotebook[], All, ButtonCell], FrontEndToken["OpenCloseGroup"]}]& ),  
    ButtonFrame->"None", Evaluator->None, Method->"Preemptive"] *)

WrapInOpenCloseButton[expr_]:=
ButtonBox[expr, 
  BaseStyle->None,
  ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[FrontEnd`SelectedNotebook[], All, ButtonCell],
    FrontEndToken["OpenCloseGroup"],
    FrontEnd`SelectionMove[FrontEnd`SelectedNotebook[], After, CellContents]
  }]& ),  
  ButtonFrame->"None", Evaluator->None, Method->"Preemptive"]


(******************************************************************************)
(** AddHeaderLinkTrail - Internal **)

AddHeaderLinkTrail[nb_Notebook, uri_String, lang_String:"English"] :=
  Module[{trailCell, expr=nb},
    BuildLog["-- Add category trail: "<>uri];
    trailCell = If[uri==="webMathematica/guide/webMathematica", {Cell[CellGroupData[{Cell[TextData[{ButtonBox["webMathematica",ButtonData->"paclet:webMathematica/guide/webMathematica",
    	BaseStyle->{"Link","LinkTrail"}],StyleBox[" > ","LinkTrailSeparator"]}],"LinkTrail",CellID->0]}]]},
    	MakeLinkTrailExpr[uri, lang]];
    BuildLog[ToString@trailCell];
    If[ListQ@trailCell && (trailCell =!= {}), 
      expr = MapAt[Flatten@Insert[#, {trailCell}, 1] &, expr, 1]
    ];
    expr
  ];


(******************************************************************************)
(** AddHeaderColorBar - Internal **)

AddHeaderColorBar[nb_Notebook, info_List, opts___?OptionQ] :=
  Module[{colorBarCell, type, linkTrailQ, expr=nb},
    linkTrailQ = If[Length@Flatten@Cases[nb, Cell[_,"LinkTrail",___], Infinity] > 0, True, False];
    type = Switch["EntityType" /. info,
      "Guide" | "Guide Overview" | "Alphabetical Listing", "Guide",
      "Tutorial", "Tutorial", "HowTo", "Tutorial",
      "Root Guide", None,
      _ , "Symbol"
    ];
    If[type =!= None,
      colorBarCell = If[linkTrailQ,
        Cell[" ", type<>"ColorBar", CellMargins->{{Inherited,Inherited},{-5,0}}],
        Cell[" ", type<>"ColorBar"]];
      MapAt[Flatten@Insert[#, {colorBarCell}, 1] &, expr, 1],
      expr
    ]
  ];


(******************************************************************************)
(** AddNotebookLabel - Internal **)

Options[AddNotebookLabel] = {"BuildMode"->"Normal", "PacletName"->"Mathematica", "Rebranded"->False};

AddNotebookLabel[nb_Notebook, info_List, lang_String, opts___?OptionQ]:=
  Module[{expr=nb, anchorBar, pacletNameCell, anchorBarGrid, gridOpts, buildmode, pacletName, URI, $Rebranded},
    (* do not include nb label on Guide or Root Guide pages *)
    If[MemberQ[{(*"Guide", *)"Root Guide"}, ("EntityType" /. info)], Return[expr] ];
	(* make anchor bar *)
	buildmode = "BuildMode" /. {opts} /. Options[AddNotebookLabel];
	pacletName = "PacletName" /. {opts} /. Options[AddNotebookLabel];
	URI = "URI" /. {info} /. Options[AddNotebookLabel];
	$Rebranded = "Rebranded" /. {opts} /. Options[AddNotebookLabel];
    anchorBar = MakeAnchorBarCell[expr, info, lang,
    	Which[ ("EntityType" /. info) === "Tutorial", 
    	  	"IncludeSections"->{"RelatedTutorialsSection", "TutorialMoreAboutSection", "TutorialFunction"},
    	  	("EntityType" /. info) === "Guide" && URI === {"guide/AlphabeticalListing"},
    	  	"IncludeSections"->{"GuideTutorial", "GuideMoreAbout"},
    	  	("EntityType" /. info) === "Guide", 
    	  	"IncludeSections"->{"GuideTutorial", "GuideMoreAbout", "GuideFunction"},
    	  	MatchQ[("EntityType" /. info), "ExamplePage" | "Example"], 
    	  	"IncludeSections"->{"FeaturedExampleTutorialsSection", "FeaturedExampleMoreAboutSection"},
    	  	True, "IncludeSections"->Automatic ],
          If[Flatten[{Cases[expr, StyleBox[_, "ModInfo", ___], Infinity], 
              Cases[expr, Cell[_, "ModInfo", ___], Infinity]}] =!= {}, 
            (* FIXME: This will prohibit non-M- documentation to not have a "New in ..." graphic 
               This isn't an issue since we don't support that option in the authoring environment
               today, but might someday *)
            "NewIn"->If[("PacletName" /. info) =!= "Mathematica", False, True], 
            "NewIn"->False], "BuildMode"->buildmode, "PacletName"->pacletName
        ];  
	gridOpts = 
  	  If[Length@Cases[anchorBar, _ActionMenuBox, Infinity] < 3,
        GridBoxOptions -> { GridBoxItemSize->{"Columns" -> { Scaled[0.65], { Scaled[0.34]}}, "ColumnsIndexed" -> {}, "Rows" -> {{1.}}, "RowsIndexed" -> {}}}
        ,
        CellID -> 1
      ];
    BuildLog["-- Assemble anchor bar grid"];
    (* V9 EDIT BELOW *)
    pacletNameCell = 
      CapitalizeString[Transmogrify`ConvertToString[GetClassificationBoxes[info, lang, "Rebranded"->$Rebranded]]];
    anchorBarGrid = 
    	Cell[BoxData[GridBox[{
			{If[pacletNameCell === " ", " ", Cell[pacletNameCell, "PacletNameCell"]], anchorBar}
		}]], "AnchorBarGrid", gridOpts, CellID->1];
    (* do not include nb label on Guide or Root Guide pages *)
    MapAt[Flatten@Insert[#, {anchorBarGrid}, 1] &, expr, 1]
  ];


(******************************************************************************)
(** CheckVersion - Internal **)
(** Returns:
      Infinity if s is greater than ReleaseVersion
      0 if s is less than Threshold Version
      some positive integer if s is between ThresholdVersion and ReleaseVersion
    Note that t<=s<=r currently only returns 1, but it might return other
    integers if we implement multiple threshholds.
**)
CheckVersion[s_String, opts___?OptionQ] :=
Module[{rversion, tversion, rtup, ttup, stup},
  (*Global`AntLog["Starting CheckVersion with s = " <> s];*)
  (* test for Future tagged Modinfo *)
  If[
  	StringMatchQ[s, RegularExpression["\\s*F\\s*"]] ||
  	!StringFreeQ[s, CharacterRange["a","z"]],
  	(*Global`AntLog["Ending CheckVersion (Returned s)"];*)Return[Infinity];
  ];
  rversion = "ReleaseVersion" /. {opts} /. Options[MakeNotebook];
  tversion = "ThresholdVersion" /. {opts} /. Options[MakeNotebook];
  If[tversion === Automatic, tversion = StringSplit[rversion, "."][[1]] ];
  {rtup, ttup, stup} = Map[
    Function[{vers}, Module[{v, tup},
      v = StringCases[vers, RegularExpression["^[0-9\\.]+"]];
      If[v === {}, Return[{}], v = v[[1]]];
      tup = StringSplit[v, "."];
      ToExpression /@ tup
    ]],
    {rversion, tversion, s}];
  Which[
    LexCmp[stup, rtup] === 1, (*Global`AntLog["Ending CheckVersion (Infinity)"];*)Infinity,
    LexCmp[stup, ttup] >= 0, (*Global`AntLog["Ending CheckVersion (1)"];*)1,
    True, (*Global`AntLog["Ending CheckVersion (0)"];*)0
]];


(******************************************************************************)
(** LexCmp - Internal **)
LexCmp[s1_, s2_] :=
  Which[
    s1 === s2 === {}, 0,
    s1 === {}, -1,
    s2 === {}, 1,
    s1[[1]] < s2[[1]], -1,
    s1[[1]] > s2[[1]], 1,
    True, LexCmp[Rest[s1], Rest[s2]]
  ];


(******************************************************************************)
(** ShippingFunctionQ - Internal **)
ShippingFunctionQ[___]:= False;
ShippingFunctionQ[func_String, ""] := ShippingFunctionQ[func];
ShippingFunctionQ[func_String, pacletName_String:"Mathematica"] := 
  Module[{ref, flag, base=pacletName},
  	base = If[StringMatchQ[base, "Mathematica"], "", base<>"/"];
    ref = If[StringMatchQ[func, "paclet:*"], func, "paclet:"<>base<>"ref/"<>func];
    (*Avoid looking at applications outside of Mathematica*)
    If[MemberQ[{"", "ANOVA", "Audio", "BarCharts", "Benchmarking", "BlackBodyRadiation", "Calendar", "CCompilerDriver", "ClusterIntegration", "Combinatorica", "Compatibility", "ComputationalGeometry", "ComputerArithmetic", "DatabaseLink", "Developer", "DLLTools", "EquationTrekker", "ErrorBarPlots", "Experimental", "FiniteFields", "FourierSeries", "FunctionApproximations", "Geodesy", "GraphUtilities", "GUIKit", "HierarchicalClustering", "Histograms", "HypothesisTesting", "JLink", "LightweightGridClient", "LinearRegression", "Mathematica", "Messages", "MultivariateStatistics", "Music", "NETLink", "NonlinearRegression", "Notation", "NumericalCalculus", "NumericalDifferentialEquationAnalysis", "Parallel", "ParallelTools", "PhysicalConstants", "PieCharts", "PlotLegends", "PolyhedronOperations", "Polytopes", "PrimalityProving", "Quaternions", "RawGuides", "RegressionCommon","RemoteSLightweightGridCliental", "ResonanceAbsorptionLines","Splines", "StandardAtmosphere", "StatisticalPlots", "Symbols","Units", "VariationalMethods", "VectorAnalysis", "VectorFieldPlots","WebServices", "WorldPlot", "XML"}
    	, pacletName],
    If[HasSourceInformation[ref, "English"], 
      flag = DocumentationBuild`Info`GetSourceInformation[ref, "English", "Flag"];
      FreeQ[{"ObsoleteFlag", "InternalFlag", "FutureFlag", "ExcisedFlag"}, flag],
      (* Else *)
      False
  ], 
  True]
  ];


(******************************************************************************)
(** InlineFunctionButton **)
InlineFunctionButton[str_String] := InlineFunctionButton[ Cell[str,"SeeAlso"] ];
InlineFunctionButton[c___] := (Message[InlineFunctionButton::arg, c]; $Failed);
InlineFunctionButton[Cell[c_String, sty_, opts___] ] :=
Module[{n, f, fList},
  If[(Head@c === String),
    f = StringReplace[c, {"\n"->"", "\r"->""}];
    Which[
      (* . separator *)
      StringCount[f, RegularExpression["\\s+\\.\\s+"]] > 0, 
      (
        f = StringSplit[f, RegularExpression["\\s+\\.\\s+"]];
        fList = 
          Cell[BoxData[ButtonBox[#, BaseStyle -> "Link"]], "InlineFormula"] & /@ DeleteCases[f, s_String /; StringMatchQ[s, RegularExpression["XX+"]]];
        n = Flatten[Riffle[ fList, {{" ", StyleBox["\[FilledVerySmallSquare]", "InlineSeparator"], " "}}], 1];
        Cell[BoxData[RowBox[n]], sty, opts]
      ),
      (* LongDash separator *)
      StringCount[f, RegularExpression["\\s+\[LongDash]\\s+"]] > 0, 
      (* for now, do nothing *)
        Cell[c, sty, opts],
      (*  *)
      StringCount[f, RegularExpression["\\s+\[EmptyVerySmallSquare]\\s+"]] > 0, 
      (
        f = StringSplit[f, RegularExpression["\\s+\[EmptyVerySmallSquare]\\s+"]];
        (* Commented out parts of the below line to fix autolinking of purposely unlinked inline
           listings (purposely because they have EmptyVerySmallSquares but aren't links) 
           See guide/InterpretingStrings (changed 4/2/14). *)
        fList = 
          Cell[BoxData[(*ButtonBox[*)#(*, BaseStyle -> "Link"]*)], "InlineFormula", "InlineGuideFunctionListingGray"] & /@ DeleteCases[f, s_String /; StringMatchQ[s, RegularExpression["XX+"]]];
        n = Flatten[Riffle[ fList, {{" ", StyleBox["\[FilledVerySmallSquare]", "InlineSeparator"], " "}}], 1];
        Cell[BoxData[RowBox[n]], sty, opts]
      ),
     True, 
       Cell[BoxData[RowBox[{Cell[BoxData[ButtonBox[c, BaseStyle -> "Link"]], "InlineFormula"]}]], sty, opts]
    ]    
  ,
  (* Return original input *)
  Cell[c, sty, opts]
  ]
]
InlineFunctionButton::arg = "String expected as input but value was:`1`";

(******************************************************************************)
(** OtherRefAddUnfinishedDraftBanner - Internal **)
OtherRefAddUnfinishedDraftBanner[nbExpr_, info_, type_, opts___?OptionQ]:=
  Module[{expr = nbExpr, unfindraftimg, langext,pos, bannerpos},
  	langext = "Language" /. {opts} /. Options[MakeNotebook] /. {"English"->"", "Japanese"->"_jp", "ChineseSimplified"->"_zh",
  	"Spanish"->"_es" };
  	If[MatchQ[type, "Symbol"], Return[expr]];
  	expr = If[MemberQ[expr, "UnderDevelopment", Infinity] || TrueQ["Underdevelopment"/.info],
  	  unfindraftimg = Import[ToFileName[{$DocumentationBuildDirectory,
         "Internal", "online", "Images"}, "RefUnfinishedDraft"<>langext<>".m"]];
      Switch[type,
        "Device Connection",
        pos = Position[expr, Cell[___, "ObjectNameAlt", ___]];
        bannerpos = MapAt[(# + 1 &), pos, {1, -1}];
        Insert[expr, Cell[BoxData[ToBoxes[unfindraftimg]], "DeviceAbstract"], bannerpos],
        "Interpreter",
        pos = Position[expr, Cell[___, "ObjectNameAlt", ___]];
        bannerpos = MapAt[(# + 1 &), pos, {1, -1}];
        Insert[expr, Cell[BoxData[ToBoxes[unfindraftimg]], "DeviceAbstract"], bannerpos],
        _,
        expr
      ],
  	  expr];
    expr
  ];

(******************************************************************************)
(** translateTerm (from LarryA) - Internal **)
(** This function is for the translations at the tops of ref pages **)

translateTerm[term_String, language_String] :=
    DynamicModule[{translation},
     DynamicWrapper[
      Dynamic[translation],
      (
       FEPrivate`Set[translation,
        FEPrivate`TranslateKeyword[term, language]];
       FEPrivate`If[FEPrivate`SameQ[translation, $Failed],
        FEPrivate`Set[translation, ""]];
       translation
       )
      ]
     ];

(* TODO: update as new languages are added *)
translationLanguageList = {"English", "Japanese", "ChineseSimplified", "Korean", "Spanish"};

(******************************************************************************)
(** AddObjectNameGraphic - Internal **)
AddObjectNameGraphic[nbExpr_, info_, opts___?OptionQ]:=
Module[{pos, expr=nbExpr, rast, rastover, rasthide, rasthideover, unfindraftimg, langext, language, ce, dyn, histData, hist, uri, 
        translationboxes, type, NewInImgGen, NewImgNoVersion, rastNoVersion},
  LogIt["** Entering addObjectNameGraphic **"];
  (*lang = If[language === "English", "", "_jp"];*)
  
  NewInImgGen[version_String, lang:"English"]:=
   Graphics[Text[Style[Localized["BannerNewIn", lang][version],
     FontFamily->"Source Sans Pro",
     FontSize->18,FontSlant->"Italic",
     FontWeight->"Bold",FontColor->RGBColor[.968627,.466667,0]]], ImageSize->{120, 24}];
     
  NewInImgGen[version_String, lang:"Japanese"]:=
   Graphics[Text[Style[Localized["BannerNewIn", lang][version],
     FontFamily->"Hiragino Kaku Gothic ProN",
     FontSize->18,
     FontWeight->"Bold",FontColor->RGBColor[.968627,.466667,0]]], ImageSize->{120, 24}];
     
  NewInImgGen[version_String, lang:"ChineseSimplified"]:=
   Graphics[Text[Style[Localized["BannerNewIn", lang][version],
     FontFamily->"Microsoft YaHei",
     FontSize->18,
     FontWeight->"Bold",FontColor->RGBColor[.968627,.466667,0]]], ImageSize->{120, 24}];
     
  NewInImgGen[version_String, lang_] :=
   NewInImgGen[version, "English"];
  
  (* TODO:FIXME: These localization graphics should be integrated better...  *)
  (* Adding NewImgNoVersion[lang] to display "NEW" instead of "NEW IN xxx" per sw *)
  (* andyh 2016-08-03 *)
  NewImgNoVersion["English"]:=
   Graphics[Text[Style[Localized["BannerNewNoVersion", "English"],
     FontFamily->"Source Sans Pro",
     FontSize->18,FontSlant->"Italic",
     FontWeight->"Bold",FontColor->RGBColor[.968627,.466667,0]]], ImageSize->{80, 24}];
     
  NewImgNoVersion["Japanese"]:=
   Graphics[Text[Style[Localized["BannerNewNoVersion", "Japanese"],
     FontFamily->"Hiragino Kaku Gothic ProN",
     FontSize->18,
     FontWeight->"Bold",FontColor->RGBColor[.968627,.466667,0]]], ImageSize->{80, 24}];
     
  NewImgNoVersion["ChineseSimplified"]:=
   Graphics[Text[Style[Localized["BannerNewNoVersion", "ChineseSimplified"],
     FontFamily->"Microsoft YaHei",
     FontSize->18,
     FontWeight->"Bold",FontColor->RGBColor[.968627,.466667,0]]], ImageSize->{80, 24}];
     
  NewImgNoVersion[lang_] := NewImgNoVersion["English"];
  
  uri = "URI"/.info;
  type = "EntityType" /. info;
  histData = "HistoryData" /. {info} /. Options[MakeNotebook];
  hist = {histData[[1,1,2]],histData[[1,2,2]], histData[[1,3,2]], histData[[1,4,2]]};
  language = "Language" /. {opts} /. Options[MakeNotebook];
  langext = "Language" /. {opts} /. Options[MakeNotebook] /. {"English"->"", "Japanese"->"_jp", "ChineseSimplified"->"_zh",
  	"ChineseTraditional"->"", "Spanish"->"_es" };
  pos = Position[expr, Cell[_, "ObjectName"|"ObjectNameSmall"|"ObjectNameAlt", ___]];
  translationboxes = If[MatchQ[type, "Symbol"] && StringMatchQ[uri, "ref/*"], 
  	With[{title = Last[StringSplit[uri, "/"]]}, ToBoxes[
    Style[With[{defaultLang = "English"}, DynamicModule[{result, translationlang},     
     Dynamic[translationlang = FrontEnd`CurrentValue[$FrontEnd, "Language", defaultLang]; With[{translationlang = translationlang}, FE`Evaluate[FEPrivate`If[
      FEPrivate`SameQ[FEPrivate`Set[result, FEPrivate`TranslateKeyword[title, translationlang]], $Failed], "", StringJoin["(", result, ")"]]]]]]], "ObjectNameTranslation"]
   ]], ""];
  Which[
    ((Length@hist > 0) && StringMatchQ[hist[[1]], RegularExpression["\\s*11\\.*[0-9]* *"] | 
    	(* include futurized && UD pages here *)"??"] &&
    	(* exclude pages that would qualify for both new and updated tags *)
    	!StringMatchQ[hist[[2]], RegularExpression["\\s*11\\.*[0-9]* *"]]),  (
      rast = NewInImgGen[hist[[1]], language];
      (*rast = Import[ToFileName[{$DocumentationBuildDirectory,
               "Internal", "online", "Images"}, "NewIn10"<>langext<>".png"], {"PNG", "Graphics"}];*)
      unfindraftimg = Import[ToFileName[{$DocumentationBuildDirectory,
               "Internal", "online", "Images"}, "RefUnfinishedDraft"<>langext<>".m"]];
      If[StringMatchQ[uri,DocumentationBuild`URIsWithoutDesktopSupport],
        With[{icon = Import[ToFileName[{$DocumentationBuildDirectory,
               "Internal", "online", "Images"}, "cloudicon-gray.png"]]},
        c = Cell[BoxData[GridBox[{
        {First[Cases[expr, Cell[c_, b : "ObjectName" | "ObjectNameSmall" | "ObjectNameAlt", ___] :> 
        	If[MemberQ[expr, "UnderDevelopment", Infinity] || TrueQ["Underdevelopment"/.info],
        	 Cell[TextData[{Cell[c, b], Cell[BoxData[ToBoxes[Spacer[10]]]], Cell[BoxData[ToBoxes[unfindraftimg]]]}]],
        	 Cell[TextData[{Cell[c, b], Cell[BoxData[ToBoxes[Spacer[8]]]], Cell[BoxData[translationboxes], "ObjectNameTranslation"]}]]], Infinity]],
        Cell[BoxData[TooltipBox[ ToBoxes@Style[Deploy@rast,"NewIn10Graphic"], Localized["New in 10", language], ActionDelay -> .3]], Magnification->1],
        Cell[BoxData[ToBoxes[Tooltip[Deploy[icon], Localized["CloudIconTooltipText", language], TooltipStyle->"CloudIconTooltipStyle"]]], "CloudIconGraphic"]}
        }]], "ObjectNameGrid"]],c = Cell[BoxData[GridBox[{
        {First[Cases[expr, Cell[c_, b : "ObjectName" | "ObjectNameSmall" | "ObjectNameAlt", ___] :> 
        	If[MemberQ[expr, "UnderDevelopment" | "ExperimentalObject", Infinity] || TrueQ["Underdevelopment"/.info],
        	 Cell[TextData[{Cell[c, b], Cell[BoxData[ToBoxes[Spacer[10]]]], Cell[BoxData[ToBoxes[unfindraftimg]]]}]],
        	 Cell[TextData[{Cell[c, b], Cell[BoxData[ToBoxes[Spacer[8]]]], Cell[BoxData[translationboxes], "ObjectNameTranslation"]}]]], Infinity]],
        Cell[BoxData[TooltipBox[ ToBoxes@Style[Deploy@rast,"NewIn10Graphic"], Localized["New in 11", language], ActionDelay -> .3]], Magnification->1]}
        }]], "ObjectNameGrid"]];
      pos = Position[expr, Cell[_, "ObjectName"|"ObjectNameSmall" | "ObjectNameAlt", ___]];
      expr = ReplacePart[expr, First[pos] -> c];
    ),
    ((Length@hist > 1) && StringMatchQ[hist[[2]], RegularExpression["\\s*11\\.*[0-9]* *"]]), (
      (* how many actual updates in this nb *)
      dyn = Plus[
        Length[Cases[Cases[nbExpr, Cell[__, "Usage", ___], Infinity],
          {___, Dynamic[If[CurrentValue[{TaggingRules, "ModificationHighlight"}] === True, __]], ___},
          Infinity, 1]],
        Length[Cases[nbExpr,
          Cell[__, "Notes"|"FormatNotes", ___,
            Background->Dynamic[If[CurrentValue[{TaggingRules, "ModificationHighlight"}] === True, __]],
          ___], Infinity, 1]],
        Length[Cases[Cases[nbExpr, Cell[__, "2ColumnTableMod" | "3ColumnTableMod"|"3ColumnIllustratedTableMod", ___], Infinity],
          GridBox[{___, {Cell["\[FilledVerySmallSquare]", "TableRowIcon", ___], ___}, ___}, ___],
          Infinity, 1]]
      ];
      If[dyn > 0,
        rast = Import[ToFileName[{$DocumentationBuildDirectory,
               "Internal", "online", "Images"}, "UpdatedIn10-show"<>langext<>".png"], {"PNG", "Graphics"}];
        rastover = Import[ToFileName[{$DocumentationBuildDirectory,
               "Internal", "online", "Images"}, "UpdatedIn10-show-over"<>langext<>".png"], {"PNG", "Graphics"}];
        rasthide = Import[ToFileName[{$DocumentationBuildDirectory,
               "Internal", "online", "Images"}, "UpdatedIn10-hide"<>langext<>".png"], {"PNG", "Graphics"}];
        rasthideover = Import[ToFileName[{$DocumentationBuildDirectory,
               "Internal", "online", "Images"}, "UpdatedIn10-hide-over"<>langext<>".png"], {"PNG", "Graphics"}];
        ce = Cell[BoxData[TagBox[TooltipBox[
            TogglerBox[Dynamic[
            FrontEnd`CurrentValue[
            FrontEnd`EvaluationNotebook[], {
              TaggingRules, "ModificationHighlight"}, False]], 
              { True-> PaneSelectorBox[{False->ToBoxes@rasthide, True->ToBoxes@rasthideover}, Dynamic[
    CurrentValue["MouseOver"]]], False->PaneSelectorBox[{False->ToBoxes@rast, True->ToBoxes@rastover}, Dynamic[
    CurrentValue["MouseOver"]]] }, "\<\"Click me\"\>"], 
                Localized["Toggle changes in 11", language], ActionDelay -> .3], MouseAppearanceTag["LinkHand"]]], Magnification->1];
        unfindraftimg = Import[ToFileName[{$DocumentationBuildDirectory,
               "Internal", "online", "Images"}, "RefUnfinishedDraft"<>langext<>".m"]];

        (* assemble ObjectName grid *)
        If[StringMatchQ[uri,DocumentationBuild`URIsWithoutDesktopSupport],
        With[{icon = Import[ToFileName[{$DocumentationBuildDirectory,
               "Internal", "online", "Images"}, "cloudicon-gray.png"]]},
        c = Cell[BoxData[GridBox[{
          {First[Cases[expr, Cell[c_, b : "ObjectName" | "ObjectNameSmall" | "ObjectNameAlt", ___] :> 
           If[MemberQ[expr, "UnderDevelopment" | "ExperimentalObject", Infinity] || TrueQ["Underdevelopment"/.info],
        	 Cell[TextData[{Cell[c, b], Cell[BoxData[ToBoxes[Spacer[10]]]], Cell[BoxData[ToBoxes[unfindraftimg]]]}]],
        	 Cell[TextData[{Cell[c, b], Cell[BoxData[ToBoxes[Spacer[8]]]], Cell[BoxData[translationboxes], "ObjectNameTranslation"]}]]], Infinity]],
           ce, 
           Cell[BoxData[ToBoxes[
           	Tooltip[Deploy[icon], 
           		"This feature relies on access \nto the Wolfram Cloud.", 
           		TooltipStyle->"CloudIconTooltipStyle"]]], "CloudIconGraphic"]}
          }]], "ObjectNameGrid"]],
        c = Cell[BoxData[GridBox[{
          {First[Cases[expr, Cell[c_, b : "ObjectName" | "ObjectNameSmall" | "ObjectNameAlt", ___] :> 
            If[MemberQ[expr, "UnderDevelopment" | "ExperimentalObject", Infinity] || TrueQ["Underdevelopment"/.info],
        	 Cell[TextData[{Cell[c, b], Cell[BoxData[ToBoxes[Spacer[10]]]], Cell[BoxData[ToBoxes[unfindraftimg]]]}]],
        	 Cell[TextData[{Cell[c, b], Cell[BoxData[ToBoxes[Spacer[8]]]], Cell[BoxData[translationboxes], "ObjectNameTranslation"]}]]], Infinity]],
           ce}
          }]], "ObjectNameGrid"]],
      (* Else *)
        rast = Import[ToFileName[{$DocumentationBuildDirectory,
               "Internal", "online", "Images"}, "UpdatedIn10"<>langext<>".png"], {"PNG", "Graphics"}];
        If[StringMatchQ[uri,DocumentationBuild`URIsWithoutDesktopSupport],
        With[{icon = Import[ToFileName[{$DocumentationBuildDirectory,
               "Internal", "online", "Images"}, "cloudicon-gray.png"]]},
        c = Cell[BoxData[GridBox[{
          {First[Cases[expr, Cell[c_, b : "ObjectName" | "ObjectNameSmall" | "ObjectNameAlt", ___] :> 
            If[MemberQ[expr, "UnderDevelopment" | "ExperimentalObject", Infinity] || TrueQ["Underdevelopment"/.info],
        	 Cell[TextData[{Cell[c, b], Cell[BoxData[ToBoxes[Spacer[10]]]], Cell[BoxData[ToBoxes[unfindraftimg]]]}]],
        	 Cell[TextData[{Cell[c, b], Cell[BoxData[ToBoxes[Spacer[8]]]], Cell[BoxData[translationboxes], "ObjectNameTranslation"]}]]], Infinity]] (* ,
           Cell[BoxData[ToBoxes@rast], Magnification->1, Deployed->True]*),
           Cell[BoxData[ToBoxes[
           	Tooltip[Deploy[icon], 
           		"This feature relies on access \nto the Wolfram Cloud.", 
           		TooltipStyle->"CloudIconTooltipStyle"]]], "CloudIconGraphic"]}
          }]], "ObjectNameGrid"]],
        c = Cell[BoxData[GridBox[{
          {First[Cases[expr, Cell[c_, b : "ObjectName" | "ObjectNameSmall" | "ObjectNameAlt", ___] :> 
            If[MemberQ[expr, "UnderDevelopment" | "ExperimentalObject", Infinity] || TrueQ["Underdevelopment"/.info],
        	 Cell[TextData[{Cell[c, b], Cell[BoxData[ToBoxes[Spacer[10]]]], Cell[BoxData[ToBoxes[unfindraftimg]]]}]],
        	 Cell[TextData[{Cell[c, b], Cell[BoxData[ToBoxes[Spacer[8]]]], Cell[BoxData[translationboxes], "ObjectNameTranslation"]}]]], Infinity]] (* ,
           Cell[BoxData[ToBoxes@rast], Magnification->1, Deployed->True]*)}
          }]], "ObjectNameGrid"]]
      ];
        
      expr = ReplacePart[expr, First[pos] -> c];
    ),
  (* Functions which were new in previous major version are marked "New" *)
  (* From: https://jira.wolfram.com/jira/browse/WLDOCS-636 
     SW wants docs pages from the direct previous integer version of M- to say "New," 
     but newest/current version of M- to say "New in <version number>".
     For example, any new M9 docs pages will have "New," but M10 will say "New in 10," "New in 10.1," 
     "New in 10.2," etc. Then those will be refactored after the next integer release.
  *)
  (* andyh 2016-08-03 *)
  ((Length@hist > 0) && StringMatchQ[hist[[1]], RegularExpression["\\s*10\\.*[1-9]+ *"]]),
  (    
    rast = NewInImgGen[hist[[1]], language];
    rastNoVersion = NewImgNoVersion[language];
    unfindraftimg = Import[ToFileName[{$DocumentationBuildDirectory,
                           "Internal", "online", "Images"}, "RefUnfinishedDraft"<>langext<>".m"]];
    (* Check for Cloud only functions without desktop support *)
    If[StringMatchQ[uri,DocumentationBuild`URIsWithoutDesktopSupport],
       
        With[{icon = Import[ToFileName[{$DocumentationBuildDirectory,
                            "Internal", "online", "Images"}, "cloudicon-gray.png"]]},
            c = Cell[BoxData[GridBox[{
                    {First[Cases[expr, Cell[c_, b : "ObjectName" | "ObjectNameSmall" | "ObjectNameAlt", ___] :> 
                            If[MemberQ[expr, "UnderDevelopment", Infinity] || TrueQ["Underdevelopment"/.info],
                                Cell[TextData[{Cell[c, b], Cell[BoxData[ToBoxes[Spacer[10]]]], Cell[BoxData[ToBoxes[unfindraftimg]]]}]]
                            ,
                                Cell[TextData[{Cell[c, b], Cell[BoxData[ToBoxes[Spacer[8]]]], Cell[BoxData[translationboxes], "ObjectNameTranslation"]}]]
                            ], Infinity]],
                        Cell[BoxData[TooltipBox[ ToBoxes@Style[Deploy@rast,"NewIn10Graphic"], Localized["New in 10", language], ActionDelay -> .3]], Magnification->1],
                        Cell[BoxData[ToBoxes[Tooltip[Deploy[icon], Localized["CloudIconTooltipText", language], TooltipStyle->"CloudIconTooltipStyle"]]], "CloudIconGraphic"]}
                    }]], "ObjectNameGrid"]
        ]
    ,
        (* Add "NEW" banner here... *)
        c = Cell[BoxData[GridBox[{
                {First[Cases[expr, Cell[c_, b : "ObjectName" | "ObjectNameSmall" | "ObjectNameAlt", ___] :> 
                        If[MemberQ[expr, "UnderDevelopment" | "ExperimentalObject", Infinity] || TrueQ["Underdevelopment"/.info],
                            Cell[TextData[{Cell[c, b], Cell[BoxData[ToBoxes[Spacer[10]]]], Cell[BoxData[ToBoxes[unfindraftimg]]]}]]
                        ,
                            Cell[TextData[{Cell[c, b], Cell[BoxData[ToBoxes[Spacer[8]]]], Cell[BoxData[translationboxes], "ObjectNameTranslation"]}]]
                        ], Infinity]],
                 Cell[BoxData[TooltipBox[ ToBoxes@Style[Deploy@rastNoVersion,"NewIn10Graphic"], Localized["BannerNewNoVersion", language], ActionDelay -> .3]], Magnification->1]}
            }]], "ObjectNameGrid"]
    ];
    pos = Position[expr, Cell[_, "ObjectName"|"ObjectNameSmall" | "ObjectNameAlt", ___]];
    expr = ReplacePart[expr, First[pos] -> c];
  ),
  (* 
      Fall through 
  *)
  True, 
    LogIt["   Non match: "<>ToString[{hist, language}] ];
    unfindraftimg = Import[ToFileName[{$DocumentationBuildDirectory,
               "Internal", "online", "Images"}, "RefUnfinishedDraft"<>langext<>".m"]];
    If[StringMatchQ[uri,DocumentationBuild`URIsWithoutDesktopSupport],
        With[{icon = Import[ToFileName[{$DocumentationBuildDirectory,
               "Internal", "online", "Images"}, "cloudicon-gray.png"]]},
        c = Cell[BoxData[GridBox[{
          {First[Cases[expr, Cell[c_, b : "ObjectName" | "ObjectNameSmall", ___] :> 
            If[MemberQ[expr, "UnderDevelopment" | "ExperimentalObject", Infinity] || TrueQ["Underdevelopment"/.info],
        	 Cell[TextData[{Cell[c, b], Cell[BoxData[ToBoxes[Spacer[10]]]], Cell[BoxData[ToBoxes[unfindraftimg]]]}]],
        	 Cell[TextData[{Cell[c, b], Cell[BoxData[ToBoxes[Spacer[8]]]], Cell[BoxData[translationboxes], "ObjectNameTranslation"]}]]], Infinity]], "",
          Cell[BoxData[ToBoxes[
           	Tooltip[Deploy[icon], 
           		"This feature relies on access \nto the Wolfram Cloud.", 
           		TooltipStyle->"CloudIconTooltipStyle"]]], "CloudIconGraphic"]}
          }]], "ObjectNameGrid"]
        ];
        expr = ReplacePart[expr, First[pos] -> c],
      (* No graphic; just add translation boxes *)
      expr = ReplaceAll[expr, Cell[c_, b : "ObjectName" | "ObjectNameSmall", ___] :>
        If[MemberQ[expr, "UnderDevelopment" | "ExperimentalObject", Infinity] || TrueQ["Underdevelopment"/.info],
        	 Cell[TextData[{Cell[c, b], Cell[BoxData[ToBoxes[Spacer[10]]]], Cell[BoxData[ToBoxes[unfindraftimg]]]}], "ObjectNameGrid"],
        	 Cell[TextData[{Cell[c, b], Cell[BoxData[ToBoxes[Spacer[8]]]], Cell[BoxData[translationboxes], "ObjectNameTranslation"]}], "ObjectNameGrid"]]]];
    ];
  LogIt["** Leaving addObjectNameGraphic **"];
  expr
];


(******************************************************************************)
(** AddGuideNewInGraphic - Internal **)

AddGuideNewInGraphic[nb_Notebook, info_, opts___?OptionQ]:=
  Module[{pos, expr=nb, lang, langext, rast, rasthide, funcs, lang, versionNumber},
  	versionNumber = StringSplit[$Version, "."][[1]];
    lang = "Language" /. {opts} /. Options[MakeNotebook];
    langext = If[lang === "English", "", "_jp"];
    funcs = Cases[expr, ButtonBox[__, BaseStyle->"NewInVersionLink", ___], Infinity]; 
    If[Length@funcs > 0, (
      rast = Import[ToFileName[{$DocumentationBuildDirectory,
        "Internal", "online", "Images"}, "UpdatedIn"<>versionNumber<>langext<>".gif"], "GIF"];
      rasthide = Import[ToFileName[{$DocumentationBuildDirectory,
        "Internal", "online", "Images"}, "UpdatedIn"<>versionNumber<>"-hide"<>langext<>".gif"], "GIF"];
      c = Cell[BoxData[GridBox[{
        {Cell[First@Cases[expr, Cell[c_String, "GuideTitle", ___]:>c, Infinity], "GuideTitle"],
          Cell[BoxData[TagBox[TooltipBox[
            TogglerBox[Dynamic[
              FrontEnd`CurrentValue[
                FrontEnd`EvaluationNotebook[], {
                  TaggingRules, "ModificationHighlight"}, False]],
              { True-> ToBoxes@rasthide, False->ToBoxes@rast },
              "\<\"Click me\"\>"],
            (* FIXME: use ReleaseVersion *)
            Localized["HistoryNewIn", lang][versionNumber], ActionDelay -> .3], MouseAppearanceTag["LinkHand"]]]]}
        }]], "GuideTitleGrid"];
      pos = Position[expr, Cell[_, "GuideTitle", ___]];
      expr = ReplacePart[expr, First[pos] -> c];
    )];
    expr
  ];

(*AddGuideNewInGraphic[nb_Notebook, info_, opts___?OptionQ]:=
  Module[{pos, expr=nb, lang, langext, rast, rasthide, funcs, lang},
    lang = "Language" /. {opts} /. Options[MakeNotebook];
    langext = If[lang === "English", "", "_jp"];
    funcs = Cases[expr, ButtonBox[__, BaseStyle->"NewInVersionLink", ___], Infinity]; 
    If[Length@funcs > 0, (
      rast = Import[ToFileName[{$DocumentationBuildDirectory,
        "Internal", "online", "Images"}, "UpdatedIn6"<>langext<>".gif"], "GIF"];
      rasthide = Import[ToFileName[{$DocumentationBuildDirectory,
        "Internal", "online", "Images"}, "UpdatedIn6-hide"<>langext<>".gif"], "GIF"];
      c = Cell[BoxData[GridBox[{
        {Cell[First@Cases[expr, Cell[c_String, "GuideTitle", ___]:>c, Infinity], "GuideTitle"],
          Cell[BoxData[TooltipBox[
            TogglerBox[Dynamic[
              FrontEnd`CurrentValue[
                FrontEnd`EvaluationNotebook[], {
                  TaggingRules, "ModificationHighlight"}, False]],
              { True-> ToBoxes@rasthide, False->ToBoxes@rast },
              "\<\"Click me\"\>"],
            (* FIXME: use ReleaseVersion *)
            Localized["HistoryNewIn", lang]["6"], ActionDelay -> .3]]]}
        }]], "GuideTitleGrid"];
      pos = Position[expr, Cell[_, "GuideTitle", ___]];
      expr = ReplacePart[expr, First[pos] -> c];
    )];
    expr
  ];*)


(******************************************************************************)
(** addGuideTOCRightGuillemet - Internal **)
addGuideTOCRightGuillemet[nbExpr_]:=
Module[{expr = nbExpr, pos, cexpr},
      (* GuideTOCLinks need a RightGuillemet *)
      pos = Position[expr, Cell[TextData[ButtonBox[_, ___]], "GuideTOCLink", ___], Infinity ];
      Scan[(
          cexpr = Extract[expr, #];
          If[ !StringMatchQ[ ConvertToString[cexpr], "*\[RightGuillemet]"],
          cexpr = cexpr /.
              Cell[TextData[ButtonBox[c_, o1___]], "GuideTOCLink", o2___] :>
              Cell[TextData[ButtonBox[
                  If[Head@c === String,
                     c <> "\[NonBreakingSpace]\[RightGuillemet]",
                     AppendTo[c, "\[NonBreakingSpace]\[RightGuillemet]"] 
                     ],
                  o1]], "GuideTOCLink", o2];
          expr = ReplacePart[expr, # -> cexpr]
        ]
      )&,
      pos];
  expr
];


(******************************************************************************)
(** appendString - Internal **)
appendString[str_String, text_String:" \[RightGuillemet]"]:=
  str <> text;
  

(******************************************************************************)
(** ConvertValuesForSearch - Internal **)

ConvertValuesForSearch[lst_List, lang_String:"English"] := 
  Flatten[ConvertSearchText[{#}, lang] & /@ lst, 1];

ConvertSearchText[a___]:= (Message[ConvertSearchText::arg, a]; $Failed)
ConvertSearchText[{name_ -> value_String}, lang_String:"English"] :=
  {name -> Switch[lang, (*"Japanese", value, "ChineseSimplified", value,*) _, value] };

ConvertSearchText[{name_ -> value_}, lang_String:"English"] := {name -> value };
(*  *)
ConvertSearchText[{"status" -> None}, ___] := {"status" -> "" };

ConvertSearchText::arg = "List of rules expected as input but value was:`1`";










(**)


(* 
    ExportMarketingWebPage[outfile, expr, pageType, opts] 
    this is the main Transmogrify[] wrapper.
    NOTE - outfile = None just returns the raw Transmogrify output
           Test->True causes ExportMarketingWebPage to use TransmogrifyString[]
           the shorthand functions rawmogtest[] and mogtest[], respectively,
           are wrappers for these two functionalities.
*)
ExportMarketingWebPage[_,$Failed,___] = $Failed;

ExportMarketingWebPage[file_String, nb_String, language_String:"English", opts___?OptionQ]:= 
   ExportMarketingWebPage[file, DocumentationBuild`Utils`GetQuietly[nb], language, opts ];
  
ExportMarketingWebPage[file_String, nbExpr_Notebook, language_String:"English", opts___?OptionQ]:=
Developer`UseFrontEnd @
Module[{ext, properties={}, outfile=file, transform, keywordList={}, titleName, res, nbObj, nbfile, nbobj, nbobj2, nbget, tmp, parentDir, nbFile, localcss, 
  layoutfile, popupLayoutfile, urlBaseRemote, expr, infilename, filename, jatitle, description, ExamplesRawMaterialDir, WhatsNewID, MarketingCategoryName, additionalstyles, defaultstylesheet},

  LogIt["** Entering ExportMarketingPage **"];
  Global`AntLog["** Entering ExportMarketingPage **"];
  (* Add DocumentationBuild\XMLTransforms if it isn't already on path *)
  If[ !MemberQ[ToFileName[{DocumentationBuild`Common`$DocumentationBuildDirectory, "XMLTransforms"}], Transmogrify`$XMLTransformPath], 
    AppendTo[Transmogrify`$XMLTransformPath, ToFileName[{DocumentationBuild`Common`$DocumentationBuildDirectory, "XMLTransforms"}] ];
  ];

  urlBaseRemote = "http://reference.wolfram.com/language/";
  ext = Switch[language, "Japanese", "ja", "Russian", "ru", "Korean", "ko", "French", "fr", "German", "de", "Portuguese", "pt-br", "Spanish", "es","ChineseSimplified", "zh", _, "en"];
  
  Preview = "Preview" /. {opts};
  ExamplesRawMaterialDir = "ExamplesRawMaterialDir" /. {opts};
  MarketingCategoryName = "CategoryName" /. {opts};
  WhatsNewID = "WhatsNewID" /. {opts};
  infilename = "InFileName" /. {opts};
  
  transform = {"HTML","MarketingPages.m"};
  layoutfile = ToFileName[{
     DocumentationBuild`Common`$DocumentationBuildDirectory,"XMLTransforms","HTML","Layouts"}, "MarketingPages.html"];
  popupLayoutfile = ToFileName[{
     DocumentationBuild`Common`$DocumentationBuildDirectory,"XMLTransforms","HTML","Layouts"}, "MarketingPagePopup.html"];

  (* keywords *)
  description = DocumentationBuild`Info`GetNotebookDescription[nbExpr];
  keywordList = DocumentationBuild`Info`GetNotebookKeywords[nbExpr];
  synonymList = DocumentationBuild`Info`GetNotebookSynonyms[nbExpr];
  LogIt["  Keywords: "<>ToString[keywordList]];

  titleName = Switch[language, "Japanese", 
  	(* 
  	jatitle = Cases[nbExpr, Cell[c_, "MarketingTitle", ___,Rule[CellLabel,d_],___] :> d, Infinity];
    jatitle = DeleteCases[jatitle, StyleBox[_, "FullListing"], Infinity];
    jatitle = Transmogrify`ConvertToString[First[jatitle]];
    jatitle = If[StringFreeQ[jatitle, "\n"], jatitle, First@StringSplit[jatitle, "\n"]];
    jatitle = StringReplace[jatitle, {
        """ | """ | "'" | "'" -> "'",
        "\[FilledRightTriangle]" -> ">",
        "-" -> "-",
        "\"" -> ""
    }];
    jatitle
    *)
    DocumentationBuild`Info`GetNotebookTitle[nbExpr, "MarketingPage"],
    "ChineseSimplified",
  	DocumentationBuild`Info`GetNotebookTitle[nbExpr, "MarketingPage"],
  	    "Spanish",
  	DocumentationBuild`Info`GetNotebookTitle[nbExpr, "MarketingPage"],
  	  	    "Russian",
  	DocumentationBuild`Info`GetNotebookTitle[nbExpr, "MarketingPage"],
  	  	    "Korean",
  	DocumentationBuild`Info`GetNotebookTitle[nbExpr, "MarketingPage"],
  	  	  	    "German",
  	DocumentationBuild`Info`GetNotebookTitle[nbExpr, "MarketingPage"],
  	  	  	    "French",
  	DocumentationBuild`Info`GetNotebookTitle[nbExpr, "MarketingPage"],
  	  	    "Portuguese",
  	DocumentationBuild`Info`GetNotebookTitle[nbExpr, "MarketingPage"],
    _,
  	DocumentationBuild`Info`GetNotebookTitle[nbExpr, "MarketingPage"]];

  LogIt["  Title name: "<>ToString[titleName]];
  outfile = 
    If[ StringQ[ext],
      StringReplace[outfile, RegularExpression["(?<!\\.\\w\\w)(\\.html)"]->"."<>ext<>"$1"],
      outfile];
  localcss = StringReplace[DirectoryName@outfile, {DirectoryName@DirectoryName@outfile->"", "\""->"", "/"->"", $PathnameSeparator -> ""}] <> "-localcss";

  (* Create preprocess notebook *)
  LogIt["  MakeMarketingPageNotebook"];
  Global`AntLog["** MakeMarketingPageNotebook **"];
  
  expr = MakeMarketingPageNotebook[nbExpr, language, opts];

  SetOptions[$FrontEnd, StyleSheetPath -> Prepend[CurrentValue[$FrontEnd, StyleSheetPath], DocumentationBuild`$MarketingStylePath]];

  SetOptions[$FrontEndSession, Rule[AutoStyleOptions, {Rule["HighlightUndefinedSymbols", False]}]];
    
    additionalstyles =
             {Cell[StyleData["InlineFormula"], FontFamily->"Source Sans Pro", FontSize->Inherited 1.08],
             Cell[StyleData["MR"], FontFamily->"Source Sans Pro"],
             Cell[StyleData["StandardForm"],FontFamily->"Source Sans Pro"],
             Cell[StyleData["InputForm"],FontFamily->"Source Sans Pro"],
             Cell[StyleData["Input"],  FontFamily->"Source Sans Pro", FontSize->13, LineSpacing->{1.2, 0}],
             Cell[StyleData["Output"], FontSize->13]
             };
             
    defaultstylesheet= Cell[StyleData[StyleDefinitions -> FrontEnd`FileName[{"Wolfram"}, "MarketingPageStyles.nb"]]];
    
    expr = expr /. 
    {Rule[StyleDefinitions, FrontEnd`FileName[___]]:> Rule[StyleDefinitions, Notebook[Flatten[{defaultstylesheet, additionalstyles}], Rule[AutoStyleOptions,{Rule["HighlightUndefinedSymbols",False]}]]],
     Rule[StyleDefinitions, Notebook[{contents___}, options___]]:> Rule[StyleDefinitions, Notebook[Flatten[{contents, additionalstyles}], options, Rule[AutoStyleOptions,{Rule["HighlightUndefinedSymbols",False]}]]]
    };

    expr = expr /. {Notebook[{contents___}, options___]:> Notebook[{contents}, options, CellLabelAutoDelete->False]};

  Global`AntLog["** About to run a series of FrontEnd commands that apply the various stylesheet options/rules **"];
  Global`AntLog["** About to run NotebookPut **"];
  nbobj2 = NotebookPut[expr];
  Global`AntLog["** Finished running NotebookPut **"];
  Global`AntLog["** About to run NotebookGet **"];
  expr  =  NotebookGet[nbobj2];
  Global`AntLog["** Finished running NotebookGet **"];
  Global`AntLog["** About to run NotebookClose **"];
           NotebookClose[nbobj2];
  Global`AntLog["** Finished running NotebookClose **"];
  Global`AntLog["** Finished running a series of FrontEnd commands that apply the various stylesheet options/rules **"];
 
  nbFile = StringReplace[outfile, ".html"->".nb"];
  CreateDirectory[ DirectoryName@nbFile];

(*   Put[ CurrentValue[$FrontEnd, StyleSheetPath], nbFile];
   Put[Rasterize[nbget], nbFile<>"1.txt"];
   Put[Rasterize[expr], nbFile<>"2.txt"]; *)

  If[DocumentationBuild`Transform`debug === True,
    Put[ expr, nbFile];
  ];

  (* now call Transmogrify with the appropriate parameters set.
     note that language extension by default goes before the .html *)
  LogIt["  Off to transmogrify"];
  Global`AntLog["** Off to Transmogrify **"];
  res = Developer`UseFrontEnd @ Transmogrify`Transmogrify[ outfile, expr, transform, 
      Transmogrify`DefaultParameters -> {
      "layoutfile"->layoutfile, 
      "popupLayoutfile"->popupLayoutfile, 
      	"titleName"->titleName, 
      	"Language"->language, "urlBaseRemote"->urlBaseRemote, "InFileName"->infilename, "WhatsNewID"->WhatsNewID, "MarketingCategoryName"->MarketingCategoryName, "ExamplesRawMaterialDir"->ExamplesRawMaterialDir, "Preview"->Preview, "description"->description, "keywords"->keywordList, "synonyms"->synonymList, "localcss"->localcss, "filename"->outfile}
    ];
    If[Head@res =!= String, First@res, res]

];

Options[ExportMarketingWebPage] = {
  DefaultParameters->{}, Keywords->{}, LogFile->None, Test->False};

ExportMarketingCategoryWebPages[_,$Failed,___] = $Failed;

ExportMarketingCategoryWebPages[outputDir_String, nb_String, language_String:"English", marketingpagedir_, opts___?OptionQ]:= 
  ExportMarketingCategoryWebPages[outputDir, DocumentationBuild`Utils`GetQuietly[nb], language, marketingpagedir, opts ];
ExportMarketingCategoryWebPages[outputDir_String, nbExpr_Notebook, language_String:"English", marketingpagedir_, opts___?OptionQ]:=
Developer`UseFrontEnd @
Module[{ext, properties={}, transform, keywordList={}, titleName, res, nbObj, tmp, parentDir, nbFile, localcss, 
  layoutfile, popupLayoutfile, expr, filename, jatitle, marketingpagefolder=marketingpagedir},

  LogIt["** Entering ExportMarketingCategoryWebPages **"];
  (* Add DocumentationBuild\XMLTransforms if it isn't already on path *)
  If[ !MemberQ[ToFileName[{DocumentationBuild`Common`$DocumentationBuildDirectory, "XMLTransforms"}], Transmogrify`$XMLTransformPath], 
    AppendTo[Transmogrify`$XMLTransformPath, ToFileName[{DocumentationBuild`Common`$DocumentationBuildDirectory, "XMLTransforms"}] ];
  ];

  ext = Switch[language, "Japanese", "ja", "Russian", "ru", "Korean", "ko", "German", "de", "French", "fr", "Portuguese", "pt-br", "Spanish", "es", "ChineseSimplified", "zh", _, "en"];
  
  Preview = "Preview" /. {opts};
  
  transform = {"HTML","MarketingCategoryPages.m"};
  layoutfile = ToFileName[{
     DocumentationBuild`Common`$DocumentationBuildDirectory,"XMLTransforms","HTML","Layouts"}, "MarketingPages.html"];

  res = Developer`UseFrontEnd @ Transmogrify`Transmogrify[ nbExpr, transform, 
      Transmogrify`DefaultParameters -> {
      	"outputDir" -> outputDir,
      	"layoutfile"->layoutfile, 
      	"marketingpagefolder"-> marketingpagefolder,
      	"Language"->language}
    ];
    If[Head@res =!= String, First@res, res]

];

Options[ExportMarketingCategoryWebPages] = {
  DefaultParameters->{}, Keywords->{}, LogFile->None, Test->False};


(* Create Marketing page notebook *)
MakeMarketingPageNotebook[nb_String, language_String:"English", opts___?OptionQ ]:= 
  MakeMarketingPageNotebook[DocumentationBuild`Utils`GetQuietly[nb], language, opts];
MakeMarketingPageNotebook[nbExpr_Notebook, language_String:"English", opts___?OptionQ ]:= 
Module[{transform, expr=nbExpr, pageTitle = "None", keywordList = {}, seeAlsoGroup, poss, exampleCaption },

  (* Change grouped styles to aid in conversion *)
  expr = expr /. {Cell[CellGroupData[{ Cell[a_, "Delimiter", o___], Cell[b_, "MarketingText", op___], cells___ }, oc___]] :>
    Cell[CellGroupData[ Flatten[{ 
      Cell[a, "HighlightSection", o, CellGroupingRules->{"SectionGrouping", 61}], 
      Cell[b, "HighlightItem", op],
      textStyleToItem[{cells}] 
    }], oc]] };

  (* Change '>>' to \[RightGuillemet] *)
  expr = expr /. ButtonBox[">>", r___] :> ButtonBox["\[RightGuillemet]", r];
expr = expr /. Cell[a___, Rule[CellFrameLabels, {{cf1_, cf2_}, {cf3_, Cell[___, "TextAnnotation",___]}}], b___] :> Cell[a, b];
  If[("Preview"/.{opts})===True,
  expr = DeleteCases[expr, Cell[CellGroupData[{Cell[___, "FutureExample"(* | "ExcludedMaterial"*), ___], ___}, ___], ___], Infinity];
  expr = DeleteCases[expr, Cell[___, "FutureExample" | "ExcludedMaterial", ___], Infinity];
  expr = DeleteCases[expr, Cell[___, CellDingbat->Cell[__, "BrightFlag", ___], ___], Infinity];
  expr = expr /. Cell[CellGroupData[{Cell[aa___, bb_?"FutureExample" | "TentativeExample" | "ExcludedMaterial", cc___], dd___}, ee___], ff___] :> Cell[CellGroupData[{Cell[aa, cc], dd}, ee], ff];
  expr = expr /. Cell[aa___, bb_?"FutureExample" | "TentativeExample" | "ExcludedMaterial", cc___] :> Cell[aa, cc];
  expr = expr /. Cell[aa___, Rule[CellDingbat,Cell[cc___, "TentativeID" | "BrightFlag", dd___]], ee___]:> Cell[aa, ee]
  ,
  expr = DeleteCases[expr, Cell[CellGroupData[{Cell[___, "FutureExample" | "TentativeExample"(* | "ExcludedMaterial"*), ___], ___}, ___], ___], Infinity];
  expr = DeleteCases[expr, Cell[___, "FutureExample" | "TentativeExample" | "ExcludedMaterial", ___], Infinity];
  expr = DeleteCases[expr, Cell[___, CellDingbat->Cell[__, "TentativeID" | "BrightFlag", ___], ___], Infinity]
  ];
  
    expr = DeleteCases[expr, Cell[___, "History", ___], Infinity];
  (* Add in styles used for grouping *)
  expr = expr /.
   Cell[CellGroupData[{Cell[c_, "MarketingTitle", o___], p___}, q___], r___] :> 
   Cell[CellGroupData[{
     Cell[c, "MarketingTitle", o],
     Cell["SUBHEADER", "MarketingTitleSubheader", CellGroupingRules->{"SectionGrouping", 60}],
     Cell[c, "MarketingTitleAlt", o, CellGroupingRules->{"SectionGrouping", 60}],
     Cell[CellGroupData[{Cell["", "MarketingMainSection", CellGroupingRules->{"SectionGrouping", 40}],
     Cell[CellGroupData[{p}, q], r]}]]}]];
  expr = expr /. Cell[c_, "MarketingText", o___] :> 
   Cell[c, "MarketingText", o, CellGroupingRules->{"SectionGrouping", 60}];
     expr = expr /. Cell[c_, "MarketingExampleDetail", o___] :> 
   Cell[c, "MarketingExampleDetail", o, CellGroupingRules->{"SectionGrouping", 60}];

  (*  Make callouts *)
  If[Length@Cases[expr, Cell[_, "Output", ___], Infinity] :> 0,
    exampleCaption = Cases[expr, Cell[_, "MarketingExampleCaption", ___], Infinity];
    expr = DeleteCases[expr, Cell[_, "MarketingExampleCaption", ___], Infinity];
    poss = First @ Position[expr, Cell[_, "Output", ___], Infinity];
    expr = ReplacePart[expr, { poss -> Cell[CellGroupData[Flatten@{ Extract[expr, poss], exampleCaption}]] }];

    expr = expr /. Cell[c_, "MarketingExampleCaption", o___] :> 
     Cell[c, "calloutbox", o, CellGroupingRules->"OutputGrouping", FontColor->None, FontSize->1, CellMargins->{{0,0},{-2,-2}}];
    expr = expr /. Cell[c_, "MarketingExampleDetail", o___] :> 
     Cell[c, "calloutbox", o, CellGroupingRules->"OutputGrouping", FontColor->None, FontSize->1, CellMargins->{{0,0},{-2,-2}}];
  ];

    expr = expr /. Cell[con_, sty___,"AlternateThumbnail", o___] :> Cell[con, "AlternateThumbnail", o];
  

  expr = expr /. Cell[BoxData[Cell[a__, "Graphics", b__]], _, c__] :>
                   Cell[BoxData[Cell[a, NotebookDefault, b]], "Output", c];
  
  (* Change Input cells *)
  expr = expr /. Cell[con_, "Input", o___] /; DocumentationBuild`Utils`IsManipulateQ @ Cell[con, "Input", o] :> Cell[con, "Output", o]; 
  expr = expr /. Cell[con_, "Input", "PopupMaterial", o___] :> Cell[con, "InputPopupMaterial", o];
  expr = expr /. Cell[con_, "Input", "MainPageMaterial", o___] :> Cell[con, "InputMainPageMaterial", o];
  expr = expr /. Cell[con_, "Input", "MainPopupMaterial", o___] :> Cell[con, "InputMainPopupMaterial", o];

  expr = expr /. Cell[con_, "Input", "HiddenMaterial", o___] :> Cell[con, "InputHiddenMaterial", o];
  expr = expr /. Cell[con_, "Input", "SuppressInputPopupText", o___] :> Cell[con, "InputNoClipboardAction", o];
  expr = expr /. Cell[con_, "Program", "HiddenMaterial", o___] :> Cell[con, "ProgramHiddenMaterial", o];
  expr = DeleteCases[expr, Cell[con_, "Input", "SuppressedMaterial", o___], Infinity];
  expr = DeleteCases[expr, Cell[con___, "SuppressedMaterial", o___], Infinity];
  expr = expr /. Cell[con_, "Input", "ThumbnailMaterial", o___] :> Cell[con, "InputThumbnailMaterial", o];
  
  (* TODO: Fix replacement rule to catch multiple styles *)
  (* Catch other style variations *)
  expr = expr /. Cell[con_, "Input", "ConversionSpecial", o___] :> Cell[con, "OutputSpecial", o, CellTags->"NoFlash"];
  expr = expr /. Cell[con_ /; DocumentationBuild`Utils`IsManipulateQ@con, _, "ConversionSpecial", o___] :> 
    Cell[con, "OutputSpecial", o, CellTags->"NoFlash"];
  
(*  expr = expr /. Cell[con_, "Output", "ConversionSpecial", o___] :> Cell[con, "OutputSpecial", o, CellTags->"NoFlash"];*)

  expr = expr /. Cell[con_, "Input", sty_String, o___] :> Cell[con, "Input", o];

  expr = expr /. Cell[con_, sty_String, "TR", o___] :> Cell[con, sty, o, FontFamily->"Times"];
  expr = expr /. Cell[con_, sty_String, "TI", o___] :> Cell[con, sty, o, FontFamily->"Times", FontSlant->"Italic"];

  expr = expr /. Cell[con_, "Message",  "MSG", o___] :> Cell[con, "Message", o];
  
  expr = expr /. Cell[con_, sty_String, "AltThumbnail", o___] :> Cell[con, sty, o, CellTags->"AltThumbnail"];
    expr = expr /. Cell[con_, sty___,"AlternateThumbnail", o___] :> Cell[con, "AlternateThumbnail", o];
  
  expr = expr /. Cell[con_, "AltThumbnail", o___] :> Cell[con, "AltThumbnail", o, CellTags->"AltThumbnail"];

  expr = expr /. Cell[con_, "NotebookImage", "ThumbnailMaterial", o___] :> Cell[con, "NotebookImage", o];

  expr = expr /. Cell[con_, sty1_String,  sty2_String, o___] :> Cell[con, "Output", o]; 

  expr = expr /. Cell[con_, "Graphics"|"Picture"|"Output", "ThumbnailMaterial", o___] :> Cell[con, "Output", o];


  (* Remove XXX* cells *)
  expr = expr /. Cell[c_String /; StringMatchQ[c, RegularExpression["XX*"]], "MarketingText", d___] :> Cell["xxx", "MarketingText", d];
  
  expr = DeleteCases[expr, Cell[c_String /; StringMatchQ[c, RegularExpression["XX*"]], ___], Infinity];
  expr = DeleteCases[expr, Cell[BoxData[c_String /; StringMatchQ[c, RegularExpression["XX*"]]], ___], Infinity];

  (* convert Function string to buttons *)
  expr = expr /. Cell[c_String, "MarketingRelatedFunctions", o___] :> 
    DocumentationBuild`Make`Private`inlineFunctionStringToButtonBox @ Cell[c, "MarketingRelatedFunctions", o];

  (* Remove XXX* cells *)
  expr = DeleteCases[expr, Cell[c_String /; StringMatchQ[c, RegularExpression["XX*"]], ___], Infinity];
  expr = DeleteCases[expr, 
    Cell[_  sty_String /; StringMatchQ[sty, "DocumentStatusSection" | "DocumentStatus"], ___], Infinity];

  (* Remove empty cell groups *)
  expr = RemoveEmptyCellGroups[expr, {
           "KeywordsSection", "DescriptionSection", "MarketingContentsSection", "MarketingMoreAboutSection", "MarketingRelatedDemonstrationsSection", 
           "MarketingRelatedFunctionsSection", "MarketingSeeAlsoSection", "MarketingSubpageMoreAboutSection", 
           "MarketingSubpageRelatedDemonstrationsSection", "MarketingSubpageRelatedFunctionsSection", "MarketingSubpageSection",
           "MarketingSubpageSeeAlsoSection", "MarketingThumnailSection", "Keywords", "Description", "MarketingRelatedLinksSection", "DocumentStatusSection", "DocumentStatus"
           }];
           
  expr = expr /.
   Cell[CellGroupData[{Cell[c_, "MarketingSeeAlsoSection", o___], p___}, q___], r___] :> 
   Cell[CellGroupData[{
     Cell[c, "MarketingSeeAlsoSection", o],
     Cell["", "MarketingUnorderedList", CellGroupingRules->{"SectionGrouping", 60}],
     p
   }, q], r];
   
   expr = expr /.
   Cell[CellGroupData[{Cell[c_, "MarketingMoreAboutSection", o___], p___}, q___], r___] :> 
   Cell[CellGroupData[{
     Cell[c, "MarketingMoreAboutSection", o],
     Cell["", "MarketingUnorderedList", CellGroupingRules->{"SectionGrouping", 60}],
     p
   }, q], r];
   
   expr = expr /.
   Cell[CellGroupData[{Cell[c_, "MarketingRelatedDemonstrationsSection", o___], p___}, q___], r___] :> 
   Cell[CellGroupData[{
     Cell[c, "MarketingRelatedDemonstrationsSection", o],
     Cell["", "MarketingUnorderedList", CellGroupingRules->{"SectionGrouping", 60}],
     p
   }, q], r];
   
expr = expr /.
   Cell[CellGroupData[{Cell[c_, "MarketingRelatedLinksSection", o___], p___}, q___], r___] :> 
   Cell[CellGroupData[{
     Cell[c, "MarketingRelatedLinksSection", o],
     Cell["", "MarketingUnorderedList", CellGroupingRules->{"SectionGrouping", 60}],
     p
   }, q], r];
  (* Change See Also title per sw *)
  expr = expr /. Cell[_, "MarketingSeeAlsoSection", o___] :> Cell[Localized["MarketingSeeAlsoSection", language], "MarketingSeeAlsoSection", o];
  expr = expr /. Cell[_, "MarketingMoreAboutSection", o___] :> Cell[Localized["MarketingMoreAboutSection", language], "MarketingMoreAboutSection", o];
  expr = expr /. Cell[_, "MarketingRelatedFunctionsSection", o___] :> Cell[Localized["MarketingRelatedFunctionsSection", language], "MarketingRelatedFunctionsSection", o];
  expr = expr /. Cell[_, "MarketingRelatedLinksSection", o___] :> Cell[Localized["MarketingRelatedLinksSection", language], "MarketingRelatedLinksSection", o];
  expr = expr /. Cell[_, "MarketingRelatedDemonstrationsSection", o___] :> Cell[Localized["MarketingRelatedDemonstrationsSection", language], "MarketingRelatedDemonstrationsSection", o];
  (* Move See Also group to bottom per sw *)
  
  seeAlsoGroup = Cases[expr, Cell[CellGroupData[{Cell[_, "MarketingSeeAlsoSection", ___], ___},___],___], Infinity];
  expr = DeleteCases[expr, Cell[CellGroupData[{Cell[_, "MarketingSeeAlsoSection", ___], ___},___],___], Infinity];
  expr[[1]] = Flatten[Append[First[expr], seeAlsoGroup]];

  (* Change paclets to URLs *)

  expr = expr /. ButtonBox[c__, BaseStyle->_, o1___, ButtonData->str_String /; StringMatchQ[str, "paclet*"], o2___] :> 
    ButtonBox[c, BaseStyle->"Hyperlink", ButtonData->{URL[Which[Length[StringSplit[str, "#"]] === 2, 
 "http://reference.wolfram.com/language/" <> 
  StringReplace[StringSplit[str, "#"][[1]], "paclet:" -> ""] <> 
  ".html#" <> 
  StringReplace[StringSplit[str, "#"][[2]], "paclet:" -> ""], True, 
 "http://reference.wolfram.com/language/" <> 
  StringReplace[str, "paclet:" -> ""] <> ".html"]], None}, BaseStyle->"Hyperlink", o1, o2];

  expr = expr /. Cell[BoxData[ButtonBox[c_, BaseStyle->"Link", ___]], "InlineFormula", ___] :> 
    Cell[BoxData[ButtonBox[c, ButtonData->{URL["http://reference.wolfram.com/language/ref/"<> StringReplace[c, {"\""->""}] <>".html"], None}, BaseStyle->"Hyperlink"]], "InlineFormula"];

  (* manipulate some button boxen *)
   expr = ReplacePart[expr, (# -> DocumentationBuild`Utils`MakePacletButton[Extract[expr, #]]) & /@ Position[expr, ButtonBox[_String, ___]]];

  expr = expr /. Cell[con_String, "MarketingSeeAlso", o___] :> makeMarketingSeeAlsoButton@Cell[con, "MarketingSeeAlso", o];
  expr = expr /. Cell[con_String, "MarketingMoreAbout", o___] :> makeMarketingMoreAboutButton@Cell[con, "MarketingMoreAbout", o];
  expr = expr /. Cell[con_String, "MarketingRelatedFunctions", o___] :> makeMarketingRelatedFunctionsButton@Cell[con, "MarketingRelatedFunctions", o];
(*
  expr = expr //. { 
    Cell[TextData[{a___, s_String, b___}], 
      sty_String/; StringMatchQ[sty, "MarketingRelatedFunctions"], o___] :> 
    Cell[TextData[Flatten@{a, makeMarketingRelatedFunctionsButton[s], b}
      ], sty, o] };
*)  
  (* change \[EmptyVerySmallSquare] to "something else" *)
  expr = expr //. Cell[TextData[{a___, s_String /; StringMatchQ[s, RegularExpression["\\s+\[EmptyVerySmallSquare]\\s+"] ], b___}], 
    "MarketingRelatedFunctions", o___] :> 
    Cell[TextData[{a, " ", b}], "MarketingRelatedFunctions", o];

  expr = expr /. Cell[BoxData[con_], "MarketingText", o___] :> Cell[BoxData[con], "Output", o];

  (* Renumber In/Out Cell labels *)
  expr = RenumberCells[expr, {"MarketingSubpageSection","ExampleDelimiter"}, 
           InputStyles->{"InputNoClipboardAction", "InputHiddenMaterial", "InputSuppressedMaterial", "InputThumbnailMaterial", "Input"}];

  expr = DeleteCases[expr, Rule[CellID, ___], Infinity];

  (* Change WindowSize to a width for nicer line wrapping in gifs *)
  expr = DocumentationBuild`Utils`Private`ResetOptions[expr, 
           WindowSize -> {700, 700}, ShowCellBracket->True, (*PageWidth->WindowWidth,*)
           StyleDefinitions -> 
             Notebook[{Cell[ StyleData[ StyleDefinitions -> FrontEnd`FileName[{"Wolfram"}, "MarketingPageStyles.nb"]]](*, 
               Cell[StyleData["Input"], FontSize -> 11, CellID -> 1,PageWidth->525 ], 
               Cell[StyleData["Output"], FontSize -> 11, CellID -> 1], 
               Cell[StyleData["Graphics"], CellID -> 1, PageWidth->525 ], 
               Cell[StyleData["Picture"], CellID -> 1, PageWidth->525 ], 
               Cell[StyleData["InputHiddenMaterial", StyleDefinitions->StyleData["Input"]], CellID -> 1], 
               Cell[StyleData["InputNoClipboardAction", StyleDefinitions->StyleData["Input"]], CellID -> 1], 
               Cell[StyleData["InputSuppressedMaterial", StyleDefinitions->StyleData["Input"]], CellID -> 1], 
               Cell[StyleData["InputThumbnailMaterial", StyleDefinitions->StyleData["Input"]], CellID -> 1]*)
               }]];

  expr
];

(* Create Solutions page notebook *)
MakeSolutionsPageNotebook[nb_String, language_String:"English", opts___?OptionQ ]:= 
  MakeSolutionsPageNotebook[DocumentationBuild`Utils`GetQuietly[nb], language, opts];
MakeSolutionsPageNotebook[nbExpr_Notebook, language_String:"English", opts___?OptionQ ]:= 
Module[{transform, expr=nbExpr, pageTitle = "None", keywordList = {}, seeAlsoGroup, poss, exampleCaption },

  
    expr = DeleteCases[expr, Cell[___, "History", ___], Infinity];
    expr = DeleteCases[expr, ShowCellTags->True, Infinity];
    expr = DeleteCases[expr, Cell[__, "InternalNoteLink", ___],Infinity];
   	
   	expr = expr /.Cell[TextData[ButtonBox[a_String,
 	BaseStyle->"Hyperlink",
 	ButtonData->{URL[b_String/;
 		StringJoin[
 			Insert[ Riffle[Drop[Drop[StringSplit[b, "/"], 1], -1], "/"],  "http:/" , 1 ]
 		]==="http://reference.wolfram.com/language/ref"], None}, c__]], "SolutionsListing", d__]:>
 	Cell[TextData[StyleBox[ButtonBox[a, BaseStyle->"Hyperlink", ButtonData->{
   	URL[b], None}, c], "InlineFormulaListItem"]], "SolutionsListing", d];


  (* Remove XXX* cells *)
  expr = DeleteCases[expr, Cell[c_String /; StringMatchQ[c, RegularExpression["XX*"]], ___], Infinity];
  expr = DeleteCases[expr, Cell[BoxData[c_String /; StringMatchQ[c, RegularExpression["XX*"]]], ___], Infinity];

 (* Remove XXX* cells *)
  expr = DeleteCases[expr, Cell[c_String /; StringMatchQ[c, RegularExpression["XX*"]], ___], Infinity];
  expr = DeleteCases[expr, 
    Cell[_  sty_String /; StringMatchQ[sty, "DocumentStatusSection" | "DocumentStatus"], ___], Infinity];

  (* Remove empty cell groups *)
  expr = RemoveEmptyCellGroups[expr, {
           "KeywordsSection", "Keywords", "DocumentStatusSection", "DocumentStatus"
           }];

  (* Change paclets to URLs *)
  expr = expr /. ButtonBox[c__, BaseStyle->_, o1___, ButtonData->str_String /; StringMatchQ[str, "paclet*"], o2___] :> 
    ButtonBox[c, BaseStyle->"Hyperlink", ButtonData->{URL["http://reference.wolfram.com/language/"<>StringReplace[str,"paclet:"->""]<>".html"], 
    	None}, BaseStyle->"Hyperlink", o1, o2];
    	
  expr = expr /. Cell[BoxData[FormBox[ButtonBox["\[RightGuillemet]", BaseStyle->"Hyperlink",a__], TraditionalForm]]]:>
  ButtonBox["\[RightGuillemet]",BaseStyle->"Hyperlink",a];
  
  (* manipulate some button boxen *)
  expr = ReplacePart[expr, (# -> DocumentationBuild`Utils`MakePacletButton[Extract[expr, #]]) & /@ Position[expr, ButtonBox[_String, ___]]];

  (* Change WindowSize to a width for nicer line wrapping in gifs *)
  expr = DocumentationBuild`Utils`Private`ResetOptions[expr, 
           WindowSize -> {700, 700}, ShowCellBracket->False, Magnification->1,(*PageWidth->WindowWidth,*)
           StyleDefinitions -> 
             Notebook[{Cell[ StyleData[ StyleDefinitions -> FrontEnd`FileName[{"Wolfram"}, "SolutionsPageStyles.nb"]]], 
               Cell[StyleData["Input"], FontSize -> 11, CellID -> 1,PageWidth->525 ], 
               Cell[StyleData["Output"], FontSize -> 11, CellID -> 1], 
               Cell[StyleData["Graphics"], Magnification -> 1, CellID -> 1, PageWidth->525 ], 
               Cell[StyleData["Picture"], Magnification -> 1, CellID -> 1, PageWidth->525 ], 
               Cell[StyleData["InputHiddenMaterial", StyleDefinitions->StyleData["Input"]], CellID -> 1], 
               Cell[StyleData["InputSuppressedMaterial", StyleDefinitions->StyleData["Input"]], CellID -> 1], 
               Cell[StyleData["InputThumbnailMaterial", StyleDefinitions->StyleData["Input"]], CellID -> 1]
               }]];

  expr
];

(* Create Updated 2012 Solutions page notebook *)
MakeUpdatedSolutionsNotebook[nb_String, language_String:"English", opts___?OptionQ ]:= 
  MakeUpdatedSolutionsNotebook[DocumentationBuild`Utils`GetQuietly[nb], language, opts];
MakeUpdatedSolutionsNotebook[nbExpr_Notebook, language_String:"English", opts___?OptionQ ]:= 
Module[{transform, expr=nbExpr, pageTitle = "None", keywordList = {}, seeAlsoGroup, poss, exampleCaption },

  
    expr = DeleteCases[expr, Cell[___, "History", ___], Infinity];
    expr = DeleteCases[expr, ShowCellTags->True, Infinity];
    expr = DeleteCases[expr, Cell[__, "InternalNoteLink", ___],Infinity];
   	
   	expr = expr /.Cell[TextData[ButtonBox[a_String,
 	BaseStyle->"Hyperlink",
 	ButtonData->{URL[b_String/;
 		StringJoin[
 			Insert[ Riffle[Drop[Drop[StringSplit[b, "/"], 1], -1], "/"],  "http:/" , 1 ]
 		]==="http://reference.wolfram.com/language/ref"], None}, c__]], "SolutionsListing", d__]:>
 	Cell[TextData[StyleBox[ButtonBox[a, BaseStyle->"Hyperlink", ButtonData->{
   	URL[b], None}, c], "InlineFormulaListItem"]], "SolutionsListing", d];


  (* Remove XXX* cells *)
  expr = DeleteCases[expr, Cell[c_String /; StringMatchQ[c, RegularExpression["XX*"]], ___], Infinity];
  expr = DeleteCases[expr, Cell[BoxData[c_String /; StringMatchQ[c, RegularExpression["XX*"]]], ___], Infinity];

 (* Remove XXX* cells *)
  expr = DeleteCases[expr, Cell[c_String /; StringMatchQ[c, RegularExpression["XX*"]], ___], Infinity];
  expr = DeleteCases[expr, 
    Cell[_  sty_String /; StringMatchQ[sty, "DocumentStatusSection" | "DocumentStatus"], ___], Infinity];

  (* Remove empty cell groups *)
  expr = RemoveEmptyCellGroups[expr, {
           "KeywordsSection", "Keywords", "DocumentStatusSection", "DocumentStatus"
           }];

  (* Change paclets to URLs *)
  expr = expr /. ButtonBox[c__, BaseStyle->_, o1___, ButtonData->str_String /; StringMatchQ[str, "paclet*"], o2___] :> 
    ButtonBox[c, BaseStyle->"Hyperlink", ButtonData->{URL["http://reference.wolfram.com/language/"<>StringReplace[str,"paclet:"->""]<>".html"], 
    	None}, BaseStyle->"Hyperlink", o1, o2];
    	
  expr = expr /. Cell[BoxData[FormBox[ButtonBox["\[RightGuillemet]", BaseStyle->"Hyperlink",a__], TraditionalForm]]]:>
  ButtonBox["\[RightGuillemet]",BaseStyle->"Hyperlink",a];
  
  (* manipulate some button boxen *)
  expr = ReplacePart[expr, (# -> DocumentationBuild`Utils`MakePacletButton[Extract[expr, #]]) & /@ Position[expr, ButtonBox[_String, ___]]];

  (* Change WindowSize to a width for nicer line wrapping in gifs *)
  expr = DocumentationBuild`Utils`Private`ResetOptions[expr, 
           WindowSize -> {700, 700}, ShowCellBracket->False, Magnification->1,(*PageWidth->WindowWidth,*)
           StyleDefinitions -> 
             Notebook[{Cell[ StyleData[ StyleDefinitions -> FrontEnd`FileName[{"Wolfram"}, "SolutionsPageStyles.nb"]]], 
               Cell[StyleData["Input"], FontSize -> 11, CellID -> 1,PageWidth->525 ], 
               Cell[StyleData["Output"], FontSize -> 11, CellID -> 1], 
               Cell[StyleData["Graphics"], Magnification -> 1, CellID -> 1, PageWidth->525 ], 
               Cell[StyleData["Picture"], Magnification -> 1, CellID -> 1, PageWidth->525 ], 
               Cell[StyleData["InputHiddenMaterial", StyleDefinitions->StyleData["Input"]], CellID -> 1], 
               Cell[StyleData["InputSuppressedMaterial", StyleDefinitions->StyleData["Input"]], CellID -> 1], 
               Cell[StyleData["InputThumbnailMaterial", StyleDefinitions->StyleData["Input"]], CellID -> 1]
               }]];

  expr
];



(* ExportUpdatedSolutionsPage[outfile, expr, pageType,= opts] *)
ExportUpdatedSolutionsPage[_,$Failed,___] = $Failed;

Options[ExportUpdatedSolutionsPage] = {
    "DefaultParameters" -> {},
    "IsManipulate" -> True,
    "Language" -> "English",
    "SolutionsTest" -> True,
    "TMJTest" -> False
    
};

ExportUpdatedSolutionsPage[file_String, nb_String, language_String:"English", opts___?OptionQ]:= 
  ExportUpdatedSolutionsPage[file, DocumentationBuild`Utils`GetQuietly[nb], language, opts ];
ExportUpdatedSolutionsPage[file_String, nbExpr_Notebook, language_String:"English", opts___?OptionQ]:=
Developer`UseFrontEnd @
Module[{ext, properties={}, outfile=file, transform, keywordList={}, synonymList={}, summaryText="", titleName, res, nbObj, tmp, parentDir, nbFile, localcss, 
  layoutfile, popupLayoutfile, expr, filename, jatitle, altTitleName, nbFileName, isSolutionsQ, isTMJQ, descriptionText},

	isSolutionsQ = "SolutionsTest" /. {opts} /. Options[ExportUpdatedSolutionsPage];
	isTMJQ = "TMJTest" /. {opts} /. Options[ExportUpdatedSolutionsPage];
    nbFileName=opts[[2]];

  LogIt["** Entering ExportUpdatedSolutionsPage **"];
  (* Add DocumentationBuild\XMLTransforms if it isn't already on path *)
  If[ !MemberQ[ToFileName[{DocumentationBuild`Common`$DocumentationBuildDirectory, "XMLTransforms"}], Transmogrify`$XMLTransformPath], 
    AppendTo[Transmogrify`$XMLTransformPath, ToFileName[{DocumentationBuild`Common`$DocumentationBuildDirectory, "XMLTransforms"}] ];
  ];

  ext = Switch[language, 
  	"Japanese", "ja", 
  	"Spanish", "es", 
  	"ChineseSimplified", "zh", 
  	"Russian", "ru",
	"Korean", "ko",
	"German", "de",
	"French", "fr",
	"Portuguese", "pt-br", 
  	_, "en"];
  
  Preview = "Preview" /. {opts};
  
  transform = {"HTML","Solutions2012.m"};
  layoutfile = ToFileName[{
     DocumentationBuild`Common`$DocumentationBuildDirectory,"XMLTransforms","HTML","Layouts"}, "Solutions2012.html"];

  (* keywords *)
  keywordList = DocumentationBuild`Info`GetNotebookKeywords[nbExpr];
  synonymList = DocumentationBuild`Info`GetNotebookSynonyms[nbExpr];
  summaryText = DocumentationBuild`Info`GetNotebookSummary[nbExpr, "Solutions Pages"];
  
  LogIt["  Keywords: "<>ToString[keywordList]];

  titleName = DocumentationBuild`Info`GetNotebookTitle[nbExpr, "Solution"];
  altTitleName = Cases[nbExpr,Cell[a_,"Categorization",CellLabel->"ForcedPageTitle",b___]:>a,Infinity];
  descriptionText = Last[Cases[nbExpr, CellGroupData[{Cell[a_, "WebMetadataSection", b__], c___}, d___]:>c, Infinity]];

  LogIt["  Title name: "<>ToString[titleName]];
  outfile = 
    If[ StringQ[ext],
      StringReplace[outfile, RegularExpression["(?<!\\.\\w\\w)(\\.html)"]->"."<>ext<>"$1"],
      outfile];
  localcss = StringReplace[DirectoryName@outfile, {DirectoryName@DirectoryName@outfile->"", "\""->"", "/"->"", $PathnameSeparator -> ""}] <> "-localcss";

  (* Create preprocess notebook *)
  LogIt["  MakeUpdatedSolutionsNotebook"];
  
  expr = MakeUpdatedSolutionsNotebook[nbExpr, language, opts];

  LogIt["  NotebookPut"];
  nbObj = DocumentationBuild`Common`PubsEvaluateWithFE@NotebookPut[expr];

  LogIt["  Export"];
  tmp = OpenTemporary[];
  res = Export[First@tmp<>".nb", NotebookGet@nbObj, "NB"];
  FrontEnd`NotebookClose[nbObj];
  expr = Get[First@tmp<>".nb"];
  DeleteFile[First@tmp<>".nb"];
  NotebookClose[nbObj];
  
  (* Merge ButtonBoxes *)
  expr = expr /. a_TextData -> ConvertTextData[a];

  nbFile = StringReplace[outfile, ".html"->".nb"];
  CreateDirectory[ DirectoryName@nbFile];

  If[DocumentationBuild`Transform`debug === True,
    Put[ expr, nbFile];
  ];

  (* now call Transmogrify with the appropriate parameters set.
     note that language extension by default goes before the .html *)
  LogIt["  Off to transmogrify"];

  res = Developer`UseFrontEnd @ Transmogrify`Transmogrify[ outfile, expr, transform, 
      Transmogrify`DefaultParameters -> {"layoutfile"->layoutfile, "popupLayoutfile"->popupLayoutfile, "nbFileName"->nbFileName, "IsSolutionsQ"->isSolutionsQ, "IsTMJQ"->isTMJQ, 
      	"titleName"->titleName, "altTitleName"->altTitleName, "Language"->language, "Preview"->Preview,"keywords"->keywordList, "synonyms"->synonymList, 
      	"summary"->summaryText, "description"->descriptionText, "localcss"->localcss, "filename"->outfile}
    ];
    If[Head@res =!= String, First@res, res]

];



(* Create TMJ page notebook *)
MakeTMJNotebook[nb_String, language_String:"English", opts___?OptionQ ]:= 
  MakeTMJNotebook[DocumentationBuild`Utils`GetQuietly[nb], language, opts];
MakeTMJNotebook[nbExpr_Notebook, language_String:"English", opts___?OptionQ ]:= 
Module[{transform, expr=nbExpr, pageTitle = "None", keywordList = {}, seeAlsoGroup, poss, exampleCaption },

  (* Remove XXX* cells *)
  expr = DeleteCases[expr, Cell[c_String /; StringMatchQ[c, RegularExpression["XX*"]], ___], Infinity];
  expr = DeleteCases[expr, Cell[BoxData[c_String /; StringMatchQ[c, RegularExpression["XX*"]]], ___], Infinity];

 (* Remove XXX* cells *)
  expr = DeleteCases[expr, Cell[c_String /; StringMatchQ[c, RegularExpression["XX*"]], ___], Infinity];
  expr = DeleteCases[expr, 
    Cell[_  sty_String /; StringMatchQ[sty, "DocumentStatusSection" | "DocumentStatus"], ___], Infinity];

  (* Remove empty cell groups *)
  expr = RemoveEmptyCellGroups[expr, {
           "KeywordsSection", "Keywords", "DocumentStatusSection", "DocumentStatus"
           }];
           
  expr = expr /. CounterBox["Reference", a_]:> ButtonBox[CounterBox["Reference", a], "ReferenceCounterBox", ButtonData->a];
  
  expr = expr /. CounterBox["DisplayFormulaNumbered", a_]:> ButtonBox[CounterBox["DisplayFormulaNumbered", a], "DisplayFormulaNumbered", ButtonData->a];

  (* Change paclets to URLs *)
  expr = expr /. ButtonBox[c__, BaseStyle->_, o1___, ButtonData->str_String /; StringMatchQ[str, "paclet*"], o2___] :> 
    ButtonBox[c, BaseStyle->"Hyperlink", ButtonData->{URL["http://reference.wolfram.com/language/"<>StringReplace[str,"paclet:"->""]<>".html"], None}, BaseStyle->"Hyperlink", o1, o2];

  (* manipulate some button boxen *)
  expr = ReplacePart[expr, (# -> DocumentationBuild`Utils`MakePacletButton[Extract[expr, #]]) & /@ Position[expr, ButtonBox[_String, ___]]];

  expr = DocumentationBuild`Utils`Private`ResetOptions[expr, 
           WindowSize -> {700, 700}, ShowCellBracket->False, 
           StyleDefinitions -> Import[
FileNameJoin[{DocumentationBuild`Common`$DocumentationBuildDirectory, "Internal", "web", "html", "TMJExportStyle.nb"}]
, "Notebook"]
           (*, 
           StyleDefinitions -> 
             Notebook[{Cell[ StyleData[ StyleDefinitions -> FrontEnd`FileName[{"Wolfram"}, "SolutionsPageStyles.nb"]]], 
               Cell[StyleData["Input"], FontSize -> 11, CellID -> 1,PageWidth->525 ], 
               Cell[StyleData["Output"], FontSize -> 11, CellID -> 1], 
               Cell[StyleData["Graphics"], Magnification -> 1, CellID -> 1, PageWidth->525 ], 
               Cell[StyleData["InputHiddenMaterial", StyleDefinitions->StyleData["Input"]], CellID -> 1], 
               Cell[StyleData["InputSuppressedMaterial", StyleDefinitions->StyleData["Input"]], CellID -> 1], 
               Cell[StyleData["InputThumbnailMaterial", StyleDefinitions->StyleData["Input"]], CellID -> 1]
               }]*)];
               
  expr
];


(* 
    ExportTMJWebPage[outfile, expr, pageType, opts] 
*)
ExportTMJWebPage[_,$Failed,___] = $Failed;

Options[ExportTMJWebPage] = {
    "DefaultParameters" -> {},
    "CompleteHTMLQ" -> True,
    "IsManipulate" -> True,
    "ImageFileDirectory" -> "Files",
    "ImageFilePath" -> "",
    "CoreInformation"-> Automatic,
    "Language" -> "English",
    "TMJTest" -> True
};


ExportTMJWebPage[file_String, nb_String, language_String:"English", opts___?OptionQ]:= 
  ExportTMJWebPage[file, DocumentationBuild`Utils`GetQuietly[nb], language, opts ];
ExportTMJWebPage[file_String, nbExpr_Notebook, language_String:"English", opts___?OptionQ]:=
Developer`UseFrontEnd @
Module[{ext, outfile=file, transform, keywordList={}, synonymList={}, summaryText="", res, nbObj, tmp, CompleteHTMLQ, isTMJQ, 
	htmlBase = "http://reference.wolfram.com", nbFile, localcss, layoutfile, expr, imageFileDir, imageFilePath, pacletName, url="tmj/tmj", extraTransformRules,
	urlBaseRemote = "http://reference.wolfram.com/language/"},
	
  imageFileDir= "ImageFileDirectory" /. {opts} /. Options[ExportTMJWebPage];
  imageFilePath= "ImageFilePath" /. {opts} /. Options[ExportTMJWebPage];
  CompleteHTMLQ = "CompleteHTMLQ" /. {opts} /. Options[ExportTMJWebPage];
  isTMJQ = "TMJTest" /. {opts} /. Options[ExportTMJWebPage];

  LogIt["** Entering ExportSolutionsPage **"];
  (* Add DocumentationBuild\XMLTransforms if it isn't already on path *)
  If[ !MemberQ[ToFileName[{DocumentationBuild`Common`$DocumentationBuildDirectory, "XMLTransforms"}], Transmogrify`$XMLTransformPath], 
    AppendTo[Transmogrify`$XMLTransformPath, ToFileName[{DocumentationBuild`Common`$DocumentationBuildDirectory, "XMLTransforms"}] ];
  ];

  ext = Switch[language, 
  	"Japanese", "ja", 
  	"Spanish", "es", 
  	"ChineseSimplified", "zh", 
  	_, "en"];
  
  Preview = "Preview" /. {opts};
  
  transform = {"HTML","TMJ.m"};
  layoutfile = ToFileName[{
     DocumentationBuild`Common`$DocumentationBuildDirectory,"XMLTransforms","HTML","Layouts"}, "TMJ.html"];
     
  (* keywords *)
  keywordList = DocumentationBuild`Info`GetNotebookKeywords[nbExpr];
  synonymList = DocumentationBuild`Info`GetNotebookSynonyms[nbExpr];
  summaryText = DocumentationBuild`Info`GetNotebookSummary[nbExpr, "Solutions Pages"];
  
  LogIt["  Keywords: "<>ToString[keywordList]];

  LogIt["  Title name: "<>ToString[titleName]];
  outfile = 
    If[ StringQ[ext],
      StringReplace[outfile, RegularExpression["(?<!\\.\\w\\w)(\\.html)"]->"."<>ext<>"$1"],
      outfile];
  localcss = StringReplace[DirectoryName@outfile, {DirectoryName@DirectoryName@outfile->"", "\""->"", "/"->"", $PathnameSeparator -> ""}] <> "-localcss";

  (* Create preprocess notebook *)
  LogIt["  MakeSolutionsPageNotebook"];
  
  expr = MakeTMJNotebook[nbExpr, language, opts];

  LogIt["  NotebookPut"];
  (*nbObj = DocumentationBuild`Common`PubsEvaluateWithFE@NotebookPut[expr];*)
  nbObj = NotebookPut[expr];
            FrontEndExecute[{FrontEnd`NotebookDynamicToLiteral[nbObj]}];

  LogIt["  Export"];
  tmp = OpenTemporary[];
  res = Export[First@tmp<>".nb", NotebookGet@nbObj, "NB"];
  FrontEnd`NotebookClose[nbObj];
  expr = Get[First@tmp<>".nb"];
  DeleteFile[First@tmp<>".nb"];
  NotebookClose[nbObj];
  
  (* Merge ButtonBoxes *)
  expr = expr /. a_TextData -> ConvertTextData[a];
  
  (*expr = expr /. Cell[a__, "MCode"]:>StyleBox[a, "Code"];*)
  expr = expr /. Cell[a__, "MCode"] :> Cell[a, "InlineFormula"]/;MatchQ[a, BoxData[_String]];
  
  expr = expr /.  Cell[TextData[{a__}], "Reference", b___]:>
   Cell[TextData[{"[", CounterBox["ReferenceList"], "] ", a}], "Reference", b];
  expr = expr /.  Cell[a_String, "Reference", b___]:>
   Cell[TextData[{"[", CounterBox["ReferenceList"], "] ", a}], "Reference", b];
  expr = expr /.   Cell[CellGroupData[{Cell["References","Section",x__], Cell[a__, "Reference", b___],
y__},Open]] :> 
	{Cell["References","Section",x], Cell[CellGroupData[{Cell[a, "Reference", b],
y},Open]]};
expr = expr /.   Cell[CellGroupData[{Cell["References","Section",x__], Cell[a_, "Reference", b___],
y__},Open]] :> 
	Sequence[Cell["References","Section"], Cell[CellGroupData[{Cell[a, "Reference", b],
y},Open]]];

expr = expr /. 
Cell[BoxData[FormBox[a_, TraditionalForm]]] :> Cell[BoxData[
    FormBox[a, TraditionalForm]], "Math"];
    
expr = expr /. Cell[a___, TextAlignment->b__, c___] :> Cell[a,c];
expr = expr /. Cell[a___, TextJustification->b__, c___] :> Cell[a,c];
expr = expr /. Cell[a__, b_String, "MakeDisplayImage", c___] :> Cell[a,b<>"MakeDisplayImage",c]; 

extraTransformRules = "{Cell, \"ProgramMakeDisplayImage\"}";
    
expr = expr /.    
    Cell[b_, "None"]:>
    Cell[b];


  nbFile = StringReplace[outfile, ".html"->".nb"];
  Quiet[CreateDirectory[ DirectoryName@nbFile], {CreateDirectory::filex}];

  If[DocumentationBuild`Transform`debug === True,
    Put[ expr, nbFile];
  ];

  (* now call Transmogrify with the appropriate parameters set.
     note that language extension by default goes before the .html *)
  LogIt["  Off to transmogrify"];

  res = Quiet[Developer`UseFrontEnd @ Transmogrify`Transmogrify[ outfile, expr, transform, 
      Transmogrify`DefaultParameters -> {"layoutfile"->layoutfile, "Language"->language, "ImageFileDirectory"->imageFileDir, "ImageFilePath"->imageFilePath, 
      	"keywords"->keywordList, "synonyms"->synonymList, "summary"->summaryText, "localcss"->localcss, "filename"->outfile, "urlBaseRemote"->urlBaseRemote, 
      	"htmlBase"->htmlBase, "CompleteHTMLQ"->CompleteHTMLQ, "pacletName"->pacletName, "url"->url, "IsTMJQ"->isTMJQ, "ExtraRules"->extraTransformRules}
    ], {CreateDirectory::filex}];
    If[Head@res =!= String, First@res, res]

];

Options[ExportMarketingWebPage] = {
  DefaultParameters->{}, Keywords->{}, LogFile->None, Test->False};
  
  
  
  
  
  



(* Create Wolfram Alpha API page notebook *)
MakeAlphaAPINotebook[nb_String, language_String:"English", opts___?OptionQ ]:= 
  MakeAlphaAPINotebook[DocumentationBuild`Utils`GetQuietly[nb], language, opts];
MakeAlphaAPINotebook[nbExpr_Notebook, language_String:"English", opts___?OptionQ ]:= 
Module[{expr=nbExpr, stylesheet},

stylesheet = Cases[Options[expr], Rule[StyleDefinitions, a_Notebook]->a, Infinity][[1]];

  (* Remove XXX* cells *)
  expr = DeleteCases[expr, Cell[c_String /; StringMatchQ[c, RegularExpression["XX*"]], ___], Infinity];
  expr = DeleteCases[expr, Cell[BoxData[c_String /; StringMatchQ[c, RegularExpression["XX*"]]], ___], Infinity];

 (* Remove XXX* cells *)
  expr = DeleteCases[expr, Cell[c_String /; StringMatchQ[c, RegularExpression["XX*"]], ___], Infinity];
  expr = DeleteCases[expr, 
    Cell[_  sty_String /; StringMatchQ[sty, "DocumentStatusSection" | "DocumentStatus"], ___], Infinity];

  (* Remove empty cell groups *)
  expr = RemoveEmptyCellGroups[expr, {
           "KeywordsSection", "Keywords", "DocumentStatusSection", "DocumentStatus"
           }];

  (* Change paclets to URLs *)
  expr = expr /. ButtonBox[c__, BaseStyle->_, o1___, ButtonData->str_String /; StringMatchQ[str, "paclet*"], o2___] :> 
    ButtonBox[c, BaseStyle->"Hyperlink", ButtonData->{URL["http://reference.wolfram.com/language/"<>StringReplace[str,"paclet:"->""]<>".html"], None}, BaseStyle->"Hyperlink", o1, o2];

  (* manipulate some button boxen *)
  expr = ReplacePart[expr, (# -> DocumentationBuild`Utils`MakePacletButton[Extract[expr, #]]) & /@ Position[expr, ButtonBox[_String, ___]]];

  expr = DeleteCases[expr, Cell[__, "StrainedGraphics"|"EpigraphAuthor",___], Infinity];

  expr = DocumentationBuild`Utils`Private`ResetOptions[expr, 
           WindowSize -> {750, 825}, ShowCellBracket->False, 
           StyleDefinitions -> stylesheet
];
               
  expr
];


(* 
    ExportTMJWebPage[outfile, expr, pageType, opts] 
*)
ExportAlphaAPIWebPage[_,$Failed,___] = $Failed;

Options[ExportAlphaAPIWebPage] = {
    "DefaultParameters" -> {},
    "CompleteHTMLQ" -> True,
    "IsManipulate" -> True,
    "ImageFileDirectory" -> "Files",
    "ImageFilePath" -> "",
    "CoreInformation"-> Automatic,
    "Language" -> "English",
    "SolutionsTest" -> False,
    "TMJTest" -> False
};




ExportAlphaAPIWebPage[file_String, nb_String, language_String:"English", opts___?OptionQ]:= 
  ExportAlphaAPIWebPage[file, DocumentationBuild`Utils`GetQuietly[nb], language, opts ];
ExportAlphaAPIWebPage[file_String, nbExpr_Notebook, language_String:"English", opts___?OptionQ]:=
Developer`UseFrontEnd @
Module[{ext, outfile=file, transform, keywordList={}, synonymList={}, summaryText="", res, nbObj, tmp, CompleteHTMLQ, isTMJQ, isSolutionsQ, 
	htmlBase = "http://www.wolframalpha.com", nbFile, localcss, layoutfile, expr, imageFileDir, imageFilePath, pacletName, url="tmj/tmj", extraTransformRules,
	urlBaseRemote = "http://www.wolframalpha.com/api/"},
	
  imageFileDir= "ImageFileDirectory" /. {opts} /. Options[ExportAlphaAPIWebPage];
  imageFilePath= "ImageFilePath" /. {opts} /. Options[ExportAlphaAPIWebPage];
  CompleteHTMLQ = "CompleteHTMLQ" /. {opts} /. Options[ExportAlphaAPIWebPage];
  isTMJQ = "TMJTest" /. {opts} /. Options[ExportAlphaAPIWebPage];
  isSolutionsQ = "SolutionsTest" /. {opts} /. Options[ExportSolutionsWebPage];

  LogIt["** Entering ExportSolutionsPage **"];
  (* Add DocumentationBuild\XMLTransforms if it isn't already on path *)
  If[ !MemberQ[ToFileName[{DocumentationBuild`Common`$DocumentationBuildDirectory, "XMLTransforms"}], Transmogrify`$XMLTransformPath], 
    AppendTo[Transmogrify`$XMLTransformPath, ToFileName[{DocumentationBuild`Common`$DocumentationBuildDirectory, "XMLTransforms"}] ];
  ];

  ext = Switch[language, 
  	"Japanese", "ja", 
  	"Spanish", "es", 
  	"ChineseSimplified", "zh", 
  	_, "en"];
  
  Preview = "Preview" /. {opts};
  
  transform = {"HTML","WolframAlphaAPI.m"};
  layoutfile = ToFileName[{
     DocumentationBuild`Common`$DocumentationBuildDirectory,"XMLTransforms","HTML","Layouts"}, "WolframAlphaAPI.html"];
     

  LogIt["  Title name: "<>ToString[titleName]];
  outfile = 
    If[ StringQ[ext],
      StringReplace[outfile, RegularExpression["(?<!\\.\\w\\w)(\\.html)"]->"."<>ext<>"$1"],
      outfile];
  localcss = StringReplace[DirectoryName@outfile, {DirectoryName@DirectoryName@outfile->"", "\""->"", "/"->"", $PathnameSeparator -> ""}] <> "-localcss";

  (* Create preprocess notebook *)
  LogIt["  MakeAlphaAPINotebook"];
  
  expr = MakeAlphaAPINotebook[nbExpr, language, opts];

  LogIt["  NotebookPut"];
  (*nbObj = DocumentationBuild`Common`PubsEvaluateWithFE@NotebookPut[expr];*)
  nbObj = NotebookPut[expr];
            FrontEndExecute[{FrontEnd`NotebookDynamicToLiteral[nbObj]}];

  LogIt["  Export"];
  tmp = OpenTemporary[];
  res = Export[First@tmp<>".nb", NotebookGet@nbObj, "NB"];
  FrontEnd`NotebookClose[nbObj];
  expr = Get[First@tmp<>".nb"];
  DeleteFile[First@tmp<>".nb"];
  NotebookClose[nbObj];
  
  (* Merge ButtonBoxes *)
  expr = expr /. a_TextData -> ConvertTextData[a];
  
  expr = expr /. Cell[a__, "MCode"] :> Cell[a, "InlineFormula"]/;MatchQ[a, BoxData[_String]];

expr = expr /. 
Cell[BoxData[FormBox[a_, TraditionalForm]]] :> Cell[BoxData[
    FormBox[a, TraditionalForm]], "Math"];
    
expr = expr /. Cell[a___, TextAlignment->b__, c___] :> Cell[a,c];
expr = expr /. Cell[a___, TextJustification->b__, c___] :> Cell[a,c];
expr = expr /. Cell[a__, b_String, "MakeDisplayImage", c___] :> Cell[a,b<>"MakeDisplayImage",c]; 
    
expr = expr /.    
    Cell[b_, "None"]:>
    Cell[b];
    
expr = expr /. {Cell[a__String, "ModInfo"], b__}:>{b};


  nbFile = StringReplace[outfile, ".html"->".nb"];
  Quiet[CreateDirectory[ DirectoryName@nbFile], {CreateDirectory::filex}];

  If[DocumentationBuild`Transform`debug === True,
    Put[ expr, nbFile];
  ];

  (* now call Transmogrify with the appropriate parameters set.
     note that language extension by default goes before the .html *)
  LogIt["  Off to transmogrify"];

  res = Quiet[Developer`UseFrontEnd @ Transmogrify`Transmogrify[ outfile, expr, transform, 
      Transmogrify`DefaultParameters -> {"layoutfile"->layoutfile, "Language"->language, "ImageFileDirectory"->imageFileDir, "ImageFilePath"->imageFilePath, 
      	"keywords"->keywordList, "synonyms"->synonymList, "summary"->summaryText, "localcss"->localcss, "filename"->outfile, "urlBaseRemote"->urlBaseRemote, 
      	"htmlBase"->htmlBase, "CompleteHTMLQ"->CompleteHTMLQ, "pacletName"->pacletName, "url"->url, "IsTMJQ"->isTMJQ,  "IsSolutionsQ"->isSolutionsQ, "ExtraRules"->extraTransformRules}
    ], {CreateDirectory::filex}];
    If[Head@res =!= String, First@res, res]

];



(* 
    ExportSolutionsWebPage[outfile, expr, pageType,= opts] 
    this is the main Transmogrify[] wrapper.
    NOTE - outfile = None just returns the raw Transmogrify output
           Test->True causes ExportMarketingWebPage to use TransmogrifyString[]
           the shorthand functions rawmogtest[] and mogtest[], respectively,
           are wrappers for these two functionalities.
*)
ExportSolutionsWebPage[_,$Failed,___] = $Failed;

Options[ExportSolutionsWebPage] = {
    "DefaultParameters" -> {},
    "IsManipulate" -> True,
    "Language" -> "English",
    "SolutionsTest" -> True,
    "TMJTest" -> False
    
};

ExportSolutionsWebPage[file_String, nb_String, language_String:"English", opts___?OptionQ]:= 
  ExportSolutionsWebPage[file, DocumentationBuild`Utils`GetQuietly[nb], language, opts ];
ExportSolutionsWebPage[file_String, nbExpr_Notebook, language_String:"English", opts___?OptionQ]:=
Developer`UseFrontEnd @
Module[{ext, properties={}, outfile=file, transform, keywordList={}, synonymList={}, summaryText="", titleName, res, nbObj, tmp, parentDir, nbFile, localcss, 
  layoutfile, popupLayoutfile, expr, filename, jatitle, altTitleName, nbFileName, isSolutionsQ, isTMJQ, descriptionText},

	isSolutionsQ = "SolutionsTest" /. {opts} /. Options[ExportSolutionsWebPage];
	isTMJQ = "TMJTest" /. {opts} /. Options[ExportSolutionsWebPage];
    nbFileName=opts[[2]];
    
    

  LogIt["** Entering ExportSolutionsPage **"];
  (* Add DocumentationBuild\XMLTransforms if it isn't already on path *)
  If[ !MemberQ[ToFileName[{DocumentationBuild`Common`$DocumentationBuildDirectory, "XMLTransforms"}], Transmogrify`$XMLTransformPath], 
    AppendTo[Transmogrify`$XMLTransformPath, ToFileName[{DocumentationBuild`Common`$DocumentationBuildDirectory, "XMLTransforms"}] ];
  ];

  ext = Switch[language, 
  	"Japanese", "ja", 
  	"Spanish", "es", 
  	"ChineseSimplified", "zh", 
  	"Russian", "ru",
	"Korean", "ko",
	"German", "de",
    "French", "fr",
	"Portuguese", "pt-br", 
  	_, "en"];
  
  Preview = "Preview" /. {opts};
  
  transform = {"HTML","SolutionsPages.m"};
  layoutfile = ToFileName[{
     DocumentationBuild`Common`$DocumentationBuildDirectory,"XMLTransforms","HTML","Layouts"}, "SolutionsPages.html"];
  popupLayoutfile = ToFileName[{
     DocumentationBuild`Common`$DocumentationBuildDirectory,"XMLTransforms","HTML","Layouts"}, "SolutionsPagesPopup.html"];

  (* keywords *)
  keywordList = DocumentationBuild`Info`GetNotebookKeywords[nbExpr];
  synonymList = DocumentationBuild`Info`GetNotebookSynonyms[nbExpr];
  summaryText = DocumentationBuild`Info`GetNotebookSummary[nbExpr, "Solutions Pages"];
  
  LogIt["  Keywords: "<>ToString[keywordList]];

  titleName = DocumentationBuild`Info`GetNotebookTitle[nbExpr, "Solution"];
  altTitleName = Cases[nbExpr,Cell[a_,"Categorization",CellLabel->"ForcedPageTitle",b___]:>a,Infinity];
  descriptionText = Last[Cases[nbExpr, CellGroupData[{Cell[a_, "WebMetadataSection", b__], c___}, d___]:>c, Infinity]];

  LogIt["  Title name: "<>ToString[titleName]];
  outfile = 
    If[ StringQ[ext],
      StringReplace[outfile, RegularExpression["(?<!\\.\\w\\w)(\\.html)"]->"."<>ext<>"$1"],
      outfile];
  localcss = StringReplace[DirectoryName@outfile, {DirectoryName@DirectoryName@outfile->"", "\""->"", "/"->"", $PathnameSeparator -> ""}] <> "-localcss";

  (* Create preprocess notebook *)
  LogIt["  MakeSolutionsPageNotebook"];
  
  expr = MakeSolutionsPageNotebook[nbExpr, language, opts];

  LogIt["  NotebookPut"];
  nbObj = DocumentationBuild`Common`PubsEvaluateWithFE@NotebookPut[expr];

  LogIt["  Export"];
  tmp = OpenTemporary[];
  res = Export[First@tmp<>".nb", NotebookGet@nbObj, "NB"];
  FrontEnd`NotebookClose[nbObj];
  expr = Get[First@tmp<>".nb"];
  DeleteFile[First@tmp<>".nb"];
  NotebookClose[nbObj];
  
  (* Merge ButtonBoxes *)
  expr = expr /. a_TextData -> ConvertTextData[a];

  nbFile = StringReplace[outfile, ".html"->".nb"];
  CreateDirectory[ DirectoryName@nbFile];

  If[DocumentationBuild`Transform`debug === True,
    Put[ expr, nbFile];
  ];

  (* now call Transmogrify with the appropriate parameters set.
     note that language extension by default goes before the .html *)
  LogIt["  Off to transmogrify"];

  res = Developer`UseFrontEnd @ Transmogrify`Transmogrify[ outfile, expr, transform, 
      Transmogrify`DefaultParameters -> {"layoutfile"->layoutfile, "popupLayoutfile"->popupLayoutfile, "nbFileName"->nbFileName, "IsSolutionsQ"->isSolutionsQ, "IsTMJQ"->isTMJQ, 
      	"titleName"->titleName, "altTitleName"->altTitleName, "Language"->language, "Preview"->Preview,"keywords"->keywordList, "synonyms"->synonymList, 
      	"summary"->summaryText, "description"->descriptionText, "localcss"->localcss, "filename"->outfile}
    ];
    If[Head@res =!= String, First@res, res]

];



 ChangeInsiderTextStyle[text__] := Module[{blah},
   blah = 
    text /. {
      Cell[a_, "WolframInsiderSection", Rule[CellTags,"tmj"], b___]:>
      Cell[a, "WolframInsiderSection", Rule[CellTags,"journal"],b],
      Cell[a__, "WolframInsiderText", b___] :> 
      Cell[a, "WolframInsiderTextRight", b],
      ButtonBox[a__,BaseStyle->"Hyperlink",b__] :> 
      ButtonBox[a,BaseStyle->"NewHyperlink",b]
      
      };
   blah
   ];
   
   
 ChangeFeatureTextGrouping[text__] := Module[{blah},
   blah = 
    text /. {
    Cell[a__, "WolframInsiderImage", b___] :> 
      Cell[a, "WolframInsiderImage", b, CellGroupingRules->{"GroupTogetherNestedGrouping", 16000}, CellTags->"ModifyFeatureImage"], 
    Cell[a__, "WolframInsiderImageTitle", b___] :> 
      Cell[a, "WolframInsiderImageTitle", b, CellGroupingRules->{"GroupTogetherNestedGrouping", 16000}]
      };
   blah
   ];


FindAndReplaceInsiderText[exp__]:=(ReplacePart[exp, (# -> ChangeInsiderTextStyle[Extract[exp, #]]) & /@ 
	Union[Position[exp, Cell[__, "WolframInsiderText"|"WolframInsiderSection", ___]], 
		Position[exp, _ButtonBox]]
	 ]);


ModifyRightColumn[ex__]:=ReplacePart[ex, (# -> FindAndReplaceInsiderText[Extract[ex, #]]) & /@ Position[ex, 
	Cell[CellGroupData[{Cell["Column 2", "WolframInsiderRightSide",a___], b___}, 
    	c___], d___]|Cell[CellGroupData[{Cell["Column 2", "WolframInsiderRightSide", ___], __}, Open]]  ]  ];


    
    
    
(* Create Insider notebook *)
MakeInsiderNotebook[nb_String, language_String:"English", opts___?OptionQ ]:= 
  MakeInsiderNotebook[DocumentationBuild`Utils`GetQuietly[nb], language, opts];
MakeInsiderNotebook[nbExpr_Notebook, language_String:"English", opts___?OptionQ ]:= 
Module[{transform, expr=nbExpr, pageTitle = "None", keywordList = {}, seeAlsoGroup, poss, exampleCaption },
  
    expr = DeleteCases[expr, Cell[___, "History", ___], Infinity];
    expr = DeleteCases[expr, ShowCellTags->True, Infinity];
    expr = DeleteCases[expr, Cell[__, "InternalNoteLink", ___],Infinity];
   	
   	expr = expr /.Cell[TextData[ButtonBox[a_String,
 	BaseStyle->"Hyperlink",
 	ButtonData->{URL[b_String/;
 		StringJoin[
 			Insert[ Riffle[Drop[Drop[StringSplit[b, "/"], 1], -1], "/"],  "http:/" , 1 ]
 		]==="http://reference.wolfram.com/language/ref"], None}, c__]], "SolutionsListing", d__]:>
 	Cell[TextData[StyleBox[ButtonBox[a, BaseStyle->"Hyperlink", ButtonData->{
   	URL[b], None}, c], "InlineFormulaListItem"]], "SolutionsListing", d];


  (* Remove XXX* cells *)
  expr = DeleteCases[expr, Cell[c_String /; StringMatchQ[c, RegularExpression["XX*"]], ___], Infinity];
  expr = DeleteCases[expr, Cell[BoxData[c_String /; StringMatchQ[c, RegularExpression["XX*"]]], ___], Infinity];

 (* Remove XXX* cells *)
  expr = DeleteCases[expr, Cell[c_String /; StringMatchQ[c, RegularExpression["XX*"]], ___], Infinity];
  expr = DeleteCases[expr, 
    Cell[_  sty_String /; StringMatchQ[sty, "DocumentStatusSection" | "DocumentStatus"], ___], Infinity];

  (* Remove empty cell groups *)
  expr = RemoveEmptyCellGroups[expr, {
           "KeywordsSection", "Keywords", "DocumentStatusSection", "DocumentStatus"
           }];
           
  (* Remove SuppressedMaterial section cell group *)
  expr = DeleteCases[expr, Cell[CellGroupData[{Cell[a_, "WolframInsiderSection", 
    "SuppressedMaterial", b___], c___, d___}, e___]], Infinity];
  expr = DeleteCases[expr, Cell[a_, "WolframInsiderSection", 
    "SuppressedMaterial", b___], Infinity];
    


  (* Change paclets to URLs *)
  expr = expr /. ButtonBox[c__, BaseStyle->_, o1___, ButtonData->str_String /; StringMatchQ[str, "paclet*"], o2___] :> 
    ButtonBox[c, BaseStyle->"Hyperlink", ButtonData->{URL["http://reference.wolfram.com/language/"<>StringReplace[str,"paclet:"->""]<>".html"], 
    	None}, BaseStyle->"Hyperlink", o1, o2];
    	
  expr = expr /. Cell[BoxData[FormBox[ButtonBox["\[RightGuillemet]", BaseStyle->"Hyperlink",a__], TraditionalForm]]]:>
  ButtonBox["\[RightGuillemet]",BaseStyle->"Hyperlink",a];
  
  expr = expr /. ButtonBox[
  	ButtonBox[a__, BaseStyle->"Hyperlink",b___],
  		BaseStyle->"Hyperlink",
  		b___]:>
  ButtonBox[a,BaseStyle->"Hyperlink", b];
  
  expr = expr /. ButtonBox[a__, BaseStyle->"Hyperlink",
  		ButtonData->{b_String/;StringTake[b,1]==="#", None}, c___]:>
  ButtonBox[a,BaseStyle->"Hyperlink", ButtonData->b, c];
  
  expr = expr /. ButtonBox[a__, BaseStyle->"Hyperlink",
  		ButtonData->{URL[b_String], None}, c___]:>
  ButtonBox[a,BaseStyle->"Hyperlink", ButtonData->b, c];
  
  expr = expr /. ButtonBox[a_, BaseStyle->"Hyperlink", ButtonData->b_String]/;StringMatchQ[StringTake[b, 1],"#"]:>
  ButtonBox[a, BaseStyle->"InternalPageLink", ButtonData->b];
  
  
  (*Change Hyperlink inside of h2*)
  expr = expr /. Cell[TextData[ButtonBox[a__, BaseStyle->"Hyperlink",
  		b___]], "WolframInsiderHeading2", x___]:>
  Cell[TextData[ButtonBox[a,BaseStyle->"H2Hyperlink",b]], "WolframInsiderHeading2", x];
  expr = expr /. Cell[TextData[ButtonBox[a__, BaseStyle->"Hyperlink",
  		b___]], "WolframInsiderHeading3", x___]:>
  Cell[TextData[ButtonBox[a,BaseStyle->"H3Hyperlink",b]], "WolframInsiderHeading3", x];
  
  
  (* manipulate some button boxen *)
  expr = ReplacePart[expr, (# -> DocumentationBuild`Utils`MakePacletButton[Extract[expr, #]]) & /@ Position[expr, ButtonBox[_String, ___]]];
  
 (* The function below is a series of ReplacePart functions that simply finds all the Cells / Groups of a particular style and 
 rewrites them. This one rewrites the notebook expression, to rewrite the Right column, to rewrite the WolframInsiderText cells.*)
  expr = ModifyRightColumn[expr];
  
  	
    (* Extract and change Image grouping for feature section *)
  expr = ReplacePart[expr, (# -> ChangeFeatureTextGrouping[Extract[expr, #]]) & /@ Position[expr, 
  	Cell[CellGroupData[{Cell[_, "WolframInsiderSection", CellTags->"upcomingevents", ___], __}, Open]]]  ];
  

  (* Change WindowSize to a width for nicer line wrapping in gifs *)
  expr = DocumentationBuild`Utils`Private`ResetOptions[expr, 
           WindowSize -> {700, 700}, ShowCellBracket->False, Magnification->1,(*PageWidth->WindowWidth,*)
           StyleDefinitions -> 
             Notebook[{Cell[ StyleData[ StyleDefinitions -> FrontEnd`FileName[{"Wolfram"}, "WolframInsiderPageStyles.nb"]]](*, 
               Cell[StyleData["Input"], FontSize -> 11, CellID -> 1,PageWidth->525 ], 
               Cell[StyleData["Output"], FontSize -> 11, CellID -> 1], 
               Cell[StyleData["Graphics"], Magnification -> 1, CellID -> 1, PageWidth->525 ], 
               Cell[StyleData["Picture"], Magnification -> 1, CellID -> 1, PageWidth->525 ], 
               Cell[StyleData["InputHiddenMaterial", StyleDefinitions->StyleData["Input"]], CellID -> 1], 
               Cell[StyleData["InputSuppressedMaterial", StyleDefinitions->StyleData["Input"]], CellID -> 1], 
               Cell[StyleData["InputThumbnailMaterial", StyleDefinitions->StyleData["Input"]], CellID -> 1]*)
               }]];

  expr
];







(* 
    ExporInsiderWebPage[outfile, expr, pageType,= opts] 
    this is the main Transmogrify[] wrapper.
    NOTE - outfile = None just returns the raw Transmogrify output
           Test->True causes ExportMarketingWebPage to use TransmogrifyString[]
           the shorthand functions rawmogtest[] and mogtest[], respectively,
           are wrappers for these two functionalities.
*)
ExportInsiderWebPage[_,$Failed,___] = $Failed;

Options[ExportInsiderWebPage] = {
    "DefaultParameters" -> {},
    "IsManipulate" -> True,
    "Language" -> "English",
    "ReleaseBuild" -> False,
    "SolutionsTest" -> False,
    "TMJTest" -> False
    
};


ExportInsiderWebPage[file_String, nb_String, language_String:"English", opts___?OptionQ]:= 
  ExportInsiderWebPage[file, DocumentationBuild`Utils`GetQuietly[nb], language, opts ];
ExportInsiderWebPage[file_String, nbExpr_Notebook, language_String:"English", opts___?OptionQ]:=
Developer`UseFrontEnd @
Module[{ext, outfile=file, transform, keywordList={}, synonymList={}, summaryText="", titleName, res, nbObj, tmp, 
	nbFile, localcss, layoutfile, emaillayoutfile, expr, fileName, isSolutionsQ, isTMJQ, descriptionText,
	pageType, pageTitle, jira, month, year, releaseImagePath, buildTag, pathTail, isReleaseBuild},

	isSolutionsQ = "SolutionsTest" /. {opts} /. Options[ExportInsiderWebPage];
	isTMJQ = "TMJTest" /. {opts} /. Options[ExportInsiderWebPage];
	isReleaseBuild = "ReleaseBuild" /. {opts} /. Options[ExportInsiderWebPage];
    
  LogIt["** Entering ExportSolutionsPage **"];
  (* Add DocumentationBuild\XMLTransforms if it isn't already on path *)
  If[ !MemberQ[ToFileName[{DocumentationBuild`Common`$DocumentationBuildDirectory, "XMLTransforms"}], Transmogrify`$XMLTransformPath], 
    AppendTo[Transmogrify`$XMLTransformPath, ToFileName[{DocumentationBuild`Common`$DocumentationBuildDirectory, "XMLTransforms"}] ];
  ];

  ext = Switch[language, 
  	"Japanese", "ja", 
  	_, "en"];
  
  Preview = "Preview" /. {opts};
  
  transform = {"HTML","Insider.m"};
  layoutfile = ToFileName[{
     DocumentationBuild`Common`$DocumentationBuildDirectory,"XMLTransforms","HTML","Layouts"}, "Insider.html"];
  emaillayoutfile = ToFileName[{
     DocumentationBuild`Common`$DocumentationBuildDirectory,"XMLTransforms","HTML","Layouts"}, "InsiderEmail.html"];

  (* keywords *)
  keywordList = DocumentationBuild`Info`GetNotebookKeywords[nbExpr];
  synonymList = DocumentationBuild`Info`GetNotebookSynonyms[nbExpr];
  summaryText = DocumentationBuild`Info`GetNotebookSummary[nbExpr, "Solutions Pages"];
  
  LogIt["  Keywords: "<>ToString[keywordList]];

  descriptionText = Last[Cases[nbExpr, CellGroupData[{Cell[a_, "WebMetadataSection", b__], c___}, d___]:>c, Infinity]];
  pageType=Cases[nbExpr, CellGroupData[{Cell[a_, "CategorizationSection", b__], c___}, d___]:>c, Infinity][[1,1]];
  pageTitle=Cases[nbExpr, CellGroupData[{Cell[a_, "WebMetadataSection", b__], c___}, d___]:>c, Infinity][[1,1]];
  
  jira = Cases[nbExpr, Cell[a_, "Categorization", CellLabel -> "Jira Issue", b__] :> a, Infinity];
  jira = If[jira==={}, "", jira[[1]]  ];
  month = Cases[nbExpr, Cell[a_, "Categorization", CellLabel -> "Month", b__] :> a, Infinity];
  month = If[month==={}, "", month[[1]]  ];  
  year = Cases[nbExpr, Cell[a_, "Categorization", CellLabel -> "Year", b__] :> a, Infinity];
  year = If[year==={}, "", year[[1]]  ];
  
  buildTag = jira<>year<>month;
  pathTail = year<>"/"<>month<>"/"<>buildTag;
  
  releaseImagePath="http://msgcache.wolfram.com/"<>
  Switch[pageType,
  	"Commercial", "wolfram-insider/"<>pathTail,
  	"Academic", "wolfram-insider/"<>pathTail,
  	"Publishing", "publishing/"<>pathTail,
  	"Salespartner", "wolfram-insider/sales-partner/"<>pathTail,
    _, "wolfram-insider/"<>"NO_PAGE_TYPE"
  ];
		
  
  fileName=Switch[pageType,
  	"Salespartner", "Salespartner",
  	"Publishing", "Publishing",
  	"Academic", "ACAD",
  _, "COMM"];

  LogIt["  Title name: "<>ToString[titleName]];
  outfile = 
    If[ StringQ[ext],
      StringReplace[outfile, RegularExpression["(?<!\\.\\w\\w)(\\.html)"]->"."<>ext<>"$1"],
      outfile];
  localcss = StringReplace[DirectoryName@outfile, {DirectoryName@DirectoryName@outfile->"", "\""->"", "/"->"", $PathnameSeparator -> ""}] <> "-localcss";

  (* Create preprocess notebook *)
  LogIt["  MakeSolutionsPageNotebook"];
  
  expr = MakeInsiderNotebook[nbExpr, language, opts];

  LogIt["  NotebookPut"];
  nbObj = DocumentationBuild`Common`PubsEvaluateWithFE@NotebookPut[expr];

  LogIt["  Export"];
  tmp = OpenTemporary[];
  res = Export[First@tmp<>".nb", NotebookGet@nbObj, "NB"];
  FrontEnd`NotebookClose[nbObj];
  expr = Get[First@tmp<>".nb"];
  DeleteFile[First@tmp<>".nb"];
  NotebookClose[nbObj];
		
  expr = expr /. CellGroupData[{Cell[b__,"WolframInsiderSection",c__], 
		Cell[d__, "WolframInsiderListItem", e___], x___},y___] :>
		CellGroupData[{Cell[b,"WolframInsiderSection",c], 
		Cell[CellGroupData[{Cell[d, "WolframInsiderListItem", e](*, Cell[" ", "WolframInsiderListItem"]*)}]], x},y]; 
	
(*  expr = expr /. Cell[CellGroupData[{Cell["Column 2","WolframInsiderColumn",x1__],
		Cell[x2__,"WolframInsiderHeading4",x3__],
		Cell[CellGroupData[{a___, Cell[b__,"WolframInsiderListItem",c__], 
		Cell[d__, "WolframInsiderListItemDateLocation", e___], x___},y___]]},x4___]]:>
		
  Cell[CellGroupData[{Cell["Column 2","WolframInsiderColumn",x1],
		Cell[x2,"WolframInsiderHeading4",x3],
	Cell[CellGroupData[{a, 
		CellGroupData[{Cell[b,"WolframInsiderListItem25",c], Cell[d, "WolframInsiderListItemDateLocation5", e]}], x},y], "SpecialGroup"]}, x4]];
*)
  expr = expr /. CellGroupData[{a___, Cell[b__,"WolframInsiderListItem",c__], 
		Cell[d__, "WolframInsiderListItemDateLocation", e___], x___},y___]:>
	Cell[CellGroupData[{a, 
		CellGroupData[{Cell[b,"WolframInsiderListItem2",c], Cell[d, "WolframInsiderListItemDateLocation", e]}]
		, x},y], "NewGroup"];
		
  expr = expr //. CellGroupData[{a___, Cell[b__,"WolframInsiderListItem",c__], 
		Cell[d__, "WolframInsiderListItemDateLocation", e___], x___},y___]:>
	CellGroupData[{a, 
		CellGroupData[{Cell[b,"WolframInsiderListItem2",c], Cell[d, "WolframInsiderListItemDateLocation", e]}]
		, x},y];
		
  expr = expr /. CellGroupData[{a___, Cell[b__,"WolframInsiderListItem",c__], 
		Cell[d__, "WolframInsiderListAnswer", e___], x___},y___]:>
	Cell[CellGroupData[{a, 
		CellGroupData[{Cell[b,"WolframInsiderListItem3",c], Cell[d, "WolframInsiderListAnswer", e]}]
		, x},y], "NewGroup"];
		
  expr = expr //. CellGroupData[{a___, Cell[b__,"WolframInsiderListItem",c__], 
		Cell[d__, "WolframInsiderListAnswer", e___], x___},y___]:>
	CellGroupData[{a, 
		CellGroupData[{Cell[b,"WolframInsiderListItem3",c], Cell[d, "WolframInsiderListAnswer", e]}]
		, x},y];
     
  
  (* Merge ButtonBoxes *)
  expr = expr /. a_TextData -> ConvertTextData[a];

  nbFile = StringReplace[outfile, ".html"->".nb"];
  CreateDirectory[ DirectoryName@nbFile];

  If[DocumentationBuild`Transform`debug === True,
    Put[ expr, nbFile];
  ];

  (* now call Transmogrify with the appropriate parameters set.
     note that language extension by default goes before the .html *)
  LogIt["  Off to transmogrify"];

  res = Developer`UseFrontEnd @ Transmogrify`Transmogrify[ outfile, expr, transform, 
      Transmogrify`DefaultParameters -> {"layoutfile"->layoutfile, "emaillayoutfile"->emaillayoutfile, "fileName"->fileName, "IsSolutionsQ"->isSolutionsQ, "IsTMJQ"->isTMJQ, 
        "Language"->language, "Preview"->Preview,"keywords"->keywordList, "synonyms"->synonymList, "summary"->summaryText, 
      	"description"->descriptionText, "pageTitle"->pageTitle, "localcss"->localcss, "filename"->outfile, "type"->pageType, "imagepath"->releaseImagePath, 
      	"releaseBuild"->isReleaseBuild}
    ];
    If[Head@res =!= String, First@res, res]

];








Options[ExportMarketingWebPage] = {
  DefaultParameters->{}, Keywords->{}, LogFile->None, Test->False};

CellLabelInThumbnailQ[expr_]:=
TrueQ[ Length @ Cases[expr, Cell[_, "InputThumbnailMaterial", ___], Infinity] > 0];

inlineFunctionStringToButtonBox[str_String] := inlineFunctionStringToButtonBox[ Cell[str,"SeeAlso"] ];
inlineFunctionStringToButtonBox[c___] := (Message[inlineFunctionStringToButtonBox::arg, c]; $Failed);
inlineFunctionStringToButtonBox[Cell[c_String, sty_, opts___] ] :=
Module[{n, f, fList},
  If[(Head@c === String),
    f = StringReplace[c, {"\n"->"", "\r"->""}];
    Which[
      (* . separator *)
      StringCount[f, RegularExpression["\\s+\\.\\s+"]] > 0, 
      (
        f = StringSplit[f, RegularExpression["\\s+\\.\\s+"]];
        fList = 
          Cell[BoxData[ButtonBox[#, BaseStyle -> "Link"]], "InlineFormula"] & /@ f;
        n = Flatten[Riffle[ fList, {{" ", StyleBox["\[FilledVerySmallSquare]", "InlineSeparator"], " "}}], 1];
        Cell[BoxData[RowBox[n]], sty, opts]
      ),
      (* LongDash separator *)
      StringCount[f, RegularExpression["\\s+\[LongDash]\\s+"]] > 0, 
      (* for now, do nothing *)
        Cell[c, sty, opts],
      (*  *)
      StringCount[f, RegularExpression["\\s+\[EmptyVerySmallSquare]\\s+"]] > 0, 
      (
        f = StringSplit[f, RegularExpression["\\s+\[EmptyVerySmallSquare]\\s+"]];
        fList = 
          Cell[BoxData[ButtonBox[#, BaseStyle -> "Link"]], "InlineFormula"] & /@ f;
        n = Flatten[Riffle[ fList, {{" ", StyleBox["\[FilledVerySmallSquare]", "InlineSeparator"], " "}}], 1];
        Cell[BoxData[RowBox[n]], sty, opts]
      ),
     True, 
       Cell[BoxData[RowBox[{Cell[BoxData[ButtonBox[c, BaseStyle -> "Link"]], "InlineFormula"]}]], sty, opts]
    ]    
  ,
  (* Return original input *)
  Cell[c, sty, opts]
  ]
]
inlineFunctionStringToButtonBox::arg = "String expected as input but value was:`1`";


makeMarketingSeeAlsoButton[ Cell[con_, sty_, opts___] ]:=
Cell[TextData@ButtonBox[con, BaseStyle->"Hyperlink", ButtonData->{URL["/language/11/"<>ToLowerCase[StringReplace[ Transmogrify`ConvertToString[Cell[con, sty]], {"|"->"",","->"","."->"", "&" -> "And"," "->"-","\[EAcute]"->"e","\[CapitalAHat]\[CapitalZHacek]"->"e", "\[Hyphen]"->"-", "\[LongDash]"->"-",":"->"-","\[Dash]"->"-","-"->"-"}]] <> "/"], None}], sty, opts];

makeMarketingMoreAboutButton[ Cell[con_, sty_, opts___] ]:=
Cell[TextData@ButtonBox[con, BaseStyle->"Hyperlink", ButtonData->{URL["http://reference.wolfram.com/language/guide/" <> DocumentationBuild`Utils`MakeCamelCaseString@Cell[con, sty] <> ".html"], None}], sty, opts];

makeMarketingRelatedFunctionsButton[ c_ ]:=
Module[{f, fList},
  If[(Head@c === String),
    f = StringReplace[c, {"\n"->"", "\r"->""}];
    Which[
      (* . separator *)
      StringCount[f, RegularExpression["\\s+\\.\\s+"]] > 0, 
      (
        f = StringSplit[f, RegularExpression["\\s+\\.\\s+"]];
        fList = 
          Cell[BoxData[ButtonBox[#, BaseStyle->"Hyperlink", ButtonData->{URL["http://reference.wolfram.com/language/ref/" <> StringReplace[#, {"\""->""}] <> ".html"], None}]], "InlineFormula"] & /@ f;
        Flatten[Riffle[ fList, {{" ", StyleBox["\[FilledVerySmallSquare]", "InlineSeparator"], " "}}], 1]
      ),
      (* LongDash separator *)
      StringCount[f, RegularExpression["\\s+\[LongDash]\\s+"]] > 0, 
      (* for now, do nothing *)
        c,
     True, 
       ButtonBox[c, BaseStyle->"Hyperlink", ButtonData->{URL["http://reference.wolfram.com/language/ref/" <> c <> ".html"], None}]
    ]    
  ,
  (* Return original input *)
  c
  ]
];

makeMarketingRelatedFunctionsButton[ TextData[c_List] ]:= TextData[Flatten@{makeMarketingRelatedFunctionsButton/@c}];


(* Make marketing thumbnail table *)
MakeMarketingThumbnailTable[expr_]:=
Module[{tds, list },
  (* expr is a list of pairs {thumbail, caption} *)

tds = {XMLElement["td", {"class" -> "thumHolder"}, {#[[1]]}], 
       XMLElement["td", {"class" -> "thumCaption"}, {#[[2]]}]} & /@ expr;

list = Partition[tds, 3, 3, {1, 1}, "xxx"];

(* TODO: fold this line into code above  *)
list = list /. "xxx" -> {XMLElement["td", {}, {" "}], XMLElement["td", {}, {" "}]};

  XMLElement["table", {"class" -> "thumbnailTable", "border"->"0", "cellspacing"->"0", "cellpadding"->"0"},
   Flatten@{
    {XMLElement["tr", {}, Flatten@{#[[1, 1]], #[[2, 1]], #[[3, 1]]}], 
       XMLElement["tr", {}, 
        Flatten@{#[[1, 2]], #[[2, 2]], #[[3, 2]]}]} & /@ list}]
];


MakeBodyCaptionPair[expr_]:=
 Flatten@{expr, Cases[expr, Cell[_, "MarketingSubpageSection", ___], Infinity]};


MakeFirstCategory[{id_, title_, htmlpage_}] := 
 Module[{}, 
  XMLElement[
   "div", {"class" -> "sb_item " <> id}, {XMLElement[
     "a", {"href" -> htmlpage}, {title}], 
    XMLElement[
     "div", {"class" -> "sb_line"}, {XML`RawXML["<!-- -->"]}], 
    XMLElement[
     "div", {"class" -> "sb_separator"}, {XML`RawXML["<!-- -->"]}]}]];

MakeRegularCategory[{id_, title_, htmlpage_}] := 
 Module[{}, 
  XMLElement[
   "div", {"class" -> "sb_item " <> id}, {XMLElement[
     "div", {"class" -> "sb_line"}, {XML`RawXML["<!-- -->"]}], 
    XMLElement["a", {"href" -> htmlpage}, {title}], 
    XMLElement[
     "div", {"class" -> "sb_line"}, {XML`RawXML["<!-- -->"]}], 
    XMLElement[
     "div", {"class" -> "sb_separator"}, {XML`RawXML["<!-- -->"]}]}]];

MakeLastCategory[{id_, title_, htmlpage_}] := 
 Module[{}, 
  XMLElement[
   "div", {"class" -> "sb_item " <> id}, {XMLElement[
     "div", {"class" -> "sb_line"}, {XML`RawXML["<!-- -->"]}], 
    XMLElement["a", {"href" -> htmlpage}, {title}], 
    XMLElement[
     "div", {"class" -> "sb_line"}, {XML`RawXML["<!-- -->"]}]}]];

MakeOneRegularSubCategory[{id_, title_, htmlpage_}] := 
 Module[{}, 
  XMLElement[
   "div", {"class" -> "sb_item " <> id}, {XMLElement[
     "div", {"class" -> "sb_line"}, {XML`RawXML["<!-- -->"]}], 
    XMLElement["a", {"href" -> htmlpage}, {title}], 
    XMLElement[
     "div", {"class" -> "sb_line"}, {XML`RawXML["<!-- -->"]}]}]];

MakeRegularSubCategory[{subcatid_, idtitlehtmlpage_List}] := 
 Module[{}, 
  XMLElement["div", {"class" -> "subcat " <> subcatid}, 
   Join[Delete[
     Delete[Map[
       MakeOneRegularSubCategory[{#[[1]], #[[2]], #[[3]]}] &, 
       idtitlehtmlpage], { -1, -1, -1}], {1, 3, 1}], {XMLElement[
      "div", {"class" -> "sb_subline"}, {XML`RawXML["<!-- -->"]}]}]]];

MakeLastSubCategory[{subcatid_, idtitlehtmlpage_List}] := 
 Module[{}, 
  XMLElement["div", {"class" -> "subcat " <> subcatid}, 
   Delete[Delete[
     Map[MakeOneRegularSubCategory[{#[[1]], #[[2]], #[[3]]}] &, 
      idtitlehtmlpage], { -1, -1, -1}], {1, 3, 1}]]];

MakeWhatsNewSidebar[cattree_List] := 
 Module[{catlength}, catlength = Length[cattree]; 
  XMLElement["div", {"id" -> "sidebar"}, 
   Flatten[{XMLElement[
      "div", {"id" -> "sb_top"}, {XML`RawXML["<!-- -->"]}], 
     XMLElement[
      "div", {"class" -> "sb_item sb_first whatsnewAll"}, {XMLElement[
        "a", {"href" -> 
          "/mathematica/new-in-9/index.html"}, {"All What's New Features"}]}],
     XMLElement[
      "div", {"class" -> "sb_line always_on"}, {XML`RawXML[
        "<!-- -->"]}],
     Switch[catlength, 
      1, {MakeLastCategory[{cattree[[1, 1, 1]], cattree[[1, 1, 2]], 
         cattree[[1, 1, 3]]}], MakeLastSubCategory[cattree[[1, 2]]]}, 
      2, {MakeFirstCategory[{cattree[[1, 1, 1]], cattree[[1, 1, 2]], 
         cattree[[1, 1, 3]]}], 
       MakeRegularSubCategory[cattree[[1, 2]]], 
       MakeLastCategory[{cattree[[2, 1, 1]], cattree[[2, 1, 2]], 
         cattree[[2, 1, 3]]}], 
       MakeLastSubCategory[
        cattree[[2, 2]]]}, _, {MakeFirstCategory[{cattree[[1, 1, 1]], 
         cattree[[1, 1, 2]], cattree[[1, 1, 3]]}], 
       MakeRegularSubCategory[cattree[[1, 2]]], 
       Table[{MakeRegularCategory[{cattree[[i, 1, 1]], 
           cattree[[i, 1, 2]], cattree[[i, 1, 3]]}], 
         MakeRegularSubCategory[cattree[[i, 2]]]}, {i, 2, 
         catlength - 1}], 
       MakeLastCategory[{cattree[[-1, 1, 1]], cattree[[-1, 1, 2]], 
         cattree[[-1, 1, 3]]}], 
       MakeLastSubCategory[cattree[[-1, 2]]]}],
     XMLElement[
      "div", {"id" -> "sb_bot"}, {XML`RawXML["<!-- -->"]}]}]]];


ExportWhatsNewSidebar[_,$Failed,___] = $Failed;

ExportWhatsNewSidebar[outputfile_, nb_String, language_String:"English", marketingpagedir_, opts___?OptionQ]:= 
  ExportWhatsNewSidebar[outputfile, DocumentationBuild`Utils`GetQuietly[nb], language, marketingpagedir, opts ];
ExportWhatsNewSidebar[outputfile_, nbExpr_Notebook, language_String:"English", marketingpagedir_, opts___?OptionQ]:=
Developer`UseFrontEnd @
Module[{ext, transform,  res, res2, marketingpagefolder=marketingpagedir},

  LogIt["** Entering ExportWhatsNewSidebar **"];
  (* Add DocumentationBuild\XMLTransforms if it isn't already on path *)
  If[ !MemberQ[ToFileName[{DocumentationBuild`Common`$DocumentationBuildDirectory, "XMLTransforms"}], Transmogrify`$XMLTransformPath], 
    AppendTo[Transmogrify`$XMLTransformPath, ToFileName[{DocumentationBuild`Common`$DocumentationBuildDirectory, "XMLTransforms"}] ];
  ];

  ext = Switch[language, "Japanese", "ja", "ChineseSimplified", "zh", _, "en"];
  
  transform = {"HTML","MarketingSidebar.m"};

  res = Developer`UseFrontEnd @ Transmogrify`Transmogrify[ nbExpr, transform, 
      Transmogrify`DefaultParameters -> {
      	"marketingpagefolder"-> marketingpagefolder,
      	"Language"->language}
    ];
    res2 = ExportString[MakeWhatsNewSidebar[res], "XML"]<>"\n<div style=\"clear: both\"><!-- --></div>";
    
    Export[outputfile, res2 , "Lines"]

];

Options[ExportWhatsNewSidebar] = {
  DefaultParameters->{}, Keywords->{}, LogFile->None, Test->False};


(* 
    ExportComparisonWebPage[outfile, expr, pageType, opts] 
    this is the main Transmogrify[] wrapper.
    NOTE - outfile = None just returns the raw Transmogrify output
           Test->True causes ExportComparisonWebPage to use TransmogrifyString[]
           the shorthand functions rawmogtest[] and mogtest[], respectively,
           are wrappers for these two functionalities.
*)

ExportComparisonWebPage[_,$Failed,___] = $Failed;

ExportComparisonWebPage[file_String, nb_String, language_String:"English", opts___?OptionQ]:= 
  ExportComparisonWebPage[file, DocumentationBuild`Utils`GetQuietly[nb], language, opts ];
ExportComparisonWebPage[file_String, nbExpr_Notebook, language_String:"English", opts___?OptionQ]:=
Developer`UseFrontEnd @
Module[{ext, properties={}, outfile=file, transform, keywordList={}, titleName, res, nbObj, tmp },

  (* Add DocumentationBuild\XMLTransforms if it isn't already on path *)
  If[ !MemberQ[ToFileName[{DocumentationBuild`Common`$DocumentationBuildDirectory, "XMLTransforms"}], Transmogrify`$XMLTransformPath], 
    AppendTo[Transmogrify`$XMLTransformPath, ToFileName[{DocumentationBuild`Common`$DocumentationBuildDirectory, "XMLTransforms"}] ];
  ];

  ext = Switch[language, "Japanese", "ja", "ChineseSimplified", "zh", _, "en"];
  
  transform = {"HTML","ComparisonPages.m"};

  (* keywords *)
  keywordList = DocumentationBuild`Info`GetNotebookKeywords[nbExpr];
  (*titleName = DocumentationBuild`Info`GetNotebookTitle[nbExpr, "MarketingPage"] <> ": Wolfram Mathematica Comparative Analysis";*)
  titleName = Cases[nbExpr, Cell[a_, "ComparisonTitle", b__]:>a, Infinity ] <> ": Wolfram Mathematica Comparative Analysis";
  
  outfile = 
    If[ StringQ[ext],
      StringReplace[outfile,RegularExpression["(?<!\\.\\w\\w)(\\.html)"]->"."<>ext<>"$1"],
      outfile];

  (* Create preprocess notebook *)
  expr = MakeComparisonPageNotebook[nbExpr, language, opts];

  nbObj = DocumentationBuild`Common`PubsEvaluateWithFE@NotebookPut[expr];
  tmp = OpenTemporary[];
  res=Export[First@tmp<>".nb", NotebookGet@nbObj, "NB"];
  FrontEnd`NotebookClose[nbObj];
  expr = Get[First@tmp<>".nb"];
  DeleteFile[First@tmp<>".nb"];
  NotebookClose[nbObj];
    
  (* Change WindowSize to a width for nicer line wrapping in gifs *)
  expr = DocumentationBuild`Utils`Private`ResetOptions[expr, 
           WindowSize -> {565, 700}, ShowCellBracket->False (*, PageWidth->WindowWidth*)];

  (* now call Transmogrify with the appropriate parameters set.
     note that language extension by default goes before the .html *)
  res = Developer`UseFrontEnd @ Transmogrify`Transmogrify[ outfile, expr, transform, 
      Transmogrify`DefaultParameters -> {"titleName" -> titleName, "keywords"->keywordList, "Language"->language}
    ];
    If[Head@res =!= String, First@res, res]
];

Options[ExportComparisonWebPage] = {
  DefaultParameters->{}, LanguageExtension->"en", Keywords->{}, LogFile->None, Test->False};


(**

**)
ExportCapabilitiesWebPage[file_String, nb_String, language_String:"English", opts___?OptionQ]:= 
  ExportCapabilitiesWebPage[file, DocumentationBuild`Utils`GetQuietly[nb], language, opts ];
ExportCapabilitiesWebPage[file_String, nbExpr_Notebook, language_String:"English", opts___?OptionQ]:=
Developer`UseFrontEnd @
Module[{ext, properties={}, outfile=file, transform, layoutfile, keywordList={}, titleName, res, nbObj, tmp },

  (* Add DocumentationBuild\XMLTransforms if it isn't already on path *)
  If[ !MemberQ[ToFileName[{DocumentationBuild`Common`$DocumentationBuildDirectory, "XMLTransforms"}], Transmogrify`$XMLTransformPath], 
    AppendTo[Transmogrify`$XMLTransformPath, ToFileName[{DocumentationBuild`Common`$DocumentationBuildDirectory, "XMLTransforms"}] ];
  ];

  ext = LanguageExtension /. {opts} /. Options[ExportCapabilitiesWebPage];
  transform = {"HTML","CapabilitiesPages.m"};
  layoutfile = ToFileName[{
     DocumentationBuild`Common`$DocumentationBuildDirectory,"XMLTransforms","HTML","Layouts"}, "MarketingPages.html"];

  (* meta info *)
  keywordList = GetNotebookKeywords[nbExpr];
  synonymList = GetNotebookSynonyms[nbExpr];
  titleName = GetNotebookTitle[nbExpr, "MarketingPage"];

  outfile = 
    If[ StringQ[ext],
      StringReplace[outfile,RegularExpression["(?<!\\.\\w\\w)(\\.html)"]->"."<>ext<>"$1"],
      outfile];

  (* Create preprocess notebook *)
  expr = MakeCapabilityPageNotebook[nbExpr, language, opts];

  nbObj = DocumentationBuild`Common`PubsEvaluateWithFE@NotebookPut[expr];
  tmp = OpenTemporary[];
  res=Export[First@tmp<>".nb", NotebookGet@nbObj, "NB"];
  FrontEnd`NotebookClose[nbObj];
  expr = Get[First@tmp<>".nb"];
  DeleteFile[First@tmp<>".nb"];
  NotebookClose[nbObj];
    
  (* Change WindowSize to a width for nicer line wrapping in gifs *)
  expr = DocumentationBuild`Utils`ResetOptions[expr, 
           WindowSize -> {565, 700}, ShowCellBracket->False (*, PageWidth->WindowWidth*)];

  (* now call Transmogrify with the appropriate parameters set.
     note that language extension by default goes before the .html *)
  res = Developer`UseFrontEnd @ Transmogrify`Transmogrify[ outfile, expr, transform, 
      Transmogrify`DefaultParameters -> {"layoutfile"->layoutfile, "titleName" -> titleName, "keywords"->keywordList, 
	  "synonyms" -> synonymList}
    ];
    If[Head@res =!= String, First@res, res]
];

Options[ExportCapabilitiesWebPage] = {
  DefaultParameters->{}, LanguageExtension->"en", Keywords->{}, LogFile->None, Test->False};



(* Create Comparison page notebook *)
MakeCapabilityPageNotebook[nb_String, language_String:"English", opts___?OptionQ ]:= 
  MakeCapabilityPageNotebook[DocumentationBuild`Utils`GetQuietly[nb], language, opts];
MakeCapabilityPageNotebook[nbExpr_Notebook, language_String:"English", opts___?OptionQ ]:= 
Module[{transform, expr=nbExpr, pageTitle = "None", keywordList = {} },

  (* Change paclets to URLs *)
  expr = expr /. ButtonBox[c__, BaseStyle->_, o1___, ButtonData->str_String /; StringMatchQ[str, "paclet*"], o2___] :> 
    ButtonBox[c, ButtonData->{URL["http://reference.wolfram.com/language/"<>StringReplace[str,"paclet:"->""]<>".html"], None}, BaseStyle->"Hyperlink", o1, o2];

  (* Change '>>' to \[RightGuillemet] *)
  expr = expr /. ButtonBox[">>", r___] :> ButtonBox["\[RightGuillemet]", r];
  (* Remove "FutureExample" | "TentativeExample" tagged cells *)
  expr = DeleteCases[expr, Cell[CellGroupData[{Cell[___, "FutureExample" | "TentativeExample"(* | "ExcludedMaterial"*), ___], ___}, ___], ___], Infinity];
  expr = DeleteCases[expr, Cell[__, "FutureExample" | "TentativeExample" | "ExcludedMaterial", ___], Infinity];
  expr = DeleteCases[expr, Cell[__, CellDingbat->Cell[__, "TentativeID" | "BrightFlag" | "", ___], ___], Infinity];
    
  (* Remove XXX* cells *)
  expr = DeleteCases[expr, Cell[c_String /; StringMatchQ[c, RegularExpression["XX*"]], ___], Infinity];
  expr = DeleteCases[expr, 
    Cell[_  sty_String /; StringMatchQ[sty, "DocumentStatusSection" | "DocumentStatus"], ___], Infinity];
  

  (* Remove empty cell groups *)
  expr = RemoveEmptyCellGroups[expr, {
           "CapabilitiesFeaturesSection", "CapabilitiesMoreAboutSection", "CapabilitiesRelatedDemonstrationsSection", 
           "CapabilitiesRelatedFunctionsSection", "CapabilitiesRelatedLinksSection", 
           "CapabilitiesSeeAlsoSection", "CapabilitiesSlideshowSection", 
           "DocumentStatus", "DocumentStatusSection", "Keywords", "KeywordsSection"
           }];
  (* Change paclets to URLs *)
  expr = expr /. ButtonBox[c__, BaseStyle->_, o1___, ButtonData->str_String /; StringMatchQ[str, "paclet*"], o2___] :> 
    ButtonBox[c, BaseStyle->"Hyperlink", ButtonData->{URL["http://reference.wolfram.com/language/"<>StringReplace[str,"paclet:"->""]<>".html"], None}, BaseStyle->"Hyperlink", o1, o2];

  (* manipulate some button boxen *)
   expr = ReplacePart[expr, (# -> DocumentationBuild`Utils`MakePacletButton[Extract[expr, #]]) & /@ Position[expr, Position[expr, ButtonBox[_String, ___]]]];

  (* Renumber In/Out Cell labels *)
  expr = DocumentationBuild`Export`RenumberCells[expr, {"CapabilitiesFeaturesSection", "CapabilitiesSlideshowSection", "CapabilitiesItem"}];

  expr
];




(*  Comparison front page SSI file listing *)
ExportComparisonTable[file_String, dir_String] :=
Module[{d},
  (* stet as CreateDirectory rather than createDirectory since d is ignored; it doesn't matter here whether d is a dir name (pre-10.4 behavior) or $Failed (10.4 behavior) *)
  d = CreateDirectory[DirectoryName@file];
  Export[file, MakeComparisonTable[dir], "XML", "AttributeQuoting"->"\"", "Entities"->{"\[LongDash]"->"&mdash;", "'"->"'"}]
];
MakeComparisonTable[dir_] := Module[{data, data2divs, files},
  files = Sort[FileBaseName[#]& /@ FileNames["*.nb", dir], cmpfunc];
  data = Map[
    Function[{file}, Module[{nb, base},
      base = FileBaseName[file];
      nb = DocumentationBuild`Utils`GetQuietly[file];

      Transmogrify`ConvertToString /@ {
        If[Count[nb, Cell[_, "FutureFlag", ___], 2] != 0,
          "FutureFlag", ""],
        base,
        First@Cases[nb, Cell[_, "ComparisonTitle", ___], Infinity, 1],
        MakeComparisonListingCell@First@Cases[nb, Cell[_, "ComparisonListing", ___], Infinity, 1]
      }
    ]],
    ToFileName[dir, #<>".nb"]& /@ files ];
  data2divs = {
    XMLElement["div", {"class" -> "comptitle"}, {
      XMLElement["a", {"href" -> ("content/" <> #[[1]] <> ".html")}, { #[[2]] }]
    }],
    XMLElement["div", {"class" -> "complisting"}, {
      XMLElement["a", {"href" -> ("content/" <> #[[1]] <> ".html")}, { #[[3]] }]
    }]
  }&;
  XMLElement["div", {}, MapThread[
    Function[{datum, class},
      XMLElement["div", {"class" -> class}, {
        XMLElement["table", {"class" -> "comptable"}, {
          XMLElement["tr", {}, Map[
            XMLElement["td", {"valign" -> "top", "width" -> "33%"}, Flatten@Map[data2divs, #]]&,
            {Take[datum, #], Take[datum, {#+1, 2*#}], Drop[datum, 2*#]}&[
              Ceiling[Length[datum]/3]]
          ]]
        }]
      }]
    ],
    {{Cases[data, {"", r___} :> {r}], Cases[data, {"FutureFlag", r___} :> {r}]},
     {"", "Future"}}
  ]]
];


(* Sort letters first, then numbers, ... *)
cmpfunc[s1_, s2_] := Which[
  s1 === "", True, s2 === "", False,
  True, Module[{f1, f2},
    {f1, f2} = StringTake[#, 1] & /@ {s1, s2};
    Which[
      f1 === f2, Apply[cmpfunc, StringDrop[#, 1]& /@ {s1, s2}],
      LetterQ[f1] && LetterQ[f2], OrderedQ[{f1, f2}],
      LetterQ[f1], True,
      LetterQ[f2], False,
      DigitQ[f1] && DigitQ[f2], OrderedQ[{f1, f2}],
      DigitQ[f1], True,
      DigitQ[f2], False,
      True, OrderedQ[{f1, f2}]
]]];


(* Create Comparison page notebook *)
MakeComparisonPageNotebook[nb_String, language_String:"English", opts___?OptionQ ]:= 
  MakeComparisonPageNotebook[DocumentationBuild`Utils`GetQuietly[nb], language, opts];
MakeComparisonPageNotebook[nbExpr_Notebook, language_String:"English", opts___?OptionQ ]:= 
Module[{transform, expr=nbExpr, pageTitle = "None", keywordList = {} },

  (* Change paclets to URLs *)
  expr = expr /. ButtonBox[c__, BaseStyle->_, o1___, ButtonData->str_String /; StringMatchQ[str, "paclet*"], o2___] :> 
    ButtonBox[c, 
      ButtonData->{URL["http://reference.wolfram.com/language/"<>StringReplace[str,"paclet:"->""]<>".html"], None}, 
      BaseStyle->"Hyperlink", o1, o2];

  (* Change '>>' to \[RightGuillemet] *)
  expr = expr /. ButtonBox[">>", r___] :> ButtonBox["\[RightGuillemet]", r];
  (* Remove "FutureExample" | "TentativeExample" tagged cells *)
  expr = DeleteCases[expr, Cell[CellGroupData[{Cell[___, "FutureExample" | "TentativeExample"(* | "ExcludedMaterial"*), ___], ___}, ___], ___], Infinity];
  expr = DeleteCases[expr, Cell[__, "FutureExample" | "TentativeExample" | "ExcludedMaterial", ___], Infinity];
  expr = DeleteCases[expr, Cell[__, CellDingbat->Cell[__, "TentativeID" | "BrightFlag", ___], ___], Infinity];
    
  (* Remove XXX* cells *)
  expr = DeleteCases[expr, Cell[c_String /; StringMatchQ[c, RegularExpression["XX*"]], ___], Infinity];
  expr = DeleteCases[expr, 
    Cell[_  sty_String /; StringMatchQ[sty, "DocumentStatusSection" | "DocumentStatus"], ___], Infinity];
  
  expr = expr /. Cell[c_, "ComparisonAbstract", o___] :> Cell[c, "ComparisonText", o];

  expr = expr /. Cell[CellGroupData[{Cell[con___], Cell[c1_, "ComparisonListing", o1___], Cell[c2_, "ComparisonText", o2___], rest___ },g1___],g2___] :>
                  Cell[CellGroupData[{Cell[con], Cell[c1, "ComparisonListing", o1], Cell[c2, "ComparisonTextFirst", o2], rest}, g1], g2];

  expr = expr /. Cell[con_String, "ComparisonSeeAlsoSection", o___] :> Cell["See Also Analyses On:", "ComparisonSeeAlsoSection", o];
  expr = expr /. Cell[con_String, "ComparisonSeeAlso", o___] :> makeComparisonSeeAlsoButton@Cell[con, "ComparisonSeeAlso", o];

  (* Remove empty cell groups *)
  expr = RemoveEmptyCellGroups[expr, {
           "ComparisonAdvantagesSection", (*"ComparisonAdvantages", "ComparisonFeatures", *)
           "ComparisonFeaturesSection", "ComparisonInteroperabilitySection", (*"ComparisonInteroperability", *)
           "ComparisonSeeAlsoSection", "ComparisonSubpageSection", (*"ComparisonListing", "ComparisonSeeAlso", *)
           (*"ComparisonText", "ComparisonTitle",*) "DocumentStatus", "DocumentStatusSection", "Keywords", "KeywordsSection",
           "ComparisonInternalInformationSection",
           "ComparisonRelatedLinksSection"
           }];

  (* Change paclets to URLs *)
  expr = expr /. ButtonBox[c__, BaseStyle->_, o1___, ButtonData->str_String /; StringMatchQ[str, "paclet*"], o2___] :> 
    ButtonBox[c, BaseStyle->"Hyperlink", ButtonData->{URL["http://reference.wolfram.com/language/"<>StringReplace[str,"paclet:"->""]<>".html"], None}, BaseStyle->"Hyperlink", o1, o2];

  (* manipulate some button boxen *)
   expr = ReplacePart[expr, (# -> DocumentationBuild`Utils`MakePacletButton[Extract[expr, #]]) & /@ Position[expr, ButtonBox[_String, ___]]];

  If[ DocumentationBuild`Make`$ComparisonPageStyleSheet =!= Null,
    expr = expr /. Rule[StyleDefinitions, {_, _}] :> DocumentationBuild`Make`$ComparisonPageStyleSheet;
  ];

  expr
];



GetWindowSizeCellTag[expr_]:=
Module[{tags, ws, t},
  tags = Union@Flatten@Cases[expr, Cell[___, CellTags->t_, ___] :>t, Infinity];
  ws = 
    DeleteCases[
      StringCases[tags, 
        RegularExpression[
          "{WindowSize\\s*->\\s*{\\s*(\\d+)\\s*,\\s*(\\d+)\\s*}}"] -> {"$1", "$2"}], {}];
  If[Length@ws > 0, ToExpression@Flatten[{ First@ws }], {}]
];


makeComparisonSeeAlsoButton[ Cell[con_, sty_, opts___] ]:=
Cell[TextData@ButtonBox[con, BaseStyle->"Hyperlink", ButtonData->{URL["/products/mathematica/analysis/content/"<>DocumentationBuild`Utils`MakeCamelCaseString@Cell[con, sty] <> ".html"], None}], sty, opts];


textStyleToItem[l___]:=
Module[{ce=l},
  ce = ce /. Cell[c_, "MarketingText", o___] :> Cell[c, "HighlightItem", o];
  ce
];


hasMarketingGraphicQ[expr_]:=
Module[{lst, num},
  lst = Cases[expr, CellGroupData[{Cell[_, "MarketingContentsSection", ___], ___}, ___], Infinity];
  num = Cases[lst, 
          Cell[c_ /; Length @ 
            Cases[c, StyleBox[s_String /; StringMatchQ[s, "*Placeholder*"], ___], Infinity] > 0, "Output", ___], Infinity];
  If[ Length @ num > 0, False, True] 
];


HiddenInputQ[___]:= False;
HiddenInputQ[list_List]:= Apply[Or, Not /@ StringFreeQ[list, "HiddenMaterial"]];

HideCellLabelQ[___]:= False;
HideCellLabelQ[list_List] := 
 TrueQ[
    Or @@ StringMatchQ[list, "*HiddenMaterial"] || 
      !(
      Or @@ StringMatchQ[list, "Input" | "InputThumbnailMaterial" ] && 
      Or @@ StringMatchQ[list, "Output" | "OutputThumbnailMaterial" | "Graphics" | "Picture"]
      )
    ];

(* Replacements *)
characterReplace[str_String, search_String, rep_] :=
  Flatten@Riffle[StringSplit[str, search], rep]

characterReplace[Cell[TextData[lst_List], r___], search_String, rep_] :=
 Cell[TextData[Flatten[
    If[Head@# === String,
       characterReplace[#, search, rep],
       #
       ] & /@ lst]], r
  ]

characterReplace[Cell[str_String, r___], search_String, rep_] :=
  characterReplace[str, search, rep]



MakeComparisonListingCell[Cell[c_, "ComparisonListing", ___]]:= MakeComparisonListingCell @ c;
MakeComparisonListingCell[TextData[lst_List]]:= MakeComparisonListingCell @ lst;
MakeComparisonListingCell[TextData[con_]]:= MakeComparisonListingCell[{con}];
MakeComparisonListingCell[str_]:=
Module[{elems, list},
  (* split list on "," *)
  elems = Transmogrify`ConvertToString @ str;
  elems = StringSplit[ elems, RegularExpression[",\\s*"]];
  elems = DeleteCases[elems, "" | "(" | ")" | "," | ", " | "...)" | ", ...)", Infinity];
  elems = If[Head@#===String, FixedPoint[StringReplace[#, {"(" | ")" | "..." -> ""}]&, #], #]&/@elems;
  elems = DeleteCases[elems, "" | "(" | ")" | "," | ", " | "...)" | ", ...)", Infinity];
(*  list = If[(StringLength@StringJoin@@Take[elems, 3] < 35, Take[elems, 2], Take[elems, 3], elems];*)
  Cell[TextData[ 
    Flatten@Join[{"("}, Riffle[Join[If[Length@elems > 2, Take[elems, 3], elems], {"... )"}], ", "]]], "ComparisonListing"]
];


MakeComparisonItemCell[Cell[con_, sty_String, opts___]]:= 
  Cell[TextData[Flatten@{ MakeComparisonItemCell@con }], sty, opts];
MakeComparisonItemCell[TextData[lst_List]]:= MakeComparisonItemCell @ lst;
MakeComparisonItemCell[TextData[con_]]:= MakeComparisonItemCell[{con}];
MakeComparisonItemCell[con_]:=
Module[{elems, str},
  str = Transmogrify`ConvertToString @ con;
  If[ StringLength@str > 86,
    Print["Too long: "<>ToString@StringLength@str],
    Print["Just right: "<>ToString@StringLength@str]
  ];
  con
];



(* *** *)


End[]
EndPackage[]