(* :Title: Make.m *)

(* :Authors:
	Shaun McCance
	shaunm@wolfram.com
*)

(* :Package Version: 0.50 *)

(* :Mathematica Version: 6.0 *)
                     
BeginPackage["DocumentationBuild`KeywordsIndex`", {
    "DocumentationBuild`Info`",
    "DocumentationBuild`Utils`"
}];


Options[ExportKeywordsIndex] = {
  "ManualGrouping" -> False
};

$Version6ManualGrouping = {
  "!", "\\0", "3.3.1",
  "a", "add", "alg", "alt", "app", "arc", "arr", "ast", "axe",
  "b", "ber", "bio", "boo", "\\bumpeq",
  "c", "cas", "cel", "chain", "cho", "clk", "col", "com", "con", "coo", "cre", "cur",
  "d", "deb", "del", "dev", "dig", "dis", "div", "dou", "dur",
  "e", "ell", "ens", "euc", "exc", "exp", "ext",
  "f", "fil", "fin", "fla", "for", "fra", "ful",
  "g", "gen", "gis", "gra", "gre", "h", "hel", "hol", "hyp",
  "i", "ima", "incenter", "inline", "int", "\\in (TeX)", "inv", "ioc",
  "j", "k", "l", "lea", "lev", "lin", "lip", "loc", "lon",
  "m", "mas", "mat", "mau", "mer", "mkt", "mon", "multiple",
  "\\n", "neu", "nor", "not", "num", "nun", "o", "ope", "ord", "ove",
  "p", "par", "pas", "perceivable", "pie", "plu", "pol", "pop", "ppc", "pri", "pro", "pur",
  "q", "r", "ran", "raw", "rec", "reg", "rep", "rev", "rnd", "rou",
  "s", "sco", "sed", "ser", "she", "\\simeq", "sli", "sou", "sph",
  "sqr", "sta", "sti", "str", "stu", "sum", "sym", 
  "t", "tcp", "tex", "thr", "tog", "tra", "tre", "tur",
  "u", "uns", "v", "vie", "w", "win", "x", "y", "z", "$"
};


ExportKeywordsIndex::usage = "ExportKeywordsIndex[dir, file] calls MakeKeywordsIndex[dir] and exports the HTML pages.";
MakeKeywordsIndex::usage = "MakeKeywordsIndex[file] creates a list of keyword index HTML pages as symbolic XML expressions.";

Begin["`Private`"];

ExportKeywordsIndex[out_String, opts___?OptionQ] :=
	ExportKeywordsIndex[out, 
		ToFileName[{DocumentationBuild`Common`$DocumentationBuildDirectory, "Internal", "data", "keywords"}], "English", opts];

ExportKeywordsIndex[out_, in_, language_String:"English", opts___?OptionQ] :=
Module[{files, pages, data, demos},
  CreateDirectory[out];
  files = FileNames["*.m", in, 2];
  files = DeleteCases[files,
    s_ /; StringMatchQ[s,
      "*" <> $PathnameSeparator <> "bin" <> $PathnameSeparator <> "*"]];
  (* get demonstration keywords *)
  demos = GetDemonstrationKeywordList[];
  demos = If[demos === $Failed, {}, demos];
  data = {};
  Scan[(data = Join[data, demos,
      ReadList[StringToStream[Import[#, "Text", CharacterEncoding->"UTF-8"]]] ])&,
    files];
  (* remove entries without keywords *)
  data = DeleteCases[data, {"", _}];
  data = DeleteCases[data, {"\[EAcute]l\[EAcute]ments propres", _}];
  data = DeleteCases[data,
    {_, uri_ /;
      (HasSourceInformation[uri, language] &&
        MemberQ[{"ExcisedFlag", "FutureFlag", "InternalFlag", "TemporaryFlag"},
          GetSourceInformation[uri, language, "Flag"]])}];

(* save data for testing *)
  Export[ToFileName[{out}, "data.m"], data];
  (* create index pages *)
  pages = MakeKeywordsIndex[data, language, opts];
  Scan[
    Function[{page}, Module[{str, meta, html, outfile},
      meta = page[[1]];
      html = Last[page];
      outfile = ToFileName[{out}, First[meta]];
      If[Head[html] === XMLElement,
        str = ExportString[html, "XML"],
        str = ""; Scan[(str = str <> ExportString[#, "XML"])&, html] ];
      Export[outfile, str, "Text"];
      Export[outfile<>"-head", XMLElement["title", {}, {"Alphabetical Site Index: "<>ToString[ StringReplace[First[meta],".html"->""] ] }], "XML"];
    ]],
    pages];
];


groupKeywordSets[data_, func_, n_] := 
  Module[{Lind, splitdata, datalens, ind = 1, count = 0}, 
    splitdata = Split[data, func[#1[[1]], #2[[1]]]&];
    datalens = Total[#[[All, 2]]]& /@ splitdata;
    Lind = If[count + # > n, (count = #; ++ind), (count += #; ind)] & /@ datalens;
    Flatten[#, 1] & /@ 
      Split[Transpose@{Lind, splitdata}, #[[1]] == #2[[1]] &][[All, All, 2]]]

dropBS[s_String] :=
  If[(StringTake[s, 1] === "\\") && (StringLength[s] > 1), StringDrop[s, 1], s];

comparei[x_, y_, i_] := 
 ToLowerCase@StringTake[dropBS@x, Min[i, StringLength[dropBS@x]]] === 
  ToLowerCase@StringTake[dropBS@y, Min[i, StringLength[dropBS@y]]]

createSubListsOfNLength[data_List, n_Integer]:=
  groupKeywordSets[data, comparei[#1, #2, 3]&, n];

createManualSubLists[data_List, manual_] :=
  Module[{splitdata},
    splitdata = Split[data,
      Function[{a, b}, Module[{pa=1, pb=1},
        While[pa <= Length[manual],
          If[a[[1]] =!= manual[[pa]] && keywordOrderedQ[a[[1]], manual[[pa]]],
            pa--; Break[]];
          pa++];
        While[pb <= Length[manual],
          If[b[[1]] =!= manual[[pb]] && keywordOrderedQ[b[[1]], manual[[pb]]],
            pb--; Break[]];
          pb++];
        pa == pb
      ]]
    ];
    splitdata
  ];

italicizeString[s_String] := Fold[
  Function[{str, word},
    Flatten@Map[
      If[!StringQ[#], #,
        Apply[List, StringReplace[#,
          RegularExpression["\\b("<>word<>")\\b"] :> XMLElement["i", {}, {"$1"}] ]] ]&,
      str]],
  {s}, {"Mathematica", "MathLink"}];

keywordOrderedQ[aa_, bb_] :=
  Module[{a, b},
    {a, b} = Map[
      If[(StringTake[#, 1] === "\\") && (StringLength[#] > 1),
        StringTake[#, {2}], StringTake[#, 1]]&,
      {aa, bb}];
    Which[
      a == "$" && b == "$", OrderedQ[{aa, bb}],
      a == "$", False,
      b == "$", True,
      LetterQ[a] && LetterQ[b], OrderedQ[{aa, bb}],
      LetterQ[a], False,
      LetterQ[b], True,
      DigitQ[a] && DigitQ[b], OrderedQ[{aa, bb}],
      DigitQ[a], False,
      DigitQ[b], True,
      True, OrderedQ[{aa, bb}]
  ]];

MakeKeywordsIndex[data_, language_String:"English", opts___?OptionQ] :=
Module[{terms, refs, datas, file2dd, htmlwrap, tochar, manual},
  refs[_] := {};
  Scan[
    If[!MatchQ[refs[First[#]], {___, Last[#], ___}],
      refs[First[#]] = Append[refs[First[#]], Last[#]]
    ]&, data];
  terms = Sort[First /@ data, keywordOrderedQ];
  terms = First /@ Split[terms];
  datas = Split[Map[{#, Length[refs[#]]}&, terms],
    Module[{a, b},
      {a, b} = Map[
        If[(StringTake[First[#], 1] === "\\") && (StringLength[First[#]] > 1),
          StringTake[First[#], {2}],
          StringTake[First[#], 1]] &,
        {#1, #2}];
      Which[
        a == "$" || b == "$", a == "$" && b == "$",
        LetterQ[a] || LetterQ[b], ToLowerCase[a] == ToLowerCase[b],
        DigitQ[a] || DigitQ[b], DigitQ[a] && DigitQ[b],
        True, True
    ]]&];
 manual = "ManualGrouping" /. {opts} /. Options[ExportKeywordsIndex];
  Switch[manual,
    False,
      datas = Map[createSubListsOfNLength[#, 400]&, datas],
    True,
      datas = Map[createManualSubLists[#, $Version6ManualGrouping]&, datas],
    _,
      datas = Map[createManualSubLists[#, manual]&, datas]
  ];
  datas = Join@@MapIndexed[
    Function[{datum, idx}, Module[{up, file, group, start, end, label},
      up = Function[{s}, Module[{chars},
        chars = Characters[s];
        StringJoin[ToUpperCase[First[chars]], ToLowerCase /@ Rest[chars]]
      ]];
      {start, end} = Map[
        up[StringTake[#, {1, Min[3, StringLength[#]]}]]&,
        (First /@ {First[datum], Last[datum]}) ];
      If[Last[idx] === 1, start = StringTake[dropBS@start, 1]];
      group = StringTake[dropBS@start, 1];
      group = Which[
        LetterQ[group] || group === "$", ToUpperCase[group],
        DigitQ[group], "0",
        True, "#"];
      file = Which[
        group === "#",
          "char" <> ToString[First[ToCharacterCode[dropBS@start]]],
        True,
          dropBS@start
      ];
      file = StringReplace[file, "/" -> ""];
      file = file<>".html";
      label = start <> "-" <> end;
      {file, group, start, end, label} -> Map[{First[#], refs[First[#]]}&, datum]
    ]],
    datas, {2}];
  file2dd[paclet_String, dt_String] :=
    Module[{spl, uri, title},
      spl = StringSplit[paclet, RegularExpression["[:/]"]];
      uri = If[First[spl] === "http",
        StringReplace[paclet, ".devel." -> "."],
        Riffle[Join[{"/mathematica"}, Rest[spl]], "/"] <> ".html"
      ];
      title = If[spl[[2]] === "ref", Last[spl],
        "Title" /. (paclet /. Flatten[DocumentationBuild`Info`$DemonstrationKeywordTitles] 
            /. If[HasSourceInformation[paclet, language],
                {paclet -> GetSourceInformation[paclet, language]}, {}]
            /. {paclet -> {"Title" -> Last[spl]}} )];
      title = If[Head[title] =!= String,
        title //.
          {TextData[s_] :> s,
           StyleBox[s_, FontSlant->"Italic"] :> XMLElement["i", {}, {s}]},
        title];
      If[StringQ[title], title = italicizeString[title]];
      If[StringQ[title], title = {title}];
      XMLElement["dd", {}, {
        XMLElement["a", {"href" -> uri, "title" -> dt}, title], " ",
        XMLElement["span", {"class"->"keywordlabel"}, Flatten@{
        Switch[spl[[-2]],
          "program", "(External Program)",
          "format", {"(", XMLElement["i",{},{"Mathematica"}], " Format)"},
          "character", {"(", XMLElement["i",{},{"Mathematica"}], " Character)"},
          "menuitem", {"(", XMLElement["i",{},{"Mathematica"}], " Menu Item)"},
          "ref", {"(", XMLElement["i",{},{"Mathematica"}], " Symbol)"},
          "guide", {"(", XMLElement["i",{},{"Mathematica"}], " Guide)"},
          "tutorial", {"(", XMLElement["i",{},{"Mathematica"}], " Tutorial)"},
          (* other *)
          _, Which[
            	StringMatchQ[uri, "*newin6*"], 
            		{"(New in ", XMLElement["i",{},{"Mathematica"}], " 6)"},
            	StringMatchQ[uri, "*demonstrations*"], 
            		{"(", XMLElement["a", {"href"->"http://demonstrations.wolfram.com"}, {"Wolfram Demonstrations Project"}],")"}, 
            	True, "(Product Analysis)"]
            ]
        }]
      }]
    ];
  htmlwrap[expr_, meta_] := Module[{metas},
    metas = First /@ datas;
    {
      XMLElement["div", {"class"->"keywordTitleWrapper"}, Flatten@{
        XMLElement["div", {"class"->"keywordTitle"}, { "Alphabetical Site Index"}],
        XMLElement["div", {"class"->"searchWrapper"}, { 
          XML`RawXML["<!--#include virtual=\"/mathematicaIncludes/searchallsites.html\"-->"]
        }],
        XMLElement["table", {"class"->"keywordHolder"}, {
          XMLElement["tr", {"class"->"keywordLetters"},
            Map[
              Function[{tmeta},
                XMLElement["td", {}, {
                  If[tmeta[[2]] == meta[[2]], tmeta[[2]],
                    XMLElement["a", {"href" -> tmeta[[1]]}, {tmeta[[2]]}] ]
              }] ],
              First /@ Split[metas, (#1[[2]] === #2[[2]])&]
          ]],
          XMLElement["tr", {"class"->"keywordArrow"},
            Map[
              Function[{tmeta},
                XMLElement["td", {}, {
                  If[tmeta[[2]] == meta[[2]],
                    XMLElement["img", {"src" -> "http://www.wolframscience.com/nksonline/index/images/arrow-down-grey.gif",
                      "border"->"0", "height"->"6", "width"->"7", "vspace"->"0", "hspace"->"0", "alt"->""}, {}],
                    XML`RawXML["&nbsp;"]]
              }] ],
              First /@ Split[metas, (#1[[2]] === #2[[2]])&]
          ]]
        }],
        If[Count[metas, {_, meta[[2]], ___}] > 1,
          XMLElement["table", {"class"->"keywordSubholder"}, {
            XMLElement["tr", {"class"->"keywordLetters"},
              Map[
                Function[{tmeta},
                  XMLElement["td", {}, {
                    If[tmeta[[1]] == meta[[1]], XMLElement["nobr", {}, { tmeta[[5]] }],
                      XMLElement["a", {"href" -> tmeta[[1]]}, { XMLElement["nobr", {}, { tmeta[[5]] }] }] ]
                }] ],
                Cases[metas, {_, meta[[2]], ___}]
            ]],
            XMLElement["tr",  {"class"->"keywordArrow"},
              Map[
                Function[{tmeta},
                  XMLElement["td", {}, {
                    If[tmeta[[1]] == meta[[1]],
                      XMLElement["img", {"src" -> "http://www.wolframscience.com/nksonline/index/images/arrow-up-red.gif",
                        "border"->"0", "height"->"6", "width"->"7", "vspace"->"0", "hspace"->"0", "alt"->""}, {}],
                      XML`RawXML["&nbsp;"]]
                }] ],
                Cases[metas, {_, meta[[2]], ___}]
            ]]
        }], {} ]
      }],
      XMLElement["div", {"class"->"keywordIndexLetter"}, {
        If[Count[metas, {_, meta[[2]], ___}] > 1, meta[[5]], meta[[2]] ]
      }],
      XMLElement["table", {"class"->"keywordTable", "width"->"100%"}, {XMLElement["tr", {}, expr]}]
    }
  ];
  Map[
    Function[{datum}, Module[{stuff, meta, strs, elms, elmstrs, html},
      meta = First[datum];
      stuff = Last[datum];
      strs = Count[stuff, _String, Infinity];
      elms = elmstrs = 0;
      While[elmstrs < (strs / 2),
        elms++;
        elmstrs += Count[stuff[[elms]], _String, Infinity]];
      html = Map[
        XMLElement["td", {"valign" -> "top", "width" -> "50%"}, {XMLElement["dl", {}, Flatten@Map[
          Function[{one},
            {XMLElement["dt", {}, {First[one]}],
             file2dd[
               If[StringMatchQ[#, RegularExpression["^[A-Za-z0-9_]+$"]],
                 "paclet:ref/"<>#, #],
               First[one]]& /@ Last[one]
            }],
          #]]}]&,
          {Take[stuff, elms], Drop[stuff, elms]}];
      html = htmlwrap[html, meta];
      meta -> html
    ]],
    datas]
];

End[];
EndPackage[];
