(*
    This file contains code for the task of processing the specialized author notebook for
    single-function paclets into a standard paclet and auxiliary files. Paaclet building
    includes creation of a PacletInfo.m file from metadata in the notebook, pulling the
    implementation section out into a .m file, processing the docs into the standard in-product
    format, and packing everything into a .paclet file. There is also functionality for
    building a marketing page for the paclet and building HTML docs.

    This code is called from a WRI server application when authors upload function paclet
    notebooks. It can also be called on user machines to perform the processing steps locally.
*)


BeginPackage["DocumentationBuild`FunctionPaclets`",
    {"DocumentationBuild`Common`", "DocumentationBuild`SystemResourcesFromDocs`",
     "PacletManager`", "JLink`", "Transmogrify`"}]

BuildPacletFromTemplate
BuildPacletFromZip
BuildPacletWebDocs
ExportPacletMarketingPage::usage = "ExportPacletMarketingPage[file, notebook] creates Function Paclet marketing file based on source notebook.";

GetPacletThumbnail::usage = "GetPacletThumbnail[nb] returns the thumbnail graphic cell from nb, or returns None if a thumbnail is not found,";
GetPacletElementList
ValidatePacletElements


Begin["`Private`"]


(***************************************  BuildPaclet  *****************************************)

(*
    Builds a paclet layout in a temporary directory, then pack it into a .paclet file in the
    supplied output directory. The temp dir is then deleted.

    Returns: The path to the .paclet file, or {errorCode_Integer, errorText_String} on an error.
*)

BuildPacletFromTemplate[outputDir_String, buildNumber_, rules_List] :=
    Module[{tempDir, creator, creatorURL, name, version, context, description, symbolNames, docs, mVersion, otherRequirements,
              code, loading, encode, supportContact, docResult, piFile, docExtension,
                appExtension, codeFilePath, codeFileStream, pacletFile, publicSymbols, mainPage,
                  resourcesString, guideTitles, symbolPages, symbolPagePaths, guidePages, guidePagePaths},

        (* Validation has already occurred so that we can be confident that these values are present and valid. *)
        {creator, creatorURL, name, version, context, description, symbolNames, docs, otherRequirements,
         mVersion, code, loading, encode, supportContact} =
              {"Author", "AuthorURL", "PacletName", "PacletVersion", "PacletContext", "ShortDescription", "FunctionNames",
               "Documentation", "OtherRequirements", "MathematicaVersion", "Implementation", "Loading", "Encode", "SupportContact"} /. rules;

        (* Extract the symbols for which a usage message exists. These are the symbols that will cause
           autoloading of the paclet. This should be a list of length 1. Usage messages appear at level 4 if
           code expressions are separated by unnecessary semicolons; level 3 if not.
        *)
        publicSymbols = Cases["Implementation" /. rules, HoldPattern[sym_::"usage"] :> ToString[HoldForm[sym]], {3,4}];

        (* Build the paclet structure into a temp dir, which is deleted after the paclet is packed. *)
        tempDir = CreateDirectory[];

        (*******************  Write the PacletInfo.m file  ********************)

        piFile = OpenWrite[ToFileName[tempDir, "PacletInfo.m"], CharacterEncoding->"UTF8", PageWidth->Infinity];
        WriteString[piFile, "Paclet[\n"];
        WriteString[piFile, "\tName->\"" <> name <> "\",\n"];
        WriteString[piFile, "\tVersion->\"" <> version <> "\",\n"];
        WriteString[piFile, "\tBuildNumber->" <> buildNumber <> ",\n"];
        WriteString[piFile, "\tCreator->\"" <> creator <> "\",\n"];
        If[creatorURL != "N/A",
            WriteString[piFile, "\tURL->\"" <> creatorURL <> "\",\n"]
        ];
        If[supportContact != "N/A",
            WriteString[piFile, "\tSupport->\"" <> supportContact <> "\",\n"]
        ];
        WriteString[piFile, "\tDescription->"];
        WriteString[piFile, InputForm[description]];  (* Use InputForm to ensure that " chars in the text get quoted. *)
        WriteString[piFile, ",\n"];
        (* Note that M version requirements are ignored. Leave it like that until we get a cleaner system
           for ensuring people don't put in incorrect requirements (like a "maximum M version").
        *)
        WriteString[piFile, "\tMathematicaVersion->\"7+\",\n"];
        WriteString[piFile, "\tRoot->\"" <> name <> "\",\n"];
        WriteString[piFile, "\tThumbnail->\"General/Thumbnail.gif\",\n"];
        WriteString[piFile, "\tPublisher->\"Wolfram Research, Inc.\",\n"];
        WriteString[piFile, "\tLoading->\"" <> loading <> "\",\n"];
        (* Now do doc info. First, determine MainPage. Should be the first Guide page if one is present,
           otherwise first function page,
        *)
        guideTitles = fixGuideTitle /@ Cases[docs, Cell[guideTitle_, "GuideTitle", ___] :> guideTitle, Infinity];
        mainPage = If[Length[guideTitles] > 0, "Guides/" <> First[guideTitles], "ReferencePages/Symbols/" <> First[symbolNames]];
        resourcesString = Join[
                            ("\"ReferencePages/Symbols/" <> # <> "\"")& /@ symbolNames,
                            ("\"Guides/" <> # <> "\"")& /@ guideTitles
                          ];
        resourcesString = StringJoin[Riffle[resourcesString, ", "]];
        docExtension =
                "{\"Documentation\", LinkBase->\"" <> name <>
                "\", Language->\"English\", MainPage->\"" <> mainPage <>
                "\", Resources->{" <> resourcesString <> "}}";
        (* TODO: Modern name for this extension is "Kernel", which is not 6.0.2-or-lower-compatible. Change
           when paclets.wolfram.com build server is running 7.0.
        *)
        appExtension = "{\"Application\", Context->\"" <> context <> "\"" <>
                            If[MatchQ[publicSymbols, {__String}], ", Symbols->" <> ToString[publicSymbols, InputForm], ""] <> "}";
        WriteString[piFile, "\tExtensions->{\n"];
        WriteString[piFile, "\t\t" <> appExtension <> ",\n"];
        WriteString[piFile, "\t\t" <> docExtension <> "\n"];
        WriteString[piFile, "\t}\n"];  (* Close extensions *)

        WriteString[piFile, "]"];
        Close[piFile];

        (********************  Build the paclet's dir structure  *********************)

        CreateDirectory[ToFileName[tempDir, name]];
        CreateDirectory[ToFileName[{tempDir, name}, "Documentation"]];
        CreateDirectory[ToFileName[{tempDir, name, "Documentation"}, "English"]];
        If[Length[guideTitles] > 0,
            CreateDirectory[ToFileName[{tempDir, name, "Documentation", "English"}, "Guides"]]
        ];
        CreateDirectory[ToFileName[{tempDir, name, "Documentation", "English"}, "ReferencePages"]];
        CreateDirectory[ToFileName[{tempDir, name, "Documentation", "English", "ReferencePages"}, "Symbols"]];
        CreateDirectory[ToFileName[{tempDir, name}, "Kernel"]];
        CreateDirectory[ToFileName[{tempDir, name, "Kernel"}, "TextResources"]];
        CreateDirectory[ToFileName[{tempDir, name}, "General"]];

        (********************  Write the paclet's .m code file  *********************)

        (* code comes in as a HoldComplete list of exprs. *)
        codeFilePath = ToFileName[{tempDir, name, "Kernel"}, name <> ".m"];
        codeFileStream = OpenWrite[codeFilePath];
        ReleaseHold[Function[{e}, Write[codeFileStream, Unevaluated[e]]; WriteString[codeFileStream, "\n"], HoldFirst] /@
                       ReleaseHold[HoldForm @@@ code]
        ];
        Close[codeFileStream];
        If[TrueQ[encode],
            Encode[codeFilePath, codeFilePath <> "_encoded.m"];
            DeleteFile[codeFilePath];
            RenameFile[codeFilePath <> "_encoded.m", codeFilePath]
        ];

        (***********************  Convert the documentation  ************************)

        (* Doc-conversion code pulls some important info from Categorization cells in the notebook. These
           are not in the authoring notebook because we want to programmatically add them instead of asking
           authors to do it. That is what addDocCategorization[] does.
        *)
        symbolPages = Select[docs, !FreeQ[#, Cell[_, "ObjectName"|"ObjectNameSmall", ___]]&];
        symbolPages = MapThread[addDocCategorization[#1, "Symbol", name, context, name <> "/ref/" <> #2]&, {symbolPages, symbolNames}];
        symbolPagePaths = ToFileName[{tempDir, name, "Documentation", "English", "ReferencePages", "Symbols"}, # <> ".nb" ]& /@ symbolNames;
        docResult = MapThread[buildPacletDocumentation, {symbolPagePaths, symbolPages}];
        If[!MatchQ[docResult, {__String}],
            DeleteDirectory[tempDir, DeleteContents->True];
            Return[{100, "Failure building symbol pages for paclet documentation"}]
        ];
        guidePages = Select[docs, !FreeQ[#, Cell[_, "GuideTitle", ___]]&];
        guidePages = MapThread[addDocCategorization[#1, "Guide", name, context, name <> "/guide/" <> #2]&, {guidePages, guideTitles}];
        guidePagePaths = ToFileName[{tempDir, name, "Documentation", "English", "Guides"}, # <> ".nb" ]& /@ guideTitles;
        docResult = MapThread[buildPacletDocumentation, {guidePagePaths, guidePages}];
        If[Length[guidePages] > 1 && !MatchQ[docResult, {__String}],
            DeleteDirectory[tempDir, DeleteContents->True];
            Return[{101, "Failure building guide pages for paclet documentation"}]
        ];
        (* TODO: Figure out how indexer reports errors and report them here. *)
        indexNotebooks[Join[symbolPagePaths, guidePagePaths], ToFileName[{tempDir, name, "Documentation", "English"}], "English"];

        (**********************  Create the FunctionInformation.m file  *******************************)

        (* TODO: This will return $Failed if there is a problem extracting the syntax templates from
           the notebook. Check the result and report the error in some way. Probably not fatal.
        *)
        CreateFunctionTemplatesFile[context, Thread[{symbolNames, symbolPages}],
                ToFileName[{tempDir, name, "Kernel", "TextResources"}, "FunctionInformation.m"]];

        (**********************  Create the Thumbnail file  *******************************)

        Export[ToFileName[{tempDir, name, "General"}, "Thumbnail.gif"], "Thumbnail" /. rules];

        (*******************  Create the .paclet file in the output dir ********************)

        pacletFile = PackPaclet[tempDir, outputDir];
        DeleteDirectory[tempDir, DeleteContents->True];
        If[StringQ[pacletFile],
            pacletFile,
        (* else *)
            {200, "Failure packing the paclet into a .paclet file (paclet probably has invalid structure or data)"}
        ]
    ]


(* Builds a .paclet file from a zip-style submission instead of a template notebook.
   The unzipDir has the unpacked contents of the zip file that was submitted. It will be deleted by the caller.
   The outputDir is where the .paclet file will be placed.
*)
BuildPacletFromZip[unzipDir_, outputDir_] :=
  JavaBlock[
    Module[{rules, piFile, pm, paclets, pacletObj, pacletName, pacletFile, pacletDocsDir, submissionExtrasDir,
             documentationSourceDir, documentationSourceDirLength, fullDocSourcePaths, destPath, docResult},

        submissionExtrasDir = ToFileName[unzipDir, "SubmissionExtras"];
        piFile = First[FileNames["PacletInfo.m", unzipDir, 2]];
        pm = PacletManager`Package`getPacletManager[];
        LoadJavaClass["com.wolfram.paclet.PacletFactory"];
        paclets = com`wolfram`paclet`PacletFactory`createPaclets[piFile];
        If[!MatchQ[paclets, {_?JavaObjectQ}],
            Return[{150, "Problem reading PacletInfo.m file: " <> GetJavaException[]@toString[]}]
        ];
        pacletObj = First[paclets];
        pacletName = pacletObj@getName[];

        rules = GetPacletElementList[pacletObj@getPacletInfo[], submissionExtrasDir];

        Return[rules];

        (*********************  Convert docs  ***********************)
        (* We take the SourceDocumentation dir and mirror its contents into the paclet's Documentation
           dir (which we might need to create). Along the way we convert each notebook from author form
           to in-product form. The paclet might already have built docs in the Documentation dir, but we
           just overwrite. Any other files or subdir structure in the Documentation dir are left intact.
        *)
        (* TODO: It is not technically correct to just look for a Documentation dir in the paclet. The actual
           location of the paclet's docs is determined from the PacletInfo.m file (PacletRoot/DocRoot).
           But all paclets will probably use the standard layout.
        *)
        documentationSourceDir = ToFileName[submissionExtrasDir, "DocumentationSource"];
        If[FileType[documentationSourceDir] =!= Directory,
            Return[{151, "SubmissionExtras/DocumentationSource directory does not exist."}]
        ];
        pacletDocsDir =
            If[Length[FileNames["Documentation", unzipDir, 2]] > 0,
                First[FileNames["Documentation", unzipDir, 2]],
            (* else *)
                (* Documentation dir in paclet does not exist, so create it. *)
                Transmogrify`Utilities`Private`createDirectory[ToFileName[{pacletObj@getLocation[], pacletObj@getRoot[]}, "Documentation"]]
            ];
        mirrorDirStructure[documentationSourceDir, pacletDocsDir];
        documentationSourceDirLength = Length[FileNameSplit[documentationSourceDir]];
        (* Walk through the set of files in DocumentationSource dir, copying them into the paclet's Documentation
           dir at the corresponding location, building nb docs along the way. If the file already exists in the paclet's
           Documentation dir, delete it from there first. Also copy the dir hierarchy.
           Note that docResult will get $Failed for notebooks that are not intended to be built by the standard
           system. This will cause the 102 error below. For now, the way to avoid this is to not put such notebooks
           into the DocumentationSource dir--only put them into the paclet's Documentation dir.
        *)
        fullDocSourcePaths = Select[FileNames["*", documentationSourceDir, Infinity], (FileType[#] === File)&];
        docResult =
            Function[{docPath},
                destPath = ToFileName[{pacletDocsDir}, FileNameDrop[docPath, documentationSourceDirLength]];
                (* Delete the one in dest dir if it exists. *)
                If[FileExistsQ[destPath], DeleteFile[destPath]];
                If[StringMatchQ[docPath, "*.nb", IgnoreCase->True],
                    (* For notebooks, build them and put output into dest dir. *)
                    buildPacletDocumentation[destPath, Get[docPath]],
                (* else *)
                    (* Not a nb file; copy it unmodified. *)
                    CopyFile[docPath, destPath]
                ]
            ] /@ fullDocSourcePaths;
        If[!MatchQ[docResult, {__String}],
            Return[{102, "Failure building paclet documentation"}]
        ];

        (*********************  Build doc indexes  ***********************)
        (* TODO: Figure out how indexer reports errors and report them here. *)
        indexNotebooks[FileNames["*.nb", pacletDocsDir, Infinity], ToFileName[{pacletDocsDir, "English"}], "English"];

        (********************  Delete the SubmissionExtras dir  **********************)
        (* We already took everything we need from it. It just gets in the way when we pack the paclet. *)
        DeleteDirectory[submissionExtrasDir, DeleteContents->True];

        (****************  Encode .m files if requested  *****************)
        If[TrueQ["Encode" /. rules],
            Encode[#, #]& /@ Select[FileNames["*.m", unzipDir, Infinity], !StringMatchQ[#, "*PacletInfo.m"]&]
        ];

        (*********************  Pack the paclet  ***********************)
        (* It is essential that by the time we get here, everything from the unzipped submission that is not
           part of the paclet itself has been deleted.

           Two possible dir structures for unzipDir at this point:
           1)    unzipDir/
                   MyPaclet/
                       PacletInfo.m
                       ...etc.

           2)    unzipDir/
                   PacletInfo.m
                   ...etc.

            In the case of (1) we want to call PackPaclet on the MyPaclet dir; in (2) on the unzipDir itself.
        *)
        pacletFile =
            If[Length[FileNames["PacletInfo.m", unzipDir]] > 0,
                PackPaclet[unzipDir, outputDir],
            (* else *)
                PackPaclet[First[FileNames["*", unzipDir]], outputDir]
            ];
        If[StringQ[pacletFile],
            pacletFile,
        (* else *)
            {200, "Failure packing the paclet into a .paclet file (paclet probably has invalid structure or data)"}
        ]
    ]
  ]


(*
    file: Path to output file
    notebookExprOrFilePath: Author source notebook to convert
    return value: path to output file
*)
buildPacletDocumentation[file_String, notebookExprOrFilePath:(_Notebook | _String)]:=
Module[{res},
    (* convert author notebook *)
    res = DocumentationBuild`Make`MakeNotebook[notebookExprOrFilePath, "FunctionPaclet" -> True];
    If[ Head@res =!= Notebook, Return[$Failed] ];
    res = res /. Cell[p__, "Usage", o___] :> Cell[p, "PacletUsage", o];
    res = res /. Cell[p__, "NotesSection", o___] :> Cell[p, "PacletNotesSection", o];
    res = res /. FrameBox[StyleBox[_, "NotesFrameText", ___], ___] :>
      ToBoxes[Import[ToFileName[
        {$DocumentationBuildDirectory, "Internal", "web", "html", "images", "mathematicaImages"},
        "MoreInformation.jpg"], {"JPG", "Graphics"}]];
    res = res /. Cell[p__, "Input", o___] :>
      If[MemberQ[{o}, _[CellContext, _]],
        Cell[p, "Input", o],
        Cell[p, "Input", CellContext -> "Global`"]];
    (* export converted notebook *)
    Export[file, res, "NB"]
]


indexNotebooks[files:{___String}, languageSpecificDir_, language_]:=
    Module[{indexDir, indexSpellDir, metaData, plainText},
        indexDir = ToFileName[languageSpecificDir, "Index"];
        If[FileType@indexDir === Directory, DeleteDirectory[indexDir, DeleteContents -> True]];
        CreateDirectory[indexDir];
        Needs["DocumentationSearch`"];
        DocumentationBuild`indexer = DocumentationSearch`NewDocumentationNotebookIndexer[indexDir, "Language"->language];
        indexSpellDir = ToFileName[{DirectoryName[indexDir]}, "SpellIndex"];
        Function[{nbFile},
            plainText = Import[nbFile, {"NB","Plaintext"}];
            metaData = getSearchMetaDataList[nbFile];
            DocumentationSearch`AddDocumentationNotebook[DocumentationBuild`indexer, plainText, metaData]
        ] /@ files;
        DocumentationSearch`CloseDocumentationNotebookIndexer[DocumentationBuild`indexer];
        DocumentationSearch`CreateSpellIndex[indexDir, indexSpellDir];
    ]

(*  Gather search metadata list from in-product nb *)
getSearchMetaDataList[nb_String, opts___?OptionQ]:=
  getSearchMetaDataList[Get@nb, opts];
getSearchMetaDataList[c___]:= (Message[GetSearchMetaDataList::arg, c]; $Failed);
getSearchMetaDataList::arg = "Incorrect arg: `1`";
getSearchMetaDataList[nbExpr_Notebook, opts___?OptionQ]:=
Module[{expr, ret},
  expr = Rest[List @@ nbExpr];
  If[Length@expr > 0,
    If[ OptionQ[ret = TaggingRules /. expr],
      ret = Flatten[ {"Metadata" /. ret}], {}]
    ,
    Return[{}];
  ];
  If[Length@ret < 2, {}, ret]
];


(* Doc-conversion code pulls some important info from Categorization cells in the notebook. These
   are not in the authoring notebook because we want to programmatically add them instead of asking
   authors to do it. The addDocCategorization[] function adds these cells.
*)
addDocCategorization[docNb_Notebook, entityType_String, pacletName_String, pacletContext_String, pacletURI_String] :=
    Module[{categorizationCells},
        categorizationCells = {
            Cell["Categorization", "CategorizationSection"],
            Cell[entityType, "Categorization", CellLabel->"Entity Type"],
            Cell[pacletName, "Categorization", CellLabel->"Paclet Name"],
            Cell[pacletContext, "Categorization", CellLabel->"Context"],
            Cell[pacletURI, "Categorization", CellLabel->"URI"]
        };
        Fold[Insert[#1, #2, {1, 1}]&, docNb, Reverse[categorizationCells]]
    ]


(* We need to generate a file name and URI for guide pages from the template. We base it on the title cell
   of the page, but guide titles might have spaces in them, which we remove. Any further modifications
   that become necessary should be done here.
*)
fixGuideTitle[title_String] := StringReplace[title, " " -> ""]


(********************************  ExportPacletMarketingPage  **********************************)

ExportPacletMarketingPage[outFile_String, downloadDir_String, inFile_String /; FileType[inFile] === File, opts___?OptionQ]:=
PubsEvaluateWithFE@
	ExportPacletMarketingPage[ outFile, downloadDir, GetPacletElementList[ Get@inFile], opts];

ExportPacletMarketingPage[outFile_String, downloadDir_String, nbExpr_Notebook, opts___?OptionQ]:=
PubsEvaluateWithFE@
	ExportPacletMarketingPage[ outFile, downloadDir, GetPacletElementList[nbExpr], opts];

ExportPacletMarketingPage[outFile_String, downloadDir_String, ruleList_List, opts___?OptionQ]:=
PubsEvaluateWithFE@
Module[{language, layoutfile, rules, dirName, content, dir, file, pacletName, pacletVersion, thumbnailCell,
            pacletFileName, docs, engDir, symbolPage, webPage, indexDir, pacletPath=None, localBuild},

    {language, layoutfile, dirName, pacletPath, localBuild} = {"Language", "LayoutFile", "OutputDirectoryName", "PacletPath", "LocalBuild"} /. {opts} /. Options[ExportPacletMarketingPage];

    ServerLog[ separator <> "\nGetPacletElementList:"];
    ServerLog[ Column@ruleList ];

    rules = Flatten[{ convertPacletElementRules[#, "FileName"->outFile] & /@ ruleList }];

    pacletName = "PacletName" /. rules;
    pacletVersion = "PacletVersion" /. ruleList;
    pacletFileName = "PacletFileName" /. rules;

    (* for debugging only *)
    If[ FileType[pacletPath] === File,
    	initText = getPacletInitCode[pacletPath, pacletName, pacletVersion];
    ];

    ServerLog[ separator <> "\nRule list:"];
    ServerLog[ Column@rules ];

    dirName = If[dirName === None, Part[StringSplit[outFile, "/" | "\\"], -2], dirName];
    ServerLog[ separator <> "\nDir name:\n" <> dirName];

    If[StringQ[pacletFileName],
	    (* Add a rule giving the paclet download URL:
	       /download.jsp?url=downloadDir&name=Foo&version=1.0.0&fileName=Foo-1.0.0
	    *)
        AppendTo[rules, "PacletDownload" -> StringJoin[
            "<a href='/download.jsp?url=",
                downloadDir,
                "&name=", pacletName,
                "&version=", pacletVersion,
                "&fileName=", pacletFileName,
                "'>",
                "<img src='/images/download.jpg' border='0' />",
             "</a>"
            ]
	    ]
    ];

    content = Transmogrify`FillInLayout[
        layoutfile,
        "Text",
        "downloadDir" -> downloadDir,
        "initText" -> initText,
        Sequence@@DeleteCases[rules, Rule["Documentation", _], Infinity],
        Sequence@@
        	If[ localBuild,
        	{
        	"header" -> $header,
        	"footer" -> $footer }
        	,
        	{
        	"header" -> "",
        	"footer" -> "" }
        ]
    ];

    Export[outFile, content, "Text"]
]

Options[ExportPacletMarketingPage] = {
    "Language"->"English",
    "OutputForm"->"HTML",
    "OutputDirectoryName"->None,
    "PacletPath"->None,
    "LayoutFile"->ToFileName[{$DocumentationBuildDirectory, "FileTemplates"}, "PacletMarketingPage.html"],
    "LocalBuild"->False
    }


(* This is only for marketing page stuff. *)
convertPacletElementRules[ Rule[l_, v_], opts___?OptionQ]:=
Module[{lab=l, val=v, rules, transform, res, outFile, currentDir, outputDir},
    (* label must be a string *)
    If[!StringQ@lab, Return @ Rule[lab, val] ];
    (* If val is already a string, then return *)
    If[StringQ@val, Return @ Rule[lab, val] ];

    {transform, outFile} = {"Transform", "FileName"} /. {opts} /. Options[convertPacletElementRules];
    (* *)
    currentDir = Directory[];
    outputDir = DirectoryName@outFile;
    CreateDirectory[outputDir];
    SetDirectory[outputDir];

    val = If[lab === "Implementation", Cell[BoxData@ToBoxes@Rasterize[val], "Graphics"], val];
    val = If[(lab === "Documentation") && (Head@val === Notebook), Return @ Rule[lab, val], val];
    val = If[lab === "OtherRequirements", ToString[val], val];

    res =
    If[Head@val === List,
        Map[Transmogrify`Transmogrify[ Flatten[{ #}], transform]&, val],
        Transmogrify`Transmogrify[ Flatten[{ val}], transform]
    ];

    SetDirectory[currentDir];

    res = If[Head@res === List, Apply[StringJoin, exportToXML/@res], exportToXML[res] ];

    Rule[lab, res]
];
Options[convertPacletElementRules] = {
    "Transform" -> ToFileName[{$DocumentationBuildDirectory, "XMLTransforms"}, "PacletMarketingPage.m"] };


$header =
"<html>
	<head>
		<link rel='stylesheet' href='/css/paclets.css' type='text/css' media='all' />
		<link rel='stylesheet' href='/css/search.css' type='text/css' media='all' />
	</head>
<body>
";
$footer =
"
<script type='text/javascript' src='/javascript/paclets.js'></script>
</body>
</html>";

(**********************************  BuildPacletWebDocs  ************************************)

(* Takes the Notebook expression from the paclet template notebook and builds an HTML version
   of the docs. The generated HTML is placed into an "html" subdirectory of the specified outputDir.
*)
(* TODO: This function must be modified to correctly handle paclet submissions that have more than
   one function and doc page.
*)
BuildPacletWebDocs[expr:{__Notebook}, outputDir_String, rules_List, language_String, opts___?OptionQ] :=
    Module[{webPage, res, docs=expr, author, name, version, context, description, symbolName, symbolNames,
    	otherRequirements, mVersion, code, htmlTemplateDir, htmlTemplateFiles, docNotebook, guidePages, symbolPages},

    	DocumentationBuild`$FunctionPaclet = True;

        (* Validation has already occurred so that we can be confident that these values are present and valid. *)
        {author, name, version, context, description, symbolNames, docs, otherRequirements, mVersion, code} =
            {"Author", "PacletName", "PacletVersion", "PacletContext", "ShortDescription", "FunctionNames",
                "Documentation", "OtherRequirements", "MathematicaVersion", "Implementation"} /. rules;

        CreateDirectory[ToFileName[outputDir, "docs"]];
        webPage = ToFileName[{outputDir, "docs"}, "index.html"];

		ServerLog["Copying html files for local viewing."];

		htmlTemplateDir =
			ToFileName[{DocumentationBuild`Common`$DocumentationBuildDirectory, "Internal", "web", "html"}];

		(*Copy ExampleData files/dir ignoring'bad' files*)

		htmlTemplateFiles = FileNames["*", htmlTemplateDir, Infinity];

		htmlTemplateFiles =
			Select[htmlTemplateFiles, (StringFreeQ[#, $PathnameSeparator ~~ "CVS" ~~ $PathnameSeparator] && FileType[#] === File) &];

		Map[
			Function[
				outFile = ToFileName[{outputDir}, StringReplace[#, {htmlTemplateDir -> ""}]];
				CreateDirectory[DirectoryName[outFile]];
				CopyFile[#, outFile];
			]
			,
			htmlTemplateFiles];

		ServerLog[separator <> "\nWeb page:\n" <> webPage];

        guidePages = Select[docs, !FreeQ[#, Cell[_, "GuideTitle", ___]]&];
        symbolPages = Select[docs, !FreeQ[#, Cell[_, "ObjectName"|"ObjectNameSmall", ___]]&];
        (* Fix up some metadata in notebooks by calling addDocCategorization. *)
        guidePages =
            Function[{guidePage},
                addDocCategorization[guidePage, "Guide", name, context, name <> "/guide/" <>
                       fixGuideTitle[First[Cases[guidePage, Cell[guideTitle_, "GuideTitle", ___] :> guideTitle, Infinity]]]]
            ] /@ guidePages;
        symbolPages = MapThread[addDocCategorization[#1, "Symbol", name, context, name <> "/ref/" <> #2]&, {symbolPages, symbolNames}];

        (* Here is where modifications need to occur for building html docs for a multi-function paclet. The
           guidePages and symbolPages variables hold lists of Notebook expressions. Assume that the first
           guide page is the "main" page. The previous behavior for single-function paclets is maintained
           for now by simply calling First[symbolPages] below.
        *)
		res = DocumentationBuild`Export`ExportDocumentation[webPage, First[symbolPages],
			"ExportFormats" -> "HTML", "Language" -> language, "CompleteHTMLQ"->True, "FunctionPaclet"->True];

		ServerLog[ separator <> "\nWeb page result:\n" <> ToString[res] ];

		res
	];


(************************  Notebook Content Extraction Utilities  *************************)

GetPacletElementList[pacletObj_?JavaObjectQ, submissionExtrasDir_String] :=
  JavaBlock[
     Module[{pacletInfo, submissionInfo, snapshots, licenseAgreement},
        pacletInfo = pacletObj@getPacletInfo[];
        submissionInfo = List @@ Get[ToFileName[submissionExtrasDir, "SubmissionInfo.m"]];
        (* To support rules that have a LHS of either strings or symbols, convert all LHS to strings. *)
        submissionInfo = submissionInfo /. (x_ -> y_) :> (ToString[x] -> y);
        snapshots = "Snapshots" /. submissionInfo;
        If[MatchQ[snapshots, _String | {__String}] && snapshots =!= "Snapshots",
            (* Resolve the paths relative to SubmissionExtras dir and load the data. *)
            snapshots = Import /@ ToFileName[submissionExtrasDir, #]& /@ Flatten[{snapshots}],
        (* else *)
            snapshots = {}
        ];
        licenseAgreement = pacletInfo@getLicense[];
        If[StringQ[licenseAgreement] && licenseAgreement =!= "",
            (* Resolve the path relative to SubmissionExtras dir and load the data. *)
            licenseAgreement = Import[ToFileName[pacletInfo@getLocation[], licenseAgreement]],
        (* else *)
            licenseAgreement = "N/A"
        ];
        {
        "Author" -> pacletInfo@getCreator[],
        "AuthorURL" -> pacletInfo@getCreatorURL[],
        "SupportContact" -> pacletInfo@getSupportContact[],
        "PacletName" -> pacletInfo@getName[],
        "PacletVersion" -> pacletInfo@getVersion[],
        "PacletContext" -> First[PacletManager`Package`getPacletManager[]@getContexts[pacletObj]],
        "ShortDescription" -> pacletInfo@getDescription[],
        "MathematicaVersion" -> pacletInfo@getMathematicaVersion[],
        "PlatformRequirements" -> pacletInfo@getSystemIDs[],   (* {} *)
        "AdditionalRequirements" -> ("AdditionalRequirements" /. submissionInfo /. "AdditionalRequirements"->"N/A"),  (* "AdditionalRequirements" *)
        "KeyFeatures" -> ("KeyFeatures" /. submissionInfo /. "KeyFeatures"->"N/A"),    (* "KeyFeatures" *)
        "NewInVersion" -> ("NewInVersion" /. submissionInfo /. "NewInVersion"->"N/A"),   (* "NewInVersion" *)
        "Price" -> ("Price" /. submissionInfo /. "Price"->"N/A"),  (* "Price" *)
        "Thumbnail" -> pacletInfo@getThumbnail[],   (* TODO file path conversion? *)
        "Categorization" -> ("Categorization" /. submissionInfo /. "Categorization"->"N/A"),  (* "Categorization" *)
        "Description" -> ("Description" /. submissionInfo /. "Description"->"N/A"),  (* "Description" *)
        "SearchTerms" -> ("SearchTerms" /. submissionInfo /. "SearchTerms"->"N/A"),  (* "SearchTerms" *)
        "Snapshots" -> snapshots,
        "Agreement" -> licenseAgreement,
        "Loading" -> pacletInfo@getLoadingState[],
        "Encode" -> ("Encode" /. submissionInfo /. "Encode"->False)
        }
     ]
  ]

GetPacletElementList[nbExpr_Notebook] :=
    {
        "Author" -> getAuthor[nbExpr],
        "AuthorURL" -> getAuthorURL[nbExpr],
        "SupportContact" -> getSupportContact[nbExpr],
        "PacletName" -> getPacletName[nbExpr],
        "PacletContext" -> getPacletContext[nbExpr],
        "ShortDescription" -> getShortDescription[nbExpr],
        "KeyFeatures" -> getKeyFeatures[nbExpr],
        "NewInVersion" -> getNewInVersion[nbExpr],
        "PacletVersion" -> getPacletVersion[nbExpr],
        "MathematicaVersion" -> getMathematicaVersion[nbExpr],
        "PlatformRequirements" -> getPlatformRequirements[nbExpr],
        "AdditionalRequirements" -> getAdditionalRequirements[nbExpr],
        "AdditionalRequirementsQ" -> additionalRequirementsQ[nbExpr],
        "Price" -> getPrice[nbExpr],
        "Thumbnail" -> getThumbnail[nbExpr],
        "Categorization" -> getCategorization[nbExpr],
        "Description" -> getDescription[nbExpr],
        "SearchTerms" -> getSearchTerms[nbExpr],
        "Documentation" -> getDocumentation[nbExpr],
        (* Although paclets can have more than one function, for compatibility we keep "FunctionName",
           which now refers to "main" function.
        *)
        "FunctionName" -> First[getFunctionNames[nbExpr]],
        "FunctionNames" -> getFunctionNames[nbExpr],
        "Implementation" -> getImplementation[nbExpr],
        "TemplateVersion" -> getTemplateVersion[nbExpr],
        "Snapshots" -> getSnapshots[nbExpr],
        "Agreement" -> getAgreement[nbExpr],
        "Loading" -> getLoadingBehavior[nbExpr],
        "Encode" -> getEncode[nbExpr]
    }


GetPacletThumbnail[file_String /; FileType[file] === File] :=
	GetPacletThumbnail[ Get@file];

GetPacletThumbnail[expr_Notebook] :=
Module[{res},
    res = getThumbnail[expr];
    If[Head[res] === Cell,
        res,
    (* else *)
        None
    ]
]


(***********  Extraction functions for template content.  ***********)

(* General rule for these extractors is that if they don't find the content they return either {} or "N/A".
   If the successful result is a list, then the failure result should be {}; if the successful result is
   something else the failure result should be "N/A".
*)

getAuthor[nb_Notebook] :=
Module[{n},
    n = Cases[nb, Cell[c_, "Author" | "Creator", ___] :> c, Infinity];
    If[!stringValuePresent[n], "N/A", trimSpaces@convertToString@First@n]
]

getAuthorURL[nb_Notebook] :=
Module[{n},
    n = Cases[nb, Cell[c_, "AuthorURL" | "CreatorURL", ___] :> c, Infinity];
    If[!stringValuePresent[n], "N/A", trimSpaces@convertToString@First@n]
]

getSupportContact[nb_Notebook] :=
Module[{n},
    n = Cases[nb, Cell[c_, "SupportContact", ___] :> c, Infinity];
    If[!stringValuePresent[n], "N/A", trimSpaces@convertToString@First@n]
]

getPacletName[nb_Notebook] :=
Module[{n},
    n = Cases[nb, Cell[c_, "PacletName", ___] :> c, Infinity];
    If[!stringValuePresent[n], "N/A", trimSpaces@convertToString@First@n]
]

getPacletContext[nb_Notebook] :=
Module[{n, context = "N/A", found = 0},
    n = Cases[nb, Cell[c_, "Input", ___] :> c, Infinity];
    n = trimSpaces[convertToString[#]] &/@ n;
    While[ found === 0,
    	If[ Head[First@n] === String,
			context = { StringReplace[First@n, RegularExpression["BeginPackage\\[(.+).+\\]"] :> "$1"] };
			If[ Length@context > 0,
				context = StringReplace[ First@context, {"\"" -> "", ";" -> ""}];
				found = 1,
				n = Drop[n, 1];
			]
    	]
    ];
    context
]

getShortDescription[nb_Notebook] :=
Module[{n}, n = Cases[nb, Cell[c_, "ShortDescription", ___] :> c, Infinity];
    If[!stringValuePresent[n], "N/A", trimSpaces@convertToString@First@n]
]

getPacletVersion[nb_Notebook] :=
Module[{n},
    n = Cases[nb, Cell[c_, "PacletVersion", ___] :> c, Infinity];
    If[!stringValuePresent[n], "N/A", trimSpaces@convertToString@First@n]
]

getMathematicaVersion[nb_Notebook] :=
Module[{n},
    n = Cases[nb, Cell[c_, "MathematicaVersion", ___] :> c, Infinity];
    If[!stringValuePresent[n], "N/A", makeThreeDigitVersion[trimSpaces@convertToString[First@n]] ]
]

makeThreeDigitVersion[num_String]:=
Module[{versList},
  versList = StringSplit[num, "."];
  If[versList === {}, Return["N/A"] ];

  versList = If[Length@versList > 0, Flatten@AppendTo[versList, {"0","0"}], versList ];

  StringJoin @@ Riffle[Take[versList, 3], "."]

]
makeThreeDigitVersion[___]:= "N/A";


getPlatformRequirements[nb_Notebook] :=
Module[{n},
    n = Cases[nb, Cell[c_, "PlatformRequirements", ___] :> c, Infinity];
    n = getTrueCheckBoxes@getCheckBoxRules[n];
    If[n === {}, "N/A", StringJoin@@Riffle[Map[convertToString[#]&, n], ", "] ]
]


getAdditionalRequirements[nb_Notebook] :=
Module[{n},
    n = Cases[nb, Cell[c_, "AdditionalRequirements", ___] :> c, Infinity];
    n = Map[trimSpaces@convertToString[#]&, n];
    DeleteCases[n, "None"]
]

additionalRequirementsQ[nb_Notebook] :=
Module[{n},
    n = getAdditionalRequirements[nb];
    If[n === "N/A", False, True ]
]

getPrice[nb_Notebook] :=
Developer`UseFrontEnd @
Module[{obj, cells, data},
	(* need to grab radiobox dynamics and convert to literal *)
	obj = System`NotebookPut[nb];
	System`NotebookFind[obj, "Price", All, CellStyle];
	FrontEndExecute[{FrontEnd`NotebookDynamicToLiteral[
    	FrontEnd`NotebookSelection[obj]]}];
	cells = Flatten[{ System`NotebookRead[obj] }];
	System`NotebookClose[obj];
	(* now treat cells normally *)
	Which[
		Length@cells > 1,
			data = Cases[cells,
				Cell[BoxData[ RowBox[{RadioButtonBox[val_, ___],
					Cell[lab_, "TemplateNotes"], ___}]], "Price", ___] :>
						{val, lab}, Infinity];
			data = Cases[data, {a_, b_} /; StringMatchQ[b, "*" <> ToString@a <> "*"], Infinity];
			If[ Length@data > 0, trimSpaces@Part[data, 1,2], "N/A"],
		Length@cells === 1,
			trimSpaces@convertToString@Part[cells, 1, 1],
		_,
			"N/A"
	]
]

getThumbnail[nb_Notebook] :=
Module[{n},
    n = Cases[nb, Cell[c_, "Thumbnail", ___], Infinity];
    n = deletePlaceholderCells[n, "Thumbnail Placeholder"];
    If[Length[n] > 0, First[n], "N/A"]
]

getSnapshots[nb_Notebook] :=
Module[{n},
    n = getCellGroup[nb, "SnapshotsSection"];
    n = n /. { Cell[CellGroupData[{ Cell[_, "SnapshotsSection", ___], cells___},___],___] } :> { cells};
    n = n /. Cell[c_, "Graphics", o___] :> Cell[c, "Snapshot", o];
    n = deletePlaceholderCells[n, "Thumbnail Placeholder"];
    If[n === {}, {}, { Cell@CellGroupData[ Flatten@ { Cell["Additional Images", "SnapshotsSection"], n }] } ]
]

getCategorization[nb_Notebook] :=
Module[{n},
    n = getTrueCheckBoxes@getCheckBoxRules@getCellGroup[nb, "PacletCategorizationSection"];
    If[n === {}, "N/A", StringJoin@@Riffle[Map[convertToString[#]&, n], ", "] ]
]

getCellGroup[expr_Notebook, groupStyle_String]:=
    Cases[expr,
    	Cell[CellGroupData[{ Cell[_, groupStyle, ___], ___}, ___], ___],
        Infinity];

getDescription[nb_Notebook] :=
Module[{n},
    n = Cases[nb, Cell[c_, "Description", ___] :> c, Infinity];
    If[!stringValuePresent[n], "N/A", StringJoin @@ Riffle[Map[convertToString[#]&, n], "\n\n"] ]
]

getSearchTerms[nb_Notebook] :=
Module[{n},
    n = Cases[nb, Cell[c_, "SearchTerms", ___] :> c, Infinity];
    If[!stringValuePresent[n], "N/A", StringJoin@@Riffle[Map[convertToString[#]&, n], ", "] ]
]

getKeyFeatures[nb_Notebook] :=
Module[{n},
    n = deleteXXXCellsFromList @ Cases[nb, Cell[_, "KeyFeatures", ___], Infinity];
    If[n === {}, {}, { CellGroupData[ Flatten@ { Cell["Key Features", "KeyFeaturesSection"], n }] } ]
]

getNewInVersion[nb_Notebook] :=
Module[{n},
    n = deleteXXXCellsFromList @ Cases[nb, Cell[_, "NewInVersion", ___], Infinity];
    If[n === {}, {}, { CellGroupData[ Flatten@ { Cell["New In This Version", "NewInVersionSection"], n }] } ]
]

getFunctionNames[nb_Notebook] :=
Module[{docSection},
    docSection = Cases[nb, Cell[CellGroupData[{Cell[__, CellTags->"DocumentationSection", ___], ___}, ___], ___], Infinity];
    Cases[docSection, Cell[c_, "ObjectName"|"ObjectNameSmall", ___] :> trimSpaces@convertToString@c, Infinity]
]

getAgreement[nb_Notebook] :=
Module[{n},
    n = Cases[nb, Cell[c_, "Agreement", ___] :> c, Infinity];
    n = Cases[n, CheckboxBox[v_] :> trimSpaces@convertToString@v, Infinity];
    If[n =!= {}, First[n], "N/A"]
]

getEncode[nb_Notebook] :=
    Flatten[
        Cases[nb, Cell[c_, "CodeAccess", ___] :> Cases[c, CheckboxBox[chk_] :> TrueQ[chk], Infinity], Infinity]
    ] === {True}

getDocumentation[nbexpr_Notebook] :=
Module[{expr, nbs, opts, OnePage},
  (* grab nb options, especially style sheet *)
  opts = Rest[List@@nbexpr];
  (* Extract the documentation content as a list or elements wrapped in the inert OnePage wrapper.
     Each OnePage element is, of course, a single doc page from the template notebook.
   *)
  expr = Flatten @ Cases[nbexpr,
    Cell[CellGroupData[{Cell[_, "Section", ___, CellTags -> "DocumentationSection", ___], con___}, ___], ___]
      :> Cases[{con}, Cell[CellGroupData[{Cell[_, "Subsection", ___], docs___}, ___], ___] :> OnePage[{docs}], 1],
    Infinity];
  (* This replaces the OnePage wrapper on each page with Notebook. *)
  nbs = Notebook[#,
          ShowCellBracket->False, DockedCells->{}, StyleDefinitions -> FrontEnd`FileName[{"Wolfram"}, "Reference.nb"]
        ]& @@@ expr
]

getImplementation[nb_Notebook] :=
Module[{n},
    n = flattenCellGroups @
            Cases[nb, Cell[CellGroupData[{Cell[_, "CodeSection", ___], ___}, ___], ___], Infinity];
    n = Flatten@DeleteCases[n, Cell[_, "CodeSection", ___], 2];
    (* only allow Input | Code styled cells into package *)
    n = Flatten@DeleteCases[n, Cell[_, s_String /; !StringMatchQ[s, "Input"|"Code"], ___], 2];
    (* turn cells into StandardForm and HoldComplete *)
    Thread[Flatten[cellMakeExpression[#] & /@ n], HoldComplete]
]

getTemplateVersion[nb_Notebook] :=
Module[{n},
    n = "Revision" /. (TaggingRules /. Rest[List @@ nb]);
    If[n =!= "Revision", n, "N/A"]
]

getLoadingBehavior[nb_Notebook] :=
    First[Cases[nb, Literal[DynamicModuleBox[{$CellContext`LoadingBehavior = behav_}, __]] :> behav, Infinity]]

(* *)
getCheckBoxRules[expr_Notebook, groupStyle]:=
Module[{grp},
    grp = getCellGroup[expr, groupStyle];
    Flatten@{getCellTagCheckBoxPair /@ Flatten @ Cases[grp, Cell[_, "PacletCategorization", __], Infinity]}
]
getCheckBoxRules[expr_List]:=
    Flatten@{getCellTagCheckBoxPair /@ Flatten @ Cases[expr, Cell[_, "PacletCategorization", __], Infinity]}


getTrueCheckBoxes[{sugs___?OptionQ}] :=
 Cases[{sugs}, Rule[name_, True] :> name, Infinity]

getCellTagCheckBoxPair[Cell[con_, __, CellTags->t_, ___]]:=
Module[{cb, ct},
    cb = And @@ Cases[con, CheckboxBox[b_]:> b, Infinity];
    ct = If[Head@t === String, t, If[Length@t > 0, First@t, Null]];
    Rule[ct,cb]
]


stringValuePresent[cellContents_List] :=
    cellContents =!= {} && cellContents =!= {""} && cellContents =!= {"XXXX"}


(******************************  Verify paclet correctness  **********************************)

(*
    Returns a list: {errorCode_Integer, errorText_String}. The code is 0 for success (all elements valid).
*)
ValidatePacletElements[elementRules_List, isTemplatePaclet:(True | False):True] :=
    Module[{author, name, version, shortDesc, code, docs, context, symbols, desc, price},
        {author, name, version, shortDesc, docs, code, context, symbols, desc, price} =
                {"Author", "PacletName", "PacletVersion", "ShortDescription", "Documentation", "Implementation",
                 "PacletContext", "FunctionNames", "Description", "Price"} /. elementRules;

        (* Author checks *)
        If[author == "N/A",
            Return[{1001, "Paclet author is not present"}]
        ];

        (* Name checks *)
        If[name == "N/A",
            Return[{2001, "Paclet name is not present"}]
        ];
        If[!StringMatchQ[name, LetterCharacter ~~ WordCharacter ..],
            Return[{2002, "Paclet name is not legal"}]
        ];

        (* Version checks *)
        If[version == "N/A",
            Return[{3001, "Paclet version is not present"}]
        ];
        If[!StringMatchQ[version, DigitCharacter .. ~~ ("." ~~ DigitCharacter ..) ...],
            Return[{3002, "Paclet version is not legal"}]
        ];

        (* Short Description checks *)
        If[shortDesc == "N/A",
            Return[{4001, "Paclet short description is not present"}]
        ];
        If[StringLength[shortDesc] > 200,
            Return[{4002, "Paclet short description is too long"}]
        ];

        (* Abstract checks *)
        If[desc == "N/A",
            Return[{9001, "Paclet description is not present"}]
        ];

        (* Price checks *)
        If[price == "N/A",
            Return[{10001, "Paclet price is not present"}]
        ];

        (* Some tests only relevant for template-style paclets, not zip-style. *)
        If[isTemplatePaclet,
            (* Doc checks *)
            If[Length[docs] == 0,
                Return[{5001, "Paclet documentation is not present"}]
            ];

            (* Code checks *)
            If[code == "N/A",
                Return[{5001, "Paclet implementation is not present"}]
            ];
            If[!MatchQ[code, HoldComplete[_List]],
                Return[{6002, "Paclet implementation is not in correct format"}]
            ];

            (* Context checks *)
            (*** ignore for now--not present and not used...
            If[context == "N/A",
                Return[{7001, "Paclet context is not present"}]
            ];
            If[!StringMatchQ[context, LetterCharacter ~~ WordCharacter... ~~ "`"],
                Return[{7002, "Paclet context is not a legal context name"}]
            ];
            ***)

            (* Function name checks *)
            If[Length[symbols] == 0,
                Return[{8001, "Function name is not specified"}]
            ];
            If[Not[And @@ (StringMatchQ[#, LetterCharacter ~~ WordCharacter...]& /@ symbols)],
                Return[{8002, "Function name is not a legal Mathematica symbol"}]
            ]
        ];

        (* No tests yet for:
            "OtherRequirements",
            "Thumbnail",
            "Categorization",
            "SearchTerms"
        *)

        (* Success value *)
        {0, "No errors"}
    ]


(******************************  Miscellaneous Utilities  **********************************)

(* turn cells into HeldComplete SandardForm *)
cellMakeExpression[a_]:=cellMakeExpression[a,StandardForm]
cellMakeExpression[Notebook[a_List, ___],fmt_]:=cellMakeExpression[a,fmt]
cellMakeExpression[Cell[CellGroupData[a_List,___],___],fmt_]:=cellMakeExpression[a,fmt]
cellMakeExpression[a_List, fmt_] := cellMakeExpression[#, fmt]& /@ a
cellMakeExpression[Cell[BoxData[a_List],___],fmt_]:=cellMakeExpression[a,fmt]
cellMakeExpression[Cell[BoxData[boxes_],___],fmt_]:=cellMakeExpression[boxes,fmt]
cellMakeExpression[FormBox[a_,fmt_],_]:=cellMakeExpression[a,fmt]
cellMakeExpression[boxes_,fmt_]:=MakeExpression[StripBoxes[boxes],fmt]


exportToXML[con_]:=
Switch[Head@con,
    XMLElement, ExportString[ con, "XML"],
    String, con,
    _, Message[exportToXML::val, Head@con]; con
];
exportToXML::val = "Expected String or XMLElement, found `1`.";


(* extract init file from paclet *)
getPacletInitCode[path_String, name_String, version_String]:=
Module[{tempDir = $TemporaryPrefix<>"-"<>StringJoin @@ ToString /@ DateList[Date[]], res, initFilePath, init},
	res = UnpackPaclet[path, CreateDirectory@tempDir];
	initFilePath = ToFileName[ {tempDir, name<>"-"<>version, name, "Kernel"}, name<>".m"];
	init = Import[ initFilePath, "Text"];
	(* delete directory *)
	DeleteDirectory[tempDir, DeleteContents->True];
	(* return init file *)
	init
];


cleanFileNameString[str_String] :=
Module[{res=str},
  res = StringReplace[trimSpaces@res, RegularExpression["[^a-zA-Z0-9$_\\-\\./\\\\]"] -> ""];
  If[Head@res =!= String, Message[cleanFileNameString::reg, str]; str, res]
];
cleanFileNameString[str___] := (Message[cleanFileNameString::arg, str]; $Failed)
cleanFileNameString::reg = "StringReplace error in `1`";
cleanFileNameString::arg = "String expected, found `1`";


deleteXXXCellsFromList[expr_List]:=
DeleteCases[expr,
	Alternatives[
        Cell[s_String /; StringMatchQ[s, RegularExpression["X+"]], ___],
        Cell[TextData[s_List /; Apply[ Or,
        	StringMatchQ[Transmogrify`ConvertToString[#],
        		RegularExpression[".*XXX+.*"]] & /@ s]], ___]
	], Infinity];


trimSpaces[s_String] :=
  StringReplace[s, { StartOfString ~~ Whitespace :> "", Whitespace ~~ EndOfString :> "" } ];

(* from loumeister *)
flattenCellGroups[nb_NotebookObject?NotebookOpenQ] := flattenCellGroups[NotebookGet @ nb]
flattenCellGroups[Notebook[cells_, opts___]] := Notebook[flattenCellGroups[cells], opts]
flattenCellGroups[{c__Cell}] := Flatten[flattenCellGroups /@ {c}]
flattenCellGroups[Cell[CellGroupData[{c__Cell}, ___], ___]] := flattenCellGroups /@ {c}
flattenCellGroups[x_] := x


(* in need a stringifyer... *)
convertToString[c_String] := c;
convertToString[c_Cell] := convertToString@c[[1]];
convertToString[c_List] := StringJoin[convertToString /@ c];
convertToString[c_ButtonBox] := convertToString@c[[1]];
convertToString[c_BoxData] := convertToString@c[[1]];
convertToString[c_StyleBox] := convertToString@c[[1]];
convertToString[c_RowBox] := StringJoin[convertToString /@ c[[1]]];
convertToString[c_TextData] := convertToString@c[[1]];
convertToString[c_FormBox] := convertToString@c[[1]];
convertToString[c_AdjustmentBox] := convertToString@c[[1]];
convertToString[c_SuperscriptBox] := StringJoin[convertToString@c[[1]] <> "^"<>convertToString@c[[2]]];
convertToString[c_SubscriptBox] := StringJoin[convertToString@c[[1]] <> "_"<>convertToString@c[[2]]];
convertToString[c_] := ToString@c;
(* warn if a sequence *)
convertToString[c__] := (Message[convertToString::sequence]; StringJoin[convertToString /@ {c}]);
convertToString::sequence = "Warning: Sequence used as an argument, should be List";

deletePlaceholderCells[lst_List, str_String]:=
DeleteCases[lst,
 Cell[c_ /;
   Length[Cases[c,
      StyleBox[s_String /; StringMatchQ[s, "*"<>str<>"*"], ___],
      Infinity]] > 0, ___], Infinity]


(* Mirrors the subdir hierarchy in srcDir into destDir. Dirs only; files are completely ignored.
   Doesn't matter how much, if any, subdir structure exists in destDir. Any existing
   dirs will be ignored. destDir must exist prior to calling.
*)
mirrorDirStructure[srcDir_, destDir_] :=
    Module[{srcSubdirs, destSubdir, relSrcSubdir},
        srcSubdirs = Select[FileNames["*", srcDir], (FileType[#] === Directory)&];
        Function[srcSubdir,
            relSrcSubdir = FileNameDrop[srcSubdir, FileNameDepth[srcDir]];
            destSubdir = ToFileName[destDir, relSrcSubdir];
            If[FileType[destSubdir] =!= Directory, CreateDirectory[destSubdir]];
            mirrorDirStructure[srcSubdir, destSubdir]
        ] /@ srcSubdirs
    ]



End[]

EndPackage[]