(* ::Package:: *)

(* :Title: DocumentationBuild.m *)

(* :Authors:
	Andrew Hunt, Shaun McCance
	andy@wolfram.com, shaunm@wolfram.com
*)

(* :Package Version: 0.50 *)

(* :Mathematica Version: 6.0 *)
                     
(* :Copyright: (c) 2005, Andrew Hunt, Shaun McCance
               All rights reserved. 
*)

(* :Requirements: *)
(* :Discussion:
	Documentation conversion from :
		* authored nbs -> in-product notebooks
		* authored nbs -> web pages
		* authored nbs -> ___
*)

BeginPackage["DocumentationBuild`Common`", {
  "JLink`",
  "Transmogrify`",
  "XML`"
}]
(* Exported symbols added here with SymbolName::usage *)  


$DocumentationBuildDirectory::usage = "Path the DocumentationBuild base directory.";
$DocumentationBuildTransformDirectory::usage = "Path the DocumentationBuild XMLTransform directory.";

$DocumentationBuildDebug::usage = "";

ServerLog;
$ServerBuildLogQ;
separator;

PubsEvaluateWithFE::usage = "PubsEvaluateWithFE[expr] is a wrapper for Developer`UseFrontEnd[...] which also sets various options useful for building documentation.";

`$ImageCount = 0;

FilePathToDocURI::usage = "FilePathToDocURI[path] returns the documentation url form of path.";
EmptyStylesheetCache::usage = "";

ExcludeFilesFromList::usage = "ExcludeFilesFromList[list, excludedFiles] removes files within list which match patterns defined in excludedFiles";
ExcludePathsFromList::usage = "ExcludePathsFromList[list, excludedPaths] removes paths within list which match patterns defined in excludedPaths";
ConvertGraphicsToBitmaps::usage = "";
TagBitmapConditionalizedGraphics::usage = "";


FilepathToURI::usage = "FilepathToURI[path] returns the documentation url form of path.";

notebookFileNames::usage = "";
Options[notebookFileNames] = {
  ExcludeDirectories->{"CVS", "Internal", "RawMaterial", "ExampleData", "Future", "RawGuides", "Future", "PackagesMirror", (*"Messages", "Compatibility",*)
  (* exclude per lbyrge *)
  "AppleScript", "XMLCapabilities",
  "BarAndPieCharts"~~$PathnameSeparator,
  "ClusterAnalysis"~~$PathnameSeparator,
  "ComputationalGeometry2D"~~$PathnameSeparator,
  "ComputationalGeometry3D"~~$PathnameSeparator,
  "DeveloperUtilities"~~$PathnameSeparator,
  "ErrorBarPlot"~~$PathnameSeparator,
  "ExperimentalFunctions"~~$PathnameSeparator,
  "FourierTrig"~~$PathnameSeparator,
  "StatisticsCommon"~~$PathnameSeparator,
  "SIUnits"~~$PathnameSeparator
  (* Remove documentation within Pubs cvs tree which are being provided by developers within their applications  *)
(*    "Packages"~~$PathnameSeparator~~"DatabaseLink",
    "Packages"~~$PathnameSeparator~~"GUIKit",
    "Packages"~~$PathnameSeparator~~"JLink",
    "Packages"~~$PathnameSeparator~~"NETLink",
    "Packages"~~$PathnameSeparator~~"WebServices",*)
    }, 
  ExcludeFiles -> Flatten[{StartOfString|$PathnameSeparator~~".",   "VirtualBookContents.nb" }],
  HoursAgo->Infinity,
  FileList->All
};

DocumentationBuild`$FunctionPaclet = False;


(***************************************
  Begin Package
***************************************)

Begin["`Private`"]


(* Print full messages *)
$MessagePrePrint = .;

(* Set base directory *)
$DocumentationBuildDirectory = DirectoryName[ System`Private`$InputFileName ];
$DocumentationBuildTransformDirectory = ToFileName[{$DocumentationBuildDirectory, "XMLTransforms"}];


(* Set Debug to False *)
$DocumentationBuildDebug = False;

(* log messages on server *)
ServerLog[mes_]:= If[ $ServerBuildLogQ === True, Print[mes]];
(*  *)
separator = StringJoin @@ Table["*", {50}];


(* Add DocumentationBuild\XMLTransforms if it isn't already on path *)
If[ !MemberQ[$DocumentationBuildTransformDirectory, Transmogrify`$XMLTransformPath], 
	PrependTo[Transmogrify`$XMLTransformPath, $DocumentationBuildTransformDirectory ];
	(* force transforms to be a part of path TODO: this shouldn't be necessary *)
	If[ !MemberQ[#, Transmogrify`$XMLTransforms], 
		PrependTo[Transmogrify`$XMLTransforms, #]] & /@ 
			Select[ FileNames["*.m", $DocumentationBuildTransformDirectory, Infinity], Transmogrify`XMLTransformQ[#]& ]
];


(************************  FrontEnd Functions  *************************)


PubsStartFrontEnd[] := Module[{oldProt, x},
  If[Head[$PubsFELink] =!= LinkObject || (LinkReadyQ[$PubsFELink];First[LinkError[$PubsFELink]]) =!= 0,
    $PubsFELink = Developer`InstallFrontEnd[];
    (* empty stylesheet cache to speed up build time *)
    EmptyStylesheetCache[];
    (* start a Kernel so that syntax coloring works *)
    (*Developer`UseFrontEnd @ FrontEndExecute[FrontEndToken["EvaluatorStart","Local"]];*)
     ExportString[Manipulate[x,{x,1,2}], "GIF"];
(*    oldProt = Unprotect[MathLink`CreateFrontEndLinks];
    Block[{MathLink`CreateFrontEndLinks, $ParentLink = Developer`InstallFrontEnd[]},
      $PubsFELink = $ParentLink;
      Get[ToFileName[{$InstallationDirectory,
        "SystemFiles", "FrontEnd", "TextResources"}, "GetFEKernelInit.tr"]]
    ];
    Protect[oldProt];
    Developer`UseFrontEnd[SetOptions[$FrontEndSession, Evaluator->"$ParentLink"]];*)
  ]
];

(* FE version of TimeConstrained *)
FETimeConstrained[func_, time_]:=
MathLink`CallFrontEnd @ FrontEnd`TimeConstrained[func, time];

PubsEvaluateWithFE[expr___] := (PubsStartFrontEnd[]; Developer`UseFrontEnd[expr]);
SetAttributes[PubsEvaluateWithFE, HoldAll];


(* Here we shrink notebooks by converting graphics to bitmaps (but only if the resulting
   bitmap is smaller than the original Graphics expression). Also, as an optimization
   for the time it takes for this conversion process, only operate on notebooks that
   are at least a certain size. This process modifies the notebooks in place.
*)

TagBitmapConditionalizedGraphics[nb_, ids_] := 
 Module[{newids},
	  (*NotebookFind[nb, "Output", All, CellStyle, AutoScroll -> False]; *)
	  newids = "CellID" /. Developer`CellInformation[nb];
	  SelectionMove[nb, Before, Notebook];
	  MapThread[(NotebookFind[nb, "Output", Next, CellStyle, AutoScroll -> False]; If[#1=!=#2, FrontEndExecute[FrontEnd`SelectionAddCellTags[nb, "RasterizedOutput"]]])&, {ids, newids}]
 ]

sizeThresholdToRaster = 2048; (* Bytes *)
badAncestors = Select[Names["System`*"], MemberQ[Attributes[#], HoldAll | HoldAllComplete] &];
Options[ConvertGraphicsToBitmaps] = {"SafeRaster"->True};
ConvertGraphicsToBitmaps[nb_NotebookObject, entityType_, uri_, opts___?OptionQ] :=
 If[entityType === "Symbol" || entityType === "Tutorial" || entityType === "Format" || entityType === "HowTo",
    If[MemberQ[{(*"ref/StationaryWaveletPacketTransform", "ref/PairedHistogram", "ref/StationaryWaveletTransform", "ref/ContourPlot","ref/BarChart"*)}, uri], 
    PubsEvaluateWithFE[
      NotebookFind[nb, "Output", All, CellStyle, AutoScroll -> False]; 
      FrontEndTokenExecute[nb, "SelectionConvert", "Bitmap"]],
      PubsEvaluateWithFE[
      NotebookFind[nb, "Output", All, CellStyle, AutoScroll -> False]; 
	  oldids = "CellID" /. Developer`CellInformation[nb];
      FrontEndTokenExecute[nb, "SelectionConvert", "BitmapConditional"];
	  TagBitmapConditionalizedGraphics[nb, oldids];]
    ]
  ];



(* These functions prevent the FE from constantly reloading stylesheets by creating a
   persistent notebook to keep the stylesheet in memory.  Also, creates a temporary
   inherited stylesheet which kills the docked cell.  Speeds things up. *)
CachedStylesheetPath[file_] := ToFileName[
  {$UserBaseDirectory, "SystemFiles", "FrontEnd", "StyleSheets"},
  PubsEvaluateWithFE[ToFileName[file]]];

CacheStylesheet[nbexpr_, logfile_:Null] :=
  Module[{stylesheet = StyleDefinitions /. Options[nbexpr]},
    If[!ListQ[$StylesheetList], $StylesheetList = {}];
    If[Head[stylesheet] =!= Notebook && Head[stylesheet] =!= Symbol &&!MemberQ[$StylesheetList, stylesheet],
      Print["CacheStylesheet running and about to cache style sheet"];
      If[logfile =!= NULL, WriteLog[logfile,
        "Caching style sheet: " <> ToString[stylesheet], LineSeparator->None, TimeStamp->None]];
      AppendTo[$StylesheetList, stylesheet];
      Quiet[CreateDirectory[DirectoryName[CachedStylesheetPath[stylesheet]]]];
      Put[
        Notebook[{
          Cell[StyleData[StyleDefinitions->stylesheet]],
          Cell[StyleData["Notebook"], DockedCells->{}]}],
        CachedStylesheetPath[stylesheet]
      ];
      PubsEvaluateWithFE[NotebookPut[Notebook[{},StyleDefinitions->stylesheet]]]
    ]
  ];

EmptyStylesheetCache[] :=
  (
  If[ListQ[$StylesheetList],
    Scan[DeleteFile[CachedStylesheetPath[#]]&, $StylesheetList]
  ]; $StylesheetList=.);


(************************  Miscellaneous Utilities  *************************)

exportToXML[con_]:=
Switch[Head@con,
	XMLElement, ExportString[ con, "XML"],
	String, con,
	_, Message[exportToXML::val, Head@con]; con
];
exportToXML::val = "Expected String or XMLElement, found `1`.";



cleanFileNameString[str_String] := 
Module[{res=str},
  res = StringReplace[trimSpaces@res, RegularExpression["[^a-zA-Z0-9$_\\-\\./\\\\]"] -> ""];
  If[Head@res =!= String, Message[cleanFileNameString::reg, str]; str, res]
];
cleanFileNameString[str___] := (Message[cleanFileNameString::arg, str]; $Failed)
cleanFileNameString::reg = "StringReplace error in `1`";
cleanFileNameString::arg = "String expected, found `1`";


trimSpaces[s_String] :=
  StringReplace[s, { StartOfString ~~ Whitespace :> "", Whitespace ~~ EndOfString :> "" } ];


(* in need a stringifyer... *)
convertToString[c_String] := c;
convertToString[c_Cell] := convertToString@c[[1]];
convertToString[c_List] := StringJoin[convertToString /@ c];
convertToString[c_ButtonBox] := convertToString@c[[1]];
convertToString[c_BoxData] := convertToString@c[[1]];
convertToString[c_StyleBox] := convertToString@c[[1]];
convertToString[c_RowBox] := StringJoin[convertToString /@ c[[1]]];
convertToString[c_TextData] := convertToString@c[[1]];
convertToString[c_FormBox] := convertToString@c[[1]];
convertToString[c_AdjustmentBox] := convertToString@c[[1]];
convertToString[c_SuperscriptBox] := StringJoin[convertToString@c[[1]] <> "^"<>convertToString@c[[2]]];
convertToString[c_SubscriptBox] := StringJoin[convertToString@c[[1]] <> "_"<>convertToString@c[[2]]];
convertToString[c_] := ToString@c;
(* warn if a sequence *)
convertToString[c__] := (Message[convertToString::sequence]; StringJoin[convertToString /@ {c}]);
convertToString::sequence = "Warning: Sequence used as an argument, should be List";



(*
	ExcludePathsFromList[list, excludedPaths] 
	removes paths within list which match patterns defined in excludedPaths
*)
ExcludePathsFromList[paths_List, dirs_List] :=
 (DeleteCases[paths, 
     f_ /; ! StringFreeQ[f, Alternatives @@ #, 
        IgnoreCase -> False]] &[
   If[StringQ[#] && 
       StringFreeQ[#, $PathnameSeparator ~~ 
         EndOfString], # <> $PathnameSeparator, #] & /@ dirs])



(*
	ExcludeFilesFromList[list, excludedPaths] 
	removes files within list which match patterns defined in excludedPaths
*)
ExcludeFilesFromList[paths_List, files_List] := 
 DeleteCases[paths, f_ /;
      MemberQ[Alternatives@@files, StringReplace[f, {DirectoryName[f] -> ""}]]]



notebookFileNames[in_String, opts___?OptionQ]:=
Module[{inputDir=in, files={}, logfile, excluded, excludef, filelist, hoursAgo, total},

  {excluded, excludef, filelist, hoursAgo} = 
    {ExcludeDirectories, ExcludeFiles, FileList, HoursAgo} /. {opts} /. Options[notebookFileNames];

  (* quick ExcludeDirectories/ExcludeFiles error checking *)
  If[ !MatchQ[excluded,{(_String|_RegularExpression|_StringExpression)...}],
    Message[ExportOnlineNotebooks::exclude,excluded];
    excluded={}
  ];
  If[ !MatchQ[excludef,{(_String|_RegularExpression|_StringExpression)...}],
    Message[ExportOnlineNotebooks::exclude,excludef];
    excludef={}
  ];

  (* get rid of potentially dangerous pattern *)
  {excluded, excludef} = DeleteCases[#,"",1]&/@{excluded,excludef};
  
  (* add a trailing slash for consistency *)
  If[StringFreeQ[inputDir,$PathnameSeparator~~EndOfString],
    inputDir=inputDir<>$PathnameSeparator
  ];

  (* get /all/ the filenames in inputDir, including unwanted stuff
     this used to only get *.nb files, but it didn't work for directories
     without nbs in them.  this is much more liberal, and as such, is probably
     more prone to mistakes.  use ExcludeFiles (rather than ExcludeDirectories)
     to not create empty directories *)
  If[filelist===All,
    files = FileNames["*.*", inputDir, Infinity]; ,
    files = Union@Flatten@ModifiedFiles[inputDir, "*.*", Infinity, HoursAgo->hoursAgo]  ];
  
  (* now filter out unwanted files based on directories *)
  If[ Length@excluded > 0,
    files = DocumentationBuild`Common`ExcludePathsFromList[files, excluded];
    WriteLog[logfile,
      "Filtered "<>ToString[total-(total=Length[files])]<>" files with Directory filters",
      LineSeparator->None]
  ];

  (* remove non-notebooks *)
  files = Cases[files, f_ /; !StringFreeQ[f,".nb"~~EndOfString, IgnoreCase->True], Infinity];

  (* and any generic files we don't want to process *)
  If[ Length@excludef > 0,
    files = DocumentationBuild`Common`ExcludeFilesFromList[files, excludef];
    WriteLog[logfile,
      "Filtered "<>ToString[total-(total=Length[files])]<>" files with file filters",
      LineSeparator->None]
  ];

 files
];




pathToForwardSlash[s_String]:= StringReplace[s, "\\" -> "/"];
pathToForwardSlash[{s_String}]:= pathToForwardSlash[s];
pathToForwardSlash[s_]:= s;



FilepathToURI[filepath_String]:= FilepathToURI[filepath, "Mathematica"]
FilepathToURI[filepath_String, pacletname_String]:=
Module[{u, paclet=pacletname, lst, path=filepath},
  (* make sure that paclet is named *)
  paclet = 
    If[StringLength[paclet] < 1, 
  	  "Mathematica", 
  	  StringReplace[pacletname, RegularExpression["(.+)\\s+Package"] -> "$1"]
    ];
  (* convert all slashes to foreward *)
  path = DocumentationBuild`Common`Private`pathToForwardSlash @ path;
  (* resolve any relative paths *)
  path = StringReplace[path, RegularExpression["/[^/]+/\\.\\./"] -> "/"]; 
  (* split path on 'paclet' and return second half *)
  (* Hack for webMathematica, which doesn't follow the standard layout *)
  If[StringCount[path, "/"<>paclet<>"/"] > 0,
    lst = StringSplit[path, "/"<>paclet<>"/"],
    lst = StringSplit[path, "/"<>paclet<>"Documentation/"]
  ];
  (* handle files which are named "<paclet>.nb" *)
  u = If[MemberQ[lst, ".nb"], StringJoin[lst[[-2]], paclet, lst[[-1]]], Last@lst];
  u = StringReplace[u, RegularExpression[".*Documentation/(English|Japanese|ChineseSimplified|Spanish)(.+)"] -> "$2"]; 
  (* make sure that incoming path starts with a separator *)
  u = If[StringMatchQ[u, "/*"], u, "/"<>u];
  u = DocumentationBuild`Common`FilePathToDocURI[ u, "/"];
  u = StringReplace[u, RegularExpression["/*(.+)"] -> "$1"]; 

  (* Add paclet name if appropriate *)
  u = If[paclet =!= "Mathematica", StringJoin[paclet, "/", u], u]
  ];


(*  *)
(* TODO: insanely stupid *)
FilePathToDocURI[p_String, l_String:"\\"]:=
  Module[{path = p, ll = If[l === "\\", "\\\\", l]},
  (* lots of replacement rules... *)
  FixedPoint[StringReplace[#, {

    (* FIXME: ugly, ugly mess *)
 	l<>"ReferencePages"<>l<>"C"<>l -> l<>"ref"<>l<>"c"<>l,
 	l<>"ReferencePages"<>l<>"AppleScript"<>l -> l<>"ref"<>l<>"applescript"<>l,
 	l<>"ReferencePages"<>l<>"Characters"<>l -> l<>"ref"<>l<>"character"<>l,
 	l<>"ReferencePages"<>l<>"Formats"<>l -> l<>"ref"<>l<>"format"<>l,
 	l<>"ReferencePages"<>l<>"FrontEndObjects"<>l -> l<>"ref"<>l<>"frontendobject"<>l,
 	l<>"ReferencePages"<>l<>"Indicators"<>l -> l<>"ref"<>l<>"indicator"<>l,
 	l<>"ReferencePages"<>l<>"MenuItems"<>l -> l<>"ref"<>l<>"menuitem"<>l, 
 	l<>"ReferencePages"<>l<>"Messages"<>l -> l<>"ref"<>l<>"message"<>l,
 	l<>"ReferencePages"<>l<>"Methods"<>l -> l<>"ref"<>l<>"method"<>l,
 	l<>"ReferencePages"<>l<>"Programs"<>l -> l<>"ref"<>l<>"program"<>l,
 	l<>"ReferencePages"<>l<>"Files"<>l -> l<>"ref"<>l<>"file"<>l,
 	l<>"ReferencePages"<>l<>"Widgets"<>l -> l<>"ref"<>l<>"widget"<>l,
 	l<>"ReferencePages"<>l<>"Callbacks"<>l -> l<>"ref"<>l<>"callback"<>l,

 	l<>"ReferencePages"<>l<>"Symbols"<>l -> l<>"ref"<>l,

    (* lowercase dirs after References (TODO: sigular v. plural?) *)
 	RegularExpression["[\\\\|/]ReferencePages[\\\\|/](\\w+)[\\\\|/]"] :> ToLowerCase[l<>"ref"<>l<>"$1"<>l],

	l<>"Notes"<>l -> l<>"note"<>l,
  	l<>"Guides"<>l -> l<>"guide"<>l,
  	l<>"ExamplePages"<>l -> l<>"example"<>l,
 	l<>"Tutorials"<>l -> l<>"tutorial"<>l,
 	l<>"HowTo"<>l -> l<>"howto"<>l,
 	l<>"HowTos"<>l -> l<>"howto"<>l,
 	l<>"Screencasts"<>l -> l<>"screencast"<>l,
 	
    l<>"System"<>l -> l,

    l<>"Packages"<>l -> l,
    RegularExpression[ll<>"*Documentation"<>ll<>"English"<>ll ] -> l,
    RegularExpression[ll<>"*Documentation"<>ll<>"Japanese"<>ll ] -> l,
    RegularExpression[ll<>"*Documentation"<>ll<>"ChineseSimplified"<>ll ] -> l,
    RegularExpression[ll<>"*Documentation"<>ll<>"Spanish"<>ll ] -> l,

  	(* Remove nb extension *)
    RegularExpression["(.+)\\.nb$"] -> "$1"

  }]&, path]];



(*
  Add support for running in pre Version 7 for new file operations. 
  These can be added to as needed.   They can be dropped when we 
  don't want to support anything for Version 6.
  
  TWJ
*)

getSeparators[] :=
 	If[$PathnameSeparator === "\\", {"\\", "/"}, {"/"}]

splitFileName[ f_String] :=
 	StringSplit[f, getSeparators[]]

getFileName[f_String] :=
 	Module[{split},
  		split = splitFileName[f];
  		If[ Length[split] === 0, "", Last[split]]
  	]

getFileBaseName[f_String] :=
 	Module[{fName},
  		fName = getFileName[f];
  		StringReplace[fName, "." ~~ (Except["."] ..) ~~ EndOfString -> ""]
  	]

If[ $VersionNumber < 7, ToExpression["DocumentationBuild`Common`FileBaseName = getFileBaseName"]]


End[] (* End Private Context *)

EndPackage[]
